# $Header: /home/mike/cvs/mike/zSQLgate/lib/Net/Z3950/DBIServer/Exception.pm,v 1.3 2002-03-24 23:29:34 mike Exp $

package Net::Z3950::DBIServer::Exception;
use strict;

=head1 NAME

Net::Z3950::DBIServer::Exception - exception thrown by DBIServer module

=head1 SYNOPSIS

	# Somewhere in a function deep in the call tree
	die new Net::Z3950::DBIServer::Exception($bib1ErrorCode, $addInfo);

	# And higher up the call tree we invoke and check like this:
	eval {
	    functionWhichMightThrowAnException();
	}; if ($@ && $@->isa('Net::Z3950::DBIServer::Exception')) {
	    warn "Bib-1 error: " . $@->code() . ", " . $@->addinfo();
	} elsif ($@) {
	    die $@;
	}

=head1 DESCRIPTION

A tiny class which merely encapsulates a BIB1 diagnostic so that it
can be generated by low-level functions and picked up at a higher
level.  Exists only to be thrown (oh, alright then, C<die()>d :-)

Note that code which catches exceptions thrown in this way needs to be
careful that it only catches the right kinds of objects, as the code
above does.  This is essential as Perl itself will call C<die()> -
with a string argument - when certain ``real errors'' occur, such as
trying to call a function which has not been defined.

=head1 METHODS

=head2 new()

	new Net::Z3950::DBIServer::Exception($bib1ErrorCode, $addInfo);

Creates and returns a new Exception object with the specified Bib-1
error code and associated additional information.  The only thing you
can do with the object thereafter is retrieve the error code and
additional information with which it was created.  It is intended to
be used as the argument to C<die()>, which is Perl's, shall we say,
idiosyncratic exception-throwing primitive.

=cut

sub new {
    my $class = shift();
    my($code, $addinfo) = @_;

    return bless {
	code => $code,
	addinfo => $addinfo || "",
    }, $class;
}


=head2 throw()

	Net::Z3950::DBIServer::Exception::throw($bib1ErrorCode, $addInfo);

This convenience method is exactly equivalent to

	die new Net::Z3950::DBIServer::Exception($bib1ErrorCode, $addInfo);

but one character shorter!

=cut

sub throw {
    my($code, $addinfo) = @_;

    die __PACKAGE__->new($code, $addinfo);
}


=head2 code()

	$bib1ErrorCode = $exception->code();

Returns the Bib-1 error code with which the exception was created.

=cut

sub code {
    my $this = shift();

    return $this->{code};
}


=head2 addinfo()

	$addInfo = $exception->addinfo();

Returns the additional information with which the exception was created.

=cut

sub addinfo {
    my $this = shift();

    return $this->{addinfo};
}


=head1 AUTHOR

Mike Taylor E<lt>mike@miketaylor.org.ukE<gt>

First version Saturday 23rd February 2002.

=head1 SEE ALSO

C<Net::Z3950::DBIServer>
and
C<Net::Z3950::DBIServer::ResultSet>
are the modules that use this.

The numeric codes, their interpretations, and the form of additional
information that should accompany them are described in Appendix ERR.1
of the Z39.50-1995 standard document at
http://lcweb.loc.gov/z3950/agency/defns/bib1diag.html

Isn't it great that the standard is freely available on the web?  Hang
your heads in shame, ISO!

=cut


1;
