package PDL::Demos::PGPLOT;
use PDL::Graphics::PGPLOT;

sub info {('pgplot', 'PGPLOT graphics output')}

sub init {'
$ENV{PGPLOT_XW_WIDTH}=0.3;
$ENV{PGPLOT_DEV}=$^O =~ /MSWin32/ ? "/GW" : "/XW";
use PDL::Graphics::PGPLOT;
'}
my @demo = (
[comment => q|
    Welcome to this tour of the PDL's PGPLOT interface.

    This tour will introduce the PDL's PGPLOT plotting module and show
    what this powerful package can provide in terms of plotting. It is
    not designed to give a full tour of PGPLOT, you are advised to see
    the routines provided with pgperl for that.

    The PDL::Graphics::PGPLOT module provides a high-level interface
    to PGPLOT. However if you want even better control of your plots
    you might want to include the PGPLOT module specifically:

       use PGPLOT;

    One aspect of PGPLOT that requires mention is the use of devices:
    Normally PGPLOT will inquire you about what device you want to use,
    with the prompt:

        Graphics device/type (? to see list, default /NULL):
|],

[act => q|
    # ensure the module is loaded
    use PDL::Graphics::PGPLOT;
    # The size of the window can be specified
    $ENV{PGPLOT_XW_WIDTH}=0.3;
    # You can set your device explicitly
    $id=dev($ENV{PGPLOT_DEV}); # '/XW' on X, '/GW' on Win32
|],

[act => q|
    # First we define some variables to use for the rest of the demo.
    $x=sequence(10);
    $y=2*$x**2;

    # Now a simple plot with points
    points $x, $y;
|],

[act => q|
    # Here is the same with lines
    line $x, $y;
|],

[act => q|
    # If you want to overlay one plot you can use the command
    # 'hold' to put the graphics on hold and 'release' to
    # revert the effect

    points $x, $y, {SYMBOL=>4};  # The last argument sets symboltype
    hold;
    # Now draw lines between the points
    line $x, $y;
    # Plot errorbars over the points
    $yerr=sqrt($y);
    errb $x, $y, $yerr;

    # To revert to old behaviour, use release
    release;
|],

[act => q|
    bin $x, $y;

    # This plots a binned histogram of the data and as you can
    # see it made a new plot.
|],

[act => q|
    # 2D data can also easily be accommodated:

    # First make a simple image
    $gradient=sequence(40,40);

    # Then display it.
    imag $gradient;

    # And overlay a contour plot over it:
    hold;
    cont $gradient;
    release;
|],

[act => q|
  # PDL::Graphics::PGPLOT contains several colour tables,
  # a more extensive collection can be found in
  # PDL::Graphics::LUT
  #
  # (note: the call to lut_names() can take a few seconds to execute)
  #
  use PDL::Graphics::LUT;
  @names = lut_names();
  print "Available tables: [ ", @names, " ]\n";

  # use the first table
  ctab( lut_data($names[0]) );
  use PGPLOT;
  pglabel "", "", "Colour table: $names[0]";
|],

[act => q|
    # To change plot specifics you can either use the specific PGPLOT
    # commands - recommended if you need lots of control over your
    # plot.
    #
    # Or you can use the new option specifications:

    # To plot our first graph again with blue color, dashed line
    # and a thickness of 10 we can do:

    line $x, $y, {COLOR=>5, LINESTYLE=>'dashed', LINEWIDTH=>10};
|],

[act => q|

  # Now for a more complicated example.
  # First create some data
  $x=sequence(360)*3.1415/180.;
  $y=sin($x)*transpose(cos($x));

  # Make an ndarray with the wanted contours
  $contours=pdl [0.1,0.5,1.0];
  # And an array (reference to an array) with labels
  $labels=['A', 'B', 'C'];
  # Create a contour map of the data - note that we can set the colour of
  # the labels.
  cont($y, {CONTOURS=>$contours, linest=>'DASHED',
	    LINEWIDTH=>3, COLOR=>2, LABELCOL=>4});
  hold;

  pgqlw($linewidth);

  points $x->slice('0:-1:4')*180./3.1415;
  release;
|],

[act => q|
  #
  # More examples of changing the plot defaults
  #
  $x = 1+sequence(10);
  $y = $x*2;
  $bord_opt = { TYPE => 'RELATIVE', VALUE => 0.1 };
  line log10($x), $y, { AXIS => 'LOGX', BORDER => $bord_opt };
|],

[act => q|
  #
  # We can also create vector maps of data
  # This requires one array for the horizontal component and
  # one for the vertical component
  #
  $horizontal=sequence(10,10);
  $vertical=transpose($horizontal)+random(10,10)*$horizontal/10.;

  $arrow={ARROW=> {FS=>1, ANGLE=>25, VENT=>0.7, SIZE=>3}};
  vect $horizontal, $vertical, {ARROW=>$arrow, COLOR=>RED};
  hold;
  cont $vertical-$horizontal, {COLOR=>YELLOW};
  release;
|],

[act => q|
  #
  # To draw [filled] polygons, the command poly is handy:
  #

  $x=sequence(10)/5;
  poly $x, $x**2, {FILL=>HATCHED, COLOR=>BLUE};
|],

[act => q|
  #
  # the latest feature of PDL are complex numbers
  # so let's play with a simple example
  #


  $z50 = zeroes(50);
  $c = czip($z50->xlinvals(0,7), $z50->xlinvals(2,4));
  line sin($c)->im; hold;      # the imaginary part
  line sin($c)->re;            # real
  line abs sin $c; release;  # and the modulus
|],

[act => q|
  #
  # more complex numbers
  #

  $c =  czip(zeroes(300)->xlinvals(0,12), zeroes(300)->xlinvals(2,10));
  $sin = sin $c;
  line $sin->im, $sin->re;   # look at the result in the complex plane
|],

[act => q|
  #close the window--we're done!
  close_window($id);
  print "On X Windows, you need to close the 'PGPLOT Server' window.\n";
|],
);

sub demo { @demo }

1;

=head1 NAME

PDL::Demos::PGPLOT - demonstrate PDL::Graphics::PGPLOT capabilities

=head1 SYNOPSIS

  pdl> demo pgplot

=cut
