package OpenInteract::User;

# $Id: User.pm,v 1.11 2002/01/28 13:02:27 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level :scope );

$OpenInteract::User::VERSION = sprintf("%d.%02d", q$Revision: 1.11 $ =~ /(\d+)\.(\d+)/);
$OpenInteract::User::crypt_password = undef;


sub _class_initialize {
    my ( $class, $CONFIG ) = @_;
    no strict 'refs';
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 2, "Set $class to use crypt ($CONFIG->{login}{crypt_password})" );
    ${ $class . '::crypt_password' } = $CONFIG->{login}{crypt_password};
}


########################################
# CLASS METHODS

sub fetch_by_login_name {
    my ( $class, $login_name, $p ) = @_;
    $p ||= {};
    my $oper = $class->sql_case_insensitive;
    my $user_list = $class->fetch_group({ where => "login_name $oper ?",
                                          value => [ $login_name ],
                                          %{ $p } });
    if ( $p->{return_single} ) {
        return $user_list->[0];
    }
    return $user_list;
}


########################################
# OBJECT METHODS

sub make_public {
    my ( $self ) = @_;
    my $R = OpenInteract::Request->instance;

 # First find the public group

    my $groups = eval { $R->group->fetch_group({ where => 'name = ?',
                                                 value => [ 'public' ] }) };
    if ( my $public = $groups->[0] ) {

        # Then add the user to it

        eval { $self->group_add( [ $public->{group_id} ] ); };
        if ( $@ ) {
            OpenInteract::Error->set( SPOPS::Error->get );
            $OpenInteract::Error::user_msg = 'Cannot make user part of public group.';
            die $OpenInteract::Error::user_msg;
        }

        # Then ensure the public can see (for now) this user

        eval { $self->set_security({ scope    => SEC_SCOPE_GROUP,
                                     scope_id => $public->{group_id},
                                     level    => SEC_LEVEL_READ }) };
        if ( $@ ) {
            OpenInteract::Error->set( SPOPS::Error->get );
            $OpenInteract::Error::user_msg = 'User is part of public group, but public group cannot see user.';
            die $OpenInteract::Error::user_msg;
        }
    }
    return 1;
}


sub full_name { return join ' ', $_[0]->{first_name}, $_[0]->{last_name} }


sub increment_login {
    my ( $self ) = @_;
    $self->{last_login} = $self->now;
    $self->{num_logins}++;
    eval { $self->save };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $OpenInteract::Error::user_msg = 'Cannot incremement login number or set the login date to now.';
        die $OpenInteract::Error::user_msg;
    }
    return 1;
}



sub check_password {
    my ( $self, $check_pw ) = @_;
    return undef unless ( $check_pw );
    my $R = OpenInteract::Request->instance;
    my $exist_pw = $self->{password};
    no strict 'refs';
    my $class = ref $self;
    my $use_crypt = ${ $class . '::crypt_password' };
    if ( $use_crypt ) {
        $R->DEBUG && $R->scrib( 1, "Checking using the crypt() function." );
        return ( crypt( $check_pw, $exist_pw ) eq $exist_pw );
    } 
    return ( $check_pw eq $exist_pw );
}


sub is_in_group {
    my ( $self, $group_spec ) = @_;
    my $R = OpenInteract::Request->instance;
    my $check_group_id = ( ref $group_spec )
                           ? $group_spec->id : $group_spec;
    my $type = $R->CONFIG->{id}{group_type};
    foreach my $group ( @{ $R->{auth}{group} } ) {
        return 1 if ( $type eq 'int' and $group->id == $check_group_id );
        return 1 if ( $type eq 'char' and $group->id eq $check_group_id );
    }
    return undef;
}

1;

__END__

=pod

=head1 NAME

OpenInteract::User - Create and manipulate users. 

=head1 SYNOPSIS

  use OpenInteract::User;
  $user = OpenInteract::User->new();

  # Increment the user's login total
  $user->increment_login();
  print "Username: $user->{username}\n";

  # See if the user is in a particular group
  if ( $user->is_in_group( $group ) ) {
     print "Enter!";
  }


=head1 DESCRIPTION

Basic methods for user objects

=head1 METHODS

=head2 Class Methods

B<fetch_by_login_name( $login_name, [ \%params ] )>

Retrieve a single user by C<$login_name>. This performs a
case-insensitive search (if available). Pass a true value for
'return_single' in C<\%params> to return a single object, and pass a
true value for 'skip_security' in C<\%params> to skip security
checks. (Useful when finding a user to login with.)

=head2 Object Methods

B<full_name()>

Returns the full name -- it is accessed often enough that we just made
an alias for concatenating the first and last names.

B<increment_login()>

Increments the number of logins for this user and sets the lastlogin
date to today.

B<check_password( $pw )>

Return a 1 if the password matches what is in the database, a 0 if
not.

B<is_in_group( $group | $group_id )>

Ask a user object if it belongs to a particular group. You can pass
either a group object or its ID.

Returns true if the user is in the group, false, if not.

=head1 TO DO

Nothing known.

=head1 BUGS

None known.

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
