use Test::More;
use strict;

use Geo::Address::Mail::US;
use Geo::Address::Mail::Standardizer::USPS;

my $std = Geo::Address::Mail::Standardizer::USPS->new;

my %state_or_province_designator = (
    'AA' => [
        'A F A',                  'A F A.',
        'A F AM',                 'A F AM.',
        'A F AMER',               'A F AMER.',
        'A F AMERICA',            'A F. A',
        'A F. A.',                'A F. AM',
        'A F. AM.',               'A F. AMER',
        'A F. AMER.',             'A F. AMERICA',
        'A FOR A',                'A FOR A.',
        'A FOR AM',               'A FOR AM.',
        'A FOR AMER',             'A FOR AMER.',
        'A FOR AMERICA',          'A FOR. A',
        'A FOR. A.',              'A FOR. AM',
        'A FOR. AM.',             'A FOR. AMER',
        'A FOR. AMER.',           'A FOR. AMERICA',
        'A FORCE A',              'A FORCE A.',
        'A FORCE AM',             'A FORCE AM.',
        'A FORCE AMER',           'A FORCE AMER.',
        'A FORCE AMERICA',        'A FORCE. A',
        'A FORCE. A.',            'A FORCE. AM',
        'A FORCE. AM.',           'A FORCE. AMER',
        'A FORCE. AMER.',         'A FORCE. AMERICA',
        'A FORCES A',             'A FORCES A.',
        'A FORCES AM',            'A FORCES AM.',
        'A FORCES AMER',          'A FORCES AMER.',
        'A FORCES AMERICA',       'A. F A',
        'A. F A.',                'A. F AM',
        'A. F AM.',               'A. F AMER',
        'A. F AMER.',             'A. F AMERICA',
        'A. F. A',                'A. F. A.',
        'A. F. AM',               'A. F. AM.',
        'A. F. AMER',             'A. F. AMER.',
        'A. F. AMERICA',          'A. FOR A',
        'A. FOR A.',              'A. FOR AM',
        'A. FOR AM.',             'A. FOR AMER',
        'A. FOR AMER.',           'A. FOR AMERICA',
        'A. FOR. A',              'A. FOR. A.',
        'A. FOR. AM',             'A. FOR. AM.',
        'A. FOR. AMER',           'A. FOR. AMER.',
        'A. FOR. AMERICA',        'A. FORCE A',
        'A. FORCE A.',            'A. FORCE AM',
        'A. FORCE AM.',           'A. FORCE AMER',
        'A. FORCE AMER.',         'A. FORCE AMERICA',
        'A. FORCE. A',            'A. FORCE. A.',
        'A. FORCE. AM',           'A. FORCE. AM.',
        'A. FORCE. AMER',         'A. FORCE. AMER.',
        'A. FORCE. AMERICA',      'A. FORCES A',
        'A. FORCES A.',           'A. FORCES AM',
        'A. FORCES AM.',          'A. FORCES AMER',
        'A. FORCES AMER.',        'A. FORCES AMERICA',
        'A.F.A',                  'A.F.A.',
        'A.F.AM',                 'A.F.AM.',
        'A.F.AMER',               'A.F.AMER.',
        'A.F.AMERICA',            'A.FA',
        'A.FA.',                  'A.FAM',
        'A.FAM.',                 'A.FAMER',
        'A.FAMER.',               'A.FAMERICA',
        'A.FOR.A',                'A.FOR.A.',
        'A.FOR.AM',               'A.FOR.AM.',
        'A.FOR.AMER',             'A.FOR.AMER.',
        'A.FOR.AMERICA',          'A.FORA',
        'A.FORA.',                'A.FORAM',
        'A.FORAM.',               'A.FORAMER',
        'A.FORAMER.',             'A.FORAMERICA',
        'A.FORCE.A',              'A.FORCE.A.',
        'A.FORCE.AM',             'A.FORCE.AM.',
        'A.FORCE.AMER',           'A.FORCE.AMER.',
        'A.FORCE.AMERICA',        'A.FORCEA',
        'A.FORCEA.',              'A.FORCEAM',
        'A.FORCEAM.',             'A.FORCEAMER',
        'A.FORCEAMER.',           'A.FORCEAMERICA',
        'A.FORCESA',              'A.FORCESA.',
        'A.FORCESAM',             'A.FORCESAM.',
        'A.FORCESAMER',           'A.FORCESAMER.',
        'A.FORCESAMERICA',        'AA',
        'AA.',                    'AF.A',
        'AF.A.',                  'AF.AM',
        'AF.AM.',                 'AF.AMER',
        'AF.AMER.',               'AF.AMERICA',
        'AFA',                    'AFA.',
        'AFAM',                   'AFAM.',
        'AFAMER',                 'AFAMER.',
        'AFAMERICA',              'AFOR.A',
        'AFOR.A.',                'AFOR.AM',
        'AFOR.AM.',               'AFOR.AMER',
        'AFOR.AMER.',             'AFOR.AMERICA',
        'AFORA',                  'AFORA.',
        'AFORAM',                 'AFORAM.',
        'AFORAMER',               'AFORAMER.',
        'AFORAMERICA',            'AFORCE.A',
        'AFORCE.A.',              'AFORCE.AM',
        'AFORCE.AM.',             'AFORCE.AMER',
        'AFORCE.AMER.',           'AFORCE.AMERICA',
        'AFORCEA',                'AFORCEA.',
        'AFORCEAM',               'AFORCEAM.',
        'AFORCEAMER',             'AFORCEAMER.',
        'AFORCEAMERICA',          'AFORCESA',
        'AFORCESA.',              'AFORCESAM',
        'AFORCESAM.',             'AFORCESAMER',
        'AFORCESAMER.',           'AFORCESAMERICA',
        'ARM F A',                'ARM F A.',
        'ARM F AM',               'ARM F AM.',
        'ARM F AMER',             'ARM F AMER.',
        'ARM F AMERICA',          'ARM F. A',
        'ARM F. A.',              'ARM F. AM',
        'ARM F. AM.',             'ARM F. AMER',
        'ARM F. AMER.',           'ARM F. AMERICA',
        'ARM FOR A',              'ARM FOR A.',
        'ARM FOR AM',             'ARM FOR AM.',
        'ARM FOR AMER',           'ARM FOR AMER.',
        'ARM FOR AMERICA',        'ARM FOR. A',
        'ARM FOR. A.',            'ARM FOR. AM',
        'ARM FOR. AM.',           'ARM FOR. AMER',
        'ARM FOR. AMER.',         'ARM FOR. AMERICA',
        'ARM FORCE A',            'ARM FORCE A.',
        'ARM FORCE AM',           'ARM FORCE AM.',
        'ARM FORCE AMER',         'ARM FORCE AMER.',
        'ARM FORCE AMERICA',      'ARM FORCE. A',
        'ARM FORCE. A.',          'ARM FORCE. AM',
        'ARM FORCE. AM.',         'ARM FORCE. AMER',
        'ARM FORCE. AMER.',       'ARM FORCE. AMERICA',
        'ARM FORCES A',           'ARM FORCES A.',
        'ARM FORCES AM',          'ARM FORCES AM.',
        'ARM FORCES AMER',        'ARM FORCES AMER.',
        'ARM FORCES AMERICA',     'ARM\'D F A',
        'ARM\'D F A.',            'ARM\'D F AM',
        'ARM\'D F AM.',           'ARM\'D F AMER',
        'ARM\'D F AMER.',         'ARM\'D F AMERICA',
        'ARM\'D F. A',            'ARM\'D F. A.',
        'ARM\'D F. AM',           'ARM\'D F. AM.',
        'ARM\'D F. AMER',         'ARM\'D F. AMER.',
        'ARM\'D F. AMERICA',      'ARM\'D FOR A',
        'ARM\'D FOR A.',          'ARM\'D FOR AM',
        'ARM\'D FOR AM.',         'ARM\'D FOR AMER',
        'ARM\'D FOR AMER.',       'ARM\'D FOR AMERICA',
        'ARM\'D FOR. A',          'ARM\'D FOR. A.',
        'ARM\'D FOR. AM',         'ARM\'D FOR. AM.',
        'ARM\'D FOR. AMER',       'ARM\'D FOR. AMER.',
        'ARM\'D FOR. AMERICA',    'ARM\'D FORCE A',
        'ARM\'D FORCE A.',        'ARM\'D FORCE AM',
        'ARM\'D FORCE AM.',       'ARM\'D FORCE AMER',
        'ARM\'D FORCE AMER.',     'ARM\'D FORCE AMERICA',
        'ARM\'D FORCE. A',        'ARM\'D FORCE. A.',
        'ARM\'D FORCE. AM',       'ARM\'D FORCE. AM.',
        'ARM\'D FORCE. AMER',     'ARM\'D FORCE. AMER.',
        'ARM\'D FORCE. AMERICA',  'ARM\'D FORCES A',
        'ARM\'D FORCES A.',       'ARM\'D FORCES AM',
        'ARM\'D FORCES AM.',      'ARM\'D FORCES AMER',
        'ARM\'D FORCES AMER.',    'ARM\'D FORCES AMERICA',
        'ARM\'D. F A',            'ARM\'D. F A.',
        'ARM\'D. F AM',           'ARM\'D. F AM.',
        'ARM\'D. F AMER',         'ARM\'D. F AMER.',
        'ARM\'D. F AMERICA',      'ARM\'D. F. A',
        'ARM\'D. F. A.',          'ARM\'D. F. AM',
        'ARM\'D. F. AM.',         'ARM\'D. F. AMER',
        'ARM\'D. F. AMER.',       'ARM\'D. F. AMERICA',
        'ARM\'D. FOR A',          'ARM\'D. FOR A.',
        'ARM\'D. FOR AM',         'ARM\'D. FOR AM.',
        'ARM\'D. FOR AMER',       'ARM\'D. FOR AMER.',
        'ARM\'D. FOR AMERICA',    'ARM\'D. FOR. A',
        'ARM\'D. FOR. A.',        'ARM\'D. FOR. AM',
        'ARM\'D. FOR. AM.',       'ARM\'D. FOR. AMER',
        'ARM\'D. FOR. AMER.',     'ARM\'D. FOR. AMERICA',
        'ARM\'D. FORCE A',        'ARM\'D. FORCE A.',
        'ARM\'D. FORCE AM',       'ARM\'D. FORCE AM.',
        'ARM\'D. FORCE AMER',     'ARM\'D. FORCE AMER.',
        'ARM\'D. FORCE AMERICA',  'ARM\'D. FORCE. A',
        'ARM\'D. FORCE. A.',      'ARM\'D. FORCE. AM',
        'ARM\'D. FORCE. AM.',     'ARM\'D. FORCE. AMER',
        'ARM\'D. FORCE. AMER.',   'ARM\'D. FORCE. AMERICA',
        'ARM\'D. FORCES A',       'ARM\'D. FORCES A.',
        'ARM\'D. FORCES AM',      'ARM\'D. FORCES AM.',
        'ARM\'D. FORCES AMER',    'ARM\'D. FORCES AMER.',
        'ARM\'D. FORCES AMERICA', 'ARM\'D.F.A',
        'ARM\'D.F.A.',            'ARM\'D.F.AM',
        'ARM\'D.F.AM.',           'ARM\'D.F.AMER',
        'ARM\'D.F.AMER.',         'ARM\'D.F.AMERICA',
        'ARM\'D.FA',              'ARM\'D.FA.',
        'ARM\'D.FAM',             'ARM\'D.FAM.',
        'ARM\'D.FAMER',           'ARM\'D.FAMER.',
        'ARM\'D.FAMERICA',        'ARM\'D.FOR.A',
        'ARM\'D.FOR.A.',          'ARM\'D.FOR.AM',
        'ARM\'D.FOR.AM.',         'ARM\'D.FOR.AMER',
        'ARM\'D.FOR.AMER.',       'ARM\'D.FOR.AMERICA',
        'ARM\'D.FORA',            'ARM\'D.FORA.',
        'ARM\'D.FORAM',           'ARM\'D.FORAM.',
        'ARM\'D.FORAMER',         'ARM\'D.FORAMER.',
        'ARM\'D.FORAMERICA',      'ARM\'D.FORCE.A',
        'ARM\'D.FORCE.A.',        'ARM\'D.FORCE.AM',
        'ARM\'D.FORCE.AM.',       'ARM\'D.FORCE.AMER',
        'ARM\'D.FORCE.AMER.',     'ARM\'D.FORCE.AMERICA',
        'ARM\'D.FORCEA',          'ARM\'D.FORCEA.',
        'ARM\'D.FORCEAM',         'ARM\'D.FORCEAM.',
        'ARM\'D.FORCEAMER',       'ARM\'D.FORCEAMER.',
        'ARM\'D.FORCEAMERICA',    'ARM\'D.FORCESA',
        'ARM\'D.FORCESA.',        'ARM\'D.FORCESAM',
        'ARM\'D.FORCESAM.',       'ARM\'D.FORCESAMER',
        'ARM\'D.FORCESAMER.',     'ARM\'D.FORCESAMERICA',
        'ARM\'DF.A',              'ARM\'DF.A.',
        'ARM\'DF.AM',             'ARM\'DF.AM.',
        'ARM\'DF.AMER',           'ARM\'DF.AMER.',
        'ARM\'DF.AMERICA',        'ARM\'DFA',
        'ARM\'DFA.',              'ARM\'DFAM',
        'ARM\'DFAM.',             'ARM\'DFAMER',
        'ARM\'DFAMER.',           'ARM\'DFAMERICA',
        'ARM\'DFOR.A',            'ARM\'DFOR.A.',
        'ARM\'DFOR.AM',           'ARM\'DFOR.AM.',
        'ARM\'DFOR.AMER',         'ARM\'DFOR.AMER.',
        'ARM\'DFOR.AMERICA',      'ARM\'DFORA',
        'ARM\'DFORA.',            'ARM\'DFORAM',
        'ARM\'DFORAM.',           'ARM\'DFORAMER',
        'ARM\'DFORAMER.',         'ARM\'DFORAMERICA',
        'ARM\'DFORCE.A',          'ARM\'DFORCE.A.',
        'ARM\'DFORCE.AM',         'ARM\'DFORCE.AM.',
        'ARM\'DFORCE.AMER',       'ARM\'DFORCE.AMER.',
        'ARM\'DFORCE.AMERICA',    'ARM\'DFORCEA',
        'ARM\'DFORCEA.',          'ARM\'DFORCEAM',
        'ARM\'DFORCEAM.',         'ARM\'DFORCEAMER',
        'ARM\'DFORCEAMER.',       'ARM\'DFORCEAMERICA',
        'ARM\'DFORCESA',          'ARM\'DFORCESA.',
        'ARM\'DFORCESAM',         'ARM\'DFORCESAM.',
        'ARM\'DFORCESAMER',       'ARM\'DFORCESAMER.',
        'ARM\'DFORCESAMERICA',    'ARM. F A',
        'ARM. F A.',              'ARM. F AM',
        'ARM. F AM.',             'ARM. F AMER',
        'ARM. F AMER.',           'ARM. F AMERICA',
        'ARM. F. A',              'ARM. F. A.',
        'ARM. F. AM',             'ARM. F. AM.',
        'ARM. F. AMER',           'ARM. F. AMER.',
        'ARM. F. AMERICA',        'ARM. FOR A',
        'ARM. FOR A.',            'ARM. FOR AM',
        'ARM. FOR AM.',           'ARM. FOR AMER',
        'ARM. FOR AMER.',         'ARM. FOR AMERICA',
        'ARM. FOR. A',            'ARM. FOR. A.',
        'ARM. FOR. AM',           'ARM. FOR. AM.',
        'ARM. FOR. AMER',         'ARM. FOR. AMER.',
        'ARM. FOR. AMERICA',      'ARM. FORCE A',
        'ARM. FORCE A.',          'ARM. FORCE AM',
        'ARM. FORCE AM.',         'ARM. FORCE AMER',
        'ARM. FORCE AMER.',       'ARM. FORCE AMERICA',
        'ARM. FORCE. A',          'ARM. FORCE. A.',
        'ARM. FORCE. AM',         'ARM. FORCE. AM.',
        'ARM. FORCE. AMER',       'ARM. FORCE. AMER.',
        'ARM. FORCE. AMERICA',    'ARM. FORCES A',
        'ARM. FORCES A.',         'ARM. FORCES AM',
        'ARM. FORCES AM.',        'ARM. FORCES AMER',
        'ARM. FORCES AMER.',      'ARM. FORCES AMERICA',
        'ARM.F.A',                'ARM.F.A.',
        'ARM.F.AM',               'ARM.F.AM.',
        'ARM.F.AMER',             'ARM.F.AMER.',
        'ARM.F.AMERICA',          'ARM.FA',
        'ARM.FA.',                'ARM.FAM',
        'ARM.FAM.',               'ARM.FAMER',
        'ARM.FAMER.',             'ARM.FAMERICA',
        'ARM.FOR.A',              'ARM.FOR.A.',
        'ARM.FOR.AM',             'ARM.FOR.AM.',
        'ARM.FOR.AMER',           'ARM.FOR.AMER.',
        'ARM.FOR.AMERICA',        'ARM.FORA',
        'ARM.FORA.',              'ARM.FORAM',
        'ARM.FORAM.',             'ARM.FORAMER',
        'ARM.FORAMER.',           'ARM.FORAMERICA',
        'ARM.FORCE.A',            'ARM.FORCE.A.',
        'ARM.FORCE.AM',           'ARM.FORCE.AM.',
        'ARM.FORCE.AMER',         'ARM.FORCE.AMER.',
        'ARM.FORCE.AMERICA',      'ARM.FORCEA',
        'ARM.FORCEA.',            'ARM.FORCEAM',
        'ARM.FORCEAM.',           'ARM.FORCEAMER',
        'ARM.FORCEAMER.',         'ARM.FORCEAMERICA',
        'ARM.FORCESA',            'ARM.FORCESA.',
        'ARM.FORCESAM',           'ARM.FORCESAM.',
        'ARM.FORCESAMER',         'ARM.FORCESAMER.',
        'ARM.FORCESAMERICA',      'ARMED F A',
        'ARMED F A.',             'ARMED F AM',
        'ARMED F AM.',            'ARMED F AMER',
        'ARMED F AMER.',          'ARMED F AMERICA',
        'ARMED F. A',             'ARMED F. A.',
        'ARMED F. AM',            'ARMED F. AM.',
        'ARMED F. AMER',          'ARMED F. AMER.',
        'ARMED F. AMERICA',       'ARMED FOR A',
        'ARMED FOR A.',           'ARMED FOR AM',
        'ARMED FOR AM.',          'ARMED FOR AMER',
        'ARMED FOR AMER.',        'ARMED FOR AMERICA',
        'ARMED FOR. A',           'ARMED FOR. A.',
        'ARMED FOR. AM',          'ARMED FOR. AM.',
        'ARMED FOR. AMER',        'ARMED FOR. AMER.',
        'ARMED FOR. AMERICA',     'ARMED FORCE A',
        'ARMED FORCE A.',         'ARMED FORCE AM',
        'ARMED FORCE AM.',        'ARMED FORCE AMER',
        'ARMED FORCE AMER.',      'ARMED FORCE AMERICA',
        'ARMED FORCE. A',         'ARMED FORCE. A.',
        'ARMED FORCE. AM',        'ARMED FORCE. AM.',
        'ARMED FORCE. AMER',      'ARMED FORCE. AMER.',
        'ARMED FORCE. AMERICA',   'ARMED FORCES A',
        'ARMED FORCES A.',        'ARMED FORCES AM',
        'ARMED FORCES AM.',       'ARMED FORCES AMER',
        'ARMED FORCES AMER.',     'ARMED FORCES AMERICA',
        'ARMEDF.A',               'ARMEDF.A.',
        'ARMEDF.AM',              'ARMEDF.AM.',
        'ARMEDF.AMER',            'ARMEDF.AMER.',
        'ARMEDF.AMERICA',         'ARMEDFA',
        'ARMEDFA.',               'ARMEDFAM',
        'ARMEDFAM.',              'ARMEDFAMER',
        'ARMEDFAMER.',            'ARMEDFAMERICA',
        'ARMEDFOR.A',             'ARMEDFOR.A.',
        'ARMEDFOR.AM',            'ARMEDFOR.AM.',
        'ARMEDFOR.AMER',          'ARMEDFOR.AMER.',
        'ARMEDFOR.AMERICA',       'ARMEDFORA',
        'ARMEDFORA.',             'ARMEDFORAM',
        'ARMEDFORAM.',            'ARMEDFORAMER',
        'ARMEDFORAMER.',          'ARMEDFORAMERICA',
        'ARMEDFORCE.A',           'ARMEDFORCE.A.',
        'ARMEDFORCE.AM',          'ARMEDFORCE.AM.',
        'ARMEDFORCE.AMER',        'ARMEDFORCE.AMER.',
        'ARMEDFORCE.AMERICA',     'ARMEDFORCEA',
        'ARMEDFORCEA.',           'ARMEDFORCEAM',
        'ARMEDFORCEAM.',          'ARMEDFORCEAMER',
        'ARMEDFORCEAMER.',        'ARMEDFORCEAMERICA',
        'ARMEDFORCESA',           'ARMEDFORCESA.',
        'ARMEDFORCESAM',          'ARMEDFORCESAM.',
        'ARMEDFORCESAMER',        'ARMEDFORCESAMER.',
        'ARMEDFORCESAMERICA',     'ARMF.A',
        'ARMF.A.',                'ARMF.AM',
        'ARMF.AM.',               'ARMF.AMER',
        'ARMF.AMER.',             'ARMF.AMERICA',
        'ARMFA',                  'ARMFA.',
        'ARMFAM',                 'ARMFAM.',
        'ARMFAMER',               'ARMFAMER.',
        'ARMFAMERICA',            'ARMFOR.A',
        'ARMFOR.A.',              'ARMFOR.AM',
        'ARMFOR.AM.',             'ARMFOR.AMER',
        'ARMFOR.AMER.',           'ARMFOR.AMERICA',
        'ARMFORA',                'ARMFORA.',
        'ARMFORAM',               'ARMFORAM.',
        'ARMFORAMER',             'ARMFORAMER.',
        'ARMFORAMERICA',          'ARMFORCE.A',
        'ARMFORCE.A.',            'ARMFORCE.AM',
        'ARMFORCE.AM.',           'ARMFORCE.AMER',
        'ARMFORCE.AMER.',         'ARMFORCE.AMERICA',
        'ARMFORCEA',              'ARMFORCEA.',
        'ARMFORCEAM',             'ARMFORCEAM.',
        'ARMFORCEAMER',           'ARMFORCEAMER.',
        'ARMFORCEAMERICA',        'ARMFORCESA',
        'ARMFORCESA.',            'ARMFORCESAM',
        'ARMFORCESAM.',           'ARMFORCESAMER',
        'ARMFORCESAMER.',         'ARMFORCESAMERICA',
        'ARM`D F A',              'ARM`D F A.',
        'ARM`D F AM',             'ARM`D F AM.',
        'ARM`D F AMER',           'ARM`D F AMER.',
        'ARM`D F AMERICA',        'ARM`D F. A',
        'ARM`D F. A.',            'ARM`D F. AM',
        'ARM`D F. AM.',           'ARM`D F. AMER',
        'ARM`D F. AMER.',         'ARM`D F. AMERICA',
        'ARM`D FOR A',            'ARM`D FOR A.',
        'ARM`D FOR AM',           'ARM`D FOR AM.',
        'ARM`D FOR AMER',         'ARM`D FOR AMER.',
        'ARM`D FOR AMERICA',      'ARM`D FOR. A',
        'ARM`D FOR. A.',          'ARM`D FOR. AM',
        'ARM`D FOR. AM.',         'ARM`D FOR. AMER',
        'ARM`D FOR. AMER.',       'ARM`D FOR. AMERICA',
        'ARM`D FORCE A',          'ARM`D FORCE A.',
        'ARM`D FORCE AM',         'ARM`D FORCE AM.',
        'ARM`D FORCE AMER',       'ARM`D FORCE AMER.',
        'ARM`D FORCE AMERICA',    'ARM`D FORCE. A',
        'ARM`D FORCE. A.',        'ARM`D FORCE. AM',
        'ARM`D FORCE. AM.',       'ARM`D FORCE. AMER',
        'ARM`D FORCE. AMER.',     'ARM`D FORCE. AMERICA',
        'ARM`D FORCES A',         'ARM`D FORCES A.',
        'ARM`D FORCES AM',        'ARM`D FORCES AM.',
        'ARM`D FORCES AMER',      'ARM`D FORCES AMER.',
        'ARM`D FORCES AMERICA',   'ARM`D. F A',
        'ARM`D. F A.',            'ARM`D. F AM',
        'ARM`D. F AM.',           'ARM`D. F AMER',
        'ARM`D. F AMER.',         'ARM`D. F AMERICA',
        'ARM`D. F. A',            'ARM`D. F. A.',
        'ARM`D. F. AM',           'ARM`D. F. AM.',
        'ARM`D. F. AMER',         'ARM`D. F. AMER.',
        'ARM`D. F. AMERICA',      'ARM`D. FOR A',
        'ARM`D. FOR A.',          'ARM`D. FOR AM',
        'ARM`D. FOR AM.',         'ARM`D. FOR AMER',
        'ARM`D. FOR AMER.',       'ARM`D. FOR AMERICA',
        'ARM`D. FOR. A',          'ARM`D. FOR. A.',
        'ARM`D. FOR. AM',         'ARM`D. FOR. AM.',
        'ARM`D. FOR. AMER',       'ARM`D. FOR. AMER.',
        'ARM`D. FOR. AMERICA',    'ARM`D. FORCE A',
        'ARM`D. FORCE A.',        'ARM`D. FORCE AM',
        'ARM`D. FORCE AM.',       'ARM`D. FORCE AMER',
        'ARM`D. FORCE AMER.',     'ARM`D. FORCE AMERICA',
        'ARM`D. FORCE. A',        'ARM`D. FORCE. A.',
        'ARM`D. FORCE. AM',       'ARM`D. FORCE. AM.',
        'ARM`D. FORCE. AMER',     'ARM`D. FORCE. AMER.',
        'ARM`D. FORCE. AMERICA',  'ARM`D. FORCES A',
        'ARM`D. FORCES A.',       'ARM`D. FORCES AM',
        'ARM`D. FORCES AM.',      'ARM`D. FORCES AMER',
        'ARM`D. FORCES AMER.',    'ARM`D. FORCES AMERICA',
        'ARM`D.F.A',              'ARM`D.F.A.',
        'ARM`D.F.AM',             'ARM`D.F.AM.',
        'ARM`D.F.AMER',           'ARM`D.F.AMER.',
        'ARM`D.F.AMERICA',        'ARM`D.FA',
        'ARM`D.FA.',              'ARM`D.FAM',
        'ARM`D.FAM.',             'ARM`D.FAMER',
        'ARM`D.FAMER.',           'ARM`D.FAMERICA',
        'ARM`D.FOR.A',            'ARM`D.FOR.A.',
        'ARM`D.FOR.AM',           'ARM`D.FOR.AM.',
        'ARM`D.FOR.AMER',         'ARM`D.FOR.AMER.',
        'ARM`D.FOR.AMERICA',      'ARM`D.FORA',
        'ARM`D.FORA.',            'ARM`D.FORAM',
        'ARM`D.FORAM.',           'ARM`D.FORAMER',
        'ARM`D.FORAMER.',         'ARM`D.FORAMERICA',
        'ARM`D.FORCE.A',          'ARM`D.FORCE.A.',
        'ARM`D.FORCE.AM',         'ARM`D.FORCE.AM.',
        'ARM`D.FORCE.AMER',       'ARM`D.FORCE.AMER.',
        'ARM`D.FORCE.AMERICA',    'ARM`D.FORCEA',
        'ARM`D.FORCEA.',          'ARM`D.FORCEAM',
        'ARM`D.FORCEAM.',         'ARM`D.FORCEAMER',
        'ARM`D.FORCEAMER.',       'ARM`D.FORCEAMERICA',
        'ARM`D.FORCESA',          'ARM`D.FORCESA.',
        'ARM`D.FORCESAM',         'ARM`D.FORCESAM.',
        'ARM`D.FORCESAMER',       'ARM`D.FORCESAMER.',
        'ARM`D.FORCESAMERICA',    'ARM`DF.A',
        'ARM`DF.A.',              'ARM`DF.AM',
        'ARM`DF.AM.',             'ARM`DF.AMER',
        'ARM`DF.AMER.',           'ARM`DF.AMERICA',
        'ARM`DFA',                'ARM`DFA.',
        'ARM`DFAM',               'ARM`DFAM.',
        'ARM`DFAMER',             'ARM`DFAMER.',
        'ARM`DFAMERICA',          'ARM`DFOR.A',
        'ARM`DFOR.A.',            'ARM`DFOR.AM',
        'ARM`DFOR.AM.',           'ARM`DFOR.AMER',
        'ARM`DFOR.AMER.',         'ARM`DFOR.AMERICA',
        'ARM`DFORA',              'ARM`DFORA.',
        'ARM`DFORAM',             'ARM`DFORAM.',
        'ARM`DFORAMER',           'ARM`DFORAMER.',
        'ARM`DFORAMERICA',        'ARM`DFORCE.A',
        'ARM`DFORCE.A.',          'ARM`DFORCE.AM',
        'ARM`DFORCE.AM.',         'ARM`DFORCE.AMER',
        'ARM`DFORCE.AMER.',       'ARM`DFORCE.AMERICA',
        'ARM`DFORCEA',            'ARM`DFORCEA.',
        'ARM`DFORCEAM',           'ARM`DFORCEAM.',
        'ARM`DFORCEAMER',         'ARM`DFORCEAMER.',
        'ARM`DFORCEAMERICA',      'ARM`DFORCESA',
        'ARM`DFORCESA.',          'ARM`DFORCESAM',
        'ARM`DFORCESAM.',         'ARM`DFORCESAMER',
        'ARM`DFORCESAMER.',       'ARM`DFORCESAMERICA'
    ],
    'AE' => [
        'A F AF',                   'A F AF.',
        'A F AFR',                  'A F AFR.',
        'A F AFRICA',               'A F CA',
        'A F CA.',                  'A F CAN',
        'A F CAN.',                 'A F CANADA',
        'A F E',                    'A F E.',
        'A F EU',                   'A F EU.',
        'A F EUR',                  'A F EUR.',
        'A F EUROPE',               'A F M E',
        'A F M. E.',                'A F MID EST',
        'A F MID. EST.',            'A F MIDDLE EAST',
        'A F. AF',                  'A F. AF.',
        'A F. AFR',                 'A F. AFR.',
        'A F. AFRICA',              'A F. CA',
        'A F. CA.',                 'A F. CAN',
        'A F. CAN.',                'A F. CANADA',
        'A F. E',                   'A F. E.',
        'A F. EU',                  'A F. EU.',
        'A F. EUR',                 'A F. EUR.',
        'A F. EUROPE',              'A F. M E',
        'A F. M. E.',               'A F. MID EST',
        'A F. MID. EST.',           'A F. MIDDLE EAST',
        'A FOR AF',                 'A FOR AF.',
        'A FOR AFR',                'A FOR AFR.',
        'A FOR AFRICA',             'A FOR CA',
        'A FOR CA.',                'A FOR CAN',
        'A FOR CAN.',               'A FOR CANADA',
        'A FOR E',                  'A FOR E.',
        'A FOR EU',                 'A FOR EU.',
        'A FOR EUR',                'A FOR EUR.',
        'A FOR EUROPE',             'A FOR M E',
        'A FOR M. E.',              'A FOR MID EST',
        'A FOR MID. EST.',          'A FOR MIDDLE EAST',
        'A FOR. AF',                'A FOR. AF.',
        'A FOR. AFR',               'A FOR. AFR.',
        'A FOR. AFRICA',            'A FOR. CA',
        'A FOR. CA.',               'A FOR. CAN',
        'A FOR. CAN.',              'A FOR. CANADA',
        'A FOR. E',                 'A FOR. E.',
        'A FOR. EU',                'A FOR. EU.',
        'A FOR. EUR',               'A FOR. EUR.',
        'A FOR. EUROPE',            'A FOR. M E',
        'A FOR. M. E.',             'A FOR. MID EST',
        'A FOR. MID. EST.',         'A FOR. MIDDLE EAST',
        'A FORCE AF',               'A FORCE AF.',
        'A FORCE AFR',              'A FORCE AFR.',
        'A FORCE AFRICA',           'A FORCE CA',
        'A FORCE CA.',              'A FORCE CAN',
        'A FORCE CAN.',             'A FORCE CANADA',
        'A FORCE E',                'A FORCE E.',
        'A FORCE EU',               'A FORCE EU.',
        'A FORCE EUR',              'A FORCE EUR.',
        'A FORCE EUROPE',           'A FORCE M E',
        'A FORCE M. E.',            'A FORCE MID EST',
        'A FORCE MID. EST.',        'A FORCE MIDDLE EAST',
        'A FORCE. AF',              'A FORCE. AF.',
        'A FORCE. AFR',             'A FORCE. AFR.',
        'A FORCE. AFRICA',          'A FORCE. CA',
        'A FORCE. CA.',             'A FORCE. CAN',
        'A FORCE. CAN.',            'A FORCE. CANADA',
        'A FORCE. E',               'A FORCE. E.',
        'A FORCE. EU',              'A FORCE. EU.',
        'A FORCE. EUR',             'A FORCE. EUR.',
        'A FORCE. EUROPE',          'A FORCE. M E',
        'A FORCE. M. E.',           'A FORCE. MID EST',
        'A FORCE. MID. EST.',       'A FORCE. MIDDLE EAST',
        'A FORCES AF',              'A FORCES AF.',
        'A FORCES AFR',             'A FORCES AFR.',
        'A FORCES AFRICA',          'A FORCES CA',
        'A FORCES CA.',             'A FORCES CAN',
        'A FORCES CAN.',            'A FORCES CANADA',
        'A FORCES E',               'A FORCES E.',
        'A FORCES EU',              'A FORCES EU.',
        'A FORCES EUR',             'A FORCES EUR.',
        'A FORCES EUROPE',          'A FORCES M E',
        'A FORCES M. E.',           'A FORCES MID EST',
        'A FORCES MID. EST.',       'A FORCES MIDDLE EAST',
        'A. F AF',                  'A. F AF.',
        'A. F AFR',                 'A. F AFR.',
        'A. F AFRICA',              'A. F CA',
        'A. F CA.',                 'A. F CAN',
        'A. F CAN.',                'A. F CANADA',
        'A. F E',                   'A. F E.',
        'A. F EU',                  'A. F EU.',
        'A. F EUR',                 'A. F EUR.',
        'A. F EUROPE',              'A. F M E',
        'A. F M. E.',               'A. F MID EST',
        'A. F MID. EST.',           'A. F MIDDLE EAST',
        'A. F. AF',                 'A. F. AF.',
        'A. F. AFR',                'A. F. AFR.',
        'A. F. AFRICA',             'A. F. CA',
        'A. F. CA.',                'A. F. CAN',
        'A. F. CAN.',               'A. F. CANADA',
        'A. F. E',                  'A. F. E.',
        'A. F. EU',                 'A. F. EU.',
        'A. F. EUR',                'A. F. EUR.',
        'A. F. EUROPE',             'A. F. M E',
        'A. F. M. E.',              'A. F. MID EST',
        'A. F. MID. EST.',          'A. F. MIDDLE EAST',
        'A. FOR AF',                'A. FOR AF.',
        'A. FOR AFR',               'A. FOR AFR.',
        'A. FOR AFRICA',            'A. FOR CA',
        'A. FOR CA.',               'A. FOR CAN',
        'A. FOR CAN.',              'A. FOR CANADA',
        'A. FOR E',                 'A. FOR E.',
        'A. FOR EU',                'A. FOR EU.',
        'A. FOR EUR',               'A. FOR EUR.',
        'A. FOR EUROPE',            'A. FOR M E',
        'A. FOR M. E.',             'A. FOR MID EST',
        'A. FOR MID. EST.',         'A. FOR MIDDLE EAST',
        'A. FOR. AF',               'A. FOR. AF.',
        'A. FOR. AFR',              'A. FOR. AFR.',
        'A. FOR. AFRICA',           'A. FOR. CA',
        'A. FOR. CA.',              'A. FOR. CAN',
        'A. FOR. CAN.',             'A. FOR. CANADA',
        'A. FOR. E',                'A. FOR. E.',
        'A. FOR. EU',               'A. FOR. EU.',
        'A. FOR. EUR',              'A. FOR. EUR.',
        'A. FOR. EUROPE',           'A. FOR. M E',
        'A. FOR. M. E.',            'A. FOR. MID EST',
        'A. FOR. MID. EST.',        'A. FOR. MIDDLE EAST',
        'A. FORCE AF',              'A. FORCE AF.',
        'A. FORCE AFR',             'A. FORCE AFR.',
        'A. FORCE AFRICA',          'A. FORCE CA',
        'A. FORCE CA.',             'A. FORCE CAN',
        'A. FORCE CAN.',            'A. FORCE CANADA',
        'A. FORCE E',               'A. FORCE E.',
        'A. FORCE EU',              'A. FORCE EU.',
        'A. FORCE EUR',             'A. FORCE EUR.',
        'A. FORCE EUROPE',          'A. FORCE M E',
        'A. FORCE M. E.',           'A. FORCE MID EST',
        'A. FORCE MID. EST.',       'A. FORCE MIDDLE EAST',
        'A. FORCE. AF',             'A. FORCE. AF.',
        'A. FORCE. AFR',            'A. FORCE. AFR.',
        'A. FORCE. AFRICA',         'A. FORCE. CA',
        'A. FORCE. CA.',            'A. FORCE. CAN',
        'A. FORCE. CAN.',           'A. FORCE. CANADA',
        'A. FORCE. E',              'A. FORCE. E.',
        'A. FORCE. EU',             'A. FORCE. EU.',
        'A. FORCE. EUR',            'A. FORCE. EUR.',
        'A. FORCE. EUROPE',         'A. FORCE. M E',
        'A. FORCE. M. E.',          'A. FORCE. MID EST',
        'A. FORCE. MID. EST.',      'A. FORCE. MIDDLE EAST',
        'A. FORCES AF',             'A. FORCES AF.',
        'A. FORCES AFR',            'A. FORCES AFR.',
        'A. FORCES AFRICA',         'A. FORCES CA',
        'A. FORCES CA.',            'A. FORCES CAN',
        'A. FORCES CAN.',           'A. FORCES CANADA',
        'A. FORCES E',              'A. FORCES E.',
        'A. FORCES EU',             'A. FORCES EU.',
        'A. FORCES EUR',            'A. FORCES EUR.',
        'A. FORCES EUROPE',         'A. FORCES M E',
        'A. FORCES M. E.',          'A. FORCES MID EST',
        'A. FORCES MID. EST.',      'A. FORCES MIDDLE EAST',
        'A.F.AF',                   'A.F.AF.',
        'A.F.AFR',                  'A.F.AFR.',
        'A.F.AFRICA',               'A.F.CA',
        'A.F.CA.',                  'A.F.CAN',
        'A.F.CAN.',                 'A.F.CANADA',
        'A.F.E',                    'A.F.E.',
        'A.F.EU',                   'A.F.EU.',
        'A.F.EUR',                  'A.F.EUR.',
        'A.F.EUROPE',               'A.F.M.E.',
        'A.F.ME',                   'A.F.MID.EST.',
        'A.F.MIDDLEEAST',           'A.F.MIDEST',
        'A.FAF',                    'A.FAF.',
        'A.FAFR',                   'A.FAFR.',
        'A.FAFRICA',                'A.FCA',
        'A.FCA.',                   'A.FCAN',
        'A.FCAN.',                  'A.FCANADA',
        'A.FE',                     'A.FE.',
        'A.FEU',                    'A.FEU.',
        'A.FEUR',                   'A.FEUR.',
        'A.FEUROPE',                'A.FM.E.',
        'A.FME',                    'A.FMID.EST.',
        'A.FMIDDLEEAST',            'A.FMIDEST',
        'A.FOR.AF',                 'A.FOR.AF.',
        'A.FOR.AFR',                'A.FOR.AFR.',
        'A.FOR.AFRICA',             'A.FOR.CA',
        'A.FOR.CA.',                'A.FOR.CAN',
        'A.FOR.CAN.',               'A.FOR.CANADA',
        'A.FOR.E',                  'A.FOR.E.',
        'A.FOR.EU',                 'A.FOR.EU.',
        'A.FOR.EUR',                'A.FOR.EUR.',
        'A.FOR.EUROPE',             'A.FOR.M.E.',
        'A.FOR.ME',                 'A.FOR.MID.EST.',
        'A.FOR.MIDDLEEAST',         'A.FOR.MIDEST',
        'A.FORAF',                  'A.FORAF.',
        'A.FORAFR',                 'A.FORAFR.',
        'A.FORAFRICA',              'A.FORCA',
        'A.FORCA.',                 'A.FORCAN',
        'A.FORCAN.',                'A.FORCANADA',
        'A.FORCE.AF',               'A.FORCE.AF.',
        'A.FORCE.AFR',              'A.FORCE.AFR.',
        'A.FORCE.AFRICA',           'A.FORCE.CA',
        'A.FORCE.CA.',              'A.FORCE.CAN',
        'A.FORCE.CAN.',             'A.FORCE.CANADA',
        'A.FORCE.E',                'A.FORCE.E.',
        'A.FORCE.EU',               'A.FORCE.EU.',
        'A.FORCE.EUR',              'A.FORCE.EUR.',
        'A.FORCE.EUROPE',           'A.FORCE.M.E.',
        'A.FORCE.ME',               'A.FORCE.MID.EST.',
        'A.FORCE.MIDDLEEAST',       'A.FORCE.MIDEST',
        'A.FORCEAF',                'A.FORCEAF.',
        'A.FORCEAFR',               'A.FORCEAFR.',
        'A.FORCEAFRICA',            'A.FORCECA',
        'A.FORCECA.',               'A.FORCECAN',
        'A.FORCECAN.',              'A.FORCECANADA',
        'A.FORCEE',                 'A.FORCEE.',
        'A.FORCEEU',                'A.FORCEEU.',
        'A.FORCEEUR',               'A.FORCEEUR.',
        'A.FORCEEUROPE',            'A.FORCEM.E.',
        'A.FORCEME',                'A.FORCEMID.EST.',
        'A.FORCEMIDDLEEAST',        'A.FORCEMIDEST',
        'A.FORCESAF',               'A.FORCESAF.',
        'A.FORCESAFR',              'A.FORCESAFR.',
        'A.FORCESAFRICA',           'A.FORCESCA',
        'A.FORCESCA.',              'A.FORCESCAN',
        'A.FORCESCAN.',             'A.FORCESCANADA',
        'A.FORCESE',                'A.FORCESE.',
        'A.FORCESEU',               'A.FORCESEU.',
        'A.FORCESEUR',              'A.FORCESEUR.',
        'A.FORCESEUROPE',           'A.FORCESM.E.',
        'A.FORCESME',               'A.FORCESMID.EST.',
        'A.FORCESMIDDLEEAST',       'A.FORCESMIDEST',
        'A.FORE',                   'A.FORE.',
        'A.FOREU',                  'A.FOREU.',
        'A.FOREUR',                 'A.FOREUR.',
        'A.FOREUROPE',              'A.FORM.E.',
        'A.FORME',                  'A.FORMID.EST.',
        'A.FORMIDDLEEAST',          'A.FORMIDEST',
        'AE',                       'AE.',
        'AF.AF',                    'AF.AF.',
        'AF.AFR',                   'AF.AFR.',
        'AF.AFRICA',                'AF.CA',
        'AF.CA.',                   'AF.CAN',
        'AF.CAN.',                  'AF.CANADA',
        'AF.E',                     'AF.E.',
        'AF.EU',                    'AF.EU.',
        'AF.EUR',                   'AF.EUR.',
        'AF.EUROPE',                'AF.M.E.',
        'AF.ME',                    'AF.MID.EST.',
        'AF.MIDDLEEAST',            'AF.MIDEST',
        'AFAF',                     'AFAF.',
        'AFAFR',                    'AFAFR.',
        'AFAFRICA',                 'AFCA',
        'AFCA.',                    'AFCAN',
        'AFCAN.',                   'AFCANADA',
        'AFE',                      'AFE.',
        'AFEU',                     'AFEU.',
        'AFEUR',                    'AFEUR.',
        'AFEUROPE',                 'AFM.E.',
        'AFME',                     'AFMID.EST.',
        'AFMIDDLEEAST',             'AFMIDEST',
        'AFOR.AF',                  'AFOR.AF.',
        'AFOR.AFR',                 'AFOR.AFR.',
        'AFOR.AFRICA',              'AFOR.CA',
        'AFOR.CA.',                 'AFOR.CAN',
        'AFOR.CAN.',                'AFOR.CANADA',
        'AFOR.E',                   'AFOR.E.',
        'AFOR.EU',                  'AFOR.EU.',
        'AFOR.EUR',                 'AFOR.EUR.',
        'AFOR.EUROPE',              'AFOR.M.E.',
        'AFOR.ME',                  'AFOR.MID.EST.',
        'AFOR.MIDDLEEAST',          'AFOR.MIDEST',
        'AFORAF',                   'AFORAF.',
        'AFORAFR',                  'AFORAFR.',
        'AFORAFRICA',               'AFORCA',
        'AFORCA.',                  'AFORCAN',
        'AFORCAN.',                 'AFORCANADA',
        'AFORCE.AF',                'AFORCE.AF.',
        'AFORCE.AFR',               'AFORCE.AFR.',
        'AFORCE.AFRICA',            'AFORCE.CA',
        'AFORCE.CA.',               'AFORCE.CAN',
        'AFORCE.CAN.',              'AFORCE.CANADA',
        'AFORCE.E',                 'AFORCE.E.',
        'AFORCE.EU',                'AFORCE.EU.',
        'AFORCE.EUR',               'AFORCE.EUR.',
        'AFORCE.EUROPE',            'AFORCE.M.E.',
        'AFORCE.ME',                'AFORCE.MID.EST.',
        'AFORCE.MIDDLEEAST',        'AFORCE.MIDEST',
        'AFORCEAF',                 'AFORCEAF.',
        'AFORCEAFR',                'AFORCEAFR.',
        'AFORCEAFRICA',             'AFORCECA',
        'AFORCECA.',                'AFORCECAN',
        'AFORCECAN.',               'AFORCECANADA',
        'AFORCEE',                  'AFORCEE.',
        'AFORCEEU',                 'AFORCEEU.',
        'AFORCEEUR',                'AFORCEEUR.',
        'AFORCEEUROPE',             'AFORCEM.E.',
        'AFORCEME',                 'AFORCEMID.EST.',
        'AFORCEMIDDLEEAST',         'AFORCEMIDEST',
        'AFORCESAF',                'AFORCESAF.',
        'AFORCESAFR',               'AFORCESAFR.',
        'AFORCESAFRICA',            'AFORCESCA',
        'AFORCESCA.',               'AFORCESCAN',
        'AFORCESCAN.',              'AFORCESCANADA',
        'AFORCESE',                 'AFORCESE.',
        'AFORCESEU',                'AFORCESEU.',
        'AFORCESEUR',               'AFORCESEUR.',
        'AFORCESEUROPE',            'AFORCESM.E.',
        'AFORCESME',                'AFORCESMID.EST.',
        'AFORCESMIDDLEEAST',        'AFORCESMIDEST',
        'AFORE',                    'AFORE.',
        'AFOREU',                   'AFOREU.',
        'AFOREUR',                  'AFOREUR.',
        'AFOREUROPE',               'AFORM.E.',
        'AFORME',                   'AFORMID.EST.',
        'AFORMIDDLEEAST',           'AFORMIDEST',
        'ARM F AF',                 'ARM F AF.',
        'ARM F AFR',                'ARM F AFR.',
        'ARM F AFRICA',             'ARM F CA',
        'ARM F CA.',                'ARM F CAN',
        'ARM F CAN.',               'ARM F CANADA',
        'ARM F E',                  'ARM F E.',
        'ARM F EU',                 'ARM F EU.',
        'ARM F EUR',                'ARM F EUR.',
        'ARM F EUROPE',             'ARM F M E',
        'ARM F M. E.',              'ARM F MID EST',
        'ARM F MID. EST.',          'ARM F MIDDLE EAST',
        'ARM F. AF',                'ARM F. AF.',
        'ARM F. AFR',               'ARM F. AFR.',
        'ARM F. AFRICA',            'ARM F. CA',
        'ARM F. CA.',               'ARM F. CAN',
        'ARM F. CAN.',              'ARM F. CANADA',
        'ARM F. E',                 'ARM F. E.',
        'ARM F. EU',                'ARM F. EU.',
        'ARM F. EUR',               'ARM F. EUR.',
        'ARM F. EUROPE',            'ARM F. M E',
        'ARM F. M. E.',             'ARM F. MID EST',
        'ARM F. MID. EST.',         'ARM F. MIDDLE EAST',
        'ARM FOR AF',               'ARM FOR AF.',
        'ARM FOR AFR',              'ARM FOR AFR.',
        'ARM FOR AFRICA',           'ARM FOR CA',
        'ARM FOR CA.',              'ARM FOR CAN',
        'ARM FOR CAN.',             'ARM FOR CANADA',
        'ARM FOR E',                'ARM FOR E.',
        'ARM FOR EU',               'ARM FOR EU.',
        'ARM FOR EUR',              'ARM FOR EUR.',
        'ARM FOR EUROPE',           'ARM FOR M E',
        'ARM FOR M. E.',            'ARM FOR MID EST',
        'ARM FOR MID. EST.',        'ARM FOR MIDDLE EAST',
        'ARM FOR. AF',              'ARM FOR. AF.',
        'ARM FOR. AFR',             'ARM FOR. AFR.',
        'ARM FOR. AFRICA',          'ARM FOR. CA',
        'ARM FOR. CA.',             'ARM FOR. CAN',
        'ARM FOR. CAN.',            'ARM FOR. CANADA',
        'ARM FOR. E',               'ARM FOR. E.',
        'ARM FOR. EU',              'ARM FOR. EU.',
        'ARM FOR. EUR',             'ARM FOR. EUR.',
        'ARM FOR. EUROPE',          'ARM FOR. M E',
        'ARM FOR. M. E.',           'ARM FOR. MID EST',
        'ARM FOR. MID. EST.',       'ARM FOR. MIDDLE EAST',
        'ARM FORCE AF',             'ARM FORCE AF.',
        'ARM FORCE AFR',            'ARM FORCE AFR.',
        'ARM FORCE AFRICA',         'ARM FORCE CA',
        'ARM FORCE CA.',            'ARM FORCE CAN',
        'ARM FORCE CAN.',           'ARM FORCE CANADA',
        'ARM FORCE E',              'ARM FORCE E.',
        'ARM FORCE EU',             'ARM FORCE EU.',
        'ARM FORCE EUR',            'ARM FORCE EUR.',
        'ARM FORCE EUROPE',         'ARM FORCE M E',
        'ARM FORCE M. E.',          'ARM FORCE MID EST',
        'ARM FORCE MID. EST.',      'ARM FORCE MIDDLE EAST',
        'ARM FORCE. AF',            'ARM FORCE. AF.',
        'ARM FORCE. AFR',           'ARM FORCE. AFR.',
        'ARM FORCE. AFRICA',        'ARM FORCE. CA',
        'ARM FORCE. CA.',           'ARM FORCE. CAN',
        'ARM FORCE. CAN.',          'ARM FORCE. CANADA',
        'ARM FORCE. E',             'ARM FORCE. E.',
        'ARM FORCE. EU',            'ARM FORCE. EU.',
        'ARM FORCE. EUR',           'ARM FORCE. EUR.',
        'ARM FORCE. EUROPE',        'ARM FORCE. M E',
        'ARM FORCE. M. E.',         'ARM FORCE. MID EST',
        'ARM FORCE. MID. EST.',     'ARM FORCE. MIDDLE EAST',
        'ARM FORCES AF',            'ARM FORCES AF.',
        'ARM FORCES AFR',           'ARM FORCES AFR.',
        'ARM FORCES AFRICA',        'ARM FORCES CA',
        'ARM FORCES CA.',           'ARM FORCES CAN',
        'ARM FORCES CAN.',          'ARM FORCES CANADA',
        'ARM FORCES E',             'ARM FORCES E.',
        'ARM FORCES EU',            'ARM FORCES EU.',
        'ARM FORCES EUR',           'ARM FORCES EUR.',
        'ARM FORCES EUROPE',        'ARM FORCES M E',
        'ARM FORCES M. E.',         'ARM FORCES MID EST',
        'ARM FORCES MID. EST.',     'ARM FORCES MIDDLE EAST',
        'ARM\'D F AF',              'ARM\'D F AF.',
        'ARM\'D F AFR',             'ARM\'D F AFR.',
        'ARM\'D F AFRICA',          'ARM\'D F CA',
        'ARM\'D F CA.',             'ARM\'D F CAN',
        'ARM\'D F CAN.',            'ARM\'D F CANADA',
        'ARM\'D F E',               'ARM\'D F E.',
        'ARM\'D F EU',              'ARM\'D F EU.',
        'ARM\'D F EUR',             'ARM\'D F EUR.',
        'ARM\'D F EUROPE',          'ARM\'D F M E',
        'ARM\'D F M. E.',           'ARM\'D F MID EST',
        'ARM\'D F MID. EST.',       'ARM\'D F MIDDLE EAST',
        'ARM\'D F. AF',             'ARM\'D F. AF.',
        'ARM\'D F. AFR',            'ARM\'D F. AFR.',
        'ARM\'D F. AFRICA',         'ARM\'D F. CA',
        'ARM\'D F. CA.',            'ARM\'D F. CAN',
        'ARM\'D F. CAN.',           'ARM\'D F. CANADA',
        'ARM\'D F. E',              'ARM\'D F. E.',
        'ARM\'D F. EU',             'ARM\'D F. EU.',
        'ARM\'D F. EUR',            'ARM\'D F. EUR.',
        'ARM\'D F. EUROPE',         'ARM\'D F. M E',
        'ARM\'D F. M. E.',          'ARM\'D F. MID EST',
        'ARM\'D F. MID. EST.',      'ARM\'D F. MIDDLE EAST',
        'ARM\'D FOR AF',            'ARM\'D FOR AF.',
        'ARM\'D FOR AFR',           'ARM\'D FOR AFR.',
        'ARM\'D FOR AFRICA',        'ARM\'D FOR CA',
        'ARM\'D FOR CA.',           'ARM\'D FOR CAN',
        'ARM\'D FOR CAN.',          'ARM\'D FOR CANADA',
        'ARM\'D FOR E',             'ARM\'D FOR E.',
        'ARM\'D FOR EU',            'ARM\'D FOR EU.',
        'ARM\'D FOR EUR',           'ARM\'D FOR EUR.',
        'ARM\'D FOR EUROPE',        'ARM\'D FOR M E',
        'ARM\'D FOR M. E.',         'ARM\'D FOR MID EST',
        'ARM\'D FOR MID. EST.',     'ARM\'D FOR MIDDLE EAST',
        'ARM\'D FOR. AF',           'ARM\'D FOR. AF.',
        'ARM\'D FOR. AFR',          'ARM\'D FOR. AFR.',
        'ARM\'D FOR. AFRICA',       'ARM\'D FOR. CA',
        'ARM\'D FOR. CA.',          'ARM\'D FOR. CAN',
        'ARM\'D FOR. CAN.',         'ARM\'D FOR. CANADA',
        'ARM\'D FOR. E',            'ARM\'D FOR. E.',
        'ARM\'D FOR. EU',           'ARM\'D FOR. EU.',
        'ARM\'D FOR. EUR',          'ARM\'D FOR. EUR.',
        'ARM\'D FOR. EUROPE',       'ARM\'D FOR. M E',
        'ARM\'D FOR. M. E.',        'ARM\'D FOR. MID EST',
        'ARM\'D FOR. MID. EST.',    'ARM\'D FOR. MIDDLE EAST',
        'ARM\'D FORCE AF',          'ARM\'D FORCE AF.',
        'ARM\'D FORCE AFR',         'ARM\'D FORCE AFR.',
        'ARM\'D FORCE AFRICA',      'ARM\'D FORCE CA',
        'ARM\'D FORCE CA.',         'ARM\'D FORCE CAN',
        'ARM\'D FORCE CAN.',        'ARM\'D FORCE CANADA',
        'ARM\'D FORCE E',           'ARM\'D FORCE E.',
        'ARM\'D FORCE EU',          'ARM\'D FORCE EU.',
        'ARM\'D FORCE EUR',         'ARM\'D FORCE EUR.',
        'ARM\'D FORCE EUROPE',      'ARM\'D FORCE M E',
        'ARM\'D FORCE M. E.',       'ARM\'D FORCE MID EST',
        'ARM\'D FORCE MID. EST.',   'ARM\'D FORCE MIDDLE EAST',
        'ARM\'D FORCE. AF',         'ARM\'D FORCE. AF.',
        'ARM\'D FORCE. AFR',        'ARM\'D FORCE. AFR.',
        'ARM\'D FORCE. AFRICA',     'ARM\'D FORCE. CA',
        'ARM\'D FORCE. CA.',        'ARM\'D FORCE. CAN',
        'ARM\'D FORCE. CAN.',       'ARM\'D FORCE. CANADA',
        'ARM\'D FORCE. E',          'ARM\'D FORCE. E.',
        'ARM\'D FORCE. EU',         'ARM\'D FORCE. EU.',
        'ARM\'D FORCE. EUR',        'ARM\'D FORCE. EUR.',
        'ARM\'D FORCE. EUROPE',     'ARM\'D FORCE. M E',
        'ARM\'D FORCE. M. E.',      'ARM\'D FORCE. MID EST',
        'ARM\'D FORCE. MID. EST.',  'ARM\'D FORCE. MIDDLE EAST',
        'ARM\'D FORCES AF',         'ARM\'D FORCES AF.',
        'ARM\'D FORCES AFR',        'ARM\'D FORCES AFR.',
        'ARM\'D FORCES AFRICA',     'ARM\'D FORCES CA',
        'ARM\'D FORCES CA.',        'ARM\'D FORCES CAN',
        'ARM\'D FORCES CAN.',       'ARM\'D FORCES CANADA',
        'ARM\'D FORCES E',          'ARM\'D FORCES E.',
        'ARM\'D FORCES EU',         'ARM\'D FORCES EU.',
        'ARM\'D FORCES EUR',        'ARM\'D FORCES EUR.',
        'ARM\'D FORCES EUROPE',     'ARM\'D FORCES M E',
        'ARM\'D FORCES M. E.',      'ARM\'D FORCES MID EST',
        'ARM\'D FORCES MID. EST.',  'ARM\'D FORCES MIDDLE EAST',
        'ARM\'D. F AF',             'ARM\'D. F AF.',
        'ARM\'D. F AFR',            'ARM\'D. F AFR.',
        'ARM\'D. F AFRICA',         'ARM\'D. F CA',
        'ARM\'D. F CA.',            'ARM\'D. F CAN',
        'ARM\'D. F CAN.',           'ARM\'D. F CANADA',
        'ARM\'D. F E',              'ARM\'D. F E.',
        'ARM\'D. F EU',             'ARM\'D. F EU.',
        'ARM\'D. F EUR',            'ARM\'D. F EUR.',
        'ARM\'D. F EUROPE',         'ARM\'D. F M E',
        'ARM\'D. F M. E.',          'ARM\'D. F MID EST',
        'ARM\'D. F MID. EST.',      'ARM\'D. F MIDDLE EAST',
        'ARM\'D. F. AF',            'ARM\'D. F. AF.',
        'ARM\'D. F. AFR',           'ARM\'D. F. AFR.',
        'ARM\'D. F. AFRICA',        'ARM\'D. F. CA',
        'ARM\'D. F. CA.',           'ARM\'D. F. CAN',
        'ARM\'D. F. CAN.',          'ARM\'D. F. CANADA',
        'ARM\'D. F. E',             'ARM\'D. F. E.',
        'ARM\'D. F. EU',            'ARM\'D. F. EU.',
        'ARM\'D. F. EUR',           'ARM\'D. F. EUR.',
        'ARM\'D. F. EUROPE',        'ARM\'D. F. M E',
        'ARM\'D. F. M. E.',         'ARM\'D. F. MID EST',
        'ARM\'D. F. MID. EST.',     'ARM\'D. F. MIDDLE EAST',
        'ARM\'D. FOR AF',           'ARM\'D. FOR AF.',
        'ARM\'D. FOR AFR',          'ARM\'D. FOR AFR.',
        'ARM\'D. FOR AFRICA',       'ARM\'D. FOR CA',
        'ARM\'D. FOR CA.',          'ARM\'D. FOR CAN',
        'ARM\'D. FOR CAN.',         'ARM\'D. FOR CANADA',
        'ARM\'D. FOR E',            'ARM\'D. FOR E.',
        'ARM\'D. FOR EU',           'ARM\'D. FOR EU.',
        'ARM\'D. FOR EUR',          'ARM\'D. FOR EUR.',
        'ARM\'D. FOR EUROPE',       'ARM\'D. FOR M E',
        'ARM\'D. FOR M. E.',        'ARM\'D. FOR MID EST',
        'ARM\'D. FOR MID. EST.',    'ARM\'D. FOR MIDDLE EAST',
        'ARM\'D. FOR. AF',          'ARM\'D. FOR. AF.',
        'ARM\'D. FOR. AFR',         'ARM\'D. FOR. AFR.',
        'ARM\'D. FOR. AFRICA',      'ARM\'D. FOR. CA',
        'ARM\'D. FOR. CA.',         'ARM\'D. FOR. CAN',
        'ARM\'D. FOR. CAN.',        'ARM\'D. FOR. CANADA',
        'ARM\'D. FOR. E',           'ARM\'D. FOR. E.',
        'ARM\'D. FOR. EU',          'ARM\'D. FOR. EU.',
        'ARM\'D. FOR. EUR',         'ARM\'D. FOR. EUR.',
        'ARM\'D. FOR. EUROPE',      'ARM\'D. FOR. M E',
        'ARM\'D. FOR. M. E.',       'ARM\'D. FOR. MID EST',
        'ARM\'D. FOR. MID. EST.',   'ARM\'D. FOR. MIDDLE EAST',
        'ARM\'D. FORCE AF',         'ARM\'D. FORCE AF.',
        'ARM\'D. FORCE AFR',        'ARM\'D. FORCE AFR.',
        'ARM\'D. FORCE AFRICA',     'ARM\'D. FORCE CA',
        'ARM\'D. FORCE CA.',        'ARM\'D. FORCE CAN',
        'ARM\'D. FORCE CAN.',       'ARM\'D. FORCE CANADA',
        'ARM\'D. FORCE E',          'ARM\'D. FORCE E.',
        'ARM\'D. FORCE EU',         'ARM\'D. FORCE EU.',
        'ARM\'D. FORCE EUR',        'ARM\'D. FORCE EUR.',
        'ARM\'D. FORCE EUROPE',     'ARM\'D. FORCE M E',
        'ARM\'D. FORCE M. E.',      'ARM\'D. FORCE MID EST',
        'ARM\'D. FORCE MID. EST.',  'ARM\'D. FORCE MIDDLE EAST',
        'ARM\'D. FORCE. AF',        'ARM\'D. FORCE. AF.',
        'ARM\'D. FORCE. AFR',       'ARM\'D. FORCE. AFR.',
        'ARM\'D. FORCE. AFRICA',    'ARM\'D. FORCE. CA',
        'ARM\'D. FORCE. CA.',       'ARM\'D. FORCE. CAN',
        'ARM\'D. FORCE. CAN.',      'ARM\'D. FORCE. CANADA',
        'ARM\'D. FORCE. E',         'ARM\'D. FORCE. E.',
        'ARM\'D. FORCE. EU',        'ARM\'D. FORCE. EU.',
        'ARM\'D. FORCE. EUR',       'ARM\'D. FORCE. EUR.',
        'ARM\'D. FORCE. EUROPE',    'ARM\'D. FORCE. M E',
        'ARM\'D. FORCE. M. E.',     'ARM\'D. FORCE. MID EST',
        'ARM\'D. FORCE. MID. EST.', 'ARM\'D. FORCE. MIDDLE EAST',
        'ARM\'D. FORCES AF',        'ARM\'D. FORCES AF.',
        'ARM\'D. FORCES AFR',       'ARM\'D. FORCES AFR.',
        'ARM\'D. FORCES AFRICA',    'ARM\'D. FORCES CA',
        'ARM\'D. FORCES CA.',       'ARM\'D. FORCES CAN',
        'ARM\'D. FORCES CAN.',      'ARM\'D. FORCES CANADA',
        'ARM\'D. FORCES E',         'ARM\'D. FORCES E.',
        'ARM\'D. FORCES EU',        'ARM\'D. FORCES EU.',
        'ARM\'D. FORCES EUR',       'ARM\'D. FORCES EUR.',
        'ARM\'D. FORCES EUROPE',    'ARM\'D. FORCES M E',
        'ARM\'D. FORCES M. E.',     'ARM\'D. FORCES MID EST',
        'ARM\'D. FORCES MID. EST.', 'ARM\'D. FORCES MIDDLE EAST',
        'ARM\'D.F.AF',              'ARM\'D.F.AF.',
        'ARM\'D.F.AFR',             'ARM\'D.F.AFR.',
        'ARM\'D.F.AFRICA',          'ARM\'D.F.CA',
        'ARM\'D.F.CA.',             'ARM\'D.F.CAN',
        'ARM\'D.F.CAN.',            'ARM\'D.F.CANADA',
        'ARM\'D.F.E',               'ARM\'D.F.E.',
        'ARM\'D.F.EU',              'ARM\'D.F.EU.',
        'ARM\'D.F.EUR',             'ARM\'D.F.EUR.',
        'ARM\'D.F.EUROPE',          'ARM\'D.F.M.E.',
        'ARM\'D.F.ME',              'ARM\'D.F.MID.EST.',
        'ARM\'D.F.MIDDLEEAST',      'ARM\'D.F.MIDEST',
        'ARM\'D.FAF',               'ARM\'D.FAF.',
        'ARM\'D.FAFR',              'ARM\'D.FAFR.',
        'ARM\'D.FAFRICA',           'ARM\'D.FCA',
        'ARM\'D.FCA.',              'ARM\'D.FCAN',
        'ARM\'D.FCAN.',             'ARM\'D.FCANADA',
        'ARM\'D.FE',                'ARM\'D.FE.',
        'ARM\'D.FEU',               'ARM\'D.FEU.',
        'ARM\'D.FEUR',              'ARM\'D.FEUR.',
        'ARM\'D.FEUROPE',           'ARM\'D.FM.E.',
        'ARM\'D.FME',               'ARM\'D.FMID.EST.',
        'ARM\'D.FMIDDLEEAST',       'ARM\'D.FMIDEST',
        'ARM\'D.FOR.AF',            'ARM\'D.FOR.AF.',
        'ARM\'D.FOR.AFR',           'ARM\'D.FOR.AFR.',
        'ARM\'D.FOR.AFRICA',        'ARM\'D.FOR.CA',
        'ARM\'D.FOR.CA.',           'ARM\'D.FOR.CAN',
        'ARM\'D.FOR.CAN.',          'ARM\'D.FOR.CANADA',
        'ARM\'D.FOR.E',             'ARM\'D.FOR.E.',
        'ARM\'D.FOR.EU',            'ARM\'D.FOR.EU.',
        'ARM\'D.FOR.EUR',           'ARM\'D.FOR.EUR.',
        'ARM\'D.FOR.EUROPE',        'ARM\'D.FOR.M.E.',
        'ARM\'D.FOR.ME',            'ARM\'D.FOR.MID.EST.',
        'ARM\'D.FOR.MIDDLEEAST',    'ARM\'D.FOR.MIDEST',
        'ARM\'D.FORAF',             'ARM\'D.FORAF.',
        'ARM\'D.FORAFR',            'ARM\'D.FORAFR.',
        'ARM\'D.FORAFRICA',         'ARM\'D.FORCA',
        'ARM\'D.FORCA.',            'ARM\'D.FORCAN',
        'ARM\'D.FORCAN.',           'ARM\'D.FORCANADA',
        'ARM\'D.FORCE.AF',          'ARM\'D.FORCE.AF.',
        'ARM\'D.FORCE.AFR',         'ARM\'D.FORCE.AFR.',
        'ARM\'D.FORCE.AFRICA',      'ARM\'D.FORCE.CA',
        'ARM\'D.FORCE.CA.',         'ARM\'D.FORCE.CAN',
        'ARM\'D.FORCE.CAN.',        'ARM\'D.FORCE.CANADA',
        'ARM\'D.FORCE.E',           'ARM\'D.FORCE.E.',
        'ARM\'D.FORCE.EU',          'ARM\'D.FORCE.EU.',
        'ARM\'D.FORCE.EUR',         'ARM\'D.FORCE.EUR.',
        'ARM\'D.FORCE.EUROPE',      'ARM\'D.FORCE.M.E.',
        'ARM\'D.FORCE.ME',          'ARM\'D.FORCE.MID.EST.',
        'ARM\'D.FORCE.MIDDLEEAST',  'ARM\'D.FORCE.MIDEST',
        'ARM\'D.FORCEAF',           'ARM\'D.FORCEAF.',
        'ARM\'D.FORCEAFR',          'ARM\'D.FORCEAFR.',
        'ARM\'D.FORCEAFRICA',       'ARM\'D.FORCECA',
        'ARM\'D.FORCECA.',          'ARM\'D.FORCECAN',
        'ARM\'D.FORCECAN.',         'ARM\'D.FORCECANADA',
        'ARM\'D.FORCEE',            'ARM\'D.FORCEE.',
        'ARM\'D.FORCEEU',           'ARM\'D.FORCEEU.',
        'ARM\'D.FORCEEUR',          'ARM\'D.FORCEEUR.',
        'ARM\'D.FORCEEUROPE',       'ARM\'D.FORCEM.E.',
        'ARM\'D.FORCEME',           'ARM\'D.FORCEMID.EST.',
        'ARM\'D.FORCEMIDDLEEAST',   'ARM\'D.FORCEMIDEST',
        'ARM\'D.FORCESAF',          'ARM\'D.FORCESAF.',
        'ARM\'D.FORCESAFR',         'ARM\'D.FORCESAFR.',
        'ARM\'D.FORCESAFRICA',      'ARM\'D.FORCESCA',
        'ARM\'D.FORCESCA.',         'ARM\'D.FORCESCAN',
        'ARM\'D.FORCESCAN.',        'ARM\'D.FORCESCANADA',
        'ARM\'D.FORCESE',           'ARM\'D.FORCESE.',
        'ARM\'D.FORCESEU',          'ARM\'D.FORCESEU.',
        'ARM\'D.FORCESEUR',         'ARM\'D.FORCESEUR.',
        'ARM\'D.FORCESEUROPE',      'ARM\'D.FORCESM.E.',
        'ARM\'D.FORCESME',          'ARM\'D.FORCESMID.EST.',
        'ARM\'D.FORCESMIDDLEEAST',  'ARM\'D.FORCESMIDEST',
        'ARM\'D.FORE',              'ARM\'D.FORE.',
        'ARM\'D.FOREU',             'ARM\'D.FOREU.',
        'ARM\'D.FOREUR',            'ARM\'D.FOREUR.',
        'ARM\'D.FOREUROPE',         'ARM\'D.FORM.E.',
        'ARM\'D.FORME',             'ARM\'D.FORMID.EST.',
        'ARM\'D.FORMIDDLEEAST',     'ARM\'D.FORMIDEST',
        'ARM\'DF.AF',               'ARM\'DF.AF.',
        'ARM\'DF.AFR',              'ARM\'DF.AFR.',
        'ARM\'DF.AFRICA',           'ARM\'DF.CA',
        'ARM\'DF.CA.',              'ARM\'DF.CAN',
        'ARM\'DF.CAN.',             'ARM\'DF.CANADA',
        'ARM\'DF.E',                'ARM\'DF.E.',
        'ARM\'DF.EU',               'ARM\'DF.EU.',
        'ARM\'DF.EUR',              'ARM\'DF.EUR.',
        'ARM\'DF.EUROPE',           'ARM\'DF.M.E.',
        'ARM\'DF.ME',               'ARM\'DF.MID.EST.',
        'ARM\'DF.MIDDLEEAST',       'ARM\'DF.MIDEST',
        'ARM\'DFAF',                'ARM\'DFAF.',
        'ARM\'DFAFR',               'ARM\'DFAFR.',
        'ARM\'DFAFRICA',            'ARM\'DFCA',
        'ARM\'DFCA.',               'ARM\'DFCAN',
        'ARM\'DFCAN.',              'ARM\'DFCANADA',
        'ARM\'DFE',                 'ARM\'DFE.',
        'ARM\'DFEU',                'ARM\'DFEU.',
        'ARM\'DFEUR',               'ARM\'DFEUR.',
        'ARM\'DFEUROPE',            'ARM\'DFM.E.',
        'ARM\'DFME',                'ARM\'DFMID.EST.',
        'ARM\'DFMIDDLEEAST',        'ARM\'DFMIDEST',
        'ARM\'DFOR.AF',             'ARM\'DFOR.AF.',
        'ARM\'DFOR.AFR',            'ARM\'DFOR.AFR.',
        'ARM\'DFOR.AFRICA',         'ARM\'DFOR.CA',
        'ARM\'DFOR.CA.',            'ARM\'DFOR.CAN',
        'ARM\'DFOR.CAN.',           'ARM\'DFOR.CANADA',
        'ARM\'DFOR.E',              'ARM\'DFOR.E.',
        'ARM\'DFOR.EU',             'ARM\'DFOR.EU.',
        'ARM\'DFOR.EUR',            'ARM\'DFOR.EUR.',
        'ARM\'DFOR.EUROPE',         'ARM\'DFOR.M.E.',
        'ARM\'DFOR.ME',             'ARM\'DFOR.MID.EST.',
        'ARM\'DFOR.MIDDLEEAST',     'ARM\'DFOR.MIDEST',
        'ARM\'DFORAF',              'ARM\'DFORAF.',
        'ARM\'DFORAFR',             'ARM\'DFORAFR.',
        'ARM\'DFORAFRICA',          'ARM\'DFORCA',
        'ARM\'DFORCA.',             'ARM\'DFORCAN',
        'ARM\'DFORCAN.',            'ARM\'DFORCANADA',
        'ARM\'DFORCE.AF',           'ARM\'DFORCE.AF.',
        'ARM\'DFORCE.AFR',          'ARM\'DFORCE.AFR.',
        'ARM\'DFORCE.AFRICA',       'ARM\'DFORCE.CA',
        'ARM\'DFORCE.CA.',          'ARM\'DFORCE.CAN',
        'ARM\'DFORCE.CAN.',         'ARM\'DFORCE.CANADA',
        'ARM\'DFORCE.E',            'ARM\'DFORCE.E.',
        'ARM\'DFORCE.EU',           'ARM\'DFORCE.EU.',
        'ARM\'DFORCE.EUR',          'ARM\'DFORCE.EUR.',
        'ARM\'DFORCE.EUROPE',       'ARM\'DFORCE.M.E.',
        'ARM\'DFORCE.ME',           'ARM\'DFORCE.MID.EST.',
        'ARM\'DFORCE.MIDDLEEAST',   'ARM\'DFORCE.MIDEST',
        'ARM\'DFORCEAF',            'ARM\'DFORCEAF.',
        'ARM\'DFORCEAFR',           'ARM\'DFORCEAFR.',
        'ARM\'DFORCEAFRICA',        'ARM\'DFORCECA',
        'ARM\'DFORCECA.',           'ARM\'DFORCECAN',
        'ARM\'DFORCECAN.',          'ARM\'DFORCECANADA',
        'ARM\'DFORCEE',             'ARM\'DFORCEE.',
        'ARM\'DFORCEEU',            'ARM\'DFORCEEU.',
        'ARM\'DFORCEEUR',           'ARM\'DFORCEEUR.',
        'ARM\'DFORCEEUROPE',        'ARM\'DFORCEM.E.',
        'ARM\'DFORCEME',            'ARM\'DFORCEMID.EST.',
        'ARM\'DFORCEMIDDLEEAST',    'ARM\'DFORCEMIDEST',
        'ARM\'DFORCESAF',           'ARM\'DFORCESAF.',
        'ARM\'DFORCESAFR',          'ARM\'DFORCESAFR.',
        'ARM\'DFORCESAFRICA',       'ARM\'DFORCESCA',
        'ARM\'DFORCESCA.',          'ARM\'DFORCESCAN',
        'ARM\'DFORCESCAN.',         'ARM\'DFORCESCANADA',
        'ARM\'DFORCESE',            'ARM\'DFORCESE.',
        'ARM\'DFORCESEU',           'ARM\'DFORCESEU.',
        'ARM\'DFORCESEUR',          'ARM\'DFORCESEUR.',
        'ARM\'DFORCESEUROPE',       'ARM\'DFORCESM.E.',
        'ARM\'DFORCESME',           'ARM\'DFORCESMID.EST.',
        'ARM\'DFORCESMIDDLEEAST',   'ARM\'DFORCESMIDEST',
        'ARM\'DFORE',               'ARM\'DFORE.',
        'ARM\'DFOREU',              'ARM\'DFOREU.',
        'ARM\'DFOREUR',             'ARM\'DFOREUR.',
        'ARM\'DFOREUROPE',          'ARM\'DFORM.E.',
        'ARM\'DFORME',              'ARM\'DFORMID.EST.',
        'ARM\'DFORMIDDLEEAST',      'ARM\'DFORMIDEST',
        'ARM. F AF',                'ARM. F AF.',
        'ARM. F AFR',               'ARM. F AFR.',
        'ARM. F AFRICA',            'ARM. F CA',
        'ARM. F CA.',               'ARM. F CAN',
        'ARM. F CAN.',              'ARM. F CANADA',
        'ARM. F E',                 'ARM. F E.',
        'ARM. F EU',                'ARM. F EU.',
        'ARM. F EUR',               'ARM. F EUR.',
        'ARM. F EUROPE',            'ARM. F M E',
        'ARM. F M. E.',             'ARM. F MID EST',
        'ARM. F MID. EST.',         'ARM. F MIDDLE EAST',
        'ARM. F. AF',               'ARM. F. AF.',
        'ARM. F. AFR',              'ARM. F. AFR.',
        'ARM. F. AFRICA',           'ARM. F. CA',
        'ARM. F. CA.',              'ARM. F. CAN',
        'ARM. F. CAN.',             'ARM. F. CANADA',
        'ARM. F. E',                'ARM. F. E.',
        'ARM. F. EU',               'ARM. F. EU.',
        'ARM. F. EUR',              'ARM. F. EUR.',
        'ARM. F. EUROPE',           'ARM. F. M E',
        'ARM. F. M. E.',            'ARM. F. MID EST',
        'ARM. F. MID. EST.',        'ARM. F. MIDDLE EAST',
        'ARM. FOR AF',              'ARM. FOR AF.',
        'ARM. FOR AFR',             'ARM. FOR AFR.',
        'ARM. FOR AFRICA',          'ARM. FOR CA',
        'ARM. FOR CA.',             'ARM. FOR CAN',
        'ARM. FOR CAN.',            'ARM. FOR CANADA',
        'ARM. FOR E',               'ARM. FOR E.',
        'ARM. FOR EU',              'ARM. FOR EU.',
        'ARM. FOR EUR',             'ARM. FOR EUR.',
        'ARM. FOR EUROPE',          'ARM. FOR M E',
        'ARM. FOR M. E.',           'ARM. FOR MID EST',
        'ARM. FOR MID. EST.',       'ARM. FOR MIDDLE EAST',
        'ARM. FOR. AF',             'ARM. FOR. AF.',
        'ARM. FOR. AFR',            'ARM. FOR. AFR.',
        'ARM. FOR. AFRICA',         'ARM. FOR. CA',
        'ARM. FOR. CA.',            'ARM. FOR. CAN',
        'ARM. FOR. CAN.',           'ARM. FOR. CANADA',
        'ARM. FOR. E',              'ARM. FOR. E.',
        'ARM. FOR. EU',             'ARM. FOR. EU.',
        'ARM. FOR. EUR',            'ARM. FOR. EUR.',
        'ARM. FOR. EUROPE',         'ARM. FOR. M E',
        'ARM. FOR. M. E.',          'ARM. FOR. MID EST',
        'ARM. FOR. MID. EST.',      'ARM. FOR. MIDDLE EAST',
        'ARM. FORCE AF',            'ARM. FORCE AF.',
        'ARM. FORCE AFR',           'ARM. FORCE AFR.',
        'ARM. FORCE AFRICA',        'ARM. FORCE CA',
        'ARM. FORCE CA.',           'ARM. FORCE CAN',
        'ARM. FORCE CAN.',          'ARM. FORCE CANADA',
        'ARM. FORCE E',             'ARM. FORCE E.',
        'ARM. FORCE EU',            'ARM. FORCE EU.',
        'ARM. FORCE EUR',           'ARM. FORCE EUR.',
        'ARM. FORCE EUROPE',        'ARM. FORCE M E',
        'ARM. FORCE M. E.',         'ARM. FORCE MID EST',
        'ARM. FORCE MID. EST.',     'ARM. FORCE MIDDLE EAST',
        'ARM. FORCE. AF',           'ARM. FORCE. AF.',
        'ARM. FORCE. AFR',          'ARM. FORCE. AFR.',
        'ARM. FORCE. AFRICA',       'ARM. FORCE. CA',
        'ARM. FORCE. CA.',          'ARM. FORCE. CAN',
        'ARM. FORCE. CAN.',         'ARM. FORCE. CANADA',
        'ARM. FORCE. E',            'ARM. FORCE. E.',
        'ARM. FORCE. EU',           'ARM. FORCE. EU.',
        'ARM. FORCE. EUR',          'ARM. FORCE. EUR.',
        'ARM. FORCE. EUROPE',       'ARM. FORCE. M E',
        'ARM. FORCE. M. E.',        'ARM. FORCE. MID EST',
        'ARM. FORCE. MID. EST.',    'ARM. FORCE. MIDDLE EAST',
        'ARM. FORCES AF',           'ARM. FORCES AF.',
        'ARM. FORCES AFR',          'ARM. FORCES AFR.',
        'ARM. FORCES AFRICA',       'ARM. FORCES CA',
        'ARM. FORCES CA.',          'ARM. FORCES CAN',
        'ARM. FORCES CAN.',         'ARM. FORCES CANADA',
        'ARM. FORCES E',            'ARM. FORCES E.',
        'ARM. FORCES EU',           'ARM. FORCES EU.',
        'ARM. FORCES EUR',          'ARM. FORCES EUR.',
        'ARM. FORCES EUROPE',       'ARM. FORCES M E',
        'ARM. FORCES M. E.',        'ARM. FORCES MID EST',
        'ARM. FORCES MID. EST.',    'ARM. FORCES MIDDLE EAST',
        'ARM.F.AF',                 'ARM.F.AF.',
        'ARM.F.AFR',                'ARM.F.AFR.',
        'ARM.F.AFRICA',             'ARM.F.CA',
        'ARM.F.CA.',                'ARM.F.CAN',
        'ARM.F.CAN.',               'ARM.F.CANADA',
        'ARM.F.E',                  'ARM.F.E.',
        'ARM.F.EU',                 'ARM.F.EU.',
        'ARM.F.EUR',                'ARM.F.EUR.',
        'ARM.F.EUROPE',             'ARM.F.M.E.',
        'ARM.F.ME',                 'ARM.F.MID.EST.',
        'ARM.F.MIDDLEEAST',         'ARM.F.MIDEST',
        'ARM.FAF',                  'ARM.FAF.',
        'ARM.FAFR',                 'ARM.FAFR.',
        'ARM.FAFRICA',              'ARM.FCA',
        'ARM.FCA.',                 'ARM.FCAN',
        'ARM.FCAN.',                'ARM.FCANADA',
        'ARM.FE',                   'ARM.FE.',
        'ARM.FEU',                  'ARM.FEU.',
        'ARM.FEUR',                 'ARM.FEUR.',
        'ARM.FEUROPE',              'ARM.FM.E.',
        'ARM.FME',                  'ARM.FMID.EST.',
        'ARM.FMIDDLEEAST',          'ARM.FMIDEST',
        'ARM.FOR.AF',               'ARM.FOR.AF.',
        'ARM.FOR.AFR',              'ARM.FOR.AFR.',
        'ARM.FOR.AFRICA',           'ARM.FOR.CA',
        'ARM.FOR.CA.',              'ARM.FOR.CAN',
        'ARM.FOR.CAN.',             'ARM.FOR.CANADA',
        'ARM.FOR.E',                'ARM.FOR.E.',
        'ARM.FOR.EU',               'ARM.FOR.EU.',
        'ARM.FOR.EUR',              'ARM.FOR.EUR.',
        'ARM.FOR.EUROPE',           'ARM.FOR.M.E.',
        'ARM.FOR.ME',               'ARM.FOR.MID.EST.',
        'ARM.FOR.MIDDLEEAST',       'ARM.FOR.MIDEST',
        'ARM.FORAF',                'ARM.FORAF.',
        'ARM.FORAFR',               'ARM.FORAFR.',
        'ARM.FORAFRICA',            'ARM.FORCA',
        'ARM.FORCA.',               'ARM.FORCAN',
        'ARM.FORCAN.',              'ARM.FORCANADA',
        'ARM.FORCE.AF',             'ARM.FORCE.AF.',
        'ARM.FORCE.AFR',            'ARM.FORCE.AFR.',
        'ARM.FORCE.AFRICA',         'ARM.FORCE.CA',
        'ARM.FORCE.CA.',            'ARM.FORCE.CAN',
        'ARM.FORCE.CAN.',           'ARM.FORCE.CANADA',
        'ARM.FORCE.E',              'ARM.FORCE.E.',
        'ARM.FORCE.EU',             'ARM.FORCE.EU.',
        'ARM.FORCE.EUR',            'ARM.FORCE.EUR.',
        'ARM.FORCE.EUROPE',         'ARM.FORCE.M.E.',
        'ARM.FORCE.ME',             'ARM.FORCE.MID.EST.',
        'ARM.FORCE.MIDDLEEAST',     'ARM.FORCE.MIDEST',
        'ARM.FORCEAF',              'ARM.FORCEAF.',
        'ARM.FORCEAFR',             'ARM.FORCEAFR.',
        'ARM.FORCEAFRICA',          'ARM.FORCECA',
        'ARM.FORCECA.',             'ARM.FORCECAN',
        'ARM.FORCECAN.',            'ARM.FORCECANADA',
        'ARM.FORCEE',               'ARM.FORCEE.',
        'ARM.FORCEEU',              'ARM.FORCEEU.',
        'ARM.FORCEEUR',             'ARM.FORCEEUR.',
        'ARM.FORCEEUROPE',          'ARM.FORCEM.E.',
        'ARM.FORCEME',              'ARM.FORCEMID.EST.',
        'ARM.FORCEMIDDLEEAST',      'ARM.FORCEMIDEST',
        'ARM.FORCESAF',             'ARM.FORCESAF.',
        'ARM.FORCESAFR',            'ARM.FORCESAFR.',
        'ARM.FORCESAFRICA',         'ARM.FORCESCA',
        'ARM.FORCESCA.',            'ARM.FORCESCAN',
        'ARM.FORCESCAN.',           'ARM.FORCESCANADA',
        'ARM.FORCESE',              'ARM.FORCESE.',
        'ARM.FORCESEU',             'ARM.FORCESEU.',
        'ARM.FORCESEUR',            'ARM.FORCESEUR.',
        'ARM.FORCESEUROPE',         'ARM.FORCESM.E.',
        'ARM.FORCESME',             'ARM.FORCESMID.EST.',
        'ARM.FORCESMIDDLEEAST',     'ARM.FORCESMIDEST',
        'ARM.FORE',                 'ARM.FORE.',
        'ARM.FOREU',                'ARM.FOREU.',
        'ARM.FOREUR',               'ARM.FOREUR.',
        'ARM.FOREUROPE',            'ARM.FORM.E.',
        'ARM.FORME',                'ARM.FORMID.EST.',
        'ARM.FORMIDDLEEAST',        'ARM.FORMIDEST',
        'ARMED F AF',               'ARMED F AF.',
        'ARMED F AFR',              'ARMED F AFR.',
        'ARMED F AFRICA',           'ARMED F CA',
        'ARMED F CA.',              'ARMED F CAN',
        'ARMED F CAN.',             'ARMED F CANADA',
        'ARMED F E',                'ARMED F E.',
        'ARMED F EU',               'ARMED F EU.',
        'ARMED F EUR',              'ARMED F EUR.',
        'ARMED F EUROPE',           'ARMED F M E',
        'ARMED F M. E.',            'ARMED F MID EST',
        'ARMED F MID. EST.',        'ARMED F MIDDLE EAST',
        'ARMED F. AF',              'ARMED F. AF.',
        'ARMED F. AFR',             'ARMED F. AFR.',
        'ARMED F. AFRICA',          'ARMED F. CA',
        'ARMED F. CA.',             'ARMED F. CAN',
        'ARMED F. CAN.',            'ARMED F. CANADA',
        'ARMED F. E',               'ARMED F. E.',
        'ARMED F. EU',              'ARMED F. EU.',
        'ARMED F. EUR',             'ARMED F. EUR.',
        'ARMED F. EUROPE',          'ARMED F. M E',
        'ARMED F. M. E.',           'ARMED F. MID EST',
        'ARMED F. MID. EST.',       'ARMED F. MIDDLE EAST',
        'ARMED FOR AF',             'ARMED FOR AF.',
        'ARMED FOR AFR',            'ARMED FOR AFR.',
        'ARMED FOR AFRICA',         'ARMED FOR CA',
        'ARMED FOR CA.',            'ARMED FOR CAN',
        'ARMED FOR CAN.',           'ARMED FOR CANADA',
        'ARMED FOR E',              'ARMED FOR E.',
        'ARMED FOR EU',             'ARMED FOR EU.',
        'ARMED FOR EUR',            'ARMED FOR EUR.',
        'ARMED FOR EUROPE',         'ARMED FOR M E',
        'ARMED FOR M. E.',          'ARMED FOR MID EST',
        'ARMED FOR MID. EST.',      'ARMED FOR MIDDLE EAST',
        'ARMED FOR. AF',            'ARMED FOR. AF.',
        'ARMED FOR. AFR',           'ARMED FOR. AFR.',
        'ARMED FOR. AFRICA',        'ARMED FOR. CA',
        'ARMED FOR. CA.',           'ARMED FOR. CAN',
        'ARMED FOR. CAN.',          'ARMED FOR. CANADA',
        'ARMED FOR. E',             'ARMED FOR. E.',
        'ARMED FOR. EU',            'ARMED FOR. EU.',
        'ARMED FOR. EUR',           'ARMED FOR. EUR.',
        'ARMED FOR. EUROPE',        'ARMED FOR. M E',
        'ARMED FOR. M. E.',         'ARMED FOR. MID EST',
        'ARMED FOR. MID. EST.',     'ARMED FOR. MIDDLE EAST',
        'ARMED FORCE AF',           'ARMED FORCE AF.',
        'ARMED FORCE AFR',          'ARMED FORCE AFR.',
        'ARMED FORCE AFRICA',       'ARMED FORCE CA',
        'ARMED FORCE CA.',          'ARMED FORCE CAN',
        'ARMED FORCE CAN.',         'ARMED FORCE CANADA',
        'ARMED FORCE E',            'ARMED FORCE E.',
        'ARMED FORCE EU',           'ARMED FORCE EU.',
        'ARMED FORCE EUR',          'ARMED FORCE EUR.',
        'ARMED FORCE EUROPE',       'ARMED FORCE M E',
        'ARMED FORCE M. E.',        'ARMED FORCE MID EST',
        'ARMED FORCE MID. EST.',    'ARMED FORCE MIDDLE EAST',
        'ARMED FORCE. AF',          'ARMED FORCE. AF.',
        'ARMED FORCE. AFR',         'ARMED FORCE. AFR.',
        'ARMED FORCE. AFRICA',      'ARMED FORCE. CA',
        'ARMED FORCE. CA.',         'ARMED FORCE. CAN',
        'ARMED FORCE. CAN.',        'ARMED FORCE. CANADA',
        'ARMED FORCE. E',           'ARMED FORCE. E.',
        'ARMED FORCE. EU',          'ARMED FORCE. EU.',
        'ARMED FORCE. EUR',         'ARMED FORCE. EUR.',
        'ARMED FORCE. EUROPE',      'ARMED FORCE. M E',
        'ARMED FORCE. M. E.',       'ARMED FORCE. MID EST',
        'ARMED FORCE. MID. EST.',   'ARMED FORCE. MIDDLE EAST',
        'ARMED FORCES AF',          'ARMED FORCES AF.',
        'ARMED FORCES AFR',         'ARMED FORCES AFR.',
        'ARMED FORCES AFRICA',      'ARMED FORCES CA',
        'ARMED FORCES CA.',         'ARMED FORCES CAN',
        'ARMED FORCES CAN.',        'ARMED FORCES CANADA',
        'ARMED FORCES E',           'ARMED FORCES E.',
        'ARMED FORCES EU',          'ARMED FORCES EU.',
        'ARMED FORCES EUR',         'ARMED FORCES EUR.',
        'ARMED FORCES EUROPE',      'ARMED FORCES M E',
        'ARMED FORCES M. E.',       'ARMED FORCES MID EST',
        'ARMED FORCES MID. EST.',   'ARMED FORCES MIDDLE EAST',
        'ARMEDF.AF',                'ARMEDF.AF.',
        'ARMEDF.AFR',               'ARMEDF.AFR.',
        'ARMEDF.AFRICA',            'ARMEDF.CA',
        'ARMEDF.CA.',               'ARMEDF.CAN',
        'ARMEDF.CAN.',              'ARMEDF.CANADA',
        'ARMEDF.E',                 'ARMEDF.E.',
        'ARMEDF.EU',                'ARMEDF.EU.',
        'ARMEDF.EUR',               'ARMEDF.EUR.',
        'ARMEDF.EUROPE',            'ARMEDF.M.E.',
        'ARMEDF.ME',                'ARMEDF.MID.EST.',
        'ARMEDF.MIDDLEEAST',        'ARMEDF.MIDEST',
        'ARMEDFAF',                 'ARMEDFAF.',
        'ARMEDFAFR',                'ARMEDFAFR.',
        'ARMEDFAFRICA',             'ARMEDFCA',
        'ARMEDFCA.',                'ARMEDFCAN',
        'ARMEDFCAN.',               'ARMEDFCANADA',
        'ARMEDFE',                  'ARMEDFE.',
        'ARMEDFEU',                 'ARMEDFEU.',
        'ARMEDFEUR',                'ARMEDFEUR.',
        'ARMEDFEUROPE',             'ARMEDFM.E.',
        'ARMEDFME',                 'ARMEDFMID.EST.',
        'ARMEDFMIDDLEEAST',         'ARMEDFMIDEST',
        'ARMEDFOR.AF',              'ARMEDFOR.AF.',
        'ARMEDFOR.AFR',             'ARMEDFOR.AFR.',
        'ARMEDFOR.AFRICA',          'ARMEDFOR.CA',
        'ARMEDFOR.CA.',             'ARMEDFOR.CAN',
        'ARMEDFOR.CAN.',            'ARMEDFOR.CANADA',
        'ARMEDFOR.E',               'ARMEDFOR.E.',
        'ARMEDFOR.EU',              'ARMEDFOR.EU.',
        'ARMEDFOR.EUR',             'ARMEDFOR.EUR.',
        'ARMEDFOR.EUROPE',          'ARMEDFOR.M.E.',
        'ARMEDFOR.ME',              'ARMEDFOR.MID.EST.',
        'ARMEDFOR.MIDDLEEAST',      'ARMEDFOR.MIDEST',
        'ARMEDFORAF',               'ARMEDFORAF.',
        'ARMEDFORAFR',              'ARMEDFORAFR.',
        'ARMEDFORAFRICA',           'ARMEDFORCA',
        'ARMEDFORCA.',              'ARMEDFORCAN',
        'ARMEDFORCAN.',             'ARMEDFORCANADA',
        'ARMEDFORCE.AF',            'ARMEDFORCE.AF.',
        'ARMEDFORCE.AFR',           'ARMEDFORCE.AFR.',
        'ARMEDFORCE.AFRICA',        'ARMEDFORCE.CA',
        'ARMEDFORCE.CA.',           'ARMEDFORCE.CAN',
        'ARMEDFORCE.CAN.',          'ARMEDFORCE.CANADA',
        'ARMEDFORCE.E',             'ARMEDFORCE.E.',
        'ARMEDFORCE.EU',            'ARMEDFORCE.EU.',
        'ARMEDFORCE.EUR',           'ARMEDFORCE.EUR.',
        'ARMEDFORCE.EUROPE',        'ARMEDFORCE.M.E.',
        'ARMEDFORCE.ME',            'ARMEDFORCE.MID.EST.',
        'ARMEDFORCE.MIDDLEEAST',    'ARMEDFORCE.MIDEST',
        'ARMEDFORCEAF',             'ARMEDFORCEAF.',
        'ARMEDFORCEAFR',            'ARMEDFORCEAFR.',
        'ARMEDFORCEAFRICA',         'ARMEDFORCECA',
        'ARMEDFORCECA.',            'ARMEDFORCECAN',
        'ARMEDFORCECAN.',           'ARMEDFORCECANADA',
        'ARMEDFORCEE',              'ARMEDFORCEE.',
        'ARMEDFORCEEU',             'ARMEDFORCEEU.',
        'ARMEDFORCEEUR',            'ARMEDFORCEEUR.',
        'ARMEDFORCEEUROPE',         'ARMEDFORCEM.E.',
        'ARMEDFORCEME',             'ARMEDFORCEMID.EST.',
        'ARMEDFORCEMIDDLEEAST',     'ARMEDFORCEMIDEST',
        'ARMEDFORCESAF',            'ARMEDFORCESAF.',
        'ARMEDFORCESAFR',           'ARMEDFORCESAFR.',
        'ARMEDFORCESAFRICA',        'ARMEDFORCESCA',
        'ARMEDFORCESCA.',           'ARMEDFORCESCAN',
        'ARMEDFORCESCAN.',          'ARMEDFORCESCANADA',
        'ARMEDFORCESE',             'ARMEDFORCESE.',
        'ARMEDFORCESEU',            'ARMEDFORCESEU.',
        'ARMEDFORCESEUR',           'ARMEDFORCESEUR.',
        'ARMEDFORCESEUROPE',        'ARMEDFORCESM.E.',
        'ARMEDFORCESME',            'ARMEDFORCESMID.EST.',
        'ARMEDFORCESMIDDLEEAST',    'ARMEDFORCESMIDEST',
        'ARMEDFORE',                'ARMEDFORE.',
        'ARMEDFOREU',               'ARMEDFOREU.',
        'ARMEDFOREUR',              'ARMEDFOREUR.',
        'ARMEDFOREUROPE',           'ARMEDFORM.E.',
        'ARMEDFORME',               'ARMEDFORMID.EST.',
        'ARMEDFORMIDDLEEAST',       'ARMEDFORMIDEST',
        'ARMF.AF',                  'ARMF.AF.',
        'ARMF.AFR',                 'ARMF.AFR.',
        'ARMF.AFRICA',              'ARMF.CA',
        'ARMF.CA.',                 'ARMF.CAN',
        'ARMF.CAN.',                'ARMF.CANADA',
        'ARMF.E',                   'ARMF.E.',
        'ARMF.EU',                  'ARMF.EU.',
        'ARMF.EUR',                 'ARMF.EUR.',
        'ARMF.EUROPE',              'ARMF.M.E.',
        'ARMF.ME',                  'ARMF.MID.EST.',
        'ARMF.MIDDLEEAST',          'ARMF.MIDEST',
        'ARMFAF',                   'ARMFAF.',
        'ARMFAFR',                  'ARMFAFR.',
        'ARMFAFRICA',               'ARMFCA',
        'ARMFCA.',                  'ARMFCAN',
        'ARMFCAN.',                 'ARMFCANADA',
        'ARMFE',                    'ARMFE.',
        'ARMFEU',                   'ARMFEU.',
        'ARMFEUR',                  'ARMFEUR.',
        'ARMFEUROPE',               'ARMFM.E.',
        'ARMFME',                   'ARMFMID.EST.',
        'ARMFMIDDLEEAST',           'ARMFMIDEST',
        'ARMFOR.AF',                'ARMFOR.AF.',
        'ARMFOR.AFR',               'ARMFOR.AFR.',
        'ARMFOR.AFRICA',            'ARMFOR.CA',
        'ARMFOR.CA.',               'ARMFOR.CAN',
        'ARMFOR.CAN.',              'ARMFOR.CANADA',
        'ARMFOR.E',                 'ARMFOR.E.',
        'ARMFOR.EU',                'ARMFOR.EU.',
        'ARMFOR.EUR',               'ARMFOR.EUR.',
        'ARMFOR.EUROPE',            'ARMFOR.M.E.',
        'ARMFOR.ME',                'ARMFOR.MID.EST.',
        'ARMFOR.MIDDLEEAST',        'ARMFOR.MIDEST',
        'ARMFORAF',                 'ARMFORAF.',
        'ARMFORAFR',                'ARMFORAFR.',
        'ARMFORAFRICA',             'ARMFORCA',
        'ARMFORCA.',                'ARMFORCAN',
        'ARMFORCAN.',               'ARMFORCANADA',
        'ARMFORCE.AF',              'ARMFORCE.AF.',
        'ARMFORCE.AFR',             'ARMFORCE.AFR.',
        'ARMFORCE.AFRICA',          'ARMFORCE.CA',
        'ARMFORCE.CA.',             'ARMFORCE.CAN',
        'ARMFORCE.CAN.',            'ARMFORCE.CANADA',
        'ARMFORCE.E',               'ARMFORCE.E.',
        'ARMFORCE.EU',              'ARMFORCE.EU.',
        'ARMFORCE.EUR',             'ARMFORCE.EUR.',
        'ARMFORCE.EUROPE',          'ARMFORCE.M.E.',
        'ARMFORCE.ME',              'ARMFORCE.MID.EST.',
        'ARMFORCE.MIDDLEEAST',      'ARMFORCE.MIDEST',
        'ARMFORCEAF',               'ARMFORCEAF.',
        'ARMFORCEAFR',              'ARMFORCEAFR.',
        'ARMFORCEAFRICA',           'ARMFORCECA',
        'ARMFORCECA.',              'ARMFORCECAN',
        'ARMFORCECAN.',             'ARMFORCECANADA',
        'ARMFORCEE',                'ARMFORCEE.',
        'ARMFORCEEU',               'ARMFORCEEU.',
        'ARMFORCEEUR',              'ARMFORCEEUR.',
        'ARMFORCEEUROPE',           'ARMFORCEM.E.',
        'ARMFORCEME',               'ARMFORCEMID.EST.',
        'ARMFORCEMIDDLEEAST',       'ARMFORCEMIDEST',
        'ARMFORCESAF',              'ARMFORCESAF.',
        'ARMFORCESAFR',             'ARMFORCESAFR.',
        'ARMFORCESAFRICA',          'ARMFORCESCA',
        'ARMFORCESCA.',             'ARMFORCESCAN',
        'ARMFORCESCAN.',            'ARMFORCESCANADA',
        'ARMFORCESE',               'ARMFORCESE.',
        'ARMFORCESEU',              'ARMFORCESEU.',
        'ARMFORCESEUR',             'ARMFORCESEUR.',
        'ARMFORCESEUROPE',          'ARMFORCESM.E.',
        'ARMFORCESME',              'ARMFORCESMID.EST.',
        'ARMFORCESMIDDLEEAST',      'ARMFORCESMIDEST',
        'ARMFORE',                  'ARMFORE.',
        'ARMFOREU',                 'ARMFOREU.',
        'ARMFOREUR',                'ARMFOREUR.',
        'ARMFOREUROPE',             'ARMFORM.E.',
        'ARMFORME',                 'ARMFORMID.EST.',
        'ARMFORMIDDLEEAST',         'ARMFORMIDEST',
        'ARM`D F AF',               'ARM`D F AF.',
        'ARM`D F AFR',              'ARM`D F AFR.',
        'ARM`D F AFRICA',           'ARM`D F CA',
        'ARM`D F CA.',              'ARM`D F CAN',
        'ARM`D F CAN.',             'ARM`D F CANADA',
        'ARM`D F E',                'ARM`D F E.',
        'ARM`D F EU',               'ARM`D F EU.',
        'ARM`D F EUR',              'ARM`D F EUR.',
        'ARM`D F EUROPE',           'ARM`D F M E',
        'ARM`D F M. E.',            'ARM`D F MID EST',
        'ARM`D F MID. EST.',        'ARM`D F MIDDLE EAST',
        'ARM`D F. AF',              'ARM`D F. AF.',
        'ARM`D F. AFR',             'ARM`D F. AFR.',
        'ARM`D F. AFRICA',          'ARM`D F. CA',
        'ARM`D F. CA.',             'ARM`D F. CAN',
        'ARM`D F. CAN.',            'ARM`D F. CANADA',
        'ARM`D F. E',               'ARM`D F. E.',
        'ARM`D F. EU',              'ARM`D F. EU.',
        'ARM`D F. EUR',             'ARM`D F. EUR.',
        'ARM`D F. EUROPE',          'ARM`D F. M E',
        'ARM`D F. M. E.',           'ARM`D F. MID EST',
        'ARM`D F. MID. EST.',       'ARM`D F. MIDDLE EAST',
        'ARM`D FOR AF',             'ARM`D FOR AF.',
        'ARM`D FOR AFR',            'ARM`D FOR AFR.',
        'ARM`D FOR AFRICA',         'ARM`D FOR CA',
        'ARM`D FOR CA.',            'ARM`D FOR CAN',
        'ARM`D FOR CAN.',           'ARM`D FOR CANADA',
        'ARM`D FOR E',              'ARM`D FOR E.',
        'ARM`D FOR EU',             'ARM`D FOR EU.',
        'ARM`D FOR EUR',            'ARM`D FOR EUR.',
        'ARM`D FOR EUROPE',         'ARM`D FOR M E',
        'ARM`D FOR M. E.',          'ARM`D FOR MID EST',
        'ARM`D FOR MID. EST.',      'ARM`D FOR MIDDLE EAST',
        'ARM`D FOR. AF',            'ARM`D FOR. AF.',
        'ARM`D FOR. AFR',           'ARM`D FOR. AFR.',
        'ARM`D FOR. AFRICA',        'ARM`D FOR. CA',
        'ARM`D FOR. CA.',           'ARM`D FOR. CAN',
        'ARM`D FOR. CAN.',          'ARM`D FOR. CANADA',
        'ARM`D FOR. E',             'ARM`D FOR. E.',
        'ARM`D FOR. EU',            'ARM`D FOR. EU.',
        'ARM`D FOR. EUR',           'ARM`D FOR. EUR.',
        'ARM`D FOR. EUROPE',        'ARM`D FOR. M E',
        'ARM`D FOR. M. E.',         'ARM`D FOR. MID EST',
        'ARM`D FOR. MID. EST.',     'ARM`D FOR. MIDDLE EAST',
        'ARM`D FORCE AF',           'ARM`D FORCE AF.',
        'ARM`D FORCE AFR',          'ARM`D FORCE AFR.',
        'ARM`D FORCE AFRICA',       'ARM`D FORCE CA',
        'ARM`D FORCE CA.',          'ARM`D FORCE CAN',
        'ARM`D FORCE CAN.',         'ARM`D FORCE CANADA',
        'ARM`D FORCE E',            'ARM`D FORCE E.',
        'ARM`D FORCE EU',           'ARM`D FORCE EU.',
        'ARM`D FORCE EUR',          'ARM`D FORCE EUR.',
        'ARM`D FORCE EUROPE',       'ARM`D FORCE M E',
        'ARM`D FORCE M. E.',        'ARM`D FORCE MID EST',
        'ARM`D FORCE MID. EST.',    'ARM`D FORCE MIDDLE EAST',
        'ARM`D FORCE. AF',          'ARM`D FORCE. AF.',
        'ARM`D FORCE. AFR',         'ARM`D FORCE. AFR.',
        'ARM`D FORCE. AFRICA',      'ARM`D FORCE. CA',
        'ARM`D FORCE. CA.',         'ARM`D FORCE. CAN',
        'ARM`D FORCE. CAN.',        'ARM`D FORCE. CANADA',
        'ARM`D FORCE. E',           'ARM`D FORCE. E.',
        'ARM`D FORCE. EU',          'ARM`D FORCE. EU.',
        'ARM`D FORCE. EUR',         'ARM`D FORCE. EUR.',
        'ARM`D FORCE. EUROPE',      'ARM`D FORCE. M E',
        'ARM`D FORCE. M. E.',       'ARM`D FORCE. MID EST',
        'ARM`D FORCE. MID. EST.',   'ARM`D FORCE. MIDDLE EAST',
        'ARM`D FORCES AF',          'ARM`D FORCES AF.',
        'ARM`D FORCES AFR',         'ARM`D FORCES AFR.',
        'ARM`D FORCES AFRICA',      'ARM`D FORCES CA',
        'ARM`D FORCES CA.',         'ARM`D FORCES CAN',
        'ARM`D FORCES CAN.',        'ARM`D FORCES CANADA',
        'ARM`D FORCES E',           'ARM`D FORCES E.',
        'ARM`D FORCES EU',          'ARM`D FORCES EU.',
        'ARM`D FORCES EUR',         'ARM`D FORCES EUR.',
        'ARM`D FORCES EUROPE',      'ARM`D FORCES M E',
        'ARM`D FORCES M. E.',       'ARM`D FORCES MID EST',
        'ARM`D FORCES MID. EST.',   'ARM`D FORCES MIDDLE EAST',
        'ARM`D. F AF',              'ARM`D. F AF.',
        'ARM`D. F AFR',             'ARM`D. F AFR.',
        'ARM`D. F AFRICA',          'ARM`D. F CA',
        'ARM`D. F CA.',             'ARM`D. F CAN',
        'ARM`D. F CAN.',            'ARM`D. F CANADA',
        'ARM`D. F E',               'ARM`D. F E.',
        'ARM`D. F EU',              'ARM`D. F EU.',
        'ARM`D. F EUR',             'ARM`D. F EUR.',
        'ARM`D. F EUROPE',          'ARM`D. F M E',
        'ARM`D. F M. E.',           'ARM`D. F MID EST',
        'ARM`D. F MID. EST.',       'ARM`D. F MIDDLE EAST',
        'ARM`D. F. AF',             'ARM`D. F. AF.',
        'ARM`D. F. AFR',            'ARM`D. F. AFR.',
        'ARM`D. F. AFRICA',         'ARM`D. F. CA',
        'ARM`D. F. CA.',            'ARM`D. F. CAN',
        'ARM`D. F. CAN.',           'ARM`D. F. CANADA',
        'ARM`D. F. E',              'ARM`D. F. E.',
        'ARM`D. F. EU',             'ARM`D. F. EU.',
        'ARM`D. F. EUR',            'ARM`D. F. EUR.',
        'ARM`D. F. EUROPE',         'ARM`D. F. M E',
        'ARM`D. F. M. E.',          'ARM`D. F. MID EST',
        'ARM`D. F. MID. EST.',      'ARM`D. F. MIDDLE EAST',
        'ARM`D. FOR AF',            'ARM`D. FOR AF.',
        'ARM`D. FOR AFR',           'ARM`D. FOR AFR.',
        'ARM`D. FOR AFRICA',        'ARM`D. FOR CA',
        'ARM`D. FOR CA.',           'ARM`D. FOR CAN',
        'ARM`D. FOR CAN.',          'ARM`D. FOR CANADA',
        'ARM`D. FOR E',             'ARM`D. FOR E.',
        'ARM`D. FOR EU',            'ARM`D. FOR EU.',
        'ARM`D. FOR EUR',           'ARM`D. FOR EUR.',
        'ARM`D. FOR EUROPE',        'ARM`D. FOR M E',
        'ARM`D. FOR M. E.',         'ARM`D. FOR MID EST',
        'ARM`D. FOR MID. EST.',     'ARM`D. FOR MIDDLE EAST',
        'ARM`D. FOR. AF',           'ARM`D. FOR. AF.',
        'ARM`D. FOR. AFR',          'ARM`D. FOR. AFR.',
        'ARM`D. FOR. AFRICA',       'ARM`D. FOR. CA',
        'ARM`D. FOR. CA.',          'ARM`D. FOR. CAN',
        'ARM`D. FOR. CAN.',         'ARM`D. FOR. CANADA',
        'ARM`D. FOR. E',            'ARM`D. FOR. E.',
        'ARM`D. FOR. EU',           'ARM`D. FOR. EU.',
        'ARM`D. FOR. EUR',          'ARM`D. FOR. EUR.',
        'ARM`D. FOR. EUROPE',       'ARM`D. FOR. M E',
        'ARM`D. FOR. M. E.',        'ARM`D. FOR. MID EST',
        'ARM`D. FOR. MID. EST.',    'ARM`D. FOR. MIDDLE EAST',
        'ARM`D. FORCE AF',          'ARM`D. FORCE AF.',
        'ARM`D. FORCE AFR',         'ARM`D. FORCE AFR.',
        'ARM`D. FORCE AFRICA',      'ARM`D. FORCE CA',
        'ARM`D. FORCE CA.',         'ARM`D. FORCE CAN',
        'ARM`D. FORCE CAN.',        'ARM`D. FORCE CANADA',
        'ARM`D. FORCE E',           'ARM`D. FORCE E.',
        'ARM`D. FORCE EU',          'ARM`D. FORCE EU.',
        'ARM`D. FORCE EUR',         'ARM`D. FORCE EUR.',
        'ARM`D. FORCE EUROPE',      'ARM`D. FORCE M E',
        'ARM`D. FORCE M. E.',       'ARM`D. FORCE MID EST',
        'ARM`D. FORCE MID. EST.',   'ARM`D. FORCE MIDDLE EAST',
        'ARM`D. FORCE. AF',         'ARM`D. FORCE. AF.',
        'ARM`D. FORCE. AFR',        'ARM`D. FORCE. AFR.',
        'ARM`D. FORCE. AFRICA',     'ARM`D. FORCE. CA',
        'ARM`D. FORCE. CA.',        'ARM`D. FORCE. CAN',
        'ARM`D. FORCE. CAN.',       'ARM`D. FORCE. CANADA',
        'ARM`D. FORCE. E',          'ARM`D. FORCE. E.',
        'ARM`D. FORCE. EU',         'ARM`D. FORCE. EU.',
        'ARM`D. FORCE. EUR',        'ARM`D. FORCE. EUR.',
        'ARM`D. FORCE. EUROPE',     'ARM`D. FORCE. M E',
        'ARM`D. FORCE. M. E.',      'ARM`D. FORCE. MID EST',
        'ARM`D. FORCE. MID. EST.',  'ARM`D. FORCE. MIDDLE EAST',
        'ARM`D. FORCES AF',         'ARM`D. FORCES AF.',
        'ARM`D. FORCES AFR',        'ARM`D. FORCES AFR.',
        'ARM`D. FORCES AFRICA',     'ARM`D. FORCES CA',
        'ARM`D. FORCES CA.',        'ARM`D. FORCES CAN',
        'ARM`D. FORCES CAN.',       'ARM`D. FORCES CANADA',
        'ARM`D. FORCES E',          'ARM`D. FORCES E.',
        'ARM`D. FORCES EU',         'ARM`D. FORCES EU.',
        'ARM`D. FORCES EUR',        'ARM`D. FORCES EUR.',
        'ARM`D. FORCES EUROPE',     'ARM`D. FORCES M E',
        'ARM`D. FORCES M. E.',      'ARM`D. FORCES MID EST',
        'ARM`D. FORCES MID. EST.',  'ARM`D. FORCES MIDDLE EAST',
        'ARM`D.F.AF',               'ARM`D.F.AF.',
        'ARM`D.F.AFR',              'ARM`D.F.AFR.',
        'ARM`D.F.AFRICA',           'ARM`D.F.CA',
        'ARM`D.F.CA.',              'ARM`D.F.CAN',
        'ARM`D.F.CAN.',             'ARM`D.F.CANADA',
        'ARM`D.F.E',                'ARM`D.F.E.',
        'ARM`D.F.EU',               'ARM`D.F.EU.',
        'ARM`D.F.EUR',              'ARM`D.F.EUR.',
        'ARM`D.F.EUROPE',           'ARM`D.F.M.E.',
        'ARM`D.F.ME',               'ARM`D.F.MID.EST.',
        'ARM`D.F.MIDDLEEAST',       'ARM`D.F.MIDEST',
        'ARM`D.FAF',                'ARM`D.FAF.',
        'ARM`D.FAFR',               'ARM`D.FAFR.',
        'ARM`D.FAFRICA',            'ARM`D.FCA',
        'ARM`D.FCA.',               'ARM`D.FCAN',
        'ARM`D.FCAN.',              'ARM`D.FCANADA',
        'ARM`D.FE',                 'ARM`D.FE.',
        'ARM`D.FEU',                'ARM`D.FEU.',
        'ARM`D.FEUR',               'ARM`D.FEUR.',
        'ARM`D.FEUROPE',            'ARM`D.FM.E.',
        'ARM`D.FME',                'ARM`D.FMID.EST.',
        'ARM`D.FMIDDLEEAST',        'ARM`D.FMIDEST',
        'ARM`D.FOR.AF',             'ARM`D.FOR.AF.',
        'ARM`D.FOR.AFR',            'ARM`D.FOR.AFR.',
        'ARM`D.FOR.AFRICA',         'ARM`D.FOR.CA',
        'ARM`D.FOR.CA.',            'ARM`D.FOR.CAN',
        'ARM`D.FOR.CAN.',           'ARM`D.FOR.CANADA',
        'ARM`D.FOR.E',              'ARM`D.FOR.E.',
        'ARM`D.FOR.EU',             'ARM`D.FOR.EU.',
        'ARM`D.FOR.EUR',            'ARM`D.FOR.EUR.',
        'ARM`D.FOR.EUROPE',         'ARM`D.FOR.M.E.',
        'ARM`D.FOR.ME',             'ARM`D.FOR.MID.EST.',
        'ARM`D.FOR.MIDDLEEAST',     'ARM`D.FOR.MIDEST',
        'ARM`D.FORAF',              'ARM`D.FORAF.',
        'ARM`D.FORAFR',             'ARM`D.FORAFR.',
        'ARM`D.FORAFRICA',          'ARM`D.FORCA',
        'ARM`D.FORCA.',             'ARM`D.FORCAN',
        'ARM`D.FORCAN.',            'ARM`D.FORCANADA',
        'ARM`D.FORCE.AF',           'ARM`D.FORCE.AF.',
        'ARM`D.FORCE.AFR',          'ARM`D.FORCE.AFR.',
        'ARM`D.FORCE.AFRICA',       'ARM`D.FORCE.CA',
        'ARM`D.FORCE.CA.',          'ARM`D.FORCE.CAN',
        'ARM`D.FORCE.CAN.',         'ARM`D.FORCE.CANADA',
        'ARM`D.FORCE.E',            'ARM`D.FORCE.E.',
        'ARM`D.FORCE.EU',           'ARM`D.FORCE.EU.',
        'ARM`D.FORCE.EUR',          'ARM`D.FORCE.EUR.',
        'ARM`D.FORCE.EUROPE',       'ARM`D.FORCE.M.E.',
        'ARM`D.FORCE.ME',           'ARM`D.FORCE.MID.EST.',
        'ARM`D.FORCE.MIDDLEEAST',   'ARM`D.FORCE.MIDEST',
        'ARM`D.FORCEAF',            'ARM`D.FORCEAF.',
        'ARM`D.FORCEAFR',           'ARM`D.FORCEAFR.',
        'ARM`D.FORCEAFRICA',        'ARM`D.FORCECA',
        'ARM`D.FORCECA.',           'ARM`D.FORCECAN',
        'ARM`D.FORCECAN.',          'ARM`D.FORCECANADA',
        'ARM`D.FORCEE',             'ARM`D.FORCEE.',
        'ARM`D.FORCEEU',            'ARM`D.FORCEEU.',
        'ARM`D.FORCEEUR',           'ARM`D.FORCEEUR.',
        'ARM`D.FORCEEUROPE',        'ARM`D.FORCEM.E.',
        'ARM`D.FORCEME',            'ARM`D.FORCEMID.EST.',
        'ARM`D.FORCEMIDDLEEAST',    'ARM`D.FORCEMIDEST',
        'ARM`D.FORCESAF',           'ARM`D.FORCESAF.',
        'ARM`D.FORCESAFR',          'ARM`D.FORCESAFR.',
        'ARM`D.FORCESAFRICA',       'ARM`D.FORCESCA',
        'ARM`D.FORCESCA.',          'ARM`D.FORCESCAN',
        'ARM`D.FORCESCAN.',         'ARM`D.FORCESCANADA',
        'ARM`D.FORCESE',            'ARM`D.FORCESE.',
        'ARM`D.FORCESEU',           'ARM`D.FORCESEU.',
        'ARM`D.FORCESEUR',          'ARM`D.FORCESEUR.',
        'ARM`D.FORCESEUROPE',       'ARM`D.FORCESM.E.',
        'ARM`D.FORCESME',           'ARM`D.FORCESMID.EST.',
        'ARM`D.FORCESMIDDLEEAST',   'ARM`D.FORCESMIDEST',
        'ARM`D.FORE',               'ARM`D.FORE.',
        'ARM`D.FOREU',              'ARM`D.FOREU.',
        'ARM`D.FOREUR',             'ARM`D.FOREUR.',
        'ARM`D.FOREUROPE',          'ARM`D.FORM.E.',
        'ARM`D.FORME',              'ARM`D.FORMID.EST.',
        'ARM`D.FORMIDDLEEAST',      'ARM`D.FORMIDEST',
        'ARM`DF.AF',                'ARM`DF.AF.',
        'ARM`DF.AFR',               'ARM`DF.AFR.',
        'ARM`DF.AFRICA',            'ARM`DF.CA',
        'ARM`DF.CA.',               'ARM`DF.CAN',
        'ARM`DF.CAN.',              'ARM`DF.CANADA',
        'ARM`DF.E',                 'ARM`DF.E.',
        'ARM`DF.EU',                'ARM`DF.EU.',
        'ARM`DF.EUR',               'ARM`DF.EUR.',
        'ARM`DF.EUROPE',            'ARM`DF.M.E.',
        'ARM`DF.ME',                'ARM`DF.MID.EST.',
        'ARM`DF.MIDDLEEAST',        'ARM`DF.MIDEST',
        'ARM`DFAF',                 'ARM`DFAF.',
        'ARM`DFAFR',                'ARM`DFAFR.',
        'ARM`DFAFRICA',             'ARM`DFCA',
        'ARM`DFCA.',                'ARM`DFCAN',
        'ARM`DFCAN.',               'ARM`DFCANADA',
        'ARM`DFE',                  'ARM`DFE.',
        'ARM`DFEU',                 'ARM`DFEU.',
        'ARM`DFEUR',                'ARM`DFEUR.',
        'ARM`DFEUROPE',             'ARM`DFM.E.',
        'ARM`DFME',                 'ARM`DFMID.EST.',
        'ARM`DFMIDDLEEAST',         'ARM`DFMIDEST',
        'ARM`DFOR.AF',              'ARM`DFOR.AF.',
        'ARM`DFOR.AFR',             'ARM`DFOR.AFR.',
        'ARM`DFOR.AFRICA',          'ARM`DFOR.CA',
        'ARM`DFOR.CA.',             'ARM`DFOR.CAN',
        'ARM`DFOR.CAN.',            'ARM`DFOR.CANADA',
        'ARM`DFOR.E',               'ARM`DFOR.E.',
        'ARM`DFOR.EU',              'ARM`DFOR.EU.',
        'ARM`DFOR.EUR',             'ARM`DFOR.EUR.',
        'ARM`DFOR.EUROPE',          'ARM`DFOR.M.E.',
        'ARM`DFOR.ME',              'ARM`DFOR.MID.EST.',
        'ARM`DFOR.MIDDLEEAST',      'ARM`DFOR.MIDEST',
        'ARM`DFORAF',               'ARM`DFORAF.',
        'ARM`DFORAFR',              'ARM`DFORAFR.',
        'ARM`DFORAFRICA',           'ARM`DFORCA',
        'ARM`DFORCA.',              'ARM`DFORCAN',
        'ARM`DFORCAN.',             'ARM`DFORCANADA',
        'ARM`DFORCE.AF',            'ARM`DFORCE.AF.',
        'ARM`DFORCE.AFR',           'ARM`DFORCE.AFR.',
        'ARM`DFORCE.AFRICA',        'ARM`DFORCE.CA',
        'ARM`DFORCE.CA.',           'ARM`DFORCE.CAN',
        'ARM`DFORCE.CAN.',          'ARM`DFORCE.CANADA',
        'ARM`DFORCE.E',             'ARM`DFORCE.E.',
        'ARM`DFORCE.EU',            'ARM`DFORCE.EU.',
        'ARM`DFORCE.EUR',           'ARM`DFORCE.EUR.',
        'ARM`DFORCE.EUROPE',        'ARM`DFORCE.M.E.',
        'ARM`DFORCE.ME',            'ARM`DFORCE.MID.EST.',
        'ARM`DFORCE.MIDDLEEAST',    'ARM`DFORCE.MIDEST',
        'ARM`DFORCEAF',             'ARM`DFORCEAF.',
        'ARM`DFORCEAFR',            'ARM`DFORCEAFR.',
        'ARM`DFORCEAFRICA',         'ARM`DFORCECA',
        'ARM`DFORCECA.',            'ARM`DFORCECAN',
        'ARM`DFORCECAN.',           'ARM`DFORCECANADA',
        'ARM`DFORCEE',              'ARM`DFORCEE.',
        'ARM`DFORCEEU',             'ARM`DFORCEEU.',
        'ARM`DFORCEEUR',            'ARM`DFORCEEUR.',
        'ARM`DFORCEEUROPE',         'ARM`DFORCEM.E.',
        'ARM`DFORCEME',             'ARM`DFORCEMID.EST.',
        'ARM`DFORCEMIDDLEEAST',     'ARM`DFORCEMIDEST',
        'ARM`DFORCESAF',            'ARM`DFORCESAF.',
        'ARM`DFORCESAFR',           'ARM`DFORCESAFR.',
        'ARM`DFORCESAFRICA',        'ARM`DFORCESCA',
        'ARM`DFORCESCA.',           'ARM`DFORCESCAN',
        'ARM`DFORCESCAN.',          'ARM`DFORCESCANADA',
        'ARM`DFORCESE',             'ARM`DFORCESE.',
        'ARM`DFORCESEU',            'ARM`DFORCESEU.',
        'ARM`DFORCESEUR',           'ARM`DFORCESEUR.',
        'ARM`DFORCESEUROPE',        'ARM`DFORCESM.E.',
        'ARM`DFORCESME',            'ARM`DFORCESMID.EST.',
        'ARM`DFORCESMIDDLEEAST',    'ARM`DFORCESMIDEST',
        'ARM`DFORE',                'ARM`DFORE.',
        'ARM`DFOREU',               'ARM`DFOREU.',
        'ARM`DFOREUR',              'ARM`DFOREUR.',
        'ARM`DFOREUROPE',           'ARM`DFORM.E.',
        'ARM`DFORME',               'ARM`DFORMID.EST.',
        'ARM`DFORMIDDLEEAST',       'ARM`DFORMIDEST'
    ],
    'AK' => [ 'AK', 'AK.', 'ALASKA' ],
    'AL' => [ 'AL', 'AL.', 'ALA', 'ALA.', 'ALABAMA' ],
    'AP' => [
        'A F P',                  'A F P.',
        'A F PAC',                'A F PAC.',
        'A F PACIFIC',            'A F. P',
        'A F. P.',                'A F. PAC',
        'A F. PAC.',              'A F. PACIFIC',
        'A FOR P',                'A FOR P.',
        'A FOR PAC',              'A FOR PAC.',
        'A FOR PACIFIC',          'A FOR. P',
        'A FOR. P.',              'A FOR. PAC',
        'A FOR. PAC.',            'A FOR. PACIFIC',
        'A FORCE P',              'A FORCE P.',
        'A FORCE PAC',            'A FORCE PAC.',
        'A FORCE PACIFIC',        'A FORCE. P',
        'A FORCE. P.',            'A FORCE. PAC',
        'A FORCE. PAC.',          'A FORCE. PACIFIC',
        'A FORCES P',             'A FORCES P.',
        'A FORCES PAC',           'A FORCES PAC.',
        'A FORCES PACIFIC',       'A. F P',
        'A. F P.',                'A. F PAC',
        'A. F PAC.',              'A. F PACIFIC',
        'A. F. P',                'A. F. P.',
        'A. F. PAC',              'A. F. PAC.',
        'A. F. PACIFIC',          'A. FOR P',
        'A. FOR P.',              'A. FOR PAC',
        'A. FOR PAC.',            'A. FOR PACIFIC',
        'A. FOR. P',              'A. FOR. P.',
        'A. FOR. PAC',            'A. FOR. PAC.',
        'A. FOR. PACIFIC',        'A. FORCE P',
        'A. FORCE P.',            'A. FORCE PAC',
        'A. FORCE PAC.',          'A. FORCE PACIFIC',
        'A. FORCE. P',            'A. FORCE. P.',
        'A. FORCE. PAC',          'A. FORCE. PAC.',
        'A. FORCE. PACIFIC',      'A. FORCES P',
        'A. FORCES P.',           'A. FORCES PAC',
        'A. FORCES PAC.',         'A. FORCES PACIFIC',
        'A.F.P',                  'A.F.P.',
        'A.F.PAC',                'A.F.PAC.',
        'A.F.PACIFIC',            'A.FOR.P',
        'A.FOR.P.',               'A.FOR.PAC',
        'A.FOR.PAC.',             'A.FOR.PACIFIC',
        'A.FORCE.P',              'A.FORCE.P.',
        'A.FORCE.PAC',            'A.FORCE.PAC.',
        'A.FORCE.PACIFIC',        'A.FORCEP',
        'A.FORCEP.',              'A.FORCEPAC',
        'A.FORCEPAC.',            'A.FORCEPACIFIC',
        'A.FORCESP',              'A.FORCESP.',
        'A.FORCESPAC',            'A.FORCESPAC.',
        'A.FORCESPACIFIC',        'A.FORP',
        'A.FORP.',                'A.FORPAC',
        'A.FORPAC.',              'A.FORPACIFIC',
        'A.FP',                   'A.FP.',
        'A.FPAC',                 'A.FPAC.',
        'A.FPACIFIC',             'AF.P',
        'AF.P.',                  'AF.PAC',
        'AF.PAC.',                'AF.PACIFIC',
        'AFOR.P',                 'AFOR.P.',
        'AFOR.PAC',               'AFOR.PAC.',
        'AFOR.PACIFIC',           'AFORCE.P',
        'AFORCE.P.',              'AFORCE.PAC',
        'AFORCE.PAC.',            'AFORCE.PACIFIC',
        'AFORCEP',                'AFORCEP.',
        'AFORCEPAC',              'AFORCEPAC.',
        'AFORCEPACIFIC',          'AFORCESP',
        'AFORCESP.',              'AFORCESPAC',
        'AFORCESPAC.',            'AFORCESPACIFIC',
        'AFORP',                  'AFORP.',
        'AFORPAC',                'AFORPAC.',
        'AFORPACIFIC',            'AFP',
        'AFP.',                   'AFPAC',
        'AFPAC.',                 'AFPACIFIC',
        'AP',                     'AP.',
        'ARM F P',                'ARM F P.',
        'ARM F PAC',              'ARM F PAC.',
        'ARM F PACIFIC',          'ARM F. P',
        'ARM F. P.',              'ARM F. PAC',
        'ARM F. PAC.',            'ARM F. PACIFIC',
        'ARM FOR P',              'ARM FOR P.',
        'ARM FOR PAC',            'ARM FOR PAC.',
        'ARM FOR PACIFIC',        'ARM FOR. P',
        'ARM FOR. P.',            'ARM FOR. PAC',
        'ARM FOR. PAC.',          'ARM FOR. PACIFIC',
        'ARM FORCE P',            'ARM FORCE P.',
        'ARM FORCE PAC',          'ARM FORCE PAC.',
        'ARM FORCE PACIFIC',      'ARM FORCE. P',
        'ARM FORCE. P.',          'ARM FORCE. PAC',
        'ARM FORCE. PAC.',        'ARM FORCE. PACIFIC',
        'ARM FORCES P',           'ARM FORCES P.',
        'ARM FORCES PAC',         'ARM FORCES PAC.',
        'ARM FORCES PACIFIC',     'ARM\'D F P',
        'ARM\'D F P.',            'ARM\'D F PAC',
        'ARM\'D F PAC.',          'ARM\'D F PACIFIC',
        'ARM\'D F. P',            'ARM\'D F. P.',
        'ARM\'D F. PAC',          'ARM\'D F. PAC.',
        'ARM\'D F. PACIFIC',      'ARM\'D FOR P',
        'ARM\'D FOR P.',          'ARM\'D FOR PAC',
        'ARM\'D FOR PAC.',        'ARM\'D FOR PACIFIC',
        'ARM\'D FOR. P',          'ARM\'D FOR. P.',
        'ARM\'D FOR. PAC',        'ARM\'D FOR. PAC.',
        'ARM\'D FOR. PACIFIC',    'ARM\'D FORCE P',
        'ARM\'D FORCE P.',        'ARM\'D FORCE PAC',
        'ARM\'D FORCE PAC.',      'ARM\'D FORCE PACIFIC',
        'ARM\'D FORCE. P',        'ARM\'D FORCE. P.',
        'ARM\'D FORCE. PAC',      'ARM\'D FORCE. PAC.',
        'ARM\'D FORCE. PACIFIC',  'ARM\'D FORCES P',
        'ARM\'D FORCES P.',       'ARM\'D FORCES PAC',
        'ARM\'D FORCES PAC.',     'ARM\'D FORCES PACIFIC',
        'ARM\'D. F P',            'ARM\'D. F P.',
        'ARM\'D. F PAC',          'ARM\'D. F PAC.',
        'ARM\'D. F PACIFIC',      'ARM\'D. F. P',
        'ARM\'D. F. P.',          'ARM\'D. F. PAC',
        'ARM\'D. F. PAC.',        'ARM\'D. F. PACIFIC',
        'ARM\'D. FOR P',          'ARM\'D. FOR P.',
        'ARM\'D. FOR PAC',        'ARM\'D. FOR PAC.',
        'ARM\'D. FOR PACIFIC',    'ARM\'D. FOR. P',
        'ARM\'D. FOR. P.',        'ARM\'D. FOR. PAC',
        'ARM\'D. FOR. PAC.',      'ARM\'D. FOR. PACIFIC',
        'ARM\'D. FORCE P',        'ARM\'D. FORCE P.',
        'ARM\'D. FORCE PAC',      'ARM\'D. FORCE PAC.',
        'ARM\'D. FORCE PACIFIC',  'ARM\'D. FORCE. P',
        'ARM\'D. FORCE. P.',      'ARM\'D. FORCE. PAC',
        'ARM\'D. FORCE. PAC.',    'ARM\'D. FORCE. PACIFIC',
        'ARM\'D. FORCES P',       'ARM\'D. FORCES P.',
        'ARM\'D. FORCES PAC',     'ARM\'D. FORCES PAC.',
        'ARM\'D. FORCES PACIFIC', 'ARM\'D.F.P',
        'ARM\'D.F.P.',            'ARM\'D.F.PAC',
        'ARM\'D.F.PAC.',          'ARM\'D.F.PACIFIC',
        'ARM\'D.FOR.P',           'ARM\'D.FOR.P.',
        'ARM\'D.FOR.PAC',         'ARM\'D.FOR.PAC.',
        'ARM\'D.FOR.PACIFIC',     'ARM\'D.FORCE.P',
        'ARM\'D.FORCE.P.',        'ARM\'D.FORCE.PAC',
        'ARM\'D.FORCE.PAC.',      'ARM\'D.FORCE.PACIFIC',
        'ARM\'D.FORCEP',          'ARM\'D.FORCEP.',
        'ARM\'D.FORCEPAC',        'ARM\'D.FORCEPAC.',
        'ARM\'D.FORCEPACIFIC',    'ARM\'D.FORCESP',
        'ARM\'D.FORCESP.',        'ARM\'D.FORCESPAC',
        'ARM\'D.FORCESPAC.',      'ARM\'D.FORCESPACIFIC',
        'ARM\'D.FORP',            'ARM\'D.FORP.',
        'ARM\'D.FORPAC',          'ARM\'D.FORPAC.',
        'ARM\'D.FORPACIFIC',      'ARM\'D.FP',
        'ARM\'D.FP.',             'ARM\'D.FPAC',
        'ARM\'D.FPAC.',           'ARM\'D.FPACIFIC',
        'ARM\'DF.P',              'ARM\'DF.P.',
        'ARM\'DF.PAC',            'ARM\'DF.PAC.',
        'ARM\'DF.PACIFIC',        'ARM\'DFOR.P',
        'ARM\'DFOR.P.',           'ARM\'DFOR.PAC',
        'ARM\'DFOR.PAC.',         'ARM\'DFOR.PACIFIC',
        'ARM\'DFORCE.P',          'ARM\'DFORCE.P.',
        'ARM\'DFORCE.PAC',        'ARM\'DFORCE.PAC.',
        'ARM\'DFORCE.PACIFIC',    'ARM\'DFORCEP',
        'ARM\'DFORCEP.',          'ARM\'DFORCEPAC',
        'ARM\'DFORCEPAC.',        'ARM\'DFORCEPACIFIC',
        'ARM\'DFORCESP',          'ARM\'DFORCESP.',
        'ARM\'DFORCESPAC',        'ARM\'DFORCESPAC.',
        'ARM\'DFORCESPACIFIC',    'ARM\'DFORP',
        'ARM\'DFORP.',            'ARM\'DFORPAC',
        'ARM\'DFORPAC.',          'ARM\'DFORPACIFIC',
        'ARM\'DFP',               'ARM\'DFP.',
        'ARM\'DFPAC',             'ARM\'DFPAC.',
        'ARM\'DFPACIFIC',         'ARM. F P',
        'ARM. F P.',              'ARM. F PAC',
        'ARM. F PAC.',            'ARM. F PACIFIC',
        'ARM. F. P',              'ARM. F. P.',
        'ARM. F. PAC',            'ARM. F. PAC.',
        'ARM. F. PACIFIC',        'ARM. FOR P',
        'ARM. FOR P.',            'ARM. FOR PAC',
        'ARM. FOR PAC.',          'ARM. FOR PACIFIC',
        'ARM. FOR. P',            'ARM. FOR. P.',
        'ARM. FOR. PAC',          'ARM. FOR. PAC.',
        'ARM. FOR. PACIFIC',      'ARM. FORCE P',
        'ARM. FORCE P.',          'ARM. FORCE PAC',
        'ARM. FORCE PAC.',        'ARM. FORCE PACIFIC',
        'ARM. FORCE. P',          'ARM. FORCE. P.',
        'ARM. FORCE. PAC',        'ARM. FORCE. PAC.',
        'ARM. FORCE. PACIFIC',    'ARM. FORCES P',
        'ARM. FORCES P.',         'ARM. FORCES PAC',
        'ARM. FORCES PAC.',       'ARM. FORCES PACIFIC',
        'ARM.F.P',                'ARM.F.P.',
        'ARM.F.PAC',              'ARM.F.PAC.',
        'ARM.F.PACIFIC',          'ARM.FOR.P',
        'ARM.FOR.P.',             'ARM.FOR.PAC',
        'ARM.FOR.PAC.',           'ARM.FOR.PACIFIC',
        'ARM.FORCE.P',            'ARM.FORCE.P.',
        'ARM.FORCE.PAC',          'ARM.FORCE.PAC.',
        'ARM.FORCE.PACIFIC',      'ARM.FORCEP',
        'ARM.FORCEP.',            'ARM.FORCEPAC',
        'ARM.FORCEPAC.',          'ARM.FORCEPACIFIC',
        'ARM.FORCESP',            'ARM.FORCESP.',
        'ARM.FORCESPAC',          'ARM.FORCESPAC.',
        'ARM.FORCESPACIFIC',      'ARM.FORP',
        'ARM.FORP.',              'ARM.FORPAC',
        'ARM.FORPAC.',            'ARM.FORPACIFIC',
        'ARM.FP',                 'ARM.FP.',
        'ARM.FPAC',               'ARM.FPAC.',
        'ARM.FPACIFIC',           'ARMED F P',
        'ARMED F P.',             'ARMED F PAC',
        'ARMED F PAC.',           'ARMED F PACIFIC',
        'ARMED F. P',             'ARMED F. P.',
        'ARMED F. PAC',           'ARMED F. PAC.',
        'ARMED F. PACIFIC',       'ARMED FOR P',
        'ARMED FOR P.',           'ARMED FOR PAC',
        'ARMED FOR PAC.',         'ARMED FOR PACIFIC',
        'ARMED FOR. P',           'ARMED FOR. P.',
        'ARMED FOR. PAC',         'ARMED FOR. PAC.',
        'ARMED FOR. PACIFIC',     'ARMED FORCE P',
        'ARMED FORCE P.',         'ARMED FORCE PAC',
        'ARMED FORCE PAC.',       'ARMED FORCE PACIFIC',
        'ARMED FORCE. P',         'ARMED FORCE. P.',
        'ARMED FORCE. PAC',       'ARMED FORCE. PAC.',
        'ARMED FORCE. PACIFIC',   'ARMED FORCES P',
        'ARMED FORCES P.',        'ARMED FORCES PAC',
        'ARMED FORCES PAC.',      'ARMED FORCES PACIFIC',
        'ARMEDF.P',               'ARMEDF.P.',
        'ARMEDF.PAC',             'ARMEDF.PAC.',
        'ARMEDF.PACIFIC',         'ARMEDFOR.P',
        'ARMEDFOR.P.',            'ARMEDFOR.PAC',
        'ARMEDFOR.PAC.',          'ARMEDFOR.PACIFIC',
        'ARMEDFORCE.P',           'ARMEDFORCE.P.',
        'ARMEDFORCE.PAC',         'ARMEDFORCE.PAC.',
        'ARMEDFORCE.PACIFIC',     'ARMEDFORCEP',
        'ARMEDFORCEP.',           'ARMEDFORCEPAC',
        'ARMEDFORCEPAC.',         'ARMEDFORCEPACIFIC',
        'ARMEDFORCESP',           'ARMEDFORCESP.',
        'ARMEDFORCESPAC',         'ARMEDFORCESPAC.',
        'ARMEDFORCESPACIFIC',     'ARMEDFORP',
        'ARMEDFORP.',             'ARMEDFORPAC',
        'ARMEDFORPAC.',           'ARMEDFORPACIFIC',
        'ARMEDFP',                'ARMEDFP.',
        'ARMEDFPAC',              'ARMEDFPAC.',
        'ARMEDFPACIFIC',          'ARMF.P',
        'ARMF.P.',                'ARMF.PAC',
        'ARMF.PAC.',              'ARMF.PACIFIC',
        'ARMFOR.P',               'ARMFOR.P.',
        'ARMFOR.PAC',             'ARMFOR.PAC.',
        'ARMFOR.PACIFIC',         'ARMFORCE.P',
        'ARMFORCE.P.',            'ARMFORCE.PAC',
        'ARMFORCE.PAC.',          'ARMFORCE.PACIFIC',
        'ARMFORCEP',              'ARMFORCEP.',
        'ARMFORCEPAC',            'ARMFORCEPAC.',
        'ARMFORCEPACIFIC',        'ARMFORCESP',
        'ARMFORCESP.',            'ARMFORCESPAC',
        'ARMFORCESPAC.',          'ARMFORCESPACIFIC',
        'ARMFORP',                'ARMFORP.',
        'ARMFORPAC',              'ARMFORPAC.',
        'ARMFORPACIFIC',          'ARMFP',
        'ARMFP.',                 'ARMFPAC',
        'ARMFPAC.',               'ARMFPACIFIC',
        'ARM`D F P',              'ARM`D F P.',
        'ARM`D F PAC',            'ARM`D F PAC.',
        'ARM`D F PACIFIC',        'ARM`D F. P',
        'ARM`D F. P.',            'ARM`D F. PAC',
        'ARM`D F. PAC.',          'ARM`D F. PACIFIC',
        'ARM`D FOR P',            'ARM`D FOR P.',
        'ARM`D FOR PAC',          'ARM`D FOR PAC.',
        'ARM`D FOR PACIFIC',      'ARM`D FOR. P',
        'ARM`D FOR. P.',          'ARM`D FOR. PAC',
        'ARM`D FOR. PAC.',        'ARM`D FOR. PACIFIC',
        'ARM`D FORCE P',          'ARM`D FORCE P.',
        'ARM`D FORCE PAC',        'ARM`D FORCE PAC.',
        'ARM`D FORCE PACIFIC',    'ARM`D FORCE. P',
        'ARM`D FORCE. P.',        'ARM`D FORCE. PAC',
        'ARM`D FORCE. PAC.',      'ARM`D FORCE. PACIFIC',
        'ARM`D FORCES P',         'ARM`D FORCES P.',
        'ARM`D FORCES PAC',       'ARM`D FORCES PAC.',
        'ARM`D FORCES PACIFIC',   'ARM`D. F P',
        'ARM`D. F P.',            'ARM`D. F PAC',
        'ARM`D. F PAC.',          'ARM`D. F PACIFIC',
        'ARM`D. F. P',            'ARM`D. F. P.',
        'ARM`D. F. PAC',          'ARM`D. F. PAC.',
        'ARM`D. F. PACIFIC',      'ARM`D. FOR P',
        'ARM`D. FOR P.',          'ARM`D. FOR PAC',
        'ARM`D. FOR PAC.',        'ARM`D. FOR PACIFIC',
        'ARM`D. FOR. P',          'ARM`D. FOR. P.',
        'ARM`D. FOR. PAC',        'ARM`D. FOR. PAC.',
        'ARM`D. FOR. PACIFIC',    'ARM`D. FORCE P',
        'ARM`D. FORCE P.',        'ARM`D. FORCE PAC',
        'ARM`D. FORCE PAC.',      'ARM`D. FORCE PACIFIC',
        'ARM`D. FORCE. P',        'ARM`D. FORCE. P.',
        'ARM`D. FORCE. PAC',      'ARM`D. FORCE. PAC.',
        'ARM`D. FORCE. PACIFIC',  'ARM`D. FORCES P',
        'ARM`D. FORCES P.',       'ARM`D. FORCES PAC',
        'ARM`D. FORCES PAC.',     'ARM`D. FORCES PACIFIC',
        'ARM`D.F.P',              'ARM`D.F.P.',
        'ARM`D.F.PAC',            'ARM`D.F.PAC.',
        'ARM`D.F.PACIFIC',        'ARM`D.FOR.P',
        'ARM`D.FOR.P.',           'ARM`D.FOR.PAC',
        'ARM`D.FOR.PAC.',         'ARM`D.FOR.PACIFIC',
        'ARM`D.FORCE.P',          'ARM`D.FORCE.P.',
        'ARM`D.FORCE.PAC',        'ARM`D.FORCE.PAC.',
        'ARM`D.FORCE.PACIFIC',    'ARM`D.FORCEP',
        'ARM`D.FORCEP.',          'ARM`D.FORCEPAC',
        'ARM`D.FORCEPAC.',        'ARM`D.FORCEPACIFIC',
        'ARM`D.FORCESP',          'ARM`D.FORCESP.',
        'ARM`D.FORCESPAC',        'ARM`D.FORCESPAC.',
        'ARM`D.FORCESPACIFIC',    'ARM`D.FORP',
        'ARM`D.FORP.',            'ARM`D.FORPAC',
        'ARM`D.FORPAC.',          'ARM`D.FORPACIFIC',
        'ARM`D.FP',               'ARM`D.FP.',
        'ARM`D.FPAC',             'ARM`D.FPAC.',
        'ARM`D.FPACIFIC',         'ARM`DF.P',
        'ARM`DF.P.',              'ARM`DF.PAC',
        'ARM`DF.PAC.',            'ARM`DF.PACIFIC',
        'ARM`DFOR.P',             'ARM`DFOR.P.',
        'ARM`DFOR.PAC',           'ARM`DFOR.PAC.',
        'ARM`DFOR.PACIFIC',       'ARM`DFORCE.P',
        'ARM`DFORCE.P.',          'ARM`DFORCE.PAC',
        'ARM`DFORCE.PAC.',        'ARM`DFORCE.PACIFIC',
        'ARM`DFORCEP',            'ARM`DFORCEP.',
        'ARM`DFORCEPAC',          'ARM`DFORCEPAC.',
        'ARM`DFORCEPACIFIC',      'ARM`DFORCESP',
        'ARM`DFORCESP.',          'ARM`DFORCESPAC',
        'ARM`DFORCESPAC.',        'ARM`DFORCESPACIFIC',
        'ARM`DFORP',              'ARM`DFORP.',
        'ARM`DFORPAC',            'ARM`DFORPAC.',
        'ARM`DFORPACIFIC',        'ARM`DFP',
        'ARM`DFP.',               'ARM`DFPAC',
        'ARM`DFPAC.',             'ARM`DFPACIFIC'
    ],
    'AR' => [ 'AR', 'AR.', 'ARK.', 'ARKANSAS' ],
    'AS' => [
        'A S',           'A S.',
        'A SAM',         'A SAM.',
        'A SAMOA',       'A. S',
        'A. S.',         'A. SAM',
        'A. SAM.',       'A. SAMOA',
        'A.S',           'A.S.',
        'A.SAM',         'A.SAM.',
        'A.SAMOA',       'AM S',
        'AM S.',         'AM SAM',
        'AM SAM.',       'AM SAMOA',
        'AM. S',         'AM. S.',
        'AM. SAM',       'AM. SAM.',
        'AM. SAMOA',     'AM.S',
        'AM.S.',         'AM.SAM',
        'AM.SAM.',       'AM.SAMOA',
        'AMER S',        'AMER S.',
        'AMER SAM',      'AMER SAM.',
        'AMER SAMOA',    'AMER. S',
        'AMER. S.',      'AMER. SAM',
        'AMER. SAM.',    'AMER. SAMOA',
        'AMER.S',        'AMER.S.',
        'AMER.SAM',      'AMER.SAM.',
        'AMER.SAMOA',    'AMERICAN S',
        'AMERICAN S.',   'AMERICAN SAM',
        'AMERICAN SAM.', 'AMERICAN SAMOA',
        'AMERICANS',     'AMERICANS.',
        'AMERICANSAM',   'AMERICANSAM.',
        'AMERICANSAMOA', 'AMERS',
        'AMERS.',        'AMERSAM',
        'AMERSAM.',      'AMERSAMOA',
        'AMS',           'AMS.',
        'AMSAM',         'AMSAM.',
        'AMSAMOA',       'AS',
        'AS.',           'ASAM',
        'ASAM.',         'ASAMOA'
    ],
    'AZ' => [ 'ARIZ', 'ARIZ.', 'ARIZONA', 'AZ',    'AZ.' ],
    'CA' => [ 'CA',   'CA.',   'CAL.',    'CALIF', 'CALIF.', 'CALIFORNIA' ],
    'CO' => [ 'CO', 'CO.', 'COL', 'COL.', 'COLO', 'COLO.', 'COLORADO' ],
    'CT' => [ 'CONN', 'CONN.', 'CONNECTICUT', 'CT', 'CT.' ],
    'DC' => [
        'D C', 'D C.', 'D. C', 'D. C.', 'D.C', 'D.C.', 'DC', 'DC.',
        'DIST OF COLUMBIA',
        'DIST. OF COLUMBIA',
        'DISTRICT OF COLUMBIA',
    ],
    'DE' => [ 'DE', 'DE.', 'DEL',  'DEL.', 'DELEWARE' ],
    'FL' => [ 'FL', 'FL.', 'FLA.', 'FLORIDA' ],
    'FM' => [
        'F S M',                       'F S M.',
        'F S MIC',                     'F S MIC.',
        'F S MICRO',                   'F S MICRO.',
        'F S MICRONESIA',              'F S O M',
        'F S O. M.',                   'F S OF MIC',
        'F S. M',                      'F S. M.',
        'F S. MIC',                    'F S. MIC.',
        'F S. MICRO',                  'F S. MICRO.',
        'F S. MICRONESIA',             'F S. O M',
        'F S. O. M.',                  'F S. OF MIC',
        'F ST M',                      'F ST M.',
        'F ST MIC',                    'F ST MIC.',
        'F ST MICRO',                  'F ST MICRO.',
        'F ST MICRONESIA',             'F ST O M',
        'F ST O. M.',                  'F ST OF MIC',
        'F ST. M',                     'F ST. M.',
        'F ST. MIC',                   'F ST. MIC.',
        'F ST. MICRO',                 'F ST. MICRO.',
        'F ST. MICRONESIA',            'F ST. O M',
        'F ST. O. M.',                 'F ST. OF MIC',
        'F STATE M',                   'F STATE M.',
        'F STATE MIC',                 'F STATE MIC.',
        'F STATE MICRO',               'F STATE MICRO.',
        'F STATE MICRONESIA',          'F STATE O M',
        'F STATE O. M.',               'F STATE OF MIC',
        'F STATES M',                  'F STATES M.',
        'F STATES MIC',                'F STATES MIC.',
        'F STATES MICRO',              'F STATES MICRO.',
        'F STATES MICRONESIA',         'F STATES O M',
        'F STATES O. M.',              'F STATES OF MIC',
        'F. S M',                      'F. S M.',
        'F. S MIC',                    'F. S MIC.',
        'F. S MICRO',                  'F. S MICRO.',
        'F. S MICRONESIA',             'F. S O M',
        'F. S O. M.',                  'F. S OF MIC',
        'F. S. M',                     'F. S. M.',
        'F. S. MIC',                   'F. S. MIC.',
        'F. S. MICRO',                 'F. S. MICRO.',
        'F. S. MICRONESIA',            'F. S. O M',
        'F. S. O. M.',                 'F. S. OF MIC',
        'F. ST M',                     'F. ST M.',
        'F. ST MIC',                   'F. ST MIC.',
        'F. ST MICRO',                 'F. ST MICRO.',
        'F. ST MICRONESIA',            'F. ST O M',
        'F. ST O. M.',                 'F. ST OF MIC',
        'F. ST. M',                    'F. ST. M.',
        'F. ST. MIC',                  'F. ST. MIC.',
        'F. ST. MICRO',                'F. ST. MICRO.',
        'F. ST. MICRONESIA',           'F. ST. O M',
        'F. ST. O. M.',                'F. ST. OF MIC',
        'F. STATE M',                  'F. STATE M.',
        'F. STATE MIC',                'F. STATE MIC.',
        'F. STATE MICRO',              'F. STATE MICRO.',
        'F. STATE MICRONESIA',         'F. STATE O M',
        'F. STATE O. M.',              'F. STATE OF MIC',
        'F. STATES M',                 'F. STATES M.',
        'F. STATES MIC',               'F. STATES MIC.',
        'F. STATES MICRO',             'F. STATES MICRO.',
        'F. STATES MICRONESIA',        'F. STATES O M',
        'F. STATES O. M.',             'F. STATES OF MIC',
        'F.S.M',                       'F.S.M.',
        'F.S.MIC',                     'F.S.MIC.',
        'F.S.MICRO',                   'F.S.MICRO.',
        'F.S.MICRONESIA',              'F.S.O.M.',
        'F.S.OFMIC',                   'F.S.OM',
        'F.SM',                        'F.SM.',
        'F.SMIC',                      'F.SMIC.',
        'F.SMICRO',                    'F.SMICRO.',
        'F.SMICRONESIA',               'F.SO.M.',
        'F.SOFMIC',                    'F.SOM',
        'F.ST.M',                      'F.ST.M.',
        'F.ST.MIC',                    'F.ST.MIC.',
        'F.ST.MICRO',                  'F.ST.MICRO.',
        'F.ST.MICRONESIA',             'F.ST.O.M.',
        'F.ST.OFMIC',                  'F.ST.OM',
        'F.STATEM',                    'F.STATEM.',
        'F.STATEMIC',                  'F.STATEMIC.',
        'F.STATEMICRO',                'F.STATEMICRO.',
        'F.STATEMICRONESIA',           'F.STATEO.M.',
        'F.STATEOFMIC',                'F.STATEOM',
        'F.STATESM',                   'F.STATESM.',
        'F.STATESMIC',                 'F.STATESMIC.',
        'F.STATESMICRO',               'F.STATESMICRO.',
        'F.STATESMICRONESIA',          'F.STATESO.M.',
        'F.STATESOFMIC',               'F.STATESOM',
        'F.STM',                       'F.STM.',
        'F.STMIC',                     'F.STMIC.',
        'F.STMICRO',                   'F.STMICRO.',
        'F.STMICRONESIA',              'F.STO.M.',
        'F.STOFMIC',                   'F.STOM',
        'FED S M',                     'FED S M.',
        'FED S MIC',                   'FED S MIC.',
        'FED S MICRO',                 'FED S MICRO.',
        'FED S MICRONESIA',            'FED S O M',
        'FED S O. M.',                 'FED S OF MIC',
        'FED S. M',                    'FED S. M.',
        'FED S. MIC',                  'FED S. MIC.',
        'FED S. MICRO',                'FED S. MICRO.',
        'FED S. MICRONESIA',           'FED S. O M',
        'FED S. O. M.',                'FED S. OF MIC',
        'FED ST M',                    'FED ST M.',
        'FED ST MIC',                  'FED ST MIC.',
        'FED ST MICRO',                'FED ST MICRO.',
        'FED ST MICRONESIA',           'FED ST O M',
        'FED ST O. M.',                'FED ST OF MIC',
        'FED ST. M',                   'FED ST. M.',
        'FED ST. MIC',                 'FED ST. MIC.',
        'FED ST. MICRO',               'FED ST. MICRO.',
        'FED ST. MICRONESIA',          'FED ST. O M',
        'FED ST. O. M.',               'FED ST. OF MIC',
        'FED STATE M',                 'FED STATE M.',
        'FED STATE MIC',               'FED STATE MIC.',
        'FED STATE MICRO',             'FED STATE MICRO.',
        'FED STATE MICRONESIA',        'FED STATE O M',
        'FED STATE O. M.',             'FED STATE OF MIC',
        'FED STATES M',                'FED STATES M.',
        'FED STATES MIC',              'FED STATES MIC.',
        'FED STATES MICRO',            'FED STATES MICRO.',
        'FED STATES MICRONESIA',       'FED STATES O M',
        'FED STATES O. M.',            'FED STATES OF MIC',
        'FED. S M',                    'FED. S M.',
        'FED. S MIC',                  'FED. S MIC.',
        'FED. S MICRO',                'FED. S MICRO.',
        'FED. S MICRONESIA',           'FED. S O M',
        'FED. S O. M.',                'FED. S OF MIC',
        'FED. S. M',                   'FED. S. M.',
        'FED. S. MIC',                 'FED. S. MIC.',
        'FED. S. MICRO',               'FED. S. MICRO.',
        'FED. S. MICRONESIA',          'FED. S. O M',
        'FED. S. O. M.',               'FED. S. OF MIC',
        'FED. ST M',                   'FED. ST M.',
        'FED. ST MIC',                 'FED. ST MIC.',
        'FED. ST MICRO',               'FED. ST MICRO.',
        'FED. ST MICRONESIA',          'FED. ST O M',
        'FED. ST O. M.',               'FED. ST OF MIC',
        'FED. ST. M',                  'FED. ST. M.',
        'FED. ST. MIC',                'FED. ST. MIC.',
        'FED. ST. MICRO',              'FED. ST. MICRO.',
        'FED. ST. MICRONESIA',         'FED. ST. O M',
        'FED. ST. O. M.',              'FED. ST. OF MIC',
        'FED. STATE M',                'FED. STATE M.',
        'FED. STATE MIC',              'FED. STATE MIC.',
        'FED. STATE MICRO',            'FED. STATE MICRO.',
        'FED. STATE MICRONESIA',       'FED. STATE O M',
        'FED. STATE O. M.',            'FED. STATE OF MIC',
        'FED. STATES M',               'FED. STATES M.',
        'FED. STATES MIC',             'FED. STATES MIC.',
        'FED. STATES MICRO',           'FED. STATES MICRO.',
        'FED. STATES MICRONESIA',      'FED. STATES O M',
        'FED. STATES O. M.',           'FED. STATES OF MIC',
        'FED.S.M',                     'FED.S.M.',
        'FED.S.MIC',                   'FED.S.MIC.',
        'FED.S.MICRO',                 'FED.S.MICRO.',
        'FED.S.MICRONESIA',            'FED.S.O.M.',
        'FED.S.OFMIC',                 'FED.S.OM',
        'FED.SM',                      'FED.SM.',
        'FED.SMIC',                    'FED.SMIC.',
        'FED.SMICRO',                  'FED.SMICRO.',
        'FED.SMICRONESIA',             'FED.SO.M.',
        'FED.SOFMIC',                  'FED.SOM',
        'FED.ST.M',                    'FED.ST.M.',
        'FED.ST.MIC',                  'FED.ST.MIC.',
        'FED.ST.MICRO',                'FED.ST.MICRO.',
        'FED.ST.MICRONESIA',           'FED.ST.O.M.',
        'FED.ST.OFMIC',                'FED.ST.OM',
        'FED.STATEM',                  'FED.STATEM.',
        'FED.STATEMIC',                'FED.STATEMIC.',
        'FED.STATEMICRO',              'FED.STATEMICRO.',
        'FED.STATEMICRONESIA',         'FED.STATEO.M.',
        'FED.STATEOFMIC',              'FED.STATEOM',
        'FED.STATESM',                 'FED.STATESM.',
        'FED.STATESMIC',               'FED.STATESMIC.',
        'FED.STATESMICRO',             'FED.STATESMICRO.',
        'FED.STATESMICRONESIA',        'FED.STATESO.M.',
        'FED.STATESOFMIC',             'FED.STATESOM',
        'FED.STM',                     'FED.STM.',
        'FED.STMIC',                   'FED.STMIC.',
        'FED.STMICRO',                 'FED.STMICRO.',
        'FED.STMICRONESIA',            'FED.STO.M.',
        'FED.STOFMIC',                 'FED.STOM',
        'FEDERATED S M',               'FEDERATED S M.',
        'FEDERATED S MIC',             'FEDERATED S MIC.',
        'FEDERATED S MICRO',           'FEDERATED S MICRO.',
        'FEDERATED S MICRONESIA',      'FEDERATED S O M',
        'FEDERATED S O. M.',           'FEDERATED S OF MIC',
        'FEDERATED S. M',              'FEDERATED S. M.',
        'FEDERATED S. MIC',            'FEDERATED S. MIC.',
        'FEDERATED S. MICRO',          'FEDERATED S. MICRO.',
        'FEDERATED S. MICRONESIA',     'FEDERATED S. O M',
        'FEDERATED S. O. M.',          'FEDERATED S. OF MIC',
        'FEDERATED ST M',              'FEDERATED ST M.',
        'FEDERATED ST MIC',            'FEDERATED ST MIC.',
        'FEDERATED ST MICRO',          'FEDERATED ST MICRO.',
        'FEDERATED ST MICRONESIA',     'FEDERATED ST O M',
        'FEDERATED ST O. M.',          'FEDERATED ST OF MIC',
        'FEDERATED ST. M',             'FEDERATED ST. M.',
        'FEDERATED ST. MIC',           'FEDERATED ST. MIC.',
        'FEDERATED ST. MICRO',         'FEDERATED ST. MICRO.',
        'FEDERATED ST. MICRONESIA',    'FEDERATED ST. O M',
        'FEDERATED ST. O. M.',         'FEDERATED ST. OF MIC',
        'FEDERATED STATE M',           'FEDERATED STATE M.',
        'FEDERATED STATE MIC',         'FEDERATED STATE MIC.',
        'FEDERATED STATE MICRO',       'FEDERATED STATE MICRO.',
        'FEDERATED STATE MICRONESIA',  'FEDERATED STATE O M',
        'FEDERATED STATE O. M.',       'FEDERATED STATE OF MIC',
        'FEDERATED STATES M',          'FEDERATED STATES M.',
        'FEDERATED STATES MIC',        'FEDERATED STATES MIC.',
        'FEDERATED STATES MICRO',      'FEDERATED STATES MICRO.',
        'FEDERATED STATES MICRONESIA', 'FEDERATED STATES O M',
        'FEDERATED STATES O. M.',      'FEDERATED STATES OF MIC',
        'FEDERATEDS.M',                'FEDERATEDS.M.',
        'FEDERATEDS.MIC',              'FEDERATEDS.MIC.',
        'FEDERATEDS.MICRO',            'FEDERATEDS.MICRO.',
        'FEDERATEDS.MICRONESIA',       'FEDERATEDS.O.M.',
        'FEDERATEDS.OFMIC',            'FEDERATEDS.OM',
        'FEDERATEDSM',                 'FEDERATEDSM.',
        'FEDERATEDSMIC',               'FEDERATEDSMIC.',
        'FEDERATEDSMICRO',             'FEDERATEDSMICRO.',
        'FEDERATEDSMICRONESIA',        'FEDERATEDSO.M.',
        'FEDERATEDSOFMIC',             'FEDERATEDSOM',
        'FEDERATEDST.M',               'FEDERATEDST.M.',
        'FEDERATEDST.MIC',             'FEDERATEDST.MIC.',
        'FEDERATEDST.MICRO',           'FEDERATEDST.MICRO.',
        'FEDERATEDST.MICRONESIA',      'FEDERATEDST.O.M.',
        'FEDERATEDST.OFMIC',           'FEDERATEDST.OM',
        'FEDERATEDSTATEM',             'FEDERATEDSTATEM.',
        'FEDERATEDSTATEMIC',           'FEDERATEDSTATEMIC.',
        'FEDERATEDSTATEMICRO',         'FEDERATEDSTATEMICRO.',
        'FEDERATEDSTATEMICRONESIA',    'FEDERATEDSTATEO.M.',
        'FEDERATEDSTATEOFMIC',         'FEDERATEDSTATEOM',
        'FEDERATEDSTATESM',            'FEDERATEDSTATESM.',
        'FEDERATEDSTATESMIC',          'FEDERATEDSTATESMIC.',
        'FEDERATEDSTATESMICRO',        'FEDERATEDSTATESMICRO.',
        'FEDERATEDSTATESMICRONESIA',   'FEDERATEDSTATESO.M.',
        'FEDERATEDSTATESOFMIC',        'FEDERATEDSTATESOM',
        'FEDERATEDSTM',                'FEDERATEDSTM.',
        'FEDERATEDSTMIC',              'FEDERATEDSTMIC.',
        'FEDERATEDSTMICRO',            'FEDERATEDSTMICRO.',
        'FEDERATEDSTMICRONESIA',       'FEDERATEDSTO.M.',
        'FEDERATEDSTOFMIC',            'FEDERATEDSTOM',
        'FEDS.M',                      'FEDS.M.',
        'FEDS.MIC',                    'FEDS.MIC.',
        'FEDS.MICRO',                  'FEDS.MICRO.',
        'FEDS.MICRONESIA',             'FEDS.O.M.',
        'FEDS.OFMIC',                  'FEDS.OM',
        'FEDSM',                       'FEDSM.',
        'FEDSMIC',                     'FEDSMIC.',
        'FEDSMICRO',                   'FEDSMICRO.',
        'FEDSMICRONESIA',              'FEDSO.M.',
        'FEDSOFMIC',                   'FEDSOM',
        'FEDST.M',                     'FEDST.M.',
        'FEDST.MIC',                   'FEDST.MIC.',
        'FEDST.MICRO',                 'FEDST.MICRO.',
        'FEDST.MICRONESIA',            'FEDST.O.M.',
        'FEDST.OFMIC',                 'FEDST.OM',
        'FEDSTATEM',                   'FEDSTATEM.',
        'FEDSTATEMIC',                 'FEDSTATEMIC.',
        'FEDSTATEMICRO',               'FEDSTATEMICRO.',
        'FEDSTATEMICRONESIA',          'FEDSTATEO.M.',
        'FEDSTATEOFMIC',               'FEDSTATEOM',
        'FEDSTATESM',                  'FEDSTATESM.',
        'FEDSTATESMIC',                'FEDSTATESMIC.',
        'FEDSTATESMICRO',              'FEDSTATESMICRO.',
        'FEDSTATESMICRONESIA',         'FEDSTATESO.M.',
        'FEDSTATESOFMIC',              'FEDSTATESOM',
        'FEDSTM',                      'FEDSTM.',
        'FEDSTMIC',                    'FEDSTMIC.',
        'FEDSTMICRO',                  'FEDSTMICRO.',
        'FEDSTMICRONESIA',             'FEDSTO.M.',
        'FEDSTOFMIC',                  'FEDSTOM',
        'FM',                          'FM.',
        'FS.M',                        'FS.M.',
        'FS.MIC',                      'FS.MIC.',
        'FS.MICRO',                    'FS.MICRO.',
        'FS.MICRONESIA',               'FS.O.M.',
        'FS.OFMIC',                    'FS.OM',
        'FSM',                         'FSM.',
        'FSMIC',                       'FSMIC.',
        'FSMICRO',                     'FSMICRO.',
        'FSMICRONESIA',                'FSO.M.',
        'FSOFMIC',                     'FSOM',
        'FST.M',                       'FST.M.',
        'FST.MIC',                     'FST.MIC.',
        'FST.MICRO',                   'FST.MICRO.',
        'FST.MICRONESIA',              'FST.O.M.',
        'FST.OFMIC',                   'FST.OM',
        'FSTATEM',                     'FSTATEM.',
        'FSTATEMIC',                   'FSTATEMIC.',
        'FSTATEMICRO',                 'FSTATEMICRO.',
        'FSTATEMICRONESIA',            'FSTATEO.M.',
        'FSTATEOFMIC',                 'FSTATEOM',
        'FSTATESM',                    'FSTATESM.',
        'FSTATESMIC',                  'FSTATESMIC.',
        'FSTATESMICRO',                'FSTATESMICRO.',
        'FSTATESMICRONESIA',           'FSTATESO.M.',
        'FSTATESOFMIC',                'FSTATESOM',
        'FSTM',                        'FSTM.',
        'FSTMIC',                      'FSTMIC.',
        'FSTMICRO',                    'FSTMICRO.',
        'FSTMICRONESIA',               'FSTO.M.',
        'FSTOFMIC',                    'FSTOM'
    ],
    'GA' => [ 'GA',     'GA.',  'GEORGIA' ],
    'GU' => [ 'GU',     'GU.',  'GUAM' ],
    'HI' => [ 'HAWAII', 'HI',   'HI.' ],
    'IA' => [ 'IA',     'IA.',  'IOWA' ],
    'ID' => [ 'ID',     'ID.',  'IDAHO' ],
    'IL' => [ 'IL',     'IL.',  'ILL.', 'ILLINOIS' ],
    'IN' => [ 'IN',     'IN.',  'IND', 'IND.', 'INDIANA' ],
    'KS' => [ 'KAN',    'KAN.', 'KANS', 'KANS.', 'KANSAS', 'KS', 'KS.' ],
    'KY' => [ 'KEN', 'KEN.', 'KENTUCK', 'KENTUCK.', 'KENTUCKY', 'KY', 'KY.' ],
    'LA' => [ 'LA',       'LA.', 'LOUIS', 'LOUIS.', 'LOUISIANA' ],
    'MA' => [ 'MA',       'MA.', 'MASS',  'MASS.',  'MASSACHUSETTS' ],
    'MD' => [ 'MARYLAND', 'MD',  'MD.' ],
    'ME' => [ 'MAINE',    'ME',  'ME.' ],
    'MH' => [
        'MARSH IS',         'MARSH IS.',
        'MARSH ISLAND',     'MARSH ISLAND.',
        'MARSH ISLANDS',    'MARSH. IS',
        'MARSH. IS.',       'MARSH. ISLAND',
        'MARSH. ISLAND.',   'MARSH. ISLANDS',
        'MARSH.IS',         'MARSH.IS.',
        'MARSH.ISLAND',     'MARSH.ISLAND.',
        'MARSH.ISLANDS',    'MARSHALL IS',
        'MARSHALL IS.',     'MARSHALL ISLAND',
        'MARSHALL ISLAND.', 'MARSHALL ISLANDS',
        'MARSHALLIS',       'MARSHALLIS.',
        'MARSHALLISLAND',   'MARSHALLISLAND.',
        'MARSHALLISLANDS',  'MARSHIS',
        'MARSHIS.',         'MARSHISLAND',
        'MARSHISLAND.',     'MARSHISLANDS',
        'MH',               'MH.',
    ],
    'MI' => [ 'MI',       'MI.',   'MICH',      'MICH.', 'MICHIGAN' ],
    'MN' => [ 'MINN',     'MINN.', 'MINNESOTA', 'MN',    'MN.' ],
    'MO' => [ 'MISSOURI', 'MO',    'MO.' ],
    'MP' => [
        'MP',                       'MP.',
        'N MARIANA I',              'N MARIANA I.',
        'N MARIANA IS',             'N MARIANA IS.',
        'N MARIANA ISLAND',         'N MARIANA ISLAND.',
        'N MARIANA ISLANDS',        'N. MARIANA I',
        'N. MARIANA I.',            'N. MARIANA IS',
        'N. MARIANA IS.',           'N. MARIANA ISLAND',
        'N. MARIANA ISLAND.',       'N. MARIANA ISLANDS',
        'N.MARIANAI',               'N.MARIANAI.',
        'N.MARIANAIS',              'N.MARIANAIS.',
        'N.MARIANAISLAND',          'N.MARIANAISLAND.',
        'N.MARIANAISLANDS',         'NMARIANAI',
        'NMARIANAI.',               'NMARIANAIS',
        'NMARIANAIS.',              'NMARIANAISLAND',
        'NMARIANAISLAND.',          'NMARIANAISLANDS',
        'NOR MARIANA I',            'NOR MARIANA I.',
        'NOR MARIANA IS',           'NOR MARIANA IS.',
        'NOR MARIANA ISLAND',       'NOR MARIANA ISLAND.',
        'NOR MARIANA ISLANDS',      'NOR. MARIANA I',
        'NOR. MARIANA I.',          'NOR. MARIANA IS',
        'NOR. MARIANA IS.',         'NOR. MARIANA ISLAND',
        'NOR. MARIANA ISLAND.',     'NOR. MARIANA ISLANDS',
        'NOR.MARIANAI',             'NOR.MARIANAI.',
        'NOR.MARIANAIS',            'NOR.MARIANAIS.',
        'NOR.MARIANAISLAND',        'NOR.MARIANAISLAND.',
        'NOR.MARIANAISLANDS',       'NORMARIANAI',
        'NORMARIANAI.',             'NORMARIANAIS',
        'NORMARIANAIS.',            'NORMARIANAISLAND',
        'NORMARIANAISLAND.',        'NORMARIANAISLANDS',
        'NORTH MARIANA I',          'NORTH MARIANA I.',
        'NORTH MARIANA IS',         'NORTH MARIANA IS.',
        'NORTH MARIANA ISLAND',     'NORTH MARIANA ISLAND.',
        'NORTH MARIANA ISLANDS',    'NORTH. MARIANA I',
        'NORTH. MARIANA I.',        'NORTH. MARIANA IS',
        'NORTH. MARIANA IS.',       'NORTH. MARIANA ISLAND',
        'NORTH. MARIANA ISLAND.',   'NORTH. MARIANA ISLANDS',
        'NORTH.MARIANAI',           'NORTH.MARIANAI.',
        'NORTH.MARIANAIS',          'NORTH.MARIANAIS.',
        'NORTH.MARIANAISLAND',      'NORTH.MARIANAISLAND.',
        'NORTH.MARIANAISLANDS',     'NORTHERN MARIANA I',
        'NORTHERN MARIANA I.',      'NORTHERN MARIANA IS',
        'NORTHERN MARIANA IS.',     'NORTHERN MARIANA ISLAND',
        'NORTHERN MARIANA ISLAND.', 'NORTHERN MARIANA ISLANDS',
        'NORTHERNMARIANAI',         'NORTHERNMARIANAI.',
        'NORTHERNMARIANAIS',        'NORTHERNMARIANAIS.',
        'NORTHERNMARIANAISLAND',    'NORTHERNMARIANAISLAND.',
        'NORTHERNMARIANAISLANDS',   'NORTHMARIANAI',
        'NORTHMARIANAI.',           'NORTHMARIANAIS',
        'NORTHMARIANAIS.',          'NORTHMARIANAISLAND',
        'NORTHMARIANAISLAND.',      'NORTHMARIANAISLANDS',
    ],
    'MS' => [ 'MISS', 'MISS.', 'MISSISSIPPI', 'MS', 'MS.' ],
    'MT' => [ 'MONT', 'MONT.', 'MONTANA',     'MT', 'MT.' ],
    'NC' => [
        'N C',            'N C.',
        'N CAR',          'N CAR.',
        'N CAROLINA',     'N. C',
        'N. C.',          'N. CAR',
        'N. CAR.',        'N. CAROLINA',
        'N.C',            'N.C.',
        'N.CAR',          'N.CAR.',
        'N.CAROLINA',     'NC',
        'NC.',            'NCAR',
        'NCAR.',          'NCAROLINA',
        'NOR C',          'NOR C.',
        'NOR CAR',        'NOR CAR.',
        'NOR CAROLINA',   'NOR. C',
        'NOR. C.',        'NOR. CAR',
        'NOR. CAR.',      'NOR. CAROLINA',
        'NOR.C',          'NOR.C.',
        'NOR.CAR',        'NOR.CAR.',
        'NOR.CAROLINA',   'NORC',
        'NORC.',          'NORCAR',
        'NORCAR.',        'NORCAROLINA',
        'NORTH C',        'NORTH C.',
        'NORTH CAR',      'NORTH CAR.',
        'NORTH CAROLINA', 'NORTHC',
        'NORTHC.',        'NORTHCAR',
        'NORTHCAR.',      'NORTHCAROLINA'
    ],
    'ND' => [
        'N D',          'N D.',        'N DAK',      'N DAK.',
        'N DAKOTA',     'N. D',        'N. D.',      'N. DAK',
        'N. DAK.',      'N. DAKOTA',   'N.D',        'N.D.',
        'N.DAK',        'N.DAK.',      'N.DAKOTA',   'ND',
        'ND.',          'NDAK',        'NDAK.',      'NDAKOTA',
        'NOR D',        'NOR D.',      'NOR DAK',    'NOR DAK.',
        'NOR DAKOTA',   'NOR. D',      'NOR. D.',    'NOR. DAK',
        'NOR. DAK.',    'NOR. DAKOTA', 'NOR.D',      'NOR.D.',
        'NOR.DAK',      'NOR.DAK.',    'NOR.DAKOTA', 'NORD',
        'NORD.',        'NORDAK',      'NORDAK.',    'NORDAKOTA',
        'NORTH D',      'NORTH D.',    'NORTH DAK',  'NORTH DAK.',
        'NORTH DAKOTA', 'NORTHD',      'NORTHD.',    'NORTHDAK',
        'NORTHDAK.',    'NORTHDAKOTA'
    ],
    'NE' => [ 'NE', 'NE.', 'NEB', 'NEB.', 'NEBR', 'NEBR.', 'NEBRASKA' ],
    'NH' => [
        'N H',         'N H.',         'N HAMP',       'N HAMP.',
        'N HAMPS',     'N HAMPS.',     'N HAMPSHIRE',  'N. H',
        'N. H.',       'N. HAMP',      'N. HAMP.',     'N. HAMPS',
        'N. HAMPS.',   'N. HAMPSHIRE', 'N.H',          'N.H.',
        'N.HAMP',      'N.HAMP.',      'N.HAMPS',      'N.HAMPS.',
        'N.HAMPSHIRE', 'NEW H',        'NEW H.',       'NEW HAMP',
        'NEW HAMP.',   'NEW HAMPS',    'NEW HAMPS.',   'NEW HAMPSHIRE',
        'NEWH',        'NEWH.',        'NEWHAMP',      'NEWHAMP.',
        'NEWHAMPS',    'NEWHAMPS.',    'NEWHAMPSHIRE', 'NH',
        'NH.',         'NHAMP',        'NHAMP.',       'NHAMPS',
        'NHAMPS.',     'NHAMPSHIRE'
    ],
    'NJ' => [
        'N J',      'N J.',      'N JER',     'N JER.',
        'N JERS',   'N JERS.',   'N JERSEY',  'N. J',
        'N. J.',    'N. JER',    'N. JER.',   'N. JERS',
        'N. JERS.', 'N. JERSEY', 'N.J',       'N.J.',
        'N.JER',    'N.JER.',    'N.JERS',    'N.JERS.',
        'N.JERSEY', 'NEW J',     'NEW J.',    'NEW JER',
        'NEW JER.', 'NEW JERS',  'NEW JERS.', 'NEW JERSEY',
        'NEWJ',     'NEWJ.',     'NEWJER',    'NEWJER.',
        'NEWJERS',  'NEWJERS.',  'NEWJERSEY', 'NJ',
        'NJ.',      'NJER',      'NJER.',     'NJERS',
        'NJERS.',   'NJERSEY'
    ],
    'NM' => [
        'N M',       'N M.',      'N MEX',    'N MEX.',
        'N MEXICO',  'N. M',      'N. M.',    'N. MEX',
        'N. MEX.',   'N. MEXICO', 'N.M',      'N.M.',
        'N.MEX',     'N.MEX.',    'N.MEXICO', 'NEW M',
        'NEW M.',    'NEW MEX',   'NEW MEX.', 'NEW MEXICO',
        'NEWM',      'NEWM.',     'NEWMEX',   'NEWMEX.',
        'NEWMEXICO', 'NM',        'NM.',      'NMEX',
        'NMEX.',     'NMEXICO'
    ],
    'NV' => [ 'NEV', 'NEV.', 'NEVADA', 'NV', 'NV.' ],
    'NY' => [
        'N Y',  'N Y.',  'N YORK',  'N. Y',  'N. Y.',  'N. YORK',
        'N.Y',  'N.Y.',  'N.YORK',  'NEW Y', 'NEW Y.', 'NEW YORK',
        'NEWY', 'NEWY.', 'NEWYORK', 'NY',    'NY.',    'NYORK'
    ],
    'OH' => [ 'OH', 'OH.', 'OHIO' ],
    'OK' => [ 'OK', 'OK.', 'OKLA', 'OKLA.', 'OKLAHOMA' ],
    'OR' => [ 'OR', 'OR.', 'ORE', 'ORE.', 'OREG', 'OREG.', 'OREGON' ],
    'PA' => [ 'PA', 'PA.', 'PENN', 'PENN.', 'PENNSYLVANIA' ],
    'PR' => [
        'P R',         'P R.',       'P RI',      'P RI.',
        'P RICO',      'P. R',       'P. R.',     'P. RI',
        'P. RI.',      'P. RICO',    'P.R',       'P.R.',
        'P.RI',        'P.RI.',      'P.RICO',    'PR',
        'PR.',         'PRI',        'PRI.',      'PRICO',
        'PUER R',      'PUER R.',    'PUER RI',   'PUER RI.',
        'PUER RICO',   'PUER. R',    'PUER. R.',  'PUER. RI',
        'PUER. RI.',   'PUER. RICO', 'PUER.R',    'PUER.R.',
        'PUER.RI',     'PUER.RI.',   'PUER.RICO', 'PUERR',
        'PUERR.',      'PUERRI',     'PUERRI.',   'PUERRICO',
        'PUERTO R',    'PUERTO R.',  'PUERTO RI', 'PUERTO RI.',
        'PUERTO RICO', 'PUERTOR',    'PUERTOR.',  'PUERTORI',
        'PUERTORI.',   'PUERTORICO'
    ],
    'PW' => [ 'PAL', 'PAL.', 'PALAU', 'PW', 'PW.' ],
    'RI' => [
        'R I',         'R I.',      'R IS',      'R IS.',
        'R ISLAND',    'R. I',      'R. I.',     'R. IS',
        'R. IS.',      'R. ISLAND', 'R.I',       'R.I.',
        'R.IS',        'R.IS.',     'R.ISLAND',  'RH I',
        'RH I.',       'RH IS',     'RH IS.',    'RH ISLAND',
        'RH. I',       'RH. I.',    'RH. IS',    'RH. IS.',
        'RH. ISLAND',  'RH.I',      'RH.I.',     'RH.IS',
        'RH.IS.',      'RH.ISLAND', 'RHI',       'RHI.',
        'RHIS',        'RHIS.',     'RHISLAND',  'RHODE I',
        'RHODE I.',    'RHODE IS',  'RHODE IS.', 'RHODE ISLAND',
        'RHODEI',      'RHODEI.',   'RHODEIS',   'RHODEIS.',
        'RHODEISLAND', 'RI',        'RI.',       'RIS',
        'RIS.',        'RISLAND'
    ],
    'SC' => [
        'S C',            'S C.',
        'S CAR',          'S CAR.',
        'S CAROLINA',     'S. C',
        'S. C.',          'S. CAR',
        'S. CAR.',        'S. CAROLINA',
        'S.C',            'S.C.',
        'S.CAR',          'S.CAR.',
        'S.CAROLINA',     'SC',
        'SC.',            'SCAR',
        'SCAR.',          'SCAROLINA',
        'SOU C',          'SOU C.',
        'SOU CAR',        'SOU CAR.',
        'SOU CAROLINA',   'SOU. C',
        'SOU. C.',        'SOU. CAR',
        'SOU. CAR.',      'SOU. CAROLINA',
        'SOU.C',          'SOU.C.',
        'SOU.CAR',        'SOU.CAR.',
        'SOU.CAROLINA',   'SOUC',
        'SOUC.',          'SOUCAR',
        'SOUCAR.',        'SOUCAROLINA',
        'SOUTH C',        'SOUTH C.',
        'SOUTH CAR',      'SOUTH CAR.',
        'SOUTH CAROLINA', 'SOUTHC',
        'SOUTHC.',        'SOUTHCAR',
        'SOUTHCAR.',      'SOUTHCAROLINA',
        'STH C',          'STH C.',
        'STH CAR',        'STH CAR.',
        'STH CAROLINA',   'STH. C',
        'STH. C.',        'STH. CAR',
        'STH. CAR.',      'STH. CAROLINA',
        'STH.C',          'STH.C.',
        'STH.CAR',        'STH.CAR.',
        'STH.CAROLINA',   'STHC',
        'STHC.',          'STHCAR',
        'STHCAR.',        'STHCAROLINA'
    ],
    'SD' => [
        'S D',          'S D.',        'S DAK',      'S DAK.',
        'S DAKOTA',     'S. D',        'S. D.',      'S. DAK',
        'S. DAK.',      'S. DAKOTA',   'S.D',        'S.D.',
        'S.DAK',        'S.DAK.',      'S.DAKOTA',   'SD',
        'SD.',          'SDAK',        'SDAK.',      'SDAKOTA',
        'SOU D',        'SOU D.',      'SOU DAK',    'SOU DAK.',
        'SOU DAKOTA',   'SOU. D',      'SOU. D.',    'SOU. DAK',
        'SOU. DAK.',    'SOU. DAKOTA', 'SOU.D',      'SOU.D.',
        'SOU.DAK',      'SOU.DAK.',    'SOU.DAKOTA', 'SOUD',
        'SOUD.',        'SOUDAK',      'SOUDAK.',    'SOUDAKOTA',
        'SOUTH D',      'SOUTH D.',    'SOUTH DAK',  'SOUTH DAK.',
        'SOUTH DAKOTA', 'SOUTHD',      'SOUTHD.',    'SOUTHDAK',
        'SOUTHDAK.',    'SOUTHDAKOTA', 'STH D',      'STH D.',
        'STH DAK',      'STH DAK.',    'STH DAKOTA', 'STH. D',
        'STH. D.',      'STH. DAK',    'STH. DAK.',  'STH. DAKOTA',
        'STH.D',        'STH.D.',      'STH.DAK',    'STH.DAK.',
        'STH.DAKOTA',   'STHD',        'STHD.',      'STHDAK',
        'STHDAK.',      'STHDAKOTA'
    ],
    'TN' => [ 'TEN', 'TEN.', 'TENN',  'TENN.', 'TENNESSEE', 'TN', 'TN.' ],
    'TX' => [ 'TEX', 'TEX.', 'TEXAS', 'TX',    'TX.' ],
    'UT' => [ 'UT',  'UT.',  'UTAH' ],
    'VA' => [ 'VA', 'VA.', 'VIR', 'VIR.', 'VIRGINIA' ],
    'VI' => [
        'V IS',
        'V IS.',
        'V ISLAND',
        'V ISLAND.',
        'V ISLANDS',
        'V. IS',
        'V. IS.',
        'V. ISLAND',
        'V. ISLAND.',
        'V. ISLANDS',
        'V.IS',
        'V.IS.',
        'V.ISLAND',
        'V.ISLAND.',
        'V.ISLANDS',
        'VIRGIN I',
        'VIRGIN I.',
        'VIRGIN IS',
        'VIRGIN IS.',
        'VIRGIN ISLAND',
        'VIRGIN ISLAND.',
        'VIRGIN ISLANDS',
        'VIRGINI',
        'VIRGINI.',
        'VIRGINIS',
        'VIRGINIS.',
        'VIRGINISLAND',
        'VIRGINISLAND.',
        'VIRGINISLANDS',
    ],
    'VT' => [
        'VER', 'VER.', 'VERMNT', 'VERMONT', 'VERMT', 'VERMT.', 'VT', 'VT.'
    ],
    'WA' => [ 'WA', 'WA.', 'WASH', 'WASH.',  'WASHINGTON' ],
    'WI' => [ 'WI', 'WI.', 'WIS',  'WISCON', 'WISCONS', 'WISCONSIN' ],
    'WV' => [
        'W V',        'W V.',        'W VA',         'W VA.',
        'W VIR',      'W VIR.',      'W VIRGINIA',   'W. V',
        'W. V.',      'W. VA',       'W. VA.',       'W. VIR',
        'W. VIR.',    'W. VIRGINIA', 'W.V',          'W.V.',
        'W.VA',       'W.VA.',       'W.VIR',        'W.VIR.',
        'W.VIRGINIA', 'WEST V',      'WEST V.',      'WEST VA',
        'WEST VA.',   'WEST VIR',    'WEST VIR.',    'WEST VIRGINIA',
        'WESTV',      'WESTV.',      'WESTVA',       'WESTVA.',
        'WESTVIR',    'WESTVIR.',    'WESTVIRGINIA', 'WV',
        'WV.',        'WVA',         'WVA.',         'WVIR',
        'WVIR.',      'WVIRGINIA'
    ],
    'WY' => [ 'WY', 'WY.', 'WYO', 'WYO.', 'WYOMING' ]
);

foreach my $k ( sort keys %state_or_province_designator ) {
    my @case = ( @{ $state_or_province_designator{$k} } );
    foreach my $test_case ( 0 .. $#case ) {
        my $address = Geo::Address::Mail::US->new(
            name        => 'Test Testerson',
            street      => '23 Testing Dr',
            street2     => 'Unit 1',
            city        => 'Testville',
            state       => $case[$test_case],
            postal_code => '12345'
        );

        my $res  = $std->standardize($address);
        my $corr = $res->standardized_address;
        cmp_ok(
                $res->standardized_address->state,
                'eq', $k,
                sprintf(
                    q{%s-%02d: %s -> %s},
                    $k, $test_case, $case[$test_case], $k,
                )
        );
    }
}

done_testing;

