package Color::Library::Dictionary::NBS_ISCC::A;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::A;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::A - (NBS/ISCC A) Dye Colors

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (A) Dye Colors

    American Association of Textile Chemists and
    Colorists and Society of Dyers and Colourists 

This combinatorial dictionary's 88 names map to 350 colors; 263 out of 267 centroids are covered.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#A>

=head1 COLORS

	black                   black                 #222222

	blue                    blue                  #4997d0

	blue                    blue                  #0067a5

	blue                    blue                  #00416a

	blue                    blue                  #a1caf1

	blue                    blue                  #70a3cc

	blue                    blue                  #436b95

	blue                    blue                  #00304e

	blue                    blue                  #b3bce2

	blue                    blue                  #8791bf

	blue                    blue                  #252440

	bluish black            bluishblack           #202428

	bluish gray             bluishgray            #b4bcc0

	bluish gray             bluishgray            #81878b

	bluish gray             bluishgray            #51585e

	bluish green            bluishgreen           #00a693

	bluish green            bluishgreen           #007a74

	bluish green            bluishgreen           #00443f

	bluish green            bluishgreen           #96ded1

	bluish green            bluishgreen           #66ada4

	bluish green            bluishgreen           #317873

	bluish green            bluishgreen           #004b49

	bluish green            bluishgreen           #002a29

	bluish olive            bluisholive           #404f00

	bluish olive            bluisholive           #232f00

	bluish olive            bluisholive           #4a5d23

	bluish olive            bluisholive           #2b3d26

	bluish olive            bluisholive           #515744

	bluish olive            bluisholive           #31362b

	bluish pink             bluishpink            #ffc8d6

	bluish pink             bluishpink            #e68fac

	bluish pink             bluishpink            #de6fa1

	bluish pink             bluishpink            #efbbcc

	bluish pink             bluishpink            #d597ae

	bluish pink             bluishpink            #c17e91

	bluish pink             bluishpink            #e8ccd7

	bluish pink             bluishpink            #c3a6b1

	bluish pink             bluishpink            #af868e

	bluish red              bluishred             #b3446c

	bluish red              bluishred             #78184a

	bluish red              bluishred             #54133b

	bluish red              bluishred             #a8516e

	bluish violet           bluishviolet          #7e73b8

	bluish violet           bluishviolet          #604e97

	bluish violet           bluishviolet          #32174d

	bluish violet           bluishviolet          #dcd0ff

	bluish violet           bluishviolet          #8c82b6

	bluish violet           bluishviolet          #604e81

	bluish violet           bluishviolet          #2f2140

	bluish white            bluishwhite           #e9e9ed

	bordeaux                bordeaux              #673147

	bordeaux                bordeaux              #38152c

	bordeaux                bordeaux              #915f6d

	bright blue             brightblue            #00a1c2

	bright bluish green     brightbluishgreen     #008882

	bright bluish red       brightbluishred       #ce4676

	bright bluish violet    brightbluishviolet    #9065ca

	bright green            brightgreen           #008856

	bright greenish blue    brightgreenishblue    #0085a1

	bright greenish yellow  brightgreenishyellow  #dcd300

	bright orange           brightorange          #f38400

	bright red              brightred             #be0032

	bright reddish blue     brightreddishblue     #30267a

	bright reddish orange   brightreddishorange   #e25822

	bright reddish violet   brightreddishviolet   #870074

	bright reddish yellow   brightreddishyellow   #f6a600

	bright reddish yellow   brightreddishyellow   #f3c300

	bright violet           brightviolet          #9a4eae

	bright yellow           brightyellow          #f3c300

	bright yellowish green  brightyellowishgreen  #8db600

	bright yellowish green  brightyellowishgreen  #27a64c

	bright yellowish orange brightyellowishorange #f6a600

	bright yellowish red    brightyellowishred    #be0032

	bright yellowish red    brightyellowishred    #e25822

	brown                   brown                 #80461b

	brown                   brown                 #593319

	brown                   brown                 #a67b5b

	brown                   brown                 #6f4e37

	brown                   brown                 #422518

	brownish black          brownishblack         #28201c

	brownish gray           brownishgray          #8e8279

	brownish gray           brownishgray          #5b504f

	brownish gray           brownishgray          #bfb8a5

	brownish olive          brownisholive         #967117

	brownish olive          brownisholive         #6c541e

	brownish white          brownishwhite         #f0ead6

	dull blue               dullblue              #bcd4e6

	dull blue               dullblue              #91a3b0

	dull blue               dullblue              #536878

	dull blue               dullblue              #36454f

	dull blue               dullblue              #202830

	dull blue               dullblue              #c0c8e1

	dull blue               dullblue              #8c92ac

	dull blue               dullblue              #4c516d

	dull bluish green       dullbluishgreen       #c7e6d7

	dull bluish green       dullbluishgreen       #8da399

	dull bluish green       dullbluishgreen       #5e716a

	dull bluish green       dullbluishgreen       #3a4b47

	dull bluish green       dullbluishgreen       #1a2421

	dull bluish red         dullbluishred         #a8516e

	dull bluish red         dullbluishred         #673147

	dull bluish red         dullbluishred         #38152c

	dull bluish violet      dullbluishviolet      #c4c3dd

	dull bluish violet      dullbluishviolet      #9690ab

	dull bluish violet      dullbluishviolet      #554c69

	dull bluish violet      dullbluishviolet      #d6cadd

	dull bluish violet      dullbluishviolet      #aa98a9

	dull bluish violet      dullbluishviolet      #796878

	dull bluish violet      dullbluishviolet      #50404d

	dull bluish violet      dullbluishviolet      #291e29

	dull bordeaux           dullbordeaux          #796878

	dull bordeaux           dullbordeaux          #50404d

	dull bordeaux           dullbordeaux          #291e29

	dull bordeaux           dullbordeaux          #673147

	dull bordeaux           dullbordeaux          #38152c

	dull brownish olive     dullbrownisholive     #3b3121

	dull green              dullgreen             #c7e6d7

	dull green              dullgreen             #8da399

	dull green              dullgreen             #5e716a

	dull green              dullgreen             #3a4b47

	dull green              dullgreen             #1a2421

	dull greenish blue      dullgreenishblue      #bcd4e6

	dull greenish blue      dullgreenishblue      #91a3b0

	dull greenish blue      dullgreenishblue      #536878

	dull greenish blue      dullgreenishblue      #36454f

	dull greenish blue      dullgreenishblue      #202830

	dull greenish yellow    dullgreenishyellow    #ebe8a4

	dull greenish yellow    dullgreenishyellow    #b9b57d

	dull greenish yellow    dullgreenishyellow    #867e36

	dull orange             dullorange            #d9a6a9

	dull orange             dullorange            #a67b5b

	dull red                dullred               #ab4e52

	dull red                dullred               #722f37

	dull red                dullred               #3f1728

	dull red                dullred               #905d5d

	dull red                dullred               #56070c

	dull red                dullred               #79443b

	dull reddish blue       dullreddishblue       #bcd4e6

	dull reddish blue       dullreddishblue       #91a3b0

	dull reddish blue       dullreddishblue       #536878

	dull reddish blue       dullreddishblue       #36454f

	dull reddish blue       dullreddishblue       #202830

	dull reddish blue       dullreddishblue       #c0c8e1

	dull reddish blue       dullreddishblue       #8c92ac

	dull reddish blue       dullreddishblue       #4c516d

	dull reddish brown      dullreddishbrown      #722f37

	dull reddish brown      dullreddishbrown      #3f1728

	dull reddish brown      dullreddishbrown      #905d5d

	dull reddish brown      dullreddishbrown      #543d3f

	dull reddish brown      dullreddishbrown      #2e1d21

	dull reddish brown      dullreddishbrown      #a87c6d

	dull reddish brown      dullreddishbrown      #79443b

	dull reddish brown      dullreddishbrown      #3e1d1e

	dull reddish brown      dullreddishbrown      #977f73

	dull reddish brown      dullreddishbrown      #674c47

	dull reddish brown      dullreddishbrown      #43302e

	dull reddish orange     dullreddishorange     #b4745e

	dull reddish violet     dullreddishviolet     #aa98a9

	dull reddish violet     dullreddishviolet     #796878

	dull reddish violet     dullreddishviolet     #50404d

	dull reddish violet     dullreddishviolet     #291e29

	dull reddish violet     dullreddishviolet     #aa8a9e

	dull reddish violet     dullreddishviolet     #836479

	dull reddish yellow     dullreddishyellow     #fad6a5

	dull reddish yellow     dullreddishyellow     #f3e5ab

	dull reddish yellow     dullreddishyellow     #c2b280

	dull reddish yellow     dullreddishyellow     #a18f60

	dull violet             dullviolet            #d6cadd

	dull violet             dullviolet            #aa98a9

	dull violet             dullviolet            #796878

	dull violet             dullviolet            #50404d

	dull violet             dullviolet            #291e29

	dull yellow             dullyellow            #f3e5ab

	dull yellow             dullyellow            #c2b280

	dull yellow             dullyellow            #a18f60

	dull yellowish brown    dullyellowishbrown    #c19a6b

	dull yellowish brown    dullyellowishbrown    #826644

	dull yellowish brown    dullyellowishbrown    #4b3621

	dull yellowish brown    dullyellowishbrown    #ae9b82

	dull yellowish brown    dullyellowishbrown    #7e6d5a

	dull yellowish brown    dullyellowishbrown    #483c32

	dull yellowish green    dullyellowishgreen    #dadfb7

	dull yellowish green    dullyellowishgreen    #8f9779

	dull yellowish orange   dullyellowishorange   #fad6a5

	dull yellowish red      dullyellowishred      #b4745e

	gray                    gray                  #b9b8b5

	gray                    gray                  #848482

	gray                    gray                  #555555

	green                   green                 #3eb489

	green                   green                 #007959

	green                   green                 #00543d

	green                   green                 #8ed1b2

	green                   green                 #6aab8e

	green                   green                 #3b7861

	green                   green                 #1b4d3e

	green                   green                 #1c352d

	greenish black          greenishblack         #25241d

	greenish black          greenishblack         #1e2321

	greenish blue           greenishblue          #239eba

	greenish blue           greenishblue          #007791

	greenish blue           greenishblue          #2e8495

	greenish blue           greenishblue          #9cd1dc

	greenish blue           greenishblue          #66aabc

	greenish blue           greenishblue          #367588

	greenish blue           greenishblue          #004958

	greenish blue           greenishblue          #002e3b

	greenish gray           greenishgray          #8a8776

	greenish gray           greenishgray          #57554c

	greenish gray           greenishgray          #b2beb5

	greenish gray           greenishgray          #7d8984

	greenish gray           greenishgray          #4e5755

	greenish navy           greenishnavy          #00304e

	greenish navy           greenishnavy          #202830

	greenish white          greenishwhite         #dfede8

	greenish yellow         greenishyellow        #e9e450

	greenish yellow         greenishyellow        #beb72e

	greenish yellow         greenishyellow        #9b9400

	greenish yellow         greenishyellow        #eae679

	greenish yellow         greenishyellow        #b9b459

	greenish yellow         greenishyellow        #98943e

	navy                    navy                  #00304e

	navy                    navy                  #202830

	navy                    navy                  #252440

	olive                   olive                 #867e36

	olive                   olive                 #665d1e

	olive                   olive                 #403d21

	olive                   olive                 #8c8767

	olive                   olive                 #5b5842

	olive                   olive                 #363527

	orange                  orange                #fd943f

	orange                  orange                #ed872d

	orange                  orange                #be6516

	orange                  orange                #fab57f

	orange                  orange                #d99058

	orange                  orange                #ae6938

	pink                    pink                  #ffb5ba

	pink                    pink                  #ea9399

	pink                    pink                  #e4717a

	pink                    pink                  #f9ccca

	pink                    pink                  #dea5a4

	pink                    pink                  #c08081

	pink                    pink                  #ead8d7

	pink                    pink                  #c4aead

	pink                    pink                  #ad8884

	pink                    pink                  #af868e

	red                     red                   #bc3f4a

	red                     red                   #841b2d

	red                     red                   #5c0923

	red                     red                   #ab4e52

	reddish black           reddishblack          #543d3f

	reddish black           reddishblack          #282022

	reddish black           reddishblack          #242124

	reddish blue            reddishblue           #6c79b8

	reddish blue            reddishblue           #545aa7

	reddish blue            reddishblue           #272458

	reddish blue            reddishblue           #b3bce2

	reddish blue            reddishblue           #8791bf

	reddish blue            reddishblue           #4e5180

	reddish blue            reddishblue           #252440

	reddish brown           reddishbrown          #841b2d

	reddish brown           reddishbrown          #5c0923

	reddish brown           reddishbrown          #722f37

	reddish brown           reddishbrown          #905d5d

	reddish brown           reddishbrown          #882d17

	reddish brown           reddishbrown          #56070c

	reddish brown           reddishbrown          #79443b

	reddish gray            reddishgray           #c1b6b3

	reddish gray            reddishgray           #8f817f

	reddish gray            reddishgray           #5c504f

	reddish gray            reddishgray           #bfb9bd

	reddish gray            reddishgray           #8b8589

	reddish gray            reddishgray           #5d555b

	reddish navy            reddishnavy           #2f2140

	reddish orange          reddishorange         #d9603b

	reddish orange          reddishorange         #aa381e

	reddish orange          reddishorange         #cb6d51

	reddish orange          reddishorange         #9e4732

	reddish violet          reddishviolet         #9e4f88

	reddish violet          reddishviolet         #702963

	reddish violet          reddishviolet         #54194e

	reddish violet          reddishviolet         #b784a7

	reddish violet          reddishviolet         #915c83

	reddish violet          reddishviolet         #5d3954

	reddish violet          reddishviolet         #341731

	reddish white           reddishwhite          #eae3e1

	reddish white           reddishwhite          #e8e3e5

	reddish yellow          reddishyellow         #ffc14f

	reddish yellow          reddishyellow         #eaa221

	reddish yellow          reddishyellow         #c98500

	reddish yellow          reddishyellow         #fbc97f

	reddish yellow          reddishyellow         #e3a857

	reddish yellow          reddishyellow         #be8a3d

	reddish yellow          reddishyellow         #fada5e

	reddish yellow          reddishyellow         #d4af37

	reddish yellow          reddishyellow         #af8d13

	reddish yellow          reddishyellow         #f8de7e

	reddish yellow          reddishyellow         #c9ae5d

	reddish yellow          reddishyellow         #ab9144

	violet                  violet                #d399e6

	violet                  violet                #875692

	violet                  violet                #602f6b

	violet                  violet                #401a4c

	violet                  violet                #d5badb

	violet                  violet                #b695c0

	violet                  violet                #86608e

	violet                  violet                #563c5c

	violet                  violet                #301934

	white                   white                 #f2f3f4

	yellow                  yellow                #fada5e

	yellow                  yellow                #d4af37

	yellow                  yellow                #af8d13

	yellow                  yellow                #f8de7e

	yellow                  yellow                #c9ae5d

	yellow                  yellow                #ab9144

	yellowish brown         yellowishbrown        #996515

	yellowish brown         yellowishbrown        #654522

	yellowish green         yellowishgreen        #bdda57

	yellowish green         yellowishgreen        #7e9f2e

	yellowish green         yellowishgreen        #467129

	yellowish green         yellowishgreen        #c9dc89

	yellowish green         yellowishgreen        #8a9a5b

	yellowish green         yellowishgreen        #4a5d23

	yellowish green         yellowishgreen        #83d37d

	yellowish green         yellowishgreen        #44944a

	yellowish green         yellowishgreen        #00622d

	yellowish green         yellowishgreen        #003118

	yellowish green         yellowishgreen        #b6e5af

	yellowish green         yellowishgreen        #93c592

	yellowish green         yellowishgreen        #679267

	yellowish green         yellowishgreen        #355e3b

	yellowish green         yellowishgreen        #173620

	yellowish orange        yellowishorange       #ffc14f

	yellowish orange        yellowishorange       #eaa221

	yellowish orange        yellowishorange       #c98500

	yellowish orange        yellowishorange       #fbc97f

	yellowish orange        yellowishorange       #e3a857

	yellowish orange        yellowishorange       #be8a3d

	yellowish pink          yellowishpink         #ffb7a5

	yellowish pink          yellowishpink         #f88379

	yellowish pink          yellowishpink         #e66761

	yellowish pink          yellowishpink         #f4c2c2

	yellowish pink          yellowishpink         #d9a6a9

	yellowish pink          yellowishpink         #c48379

	yellowish pink          yellowishpink         #ecd5c5

	yellowish pink          yellowishpink         #c7ada3

	yellowish red           yellowishred          #d9603b

	yellowish red           yellowishred          #aa381e

	yellowish red           yellowishred          #cb6d51

	yellowish red           yellowishred          #9e4732


=cut

sub _load_color_list() {
    return [
['nbs-iscc-a:black.267','black','black',[34,34,34],'222222',2236962],
['nbs-iscc-a:blue.177','blue','blue',[73,151,208],'4997d0',4822992],
['nbs-iscc-a:blue.178','blue','blue',[0,103,165],'0067a5',26533],
['nbs-iscc-a:blue.179','blue','blue',[0,65,106],'00416a',16746],
['nbs-iscc-a:blue.180','blue','blue',[161,202,241],'a1caf1',10603249],
['nbs-iscc-a:blue.181','blue','blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc-a:blue.182','blue','blue',[67,107,149],'436b95',4418453],
['nbs-iscc-a:blue.183','blue','blue',[0,48,78],'00304e',12366],
['nbs-iscc-a:blue.198','blue','blue',[179,188,226],'b3bce2',11779298],
['nbs-iscc-a:blue.199','blue','blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-a:blue.201','blue','blue',[37,36,64],'252440',2434112],
['nbs-iscc-a:bluishblack.193','bluishblack','bluish black',[32,36,40],'202428',2106408],
['nbs-iscc-a:bluishgray.190','bluishgray','bluish gray',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-a:bluishgray.191','bluishgray','bluish gray',[129,135,139],'81878b',8488843],
['nbs-iscc-a:bluishgray.192','bluishgray','bluish gray',[81,88,94],'51585e',5331038],
['nbs-iscc-a:bluishgreen.159','bluishgreen','bluish green',[0,166,147],'00a693',42643],
['nbs-iscc-a:bluishgreen.160','bluishgreen','bluish green',[0,122,116],'007a74',31348],
['nbs-iscc-a:bluishgreen.161','bluishgreen','bluish green',[0,68,63],'00443f',17471],
['nbs-iscc-a:bluishgreen.162','bluishgreen','bluish green',[150,222,209],'96ded1',9887441],
['nbs-iscc-a:bluishgreen.163','bluishgreen','bluish green',[102,173,164],'66ada4',6729124],
['nbs-iscc-a:bluishgreen.164','bluishgreen','bluish green',[49,120,115],'317873',3242099],
['nbs-iscc-a:bluishgreen.165','bluishgreen','bluish green',[0,75,73],'004b49',19273],
['nbs-iscc-a:bluishgreen.166','bluishgreen','bluish green',[0,42,41],'002a29',10793],
['nbs-iscc-a:bluisholive.123','bluisholive','bluish olive',[64,79,0],'404f00',4214528],
['nbs-iscc-a:bluisholive.124','bluisholive','bluish olive',[35,47,0],'232f00',2305792],
['nbs-iscc-a:bluisholive.125','bluisholive','bluish olive',[74,93,35],'4a5d23',4873507],
['nbs-iscc-a:bluisholive.126','bluisholive','bluish olive',[43,61,38],'2b3d26',2833702],
['nbs-iscc-a:bluisholive.127','bluisholive','bluish olive',[81,87,68],'515744',5330756],
['nbs-iscc-a:bluisholive.128','bluisholive','bluish olive',[49,54,43],'31362b',3225131],
['nbs-iscc-a:bluishpink.246','bluishpink','bluish pink',[255,200,214],'ffc8d6',16763094],
['nbs-iscc-a:bluishpink.247','bluishpink','bluish pink',[230,143,172],'e68fac',15110060],
['nbs-iscc-a:bluishpink.248','bluishpink','bluish pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc-a:bluishpink.249','bluishpink','bluish pink',[239,187,204],'efbbcc',15711180],
['nbs-iscc-a:bluishpink.250','bluishpink','bluish pink',[213,151,174],'d597ae',13997998],
['nbs-iscc-a:bluishpink.251','bluishpink','bluish pink',[193,126,145],'c17e91',12680849],
['nbs-iscc-a:bluishpink.252','bluishpink','bluish pink',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-a:bluishpink.253','bluishpink','bluish pink',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-a:bluishpink.261','bluishpink','bluish pink',[175,134,142],'af868e',11503246],
['nbs-iscc-a:bluishred.255','bluishred','bluish red',[179,68,108],'b3446c',11748460],
['nbs-iscc-a:bluishred.256','bluishred','bluish red',[120,24,74],'78184a',7870538],
['nbs-iscc-a:bluishred.257','bluishred','bluish red',[84,19,59],'54133b',5509947],
['nbs-iscc-a:bluishred.258','bluishred','bluish red',[168,81,110],'a8516e',11030894],
['nbs-iscc-a:bluishviolet.206','bluishviolet','bluish violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc-a:bluishviolet.207','bluishviolet','bluish violet',[96,78,151],'604e97',6311575],
['nbs-iscc-a:bluishviolet.208','bluishviolet','bluish violet',[50,23,77],'32174d',3282765],
['nbs-iscc-a:bluishviolet.209','bluishviolet','bluish violet',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-a:bluishviolet.210','bluishviolet','bluish violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc-a:bluishviolet.211','bluishviolet','bluish violet',[96,78,129],'604e81',6311553],
['nbs-iscc-a:bluishviolet.212','bluishviolet','bluish violet',[47,33,64],'2f2140',3088704],
['nbs-iscc-a:bluishwhite.189','bluishwhite','bluish white',[233,233,237],'e9e9ed',15329773],
['nbs-iscc-a:bordeaux.259','bordeaux','bordeaux',[103,49,71],'673147',6762823],
['nbs-iscc-a:bordeaux.260','bordeaux','bordeaux',[56,21,44],'38152c',3675436],
['nbs-iscc-a:bordeaux.262','bordeaux','bordeaux',[145,95,109],'915f6d',9527149],
['nbs-iscc-a:brightblue.176','brightblue','bright blue',[0,161,194],'00a1c2',41410],
['nbs-iscc-a:brightbluishgreen.158','brightbluishgreen','bright bluish green',[0,136,130],'008882',34946],
['nbs-iscc-a:brightbluishred.254','brightbluishred','bright bluish red ',[206,70,118],'ce4676',13518454],
['nbs-iscc-a:brightbluishviolet.205','brightbluishviolet','bright bluish violet',[144,101,202],'9065ca',9463242],
['nbs-iscc-a:brightgreen.139','brightgreen','bright green',[0,136,86],'008856',34902],
['nbs-iscc-a:brightgreenishblue.167','brightgreenishblue','bright greenish blue',[0,133,161],'0085a1',34209],
['nbs-iscc-a:brightgreenishyellow.97','brightgreenishyellow','bright greenish yellow',[220,211,0],'dcd300',14471936],
['nbs-iscc-a:brightorange.48','brightorange','bright orange',[243,132,0],'f38400',15959040],
['nbs-iscc-a:brightred.11','brightred','bright red',[190,0,50],'be0032',12451890],
['nbs-iscc-a:brightreddishblue.194','brightreddishblue','bright reddish blue',[48,38,122],'30267a',3155578],
['nbs-iscc-a:brightreddishorange.34','brightreddishorange','bright reddish orange',[226,88,34],'e25822',14833698],
['nbs-iscc-a:brightreddishviolet.236','brightreddishviolet','bright reddish violet',[135,0,116],'870074',8847476],
['nbs-iscc-a:brightreddishyellow.66','brightreddishyellow','bright reddish yellow',[246,166,0],'f6a600',16164352],
['nbs-iscc-a:brightreddishyellow.82','brightreddishyellow','bright reddish yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-a:brightviolet.216','brightviolet','bright violet',[154,78,174],'9a4eae',10112686],
['nbs-iscc-a:brightyellow.82','brightyellow','bright yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc-a:brightyellowishgreen.115','brightyellowishgreen','bright yellowish green',[141,182,0],'8db600',9287168],
['nbs-iscc-a:brightyellowishgreen.129','brightyellowishgreen','bright yellowish green',[39,166,76],'27a64c',2598476],
['nbs-iscc-a:brightyellowishorange.66','brightyellowishorange','bright yellowish orange',[246,166,0],'f6a600',16164352],
['nbs-iscc-a:brightyellowishred.11','brightyellowishred','bright yellowish red',[190,0,50],'be0032',12451890],
['nbs-iscc-a:brightyellowishred.34','brightyellowishred','bright yellowish red',[226,88,34],'e25822',14833698],
['nbs-iscc-a:brown.55','brown','brown',[128,70,27],'80461b',8406555],
['nbs-iscc-a:brown.56','brown','brown',[89,51,25],'593319',5845785],
['nbs-iscc-a:brown.57','brown','brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc-a:brown.58','brown','brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc-a:brown.59','brown','brown',[66,37,24],'422518',4334872],
['nbs-iscc-a:brownishblack.65','brownishblack','brownish black',[40,32,28],'28201c',2629660],
['nbs-iscc-a:brownishgray.63','brownishgray','brownish gray',[142,130,121],'8e8279',9339513],
['nbs-iscc-a:brownishgray.64','brownishgray','brownish gray',[91,80,79],'5b504f',5984335],
['nbs-iscc-a:brownishgray.93','brownishgray','brownish gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-a:brownisholive.94','brownisholive','brownish olive',[150,113,23],'967117',9859351],
['nbs-iscc-a:brownisholive.95','brownisholive','brownish olive',[108,84,30],'6c541e',7099422],
['nbs-iscc-a:brownishwhite.92','brownishwhite','brownish white',[240,234,214],'f0ead6',15788758],
['nbs-iscc-a:dullblue.184','dullblue','dull blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-a:dullblue.185','dullblue','dull blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-a:dullblue.186','dullblue','dull blue',[83,104,120],'536878',5466232],
['nbs-iscc-a:dullblue.187','dullblue','dull blue',[54,69,79],'36454f',3556687],
['nbs-iscc-a:dullblue.188','dullblue','dull blue',[32,40,48],'202830',2107440],
['nbs-iscc-a:dullblue.202','dullblue','dull blue',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-a:dullblue.203','dullblue','dull blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-a:dullblue.204','dullblue','dull blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-a:dullbluishgreen.148','dullbluishgreen','dull bluish green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-a:dullbluishgreen.149','dullbluishgreen','dull bluish green',[141,163,153],'8da399',9282457],
['nbs-iscc-a:dullbluishgreen.150','dullbluishgreen','dull bluish green',[94,113,106],'5e716a',6189418],
['nbs-iscc-a:dullbluishgreen.151','dullbluishgreen','dull bluish green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-a:dullbluishgreen.152','dullbluishgreen','dull bluish green',[26,36,33],'1a2421',1713185],
['nbs-iscc-a:dullbluishred.258','dullbluishred','dull bluish red',[168,81,110],'a8516e',11030894],
['nbs-iscc-a:dullbluishred.259','dullbluishred','dull bluish red',[103,49,71],'673147',6762823],
['nbs-iscc-a:dullbluishred.260','dullbluishred','dull bluish red',[56,21,44],'38152c',3675436],
['nbs-iscc-a:dullbluishviolet.213','dullbluishviolet','dull bluish violet',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-a:dullbluishviolet.214','dullbluishviolet','dull bluish violet',[150,144,171],'9690ab',9867435],
['nbs-iscc-a:dullbluishviolet.215','dullbluishviolet','dull bluish violet',[85,76,105],'554c69',5590121],
['nbs-iscc-a:dullbluishviolet.226','dullbluishviolet','dull bluish violet',[214,202,221],'d6cadd',14076637],
['nbs-iscc-a:dullbluishviolet.227','dullbluishviolet','dull bluish violet',[170,152,169],'aa98a9',11180201],
['nbs-iscc-a:dullbluishviolet.228','dullbluishviolet','dull bluish violet',[121,104,120],'796878',7956600],
['nbs-iscc-a:dullbluishviolet.229','dullbluishviolet','dull bluish violet',[80,64,77],'50404d',5259341],
['nbs-iscc-a:dullbluishviolet.230','dullbluishviolet','dull bluish violet',[41,30,41],'291e29',2694697],
['nbs-iscc-a:dullbordeaux.228','dullbordeaux','dull bordeaux',[121,104,120],'796878',7956600],
['nbs-iscc-a:dullbordeaux.229','dullbordeaux','dull bordeaux',[80,64,77],'50404d',5259341],
['nbs-iscc-a:dullbordeaux.230','dullbordeaux','dull bordeaux',[41,30,41],'291e29',2694697],
['nbs-iscc-a:dullbordeaux.259','dullbordeaux','dull bordeaux',[103,49,71],'673147',6762823],
['nbs-iscc-a:dullbordeaux.260','dullbordeaux','dull bordeaux',[56,21,44],'38152c',3675436],
['nbs-iscc-a:dullbrownisholive.96','dullbrownisholive','dull brownish olive',[59,49,33],'3b3121',3879201],
['nbs-iscc-a:dullgreen.148','dullgreen','dull green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-a:dullgreen.149','dullgreen','dull green',[141,163,153],'8da399',9282457],
['nbs-iscc-a:dullgreen.150','dullgreen','dull green',[94,113,106],'5e716a',6189418],
['nbs-iscc-a:dullgreen.151','dullgreen','dull green',[58,75,71],'3a4b47',3820359],
['nbs-iscc-a:dullgreen.152','dullgreen','dull green',[26,36,33],'1a2421',1713185],
['nbs-iscc-a:dullgreenishblue.184','dullgreenishblue','dull greenish blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-a:dullgreenishblue.185','dullgreenishblue','dull greenish blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-a:dullgreenishblue.186','dullgreenishblue','dull greenish blue',[83,104,120],'536878',5466232],
['nbs-iscc-a:dullgreenishblue.187','dullgreenishblue','dull greenish blue',[54,69,79],'36454f',3556687],
['nbs-iscc-a:dullgreenishblue.188','dullgreenishblue','dull greenish blue',[32,40,48],'202830',2107440],
['nbs-iscc-a:dullgreenishyellow.104','dullgreenishyellow','dull greenish yellow',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-a:dullgreenishyellow.105','dullgreenishyellow','dull greenish yellow',[185,181,125],'b9b57d',12170621],
['nbs-iscc-a:dullgreenishyellow.106','dullgreenishyellow','dull greenish yellow',[134,126,54],'867e36',8814134],
['nbs-iscc-a:dullorange.29','dullorange','dull orange',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-a:dullorange.57','dullorange','dull orange',[166,123,91],'a67b5b',10910555],
['nbs-iscc-a:dullred.15','dullred','dull red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-a:dullred.16','dullred','dull red',[114,47,55],'722f37',7483191],
['nbs-iscc-a:dullred.17','dullred','dull red',[63,23,40],'3f1728',4134696],
['nbs-iscc-a:dullred.19','dullred','dull red',[144,93,93],'905d5d',9461085],
['nbs-iscc-a:dullred.41','dullred','dull red',[86,7,12],'56070c',5637900],
['nbs-iscc-a:dullred.43','dullred','dull red',[121,68,59],'79443b',7947323],
['nbs-iscc-a:dullreddishblue.184','dullreddishblue','dull reddish blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-a:dullreddishblue.185','dullreddishblue','dull reddish blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc-a:dullreddishblue.186','dullreddishblue','dull reddish blue',[83,104,120],'536878',5466232],
['nbs-iscc-a:dullreddishblue.187','dullreddishblue','dull reddish blue',[54,69,79],'36454f',3556687],
['nbs-iscc-a:dullreddishblue.188','dullreddishblue','dull reddish blue',[32,40,48],'202830',2107440],
['nbs-iscc-a:dullreddishblue.202','dullreddishblue','dull reddish blue',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-a:dullreddishblue.203','dullreddishblue','dull reddish blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc-a:dullreddishblue.204','dullreddishblue','dull reddish blue',[76,81,109],'4c516d',5001581],
['nbs-iscc-a:dullreddishbrown.16','dullreddishbrown','dull reddish brown',[114,47,55],'722f37',7483191],
['nbs-iscc-a:dullreddishbrown.17','dullreddishbrown','dull reddish brown',[63,23,40],'3f1728',4134696],
['nbs-iscc-a:dullreddishbrown.19','dullreddishbrown','dull reddish brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-a:dullreddishbrown.20','dullreddishbrown','dull reddish brown',[84,61,63],'543d3f',5520703],
['nbs-iscc-a:dullreddishbrown.21','dullreddishbrown','dull reddish brown',[46,29,33],'2e1d21',3022113],
['nbs-iscc-a:dullreddishbrown.42','dullreddishbrown','dull reddish brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc-a:dullreddishbrown.43','dullreddishbrown','dull reddish brown',[121,68,59],'79443b',7947323],
['nbs-iscc-a:dullreddishbrown.44','dullreddishbrown','dull reddish brown',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-a:dullreddishbrown.45','dullreddishbrown','dull reddish brown',[151,127,115],'977f73',9928563],
['nbs-iscc-a:dullreddishbrown.46','dullreddishbrown','dull reddish brown',[103,76,71],'674c47',6769735],
['nbs-iscc-a:dullreddishbrown.47','dullreddishbrown','dull reddish brown',[67,48,46],'43302e',4403246],
['nbs-iscc-a:dullreddishorange.39','dullreddishorange','dull reddish orange',[180,116,94],'b4745e',11826270],
['nbs-iscc-a:dullreddishviolet.227','dullreddishviolet','dull reddish violet',[170,152,169],'aa98a9',11180201],
['nbs-iscc-a:dullreddishviolet.228','dullreddishviolet','dull reddish violet',[121,104,120],'796878',7956600],
['nbs-iscc-a:dullreddishviolet.229','dullreddishviolet','dull reddish violet',[80,64,77],'50404d',5259341],
['nbs-iscc-a:dullreddishviolet.230','dullreddishviolet','dull reddish violet',[41,30,41],'291e29',2694697],
['nbs-iscc-a:dullreddishviolet.244','dullreddishviolet','dull reddish violet',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-a:dullreddishviolet.245','dullreddishviolet','dull reddish violet',[131,100,121],'836479',8610937],
['nbs-iscc-a:dullreddishyellow.73','dullreddishyellow','dull reddish yellow',[250,214,165],'fad6a5',16438949],
['nbs-iscc-a:dullreddishyellow.89','dullreddishyellow','dull reddish yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-a:dullreddishyellow.90','dullreddishyellow','dull reddish yellow',[194,178,128],'c2b280',12759680],
['nbs-iscc-a:dullreddishyellow.91','dullreddishyellow','dull reddish yellow',[161,143,96],'a18f60',10588000],
['nbs-iscc-a:dullviolet.226','dullviolet','dull violet',[214,202,221],'d6cadd',14076637],
['nbs-iscc-a:dullviolet.227','dullviolet','dull violet',[170,152,169],'aa98a9',11180201],
['nbs-iscc-a:dullviolet.228','dullviolet','dull violet',[121,104,120],'796878',7956600],
['nbs-iscc-a:dullviolet.229','dullviolet','dull violet',[80,64,77],'50404d',5259341],
['nbs-iscc-a:dullviolet.230','dullviolet','dull violet',[41,30,41],'291e29',2694697],
['nbs-iscc-a:dullyellow.89','dullyellow','dull yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-a:dullyellow.90','dullyellow','dull yellow',[194,178,128],'c2b280',12759680],
['nbs-iscc-a:dullyellow.91','dullyellow','dull yellow',[161,143,96],'a18f60',10588000],
['nbs-iscc-a:dullyellowishbrown.76','dullyellowishbrown','dull yellowish brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc-a:dullyellowishbrown.77','dullyellowishbrown','dull yellowish brown',[130,102,68],'826644',8545860],
['nbs-iscc-a:dullyellowishbrown.78','dullyellowishbrown','dull yellowish brown',[75,54,33],'4b3621',4929057],
['nbs-iscc-a:dullyellowishbrown.79','dullyellowishbrown','dull yellowish brown',[174,155,130],'ae9b82',11443074],
['nbs-iscc-a:dullyellowishbrown.80','dullyellowishbrown','dull yellowish brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-a:dullyellowishbrown.81','dullyellowishbrown','dull yellowish brown',[72,60,50],'483c32',4734002],
['nbs-iscc-a:dullyellowishgreen.121','dullyellowishgreen','dull yellowish green',[218,223,183],'dadfb7',14344119],
['nbs-iscc-a:dullyellowishgreen.122','dullyellowishgreen','dull yellowish green',[143,151,121],'8f9779',9410425],
['nbs-iscc-a:dullyellowishorange.73','dullyellowishorange','dull yellowish orange',[250,214,165],'fad6a5',16438949],
['nbs-iscc-a:dullyellowishred.39','dullyellowishred','dull yellowish red',[180,116,94],'b4745e',11826270],
['nbs-iscc-a:gray.264','gray','gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-a:gray.265','gray','gray',[132,132,130],'848482',8684674],
['nbs-iscc-a:gray.266','gray','gray',[85,85,85],'555555',5592405],
['nbs-iscc-a:green.140','green','green',[62,180,137],'3eb489',4109449],
['nbs-iscc-a:green.141','green','green',[0,121,89],'007959',31065],
['nbs-iscc-a:green.142','green','green',[0,84,61],'00543d',21565],
['nbs-iscc-a:green.143','green','green',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-a:green.144','green','green',[106,171,142],'6aab8e',6990734],
['nbs-iscc-a:green.145','green','green',[59,120,97],'3b7861',3897441],
['nbs-iscc-a:green.146','green','green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-a:green.147','green','green',[28,53,45],'1c352d',1848621],
['nbs-iscc-a:greenishblack.114','greenishblack','greenish black',[37,36,29],'25241d',2434077],
['nbs-iscc-a:greenishblack.157','greenishblack','greenish black',[30,35,33],'1e2321',1975073],
['nbs-iscc-a:greenishblue.168','greenishblue','greenish blue',[35,158,186],'239eba',2334394],
['nbs-iscc-a:greenishblue.169','greenishblue','greenish blue',[0,119,145],'007791',30609],
['nbs-iscc-a:greenishblue.170','greenishblue','greenish blue',[46,132,149],'2e8495',3048597],
['nbs-iscc-a:greenishblue.171','greenishblue','greenish blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-a:greenishblue.172','greenishblue','greenish blue',[102,170,188],'66aabc',6728380],
['nbs-iscc-a:greenishblue.173','greenishblue','greenish blue',[54,117,136],'367588',3569032],
['nbs-iscc-a:greenishblue.174','greenishblue','greenish blue',[0,73,88],'004958',18776],
['nbs-iscc-a:greenishblue.175','greenishblue','greenish blue',[0,46,59],'002e3b',11835],
['nbs-iscc-a:greenishgray.112','greenishgray','greenish gray',[138,135,118],'8a8776',9078646],
['nbs-iscc-a:greenishgray.113','greenishgray','greenish gray',[87,85,76],'57554c',5723468],
['nbs-iscc-a:greenishgray.154','greenishgray','greenish gray',[178,190,181],'b2beb5',11714229],
['nbs-iscc-a:greenishgray.155','greenishgray','greenish gray',[125,137,132],'7d8984',8227204],
['nbs-iscc-a:greenishgray.156','greenishgray','greenish gray',[78,87,85],'4e5755',5134165],
['nbs-iscc-a:greenishnavy.183','greenishnavy','greenish navy',[0,48,78],'00304e',12366],
['nbs-iscc-a:greenishnavy.188','greenishnavy','greenish navy',[32,40,48],'202830',2107440],
['nbs-iscc-a:greenishwhite.153','greenishwhite','greenish white',[223,237,232],'dfede8',14675432],
['nbs-iscc-a:greenishyellow.98','greenishyellow','greenish yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc-a:greenishyellow.99','greenishyellow','greenish yellow',[190,183,46],'beb72e',12498734],
['nbs-iscc-a:greenishyellow.100','greenishyellow','greenish yellow',[155,148,0],'9b9400',10195968],
['nbs-iscc-a:greenishyellow.101','greenishyellow','greenish yellow',[234,230,121],'eae679',15394425],
['nbs-iscc-a:greenishyellow.102','greenishyellow','greenish yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc-a:greenishyellow.103','greenishyellow','greenish yellow',[152,148,62],'98943e',9999422],
['nbs-iscc-a:navy.183','navy','navy',[0,48,78],'00304e',12366],
['nbs-iscc-a:navy.188','navy','navy',[32,40,48],'202830',2107440],
['nbs-iscc-a:navy.201','navy','navy',[37,36,64],'252440',2434112],
['nbs-iscc-a:olive.106','olive','olive',[134,126,54],'867e36',8814134],
['nbs-iscc-a:olive.107','olive','olive',[102,93,30],'665d1e',6708510],
['nbs-iscc-a:olive.108','olive','olive',[64,61,33],'403d21',4209953],
['nbs-iscc-a:olive.109','olive','olive',[140,135,103],'8c8767',9209703],
['nbs-iscc-a:olive.110','olive','olive',[91,88,66],'5b5842',5986370],
['nbs-iscc-a:olive.111','olive','olive',[54,53,39],'363527',3552551],
['nbs-iscc-a:orange.49','orange','orange',[253,148,63],'fd943f',16618559],
['nbs-iscc-a:orange.50','orange','orange',[237,135,45],'ed872d',15566637],
['nbs-iscc-a:orange.51','orange','orange',[190,101,22],'be6516',12477718],
['nbs-iscc-a:orange.52','orange','orange',[250,181,127],'fab57f',16430463],
['nbs-iscc-a:orange.53','orange','orange',[217,144,88],'d99058',14258264],
['nbs-iscc-a:orange.54','orange','orange',[174,105,56],'ae6938',11430200],
['nbs-iscc-a:pink.1','pink','pink',[255,181,186],'ffb5ba',16758202],
['nbs-iscc-a:pink.2','pink','pink',[234,147,153],'ea9399',15373209],
['nbs-iscc-a:pink.3','pink','pink',[228,113,122],'e4717a',14971258],
['nbs-iscc-a:pink.4','pink','pink',[249,204,202],'f9ccca',16370890],
['nbs-iscc-a:pink.5','pink','pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc-a:pink.6','pink','pink',[192,128,129],'c08081',12615809],
['nbs-iscc-a:pink.7','pink','pink',[234,216,215],'ead8d7',15390935],
['nbs-iscc-a:pink.8','pink','pink',[196,174,173],'c4aead',12889773],
['nbs-iscc-a:pink.18','pink','pink',[173,136,132],'ad8884',11372676],
['nbs-iscc-a:pink.261','pink','pink',[175,134,142],'af868e',11503246],
['nbs-iscc-a:red.12','red','red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-a:red.13','red','red',[132,27,45],'841b2d',8657709],
['nbs-iscc-a:red.14','red','red',[92,9,35],'5c0923',6031651],
['nbs-iscc-a:red.15','red','red',[171,78,82],'ab4e52',11226706],
['nbs-iscc-a:reddishblack.20','reddishblack','reddish black',[84,61,63],'543d3f',5520703],
['nbs-iscc-a:reddishblack.24','reddishblack','reddish black',[40,32,34],'282022',2629666],
['nbs-iscc-a:reddishblack.235','reddishblack','reddish black',[36,33,36],'242124',2367780],
['nbs-iscc-a:reddishblue.195','reddishblue','reddish blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc-a:reddishblue.196','reddishblue','reddish blue',[84,90,167],'545aa7',5528231],
['nbs-iscc-a:reddishblue.197','reddishblue','reddish blue',[39,36,88],'272458',2565208],
['nbs-iscc-a:reddishblue.198','reddishblue','reddish blue',[179,188,226],'b3bce2',11779298],
['nbs-iscc-a:reddishblue.199','reddishblue','reddish blue',[135,145,191],'8791bf',8884671],
['nbs-iscc-a:reddishblue.200','reddishblue','reddish blue',[78,81,128],'4e5180',5132672],
['nbs-iscc-a:reddishblue.201','reddishblue','reddish blue',[37,36,64],'252440',2434112],
['nbs-iscc-a:reddishbrown.13','reddishbrown','reddish brown',[132,27,45],'841b2d',8657709],
['nbs-iscc-a:reddishbrown.14','reddishbrown','reddish brown',[92,9,35],'5c0923',6031651],
['nbs-iscc-a:reddishbrown.16','reddishbrown','reddish brown',[114,47,55],'722f37',7483191],
['nbs-iscc-a:reddishbrown.19','reddishbrown','reddish brown',[144,93,93],'905d5d',9461085],
['nbs-iscc-a:reddishbrown.40','reddishbrown','reddish brown',[136,45,23],'882d17',8924439],
['nbs-iscc-a:reddishbrown.41','reddishbrown','reddish brown',[86,7,12],'56070c',5637900],
['nbs-iscc-a:reddishbrown.43','reddishbrown','reddish brown',[121,68,59],'79443b',7947323],
['nbs-iscc-a:reddishgray.10','reddishgray','reddish gray',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-a:reddishgray.22','reddishgray','reddish gray',[143,129,127],'8f817f',9404799],
['nbs-iscc-a:reddishgray.23','reddishgray','reddish gray',[92,80,79],'5c504f',6049871],
['nbs-iscc-a:reddishgray.232','reddishgray','reddish gray',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-a:reddishgray.233','reddishgray','reddish gray',[139,133,137],'8b8589',9143689],
['nbs-iscc-a:reddishgray.234','reddishgray','reddish gray',[93,85,91],'5d555b',6116699],
['nbs-iscc-a:reddishnavy.212','reddishnavy','reddish navy',[47,33,64],'2f2140',3088704],
['nbs-iscc-a:reddishorange.35','reddishorange','reddish orange',[217,96,59],'d9603b',14245947],
['nbs-iscc-a:reddishorange.36','reddishorange','reddish orange',[170,56,30],'aa381e',11155486],
['nbs-iscc-a:reddishorange.37','reddishorange','reddish orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc-a:reddishorange.38','reddishorange','reddish orange',[158,71,50],'9e4732',10372914],
['nbs-iscc-a:reddishviolet.237','reddishviolet','reddish violet',[158,79,136],'9e4f88',10375048],
['nbs-iscc-a:reddishviolet.238','reddishviolet','reddish violet',[112,41,99],'702963',7350627],
['nbs-iscc-a:reddishviolet.239','reddishviolet','reddish violet',[84,25,78],'54194e',5511502],
['nbs-iscc-a:reddishviolet.240','reddishviolet','reddish violet',[183,132,167],'b784a7',12027047],
['nbs-iscc-a:reddishviolet.241','reddishviolet','reddish violet',[145,92,131],'915c83',9526403],
['nbs-iscc-a:reddishviolet.242','reddishviolet','reddish violet',[93,57,84],'5d3954',6109524],
['nbs-iscc-a:reddishviolet.243','reddishviolet','reddish violet',[52,23,49],'341731',3413809],
['nbs-iscc-a:reddishwhite.9','reddishwhite','reddish white',[234,227,225],'eae3e1',15393761],
['nbs-iscc-a:reddishwhite.231','reddishwhite','reddish white',[232,227,229],'e8e3e5',15262693],
['nbs-iscc-a:reddishyellow.67','reddishyellow','reddish yellow',[255,193,79],'ffc14f',16761167],
['nbs-iscc-a:reddishyellow.68','reddishyellow','reddish yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc-a:reddishyellow.69','reddishyellow','reddish yellow',[201,133,0],'c98500',13206784],
['nbs-iscc-a:reddishyellow.70','reddishyellow','reddish yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc-a:reddishyellow.71','reddishyellow','reddish yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc-a:reddishyellow.72','reddishyellow','reddish yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc-a:reddishyellow.83','reddishyellow','reddish yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-a:reddishyellow.84','reddishyellow','reddish yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-a:reddishyellow.85','reddishyellow','reddish yellow',[175,141,19],'af8d13',11504915],
['nbs-iscc-a:reddishyellow.86','reddishyellow','reddish yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-a:reddishyellow.87','reddishyellow','reddish yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-a:reddishyellow.88','reddishyellow','reddish yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-a:violet.217','violet','violet',[211,153,230],'d399e6',13867494],
['nbs-iscc-a:violet.218','violet','violet',[135,86,146],'875692',8869522],
['nbs-iscc-a:violet.219','violet','violet',[96,47,107],'602f6b',6303595],
['nbs-iscc-a:violet.220','violet','violet',[64,26,76],'401a4c',4201036],
['nbs-iscc-a:violet.221','violet','violet',[213,186,219],'d5badb',14007003],
['nbs-iscc-a:violet.222','violet','violet',[182,149,192],'b695c0',11965888],
['nbs-iscc-a:violet.223','violet','violet',[134,96,142],'86608e',8806542],
['nbs-iscc-a:violet.224','violet','violet',[86,60,92],'563c5c',5651548],
['nbs-iscc-a:violet.225','violet','violet',[48,25,52],'301934',3152180],
['nbs-iscc-a:white.263','white','white',[242,243,244],'f2f3f4',15922164],
['nbs-iscc-a:yellow.83','yellow','yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc-a:yellow.84','yellow','yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc-a:yellow.85','yellow','yellow',[175,141,19],'af8d13',11504915],
['nbs-iscc-a:yellow.86','yellow','yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc-a:yellow.87','yellow','yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-a:yellow.88','yellow','yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc-a:yellowishbrown.74','yellowishbrown','yellowish brown',[153,101,21],'996515',10052885],
['nbs-iscc-a:yellowishbrown.75','yellowishbrown','yellowish brown',[101,69,34],'654522',6636834],
['nbs-iscc-a:yellowishgreen.116','yellowishgreen','yellowish green',[189,218,87],'bdda57',12442199],
['nbs-iscc-a:yellowishgreen.117','yellowishgreen','yellowish green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-a:yellowishgreen.118','yellowishgreen','yellowish green',[70,113,41],'467129',4616489],
['nbs-iscc-a:yellowishgreen.119','yellowishgreen','yellowish green',[201,220,137],'c9dc89',13229193],
['nbs-iscc-a:yellowishgreen.120','yellowishgreen','yellowish green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-a:yellowishgreen.125','yellowishgreen','yellowish green',[74,93,35],'4a5d23',4873507],
['nbs-iscc-a:yellowishgreen.130','yellowishgreen','yellowish green',[131,211,125],'83d37d',8639357],
['nbs-iscc-a:yellowishgreen.131','yellowishgreen','yellowish green',[68,148,74],'44944a',4494410],
['nbs-iscc-a:yellowishgreen.132','yellowishgreen','yellowish green',[0,98,45],'00622d',25133],
['nbs-iscc-a:yellowishgreen.133','yellowishgreen','yellowish green',[0,49,24],'003118',12568],
['nbs-iscc-a:yellowishgreen.134','yellowishgreen','yellowish green',[182,229,175],'b6e5af',11986351],
['nbs-iscc-a:yellowishgreen.135','yellowishgreen','yellowish green',[147,197,146],'93c592',9684370],
['nbs-iscc-a:yellowishgreen.136','yellowishgreen','yellowish green',[103,146,103],'679267',6787687],
['nbs-iscc-a:yellowishgreen.137','yellowishgreen','yellowish green',[53,94,59],'355e3b',3497531],
['nbs-iscc-a:yellowishgreen.138','yellowishgreen','yellowish green',[23,54,32],'173620',1521184],
['nbs-iscc-a:yellowishorange.67','yellowishorange','yellowish orange',[255,193,79],'ffc14f',16761167],
['nbs-iscc-a:yellowishorange.68','yellowishorange','yellowish orange',[234,162,33],'eaa221',15376929],
['nbs-iscc-a:yellowishorange.69','yellowishorange','yellowish orange',[201,133,0],'c98500',13206784],
['nbs-iscc-a:yellowishorange.70','yellowishorange','yellowish orange',[251,201,127],'fbc97f',16501119],
['nbs-iscc-a:yellowishorange.71','yellowishorange','yellowish orange',[227,168,87],'e3a857',14919767],
['nbs-iscc-a:yellowishorange.72','yellowishorange','yellowish orange',[190,138,61],'be8a3d',12487229],
['nbs-iscc-a:yellowishpink.25','yellowishpink','yellowish pink',[255,183,165],'ffb7a5',16758693],
['nbs-iscc-a:yellowishpink.26','yellowishpink','yellowish pink',[248,131,121],'f88379',16286585],
['nbs-iscc-a:yellowishpink.27','yellowishpink','yellowish pink',[230,103,97],'e66761',15099745],
['nbs-iscc-a:yellowishpink.28','yellowishpink','yellowish pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-a:yellowishpink.29','yellowishpink','yellowish pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-a:yellowishpink.30','yellowishpink','yellowish pink',[196,131,121],'c48379',12878713],
['nbs-iscc-a:yellowishpink.31','yellowishpink','yellowish pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-a:yellowishpink.32','yellowishpink','yellowish pink',[199,173,163],'c7ada3',13086115],
['nbs-iscc-a:yellowishred.35','yellowishred','yellowish red',[217,96,59],'d9603b',14245947],
['nbs-iscc-a:yellowishred.36','yellowishred','yellowish red',[170,56,30],'aa381e',11155486],
['nbs-iscc-a:yellowishred.37','yellowishred','yellowish red',[203,109,81],'cb6d51',13331793],
['nbs-iscc-a:yellowishred.38','yellowishred','yellowish red',[158,71,50],'9e4732',10372914]
    ];
}

sub _description {
    return {
          'subtitle' => 'Dye Colors',
          'title' => 'NBS/ISCC A',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (A) Dye Colors

    American Association of Textile Chemists and
    Colorists and Society of Dyers and Colourists 

This combinatorial dictionary\'s 88 names map to 350 colors; 263 out of 267 centroids are covered.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#A]
'
        }

}

1;
