use Cro::HTTP::Server;
use Cro::HTTP::Router;
use Cro::HTTP::Client;
use CamelPub::WebFinger;
use CamelPub::ActivityPub;
use JSON::Fast;
use URI;

module Server {
  constant META6 = from-json slurp "META6.json";
  my $DEBUG = False;

  our sub application(ActivityPub::Document $activitypub_profile) {
    my $id_path = URI.new($activitypub_profile.id).path;
    route {
      before {
        if $DEBUG {
          say request.Str
        } else {
          my $content_type_header = header_find(request.headers, "content-type");
          my $content_type = $content_type_header ?? "[{$content_type_header.value}]" !! "";
          my $accept_header = header_find(request.headers, "accept");
          my $accept = $accept_header ?? "Accept: {$accept_header.value}" !! "";
          my $query = request.query ?? "?{request.query}" !! "";
          say "\n{request.method} {request.path}{$query} {$accept} {$content_type}"
        }
      }
      after { say response.Str if $DEBUG }

      get -> '.well-known', 'nodeinfo' {
        my $host = header_find(request.headers, "host");
        content 'application/json', to-json WebFinger::jrd("http://{$host.value}/api/nodeinfo");
      }

      get -> '.well-known', 'webfinger', :$resource {
        if $resource eq "acct:{$activitypub_profile.webfinger_acct}" {
          my $webfinger_profile = WebFinger::minimal_profile($activitypub_profile.webfinger_acct, $activitypub_profile.id);
          say "-> $resource webfinger returns {$webfinger_profile.links(:rel('self'))}";
          content 'application/json', to-json $webfinger_profile.document;
        } else {
          say "webfinger profile requested for unknown resource {$resource}";
          content 'application/json', to-json {error => "No webfinger profile available for {$resource}"}
        }
      }

      get -> 'api', 'nodeinfo' {
        content 'application/json', to-json nodeinfo();
      }

      get -> *@parts {
        my $accept_types = header_find(request.headers, "accept").value.split(',').Array;

        if $accept_types.first({$_ eq "application/activity+json"}) {
          if request.path eq $id_path {
            say "-> sending activitypub profile for {$activitypub_profile.id}";
            content 'application/json', to-json $activitypub_profile.document;
          } else {
            say "activitypub profile requested for unknown path {request.path}";
            content 'application/json', to-json {error => "No profile available for {request.path}"}
          }
        } else {
          say "-> hello banner html";
          content 'text/html', "CamelPub ActivityPub Server {META6{'version'}} for {$activitypub_profile.id}"
        }
      }

      post -> *@parts {
        if request.path eq "$id_path/inbox" {
          say "-> {$activitypub_profile.id} INBOX POST"
        }
        say "-> {to-json (await request.body)}";
        content 'application/json', '{}';
      }
    }
  }

  sub header_find(List $request, str $name) returns Cro::HTTP::Header {
    request.headers.first: {.name.fc eq $name}
  }

  sub nodeinfo returns Hash {
    { "version" => "2.0",
      "software" => { "name" => "camelpub", "version" => META6{'version'} },
      "protocols" => ["activitypub"],
      "openRegistrations" => False,
      "usage" => { "localPosts" => 0, "localComments" => 0,
                   "users" => { "total" => 1, "activeHalfyear" => 0, "activeMonth" => 1 }},
      "metadata" => {}
    }
  }
}
