######################################################################
#
# JEF_by_Unicode_CultiCoLtd.pl
#
# Copyright (c) 2018 INABA Hitoshi <ina@cpan.org> in a CPAN
######################################################################

# JEF code book, Culti Co.,Ltd.
# http://www.culti.co.jp/2016/02/01/jef%e3%82%b3%e3%83%bc%e3%83%89%e3%83%96%e3%83%83%e3%82%af/

use strict;

my %JEF_by_Unicode_CultiCoLtd = ();
my %Unicode_by_JEF_CultiCoLtd = ();

while (<DATA>) {
    chomp;
    my($unicode, $jis, $sjis, $kuten, $jef) = split(/\t/, $_);

# print "($unicode)($jis)($sjis)($kuten)($jef)\n";

    if ($unicode !~ /^[0123456789ABCDEF]+$/) {
        next;
    }
    if ($jef     !~ /^[0123456789ABCDEF]+$/) {
        next;
    }

    # ignore Private Use Area
    if (('E000' le $unicode) and ($unicode le 'F8FF')) {
        next;
    }

    if (defined $JEF_by_Unicode_CultiCoLtd{$unicode}) {
        die;
    }
    if (defined $Unicode_by_JEF_CultiCoLtd{$jef}) {
        die;
    }

    $JEF_by_Unicode_CultiCoLtd{$unicode} = $jef;
    $Unicode_by_JEF_CultiCoLtd{$jef}     = $unicode;
}

sub JEF_by_Unicode_CultiCoLtd {
    my($unicode) = @_;
    return $JEF_by_Unicode_CultiCoLtd{$unicode};
}

sub keys_of_JEF_by_Unicode_CultiCoLtd {
    return keys %JEF_by_Unicode_CultiCoLtd;
}

sub values_of_JEF_by_Unicode_CultiCoLtd {
    return values %JEF_by_Unicode_CultiCoLtd;
}

1;

# Unicode	JIS	SJIS	_	JEF
__DATA__
4E02				41A1
E131				41A2
4E24				41A3
E134				41A4
E12F				41A5
E12A				41A6
E132				41A7
E130				41A8
4E28		FA68	11541	41AC
4E30				41AD
E000				41AE
4E2E				41AF
E135				41B3
4E47				41B7
51E2				41B8
4E51				41B9
E630				41BA
E138				41BB
4E42	5029	98A7	04809	41BC
4E44				41BD
4E74				41C1
4E68				41C2
E139				41C3
E13A				41C4
4E7F				41C5
4E5C				41C6
4E96				41CE
4E8D				41CF
4E97				41D0
4E9D				41D1
E13C				41D5
E13D				41D6
E001				41D7
4EAF				41D8
E002				41D9
E13E				41DA
E13F				41DB
501D				41DF
E140				41E0
E659				41E1
4F58				41E2
4F71				41E3
E141				41E4
4EC3				41E5
E003				41E6
4EDB				41E7
4EE1		FA69	11542	41E8
4EE8				41E9
4EF1				41EA
4EF3				41EB
4EFD				41EC
4EFE				41ED
4EFF				41EE
4F00		FA6B	11544	41EF
4F03		FA6C	11545	41F0
4F08				41F1
4F0B				41F2
4F12				41F3
4F15				41F4
4F19				41F5
4F02				41F6
4F2E				41F7
E144				41F8
E145				41F9
4F35				41FA
4F39		FA6D	11546	41FB
4F3B				41FC
4F3E				41FD
E146				41FE
4F42				42A1
4F48				42A2
4F49				42A3
4F4B				42A4
E147				42A5
4F52				42A6
4F54				42A7
4F56		FA6E	11547	42A8
E004				42A9
4F5F				42AA
4F63				42AB
4F6E				42AC
4F78				42AD
E148				42AE
E149				42AF
E14A				42B0
E14B				42B1
4F84				42B2
4F8A		FA70	11549	42B3
4F8E				42B4
4F92		FA6F	11548	42B5
4F93				42B6
4F94		FA72	11551	42B7
4F97				42B8
4F9A		FA71	11550	42B9
4F9F				42BA
FA30				42BB
E631				42BC
4FB1				42BD
4FB2				42BE
E14F				42BF
E14D				42C0
4FBC				42C1
4FBD				42C2
4FBE				42C3
E14E				42C4
4FC1				42C5
4FC5				42C6
4FC9		FA61	11534	42C7
4FCB				42C8
4FCC				42C9
4FCD		FA73	11552	42CA
4FCF				42CB
4FE2				42CC
E633				42CD
4FF0				42CE
E634				42CF
4FFC				42D0
4FFD				42D1
5000				42D2
5001				42D3
5004				42D4
5007				42D5
500C				42D6
500E				42D7
5010				42D8
5013				42D9
5017				42DA
5018				42DB
501B				42DC
501C				42DD
501E		FA77	11556	42DE
5022		FA75	11554	42DF
5027				42E0
502E				42E1
5030				42E2
5032				42E3
5035				42E4
E006				42E5
E005				42E6
5040		FA74	11553	42E7
5042		FA7A	11559	42E8
5046		FA78	11557	42E9
504E				42EA
E152				42EB
5057				42EC
5059				42ED
505F				42EE
5062				42EF
5063				42F0
E153				42F1
5066				42F2
5067				42F3
506A				42F4
5070		FA79	11558	42F5
5071				42F6
E154				42F7
5084				42F8
508A				42F9
508E				42FA
5093				42FB
5094		FA7B	11560	42FC
5096				42FD
4FF2				42FE
509B				43A1
509C				43A2
509E				43A3
509F				43A4
50A0				43A5
E155				43A6
E007				43A7
E635				43A8
50B9				43A9
50BD				43AA
50C0				43AB
50C7				43AC
50CC				43AD
7FDB				43AE
50CE				43AF
50D3				43B0
50D4				43B1
50D8		FA7D	11562	43B2
E636				43B3
50DC				43B4
E637				43B5
50E6				43B6
FA31				43B7
50E9				43B8
E638				43B9
50EF				43BA
50F1				43BB
50FA				43BC
E15A				43BD
5106				43BE
5107				43BF
5110				43C0
5117				43C1
5119				43C2
511D				43C3
511E				43C4
5123				43C5
5127				43C6
E15B				43C7
512D				43C8
512F				43C9
5138				43CA
4FA0	3622	8BA0	02202	43CB
50C5	364F	8BCD	02247	43CC
5026	3771	8C91	02381	43CD
5132	4C59	96D7	04457	43CE
E7BA				43CF
4FFF		FA76	11555	43D0
515F				43D2
E15D				43D3
E008				43D4
FA32				43D5
514F				43D6
5157				43D7
5158				43D8
5164		FA80	11564	43D9
514E	4546	9365	03738	43DA
5154	513D	995C	04929	43DB
E639				43DE
E15F				43DF
E632				43E0
E7B9				43E4
E160				43E5
E163				43E6
517E				43E7
E162				43E8
914B	3D36	8F55	02922	43E9
518B				43EC
E164				43ED
E7BB				43EE
E7BC				43F0
51A3				43F1
519D		FA81	11565	43F2
E168				43F5
51BC				43F6
51BE		FA82	11566	43F7
51BF				43F8
51C2				43F9
51C8				43FA
E63A				43FB
51CF				43FC
51D2				43FD
E63B				43FE
51D5				44A1
51D8				44A2
E009				44A3
51DC	7425	EAA3	08405	44A4
51B4	3A63	8DE1	02667	44A5
51CB	437C	929C	03592	44A6
51EE				44A8
51F2				44A9
51E5				44AA
51F4				44AB
E7BD				44AC
5202				44B2
E16B				44B3
5215		FA84	11568	44B4
E16D				44B5
E171				44B6
E00A				44B7
E00B				44B8
E00D				44B9
E63F				44BA
E00C				44BB
E16F				44BC
5218				44BD
5222				44BE
E170				44BF
E63C				44C0
E172				44C1
5249				44C2
E175				44C3
5257				44C4
5258				44C5
5260				44C6
E178				44C7
5277				44C8
5278				44C9
5279				44CA
5282				44CB
5293				44CC
529A				44CD
E173				44CE
E177				44CF
5265	476D	948D	03977	44D0
52A6		FA86	11570	44D2
52BD				44D3
E17C				44D4
52E5				44D5
52F6				44D6
E00E				44D7
52A4				44D8
E00F				44D9
52A5				44DA
52A7				44DB
52AF		FB77	11756	44DC
52B0				44DD
52B6				44DE
52B7				44DF
52B8				44E0
52BA				44E1
52BB				44E2
E17B				44E3
E010				44E4
52C4				44E5
52C6				44E6
FA33				44E7
E640				44E8
52CF				44E9
52D1				44EA
52D4				44EB
E17D				44EC
52DB		FA88	11572	44ED
52DC				44EE
FA34				44EF
52E1				44F0
E17F				44F1
52E9				44F2
52EA				44F3
52EC				44F4
52F1				44F5
52F4				44F6
529C		FA85	11569	44F7
8FA6				44F8
E17E				44F9
52C0		FA87	11571	44FA
E180				45A1
E641				45A2
5300		FA89	11573	45A3
5303				45A4
E181				45A5
530A				45A6
530C				45A7
E7BE				45A8
E182				45AB
E011				45AC
531B				45B0
531C				45B1
531E				45B2
E642				45B3
5325				45B4
E012				45B5
E183				45B6
5327				45B7
5328				45B8
5329				45B9
533D				45BA
E184				45BB
5332				45BC
E185				45BD
8294				45C1
5342				45C2
FA35				45C3
535B				45C4
5359				45C5
E187				45C6
E186				45C7
5361				45CB
E188				45CF
536C				45D0
536D				45D1
5372		FA8C	11576	45D2
5379				45D3
E644				45D4
537F	362A	8BA8	02210	45D5
5387				45D8
5388				45D9
538E				45DA
5393		FA8D	11577	45DB
5394				45DC
5399				45DD
539D				45DE
53A1				45DF
53AA				45E0
53AB				45E1
53B2		FA8E	11578	45E2
53B5				45E3
E7BF				45E4
53B8				45E7
53BD				45E8
53C5				45E9
53DB	4840	94BE	04032	45EC
E18B				45ED
E013				45EE
E18D				45EF
53DE				45F0
53DA				45F1
E7C0				45F2
53DD		FA8F	11579	45F3
FA38				45F4
569A				45F5
5586		FA95	11585	45F6
53E6				45F7
E192				45F8
E646				45F9
54FF		FA94	11584	45FA
E190				45FB
E647				45FC
5444				45FD
E19A				45FE
E196				46A1
E19C				46A2
E195				46A3
E19D				46A4
5547				46A5
560F				46A6
569E				46A7
56AD				46A8
E645				46AA
5421				46AB
5427				46AC
5428				46AD
542A				46AE
5435				46AF
E193				46B0
5443				46B1
E648				46B2
544D				46B3
5462				46B4
E649				46B5
5464				46B6
5466				46B7
5469				46B8
546B				46B9
546D				46BA
546E				46BB
547F				46BC
5481				46BD
5483				46BE
5488				46BF
5489				46C0
E64A				46C1
5491				46C2
5496				46C3
549C		FA91	11581	46C4
549F				46C5
54A1				46C6
54A6				46C7
54A7				46C8
54A9		FA93	11583	46C9
54AD				46CA
54B1				46CB
E198				46CC
54B7				46CD
54B9				46CE
54BA				46CF
54C6				46D0
54CA				46D1
54CD				46D2
54E0				46D3
54EA				46D4
54EC				46D5
5500				46D6
5501				46D7
5505				46D8
5509				46D9
550D				46DA
550E				46DB
552A				46DC
552B				46DD
E19E				46DE
5535				46DF
553C				46E0
5549				46E1
554A				46E2
5550				46E3
5561				46E4
5581				46E5
5588				46E6
558E				46E7
5591				46E8
5597				46E9
55A4				46EA
E1A4				46EB
E1A5				46EC
55AD				46ED
55C3				46EE
55CE				46EF
55D1				46F0
55D2				46F1
55D8				46F2
55DB				46F3
55E9				46F4
55FF				46F5
E1A7				46F6
FA37				46F7
5608				46F8
560D				46F9
560E				46FA
5610				46FB
5611				46FC
E1AB				46FD
5635				46FE
5639				47A1
563B				47A2
563F				47A3
5643				47A4
E1AC				47A5
5649				47A6
564B				47A7
5672				47A8
564D				47A9
564F				47AA
5660				47AB
566D				47AC
566F				47AD
E1AE				47AE
E64C				47AF
568B				47B0
569D				47B1
56AB				47B2
56AC				47B3
56C9				47B4
E18F				47B5
E191				47B6
5669				47B7
559D	3365	8A85	01969	47B8
5516	3022	88A0	01602	47B9
5618	3133	8952	01719	47BA
5642	313D	895C	01729	47BB
565B	337A	8A9A	01990	47BC
56DC				47BD
56E4				47BE
56E7				47BF
56ED				47C0
56F7				47C1
5701				47C2
E1B1				47C3
570A				47C4
E1B3				47C5
571A				47C6
571B				47C7
5C2D	3646	8BC4	02238	47C8
E7C1				47C9
586B	4536	9355	03722	47CA
5835	4548	9367	03740	47CB
E64D				47CC
579A				47CD
5725				47CE
5774				47CF
57FF				47D0
57F6				47D1
5723				47D2
574C				47D3
E014				47D4
57A9				47D5
57BD				47D6
57FE				47D7
E1C0				47D8
588A				47D9
588F				47DA
E1C3				47DB
FA3A				47DC
58C2				47DD
58E1				47DE
5722				47DF
E1B4				47E0
572F				47E1
5733				47E2
573F				47E3
5746				47E4
E1B5				47E5
5762				47E6
5767				47E7
E1B6				47E8
576D				47E9
576E				47EA
576F				47EB
5777				47EC
577B				47ED
577C				47EE
577E				47EF
5781				47F0
578C				47F1
5797				47F2
5799				47F3
E1B7				47F4
57A8				47F5
57C7		FA9A	11590	47F6
57C8		FA99	11589	47F7
57CC				47F8
E1B8				47F9
57CF				47FA
E1B9				47FB
57DE				47FC
57E4				47FD
57E6				47FE
57E9				48A1
57ED				48A2
57F0				48A3
57F5				48A4
5808				48A5
5809				48A6
581E				48A7
E1BC				48A8
5820				48A9
5826				48AA
582D				48AB
583F				48AC
584F				48AD
5855				48AE
5867				48AF
5868				48B0
5878				48B1
587F				48B2
E1C1				48B3
E1C2				48B4
5888				48B5
5889				48B6
588C				48B7
5894				48B8
FA39				48B9
589D				48BA
E64E				48BB
58A2				48BC
58A9				48BD
5880				48BE
58CE				48BF
58D4				48C0
58DA				48C1
E1C5				48C2
E64F				48C3
58E2				48C4
58E9				48C5
E12B				48C6
5858	4564	9384	03768	48C7
580B	5444	9AC2	05236	48C8
E7C2				48C9
5765		FA97	11587	48CA
57AC		FA98	11588	48CB
FA0F		FA9B	11591	48CC
58B2		FA9E	11594	48CD
E1C7				48CE
5906				48CF
590B		FA9F	11601	48D0
590C				48D1
E015				48D2
8641				48D3
E650				48D7
5921				48D8
5936				48DC
5933				48DD
5935				48DE
593F				48DF
5943				48E0
E1C9				48E1
5952				48E2
E1CB				48E3
E1CC				48E4
595D		FAA2	11604	48E5
595E				48E6
595F				48E7
5972				48E8
E651				48E9
E1CA				48EA
5963		FAA3	11605	48EB
596D				48EC
599F				48F0
E1CE				48F1
59F4				48F2
E1D0				48F3
5A0E				48F4
5AC8				48F5
5B4C				48F6
5AB5				48F7
5AE0				48F8
5975				48F9
5976				48FA
598B				48FB
598C				48FC
598E				48FD
5992				48FE
5995				49A1
5997				49A2
59A4		FAA4	11606	49A3
59AD				49A4
59AF				49A5
59B0				49A6
59B7				49A7
59C3				49A8
59C4				49A9
59C8				49AA
59CA				49AB
E1CF				49AC
59D2				49AD
59DD				49AE
59DE				49AF
E1D1				49B0
59E4				49B1
E652				49B2
59EF				49B3
59F2				49B4
E653				49B5
5A0C				49B6
5A0D				49B7
5A12				49B8
5A13				49B9
E654				49BA
5A1E				49BB
5A23				49BC
5A27				49BD
5A30				49BE
5A44				49BF
5A45				49C0
5A48				49C1
5A4C				49C2
5A5E				49C3
5A63				49C4
5A65				49C5
E1D3				49C6
5A67				49C7
5A6D				49C8
5A77				49C9
E1D4				49CA
E655				49CB
5A8B				49CC
5A93				49CD
5A96				49CE
5A99				49CF
5A9C				49D0
5A9E				49D1
5A9F				49D2
5AA7				49D3
5A55				49D4
5AB1				49D5
5AB3				49D6
5ABA				49D7
5ABF				49D8
5AC4				49D9
5ACF				49DA
5ADC				49DB
E1D9				49DC
5AE5				49DD
E656				49DE
5AEA				49DF
5AEE				49E0
5AFD				49E1
5B00				49E2
5B01				49E3
E657				49E4
5B08				49E5
5B19				49E6
5B1B				49E7
5B2D				49E8
E1DB				49E9
5B4B				49EA
E1DA				49EB
5ACC	3779	8C99	02389	49EC
5A9B	4932	9551	04118	49ED
5A29	4A5A	95D8	04258	49EE
5ABE	553D	9B5C	05329	49EF
59BA		FAA5	11607	49F0
5B56		FAA6	11608	49F2
E017				49F3
5B5E				49F4
5B6F				49F5
E658				49F6
5B7F				49F7
E1DD				49F8
E016				49F9
5B6E				49FA
5B81				4AA1
5B84				4AA2
E018				4AA3
5B86				4AA4
5B8A				4AA5
5B8E				4AA6
E019				4AA7
5B93				4AA8
5B96				4AA9
5BA8				4AAA
5BA9				4AAB
5BAF				4AAC
E01A				4AAD
5BB1				4AAF
E1DF				4AB0
E1E0				4AB1
E1DE				4AB2
5BB7				4AB3
E01B				4AB4
5BBA				4AB5
5BBC				4AB6
5BC1				4AB7
E1E3				4AB8
E01C				4AB9
5BCD				4ABA
5BCF				4ABB
E1E4				4ABC
752F		FAA8	11610	4ABD
5BD6				4ABE
5BD7				4ABF
5BD8		FAA9	11611	4AC0
5BD9				4AC1
5BDA				4AC2
5BE0				4AC3
E1E5				4AC4
E01D				4AC5
E1E6				4AC6
5BF4				4AC7
E7C3				4AC8
5BC0		FAA7	11609	4AC9
E1E9				4ACC
E1E8				4ACD
E1E7				4ACE
E1EA				4ACF
E65B				4AD3
5C1E		FAAB	11613	4AD4
5C2E				4AD8
5C32				4AD9
5C35				4ADA
5C2B				4ADB
5C63				4ADF
FA3B				4AE0
5C67				4AE1
5C68				4AE2
5C6D				4AE3
5C51	367D	8BFB	02293	4AE4
5C61	3C48	8EC6	02840	4AE5
5C60	454B	936A	03743	4AE6
E7C4				4AE7
5C74				4AEB
5C87				4AEC
5C9D				4AED
5CA2				4AEE
E01F				4AEF
E020				4AF0
5CA6		FAAC	11614	4AF1
E65E				4AF2
5CBA		FAAD	11615	4AF3
5CC0				4AF4
5CC9				4AF5
5CF2				4AF6
5D12				4AF7
5D2B				4AF8
E022				4AF9
5D23				4AFA
5D27		FAAF	11617	4AFB
E1F0				4AFC
E023				4AFD
E024				4AFE
5D34				4BA1
5D42		FAB2	11620	4BA2
E668				4BA3
5D61				4BA4
E025				4BA5
5D7E				4BA6
5D7F				4BA7
5D88				4BA8
5D8A				4BA9
5D94				4BAA
E028				4BAB
E66A				4BAC
E02A				4BAD
E029				4BAE
5DC3				4BAF
5DCB				4BB0
E65D				4BB1
E01E				4BB2
E021				4BB3
5D46				4BB4
5D64				4BB5
E027				4BB6
5C7A				4BB7
5C7B				4BB8
5C7C				4BB9
E1EE				4BBA
5C8F				4BBB
5C92				4BBC
E660				4BBD
5CAD				4BBE
5CB5				4BBF
E661				4BC0
5CCB				4BC1
5CD2				4BC2
5CD7				4BC3
E662				4BC4
5CEE				4BC5
5CF4				4BC6
E663				4BC7
5D06				4BC8
E664				4BC9
E665				4BCA
E666				4BCB
5D39				4BCC
5D3F				4BCD
5D43				4BCE
5D48				4BCF
FA11		FAB1	11619	4BD0
5D62				4BD1
E669				4BD2
5D6D		FAB3	11621	4BD3
5D83				4BD4
E026				4BD5
5D92				4BD6
5D95				4BD7
5D99				4BD8
5D9B				4BD9
5D9F				4BDA
5DA0				4BDB
5DA7				4BDC
5DAB				4BDD
E02B				4BDE
5DB8		FAB4	11622	4BDF
5DB9		FAB5	11623	4BE0
E02C				4BE1
E66B				4BE2
E66C				4BE3
E7C5				4BE4
5DD0		FAB6	11624	4BE5
E02D				4BED
E1F1				4BF1
E643				4BF2
5DF8				4BF3
5DF7	392B	8D4A	02511	4BF4
5DFD	4327	9246	03507	4BF5
5E00				4BF7
5E0D				4BF8
5E12				4BF9
5E18				4BFA
E1F4				4BFB
5E2E				4BFC
5E4B				4BFD
E1F8				4BFE
E02E				4CA1
5E14				4CA2
5E15				4CA3
E66E				4CA4
5E28				4CA5
E1F6				4CA6
5E3E				4CA7
E66F				4CA8
5E58				4CA9
E1F9				4CAA
5E6C				4CAB
E1FA				4CAC
E1FB				4CB0
79CA				4CB1
E670				4CB2
E02F				4CB6
5E80				4CBA
5E88				4CBB
E671				4CBC
E030				4CBD
5EA2				4CBE
5EA4				4CBF
5EA5				4CC0
5EA8				4CC1
E1FD				4CC2
E031				4CC3
5EB3				4CC4
E1FE				4CC5
5EBD				4CC6
5EBE				4CC7
E032				4CC8
E1FF				4CC9
F928				4CCA
5ED2				4CCB
5EDE				4CCC
E033				4CCD
5EDC				4CCE
5EE5				4CCF
E034				4CD0
5EE0	3E33	8FB1	03019	4CD1
5EDF	4940	955F	04132	4CD2
5E96	4A79	95F7	04289	4CD3
E205				4CD5
E206				4CD6
E12C				4CD7
E209				4CD8
5F07				4CDC
5F08				4CDD
E20B				4CDE
E035				4CDF
5F1C				4CE7
E20F				4CE8
5F40				4CE9
5F19				4CEA
E20D				4CEB
5F23				4CEC
5F30				4CED
5F34		FAB8	11626	4CEE
5F36				4CEF
E672				4CF0
5F3B				4CF1
5F3D				4CF2
E036				4CF3
E037				4CF4
5F4D				4CF5
5F45		FA67	11540	4CF6
5F50				4CF9
5F54				4CFA
5F65				4DA1
5F67		FAB9	11627	4DA2
E038				4DA3
5F63				4DA4
5F64				4DA5
E213				4DA6
E214				4DA7
5F6F				4DA8
E215				4DA9
5F78				4DAD
5F7E				4DAE
E216				4DAF
5F89				4DB0
5F8D				4DB1
E039				4DB2
E03A				4DB3
E03B				4DB4
5F9C				4DB5
5F9D				4DB6
5FA2				4DB7
E673				4DB8
E03C				4DB9
5FA7				4DBA
E218				4DBB
5FAC				4DBC
E219				4DBD
5FAF				4DBE
E21A				4DBF
E674				4DC0
5FBD	352B	8B4A	02111	4DC1
6167	3745	8C64	02337	4DC2
5FC4				4DC3
5FD2				4DC4
E21C				4DC5
E675				4DC6
E21D				4DC7
5FE2				4DC8
5FE9				4DC9
E21E				4DCA
E03E				4DCB
E222				4DCC
6067				4DCD
607F				4DCE
6086				4DCF
608A		FABE	11632	4DD0
6098				4DD1
60A5				4DD2
60A8				4DD3
E042				4DD4
E044				4DD5
E226				4DD6
6102				4DD7
E046				4DD8
E229				4DD9
E67C				4DDA
E047				4DDB
E67D				4DDC
617C				4DDD
E22D				4DDE
6197				4DDF
619D				4DE0
61A0				4DE1
61A5				4DE2
61A8				4DE3
E22E				4DE4
61BC				4DE5
E231				4DE6
E049				4DE7
E232				4DE8
6207				4DE9
E04A				4DEA
5FC7				4DEB
5FC9				4DEC
5FD3				4DED
5FD4				4DEE
E676				4DEF
5FE1				4DF0
5FEE				4DF1
5FEF				4DF2
5FF3				4DF3
5FF6				4DF4
5FFA				4DF5
6007				4DF6
6014				4DF7
6017				4DF8
E677				4DF9
6033				4DFA
E03D				4DFB
6047				4DFC
6049				4DFD
604C				4DFE
E03F				4EA1
6051				4EA2
E220				4EA3
6057				4EA4
6061				4EA5
6071				4EA6
6082				4EA7
E678				4EA8
6088				4EA9
E679				4EAA
6091				4EAB
6093				4EAC
FA3D				4EAD
6095				4EAE
6097	5824	9CA2	05604	4EAF
609D				4EB0
609E				4EB1
E040				4EB2
60A2				4EB3
60B0				4EB4
60B1				4EB5
E041				4EB6
60BB				4EB7
60C2				4EB8
E224				4EB9
60CA				4EBA
60CB				4EBB
E043				4EBC
60D5		FAC0	11634	4EBD
60D9				4EBE
60DB				4EBF
60DD				4EC0
60DE		FABF	11633	4EC1
60F2		FAC2	11636	4EC2
E67A				4EC3
E045				4EC4
E227				4EC5
6112				4EC6
6113				4EC7
6114				4EC8
6116				4EC9
E67B				4ECA
611C				4ECB
6122				4ECC
612B				4ECD
6130		FAC5	11639	4ECE
6131				4ECF
6137		FAC4	11638	4ED0
6139				4ED1
E048				4ED2
6146				4ED3
615E				4ED4
6160				4ED5
E22A				4ED6
616C				4ED7
FA3F				4ED8
6193				4ED9
6195				4EDA
6198		FAC6	11640	4EDB
619C				4EDC
619F				4EDD
61AD				4EDE
E67E				4EDF
E22F				4EE0
61B9				4EE1
E67F				4EE2
61C0				4EE3
61CE				4EE4
61CF				4EE5
61DD				4EE6
E230				4EE7
61E2				4EE8
61ED				4EE9
6062	327A	89F8	01890	4EEA
6108	4C7C	96FA	04492	4EEB
E7C6				4EEC
E7C7				4EED
6111		FAC3	11637	4EEE
E234				4EEF
E04B				4EF0
E04C				4EF1
621C				4EF2
6215				4EF3
6220				4EF4
6222				4EF5
6229				4EF6
6213		FAC7	11641	4EF7
6249	4862	94E0	04066	4EF9
E680				4EFA
E681				4EFB
E236				4EFC
6243				4EFD
E238				4EFE
62F3	377D	8C9D	02393	4FA1
6372	377E	8C9E	02394	4FA2
624C				4FA3
6310				4FA4
E254				4FA5
64CA				4FA6
64CE		FACC	11646	4FA7
64E5				4FA8
6251				4FA9
6252				4FAA
6256				4FAB
625A				4FAC
E04D				4FAD
626D				4FAE
626F				4FAF
E23C				4FB0
6273				4FB1
E23D				4FB2
E682				4FB3
6290				4FB4
62A6		FAC8	11642	4FB5
E23E				4FB6
62C4				4FB7
E683				4FB8
62CE				4FB9
E23F				4FBA
62D0	327D	89FB	01893	4FBB
E04E				4FBC
62D5				4FBD
62EA				4FBE
62F4				4FBF
62FC				4FC0
62FD				4FC1
6304				4FC2
E04F				4FC3
630A				4FC4
630D				4FC5
6313				4FC6
632A				4FC7
632D				4FC8
6335				4FC9
6336				4FCA
E684				4FCB
6339				4FCC
6341				4FCD
6343				4FCE
6344				4FCF
6346				4FD0
6352				4FD1
E246				4FD2
6358				4FD3
E248				4FD4
E050				4FD5
636C				4FD6
6374				4FD7
637D				4FD8
637F				4FD9
6382				4FDA
E249				4FDB
6387				4FDC
6395				4FDD
639E				4FDE
E24A				4FDF
63A4				4FE0
E24B				4FE1
63AE				4FE2
E686				4FE3
63C1				4FE4
E051				4FE5
63D1				4FE6
63DC				4FE7
63E0				4FE8
63EA				4FE9
E685				4FEA
63F2				4FEB
E24E				4FEC
63F5		FAC9	11643	4FED
6409				4FEE
6410				4FEF
E250				4FF0
6420				4FF1
E252				4FF2
6425				4FF3
642A				4FF4
E687				4FF5
E688				4FF6
E255				4FF7
645A				4FF8
645F				4FF9
E256				4FFA
6473				4FFB
6485				4FFC
E258				4FFD
6491				4FFE
6499				50A1
649D		FACB	11645	50A2
64A8				50A3
E052				50A4
64BF				50A5
64C4				50A6
64CB				50A7
64D0				50A8
64D1				50A9
64D7				50AA
64E9				50AB
64ED				50AC
64F0				50AD
64F5				50AE
6501				50AF
6504				50B0
6505	5A39	9DB7	05825	50B1
6509				50B2
650A				50B3
6514				50B4
6516				50B5
6529				50B6
647A	4022	90A0	03202	50B7
64B0	4071	90EF	03281	50B8
63BB	415F	917E	03363	50B9
63C3	4237	91B5	03423	50BA
63B4	444F	92CD	03647	50BB
6357	443D	92BB	03629	50BC
E25B				50BD
6531				50BE
E053				50BF
E689				50C0
657A				50C4
653C				50C5
E68A				50C6
E054				50C7
6547				50C8
E68B				50C9
FA41				50CA
6550				50CB
6554				50CC
655F				50CD
656B				50CE
E68C				50CF
E25E				50D0
E25C				50D1
E25F				50D2
E260				50D6
658A				50D7
E055				50D8
6595				50D9
659D				50DD
65A6				50E1
E235				50E2
65AE				50E3
65B4				50E4
E261				50E5
E263				50E9
E056				50EA
65C2				50EB
E262				50EC
65D0				50ED
65D4				50EE
65D8				50EF
65DF				50F0
E264				50F4
660D				50F8
667F				50F9
65F2				50FA
65F5				50FB
6602	3937	8D56	02523	50FC
E693				50FD
662A				50FE
662E		FAD2	11652	51A1
6630				51A2
6631		FA63	11536	51A3
E059				51A4
6651				51A5
6678				51A6
6677				51A7
E26D				51A8
E05C				51A9
FA43				51AA
66A0		FADB	11661	51AB
E69A				51AC
6705				51AD
E68D				51AE
E057				51AF
65F9				51B0
E691				51B1
E05A				51B2
6673		FAD9	11659	51B3
6675				51B4
668B				51B5
E69B				51B6
E270				51B7
65F0				51B8
E68E				51B9
E68F				51BA
E690				51BB
65FF				51BC
E266				51BD
6604				51BE
6608				51BF
6609		FAD1	11651	51C0
E267				51C1
E692				51C2
6611				51C3
6615		FACF	11649	51C4
E6B2				51C5
E058				51C6
661E		FAD3	11653	51C7
6621				51C8
6622				51C9
6624		FAD4	11654	51CA
662B				51CB
6633				51CC
6639				51CD
6640				51CE
6645				51CF
E26A				51D0
664A				51D1
664C				51D2
664E				51D3
6659		FAD7	11657	51D4
E695				51D5
665B				51D6
6661				51D7
E696				51D8
E697				51D9
E26C				51DA
666A				51DB
666B				51DC
667E				51DD
FA12		FAD8	11658	51DE
E05B				51DF
668C				51E0
668D				51E1
6692				51E2
E26E				51E3
E26F				51E4
6699		FADA	11660	51E5
669A				51E6
669B				51E7
669F				51E8
E698				51E9
66B1				51EA
66B2		FADC	11662	51EB
66B5				51EC
66BB				51EE
E699				51EF
E05D				51F0
66BF		FADD	11663	51F1
66C0				51F2
E05E				51F3
66C8				51F4
66D4				51F5
E05F				51F6
66DB				51F7
E060				51F8
E274				51F9
66EC				51FA
E265				51FB
E269				51FC
6666	3322	8A41	01902	51FD
66D9	3D6C	8F8C	02976	51FE
6657		FAD6	11656	52A1
66FA		FADE	11664	52A2
E276				52A4
E279				52A5
E27B				52A6
E277				52A7
E388				52A8
E389				52A9
E719				52AA
E061				52AB
81DD				52AC
E27C				52AD
6719				52AE
E27D				52AF
E27E				52B0
E69C				52B1
80A7				52B2
80AB				52B3
E278				52B4
E0B5				52B5
80D2				52B6
80D5				52B7
80D7				52B8
80D8				52B9
E38A				52BA
80E0				52BB
670E		FADF	11665	52BC
80F0				52BD
80F3				52BE
80F6				52BF
6713				52C0
E0B6				52C1
E27A				52C2
E38B				52C3
8116				52C4
8118				52C5
E0B7				52C6
E71A				52C7
E0B8				52C8
E71B				52C9
812C				52CA
8130				52CB
E0B9				52CC
671E	5B2E	9E4D	05914	52CD
813C				52CE
814A				52CF
813A				52D0
E38F				52D1
816D				52D2
816F				52D3
E390				52D4
E71C				52D5
6720				52D6
E391				52D7
8185				52D8
E0BA				52D9
819B				52DA
819E				52DB
E71D				52DC
81AE				52DD
81C5				52DE
81CA				52DF
81CF				52E0
81FD				52E1
817F	425C	91DA	03460	52E2
6813	4072	90F0	03282	52E3
68A2	3E3F	8FBD	03031	52E4
68DA	432A	9249	03510	52E5
68FD				52E6
E069				52E7
6A46		FAEF	11681	52E8
E6A0				52E9
687C				52EA
E69F				52EB
67BC				52EC
67D2				52ED
6852		FAE3	11669	52EE
684C				52EF
E71E				52F0
E66D				52F1
E6A7				52F2
68E8				52F3
6998		FAEB	11677	52F4
E6AB				52F5
69F3				52F6
E2A0				52F7
6A50				52F8
E2AA				52F9
69A6				52FA
E29E				52FB
6733				52FC
6744				52FD
6747				52FE
E69D				53A1
674C				53A2
6755				53A3
675D				53A4
676C				53A5
E69E				53A6
677B				53A7
E284				53A8
6784				53A9
678E				53AA
6793				53AB
6796				53AC
6798				53AD
6799				53AE
67B1				53AF
E6A1				53B0
67B5				53B1
67BB		FAE2	11668	53B2
67F9				53B3
67C0		FAE4	11670	53B4
67C2				53B5
67C5				53B6
67D7				53B7
67D9				53B8
E286				53B9
67E1				53BA
E062				53BB
E287				53BC
67F2				53BD
67F5	3A74	8DF2	02684	53BE
67F7				53BF
E28A				53C0
E6A3				53C1
E28B				53C2
6831				53C3
6833				53C4
683B				53C5
E28C				53C6
6844		FAE6	11672	53C7
E28D				53C8
E6A4				53C9
6857				53CA
6858				53CB
685B				53CC
E6A5				53CD
6870				53CE
6872				53CF
6875				53D0
6879				53D1
6884				53D2
FA44				53D3
E063				53D4
689A				53D5
689C				53D6
68A3				53D7
68AA				53D8
68BB				53D9
68D1				53DA
68D6				53DB
68DC				53DC
68EB				53DD
68EC				53DE
68ED				53DF
68F0				53E0
68F1				53E1
690A				53E2
E064				53E3
6935				53E4
E065				53E5
6945				53E6
6949				53E7
E6A8				53E8
E299				53E9
6963				53EA
6966				53EB
E6A9				53EC
6968		FAE9	11675	53ED
6980				53EE
6992				53EF
E6AA				53F0
69A5				53F1
69A8				53F2
69AB				53F3
69AD				53F4
69B7				53F5
69C8				53F6
E29A				53F7
E066				53F8
E067				53F9
FA45				53FA
E6AC				53FB
6A00				53FC
6A11				53FD
6A24				53FE
6A32				54A1
6A3B				54A2
6A3E				54A3
6A3F				54A4
6A45				54A5
E2A7				54A6
6A52				54A7
E6AD				54A8
6A55				54A9
6A5B				54AA
E2A4				54AB
6A6A				54AC
E6AE				54AD
E068				54AE
E2A3				54AF
6A7E		FAF1	11683	54B0
6A89				54B1
E06A				54B2
6A91				54B3
6A9D				54B4
6A9E				54B5
E6AF				54B6
6AAF				54B7
E06B				54B8
6ACC				54B9
E6B0				54BA
6AD5				54BB
E6B1				54BC
6AE4		FAF3	11685	54BD
6AE7				54BE
6AEC				54BF
F91D				54C0
6B06				54C1
6B09				54C2
6A67				54C3
69D9	4B6A	968A	04374	54C4
6B1D	3135	8954	01721	54C5
6E20	3574	8B94	02184	54C6
6ADB	367B	8BF9	02291	54C7
698A	3A67	8DE5	02671	54C8
6753	3C5D	8EDB	02861	54C9
E06C				54CA
6B2C				54CB
6B2F				54CC
E06D				54CD
E6B3				54CE
6B35				54CF
6B3B				54D0
6B46				54D1
6B4A				54D2
E2AD				54D3
E6B4				54D4
6B52				54D5
6B56				54D6
E06E				54D7
6B5D				54D8
E6B5				54DC
6B72				54DD
E06F				54DE
E6B6				54DF
6B67				54E0
6B82				54E5
6B85				54E6
6B9B				54E7
6BAE				54E8
6BC9				54EC
F970				54ED
6BBD				54EE
6BBE				54EF
E070				54F0
E6B7				54F4
6BD6		FAF4	11686	54F8
E6B8				54F9
6C02				54FD
6BFF				54FE
E2A9				55A1
6BE1				55A2
6BF1				55A3
E2AF				55A4
6C10				55A8
E6B9				55AC
6C33				55AD
6CE1	4B22	9641	04302	55AE
6E1A	3D6D	8F8D	02977	55AF
6C35				55B0
6DFC		FB45	11706	55B1
6CF4				55B2
6C97				55B3
E2B9				55B4
6C3F		FAF5	11687	55B5
6C4B				55B6
6C4D				55B7
6C4F				55B8
6C52				55B9
6C54				55BA
6C59				55BB
E2B1				55BC
6C5C		FAF6	11688	55BD
6C6D				55BE
6C6F		FAF8	11690	55BF
E2B3				55C0
6C78				55C1
6C85				55C2
6C86		FAF7	11689	55C3
6C89				55C4
6C94				55C5
6C95				55C6
E6BA				55C7
6CB2				55C8
6CB4				55C9
E2B4				55CA
6CC2				55CB
6CC6				55CC
E6BB				55CD
6CCF				55CE
6CD1				55CF
6CD2				55D0
6CD4				55D1
6CD6				55D2
6CDA		FAF9	11691	55D3
6CDC				55D4
E6BC				55D5
6CE0				55D6
E6BD				55D7
6CE7				55D8
E2B6				55D9
6CE9				55DA
6CEB				55DB
6CEE				55DC
E071				55DD
6D04		FAFA	11692	55DE
6D07				55DF
E072				55E0
6D0A				55E1
6D0E				55E2
6D0F				55E3
6D11				55E4
6D13				55E5
E073				55E6
E6BE				55E7
E6BF				55E8
6D26				55E9
6D27				55EA
6D2E				55EB
E074				55EC
6D39				55ED
6D3C				55EE
E2B8				55EF
6D3F				55F0
E6C0				55F1
E6C1				55F2
6D5E				55F3
6D5F				55F4
6D61				55F5
6D65				55F6
6D67				55F7
E2BA				55F8
E2BB				55F9
6D70				55FA
FA46				55FB
E2BC				55FC
6D7C				55FD
E6C2				55FE
E6C3				56A1
6D82				56A2
6D87		FAFB	11693	56A3
E2BD				56A4
E6C4				56A5
6D94				56A6
6D96		FB40	11701	56A7
6D97				56A8
6D98				56A9
E12E				56AA
E2BE				56AB
E6C5				56AC
6DAC		FB41	11702	56AD
E2C0				56AE
6DBD				56AF
6DBF				56B0
6DC4				56B1
E2C1				56B2
E6C6				56B3
6DDA				56B4
E075				56B5
6DE5				56B6
6DE9				56B7
E6C7				56B8
E2C2				56B9
E076				56BA
6E00				56BB
6E04				56BC
6E22				56BD
6E27		FB48	11709	56BE
E2C4				56BF
E6C8				56C0
6E36				56C1
6E39		FB46	11707	56C2
6E3C		FB49	11710	56C3
E6C9				56C4
E6CA				56C5
6E44				56C6
6E45				56C7
6E48				56C8
6E4F				56C9
6E51				56CA
6E53				56CB
6E54				56CC
E6CB				56CD
6E57				56CE
6E5C		FB47	11708	56CF
6E5E				56D0
6E62				56D1
6E63				56D2
6E68				56D3
6E73				56D4
E6CC				56D5
E077				56D6
6E8D				56D7
6E93				56D8
E6CD				56D9
6E99				56DA
E6CE				56DB
E6CF				56DC
6EB1				56DD
E6D0				56DE
6EBF		FB4A	11711	56DF
6EC3				56E0
6EC7				56E1
6EC8				56E2
6ECA				56E3
E2C9				56E4
E6D1				56E5
FA48				56E6
6EED				56E7
6EEE				56E8
E2CC				56E9
6EF9				56EA
6F04				56EB
6F08				56EC
6F0C				56ED
6F0D				56EE
E2D0				56EF
6F1A				56F0
E6D2				56F1
6F33				56F2
6F3C				56F3
E2CF				56F4
6F5A				56F5
6F61				56F6
6F62				56F7
E6D3				56F8
6F6C				56F9
E2D1				56FA
E2D2				56FB
E079				56FC
E2D3				56FD
6F88		FB4B	11712	56FE
6F8B				57A1
6F8C				57A2
6F8D				57A3
6F90				57A4
6F92				57A5
6F97	3442	8AC0	02034	57A6
6FA7				57A7
E6D4				57A8
E6D5				57A9
6FAE				57AA
6FAF				57AB
E07A				57AC
6FB0				57AD
6FB5		FB4C	11713	57AE
6FB6				57AF
E07B				57B0
E2D4				57B1
6FC7				57B2
E07C				57B3
6FDA				57B4
E6D6				57B5
E2D6				57B6
E07D				57B7
6FF5		FB4D	11714	57B8
7005		FB4E	11715	57B9
E2D9				57BA
E2DD				57BB
E6D7				57BC
E6D8				57BD
E6D9				57BE
E07E				57BF
7048				57C0
7054				57C1
705D				57C2
704E				57C3
7064				57C4
E12D				57C5
6E9D	3942	8D61	02534	57C6
6FEF	4275	91F3	03485	57C7
6DEB	307C	88FA	01692	57C8
6EA2	306E	88EC	01678	57C9
706C				57CA
E6DA				57CB
71DA				57CC
7192				57CD
706E				57CE
E6E0				57CF
7097				57D0
7085		FB51	11718	57D1
7138				57D2
E6DD				57D3
71D3				57D4
E081				57D5
E6DB				57D6
7076				57D7
E07F				57D8
7086				57D9
7094				57DA
7095				57DB
7098				57DC
70A4				57DD
70AB		FB52	11719	57DE
70B4				57DF
70CA				57E0
70D1				57E1
70D3				57E2
70D8				57E3
70DC				57E4
70E4				57E5
7107				57E6
711E				57E7
E6DC				57E8
712F				57E9
7141				57EA
7146		FB56	11723	57EB
7147		FB57	11724	57EC
714A				57ED
714B				57EE
7152				57EF
E2E3				57F0
715C		FB55	11722	57F1
7168				57F2
E6DE				57F3
7180				57F4
718C				57F5
E2E6				57F6
719A				57F7
71B3				57F8
71BA				57F9
71CB				57FA
71CC				57FB
71D7	5F73	E093	06383	57FC
7200				57FD
E2EA				57FE
E6DF				58A1
E2EB				58A2
71C4				58A3
E080				58A4
7104		FB54	11721	58A5
710F		FB53	11720	58A6
FA49				58A7
7199	7426	EAA4	08406	58A8
71FE		FB5A	11727	58A9
707C	3C5E	8EDC	02862	58AA
714E	4079	90F7	03289	58AB
717D	407A	90F8	03290	58AC
7149	4E7B	97F9	04691	58AD
E2E9				58AE
91C7	3A53	8DD1	02651	58AF
E2EC				58B2
E2EE				58B3
E2F0				58B4
724C	4757	9476	03955	58BF
7253				58C0
7255				58C1
E7C8				58C5
729B				58C9
72A8				58CA
725E				58CB
7263				58CC
E082				58CD
7277				58CE
727B				58CF
72AD				58D2
E083				58D3
E2F9				58D4
7358				58D5
72BE		FB5C	11729	58D6
E6E1				58D7
E088				58D8
7343				58D9
72B4				58DA
72C1				58DB
72C7				58DC
72C9				58DD
E084				58DE
72CC				58DF
E085				58E0
E6E2				58E1
72D5				58E2
72D6				58E3
E086				58E4
72DF				58E5
72E5				58E7
E087				58E8
72FA				58E9
E089				58EA
7307				58EB
E6E3				58EC
7324		FB5D	11730	58ED
7328				58EE
FA16		FB5E	11731	58EF
E2F6				58F0
E2F8				58F1
734D				58F2
7350				58F3
7360				58F4
736E				58F5
7377		FB5F	11732	58F6
737C				58F7
7381				58F8
7337	4D32	9751	04518	58F9
72B1		FB5B	11728	58FA
7386				58FC
E2FA				58FD
7385				58FE
E7C9				59A1
7476	6076	E0F4	06486	59A2
73CA	3B39	8E58	02725	59A3
E6E6				59A4
73CF				59A5
7439				59A6
E308				59A7
738E				59A8
E6E4				59A9
7390				59AA
E6E5				59AB
E2FB				59AC
7395				59AD
7398				59AE
739C				59AF
739E				59B0
E2FC				59B1
73A2				59B2
E08A				59B3
73A6				59B4
73AD				59B5
73A5				59B6
73B5				59B7
73B7				59B8
73B9				59B9
73BD		FB60	11733	59BA
73BF				59BB
73C9		FB61	11734	59BC
E2FE				59BD
73CB				59BE
73D3				59BF
73D6		FB62	11735	59C0
73D9				59C1
E6E7				59C2
73E3		FB63	11736	59C3
73E7				59C4
73E9				59C5
73F4				59C6
73F5		FB66	11739	59C7
73F7				59C8
73F9				59C9
73FA				59CA
E302				59CB
73FF				59CC
7401				59CD
E6E8				59CE
7404				59CF
7407		FB65	11738	59D0
E6E9				59D1
741A				59D2
741B				59D3
E6EA				59D4
E6EB				59D5
7424				59D6
7426		FB67	11740	59D7
7428				59D8
7429		FB69	11742	59D9
742A		FB68	11741	59DA
742B				59DB
742D				59DC
742E		FB6A	11743	59DD
742F				59DE
E304				59DF
7443				59E0
7444				59E1
E6EC				59E2
7446				59E3
7447				59E4
E306				59E5
744D				59E6
E307				59E7
7452				59E8
7457				59E9
745D				59EA
E08B				59EB
7467				59EC
7468				59ED
7471				59EE
7472				59EF
7480				59F0
7481				59F1
E6ED				59F2
7487				59F3
7489		FB6C	11745	59F4
7492				59F5
E6EE				59F6
7498				59F7
7499				59F8
749A				59F9
749C				59FA
749F		FB6D	11746	59FB
74A1				59FC
E309				59FD
74A6				59FE
74A8				5AA1
E30B				5AA2
74AB				5AA3
E30A				5AA4
74B2				5AA5
74B9				5AA6
74BF				5AA7
74C8				5AA8
74C9				5AA9
74DA				5AAA
73CE	605F	E0DD	06463	5AAB
73D2		FB64	11737	5AAC
E7CA				5AAD
E08C				5AAF
74DE				5AB0
E08D				5AB4
74FA				5AB5
7511	3979	8D99	02589	5AB6
7504	612B	E14A	06511	5AB7
E7CB				5AB8
7521				5ABD
7527				5ABE
E6EF				5ABF
757E				5AC7
7540				5AC8
753E				5AC9
7550				5ACA
E6F0				5ACB
E08E				5ACC
7547				5ACD
7548				5ACE
E312				5ACF
756F		FB6F	11748	5AD0
757C				5AD1
757D				5AD2
7581				5AD3
E08F				5AD4
7585				5AD5
757A				5AD6
E311				5AD7
7592				5ADE
75A4				5ADF
75BA				5AE0
75C0				5AE1
75CE				5AE2
75EC				5AE3
E6F1				5AE4
E314				5AE5
7602				5AE6
7604				5AE7
760C				5AE8
7616				5AE9
E315				5AEA
7625				5AEB
762D				5AEC
7633				5AED
7635				5AEE
7645				5AEF
E6F2				5AF0
7665				5AF1
766D				5AF2
766F				5AF3
7652	4C7E	96FC	04494	5AF4
75E9	4169	9189	03373	5AF5
E090				5AFB
7695				5AFC
769B		FB73	11752	5AFD
E6F3				5AFE
E317				5BA1
768C				5BA2
E091				5BA3
769C		FB71	11750	5BA4
E6F4				5BA5
769F				5BA6
76A5				5BA7
76A2				5BA8
76A6		FB74	11753	5BA9
76A7				5BAA
76AD				5BAB
7693	622B	E1A9	06611	5BAC
76C1				5BB3
76C5				5BB4
E31A				5BB5
6637				5BB6
E3AD				5BB7
E31C				5BB8
E31D				5BBC
7722				5BBD
77A2				5BBE
76F1				5BBF
76FC				5BC0
770E				5BC1
7712				5BC2
7719				5BC3
771C				5BC4
E6F5				5BC5
772D				5BC6
772E				5BC7
7734				5BC8
7736				5BC9
7739				5BCA
E6F6				5BCB
7746		FB76	11755	5BCC
E092				5BCD
774D				5BCE
774E				5BCF
7756				5BD0
7757				5BD1
775C				5BD2
775E				5BD3
775F				5BD4
7762				5BD5
7770				5BD6
7772				5BD7
7773				5BD8
7774				5BD9
E093				5BDA
777D				5BDB
E094				5BDC
779A				5BDD
E31F				5BDE
77B1				5BDF
E6F7				5BE0
776A				5BE1
7795				5BE2
77A5	4A4D	95CB	04245	5BE3
77DE				5BE5
77E6				5BE9
77E4				5BEA
E095				5BEB
77E9	366B	8BE9	02275	5BEC
E096				5BEF
782E				5BF0
E6FA				5BF1
789E				5BF2
78DB				5BF3
E32A				5BF4
7931				5BF5
77F4				5BF6
77F8				5BF7
77FB				5BF8
7806				5BF9
E6F8				5BFA
7843				5BFB
7821		FB78	11757	5BFC
7822				5BFD
7823				5BFE
E097				5CA1
782D				5CA2
E322				5CA3
7835				5CA4
7844				5CA5
7848				5CA6
E326				5CA7
E323				5CA8
E6F9				5CA9
785C				5CAA
E324				5CAB
7860				5CAC
7863				5CAD
7864		FB7A	11759	5CAE
7868				5CAF
E325				5CB0
786E				5CB1
E327				5CB2
FA4C				5CB3
E6FB				5CB4
78A4				5CB5
78AC				5CB6
78AD				5CB7
78C8				5CB8
E329				5CB9
78CE				5CBA
78DF				5CBB
78E0				5CBC
78E1				5CBD
78E6				5CBE
E098				5CBF
78EA				5CC0
78F3				5CC1
7900				5CC2
78F6				5CC3
78F7				5CC4
78FA				5CC5
78FB				5CC6
7906				5CC7
7925				5CC8
7929				5CC9
E32B				5CCA
E6FC				5CCB
E6FD				5CCC
787C	626F	E1ED	06679	5CCD
E7CC				5CCE
7947	3540	8B5F	02132	5CCF
7941	3737	8C56	02323	5CD0
E330				5CD1
E331				5CD2
FA4D				5CD3
FA4F				5CD4
FA4E				5CD5
FA50				5CD6
FA51				5CD7
795C				5CD8
FA52				5CD9
FA19		FB7E	11763	5CDA
E32F				5CDB
FA1A		FB80	11764	5CDC
E099				5CDD
7972				5CDE
7979				5CDF
797B				5CE0
798B				5CE1
FA53				5CE2
FA54				5CE3
FA1B		FB82	11766	5CE4
7993				5CE5
7994		FB81	11765	5CE6
7998				5CE7
799B		FB83	11767	5CE8
E6FE				5CE9
79A8				5CEA
E332				5CEB
7977	4578	9398	03788	5CEC
79B0	4729	9448	03909	5CED
79B8				5CEE
79C2				5CF2
E773				5CF3
FA55				5CF4
E6FF				5CF5
E09A				5CF6
79C7				5CF7
E09B				5CF8
79D4				5CF9
79D6				5CFA
79DA				5CFB
79DE				5CFC
79E2				5CFD
79EA				5CFE
E700				5DA1
79F1				5DA2
E701				5DA3
E09C				5DA4
79F8				5DA5
79FC				5DA6
7A02				5DA7
7A03				5DA8
E702				5DA9
E09D				5DAA
7A0A				5DAB
E335				5DAC
7A0C				5DAD
E667				5DAE
7A11				5DAF
7A1E				5DB0
E703				5DB1
E09E				5DB2
E338				5DB3
E09F				5DB4
E0A0				5DB5
7A2B				5DB6
E0A1				5DB7
7A34				5DB8
7A38				5DB9
7A39				5DBA
7A45				5DBB
7A47				5DBC
7A59				5DBD
7A5C				5DBE
7A5D				5DBF
7A5F				5DC0
7A60				5DC1
7A65				5DC2
7A67				5DC3
7A6A				5DC4
7A6D				5DC5
79E4	4769	9489	03973	5DC6
7A17	4923	9542	04103	5DC7
7A31	634A	E269	06742	5DC8
E33A				5DCB
E33C				5DCC
E3B4				5DCD
7A86				5DCE
7A8A				5DCF
E733				5DD0
7A91				5DD1
E704				5DD2
7AA0				5DD3
7AB3				5DD4
7AB9				5DD5
7ABB				5DD6
E7CD				5DD7
E33F				5DDA
E340				5DDB
E0A3				5DDC
7AD1		FB84	11768	5DDD
7ADB				5DDE
E0A2				5DDF
E705				5DE0
E706				5DE1
7AE8				5DE2
7AE9				5DE3
7AEB		FB87	11771	5DE4
FA1C		FB86	11770	5DE5
7AEC				5DE6
7AF1				5DE7
E707				5DE8
E7CE				5DEA
7BAD	407D	90FB	03293	5DEB
7AFD				5DEC
7B1F				5DED
7B23				5DEE
7B27				5DEF
7B2B				5DF0
7B2F				5DF1
E708				5DF2
7B3F				5DF3
7B40				5DF4
7B4E				5DF5
E0A4				5DF6
7B60				5DF7
7B64				5DF8
7B66				5DF9
7B69				5DFA
7B6A				5DFB
7B6D				5DFC
7B6F				5DFD
7B91				5DFE
7B96				5EA1
7B9B				5EA2
7B9E		FB88	11772	5EA3
7BA0				5EA4
7BA5				5EA5
7BB2				5EA6
E0A5				5EA7
7BBB				5EA8
7BBD				5EA9
E344				5EAA
E0A6				5EAB
E0A7				5EAC
7BD4				5EAD
7BD7				5EAE
7BD9				5EAF
7BDA				5EB0
E0A8				5EB1
7BEA				5EB2
7BFE				5EB3
E34A				5EB4
7C0F				5EB5
7C1B				5EB6
7C25				5EB7
E0A9				5EB8
E0AA				5EB9
E0AB				5EBA
E70A				5EBB
E34C				5EBC
7C5C				5EBD
7C63				5EBE
7C6D				5EBF
7C70				5EC0
E63E				5EC1
7BAA	433D	925C	03529	5EC2
7BB8	4824	94A2	04004	5EC3
7BC7	4A53	95D1	04251	5EC4
7C3E	4E7C	97FA	04692	5EC5
7B99	6439	E2B7	06825	5EC6
7CA6				5EC7
7C7C				5EC8
E350				5EC9
7C86				5ECA
E352				5ECB
6549				5ECC
E70B				5ECD
7CA0				5ECE
E0AC				5ECF
7CB6				5ED0
7CB7				5ED1
7CBA				5ED2
FA1D		FB89	11773	5ED3
7CBF				5ED4
E0AD				5ED5
7CD5				5ED6
E355				5ED7
7CD9				5ED8
7CDD				5ED9
7CE6				5EDA
7CE9				5EDB
E0AE				5EDC
7C7E	4C62	96E0	04466	5EDD
7C90	6464	E2E2	06868	5EDE
7D5C		FB8B	11775	5EE0
7DA6				5EE1
7E08				5EE2
FA5A				5EE3
7E44				5EE4
7E8D				5EE5
7E47				5EE6
E70C				5EE7
7D08				5EE8
7D09				5EE9
7D03				5EEA
E357				5EEB
E358				5EEC
7D0F				5EED
7D12				5EEE
7D13				5EEF
7D16				5EF0
E0AF				5EF1
E359				5EF2
E35A				5EF3
7D1D				5EF4
7D23				5EF5
7D31				5EF6
E70D				5EF7
7D3E				5EF8
7D40				5EF9
7D41				5EFA
E35D				5EFB
7D47				5EFC
7D48		FB8A	11774	5EFD
7D4D				5EFE
7D51				5FA1
E0B0				5FA2
7D53				5FA3
E70E				5FA4
E70F				5FA5
7D59				5FA6
E35E				5FA7
7D5D				5FA8
E35F				5FA9
7D65				5FAA
E0B1				5FAB
E360				5FAC
7D7A				5FAD
7D7F				5FAE
E361				5FAF
E363				5FB0
7D88				5FB1
7D8B				5FB2
7D8C				5FB3
7D8D				5FB4
7D9E				5FB5
E710				5FB6
E711				5FB7
7DA7				5FB8
7DB6				5FB9
E712				5FBA
7DC3				5FBB
7DC4				5FBC
E713				5FBD
E714				5FBE
E369				5FBF
7DE2				5FC0
E36A				5FC1
E36C				5FC2
E36D				5FC3
7DEB				5FC4
7DED				5FC5
E36E				5FC6
7DF1				5FC7
FA58				5FC8
E715				5FC9
7E11				5FCA
7E15				5FCB
E0B2				5FCC
E370				5FCD
7E1C				5FCE
7E1D				5FCF
E372				5FD0
7E2F				5FD1
7E33				5FD2
7E3F				5FD3
7E50				5FD4
E0B3				5FD5
7E52		FB8F	11779	5FD6
E374				5FD7
7E6E				5FD8
7E6F				5FD9
7E73				5FDA
7E7E				5FDB
7E8A		FA5C	11529	5FDC
7E91				5FDD
7E98				5FDE
7E4B	3752	8C71	02350	5FDF
7E4D	3D2B	8F4A	02911	5FE0
E7CF				5FE1
7D9F	6546	E365	06938	5FE2
7DAE	653B	E35A	06927	5FE3
7DB7		FB8C	11776	5FE4
530B				5FE5
7F44				5FE6
7F3C				5FE7
7F47		FB90	11780	5FE8
7F4F				5FE9
7F52				5FEC
7F5D				5FED
7F61				5FEE
7F63				5FEF
FA5B				5FF0
7FBC				5FF4
E375				5FF5
7F97				5FF6
E376				5FF7
E377				5FF8
7FA2				5FF9
7FAD				5FFA
E716				5FFB
7FD4	6646	E3C4	07038	5FFD
7FE0	3F69	9089	03173	5FFE
FA1E				60A1
7FBF				60A2
E37A				60A3
7FCF				60A4
E37B				60A5
7FDF				60A6
7FEC				60A7
7FEF				60A8
E381				60A9
E378				60AA
7FCE				60AB
E717				60AC
7FEE				60AD
7FFA				60AE
7FFD				60AF
7FF2				60B0
7FFF				60B1
7FF0	344D	8ACB	02045	60B2
7FEB	3465	8AE3	02069	60B3
FA5C				60B5
800A				60B6
800D				60BA
800E				60BB
8011				60BC
8013				60C0
8014				60C1
801D				60C2
801E				60C3
801F				60C4
8020				60C5
8024				60C6
8026				60C7
E384				60CB
E383				60CC
8039				60CD
803E				60CE
E718				60CF
8066				60D0
8075				60D1
E386				60D2
E0B4				60D3
E387				60D7
81E4				60DF
FA5D				60E3
E71F				60E4
81EB				60E5
81F8				60E9
E0BB				60EA
81F5				60EB
81F6				60EC
E0BC				60F4
E394				60F8
822E	6764	E484	07168	60FE
8221				61A1
8228				61A2
8234				61A3
E0BD				61A4
823A				61A5
E0BE				61A6
8245				61A7
8246				61A8
E397				61A9
824B				61AA
824F				61AB
E0BF				61AC
8263				61AD
8267				61AE
82A6	3032	88B0	01618	61B3
8328	3071	88EF	01681	61B4
845B	336B	8A8B	01975	61B5
85A9	3B27	8E46	02707	61B6
85AF	3D72	8F92	02982	61B7
85F7	3D73	8F93	02983	61B8
848B	3E55	8FD3	03053	61B9
E0C0				61BA
827F				61BB
8283				61BC
8287				61BD
828A				61BE
828E				61BF
8296				61C0
8298				61C1
829A				61C2
829B				61C3
82A0				61C4
82A1				61C5
82A3				61C6
82A7				61C7
82A8				61C8
82AE				61C9
82B0				61CA
E39D				61CB
82B7				61CC
82BE				61CD
82BF				61CE
E39B				61CF
82C6				61D0
82D0				61D1
82DA				61D2
82E2				61D3
82E4				61D4
82EA				61D5
82FD				61D6
82FE				61D7
8300				61D8
8301		FB93	11783	61D9
8307				61DA
830A				61DB
E0C1				61DC
8354				61DD
E0C2				61DE
E0C3				61DF
E721				61E0
E720				61E1
832C				61E2
E3A1				61E3
8333				61E4
8337				61E5
833D				61E6
8342				61E7
8343				61E8
8344				61E9
834A	3755	8C74	02353	61EA
8347				61EB
8351				61EC
E0C4				61ED
8378				61EE
837D				61EF
837F		FB95	11785	61F0
8386				61F1
8392				61F2
E0C5				61F3
8398				61F4
83A7				61F5
83A9				61F6
83AC				61F7
E0C6				61F8
E0C7				61F9
E3A4				61FA
83BE				61FB
83BF				61FC
83C0				61FD
83C7		FB96	11786	61FE
83C9				62A1
83CF				62A2
83D1				62A3
83D4				62A4
E3A5				62A5
E723				62A6
83EA				62A7
E3A6				62A8
E724				62A9
83F6		FB97	11787	62AA
83F9				62AB
8401				62AC
840F				62AD
8411				62AE
E725				62AF
8447				62B0
844A				62B1
844F				62B2
FA5F				62B3
845A				62B4
E726				62B5
8465				62B6
8470				62B7
8473				62B8
8474				62B9
8478				62BA
E727				62BB
849E				62BC
E3AC				62BD
84AA				62BE
84C8				62BF
E0C8				62C0
84CF				62C1
84D3				62C2
E0C9				62C3
84DC		FA60	11533	62C4
84F0				62C5
8532				62C6
E3B1				62C7
851E				62C8
852F				62C9
E0CA				62CA
8534				62CB
E0CB				62CC
E728				62CD
E0CC				62CE
854F				62CF
8551				62D0
8553		FB9A	11790	62D1
E3B2				62D2
E0CD				62D3
8559		FB9B	11791	62D4
E0CE				62D5
855C				62D6
855E				62D7
855F				62D8
856B		FB9C	11792	62D9
E729				62DA
E0CF				62DB
E3B3				62DC
8585				62DD
8586				62DE
E3B5				62DF
858F				62E0
859D				62E1
85A5				62E2
85AD				62E3
FA1F		FB9D	11793	62E4
E72A				62E5
85BC				62E6
85BE				62E7
85BF				62E8
85C7				62E9
85CE				62EA
85D8				62EB
85DF				62EC
E3BA				62ED
85F6				62EE
85FF				62EF
8604				62F0
E3BE				62F1
E3BD				62F2
8629				62F3
8618				62F4
8627				62F5
E3BF				62F6
E3C0				62F7
E0D0				62F8
E0D1				62F9
84AF				62FA
E3B9				62FB
83DF	4551	9370	03749	62FC
853D	4A43	95C1	04235	62FD
E7D0				62FE
7E6D	4B7A	969A	04390	63A1
83B1	4D69	9789	04573	63A2
84EE	4F21	9840	04701	63A3
84EC	4B29	9648	04309	63A4
82D2	6772	E492	07182	63A5
E3C1				63A6
8656				63A7
8657				63A8
8658				63A9
E72B				63AA
F936				63AB
E3C4				63AC
E3C3				63AD
E0D2				63AE
E0D3				63AF
8662				63B0
8663				63B1
E0D4				63B2
8749	4066	90E4	03270	63B3
86F8	427D	91FB	03493	63B4
874B	4F39	9858	04725	63B5
87CA				63B6
882D				63B7
E3C5				63B8
87BF				63B9
87FF				63BB
E3D3				63BC
881C				63BD
E3D0				63BE
8832				63BF
8803				63C0
867A				63C1
866C				63C2
8675				63C3
E3C7				63C4
868D				63C5
E0D5				63C6
86A8				63C7
86B1				63C8
86C5				63C9
86FA				63CA
86FD				63CB
8705				63CC
870B				63CD
8713				63CE
8719				63CF
8728				63D0
872E				63D1
8758				63D2
E3CE				63D3
E3CC				63D4
8785				63D5
8790				63D6
87AD				63D7
87D9				63D8
87DF				63D9
87E3				63DA
87ED				63DB
880B				63DC
881B				63DD
87D2	6A3D	E5BB	07429	63DE
E7D1				63DF
E7D2				63E0
E0D6				63E1
8845				63E2
884A				63E3
E3D6				63E7
8855				63E8
885A				63E9
885C				63EA
885F				63EB
8860				63EC
E0D7				63ED
8910	336C	8A8C	01976	63EE
8956	3228	89A6	01808	63EF
8864				63F0
88CA				63F1
E211				63F2
8930				63F3
E731				63F4
E3DA				63F5
E72D				63F6
E0D8				63F7
88DB				63F8
88F5		FBA2	11804	63F9
8879				63FA
E72E				63FB
E72F				63FC
E730				63FD
889A				63FE
889B				64A1
889F				64A2
88A8				64A3
88AA				64A4
88BA				64A5
88C0				64A6
88CD				64A7
88CE				64A8
88DE				64A9
88EF				64AA
88F0				64AB
8906				64AC
891A				64AD
8932				64AE
8939				64AF
8946				64B0
E0D9				64B1
895C				64B2
897B				64B3
890A	6A6F	E5ED	07479	64B4
891C		FA5D	11530	64B5
E3DB				64B7
E0DA				64B8
E3DD				64B9
E3DC				64BA
E3DE				64BB
E3E0				64BC
898D				64BF
8994				64C0
E0DD				64C1
E0DB				64C2
E0DC				64C3
FA61				64C4
89A0				64C5
E0DE				64C6
89B0				64C7
89B4				64C8
89B5				64C9
89BC				64CA
E3E1				64CB
89AF	6B32	E651	07518	64CC
89F3				64CF
89D4				64D0
89D6				64D1
89E5				64D2
89EB				64D3
E0DF				64D4
8AFA	3841	8CBF	02433	64D5
8A6E	4127	9146	03307	64D6
8B0E	4666	93E4	03870	64D7
8A3E				64D8
8A67				64D9
8A79		FBA5	11807	64DA
8AEC				64DB
8B37				64DC
8B7C				64DD
8B8B				64DE
E3F5				64DF
E0E0				64E0
E0E1				64E1
8A05				64E2
8A11				64E3
8A12		FBA3	11805	64E4
8A15				64E5
8A1F	3E59	8FD7	03057	64E6
8A20				64E7
8A22				64E8
8A24				64E9
8A2B				64EA
8A35				64EB
8A37		FBA4	11806	64EC
8A3D				64ED
E732				64EE
8A45				64EF
8A47				64F0
8A4D				64F1
8A4E				64F2
8A53				64F3
E3E7				64F4
8A56				64F5
8A57				64F6
8A58				64F7
8A5C				64F8
8A5D				64F9
8A61				64FA
8A65				64FB
E3E9				64FC
8A75				64FD
8A7F				64FE
8A86				65A1
8A8F				65A2
E3EA				65A3
8A97				65A4
8A9F				65A5
E3EB				65A6
8AA7		FBA6	11808	65A7
E734				65A8
8AB6				65A9
E0E2				65AA
E0E3				65AB
E3ED				65AC
E0E4				65AD
8AC9				65AE
8ACA				65AF
E3EE				65B0
8AD7				65B1
8ADD				65B2
8ADF		FBA8	11810	65B3
E3F0				65B4
E3F1				65B5
8AF6		FBAA	11812	65B6
FA22		FBA9	11811	65B7
E736				65B8
8AFC				65B9
8AFF				65BA
FA62				65BB
8B06				65BC
E3F3				65BD
E3F6				65BE
8B1C				65BF
8B1F				65C0
8B0A				65C1
8B2D				65C2
8B30				65C3
FA63				65C4
8B46				65C5
8B52				65C6
8B53		FBAB	11813	65C7
E0E5				65C8
8B54				65C9
E3F8				65CA
8B59				65CB
8B63				65CC
8B6A				65CD
E3F9				65CE
8B78				65CF
8B7E				65D0
E738				65D1
8B9C				65D2
8B9E				65D3
8B2C	4935	9554	04121	65D4
8ADE	6B66	E686	07570	65D5
8C39				65D8
E0E6				65D9
8C45				65DA
8C4B				65DE
8C51				65DF
8C54				65E0
8C47				65E1
8C49				65E2
E3FE				65E6
8C5B				65E7
8C8B				65EC
8C93				65ED
E400				65F1
E401				65F2
E403				65F3
FA64				65F4
E0E9				65F5
8CE1				65F6
E407				65F7
8D12		FBAF	11817	65F8
E73B				65F9
8CF8				65FA
8CA4				65FB
E0E7				65FC
E739				65FD
8CB9				65FE
8CBA				66A1
E0E8				66A2
8CC9				66A3
E404				66A4
E405				66A5
8CD6				66A6
8CD9				66A7
E73A				66A8
8CEC				66A9
8CEF				66AA
8CF0		FBAD	11815	66AB
E408				66AC
8CFF				66AD
8D03				66AE
FA65				66AF
E73C				66B0
E40A				66B1
8CED	4552	9371	03750	66B2
8D65				66B5
8D73	6C62	E6E0	07666	66B8
E73D				66B9
E40B				66BA
8D7F				66BB
8D84				66BC
8D90				66BD
8D95				66BE
E0EA				66BF
8DA6				66C0
8DAB				66C1
8DAF				66C2
E7D3				66C3
E73E				66C6
8DB7				66C7
E0EB				66C8
8DCE				66C9
8DD1				66CA
8DD7				66CB
8DD9				66CC
E40C				66CD
8DEC				66CE
8DF1				66CF
8E04				66D0
8E0B				66D1
E0EC				66D2
8E14				66D3
8E16				66D4
8E20				66D5
8E22				66D6
8E26				66D7
8E27				66D8
8E37				66D9
8E39				66DA
8E41				66DB
8E62				66DC
8E6F				66DD
8E70				66DE
8E7A				66DF
E40E				66E0
8E9D				66E1
8DDA	6C69	E6E7	07673	66E2
8EAD				66E6
8EB3				66E7
E73F				66E8
8EC7				66E9
8EAF	366D	8BEB	02277	66EA
8ECF		FBB2	11820	66ED
8EF9				66EE
8EED				66EF
E415				66F0
8F17				66F1
8F1E				66F2
8F25				66F3
8F2D				66F4
8F36				66F5
E416				66F6
E417				66F7
E0ED				66F8
8F54				66F9
8F58				66FA
8FA1				67A1
FA66				67A3
8FB5				67A4
E419				67A5
E418				67A6
E41A				67A7
E41E				67A8
8FC6				67A9
E41F				67AA
8FCA				67AB
E423				67AC
E422				67AD
E420				67AE
E424				67AF
8FD5				67B0
E421				67B1
8FE4				67B2
8FE8				67B3
E426				67B4
E428				67B5
8FEE				67B6
8C63				67B7
8FF6				67B8
E42B				67B9
E42C				67BA
E42D				67BB
E42E				67BC
E42F				67BD
9004				67BE
E42A				67BF
9008				67C0
900C				67C1
E432				67C2
E436				67C3
E437				67C4
9018				67C5
E438				67C6
E430				67C7
E439				67C8
E431				67C9
E435				67CA
E43A				67CB
902F				67CC
902A				67CD
902D				67CE
E43D				67CF
E43E				67D0
E43C				67D1
9033				67D2
9034				67D3
9037				67D4
FA67				67D5
E0EE				67D6
E0EF				67D7
E43F				67D8
9044				67D9
E446				67DA
E442				67DB
E447				67DC
E448				67DD
E449				67DE
E441				67DF
E440				67E0
E443				67E1
905B				67E2
905D				67E3
E44B				67E4
E44E				67E5
E450				67E6
E451				67E7
9074				67E8
E453				67E9
E455				67EA
E454				67EB
E456				67EC
E45A				67ED
E45B				67EE
9085				67EF
E45C				67F0
E45D				67F1
908E				67F2
9090				67F3
8FEA	6D6C	E78C	07776	67F4
901D	4042	90C0	03234	67F5
961D				67F6
9099				67F7
90A2				67F8
E45F				67F9
90B0				67FA
90B4				67FB
E740				67FC
90BD				67FD
90C3				67FE
90C4				68A1
90C5				68A2
90C7				68A3
90DC				68A4
90DD				68A5
E741				68A6
90DF				68A7
FA26		FBB7	11825	68A8
E0F0				68A9
9106				68AA
9114				68AB
E464				68AC
9116				68AD
E0F1				68AE
9122				68AF
9127				68B0
912F				68B1
9131				68B2
E459				68B3
913A				68B4
E742				68B5
9148				68B6
E768				68B7
E769				68B8
E76A				68B9
9633				68BA
963D				68BB
E76B				68BC
9641				68BD
E107				68BE
9652				68BF
E490				68C0
9656				68C1
E108				68C2
9658				68C3
9661				68C4
966E				68C5
E109				68C6
9674				68C7
967C				68C8
967F				68C9
9682				68CA
9684				68CB
F9DC		FBE9	11875	68CC
9689				68CD
E492				68CE
E10A				68CF
E494				68D0
9696				68D1
E493				68D2
E495				68D3
969D		FBEB	11877	68D4
96A4				68D5
96A5				68D6
96A6				68D7
96A9				68D8
90A3	4661	93DF	03865	68D9
96AE				68DA
96AF		FBEC	11878	68DB
E106				68DC
E460				68DD
912D	4522	9341	03702	68DE
E7D4				68DF
E7D5				68E0
9194				68E1
E0F2				68E2
E467				68E3
E468				68E4
E0F3				68E5
9159				68E6
915B				68E7
9161				68E8
E469				68E9
9164				68EA
E46A				68EB
E46B				68EC
9167				68ED
E0F4				68EE
9174				68EF
E46C				68F0
E46D				68F1
E46E				68F2
9183				68F3
9185				68F4
9198				68F5
E46F				68F6
919E				68F7
91A1				68F8
91A6				68F9
E0F5				68FA
91AE				68FB
91B2				68FC
91BB				68FD
91BF				68FE
91A4	3E5F	8FDD	03063	69A1
9197	4830	94AE	04016	69A2
9306	3B2C	8E4B	02712	69A9
939A	444A	92C8	03642	69AA
9453	4C7A	96F8	04490	69AB
946B				69AC
928E				69AD
E74C				69AE
92BA				69AF
E47A				69B0
E752				69B1
E0FB				69B2
92FB		FBD9	11859	69B3
E756				69B4
936A				69B5
938F				69B6
93A3				69B7
E0FD				69B8
E765				69B9
E100				69BA
91D3				69BB
91D7		FBBB	11829	69BC
91D9				69BD
91DA		FBBA	11828	69BE
91DE		FBBC	11830	69BF
91E4		FBBF	11833	69C0
91E5		FBC0	11834	69C1
E743				69C2
E473				69C3
91EC				69C4
91ED		FBBD	11831	69C5
91EE		FBBE	11832	69C6
91F0				69C7
91F1				69C8
E474				69C9
91F7				69CA
9201				69CB
9205				69CC
9206		FBC1	11835	69CD
E0F6				69CE
9207				69CF
920A		FBC3	11837	69D0
9210		FBC2	11836	69D1
9212				69D2
E476				69D3
9216				69D4
E0F7				69D5
9218				69D6
E744				69D7
921D				69D8
E745				69D9
9228				69DA
E746				69DB
9233				69DC
9235				69DD
9236				69DE
9238				69DF
9239		FBCA	11844	69E0
923A		FBC4	11838	69E1
923C		FBC6	11840	69E2
923E				69E3
9240		FBC5	11839	69E4
E478				69E5
9246				69E6
924A				69E7
924E		FBC7	11841	69E8
924F				69E9
9251		FBC9	11843	69EA
E747				69EB
E748				69EC
E749				69ED
9259		FBC8	11842	69EE
925C				69EF
925D				69F0
9260				69F1
9261				69F2
9265				69F3
E0F8				69F4
E0F9				69F5
926E				69F6
9277		FBCD	11847	69F7
E479				69F8
9279				69F9
927C				69FA
927D				69FB
927F				69FC
E74A				69FD
9288		FA5F	11532	69FE
9289				6AA1
9292				6AA2
9297				6AA3
929F				6AA4
E74B				6AA5
92A4				6AA6
92A7		FBCC	11846	6AA7
92AB				6AA8
E0FA				6AA9
92B2				6AAA
E74D				6AAB
92B6				6AAC
92BB				6AAD
92BC				6AAE
92BD				6AAF
92BF				6AB0
92C0				6AB1
92C3				6AB2
E74E				6AB3
E74F				6AB4
92CB				6AB5
92CC				6AB6
92CE				6AB7
92D0		FBD2	11852	6AB8
E750				6AB9
92D3		FBD6	11856	6ABA
92D5		FBD4	11854	6ABB
92D7		FBD0	11850	6ABC
92D8				6ABD
92D9		FBD1	11851	6ABE
E751				6ABF
92DD				6AC0
92DF				6AC1
92E0		FBD5	11855	6AC2
92E5				6AC3
92E7		FBCF	11849	6AC4
E0FC				6AC5
E47B				6AC6
92EE				6AC7
92F0				6AC8
92F9		FA65	11538	6AC9
92FF		FBDC	11862	6ACA
9300				6ACB
9302		FBDE	11864	6ACC
E753				6ACD
9308				6ACE
931C				6ACF
931D		FBDD	11863	6AD0
931E		FBDB	11861	6AD1
931F				6AD2
9321		FBD8	11858	6AD3
9324				6AD4
9325		FBD7	11857	6AD5
9327				6AD6
9329				6AD7
9347				6AD8
9348		FA5E	11531	6AD9
9349				6ADA
E754				6ADB
9352				6ADC
9355				6ADD
9357		FBE0	11866	6ADE
935E				6ADF
E755				6AE0
9367				6AE1
936D				6AE2
936F				6AE3
9370		FBDF	11865	6AE4
E757				6AE5
E758				6AE6
937A				6AE7
937D				6AE8
9380				6AE9
9388				6AEA
938A				6AEB
938B				6AEC
E480				6AED
E759				6AEE
9392				6AEF
E47E				6AF0
E75A				6AF1
93A4		FBE1	11867	6AF2
E481				6AF3
93A8				6AF4
93BA				6AF5
E75B				6AF6
E75C				6AF7
93C4				6AF8
93C6		FBE2	11868	6AF9
93C7				6AFA
93C9				6AFB
93CB				6AFC
93CD				6AFD
E75D				6AFE
93DB				6BA1
93DC				6BA2
93DE		FBE3	11869	6BA3
93DF				6BA4
E75E				6BA5
E482				6BA6
93E2				6BA7
E0FE				6BA8
E0FF				6BA9
93E6				6BAA
E75F				6BAB
E760				6BAC
93F7				6BAD
93F8		FBE4	11870	6BAE
93FA				6BAF
93FB				6BB0
93FD				6BB1
E761				6BB2
9404				6BB3
9408				6BB4
9409				6BB5
E762				6BB6
940F				6BB7
E483				6BB8
9416				6BB9
E484				6BBA
E763				6BBB
E764				6BBC
9431		FBE5	11871	6BBD
9432				6BBE
9433				6BBF
9434				6BC0
943B				6BC1
943F				6BC2
943D				6BC3
E766				6BC4
9445		FBE6	11872	6BC5
9448		FBE7	11873	6BC6
944A				6BC7
9459				6BC8
945C				6BC9
E486				6BCA
9461				6BCB
E101				6BCC
9468				6BCD
E487				6BCE
E48A				6BCF
9472				6BD0
E488				6BD1
9267		FBCB	11845	6BD2
E7D6				6BD3
E7D7				6BD4
9579				6BD7
E102				6BD8
9588				6BDC
958C				6BDD
958D				6BDE
958E				6BDF
E48B				6BE0
E48C				6BE1
959D				6BE2
959F				6BE3
95A1				6BE4
E103				6BE5
95A9				6BE6
95AB				6BE7
E767				6BE8
E104				6BE9
8ABE		FBA7	11809	6BEA
E105				6BEB
95BD				6BEC
E48E				6BED
E48F				6BEE
95DA				6BEF
95DE				6BF0
E497				6BF8
E10B				6BF9
96BA				6BFA
E498				6BFB
96DE				6BFC
FA68				6BFD
8000	4D54	9773	04552	6BFE
E499				6CA3
E49B				6CA4
96FA				6CA5
E10C				6CA6
9702				6CA7
9709				6CA8
971A				6CA9
E10D				6CAA
E10E				6CAB
9733		FBED	11879	6CAC
E76D				6CAD
974F		FBF1	11883	6CAE
E76C				6CAF
9743		FBEF	11881	6CB0
E7D8				6CB1
E76E				6CB3
975B				6CB4
9755		FBF3	11885	6CB5
E10F				6CBC
9773				6CC1
979F				6CC2
97AE				6CC3
97B1				6CC4
97B8				6CC5
97BC				6CC6
E110				6CC7
97C1				6CC8
E4A5				6CC9
9784	3373	8A93	01983	6CCA
9774	3724	8C43	02304	6CCB
9798	3E64	8FE2	03068	6CCC
97DE				6CCD
97DB				6CCE
E4A6				6CCF
E4A7				6CD0
97EF				6CD4
E4A8				6CD8
E4AA				6CD9
E111				6CDA
97F9				6CDB
E4A9				6CDC
980A				6CE0
980D				6CE1
980E				6CE2
E4AB				6CE3
E112				6CE4
E113				6CE5
E114				6CE6
E4AE				6CE7
E76F				6CE8
982B				6CE9
E4AF				6CEA
9833				6CEB
9835				6CEC
9825				6CED
9847				6CEE
E771				6CEF
E115				6CF0
9853				6CF1
E116				6CF2
9857		FBF4	11886	6CF3
E4B0				6CF4
F9D0				6CF5
E117				6CF6
9865		FBF5	11887	6CF7
9866				6CF8
E4B1				6CF9
986C				6CFA
983B	4951	9570	04149	6CFB
985B	453F	935E	03731	6CFC
982C	4B4B	966A	04343	6CFD
98BB				6DA1
98BF				6DA2
98AD				6DA3
98B4				6DA4
98B8				6DA5
98BA				6DA6
E4B2				6DAB
E4B7				6DAC
995C				6DAD
98E1				6DAE
E772				6DAF
E118				6DB0
E4B3				6DB1
98EA				6DB2
FA2A		FBF6	11888	6DB3
FA2B		FBF7	11889	6DB4
E4B5				6DB5
E4B6				6DB6
9902				6DB7
9907				6DB8
E4BC				6DB9
9917				6DBA
E4BD				6DBB
9926				6DBC
9927		FBF8	11890	6DBD
FA2C		FBF9	11891	6DBE
9931				6DBF
9932				6DC0
9939				6DC1
993B				6DC2
993C				6DC3
9940				6DC4
994D				6DC5
994E				6DC6
9958				6DC7
98F4	303B	88B9	01627	6DC8
990C	3142	8961	01734	6DC9
9957	3642	8BC0	02234	6DCA
9910	3B41	8E60	02733	6DCB
8755	3F2A	9049	03110	6DCC
9905	4C5F	96DD	04463	6DCD
E775				6DCF
999B				6DD0
999F				6DD1
E119				6DD2
E774				6DD3
99A6				6DD4
E11C				6DD9
E777				6DDA
E779				6DDB
9A16				6DDC
9A41				6DDD
E4C5				6DDE
E77D				6DDF
E776				6DE0
99B9				6DE1
99BA				6DE2
E11A				6DE3
E4C1				6DE4
99D4				6DE5
99D9				6DE6
99DA				6DE7
E778				6DE8
99EB				6DE9
99FE				6DEA
9A02				6DEB
9A03				6DEC
E77A				6DED
9A10				6DEE
E77B				6DEF
E11B				6DF0
9A23				6DF1
9A24				6DF2
9A2E				6DF3
9A36				6DF4
9A4A				6DF5
E4C6				6DF6
9A56				6DF7
9A28	424D	91CB	03445	6DF8
9AAE				6DFB
9AB6				6DFC
9AC1				6DFD
9AD7				6DFE
E77E				6EA4
E11D				6EA5
9ADB				6EA6
9ADC		FC40	11901	6EA7
E11E				6EA8
E4C7				6EAC
9AF9				6EAD
9B02				6EAE
9B03				6EAF
9B08				6EB0
9B0B				6EB1
E11F				6EB2
9B0D				6EB3
9B1B				6EB4
E4C8				6EB8
E4CA				6EC4
E120				6EC5
9B48				6EC6
9C2F	3073	88F1	01683	6EC7
9BD6	3B2A	8E49	02710	6EC8
9C48	432D	924C	03513	6EC9
9B9D				6ECA
E4D6				6ECB
9B61				6ECC
E121				6ECD
9B66				6ECE
9B73				6ECF
9B75		FC41	11902	6ED0
9B77				6ED1
9B7F				6ED2
9B80				6ED3
9B84				6ED4
9B89				6ED5
9B8A				6ED6
9B8B				6ED7
9B8F		FC43	11904	6ED8
9B90				6ED9
9B9E				6EDA
9BB1		FC44	11905	6EDB
9BB2				6EDC
E4CF				6EDD
E4CE				6EDE
E4D0				6EDF
9BE5				6EE0
9BEA				6EE1
E4D4				6EE2
9C00		FC46	11907	6EE3
E122				6EE4
9C11				6EE5
9C1E				6EE6
9C23				6EE7
9C36				6EE8
9C58				6EE9
9C63				6EEA
9C77				6EEB
9C52	4B70	9690	04380	6EEC
9BF2	724E	E9CC	08246	6EED
E7D9				6EEE
9BBB		FC45	11906	6EEF
E124				6EF0
9D96				6EF1
9DB1				6EF2
E125				6EF3
9D0B				6EF4
9D02				6EF5
9D07	463E	93BC	03830	6EF6
E123				6EF7
9D18				6EF8
9D1E				6EF9
9D42				6EFA
E4DB				6EFB
9D63				6EFC
9D62				6EFD
9D6B		FC48	11909	6FA1
E4DD				6FA2
9D70		FC47	11908	6FA3
9D86				6FA4
9D93				6FA5
E77F				6FA6
9DA1				6FA7
9DB5				6FA8
9DC3				6FA9
9DE0				6FAA
9E02				6FAB
9E10				6FAC
9E15				6FAD
9E19		FC4A	11911	6FAE
9E1C				6FAF
9D0E	322A	89A8	01810	6FB0
9D60	3974	8D94	02584	6FB1
E7DA				6FB2
9E78	3834	8CB2	02420	6FB5
9EA4				6FB7
E126				6FB8
9E83				6FB9
9E98				6FBA
9E9B				6FBB
E4DE				6FBF
9EAC				6FC0
E4E1				6FC1
9EB3				6FC2
9EB9	396D	8D8D	02577	6FC3
9EBA	4C4D	96CB	04445	6FC4
9EAA	7351	EA70	08349	6FC5
E4E2				6FC6
E780				6FC7
E4E3				6FC8
E781				6FCC
9ECB				6FCD
E4E4				6FCE
6A2E	5C53	9ED1	06051	6FD3
E782				6FD6
9EE7				6FD7
9EDB	4263	91E1	03467	6FD8
9EFF				6FDF
9F0F				6FE3
9F14				6FE7
9F19				6FE8
9F1A				6FE9
9F1B				6FEA
9F2A				6FEE
9F2F				6FEF
E4E6				6FF3
9F3C				6FF4
9F41				6FF5
9F55				6FFD
9F69				6FFE
E4E7				70A4
9F96				70A5
9F91				70A6
9F92				70A7
9F94				70A8
E783				70A9
9F90				70AA
E127				70AB
9F97				70AC
E128				70B0
E129				70B1
9FA3				70B5
55B0	3674	8BF2	02284	70B6
54E8	3E25	8FA3	03005	70B7
564C	4139	9158	03325	70B8
E7DB				70B9
56A2	4739	9458	03925	70BA
5539	5330	9A4F	05116	70BB
5533	533A	9A59	05126	70BC
5632	535E	9A7D	05162	70BD
64E2	4527	9346	03707	70BE
633D	4854	94D2	04052	70BF
6369	5960	9D80	05764	70C0
6406	596C	9D8C	05776	70C1
E7DC				70C2
6A3D	432E	924D	03514	70C3
69CC	4448	92C6	03640	70C4
6962	466A	93E8	03874	70C5
6A0B	4875	94F3	04085	70C6
67CA	4922	9541	04102	70C7
6994	4F31	9850	04717	70C8
67A6	5B45	9E64	05937	70C9
67B4	5B4A	9E69	05942	70CA
689B	5B6B	9E8B	05975	70CB
688D	5B74	9E94	05984	70CC
68C8		FA64	11537	70CD
6766		FAE1	11667	70CE
68CF		FAE7	11673	70CF
E7DD				70D0
E7DE				70D1
69E2		FAEC	11678	70D2
6A30		FAED	11679	70D3
6C72	3562	8B82	02166	70D4
7026	4375	9295	03585	70D5
6EBA	452E	934D	03714	70D6
6D9C	4642	93C0	03834	70D7
701E	4654	93D2	03852	70D8
7058	4667	93E5	03871	70D9
6E8C	482E	94AC	04014	70DA
7015	494E	956D	04146	70DB
6F23	4E7A	97F8	04690	70DC
6E6E	5E50	9FCE	06248	70DD
6D6F		FAFC	11694	70DE
E7DF				70DF
7114	316B	898B	01775	70E0
70BB		FA62	11535	70E1
E7E0				70E2
E7E1				70E3
7962	472A	9449	03910	70E4
E7E2				70E5
8375	683B	E4B9	07227	70E6
E7E3				70E7
E7E4				70E8
E7E5				70E9
E7E6				70EA
9065	4D5A	9779	04558	70EB
906E	3C57	8ED5	02855	70EC
907C	4E4B	97C9	04643	70ED
8FC2	312A	8949	01710	70EE
8FE6	3260	89DE	01864	70EF
9017	3F60	9080	03164	70F0
9061	414C	916B	03344	70F1
9019	4767	9487	03971	70F2
903C	492F	954E	04115	70F3
8FC4	4B78	9698	04388	70F4
9041	465B	93D9	03859	70F5
8FBF	4329	9248	03509	70F6
905C	423D	91BB	03429	70F7
E7E7				70F8
E7E8				70F9
E7E9				70FA
E7EA				70FB
8FE9	4676	93F4	03886	70FC
E4E8				71A2
E4E9				71A4
E4EA				71A6
E4EB				71A8
E4EC				71AA
E4ED				71AB
E4EE				71AC
E4EF				71AD
E4F0				71AE
E4F1				71AF
E4F2				71B0
E4F3				71B1
E4F4				71B2
E4F5				71B3
E4F6				71B4
E4F7				71B5
E4F8				71B6
E4F9				71B7
E4FA				71B8
E4FB				71B9
E4FC				71BA
E4FD				71BB
E4FE				71BC
E4FF				71BD
E500				71BE
E501				71BF
E502				71C0
E503				71C1
E504				71C2
E505				71C4
E506				71C5
E507				71C6
E508				71C7
E509				71C8
E50A				71CA
E50B				71CB
E50C				71CC
E50D				71CD
E50E				71CE
E50F				71CF
E510				71D0
E511				71D2
E512				71D3
E513				71D5
E514				71D6
E515				71D8
E516				71D9
E517				71DB
E518				71DC
E519				71DE
E51A				71DF
E51B				71E0
E51C				71E1
E51D				71E2
E51E				71E4
E51F				71E6
E520				71E8
E521				71E9
E522				71EA
E523				71EB
E524				71EC
E525				71ED
E526				71EF
E527				71F0
E528				71F1
E529				71F2
E52A				71F3
E52B				71F7
E52C				71FA
E52D				71FB
E52E				71FC
E52F				71FD
E530				71FE
E531				72A2
E532				72A4
E533				72A6
E534				72A8
E535				72AA
E536				72AB
E537				72AD
E538				72AE
E539				72AF
E53A				72B1
E53B				72B3
E53C				72B5
E53D				72B6
E53E				72B7
E53F				72B8
E540				72B9
E541				72BA
E542				72BB
E543				72BD
E544				72BF
E545				72C0
E546				72C1
E547				72C4
E548				72C5
E549				72C6
E54A				72C8
E54B				72CA
E54C				72CB
E54D				72CC
E54E				72CD
E54F				72CE
E550				72CF
E551				72D2
E552				72D3
E553				72D5
E554				72D8
E555				72DB
E556				72DE
E557				72DF
E558				72E0
E559				72E1
E55A				72E2
E55B				72E6
E55C				72E8
E55D				72E9
E55E				72EA
E55F				72EB
E560				72EC
E561				72ED
E562				72F0
E563				72F1
E564				72F2
E565				72F3
E566				72FA
722B				72FB
E567				72FC
E568				72FD
793B				72FE
E569				73A2
E56A				73A6
E56B				73A8
E56C				73AB
E56D				73AD
E56E				73AF
E56F				73B5
E570				73B6
E571				73B9
E572				73BD
E573				73BF
E574				73C4
E575				73C5
E576				73C8
E577				73CA
E578				73CB
E579				73CE
E57A				73CF
E57B				73D2
E57C				73D5
E57D				73DE
E57E				73DF
E57F				73E2
E580				73E6
E581				73E8
E582				73E9
E583				73EB
E584				73ED
E585				73F0
E586				73F1
E587				73F2
E588				73F3
E589				73FA
E58A				73FB
E58B				73FC
E7EB				74A1
E7EC				74A2
E7ED				74A3
E7EE				74A4
E7EF				74A5
E58C				74A6
E7F0				74A7
E7F1				74A8
E7F2				74A9
E7F3				74AA
E7F4				74AB
E58D				74AD
E58E				74AF
E7F5				74B0
E7F6				74B1
E7F7				74B2
E7F8				74B3
E7F9				74B4
E58F				74B5
E590				74B9
E7FA				74BA
E7FB				74BB
E7FC				74BC
E7FD				74BD
E7FE				74BE
E7FF				74BF
E800				74C0
E801				74C1
E802				74C2
E803				74C3
E591				74C4
E804				74C5
E805				74C6
E806				74C7
E807				74C8
E808				74C9
E592				74CA
E809				74CB
E80A				74CC
E80B				74CD
E80C				74CE
E593				74CF
E80D				74D0
E80E				74D1
E594				74D2
E80F				74D3
E810				74D4
E811				74D5
E812				74D6
E813				74D7
E814				74D8
E815				74D9
E816				74DA
E817				74DB
E818				74DC
E819				74DD
E595				74DE
E81A				74DF
E81B				74E0
E81C				74E1
E81D				74E2
E81E				74E3
E81F				74E4
E820				74E5
E821				74E6
E822				74E7
E823				74E8
E824				74E9
E825				74EA
E596				74EB
E826				74EC
E827				74ED
E828				74EE
E829				74EF
E82A				74F0
E82B				74F1
E597				74F2
E82C				74F3
E82D				74F4
E82E				74F5
E82F				74F6
E830				74F7
E831				74F8
E832				74F9
E833				74FA
E834				74FB
E835				74FC
E836				74FD
E837				74FE
E838				75A1
E839				75A2
E83A				75A3
E83B				75A4
E83C				75A5
E83D				75A6
E83E				75A7
E598				75AD
E83F				75AE
E840				75AF
E841				75B0
E842				75B1
E843				75B2
E844				75B3
E845				75B4
E846				75B5
E847				75B6
E848				75B7
E849				75B8
E84A				75B9
E84B				75BA
E84C				75BB
E84D				75BC
E84E				75BD
E84F				75BE
E599				75CF
E59A				75F2
FFE4		FA55	11522	76A1
FF07		FA56	11523	76A2
FF02		FA57	11524	76A3
FFE2	224C	81CA	00244	76A7
2235	2268	81E6	00272	76A8
2252	2262	81E0	00266	76A9
2261	2261	81DF	00265	76AA
25C1				76AE
25B7				76AF
21E9				76B0
21E7				76B1
21E6				76B2
21E8				76B3
25A2				76B4
2667				76B5
2661				76B6
2664				76B7
2662				76B8
339C				76BC
339D				76BD
339E				76BE
33A0				76BF
33A1				76C0
33A2				76C1
33A4				76C2
33A5				76C3
338E				76C4
338F				76C5
33C4				76C6
3397				76C7
2113				76C8
3398				76C9
33B3				76CA
33B2				76CB
33B1				76CC
33B0				76CD
3385				76CE
3386				76CF
3387				76D0
212B	2272	81F0	00282	76D1
33CB				76D2
2030	2273	81F1	00283	76D3
3390				76D4
33D4				76D5
33CC				76D6
E850				76D7
334D				76D9
3349				76DA
3322				76DB
3314				76DC
3316				76DD
3318				76DE
3315				76DF
3327				76E0
3303				76E1
3336				76E2
3326				76E3
3323				76E4
332B				76E5
330D				76E6
3357				76E7
3342				76E8
334A				76E9
3339				76EA
333B				76EB
3300				76EC
3331				76ED
3347				76EE
331E				76EF
332A				76F0
3305				76F1
E59B				76F4
337F				76F5
E59C				76F6
260E				76F7
3036				76F8
3020				76F9
E59D				76FA
E851				76FB
E852				76FC
E853				76FD
E854				76FE
E59E				77A9
2488				77AA
2489				77AB
248A				77AC
248B				77AD
248C				77AE
248D				77AF
248E				77B0
248F				77B1
2490				77B2
2474				77B4
2475				77B5
2476				77B6
2477				77B7
2478				77B8
2479				77B9
247A				77BA
247B				77BB
247C				77BC
247D				77BD
247E				77BE
247F				77BF
2480				77C0
2481				77C1
2482				77C2
2483				77C3
2484				77C4
2485				77C5
2486				77C6
2487				77C7
2460				77C9
2461				77CA
2462				77CB
2463				77CC
2464				77CD
2465				77CE
2466				77CF
2467				77D0
2468				77D1
2469				77D2
246A				77D3
246B				77D4
246C				77D5
246D				77D6
246E				77D7
246F				77D8
2470				77D9
2471				77DA
2472				77DB
2473				77DC
E855				77DD
2160		FA4A	11511	77DE
2161		FA4B	11512	77DF
2162		FA4C	11513	77E0
2163		FA4D	11514	77E1
2164		FA4E	11515	77E2
2165		FA4F	11516	77E3
2166		FA50	11517	77E4
2167		FA51	11518	77E5
2168		FA52	11519	77E6
2169		FA53	11520	77E7
2170		FA40	11501	77EB
2171		FA41	11502	77EC
2172		FA42	11503	77ED
2173		FA43	11504	77EE
2174		FA44	11505	77EF
2175		FA45	11506	77F0
2176		FA46	11507	77F1
2177		FA47	11508	77F2
2178		FA48	11509	77F3
2179		FA49	11510	77F4
E856				77F5
E857				77F6
E858				77F7
E859				77F8
E85A				77F9
E85B				77FA
E85C				77FB
E85D				77FC
E85E				77FD
E85F				77FE
249C				78A1
249D				78A2
249E				78A3
249F				78A4
24A0				78A5
24A1				78A6
24A2				78A7
24A3				78A8
24A4				78A9
24A5				78AA
24A6				78AB
24A7				78AC
24A8				78AD
24A9				78AE
24AA				78AF
24AB				78B0
24AC				78B1
24AD				78B2
24AE				78B3
24AF				78B4
24B0				78B5
24B1				78B6
24B2				78B7
24B3				78B8
24B4				78B9
24B5				78BA
3094				78C5
3243				78C9
323D				78CA
3239				78CB
323F				78CC
3234				78CD
3238				78CE
3232				78CF
3233				78D0
323C				78D1
3242				78D2
323E				78D3
3236				78D4
3231		FA58	11525	78D5
3235				78D6
323B				78D7
3240				78D8
323A				78D9
32B0				78DD
32AD				78DE
32A9				78DF
32AF				78E0
3294				78E1
32AA				78E2
3298				78E3
32AB				78E4
3292				78E5
3291				78E6
3293				78E7
32AC				78E8
32AE				78E9
3296				78EA
00E4				79A1
E860				79A2
00EB				79A3
E861				79A4
E862				79A5
00CF				79A6
00F6				79A7
00D6				79A8
00FC				79A9
00DC				79AA
00E1				79AB
00C1				79AC
00E9				79AD
00C9				79AE
00ED				79AF
00CD				79B0
00F3				79B1
00D3				79B2
00FA				79B3
00DA				79B4
00E0				79B5
E863				79B6
00E8				79B7
E864				79B8
E865				79B9
E866				79BA
E867				79BB
E868				79BC
00F9				79BD
E869				79BE
00E2				79BF
00C2				79C0
00EA				79C1
00CA				79C2
00EE				79C3
00CE				79C4
00F4				79C5
00D4				79C6
00FB				79C7
00DB				79C8
00F1				79C9
00D1				79CA
E86A				79CB
E86B				79CC
E86C				79CD
E86D				79CE
00E7				79CF
00C7				79D0
E86E				79D1
E86F				79D2
E870				79D3
E871				79D4
E872				79D5
E873				79D6
E874				79D7
0153				79D8
0152				79D9
E875				79DA
E876				79DB
E877				79DC
E878				79DD
E879				79DE
E87A				79DF
E87B				79E0
E87C				79E1
E87D				79E2
E87E				79E3
E87F				79E4
E880				79E5
E881				79E6
E882				79E7
E883				79E8
E59F				7AA1
E5A0				7AA2
E5A1				7AA3
E5A2				7AA4
E5A3				7AA5
E5A4				7AA6
E5A5				7AA7
E5A6				7AA8
E5A7				7AA9
E5A8				7AAA
E5A9				7AAB
E5AA				7AAC
E5AB				7AAD
E5AC				7AAE
E5AD				7AAF
E5AE				7AB0
E5AF				7AB1
E5B0				7AB2
E5B1				7AB3
E5B2				7AB4
E5B3				7AB5
E5B4				7AB6
E5B5				7AB7
E5B6				7AB8
E884				7AB9
E885				7ABA
E886				7ABB
E887				7ABC
E888				7ABD
E889				7ABE
E88A				7ABF
E88B				7AC0
E5B7				7AC1
E5B8				7AC2
E5B9				7AC3
E5BA				7AC4
E5BB				7AC5
E5BC				7AC6
E5BD				7AC7
E5BE				7AC8
E5BF				7AC9
E5C0				7ACA
E5C1				7ACB
E5C2				7ACC
E5C3				7ACD
E5C4				7ACE
E5C5				7ACF
E5C6				7AD0
E5C7				7AD1
E5C8				7AD2
E5C9				7AD3
E5CA				7AD4
E5CB				7AD5
E5CC				7AD6
E5CD				7AD7
E5CE				7AD8
E88C				7AD9
E88D				7ADA
E88E				7ADB
E88F				7ADC
E890				7ADD
E891				7ADE
E892				7ADF
E893				7AE0
E5CF				7AE1
E5D0				7AE2
E5D1				7AE3
E5D2				7AE4
E5D3				7AE5
E5D4				7AE6
E5D5				7AE7
E5D6				7AE8
E5D7				7AE9
E5D8				7AEA
E5D9				7AEB
E5DA				7AEC
E5DB				7AED
E5DC				7AEE
E5DD				7AEF
E5DE				7AF0
E5DF				7AF1
E5E0				7AF2
E5E1				7AF3
E5E2				7AF4
E5E3				7AF5
E5E4				7AF6
E5E5				7AF7
E5E6				7AF8
E894				7AFD
E895				7AFE
E5E7				7BA1
E5E8				7BA2
E5E9				7BA3
E5EA				7BA4
E5EB				7BA5
E5EC				7BA6
E5ED				7BA7
E5EE				7BA8
E5EF				7BA9
E5F0				7BAA
E5F1				7BAB
E5F2				7BAC
E5F3				7BAD
E5F4				7BAE
E5F5				7BAF
E5F6				7BB0
E5F7				7BB1
E5F8				7BB2
E5F9				7BB3
E5FA				7BB4
E5FB				7BB5
E5FC				7BB6
E5FD				7BB7
E5FE				7BB8
E896				7BB9
E897				7BBA
E898				7BBB
E899				7BBC
E89A				7BBD
E89B				7BBE
E89C				7BBF
E89D				7BC0
E5FF				7BC1
E600				7BC2
E601				7BC3
E602				7BC4
E603				7BC5
E604				7BC6
E605				7BC7
E606				7BC8
E607				7BC9
E608				7BCA
E609				7BCB
E60A				7BCC
E89E				7BCD
E89F				7BCE
E8A0				7BCF
E8A1				7BD0
E60B				7BD1
E60C				7BD2
E60D				7BD3
E60E				7BD4
E60F				7BD5
E610				7BD6
E611				7BD7
E612				7BD8
E613				7BD9
E614				7BDA
E615				7BDB
E616				7BDC
E8A2				7BDD
E8A3				7BDE
E8A4				7BDF
E8A5				7BE0
E617				7BE1
E618				7BE2
E619				7BE3
E61A				7BE4
E61B				7BE5
E61C				7BE6
E61D				7BE7
E61E				7BE8
E61F				7BE9
E620				7BEA
E621				7BEB
E622				7BEC
E623				7BED
E624				7BF1
E625				7BF2
E626				7BF3
E627				7BF4
E628				7BF5
E629				7BF6
E62A				7BF7
E62B				7BF8
E62C				7BF9
E62D				7BFA
E62E				7BFB
E62F				7BFC
E8A6				7BFD
E8A7				7BFE
E8A8				7CA1
E8A9				7CA2
E8AA				7CA3
E8AB				7CA4
E8AC				7CA5
E8AD				7CA6
E8AE				7CA7
E8AF				7CA8
E8B0				7CA9
E8B1				7CAA
E8B2				7CAB
E8B3				7CAC
E8B4				7CAD
E8B5				7CAE
E8B6				7CAF
E8B7				7CB0
E8B8				7CB1
E8B9				7CB2
E8BA				7CB3
E8BB				7CB4
E8BC				7CB5
E8BD				7CB6
E8BE				7CB7
E8BF				7CB8
E8C0				7CB9
E8C1				7CBA
E8C2				7CBB
E8C3				7CBC
E8C4				7CBD
E8C5				7CBE
E8C6				7CBF
2537	283A	84B8	00826	7CC0
250C	2823	84A1	00803	7CC1
2510	2824	84A2	00804	7CC2
2518	2825	84A3	00805	7CC3
2514	2826	84A4	00806	7CC4
252C	2828	84A6	00808	7CC5
2534	282A	84A8	00810	7CC6
251C	2827	84A5	00807	7CC7
2524	2829	84A7	00809	7CC8
253C	282B	84A9	00811	7CC9
2520	2837	84B5	00823	7CCA
252F	2838	84B6	00824	7CCB
2528	2839	84B7	00825	7CCC
2542	2840	84BE	00832	7CCD
253F	283B	84B9	00827	7CCE
E8C7				7CCF
E8C8				7CD0
2500	2821	849F	00801	7CD1
2502	2822	84A0	00802	7CD2
E8C9				7CD3
E8CA				7CD4
E8CB				7CD5
E8CC				7CD6
E8CD				7CD7
E8CE				7CD8
E8CF				7CD9
E8D0				7CDA
E8D1				7CDB
E8D2				7CDC
E8D3				7CDD
E8D4				7CDE
E8D5				7CDF
E8D6				7CE0
E8D7				7CE1
E8D8				7CE2
E8D9				7CE3
E8DA				7CE4
E8DB				7CE5
250F	282E	84AC	00814	7CE6
2513	282F	84AD	00815	7CE7
251B	2830	84AE	00816	7CE8
2517	2831	84AF	00817	7CE9
E8DC				7CEA
E8DD				7CEB
E8DE				7CEC
E8DF				7CED
E8E0				7CEE
E8E1				7CEF
E8E2				7CF0
2533	2833	84B1	00819	7CF1
253B	2835	84B3	00821	7CF2
2523	2832	84B0	00818	7CF3
252B	2834	84B2	00820	7CF4
254B	2836	84B4	00822	7CF5
2501	282C	84AA	00812	7CF6
2503	282D	84AB	00813	7CF7
E8E3				7CF8
E8E4				7CF9
E8E5				7CFA
E8E6				7CFB
E8E7				7CFC
E8E8				7DA1
E8E9				7DA2
E8EA				7DA3
E8EB				7DA4
E8EC				7DA5
E8ED				7DA6
E8EE				7DA7
E8EF				7DA8
E8F0				7DA9
E8F1				7DAA
E8F2				7DAB
E8F3				7DAC
E8F4				7DAD
E8F5				7DAE
E8F6				7DAF
E8F7				7DB0
E8F8				7DB1
E8F9				7DB2
E8FA				7DB3
E8FB				7DB4
E8FC				7DB5
E8FD				7DB6
E8FE				7DB7
E8FF				7DB8
E900				7DB9
E901				7DBA
E902				7DBB
E903				7DBC
E904				7DBD
E905				7DBE
E906				7DBF
E907				7DC0
E908				7DC1
E909				7DC2
E90A				7DC3
E90B				7DC4
E90C				7DC5
E90D				7DC6
E90E				7DC7
E90F				7DC8
E910				7DC9
E911				7DCA
E912				7DCB
E913				7DCC
E914				7DCD
E915				7DCE
E916				7DCF
E917				7DD0
E918				7DD1
E919				7DD2
E91A				7DD3
E91B				7DD4
E91C				7DD5
E91D				7DD6
E91E				7DD7
E91F				7DD8
E920				7DD9
E921				7DDA
E922				7DDB
E923				7DDC
E924				7DDD
E925				7DDE
E926				7DDF
E927				7DE0
E928				7DE1
E929				7DE2
E92A				7DE3
E92B				7DE4
E92C				7DE5
2538	283F	84BD	00831	7FA1
251D	283C	84BA	00828	7FA2
2530	283D	84BB	00829	7FA3
2525	283E	84BC	00830	7FA4
E92D				7FA5
E92E				7FA6
E92F				7FA7
E930				7FA8
E931				7FA9
E932				7FAA
E933				7FAB
E934				7FAC
E935				7FAD
E936				7FAE
E937				7FAF
E938				7FB0
E939				7FB1
E93A				7FB2
E93B				7FB3
E93C				7FB4
E93D				7FB5
E93E				7FB6
E93F				7FB7
E940				7FB8
E941				7FB9
E942				7FBA
E943				7FBB
E944				7FBC
E945				7FBD
E946				7FBE
E947				7FBF
E948				7FC0
E949				7FC1
E94A				7FC2
E94B				7FC3
E94C				7FC4
E94D				7FC5
E94E				7FC6
E94F				7FC7
E950				7FC8
E951				7FC9
E952				7FCA
E953				7FCB
E954				7FCC
E955				7FCD
E956				7FCE
E957				7FCF
2208	223A	81B8	00226	7FD0
220B	223B	81B9	00227	7FD1
2286	223C	81BA	00228	7FD2
2287	223D	81BB	00229	7FD3
2282	223E	81BC	00230	7FD4
2283	223F	81BD	00231	7FD5
222A	2240	81BE	00232	7FD6
2229	2241	81BF	00233	7FD7
2227	224A	81C8	00242	7FD8
2228	224B	81C9	00243	7FD9
21D2	224D	81CB	00245	7FDA
21D4	224E	81CC	00246	7FDB
2200	224F	81CD	00247	7FDC
2203	2250	81CE	00248	7FDD
2220	225C	81DA	00260	7FDE
22A5	225D	81DB	00261	7FDF
2312	225E	81DC	00262	7FE0
2202	225F	81DD	00263	7FE1
2207	2260	81DE	00264	7FE2
226A	2263	81E1	00267	7FE3
226B	2264	81E2	00268	7FE4
221A	2265	81E3	00269	7FE5
223D	2266	81E4	00270	7FE6
221D	2267	81E5	00271	7FE7
222B	2269	81E7	00273	7FE8
222C	226A	81E8	00274	7FE9
266F	2274	81F2	00284	7FEA
266D	2275	81F3	00285	7FEB
266A	2276	81F4	00286	7FEC
2020	2277	81F5	00287	7FED
2021	2278	81F6	00288	7FEE
00B6	2279	81F7	00289	7FEF
25EF	227E	81FC	00294	7FF0
E958				7FF1
E959				7FF2
E95A				7FF3
E95B				7FF4
E95C				7FF5
E95D				7FF6
E95E				7FF7
E95F				7FF8
E960				7FF9
E961				7FFA
E962				7FFB
E963				7FFC
E964				7FFD
3000	2121	8140	00101	A1A1
3001	2122	8141	00102	A1A2
3002	2123	8142	00103	A1A3
FF0C	2124	8143	00104	A1A4
FF0E	2125	8144	00105	A1A5
30FB	2126	8145	00106	A1A6
FF1A	2127	8146	00107	A1A7
FF1B	2128	8147	00108	A1A8
FF1F	2129	8148	00109	A1A9
FF01	212A	8149	00110	A1AA
309B	212B	814A	00111	A1AB
309C	212C	814B	00112	A1AC
00B4	212D	814C	00113	A1AD
FF40	212E	814D	00114	A1AE
00A8	212F	814E	00115	A1AF
FF3E	2130	814F	00116	A1B0
FFE3	2131	8150	00117	A1B1
FF3F	2132	8151	00118	A1B2
30FD	2133	8152	00119	A1B3
30FE	2134	8153	00120	A1B4
309D	2135	8154	00121	A1B5
309E	2136	8155	00122	A1B6
3003	2137	8156	00123	A1B7
4EDD	2138	8157	00124	A1B8
3005	2139	8158	00125	A1B9
3006	213A	8159	00126	A1BA
3007	213B	815A	00127	A1BB
30FC	213C	815B	00128	A1BC
2015	213D	815C	00129	A1BD
2010	213E	815D	00130	A1BE
FF0F	213F	815E	00131	A1BF
FF3C	2140	815F	00132	A1C0
FF5E	2141	8160	00133	A1C1
2225	2142	8161	00134	A1C2
FF5C	2143	8162	00135	A1C3
2026	2144	8163	00136	A1C4
2025	2145	8164	00137	A1C5
2018	2146	8165	00138	A1C6
2019	2147	8166	00139	A1C7
201C	2148	8167	00140	A1C8
201D	2149	8168	00141	A1C9
FF08	214A	8169	00142	A1CA
FF09	214B	816A	00143	A1CB
3014	214C	816B	00144	A1CC
3015	214D	816C	00145	A1CD
FF3B	214E	816D	00146	A1CE
FF3D	214F	816E	00147	A1CF
FF5B	2150	816F	00148	A1D0
FF5D	2151	8170	00149	A1D1
3008	2152	8171	00150	A1D2
3009	2153	8172	00151	A1D3
300A	2154	8173	00152	A1D4
300B	2155	8174	00153	A1D5
300C	2156	8175	00154	A1D6
300D	2157	8176	00155	A1D7
300E	2158	8177	00156	A1D8
300F	2159	8178	00157	A1D9
3010	215A	8179	00158	A1DA
3011	215B	817A	00159	A1DB
FF0B	215C	817B	00160	A1DC
FF0D	215D	817C	00161	A1DD
00B1	215E	817D	00162	A1DE
00D7	215F	817E	00163	A1DF
00F7	2160	8180	00164	A1E0
FF1D	2161	8181	00165	A1E1
2260	2162	8182	00166	A1E2
FF1C	2163	8183	00167	A1E3
FF1E	2164	8184	00168	A1E4
2266	2165	8185	00169	A1E5
2267	2166	8186	00170	A1E6
221E	2167	8187	00171	A1E7
2234	2168	8188	00172	A1E8
2642	2169	8189	00173	A1E9
2640	216A	818A	00174	A1EA
00B0	216B	818B	00175	A1EB
2032	216C	818C	00176	A1EC
2033	216D	818D	00177	A1ED
2103	216E	818E	00178	A1EE
FFE5	216F	818F	00179	A1EF
FF04	2170	8190	00180	A1F0
FFE0	2171	8191	00181	A1F1
FFE1	2172	8192	00182	A1F2
FF05	2173	8193	00183	A1F3
FF03	2174	8194	00184	A1F4
FF06	2175	8195	00185	A1F5
FF0A	2176	8196	00186	A1F6
FF20	2177	8197	00187	A1F7
00A7	2178	8198	00188	A1F8
2606	2179	8199	00189	A1F9
2605	217A	819A	00190	A1FA
25CB	217B	819B	00191	A1FB
25CF	217C	819C	00192	A1FC
25CE	217D	819D	00193	A1FD
25C7	217E	819E	00194	A1FE
25C6	2221	819F	00201	A2A1
25A1	2222	81A0	00202	A2A2
25A0	2223	81A1	00203	A2A3
25B3	2224	81A2	00204	A2A4
25B2	2225	81A3	00205	A2A5
25BD	2226	81A4	00206	A2A6
25BC	2227	81A5	00207	A2A7
203B	2228	81A6	00208	A2A8
3012	2229	81A7	00209	A2A9
2192	222A	81A8	00210	A2AA
2190	222B	81A9	00211	A2AB
2191	222C	81AA	00212	A2AC
2193	222D	81AB	00213	A2AD
3013	222E	81AC	00214	A2AE
FF10	2330	824F	00316	A3B0
FF11	2331	8250	00317	A3B1
FF12	2332	8251	00318	A3B2
FF13	2333	8252	00319	A3B3
FF14	2334	8253	00320	A3B4
FF15	2335	8254	00321	A3B5
FF16	2336	8255	00322	A3B6
FF17	2337	8256	00323	A3B7
FF18	2338	8257	00324	A3B8
FF19	2339	8258	00325	A3B9
FF21	2341	8260	00333	A3C1
FF22	2342	8261	00334	A3C2
FF23	2343	8262	00335	A3C3
FF24	2344	8263	00336	A3C4
FF25	2345	8264	00337	A3C5
FF26	2346	8265	00338	A3C6
FF27	2347	8266	00339	A3C7
FF28	2348	8267	00340	A3C8
FF29	2349	8268	00341	A3C9
FF2A	234A	8269	00342	A3CA
FF2B	234B	826A	00343	A3CB
FF2C	234C	826B	00344	A3CC
FF2D	234D	826C	00345	A3CD
FF2E	234E	826D	00346	A3CE
FF2F	234F	826E	00347	A3CF
FF30	2350	826F	00348	A3D0
FF31	2351	8270	00349	A3D1
FF32	2352	8271	00350	A3D2
FF33	2353	8272	00351	A3D3
FF34	2354	8273	00352	A3D4
FF35	2355	8274	00353	A3D5
FF36	2356	8275	00354	A3D6
FF37	2357	8276	00355	A3D7
FF38	2358	8277	00356	A3D8
FF39	2359	8278	00357	A3D9
FF3A	235A	8279	00358	A3DA
FF41	2361	8281	00365	A3E1
FF42	2362	8282	00366	A3E2
FF43	2363	8283	00367	A3E3
FF44	2364	8284	00368	A3E4
FF45	2365	8285	00369	A3E5
FF46	2366	8286	00370	A3E6
FF47	2367	8287	00371	A3E7
FF48	2368	8288	00372	A3E8
FF49	2369	8289	00373	A3E9
FF4A	236A	828A	00374	A3EA
FF4B	236B	828B	00375	A3EB
FF4C	236C	828C	00376	A3EC
FF4D	236D	828D	00377	A3ED
FF4E	236E	828E	00378	A3EE
FF4F	236F	828F	00379	A3EF
FF50	2370	8290	00380	A3F0
FF51	2371	8291	00381	A3F1
FF52	2372	8292	00382	A3F2
FF53	2373	8293	00383	A3F3
FF54	2374	8294	00384	A3F4
FF55	2375	8295	00385	A3F5
FF56	2376	8296	00386	A3F6
FF57	2377	8297	00387	A3F7
FF58	2378	8298	00388	A3F8
FF59	2379	8299	00389	A3F9
FF5A	237A	829A	00390	A3FA
3041	2421	829F	00401	A4A1
3042	2422	82A0	00402	A4A2
3043	2423	82A1	00403	A4A3
3044	2424	82A2	00404	A4A4
3045	2425	82A3	00405	A4A5
3046	2426	82A4	00406	A4A6
3047	2427	82A5	00407	A4A7
3048	2428	82A6	00408	A4A8
3049	2429	82A7	00409	A4A9
304A	242A	82A8	00410	A4AA
304B	242B	82A9	00411	A4AB
304C	242C	82AA	00412	A4AC
304D	242D	82AB	00413	A4AD
304E	242E	82AC	00414	A4AE
304F	242F	82AD	00415	A4AF
3050	2430	82AE	00416	A4B0
3051	2431	82AF	00417	A4B1
3052	2432	82B0	00418	A4B2
3053	2433	82B1	00419	A4B3
3054	2434	82B2	00420	A4B4
3055	2435	82B3	00421	A4B5
3056	2436	82B4	00422	A4B6
3057	2437	82B5	00423	A4B7
3058	2438	82B6	00424	A4B8
3059	2439	82B7	00425	A4B9
305A	243A	82B8	00426	A4BA
305B	243B	82B9	00427	A4BB
305C	243C	82BA	00428	A4BC
305D	243D	82BB	00429	A4BD
305E	243E	82BC	00430	A4BE
305F	243F	82BD	00431	A4BF
3060	2440	82BE	00432	A4C0
3061	2441	82BF	00433	A4C1
3062	2442	82C0	00434	A4C2
3063	2443	82C1	00435	A4C3
3064	2444	82C2	00436	A4C4
3065	2445	82C3	00437	A4C5
3066	2446	82C4	00438	A4C6
3067	2447	82C5	00439	A4C7
3068	2448	82C6	00440	A4C8
3069	2449	82C7	00441	A4C9
306A	244A	82C8	00442	A4CA
306B	244B	82C9	00443	A4CB
306C	244C	82CA	00444	A4CC
306D	244D	82CB	00445	A4CD
306E	244E	82CC	00446	A4CE
306F	244F	82CD	00447	A4CF
3070	2450	82CE	00448	A4D0
3071	2451	82CF	00449	A4D1
3072	2452	82D0	00450	A4D2
3073	2453	82D1	00451	A4D3
3074	2454	82D2	00452	A4D4
3075	2455	82D3	00453	A4D5
3076	2456	82D4	00454	A4D6
3077	2457	82D5	00455	A4D7
3078	2458	82D6	00456	A4D8
3079	2459	82D7	00457	A4D9
307A	245A	82D8	00458	A4DA
307B	245B	82D9	00459	A4DB
307C	245C	82DA	00460	A4DC
307D	245D	82DB	00461	A4DD
307E	245E	82DC	00462	A4DE
307F	245F	82DD	00463	A4DF
3080	2460	82DE	00464	A4E0
3081	2461	82DF	00465	A4E1
3082	2462	82E0	00466	A4E2
3083	2463	82E1	00467	A4E3
3084	2464	82E2	00468	A4E4
3085	2465	82E3	00469	A4E5
3086	2466	82E4	00470	A4E6
3087	2467	82E5	00471	A4E7
3088	2468	82E6	00472	A4E8
3089	2469	82E7	00473	A4E9
308A	246A	82E8	00474	A4EA
308B	246B	82E9	00475	A4EB
308C	246C	82EA	00476	A4EC
308D	246D	82EB	00477	A4ED
308E	246E	82EC	00478	A4EE
308F	246F	82ED	00479	A4EF
3090	2470	82EE	00480	A4F0
3091	2471	82EF	00481	A4F1
3092	2472	82F0	00482	A4F2
3093	2473	82F1	00483	A4F3
30A1	2521	8340	00501	A5A1
30A2	2522	8341	00502	A5A2
30A3	2523	8342	00503	A5A3
30A4	2524	8343	00504	A5A4
30A5	2525	8344	00505	A5A5
30A6	2526	8345	00506	A5A6
30A7	2527	8346	00507	A5A7
30A8	2528	8347	00508	A5A8
30A9	2529	8348	00509	A5A9
30AA	252A	8349	00510	A5AA
30AB	252B	834A	00511	A5AB
30AC	252C	834B	00512	A5AC
30AD	252D	834C	00513	A5AD
30AE	252E	834D	00514	A5AE
30AF	252F	834E	00515	A5AF
30B0	2530	834F	00516	A5B0
30B1	2531	8350	00517	A5B1
30B2	2532	8351	00518	A5B2
30B3	2533	8352	00519	A5B3
30B4	2534	8353	00520	A5B4
30B5	2535	8354	00521	A5B5
30B6	2536	8355	00522	A5B6
30B7	2537	8356	00523	A5B7
30B8	2538	8357	00524	A5B8
30B9	2539	8358	00525	A5B9
30BA	253A	8359	00526	A5BA
30BB	253B	835A	00527	A5BB
30BC	253C	835B	00528	A5BC
30BD	253D	835C	00529	A5BD
30BE	253E	835D	00530	A5BE
30BF	253F	835E	00531	A5BF
30C0	2540	835F	00532	A5C0
30C1	2541	8360	00533	A5C1
30C2	2542	8361	00534	A5C2
30C3	2543	8362	00535	A5C3
30C4	2544	8363	00536	A5C4
30C5	2545	8364	00537	A5C5
30C6	2546	8365	00538	A5C6
30C7	2547	8366	00539	A5C7
30C8	2548	8367	00540	A5C8
30C9	2549	8368	00541	A5C9
30CA	254A	8369	00542	A5CA
30CB	254B	836A	00543	A5CB
30CC	254C	836B	00544	A5CC
30CD	254D	836C	00545	A5CD
30CE	254E	836D	00546	A5CE
30CF	254F	836E	00547	A5CF
30D0	2550	836F	00548	A5D0
30D1	2551	8370	00549	A5D1
30D2	2552	8371	00550	A5D2
30D3	2553	8372	00551	A5D3
30D4	2554	8373	00552	A5D4
30D5	2555	8374	00553	A5D5
30D6	2556	8375	00554	A5D6
30D7	2557	8376	00555	A5D7
30D8	2558	8377	00556	A5D8
30D9	2559	8378	00557	A5D9
30DA	255A	8379	00558	A5DA
30DB	255B	837A	00559	A5DB
30DC	255C	837B	00560	A5DC
30DD	255D	837C	00561	A5DD
30DE	255E	837D	00562	A5DE
30DF	255F	837E	00563	A5DF
30E0	2560	8380	00564	A5E0
30E1	2561	8381	00565	A5E1
30E2	2562	8382	00566	A5E2
30E3	2563	8383	00567	A5E3
30E4	2564	8384	00568	A5E4
30E5	2565	8385	00569	A5E5
30E6	2566	8386	00570	A5E6
30E7	2567	8387	00571	A5E7
30E8	2568	8388	00572	A5E8
30E9	2569	8389	00573	A5E9
30EA	256A	838A	00574	A5EA
30EB	256B	838B	00575	A5EB
30EC	256C	838C	00576	A5EC
30ED	256D	838D	00577	A5ED
30EE	256E	838E	00578	A5EE
30EF	256F	838F	00579	A5EF
30F0	2570	8390	00580	A5F0
30F1	2571	8391	00581	A5F1
30F2	2572	8392	00582	A5F2
30F3	2573	8393	00583	A5F3
30F4	2574	8394	00584	A5F4
30F5	2575	8395	00585	A5F5
30F6	2576	8396	00586	A5F6
0391	2621	839F	00601	A6A1
0392	2622	83A0	00602	A6A2
0393	2623	83A1	00603	A6A3
0394	2624	83A2	00604	A6A4
0395	2625	83A3	00605	A6A5
0396	2626	83A4	00606	A6A6
0397	2627	83A5	00607	A6A7
0398	2628	83A6	00608	A6A8
0399	2629	83A7	00609	A6A9
039A	262A	83A8	00610	A6AA
039B	262B	83A9	00611	A6AB
039C	262C	83AA	00612	A6AC
039D	262D	83AB	00613	A6AD
039E	262E	83AC	00614	A6AE
039F	262F	83AD	00615	A6AF
03A0	2630	83AE	00616	A6B0
03A1	2631	83AF	00617	A6B1
03A3	2632	83B0	00618	A6B2
03A4	2633	83B1	00619	A6B3
03A5	2634	83B2	00620	A6B4
03A6	2635	83B3	00621	A6B5
03A7	2636	83B4	00622	A6B6
03A8	2637	83B5	00623	A6B7
03A9	2638	83B6	00624	A6B8
03B1	2641	83BF	00633	A6C1
03B2	2642	83C0	00634	A6C2
03B3	2643	83C1	00635	A6C3
03B4	2644	83C2	00636	A6C4
03B5	2645	83C3	00637	A6C5
03B6	2646	83C4	00638	A6C6
03B7	2647	83C5	00639	A6C7
03B8	2648	83C6	00640	A6C8
03B9	2649	83C7	00641	A6C9
03BA	264A	83C8	00642	A6CA
03BB	264B	83C9	00643	A6CB
03BC	264C	83CA	00644	A6CC
03BD	264D	83CB	00645	A6CD
03BE	264E	83CC	00646	A6CE
03BF	264F	83CD	00647	A6CF
03C0	2650	83CE	00648	A6D0
03C1	2651	83CF	00649	A6D1
03C3	2652	83D0	00650	A6D2
03C4	2653	83D1	00651	A6D3
03C5	2654	83D2	00652	A6D4
03C6	2655	83D3	00653	A6D5
03C7	2656	83D4	00654	A6D6
03C8	2657	83D5	00655	A6D7
03C9	2658	83D6	00656	A6D8
0410	2721	8440	00701	A7A1
0411	2722	8441	00702	A7A2
0412	2723	8442	00703	A7A3
0413	2724	8443	00704	A7A4
0414	2725	8444	00705	A7A5
0415	2726	8445	00706	A7A6
0401	2727	8446	00707	A7A7
0416	2728	8447	00708	A7A8
0417	2729	8448	00709	A7A9
0418	272A	8449	00710	A7AA
0419	272B	844A	00711	A7AB
041A	272C	844B	00712	A7AC
041B	272D	844C	00713	A7AD
041C	272E	844D	00714	A7AE
041D	272F	844E	00715	A7AF
041E	2730	844F	00716	A7B0
041F	2731	8450	00717	A7B1
0420	2732	8451	00718	A7B2
0421	2733	8452	00719	A7B3
0422	2734	8453	00720	A7B4
0423	2735	8454	00721	A7B5
0424	2736	8455	00722	A7B6
0425	2737	8456	00723	A7B7
0426	2738	8457	00724	A7B8
0427	2739	8458	00725	A7B9
0428	273A	8459	00726	A7BA
0429	273B	845A	00727	A7BB
042A	273C	845B	00728	A7BC
042B	273D	845C	00729	A7BD
042C	273E	845D	00730	A7BE
042D	273F	845E	00731	A7BF
042E	2740	845F	00732	A7C0
042F	2741	8460	00733	A7C1
0430	2751	8470	00749	A7D1
0431	2752	8471	00750	A7D2
0432	2753	8472	00751	A7D3
0433	2754	8473	00752	A7D4
0434	2755	8474	00753	A7D5
0435	2756	8475	00754	A7D6
0451	2757	8476	00755	A7D7
0436	2758	8477	00756	A7D8
0437	2759	8478	00757	A7D9
0438	275A	8479	00758	A7DA
0439	275B	847A	00759	A7DB
043A	275C	847B	00760	A7DC
043B	275D	847C	00761	A7DD
043C	275E	847D	00762	A7DE
043D	275F	847E	00763	A7DF
043E	2760	8480	00764	A7E0
043F	2761	8481	00765	A7E1
0440	2762	8482	00766	A7E2
0441	2763	8483	00767	A7E3
0442	2764	8484	00768	A7E4
0443	2765	8485	00769	A7E5
0444	2766	8486	00770	A7E6
0445	2767	8487	00771	A7E7
0446	2768	8488	00772	A7E8
0447	2769	8489	00773	A7E9
0448	276A	848A	00774	A7EA
0449	276B	848B	00775	A7EB
044A	276C	848C	00776	A7EC
044B	276D	848D	00777	A7ED
044C	276E	848E	00778	A7EE
044D	276F	848F	00779	A7EF
044E	2770	8490	00780	A7F0
044F	2771	8491	00781	A7F1
4E9C	3021	889F	01601	B0A1
555E				B0A2
5A03	3023	88A1	01603	B0A3
963F	3024	88A2	01604	B0A4
54C0	3025	88A3	01605	B0A5
611B	3026	88A4	01606	B0A6
6328	3027	88A5	01607	B0A7
59F6	3028	88A6	01608	B0A8
9022	3029	88A7	01609	B0A9
E7A8				B0AA
831C	302B	88A9	01611	B0AB
7A50	302C	88AA	01612	B0AC
60AA	302D	88AB	01613	B0AD
63E1	302E	88AC	01614	B0AE
6E25	302F	88AD	01615	B0AF
65ED	3030	88AE	01616	B0B0
8466	3031	88AF	01617	B0B1
E39C				B0B2
9C3A	724D	E9CB	08245	B0B3
6893	3034	88B2	01620	B0B4
5727	3035	88B3	01621	B0B5
65A1	3036	88B4	01622	B0B6
6271	3037	88B5	01623	B0B7
5B9B	3038	88B6	01624	B0B8
59D0	3039	88B7	01625	B0B9
E3C6				B0BA
E4B4				B0BB
7D62	303C	88BA	01628	B0BC
7DBE	303D	88BB	01629	B0BD
9B8E	303E	88BC	01630	B0BE
6216	303F	88BD	01631	B0BF
7C9F	3040	88BE	01632	B0C0
88B7	3041	88BF	01633	B0C1
5B89	3042	88C0	01634	B0C2
5EB5	3043	88C1	01635	B0C3
6309	3044	88C2	01636	B0C4
6697	3045	88C3	01637	B0C5
6848	3046	88C4	01638	B0C6
95C7	3047	88C5	01639	B0C7
978D	3048	88C6	01640	B0C8
674F	3049	88C7	01641	B0C9
4EE5	304A	88C8	01642	B0CA
4F0A	304B	88C9	01643	B0CB
4F4D	304C	88CA	01644	B0CC
4F9D	304D	88CB	01645	B0CD
5049	304E	88CC	01646	B0CE
56F2	304F	88CD	01647	B0CF
5937	3050	88CE	01648	B0D0
59D4	3051	88CF	01649	B0D1
5A01	3052	88D0	01650	B0D2
5C09	3053	88D1	01651	B0D3
60DF	3054	88D2	01652	B0D4
610F	3055	88D3	01653	B0D5
6170	3056	88D4	01654	B0D6
6613	3057	88D5	01655	B0D7
6905	3058	88D6	01656	B0D8
70BA	3059	88D7	01657	B0D9
754F	305A	88D8	01658	B0DA
7570	305B	88D9	01659	B0DB
79FB	305C	88DA	01660	B0DC
7DAD	305D	88DB	01661	B0DD
7DEF	305E	88DC	01662	B0DE
80C3	305F	88DD	01663	B0DF
840E	3060	88DE	01664	B0E0
8863	3061	88DF	01665	B0E1
8B02	3062	88E0	01666	B0E2
9055	3063	88E1	01667	B0E3
907A	3064	88E2	01668	B0E4
E785				B0E5
4E95	3066	88E4	01670	B0E6
4EA5	3067	88E5	01671	B0E7
57DF	3068	88E6	01672	B0E8
80B2	3069	88E7	01673	B0E9
90C1	306A	88E8	01674	B0EA
78EF	306B	88E9	01675	B0EB
4E00	306C	88EA	01676	B0EC
58F1	306D	88EB	01677	B0ED
E2C6				B0EE
9038	306F	88ED	01679	B0EF
7A32	3070	88EE	01680	B0F0
E3A0				B0F1
828B	3072	88F0	01682	B0F2
E4D5				B0F3
5141	3074	88F2	01684	B0F4
5370	3075	88F3	01685	B0F5
54BD	3076	88F4	01686	B0F6
54E1	3077	88F5	01687	B0F7
56E0	3078	88F6	01688	B0F8
59FB	3079	88F7	01689	B0F9
5F15	307A	88F8	01690	B0FA
98F2	307B	88F9	01691	B0FB
E2BF				B0FC
80E4	307D	88FB	01693	B0FD
852D	307E	88FC	01694	B0FE
9662	3121	8940	01701	B1A1
9670	3122	8941	01702	B1A2
96A0	3123	8942	01703	B1A3
97FB	3124	8943	01704	B1A4
540B	3125	8944	01705	B1A5
53F3	3126	8945	01706	B1A6
5B87	3127	8946	01707	B1A7
70CF	3128	8947	01708	B1A8
7FBD	3129	8948	01709	B1A9
E41D				B1AA
96E8	312B	894A	01711	B1AB
536F	312C	894B	01712	B1AC
9D5C	312D	894C	01713	B1AD
7ABA	312E	894D	01714	B1AE
4E11	312F	894E	01715	B1AF
7893	3130	894F	01716	B1B0
81FC	3131	8950	01717	B1B1
6E26	3132	8951	01718	B1B2
E64B				B1B3
5504	3134	8953	01720	B1B4
E2AC				B1B5
851A	3136	8955	01722	B1B6
9C3B	3137	8956	01723	B1B7
59E5	3138	8957	01724	B1B8
E18A				B1B9
6D66	313A	8959	01726	B1BA
74DC	313B	895A	01727	B1BB
958F	313C	895B	01728	B1BC
E1A9				B1BD
4E91	313E	895D	01730	B1BE
904B	313F	895E	01731	B1BF
96F2	3140	895F	01732	B1C0
834F	3141	8960	01733	B1C1
E4B9				B1C2
53E1	3143	8962	01735	B1C3
55B6	3144	8963	01736	B1C4
5B30	3145	8964	01737	B1C5
5F71	3146	8965	01738	B1C6
6620	3147	8966	01739	B1C7
66F3	3148	8967	01740	B1C8
6804	3149	8968	01741	B1C9
6C38	314A	8969	01742	B1CA
6CF3	314B	896A	01743	B1CB
6D29	314C	896B	01744	B1CC
745B	314D	896C	01745	B1CD
76C8	314E	896D	01746	B1CE
7A4E	314F	896E	01747	B1CF
9834	3150	896F	01748	B1D0
82F1	3151	8970	01749	B1D1
885B	3152	8971	01750	B1D2
8A60	3153	8972	01751	B1D3
92ED	3154	8973	01752	B1D4
6DB2	3155	8974	01753	B1D5
75AB	3156	8975	01754	B1D6
76CA	3157	8976	01755	B1D7
99C5	3158	8977	01756	B1D8
60A6	3159	8978	01757	B1D9
8B01	315A	8979	01758	B1DA
8D8A	315B	897A	01759	B1DB
95B2	315C	897B	01760	B1DC
698E	315D	897C	01761	B1DD
53AD	315E	897D	01762	B1DE
5186	315F	897E	01763	B1DF
5712	3160	8980	01764	B1E0
5830	3161	8981	01765	B1E1
5944	3162	8982	01766	B1E2
5BB4	3163	8983	01767	B1E3
5EF6	3164	8984	01768	B1E4
6028	3165	8985	01769	B1E5
63A9	3166	8986	01770	B1E6
63F4	3167	8987	01771	B1E7
E2B5				B1E8
6F14	3169	8989	01773	B1E9
E796				B1EA
7130				B1EB
7159	316C	898C	01776	B1EC
71D5	316D	898D	01777	B1ED
733F	316E	898E	01778	B1EE
E36B				B1EF
8276	3170	8990	01780	B1F0
82D1	3171	8991	01781	B1F1
8597	3172	8992	01782	B1F2
9060	3173	8993	01783	B1F3
E477				B1F4
9D1B	3175	8995	01785	B1F5
5869	3176	8996	01786	B1F6
65BC	3177	8997	01787	B1F7
6C5A	3178	8998	01788	B1F8
7525	3179	8999	01789	B1F9
51F9	317A	899A	01790	B1FA
592E	317B	899B	01791	B1FB
5965	317C	899C	01792	B1FC
5F80	317D	899D	01793	B1FD
5FDC	317E	899E	01794	B1FE
62BC	3221	899F	01801	B2A1
65FA	3222	89A0	01802	B2A2
6A2A	3223	89A1	01803	B2A3
6B27	3224	89A2	01804	B2A4
6BB4	3225	89A3	01805	B2A5
738B	3226	89A4	01806	B2A6
E379				B2A7
E3D9				B2A8
9DAF	7274	E9F2	08284	B2A9
9DD7				B2AA
9EC4	322B	89A9	01811	B2AB
5CA1	322C	89AA	01812	B2AC
6C96	322D	89AB	01813	B2AD
837B	322E	89AC	01814	B2AE
5104	322F	89AD	01815	B2AF
5C4B	3230	89AE	01816	B2B0
61B6	3231	89AF	01817	B2B1
81C6	3232	89B0	01818	B2B2
6876	3233	89B1	01819	B2B3
7261	3234	89B2	01820	B2B4
4E59	3235	89B3	01821	B2B5
4FFA	3236	89B4	01822	B2B6
5378	3237	89B5	01823	B2B7
6069	3238	89B6	01824	B2B8
6E29	3239	89B7	01825	B2B9
7A4F	323A	89B8	01826	B2BA
97F3	323B	89B9	01827	B2BB
4E0B	323C	89BA	01828	B2BC
5316	323D	89BB	01829	B2BD
4EEE	323E	89BC	01830	B2BE
4F55	323F	89BD	01831	B2BF
4F3D	3240	89BE	01832	B2C0
4FA1	3241	89BF	01833	B2C1
4F73	3242	89C0	01834	B2C2
52A0	3243	89C1	01835	B2C3
53EF	3244	89C2	01836	B2C4
5609	3245	89C3	01837	B2C5
590F	3246	89C4	01838	B2C6
E1D6				B2C7
E1E1				B2C8
5BE1	3249	89C7	01841	B2C9
79D1	324A	89C8	01842	B2CA
6687	324B	89C9	01843	B2CB
679C	324C	89CA	01844	B2CC
67B6	324D	89CB	01845	B2CD
6B4C	324E	89CC	01846	B2CE
6CB3	324F	89CD	01847	B2CF
706B	3250	89CE	01848	B2D0
73C2	3251	89CF	01849	B2D1
798D	3252	89D0	01850	B2D2
79BE	3253	89D1	01851	B2D3
E339				B2D4
7B87	3255	89D3	01853	B2D5
82B1	3256	89D4	01854	B2D6
82DB	3257	89D5	01855	B2D7
8304	3258	89D6	01856	B2D8
8377	3259	89D7	01857	B2D9
83EF	325A	89D8	01858	B2DA
83D3	325B	89D9	01859	B2DB
8766	325C	89DA	01860	B2DC
8AB2	325D	89DB	01861	B2DD
5629	325E	89DC	01862	B2DE
8CA8	325F	89DD	01863	B2DF
E427				B2E0
904E	3261	89DF	01865	B2E1
971E	3262	89E0	01866	B2E2
868A	3263	89E1	01867	B2E3
4FC4	3264	89E2	01868	B2E4
5CE8	3265	89E3	01869	B2E5
6211	3266	89E4	01870	B2E6
E2F4				B2E7
753B	3268	89E6	01872	B2E8
81E5	3269	89E7	01873	B2E9
82BD	326A	89E8	01874	B2EA
86FE	326B	89E9	01875	B2EB
8CC0	326C	89EA	01876	B2EC
96C5	326D	89EB	01877	B2ED
9913	326E	89EC	01878	B2EE
99D5	326F	89ED	01879	B2EF
4ECB	3270	89EE	01880	B2F0
4F1A	3271	89EF	01881	B2F1
89E3	3272	89F0	01882	B2F2
56DE	3273	89F1	01883	B2F3
584A	3274	89F2	01884	B2F4
58CA	3275	89F3	01885	B2F5
E208				B2F6
5FEB	3277	89F5	01887	B2F7
602A	3278	89F6	01888	B2F8
6094	3279	89F7	01889	B2F9
E21F				B2FA
61D0	327B	89F9	01891	B2FB
6212	327C	89FA	01892	B2FC
E240				B2FD
6539	327E	89FC	01894	B2FE
9B41	3321	8A40	01901	B3A1
E26B				B3A2
68B0	3323	8A42	01903	B3A3
6D77	3324	8A43	01904	B3A4
7070	3325	8A44	01905	B3A5
754C	3326	8A45	01906	B3A6
7686	3327	8A46	01907	B3A7
7D75	3328	8A47	01908	B3A8
82A5	3329	8A48	01909	B3A9
87F9	332A	8A49	01910	B3AA
958B	332B	8A4A	01911	B3AB
968E	332C	8A4B	01912	B3AC
8C9D	332D	8A4C	01913	B3AD
51F1	332E	8A4D	01914	B3AE
52BE	332F	8A4E	01915	B3AF
5916	3330	8A4F	01916	B3B0
54B3	3331	8A50	01917	B3B1
5BB3	3332	8A51	01918	B3B2
5D16	3333	8A52	01919	B3B3
6168	3334	8A53	01920	B3B4
6982	3335	8A54	01921	B3B5
6DAF	3336	8A55	01922	B3B6
788D	3337	8A56	01923	B3B7
84CB	3338	8A57	01924	B3B8
8857	3339	8A58	01925	B3B9
8A72	333A	8A59	01926	B3BA
93A7	333B	8A5A	01927	B3BB
9AB8	333C	8A5B	01928	B3BC
6D6C	333D	8A5C	01929	B3BD
99A8	333E	8A5D	01930	B3BE
86D9	333F	8A5E	01931	B3BF
57A3	3340	8A5F	01932	B3C0
67FF	3341	8A60	01933	B3C1
8823	695A	E579	07358	B3C2
920E	3343	8A62	01935	B3C3
E179				B3C4
5687	3345	8A64	01937	B3C5
5404	3346	8A65	01938	B3C6
5ED3	3347	8A66	01939	B3C7
62E1	3348	8A67	01940	B3C8
652A	5978	9D98	05788	B3C9
683C	334A	8A69	01942	B3CA
6838	334B	8A6A	01943	B3CB
E2AE				B3CC
7372	334D	8A6C	01945	B3CD
78BA	334E	8A6D	01946	B3CE
7A6B	334F	8A6E	01947	B3CF
899A	3350	8A6F	01948	B3D0
89D2	3351	8A70	01949	B3D1
8D6B	3352	8A71	01950	B3D2
8F03	3353	8A72	01951	B3D3
90ED	3354	8A73	01952	B3D4
95A3	3355	8A74	01953	B3D5
9694	3356	8A75	01954	B3D6
9769	3357	8A76	01955	B3D7
5B66	3358	8A77	01956	B3D8
5CB3	3359	8A78	01957	B3D9
697D	335A	8A79	01958	B3DA
984D	335B	8A7A	01959	B3DB
984E	335C	8A7B	01960	B3DC
639B	335D	8A7C	01961	B3DD
7B20	335E	8A7D	01962	B3DE
6A2B	335F	8A7E	01963	B3DF
6A7F	3360	8A80	01964	B3E0
68B6	3361	8A81	01965	B3E1
9C0D	3362	8A82	01966	B3E2
6F5F	3363	8A83	01967	B3E3
5272	3364	8A84	01968	B3E4
FA36				B3E5
6070	3366	8A86	01970	B3E6
62EC	3367	8A87	01971	B3E7
6D3B	3368	8A88	01972	B3E8
6E07	3369	8A89	01973	B3E9
6ED1	336A	8A8A	01974	B3EA
E3AA				B3EB
FA60				B3EC
8F44	336D	8A8D	01977	B3ED
4E14	336E	8A8E	01978	B3EE
9C39	336F	8A8F	01979	B3EF
53F6	3370	8A90	01980	B3F0
691B	3371	8A91	01981	B3F1
6A3A	3372	8A92	01982	B3F2
E4A2				B3F3
682A	3374	8A94	01984	B3F4
515C	3375	8A95	01985	B3F5
7AC3	3376	8A96	01986	B3F6
84B2	3377	8A97	01987	B3F7
E2EF				B3F8
938C	3379	8A99	01989	B3F9
E786				B3FA
9D28	337B	8A9B	01991	B3FB
6822	337C	8A9C	01992	B3FC
8305	337D	8A9D	01993	B3FD
8431	337E	8A9E	01994	B3FE
7CA5	3421	8A9F	02001	B4A1
5208	3422	8AA0	02002	B4A2
82C5	3423	8AA1	02003	B4A3
74E6	3424	8AA2	02004	B4A4
4E7E	3425	8AA3	02005	B4A5
4F83	3426	8AA4	02006	B4A6
51A0	3427	8AA5	02007	B4A7
5BD2	3428	8AA6	02008	B4A8
520A	3429	8AA7	02009	B4A9
52D8	342A	8AA8	02010	B4AA
52E7	342B	8AA9	02011	B4AB
5DFB	342C	8AAA	02012	B4AC
559A	342D	8AAB	02013	B4AD
582A	342E	8AAC	02014	B4AE
59E6	342F	8AAD	02015	B4AF
5B8C	3430	8AAE	02016	B4B0
5B98	3431	8AAF	02017	B4B1
5BDB	3432	8AB0	02018	B4B2
5E72	3433	8AB1	02019	B4B3
5E79	3434	8AB2	02020	B4B4
60A3	3435	8AB3	02021	B4B5
611F	3436	8AB4	02022	B4B6
6163	3437	8AB5	02023	B4B7
61BE	3438	8AB6	02024	B4B8
63DB	3439	8AB7	02025	B4B9
6562	343A	8AB8	02026	B4BA
67D1	343B	8AB9	02027	B4BB
6853	343C	8ABA	02028	B4BC
68FA	343D	8ABB	02029	B4BD
6B3E	343E	8ABC	02030	B4BE
6B53	343F	8ABD	02031	B4BF
6C57	3440	8ABE	02032	B4C0
6F22	3441	8ABF	02033	B4C1
E078				B4C2
704C	5E75	9FF3	06285	B4C3
74B0	3444	8AC2	02036	B4C4
7518	3445	8AC3	02037	B4C5
76E3	3446	8AC4	02038	B4C6
770B	3447	8AC5	02039	B4C7
7AFF	3448	8AC6	02040	B4C8
7BA1	3449	8AC7	02041	B4C9
7C21	344A	8AC8	02042	B4CA
7DE9	344B	8AC9	02043	B4CB
7F36	344C	8ACA	02044	B4CC
E380				B4CD
809D	344E	8ACC	02046	B4CE
8266	344F	8ACD	02047	B4CF
839E	3450	8ACE	02048	B4D0
89B3	3451	8ACF	02049	B4D1
8AEB	6B5D	E67C	07561	B4D2
8CAB	3453	8AD1	02051	B4D3
9084	3454	8AD2	02052	B4D4
9451	3455	8AD3	02053	B4D5
9593	3456	8AD4	02054	B4D6
9591	3457	8AD5	02055	B4D7
95A2	3458	8AD6	02056	B4D8
9665	3459	8AD7	02057	B4D9
97D3	345A	8AD8	02058	B4DA
9928	345B	8AD9	02059	B4DB
8218	345C	8ADA	02060	B4DC
4E38	345D	8ADB	02061	B4DD
542B	345E	8ADC	02062	B4DE
5CB8	345F	8ADD	02063	B4DF
5DCC	3460	8ADE	02064	B4E0
73A9	3461	8ADF	02065	B4E1
764C	3462	8AE0	02066	B4E2
773C	3463	8AE1	02067	B4E3
5CA9	3464	8AE2	02068	B4E4
E37E				B4E5
8D0B	3466	8AE4	02070	B4E6
96C1	3467	8AE5	02071	B4E7
9811	3468	8AE6	02072	B4E8
9854	3469	8AE7	02073	B4E9
9858	346A	8AE8	02074	B4EA
4F01	346B	8AE9	02075	B4EB
4F0E	346C	8AEA	02076	B4EC
5371	346D	8AEB	02077	B4ED
559C	346E	8AEC	02078	B4EE
5668	346F	8AED	02079	B4EF
57FA	3470	8AEE	02080	B4F0
5947	3471	8AEF	02081	B4F1
5B09	3472	8AF0	02082	B4F2
5BC4	3473	8AF1	02083	B4F3
5C90	3474	8AF2	02084	B4F4
5E0C	3475	8AF3	02085	B4F5
5E7E	3476	8AF4	02086	B4F6
5FCC	3477	8AF5	02087	B4F7
63EE	3478	8AF6	02088	B4F8
673A	3479	8AF7	02089	B4F9
65D7	347A	8AF8	02090	B4FA
65E2	347B	8AF9	02091	B4FB
671F	347C	8AFA	02092	B4FC
68CB	347D	8AFB	02093	B4FD
68C4	347E	8AFC	02094	B4FE
6A5F	3521	8B40	02101	B5A1
E1F5				B5A2
6BC5	3523	8B42	02103	B5A3
6C17	3524	8B43	02104	B5A4
6C7D	3525	8B44	02105	B5A5
757F	3526	8B45	02106	B5A6
7948	3527	8B46	02107	B5A7
5B63	3528	8B47	02108	B5A8
7A00	3529	8B48	02109	B5A9
7D00	352A	8B49	02110	B5AA
E21B				B5AB
898F	352C	8B4B	02112	B5AC
8A18	352D	8B4C	02113	B5AD
8CB4	352E	8B4D	02114	B5AE
8D77	352F	8B4E	02115	B5AF
8ECC	3530	8B4F	02116	B5B0
8F1D	3531	8B50	02117	B5B1
98E2	3532	8B51	02118	B5B2
9A0E	3533	8B52	02119	B5B3
9B3C	3534	8B53	02120	B5B4
4E80	3535	8B54	02121	B5B5
507D	3536	8B55	02122	B5B6
5100	3537	8B56	02123	B5B7
5993	3538	8B57	02124	B5B8
5B9C	3539	8B58	02125	B5B9
622F	353A	8B59	02126	B5BA
6280	353B	8B5A	02127	B5BB
64EC	353C	8B5B	02128	B5BC
6B3A	353D	8B5C	02129	B5BD
72A0	353E	8B5D	02130	B5BE
7591	353F	8B5E	02131	B5BF
E32D				B5C0
7FA9	3541	8B60	02133	B5C1
87FB	3542	8B61	02134	B5C2
8ABC	3543	8B62	02135	B5C3
8B70	3544	8B63	02136	B5C4
63AC	3545	8B64	02137	B5C5
83CA	3546	8B65	02138	B5C6
97A0	3547	8B66	02139	B5C7
5409	3548	8B67	02140	B5C8
5403	3549	8B68	02141	B5C9
55AB	354A	8B69	02142	B5CA
6854	354B	8B6A	02143	B5CB
6A58	354C	8B6B	02144	B5CC
8A70	354D	8B6C	02145	B5CD
7827	354E	8B6D	02146	B5CE
6775	354F	8B6E	02147	B5CF
9ECD	3550	8B6F	02148	B5D0
5374	3551	8B70	02149	B5D1
5BA2	3552	8B71	02150	B5D2
811A	3553	8B72	02151	B5D3
8650	3554	8B73	02152	B5D4
9006	3555	8B74	02153	B5D5
4E18	3556	8B75	02154	B5D6
E137				B5D7
4EC7	3558	8B77	02156	B5D8
4F11	3559	8B78	02157	B5D9
53CA	355A	8B79	02158	B5DA
5438	355B	8B7A	02159	B5DB
5BAE	355C	8B7B	02160	B5DC
5F13	355D	8B7C	02161	B5DD
6025	355E	8B7D	02162	B5DE
6551	355F	8B7E	02163	B5DF
673D	3560	8B80	02164	B5E0
6C42	3561	8B81	02165	B5E1
E2B2				B5E2
6CE3	3563	8B83	02167	B5E3
7078	3564	8B84	02168	B5E4
7403	3565	8B85	02169	B5E5
7A76	3566	8B86	02170	B5E6
7AAE	3567	8B87	02171	B5E7
E341				B5E8
7D1A	3569	8B89	02173	B5E9
7CFE	356A	8B8A	02174	B5EA
7D66	356B	8B8B	02175	B5EB
65E7	356C	8B8C	02176	B5EC
725B	356D	8B8D	02177	B5ED
53BB	356E	8B8E	02178	B5EE
5C45	356F	8B8F	02179	B5EF
5DE8	3570	8B90	02180	B5F0
62D2	3571	8B91	02181	B5F1
62E0	3572	8B92	02182	B5F2
6319	3573	8B93	02183	B5F3
E294				B5F4
865A	3575	8B95	02185	B5F5
8A31	3576	8B96	02186	B5F6
8DDD	3577	8B97	02187	B5F7
92F8	3578	8B98	02188	B5F8
6F01	3579	8B99	02189	B5F9
79A6	357A	8B9A	02190	B5FA
9B5A	357B	8B9B	02191	B5FB
4EA8	357C	8B9C	02192	B5FC
4EAB	357D	8B9D	02193	B5FD
4EAC	357E	8B9E	02194	B5FE
4F9B	3621	8B9F	02201	B6A1
4FE0				B6A2
50D1	3623	8BA1	02203	B6A3
5147	3624	8BA2	02204	B6A4
7AF6	3625	8BA3	02205	B6A5
5171	3626	8BA4	02206	B6A6
51F6	3627	8BA5	02207	B6A7
5354	3628	8BA6	02208	B6A8
5321	3629	8BA7	02209	B6A9
E189				B6AA
53EB	362B	8BA9	02211	B6AB
55AC	362C	8BAA	02212	B6AC
5883	362D	8BAB	02213	B6AD
5CE1	362E	8BAC	02214	B6AE
5F37	362F	8BAD	02215	B6AF
5F4A	3630	8BAE	02216	B6B0
602F	3631	8BAF	02217	B6B1
6050	3632	8BB0	02218	B6B2
606D	3633	8BB1	02219	B6B3
631F	3634	8BB2	02220	B6B4
6559	3635	8BB3	02221	B6B5
6A4B	3636	8BB4	02222	B6B6
6CC1	3637	8BB5	02223	B6B7
72C2	3638	8BB6	02224	B6B8
72ED	3639	8BB7	02225	B6B9
77EF	363A	8BB8	02226	B6BA
80F8	363B	8BB9	02227	B6BB
8105	363C	8BBA	02228	B6BC
8208	363D	8BBB	02229	B6BD
854E	363E	8BBC	02230	B6BE
90F7	363F	8BBD	02231	B6BF
93E1	3640	8BBE	02232	B6C0
97FF	3641	8BBF	02233	B6C1
E4BE				B6C2
9A5A	3643	8BC1	02235	B6C3
4EF0	3644	8BC2	02236	B6C4
51DD	3645	8BC3	02237	B6C5
582F	7421	EA9F	08401	B6C6
E790				B6C7
696D	3648	8BC6	02240	B6C8
5C40	3649	8BC7	02241	B6C9
66F2	364A	8BC8	02242	B6CA
6975	364B	8BC9	02243	B6CB
7389	364C	8BCA	02244	B6CC
6850	364D	8BCB	02245	B6CD
7C81	364E	8BCC	02246	B6CE
E157				B6CF
52E4	3650	8BCE	02248	B6D0
5747	3651	8BCF	02249	B6D1
5DFE	3652	8BD0	02250	B6D2
9326	3653	8BD1	02251	B6D3
65A4	3654	8BD2	02252	B6D4
6B23	3655	8BD3	02253	B6D5
6B3D	3656	8BD4	02254	B6D6
7434	3657	8BD5	02255	B6D7
7981	3658	8BD6	02256	B6D8
79BD	3659	8BD7	02257	B6D9
7B4B	365A	8BD8	02258	B6DA
7DCA	365B	8BD9	02259	B6DB
82B9	365C	8BDA	02260	B6DC
E7A7				B6DD
887F	365E	8BDC	02262	B6DE
895F	365F	8BDD	02263	B6DF
8B39	3660	8BDE	02264	B6E0
8FD1	3661	8BDF	02265	B6E1
91D1	3662	8BE0	02266	B6E2
541F	3663	8BE1	02267	B6E3
9280	3664	8BE2	02268	B6E4
4E5D	3665	8BE3	02269	B6E5
5036	3666	8BE4	02270	B6E6
53E5	3667	8BE5	02271	B6E7
533A	3668	8BE6	02272	B6E8
72D7	3669	8BE7	02273	B6E9
E799				B6EA
E321				B6EB
82E6	366C	8BEA	02276	B6EC
E411				B6ED
99C6	366E	8BEC	02278	B6EE
99C8	366F	8BED	02279	B6EF
99D2	3670	8BEE	02280	B6F0
5177	3671	8BEF	02281	B6F1
611A	3672	8BF0	02282	B6F2
865E	3673	8BF1	02283	B6F3
E1A3				B6F4
7A7A	3675	8BF3	02285	B6F5
5076	3676	8BF4	02286	B6F6
5BD3	3677	8BF5	02287	B6F7
9047	3678	8BF6	02288	B6F8
9685	3679	8BF7	02289	B6F9
4E32	367A	8BF8	02290	B6FA
E2A8				B6FB
91E7	367C	8BFA	02292	B6FC
E1EC				B6FD
5C48	367E	8BFC	02294	B6FE
6398	3721	8C40	02301	B7A1
7A9F	3722	8C41	02302	B7A2
6C93	3723	8C42	02303	B7A3
E4A1				B7A4
8F61	3725	8C44	02305	B7A5
7AAA	3726	8C45	02306	B7A6
718A	3727	8C46	02307	B7A7
9688	3728	8C47	02308	B7A8
7C82	3729	8C48	02309	B7A9
6817	372A	8C49	02310	B7AA
7E70	372B	8C4A	02311	B7AB
6851	372C	8C4B	02312	B7AC
936C	372D	8C4C	02313	B7AD
52F2	372E	8C4D	02314	B7AE
541B	372F	8C4E	02315	B7AF
85AB	3730	8C4F	02316	B7B0
8A13	3731	8C50	02317	B7B1
7FA4	3732	8C51	02318	B7B2
8ECD	3733	8C52	02319	B7B3
90E1	3734	8C53	02320	B7B4
5366	3735	8C54	02321	B7B5
8888	3736	8C55	02322	B7B6
E32C				B7B7
4FC2	3738	8C57	02324	B7B8
50BE	3739	8C58	02325	B7B9
5211	373A	8C59	02326	B7BA
5144	373B	8C5A	02327	B7BB
5553	373C	8C5B	02328	B7BC
572D	373D	8C5C	02329	B7BD
73EA	373E	8C5D	02330	B7BE
578B	373F	8C5E	02331	B7BF
5951	3740	8C5F	02332	B7C0
5F62	3741	8C60	02333	B7C1
5F84	3742	8C61	02334	B7C2
6075	3743	8C62	02335	B7C3
6176	3744	8C63	02336	B7C4
E22C				B7C5
61A9	3746	8C65	02338	B7C6
63B2	3747	8C66	02339	B7C7
643A	3748	8C67	02340	B7C8
656C	3749	8C68	02341	B7C9
666F	374A	8C69	02342	B7CA
6842	374B	8C6A	02343	B7CB
6E13	374C	8C6B	02344	B7CC
7566	374D	8C6C	02345	B7CD
7A3D	374E	8C6D	02346	B7CE
7CFB	374F	8C6E	02347	B7CF
7D4C	3750	8C6F	02348	B7D0
7D99	3751	8C70	02349	B7D1
7E6B				B7D2
7F6B	3753	8C72	02351	B7D3
830E	3754	8C73	02352	B7D4
E722				B7D5
86CD	3756	8C75	02354	B7D6
8A08	3757	8C76	02355	B7D7
8A63	3758	8C77	02356	B7D8
8B66	3759	8C78	02357	B7D9
8EFD	375A	8C79	02358	B7DA
9838	7074	E8F2	08084	B7DB
9D8F	375C	8C7B	02360	B7DC
82B8	375D	8C7C	02361	B7DD
8FCE	375E	8C7D	02362	B7DE
9BE8	375F	8C7E	02363	B7DF
5287	3760	8C80	02364	B7E0
621F	3761	8C81	02365	B7E1
6483	3762	8C82	02366	B7E2
6FC0	3763	8C83	02367	B7E3
E496				B7E4
6841	3765	8C85	02369	B7E5
5091	3766	8C86	02370	B7E6
6B20	3767	8C87	02371	B7E7
6C7A	3768	8C88	02372	B7E8
6F54	3769	8C89	02373	B7E9
E33B				B7EA
7D50	376B	8C8B	02375	B7EB
8840	376C	8C8C	02376	B7EC
8A23	376D	8C8D	02377	B7ED
6708	376E	8C8E	02378	B7EE
4EF6	376F	8C8F	02379	B7EF
5039	3770	8C90	02380	B7F0
E150				B7F1
5065	3772	8C92	02382	B7F2
517C	3773	8C93	02383	B7F3
5238	3774	8C94	02384	B7F4
5263	3775	8C95	02385	B7F5
55A7	3776	8C96	02386	B7F6
570F	3777	8C97	02387	B7F7
5805	3778	8C98	02388	B7F8
E1D7				B7F9
5EFA	377A	8C9A	02390	B7FA
61B2	377B	8C9B	02391	B7FB
61F8	377C	8C9C	02392	B7FC
E241				B7FD
E247				B7FE
691C	3821	8C9F	02401	B8A1
6A29	3822	8CA0	02402	B8A2
727D	3823	8CA1	02403	B8A3
72AC	3824	8CA2	02404	B8A4
732E	3825	8CA3	02405	B8A5
7814	3826	8CA4	02406	B8A6
786F	3827	8CA5	02407	B8A7
7D79	3828	8CA6	02408	B8A8
770C	3829	8CA7	02409	B8A9
80A9	382A	8CA8	02410	B8AA
898B	382B	8CA9	02411	B8AB
8B19	382C	8CAA	02412	B8AC
8CE2	382D	8CAB	02413	B8AD
8ED2	382E	8CAC	02414	B8AE
9063	382F	8CAD	02415	B8AF
E47D				B8B0
967A	3831	8CAF	02417	B8B1
9855	3832	8CB0	02418	B8B2
9A13	3833	8CB1	02419	B8B3
9E7C				B8B4
5143	3835	8CB3	02421	B8B5
539F	3836	8CB4	02422	B8B6
53B3	3837	8CB5	02423	B8B7
5E7B	3838	8CB6	02424	B8B8
5F26	3839	8CB7	02425	B8B9
6E1B	383A	8CB8	02426	B8BA
6E90	383B	8CB9	02427	B8BB
7384	383C	8CBA	02428	B8BC
73FE	383D	8CBB	02429	B8BD
7D43	383E	8CBC	02430	B8BE
8237	383F	8CBD	02431	B8BF
8A00	3840	8CBE	02432	B8C0
E3EF				B8C1
9650	3842	8CC0	02434	B8C2
4E4E	3843	8CC1	02435	B8C3
500B	3844	8CC2	02436	B8C4
53E4	3845	8CC3	02437	B8C5
547C	3846	8CC4	02438	B8C6
56FA	3847	8CC5	02439	B8C7
59D1	3848	8CC6	02440	B8C8
5B64	3849	8CC7	02441	B8C9
5DF1	384A	8CC8	02442	B8CA
5EAB	384B	8CC9	02443	B8CB
5F27	384C	8CCA	02444	B8CC
6238	384D	8CCB	02445	B8CD
6545	384E	8CCC	02446	B8CE
67AF	384F	8CCD	02447	B8CF
6E56	3850	8CCE	02448	B8D0
72D0	3851	8CCF	02449	B8D1
7CCA	3852	8CD0	02450	B8D2
88B4	3853	8CD1	02451	B8D3
80A1	3854	8CD2	02452	B8D4
80E1	3855	8CD3	02453	B8D5
83F0	3856	8CD4	02454	B8D6
864E	3857	8CD5	02455	B8D7
8A87	3858	8CD6	02456	B8D8
8DE8	3859	8CD7	02457	B8D9
9237	385A	8CD8	02458	B8DA
96C7	385B	8CD9	02459	B8DB
9867	385C	8CDA	02460	B8DC
9F13	385D	8CDB	02461	B8DD
4E94	385E	8CDC	02462	B8DE
4E92	385F	8CDD	02463	B8DF
4F0D	3860	8CDE	02464	B8E0
5348	3861	8CDF	02465	B8E1
5449	3862	8CE0	02466	B8E2
543E	3863	8CE1	02467	B8E3
5A2F	3864	8CE2	02468	B8E4
5F8C	3865	8CE3	02469	B8E5
5FA1	3866	8CE4	02470	B8E6
609F	3867	8CE5	02471	B8E7
68A7	3868	8CE6	02472	B8E8
6A8E	3869	8CE7	02473	B8E9
745A	386A	8CE8	02474	B8EA
7881	386B	8CE9	02475	B8EB
8A9E	386C	8CEA	02476	B8EC
8AA4	386D	8CEB	02477	B8ED
8B77	386E	8CEC	02478	B8EE
9190	386F	8CED	02479	B8EF
4E5E	3870	8CEE	02480	B8F0
9BC9	3871	8CEF	02481	B8F1
4EA4	3872	8CF0	02482	B8F2
E14C				B8F3
4FAF	3874	8CF2	02484	B8F4
5019	3875	8CF3	02485	B8F5
5016	3876	8CF4	02486	B8F6
5149	3877	8CF5	02487	B8F7
E161				B8F8
529F	3879	8CF7	02489	B8F9
52B9	387A	8CF8	02490	B8FA
52FE	387B	8CF9	02491	B8FB
539A	387C	8CFA	02492	B8FC
53E3	387D	8CFB	02493	B8FD
5411	387E	8CFC	02494	B8FE
540E	3921	8D40	02501	B9A1
5589	3922	8D41	02502	B9A2
5751	3923	8D42	02503	B9A3
57A2	3924	8D43	02504	B9A4
597D	3925	8D44	02505	B9A5
5B54	3926	8D45	02506	B9A6
5B5D	3927	8D46	02507	B9A7
5B8F	3928	8D47	02508	B9A8
5DE5	3929	8D48	02509	B9A9
5DE7	392A	8D49	02510	B9AA
E1F2				B9AB
5E78	392C	8D4B	02512	B9AC
5E83	392D	8D4C	02513	B9AD
5E9A	392E	8D4D	02514	B9AE
5EB7	392F	8D4E	02515	B9AF
5F18	3930	8D4F	02516	B9B0
6052	3931	8D50	02517	B9B1
614C	3932	8D51	02518	B9B2
6297	3933	8D52	02519	B9B3
62D8	3934	8D53	02520	B9B4
63A7	3935	8D54	02521	B9B5
653B	3936	8D55	02522	B9B6
E694				B9B7
6643	3938	8D57	02524	B9B8
66F4	3939	8D58	02525	B9B9
676D	393A	8D59	02526	B9BA
6821	393B	8D5A	02527	B9BB
E28F				B9BC
69CB	393D	8D5C	02529	B9BD
6C5F	393E	8D5D	02530	B9BE
6D2A	393F	8D5E	02531	B9BF
6D69	3940	8D5F	02532	B9C0
6E2F	3941	8D60	02533	B9C1
E2C8				B9C2
7532	3943	8D62	02535	B9C3
7687	3944	8D63	02536	B9C4
786C	3945	8D64	02537	B9C5
7A3F	3946	8D65	02538	B9C6
7CE0	3947	8D66	02539	B9C7
7D05	3948	8D67	02540	B9C8
7D18	3949	8D68	02541	B9C9
7D5E	394A	8D69	02542	B9CA
7DB1	394B	8D6A	02543	B9CB
8015	394C	8D6B	02544	B9CC
8003	394D	8D6C	02545	B9CD
80AF	394E	8D6D	02546	B9CE
80B1	394F	8D6E	02547	B9CF
8154	3950	8D6F	02548	B9D0
818F	3951	8D70	02549	B9D1
822A	3952	8D71	02550	B9D2
8352	3953	8D72	02551	B9D3
884C	3954	8D73	02552	B9D4
8861	3955	8D74	02553	B9D5
8B1B	3956	8D75	02554	B9D6
8CA2	3957	8D76	02555	B9D7
8CFC	3958	8D77	02556	B9D8
90CA	3959	8D78	02557	B9D9
9175	395A	8D79	02558	B9DA
9271	395B	8D7A	02559	B9DB
7926	6268	E1E6	06672	B9DC
92FC	395D	8D7C	02561	B9DD
95A4	395E	8D7D	02562	B9DE
964D	395F	8D7E	02563	B9DF
9805	3960	8D80	02564	B9E0
9999	3961	8D81	02565	B9E1
9AD8	3962	8D82	02566	B9E2
9D3B	3963	8D83	02567	B9E3
525B	3964	8D84	02568	B9E4
52AB	3965	8D85	02569	B9E5
53F7	3966	8D86	02570	B9E6
5408	3967	8D87	02571	B9E7
E1C6				B9E8
62F7	3969	8D89	02573	B9E9
E2D7				B9EA
E3FF				B9EB
8F5F	396C	8D8C	02576	B9EC
9EB4				B9ED
514B	396E	8D8E	02578	B9EE
523B	396F	8D8F	02579	B9EF
544A	3970	8D90	02580	B9F0
56FD	3971	8D91	02581	B9F1
7A40	3972	8D92	02582	B9F2
9177	3973	8D93	02583	B9F3
E4DC				B9F4
9ED2	3975	8D95	02585	B9F5
7344	3976	8D96	02586	B9F6
6F09	3977	8D97	02587	B9F7
8170	3978	8D98	02588	B9F8
E30E				B9F9
5FFD	397A	8D9A	02590	B9FA
60DA	397B	8D9B	02591	B9FB
9AA8	397C	8D9C	02592	B9FC
72DB	397D	8D9D	02593	B9FD
8FBC	397E	8D9E	02594	B9FE
6B64	3A21	8D9F	02601	BAA1
9803	3A22	8DA0	02602	BAA2
4ECA	3A23	8DA1	02603	BAA3
56F0	3A24	8DA2	02604	BAA4
5764	3A25	8DA3	02605	BAA5
58BE	3A26	8DA4	02606	BAA6
5A5A	3A27	8DA5	02607	BAA7
6068	3A28	8DA6	02608	BAA8
61C7	3A29	8DA7	02609	BAA9
660F	3A2A	8DA8	02610	BAAA
6606	3A2B	8DA9	02611	BAAB
6839	3A2C	8DAA	02612	BAAC
68B1	3A2D	8DAB	02613	BAAD
6DF7	3A2E	8DAC	02614	BAAE
75D5	3A2F	8DAD	02615	BAAF
7D3A	3A30	8DAE	02616	BAB0
826E	3A31	8DAF	02617	BAB1
9B42	3A32	8DB0	02618	BAB2
4E9B	3A33	8DB1	02619	BAB3
4F50	3A34	8DB2	02620	BAB4
E18C				BAB5
5506	3A36	8DB4	02622	BAB6
5D6F	3A37	8DB5	02623	BAB7
5DE6	3A38	8DB6	02624	BAB8
5DEE	3A39	8DB7	02625	BAB9
67FB	3A3A	8DB8	02626	BABA
6C99	3A3B	8DB9	02627	BABB
7473	3A3C	8DBA	02628	BABC
7802	3A3D	8DBB	02629	BABD
8A50	3A3E	8DBC	02630	BABE
9396	3A3F	8DBD	02631	BABF
88DF	3A40	8DBE	02632	BAC0
5750	3A41	8DBF	02633	BAC1
5EA7	3A42	8DC0	02634	BAC2
632B	3A43	8DC1	02635	BAC3
50B5	3A44	8DC2	02636	BAC4
50AC	3A45	8DC3	02637	BAC5
518D	3A46	8DC4	02638	BAC6
6700	3A47	8DC5	02639	BAC7
54C9	3A48	8DC6	02640	BAC8
585E	3A49	8DC7	02641	BAC9
59BB	3A4A	8DC8	02642	BACA
5BB0	3A4B	8DC9	02643	BACB
5F69	3A4C	8DCA	02644	BACC
624D	3A4D	8DCB	02645	BACD
63A1	3A4E	8DCC	02646	BACE
683D	3A4F	8DCD	02647	BACF
6B73	3A50	8DCE	02648	BAD0
6E08	3A51	8DCF	02649	BAD1
707D	3A52	8DD0	02650	BAD2
E471				BAD3
7280	3A54	8DD2	02652	BAD4
7815	3A55	8DD3	02653	BAD5
7826	3A56	8DD4	02654	BAD6
796D	3A57	8DD5	02655	BAD7
658E	3A58	8DD6	02656	BAD8
7D30	3A59	8DD7	02657	BAD9
83DC	3A5A	8DD8	02658	BADA
88C1	3A5B	8DD9	02659	BADB
8F09	3A5C	8DDA	02660	BADC
969B	3A5D	8DDB	02661	BADD
5264	3A5E	8DDC	02662	BADE
5728	3A5F	8DDD	02663	BADF
6750	3A60	8DDE	02664	BAE0
7F6A	3A61	8DDF	02665	BAE1
8CA1	3A62	8DE0	02666	BAE2
E169				BAE3
5742	3A64	8DE2	02668	BAE4
962A	3A65	8DE3	02669	BAE5
583A	3A66	8DE4	02670	BAE6
E29B				BAE7
80B4	3A68	8DE6	02672	BAE8
54B2	3A69	8DE7	02673	BAE9
5D0E	3A6A	8DE8	02674	BAEA
57FC	3A6B	8DE9	02675	BAEB
7895	3A6C	8DEA	02676	BAEC
9DFA	3A6D	8DEB	02677	BAED
4F5C	3A6E	8DEC	02678	BAEE
524A	3A6F	8DED	02679	BAEF
548B	3A70	8DEE	02680	BAF0
643E	3A71	8DEF	02681	BAF1
6628	3A72	8DF0	02682	BAF2
6714	3A73	8DF1	02683	BAF3
E6A2				BAF4
7A84	3A75	8DF3	02685	BAF5
7B56	3A76	8DF4	02686	BAF6
7D22	3A77	8DF5	02687	BAF7
932F	3A78	8DF6	02688	BAF8
685C	3A79	8DF7	02689	BAF9
9BAD	3A7A	8DF8	02690	BAFA
7B39	3A7B	8DF9	02691	BAFB
5319	3A7C	8DFA	02692	BAFC
518A	3A7D	8DFB	02693	BAFD
5237	3A7E	8DFC	02694	BAFE
5BDF	3B21	8E40	02701	BBA1
62F6	3B22	8E41	02702	BBA2
64AE	3B23	8E42	02703	BBA3
64E6	3B24	8E43	02704	BBA4
672D	3B25	8E44	02705	BBA5
6BBA	3B26	8E45	02706	BBA6
E3B7				BBA7
96D1	3B28	8E47	02708	BBA8
7690	3B29	8E48	02709	BBA9
E4D1				BBAA
634C	3B2B	8E4A	02711	BBAB
E47C				BBAC
E4CC				BBAD
76BF	3B2E	8E4D	02714	BBAE
E78F				BBAF
4E09	3B30	8E4F	02716	BBB0
5098	3B31	8E50	02717	BBB1
53C2	3B32	8E51	02718	BBB2
5C71	3B33	8E52	02719	BBB3
60E8	3B34	8E53	02720	BBB4
6492	3B35	8E54	02721	BBB5
6563	3B36	8E55	02722	BBB6
685F	3B37	8E56	02723	BBB7
71E6	3B38	8E57	02724	BBB8
E2FF				BBB9
7523	3B3A	8E59	02726	BBBA
7B97	3B3B	8E5A	02727	BBBB
7E82	3B3C	8E5B	02728	BBBC
8695	3B3D	8E5C	02729	BBBD
8B83	3B3E	8E5D	02730	BBBE
8CDB	3B3F	8E5E	02731	BBBF
9178	3B40	8E5F	02732	BBC0
E4BB				BBC1
65AC	3B42	8E61	02734	BBC2
66AB	3B43	8E62	02735	BBC3
6B8B	3B44	8E63	02736	BBC4
4ED5	3B45	8E64	02737	BBC5
4ED4	3B46	8E65	02738	BBC6
4F3A	3B47	8E66	02739	BBC7
4F7F	3B48	8E67	02740	BBC8
523A	3B49	8E68	02741	BBC9
53F8	3B4A	8E69	02742	BBCA
53F2	3B4B	8E6A	02743	BBCB
55E3	3B4C	8E6B	02744	BBCC
56DB	3B4D	8E6C	02745	BBCD
58EB	3B4E	8E6D	02746	BBCE
59CB	3B4F	8E6E	02747	BBCF
59C9	3B50	8E6F	02748	BBD0
59FF	3B51	8E70	02749	BBD1
5B50	3B52	8E71	02750	BBD2
5C4D	3B53	8E72	02751	BBD3
5E02	3B54	8E73	02752	BBD4
5E2B	3B55	8E74	02753	BBD5
5FD7	3B56	8E75	02754	BBD6
601D	3B57	8E76	02755	BBD7
6307	3B58	8E77	02756	BBD8
652F	3B59	8E78	02757	BBD9
5B5C	3B5A	8E79	02758	BBDA
65AF	3B5B	8E7A	02759	BBDB
65BD	3B5C	8E7B	02760	BBDC
65E8	3B5D	8E7C	02761	BBDD
679D	3B5E	8E7D	02762	BBDE
6B62	3B5F	8E7E	02763	BBDF
6B7B	3B60	8E80	02764	BBE0
6C0F	3B61	8E81	02765	BBE1
7345	3B62	8E82	02766	BBE2
7949	3B63	8E83	02767	BBE3
79C1	3B64	8E84	02768	BBE4
7CF8	3B65	8E85	02769	BBE5
7D19	3B66	8E86	02770	BBE6
7D2B	3B67	8E87	02771	BBE7
80A2	3B68	8E88	02772	BBE8
8102	3B69	8E89	02773	BBE9
81F3	3B6A	8E8A	02774	BBEA
8996	3B6B	8E8B	02775	BBEB
8A5E	3B6C	8E8C	02776	BBEC
8A69	3B6D	8E8D	02777	BBED
8A66	3B6E	8E8E	02778	BBEE
8A8C	3B6F	8E8F	02779	BBEF
8AEE	3B70	8E90	02780	BBF0
8CC7	3B71	8E91	02781	BBF1
8CDC	3B72	8E92	02782	BBF2
96CC	3B73	8E93	02783	BBF3
98FC	3B74	8E94	02784	BBF4
6B6F	3B75	8E95	02785	BBF5
4E8B	3B76	8E96	02786	BBF6
4F3C	3B77	8E97	02787	BBF7
4F8D	3B78	8E98	02788	BBF8
5150	3B79	8E99	02789	BBF9
5B57	3B7A	8E9A	02790	BBFA
5BFA	3B7B	8E9B	02791	BBFB
6148	3B7C	8E9C	02792	BBFC
6301	3B7D	8E9D	02793	BBFD
6642	3B7E	8E9E	02794	BBFE
6B21	3C21	8E9F	02801	BCA1
6ECB	3C22	8EA0	02802	BCA2
6CBB	3C23	8EA1	02803	BCA3
723E	3C24	8EA2	02804	BCA4
74BD	3C25	8EA3	02805	BCA5
75D4	3C26	8EA4	02806	BCA6
78C1	3C27	8EA5	02807	BCA7
793A	3C28	8EA6	02808	BCA8
800C	3C29	8EA7	02809	BCA9
8033	3C2A	8EA8	02810	BCAA
81EA	3C2B	8EA9	02811	BCAB
8494	3C2C	8EAA	02812	BCAC
8F9E	3C2D	8EAB	02813	BCAD
6C50	3C2E	8EAC	02814	BCAE
9E7F	3C2F	8EAD	02815	BCAF
5F0F	3C30	8EAE	02816	BCB0
8B58	3C31	8EAF	02817	BCB1
9D2B	3C32	8EB0	02818	BCB2
7AFA	3C33	8EB1	02819	BCB3
8EF8	3C34	8EB2	02820	BCB4
5B8D	3C35	8EB3	02821	BCB5
96EB	3C36	8EB4	02822	BCB6
4E03	3C37	8EB5	02823	BCB7
53F1	3C38	8EB6	02824	BCB8
57F7	3C39	8EB7	02825	BCB9
5931	3C3A	8EB8	02826	BCBA
5AC9	3C3B	8EB9	02827	BCBB
5BA4	3C3C	8EBA	02828	BCBC
6089	3C3D	8EBB	02829	BCBD
6E7F	3C3E	8EBC	02830	BCBE
6F06	3C3F	8EBD	02831	BCBF
75BE	3C40	8EBE	02832	BCC0
8CEA	3C41	8EBF	02833	BCC1
5B9F	3C42	8EC0	02834	BCC2
8500	3C43	8EC1	02835	BCC3
7BE0	3C44	8EC2	02836	BCC4
5072	3C45	8EC3	02837	BCC5
67F4	3C46	8EC4	02838	BCC6
829D	3C47	8EC5	02839	BCC7
5C62				BCC8
8602	6922	E541	07302	BCC9
7E1E	3C4A	8EC8	02842	BCCA
820E	3C4B	8EC9	02843	BCCB
5199	3C4C	8ECA	02844	BCCC
5C04	3C4D	8ECB	02845	BCCD
6368	3C4E	8ECC	02846	BCCE
8D66	3C4F	8ECD	02847	BCCF
659C	3C50	8ECE	02848	BCD0
716E	3C51	8ECF	02849	BCD1
793E	3C52	8ED0	02850	BCD2
7D17	3C53	8ED1	02851	BCD3
8005	3C54	8ED2	02852	BCD4
8B1D	3C55	8ED3	02853	BCD5
8ECA	3C56	8ED4	02854	BCD6
E452				BCD7
86C7	3C58	8ED6	02856	BCD8
90AA	3C59	8ED7	02857	BCD9
501F	3C5A	8ED8	02858	BCDA
52FA	3C5B	8ED9	02859	BCDB
5C3A	3C5C	8EDA	02860	BCDC
E281				BCDD
E2DF				BCDE
7235	3C5F	8EDD	02863	BCDF
914C	3C60	8EDE	02864	BCE0
91C8	3C61	8EDF	02865	BCE1
932B	3C62	8EE0	02866	BCE2
82E5	3C63	8EE1	02867	BCE3
5BC2	3C64	8EE2	02868	BCE4
5F31	3C65	8EE3	02869	BCE5
60F9	3C66	8EE4	02870	BCE6
4E3B	3C67	8EE5	02871	BCE7
53D6	3C68	8EE6	02872	BCE8
5B88	3C69	8EE7	02873	BCE9
624B	3C6A	8EE8	02874	BCEA
6731	3C6B	8EE9	02875	BCEB
6B8A	3C6C	8EEA	02876	BCEC
72E9	3C6D	8EEB	02877	BCED
73E0	3C6E	8EEC	02878	BCEE
7A2E	3C6F	8EED	02879	BCEF
816B	3C70	8EEE	02880	BCF0
8DA3	3C71	8EEF	02881	BCF1
9152	3C72	8EF0	02882	BCF2
9996	3C73	8EF1	02883	BCF3
5112	3C74	8EF2	02884	BCF4
53D7	3C75	8EF3	02885	BCF5
546A	3C76	8EF4	02886	BCF6
5BFF	3C77	8EF5	02887	BCF7
6388	3C78	8EF6	02888	BCF8
6A39	3C79	8EF7	02889	BCF9
7DAC	3C7A	8EF8	02890	BCFA
9700	3C7B	8EF9	02891	BCFB
56DA	3C7C	8EFA	02892	BCFC
53CE	3C7D	8EFB	02893	BCFD
5468	3C7E	8EFC	02894	BCFE
5B97	3D21	8F40	02901	BDA1
5C31	3D22	8F41	02902	BDA2
5DDE	3D23	8F42	02903	BDA3
4FEE	3D24	8F43	02904	BDA4
6101	3D25	8F44	02905	BDA5
62FE	3D26	8F45	02906	BDA6
6D32	3D27	8F46	02907	BDA7
79C0	3D28	8F47	02908	BDA8
79CB	3D29	8F48	02909	BDA9
7D42	3D2A	8F49	02910	BDAA
7E61				BDAB
7FD2	3D2C	8F4B	02912	BDAC
81ED	3D2D	8F4C	02913	BDAD
821F	3D2E	8F4D	02914	BDAE
8490	3D2F	8F4E	02915	BDAF
8846	3D30	8F4F	02916	BDB0
8972	3D31	8F50	02917	BDB1
8B90	3D32	8F51	02918	BDB2
8E74	3D33	8F52	02919	BDB3
8F2F	3D34	8F53	02920	BDB4
9031	3D35	8F54	02921	BDB5
E466				BDB6
916C	3D37	8F56	02923	BDB7
96C6	3D38	8F57	02924	BDB8
919C	3D39	8F58	02925	BDB9
4EC0	3D3A	8F59	02926	BDBA
4F4F	3D3B	8F5A	02927	BDBB
5145	3D3C	8F5B	02928	BDBC
5341	3D3D	8F5C	02929	BDBD
5F93	3D3E	8F5D	02930	BDBE
620E	3D3F	8F5E	02931	BDBF
67D4	3D40	8F5F	02932	BDC0
6C41	3D41	8F60	02933	BDC1
6E0B	3D42	8F61	02934	BDC2
7363	3D43	8F62	02935	BDC3
7E26	3D44	8F63	02936	BDC4
91CD	3D45	8F64	02937	BDC5
9283	3D46	8F65	02938	BDC6
53D4	3D47	8F66	02939	BDC7
5919	3D48	8F67	02940	BDC8
5BBF	3D49	8F68	02941	BDC9
6DD1	3D4A	8F69	02942	BDCA
795D	3D4B	8F6A	02943	BDCB
7E2E	3D4C	8F6B	02944	BDCC
7C9B	3D4D	8F6C	02945	BDCD
587E	3D4E	8F6D	02946	BDCE
719F	3D4F	8F6E	02947	BDCF
51FA	3D50	8F6F	02948	BDD0
8853	3D51	8F70	02949	BDD1
8FF0	3D52	8F71	02950	BDD2
4FCA	3D53	8F72	02951	BDD3
5CFB	3D54	8F73	02952	BDD4
6625	3D55	8F74	02953	BDD5
77AC	3D56	8F75	02954	BDD6
7AE3	3D57	8F76	02955	BDD7
821C	3D58	8F77	02956	BDD8
99FF	3D59	8F78	02957	BDD9
51C6	3D5A	8F79	02958	BDDA
5FAA	3D5B	8F7A	02959	BDDB
65EC	3D5C	8F7B	02960	BDDC
E297				BDDD
6B89	3D5E	8F7D	02962	BDDE
6DF3	3D5F	8F7E	02963	BDDF
6E96	3D60	8F80	02964	BDE0
6F64	3D61	8F81	02965	BDE1
E79F				BDE2
7D14	3D63	8F83	02967	BDE3
5DE1	3D64	8F84	02968	BDE4
9075	3D65	8F85	02969	BDE5
9187	3D66	8F86	02970	BDE6
9806	3D67	8F87	02971	BDE7
51E6	3D68	8F88	02972	BDE8
521D	3D69	8F89	02973	BDE9
6240	3D6A	8F8A	02974	BDEA
6691	3D6B	8F8B	02975	BDEB
E273				BDEC
FA47				BDED
5EB6	3D6E	8F8E	02978	BDEE
7DD2	3D6F	8F8F	02979	BDEF
7F72	3D70	8F90	02980	BDF0
66F8	3D71	8F91	02981	BDF1
E3B8				BDF2
E3BC				BDF3
8AF8	3D74	8F94	02984	BDF4
52A9	3D75	8F95	02985	BDF5
53D9	3D76	8F96	02986	BDF6
5973	3D77	8F97	02987	BDF7
5E8F	3D78	8F98	02988	BDF8
5F90	3D79	8F99	02989	BDF9
6055	3D7A	8F9A	02990	BDFA
92E4	3D7B	8F9B	02991	BDFB
9664	3D7C	8F9C	02992	BDFC
50B7	3D7D	8F9D	02993	BDFD
511F	3D7E	8F9E	02994	BDFE
52DD	3E21	8F9F	03001	BEA1
5320	3E22	8FA0	03002	BEA2
5347	3E23	8FA1	03003	BEA3
53EC	3E24	8FA2	03004	BEA4
E199				BEA5
5546	3E26	8FA4	03006	BEA6
5531	3E27	8FA5	03007	BEA7
5617	3E28	8FA6	03008	BEA8
5968	3E29	8FA7	03009	BEA9
59BE	3E2A	8FA8	03010	BEAA
5A3C	3E2B	8FA9	03011	BEAB
5BB5	3E2C	8FAA	03012	BEAC
5C06	3E2D	8FAB	03013	BEAD
5C0F	3E2E	8FAC	03014	BEAE
5C11	3E2F	8FAD	03015	BEAF
5C1A	3E30	8FAE	03016	BEB0
5E84	3E31	8FAF	03017	BEB1
5E8A	3E32	8FB0	03018	BEB2
E201				BEB3
5F70	3E34	8FB2	03020	BEB4
627F	3E35	8FB3	03021	BEB5
6284	3E36	8FB4	03022	BEB6
62DB	3E37	8FB5	03023	BEB7
638C	3E38	8FB6	03024	BEB8
6377	3E39	8FB7	03025	BEB9
6607	3E3A	8FB8	03026	BEBA
660C	3E3B	8FB9	03027	BEBB
662D	3E3C	8FBA	03028	BEBC
6676	3E3D	8FBB	03029	BEBD
E285				BEBE
E290				BEBF
6A1F	3E40	8FBE	03032	BEC0
6A35	3E41	8FBF	03033	BEC1
6CBC	3E42	8FC0	03034	BEC2
6D88	3E43	8FC1	03035	BEC3
6E09	3E44	8FC2	03036	BEC4
6E58	3E45	8FC3	03037	BEC5
713C	3E46	8FC4	03038	BEC6
7126	3E47	8FC5	03039	BEC7
7167	3E48	8FC6	03040	BEC8
75C7	3E49	8FC7	03041	BEC9
7701	3E4A	8FC8	03042	BECA
785D	3E4B	8FC9	03043	BECB
7901	3E4C	8FCA	03044	BECC
7965	3E4D	8FCB	03045	BECD
79F0	3E4E	8FCC	03046	BECE
7AE0	3E4F	8FCD	03047	BECF
7B11	3E50	8FCE	03048	BED0
7CA7	3E51	8FCF	03049	BED1
7D39	3E52	8FD0	03050	BED2
8096	3E53	8FD1	03051	BED3
83D6	3E54	8FD2	03052	BED4
8523				BED5
8549	3E56	8FD4	03054	BED6
885D	3E57	8FD5	03055	BED7
88F3	3E58	8FD6	03056	BED8
E3E5				BED9
8A3C	3E5A	8FD8	03058	BEDA
8A54	3E5B	8FD9	03059	BEDB
8A73	3E5C	8FDA	03060	BEDC
E3FD				BEDD
8CDE	3E5E	8FDC	03062	BEDE
91AC				BEDF
9266	3E60	8FDE	03064	BEE0
937E	3E61	8FDF	03065	BEE1
9418	3E62	8FE0	03066	BEE2
969C	3E63	8FE1	03067	BEE3
E4A3				BEE4
4E0A	3E65	8FE3	03069	BEE5
4E08	3E66	8FE4	03070	BEE6
4E1E	3E67	8FE5	03071	BEE7
4E57	3E68	8FE6	03072	BEE8
5197	3E69	8FE7	03073	BEE9
5270	3E6A	8FE8	03074	BEEA
57CE	3E6B	8FE9	03075	BEEB
5834	3E6C	8FEA	03076	BEEC
58CC	3E6D	8FEB	03077	BEED
5B22	3E6E	8FEC	03078	BEEE
5E38	3E6F	8FED	03079	BEEF
60C5	3E70	8FEE	03080	BEF0
64FE	3E71	8FEF	03081	BEF1
6761	3E72	8FF0	03082	BEF2
E282				BEF3
6D44	3E74	8FF2	03084	BEF4
72B6	3E75	8FF3	03085	BEF5
7573	3E76	8FF4	03086	BEF6
7A63	3E77	8FF5	03087	BEF7
84B8	3E78	8FF6	03088	BEF8
8B72	3E79	8FF7	03089	BEF9
91B8	3E7A	8FF8	03090	BEFA
9320	3E7B	8FF9	03091	BEFB
5631	3E7C	8FFA	03092	BEFC
E787				BEFD
98FE	3E7E	8FFC	03094	BEFE
62ED	3F21	9040	03101	BFA1
E791				BFA2
E792				BFA3
71ED	3F24	9043	03104	BFA4
7E54	3F25	9044	03105	BFA5
8077	3F26	9045	03106	BFA6
8272	3F27	9046	03107	BFA7
89E6	3F28	9047	03108	BFA8
98DF	3F29	9048	03109	BFA9
E3CD				BFAA
8FB1	3F2B	904A	03111	BFAB
5C3B	3F2C	904B	03112	BFAC
4F38	3F2D	904C	03113	BFAD
4FE1	3F2E	904D	03114	BFAE
4FB5	3F2F	904E	03115	BFAF
5507	3F30	904F	03116	BFB0
5A20	3F31	9050	03117	BFB1
5BDD	3F32	9051	03118	BFB2
5BE9	3F33	9052	03119	BFB3
5FC3	3F34	9053	03120	BFB4
614E	3F35	9054	03121	BFB5
632F	3F36	9055	03122	BFB6
65B0	3F37	9056	03123	BFB7
664B	3F38	9057	03124	BFB8
68EE	3F39	9058	03125	BFB9
699B	3F3A	9059	03126	BFBA
6D78	3F3B	905A	03127	BFBB
6DF1	3F3C	905B	03128	BFBC
7533	3F3D	905C	03129	BFBD
75B9	3F3E	905D	03130	BFBE
771F	3F3F	905E	03131	BFBF
795E	3F40	905F	03132	BFC0
79E6	3F41	9060	03133	BFC1
7D33	3F42	9061	03134	BFC2
81E3	3F43	9062	03135	BFC3
82AF	3F44	9063	03136	BFC4
85AA	3F45	9064	03137	BFC5
89AA	3F46	9065	03138	BFC6
8A3A	3F47	9066	03139	BFC7
8EAB	3F48	9067	03140	BFC8
8F9B	3F49	9068	03141	BFC9
9032	3F4A	9069	03142	BFCA
91DD	3F4B	906A	03143	BFCB
9707	3F4C	906B	03144	BFCC
4EBA	3F4D	906C	03145	BFCD
4EC1	3F4E	906D	03146	BFCE
5203	3F4F	906E	03147	BFCF
5875	3F50	906F	03148	BFD0
58EC	3F51	9070	03149	BFD1
5C0B	3F52	9071	03150	BFD2
751A	3F53	9072	03151	BFD3
5C3D	3F54	9073	03152	BFD4
814E	3F55	9074	03153	BFD5
E3E3				BFD6
8FC5	3F57	9076	03155	BFD7
9663	3F58	9077	03156	BFD8
E49F				BFD9
7B25	3F5A	9079	03158	BFDA
8ACF	3F5B	907A	03159	BFDB
9808	3F5C	907B	03160	BFDC
9162	3F5D	907C	03161	BFDD
56F3	3F5E	907D	03162	BFDE
53A8	3F5F	907E	03163	BFDF
E434				BFE0
5439	3F61	9081	03165	BFE1
5782	3F62	9082	03166	BFE2
5E25	3F63	9083	03167	BFE3
63A8	3F64	9084	03168	BFE4
6C34	3F65	9085	03169	BFE5
708A	3F66	9086	03170	BFE6
7761	3F67	9087	03171	BFE7
7C8B	3F68	9088	03172	BFE8
E37D				BFE9
8870	3F6A	908A	03174	BFEA
9042	3F6B	908B	03175	BFEB
9154	3F6C	908C	03176	BFEC
9310	3F6D	908D	03177	BFED
9318	3F6E	908E	03178	BFEE
968F	3F6F	908F	03179	BFEF
745E	3F70	9090	03180	BFF0
9AC4	3F71	9091	03181	BFF1
5D07	3F72	9092	03182	BFF2
5D69	3F73	9093	03183	BFF3
6570	3F74	9094	03184	BFF4
67A2	3F75	9095	03185	BFF5
8DA8	3F76	9096	03186	BFF6
96DB	3F77	9097	03187	BFF7
636E	3F78	9098	03188	BFF8
6749	3F79	9099	03189	BFF9
6919	3F7A	909A	03190	BFFA
83C5	3F7B	909B	03191	BFFB
9817	3F7C	909C	03192	BFFC
96C0	3F7D	909D	03193	BFFD
88FE	3F7E	909E	03194	BFFE
6F84	4021	909F	03201	C0A1
E253				C0A2
5BF8	4023	90A1	03203	C0A3
4E16	4024	90A2	03204	C0A4
702C	4025	90A3	03205	C0A5
755D	4026	90A4	03206	C0A6
662F	4027	90A5	03207	C0A7
51C4	4028	90A6	03208	C0A8
5236	4029	90A7	03209	C0A9
52E2	402A	90A8	03210	C0AA
59D3	402B	90A9	03211	C0AB
5F81	402C	90AA	03212	C0AC
6027	402D	90AB	03213	C0AD
6210	402E	90AC	03214	C0AE
653F	402F	90AD	03215	C0AF
6574	4030	90AE	03216	C0B0
661F	4031	90AF	03217	C0B1
6674	4032	90B0	03218	C0B2
68F2	4033	90B1	03219	C0B3
6816	4034	90B2	03220	C0B4
6B63	4035	90B3	03221	C0B5
6E05	4036	90B4	03222	C0B6
7272	4037	90B5	03223	C0B7
751F	4038	90B6	03224	C0B8
76DB	4039	90B7	03225	C0B9
7CBE	403A	90B8	03226	C0BA
8056	403B	90B9	03227	C0BB
58F0	403C	90BA	03228	C0BC
88FD	403D	90BB	03229	C0BD
897F	403E	90BC	03230	C0BE
8AA0	403F	90BD	03231	C0BF
8A93	4040	90BE	03232	C0C0
8ACB	4041	90BF	03233	C0C1
E43B				C0C2
9192	4043	90C1	03235	C0C3
9752	4044	90C2	03236	C0C4
9759	4045	90C3	03237	C0C5
6589	4046	90C4	03238	C0C6
7A0E	4047	90C5	03239	C0C7
8106	4048	90C6	03240	C0C8
96BB	4049	90C7	03241	C0C9
5E2D	404A	90C8	03242	C0CA
60DC	404B	90C9	03243	C0CB
621A	404C	90CA	03244	C0CC
65A5	404D	90CB	03245	C0CD
6614	404E	90CC	03246	C0CE
6790	404F	90CD	03247	C0CF
77F3	4050	90CE	03248	C0D0
7A4D	4051	90CF	03249	C0D1
7C4D	4052	90D0	03250	C0D2
7E3E	4053	90D1	03251	C0D3
810A	4054	90D2	03252	C0D4
8CAC	4055	90D3	03253	C0D5
8D64	4056	90D4	03254	C0D6
8DE1	4057	90D5	03255	C0D7
8E5F	4058	90D6	03256	C0D8
78A9	4059	90D7	03257	C0D9
5207	405A	90D8	03258	C0DA
62D9	405B	90D9	03259	C0DB
63A5	405C	90DA	03260	C0DC
6442	405D	90DB	03261	C0DD
6298	405E	90DC	03262	C0DE
8A2D	405F	90DD	03263	C0DF
7A83	4060	90DE	03264	C0E0
7BC0	4061	90DF	03265	C0E1
8AAC	4062	90E0	03266	C0E2
96EA	4063	90E1	03267	C0E3
7D76	4064	90E2	03268	C0E4
820C	4065	90E3	03269	C0E5
E72C				C0E6
4ED9	4067	90E5	03271	C0E7
5148	4068	90E6	03272	C0E8
5343	4069	90E7	03273	C0E9
5360	406A	90E8	03274	C0EA
5BA3	406B	90E9	03275	C0EB
5C02	406C	90EA	03276	C0EC
5C16	406D	90EB	03277	C0ED
5DDD	406E	90EC	03278	C0EE
6226	406F	90ED	03279	C0EF
6247	4070	90EE	03280	C0F0
E257				C0F1
E28E				C0F2
6834	4073	90F1	03283	C0F3
6CC9	4074	90F2	03284	C0F4
6D45	4075	90F3	03285	C0F5
6D17	4076	90F4	03286	C0F6
67D3	4077	90F5	03287	C0F7
6F5C	4078	90F6	03288	C0F8
E2E2				C0F9
E2E5				C0FA
65CB	407B	90F9	03291	C0FB
E33D				C0FC
E346				C0FD
7DDA	407E	90FC	03294	C0FE
7E4A	4121	9140	03301	C1A1
7FA8	4122	9141	03302	C1A2
817A	4123	9142	03303	C1A3
E393				C1A4
E396				C1A5
85A6	4126	9145	03306	C1A6
E3E8				C1A7
8CE4	6C4D	E6CB	07645	C1A8
8DF5	4129	9148	03309	C1A9
9078	412A	9149	03310	C1AA
9077	412B	914A	03311	C1AB
92AD	412C	914B	03312	C1AC
9291	412D	914C	03313	C1AD
9583	412E	914D	03314	C1AE
9BAE	412F	914E	03315	C1AF
524D	4130	914F	03316	C1B0
5584	4131	9150	03317	C1B1
6F38	4132	9151	03318	C1B2
7136	4133	9152	03319	C1B3
5168	4134	9153	03320	C1B4
7985	4135	9154	03321	C1B5
7E55	4136	9155	03322	C1B6
81B3	4137	9156	03323	C1B7
7CCE	4138	9157	03324	C1B8
E1A8				C1B9
5851	413A	9159	03326	C1BA
5CA8	413B	915A	03327	C1BB
63AA	413C	915B	03328	C1BC
66FE	413D	915C	03329	C1BD
66FD	413E	915D	03330	C1BE
695A	413F	915E	03331	C1BF
72D9	4140	915F	03332	C1C0
758F	4141	9160	03333	C1C1
758E	4142	9161	03334	C1C2
790E	4143	9162	03335	C1C3
7956	4144	9163	03336	C1C4
79DF	4145	9164	03337	C1C5
7C97	4146	9165	03338	C1C6
7D20	4147	9166	03339	C1C7
7D44	4148	9167	03340	C1C8
8607	4149	9168	03341	C1C9
8A34	414A	9169	03342	C1CA
963B	414B	916A	03343	C1CB
E44C				C1CC
9F20	414D	916C	03345	C1CD
50E7	414E	916D	03346	C1CE
5275	414F	916E	03347	C1CF
53CC	4150	916F	03348	C1D0
53E2	4151	9170	03349	C1D1
5009	4152	9171	03350	C1D2
55AA	4153	9172	03351	C1D3
58EE	4154	9173	03352	C1D4
594F	4155	9174	03353	C1D5
723D	4156	9175	03354	C1D6
5B8B	4157	9176	03355	C1D7
5C64	4158	9177	03356	C1D8
531D	4159	9178	03357	C1D9
60E3	415A	9179	03358	C1DA
60F3	415B	917A	03359	C1DB
635C	415C	917B	03360	C1DC
6383	415D	917C	03361	C1DD
633F	415E	917D	03362	C1DE
6414				C1DF
64CD	4160	9180	03364	C1E0
65E9	4161	9181	03365	C1E1
66F9	4162	9182	03366	C1E2
5DE3	4163	9183	03367	C1E3
69CD	4164	9184	03368	C1E4
69FD	4165	9185	03369	C1E5
6F15	4166	9186	03370	C1E6
71E5	4167	9187	03371	C1E7
4E89	4168	9188	03372	C1E8
7626				C1E9
76F8	416A	918A	03374	C1EA
7A93	416B	918B	03375	C1EB
7CDF	416C	918C	03376	C1EC
E367				C1ED
7D9C	416E	918E	03378	C1EE
E7A4				C1EF
8349	4170	9190	03380	C1F0
E7A5				C1F1
846C	4172	9192	03382	C1F2
84BC	4173	9193	03383	C1F3
85FB	4174	9194	03384	C1F4
88C5	4175	9195	03385	C1F5
8D70	4176	9196	03386	C1F6
9001	4177	9197	03387	C1F7
906D	4178	9198	03388	C1F8
9397	4179	9199	03389	C1F9
971C	417A	919A	03390	C1FA
9A12	417B	919B	03391	C1FB
E159				C1FC
5897	417D	919D	03393	C1FD
618E	417E	919E	03394	C1FE
81D3	4221	919F	03401	C2A1
8535	4222	91A0	03402	C2A2
8D08	4223	91A1	03403	C2A3
9020	4224	91A2	03404	C2A4
4FC3	4225	91A3	03405	C2A5
5074	4226	91A4	03406	C2A6
5247	4227	91A5	03407	C2A7
5373	4228	91A6	03408	C2A8
606F	4229	91A7	03409	C2A9
6349	422A	91A8	03410	C2AA
675F	422B	91A9	03411	C2AB
6E2C	422C	91AA	03412	C2AC
8DB3	422D	91AB	03413	C2AD
901F	422E	91AC	03414	C2AE
4FD7	422F	91AD	03415	C2AF
5C5E	4230	91AE	03416	C2B0
8CCA	4231	91AF	03417	C2B1
65CF	4232	91B0	03418	C2B2
7D9A	4233	91B1	03419	C2B3
5352	4234	91B2	03420	C2B4
8896	4235	91B3	03421	C2B5
5176	4236	91B4	03422	C2B6
E24D				C2B7
5B58	4238	91B6	03424	C2B8
5B6B	4239	91B7	03425	C2B9
5C0A	423A	91B8	03426	C2BA
640D	423B	91B9	03427	C2BB
6751	423C	91BA	03428	C2BC
E44D				C2BD
4ED6	423E	91BC	03430	C2BE
591A	423F	91BD	03431	C2BF
592A	4240	91BE	03432	C2C0
6C70	4241	91BF	03433	C2C1
8A51	4242	91C0	03434	C2C2
553E	4243	91C1	03435	C2C3
5815	4244	91C2	03436	C2C4
59A5	4245	91C3	03437	C2C5
60F0	4246	91C4	03438	C2C6
6253	4247	91C5	03439	C2C7
67C1	4248	91C6	03440	C2C8
8235	4249	91C7	03441	C2C9
6955	424A	91C8	03442	C2CA
9640	424B	91C9	03443	C2CB
99C4	424C	91CA	03444	C2CC
E77C				C2CD
4F53	424E	91CC	03446	C2CE
5806	424F	91CD	03447	C2CF
5BFE	4250	91CE	03448	C2D0
8010	4251	91CF	03449	C2D1
5CB1	4252	91D0	03450	C2D2
5E2F	4253	91D1	03451	C2D3
5F85	4254	91D2	03452	C2D4
6020	4255	91D3	03453	C2D5
614B	4256	91D4	03454	C2D6
6234	4257	91D5	03455	C2D7
66FF	4258	91D6	03456	C2D8
6CF0	4259	91D7	03457	C2D9
6EDE	425A	91D8	03458	C2DA
80CE	425B	91D9	03459	C2DB
E392				C2DC
82D4	425D	91DB	03461	C2DD
888B	425E	91DC	03462	C2DE
8CB8	425F	91DD	03463	C2DF
9000	4260	91DE	03464	C2E0
902E	4261	91DF	03465	C2E1
E491				C2E2
E4E5				C2E3
9BDB	4264	91E2	03468	C2E4
4EE3	4265	91E3	03469	C2E5
53F0	4266	91E4	03470	C2E6
5927	4267	91E5	03471	C2E7
7B2C	4268	91E6	03472	C2E8
918D	4269	91E7	03473	C2E9
984C	426A	91E8	03474	C2EA
9DF9	426B	91E9	03475	C2EB
6EDD	426C	91EA	03476	C2EC
7027	426D	91EB	03477	C2ED
5353	426E	91EC	03478	C2EE
E1A0				C2EF
5B85	4270	91EE	03480	C2F0
6258	4271	91EF	03481	C2F1
629E	4272	91F0	03482	C2F2
62D3	4273	91F1	03483	C2F3
6CA2	4274	91F2	03484	C2F4
E2D8				C2F5
E303				C2F6
8A17	4277	91F5	03487	C2F7
9438	4278	91F6	03488	C2F8
6FC1	4279	91F7	03489	C2F9
8AFE	427A	91F8	03490	C2FA
8338	427B	91F9	03491	C2FB
51E7	427C	91FA	03492	C2FC
E3CA				C2FD
53EA	427E	91FC	03494	C2FE
53E9	4321	9240	03501	C3A1
4F46	4322	9241	03502	C3A2
9054	4323	9242	03503	C3A3
8FB0	4324	9243	03504	C3A4
596A	4325	9244	03505	C3A5
8131	4326	9245	03506	C3A6
E1F3				C3A7
7AEA	4328	9247	03508	C3A8
E41C				C3A9
E293				C3AA
8C37	432B	924A	03511	C3AB
72F8	432C	924B	03512	C3AC
E4D7				C3AD
E2A2				C3AE
8AB0	432F	924E	03515	C3AF
4E39	4330	924F	03516	C3B0
5358	4331	9250	03517	C3B1
5606	4332	9251	03518	C3B2
5766	4333	9252	03519	C3B3
62C5	4334	9253	03520	C3B4
63A2	4335	9254	03521	C3B5
65E6	4336	9255	03522	C3B6
6B4E	4337	9256	03523	C3B7
E793				C3B8
6E5B	4339	9258	03525	C3B9
70AD	433A	9259	03526	C3BA
77ED	433B	925A	03527	C3BB
7AEF	433C	925B	03528	C3BC
E709				C3BD
7DBB	433E	925D	03530	C3BE
803D	433F	925E	03531	C3BF
80C6	4340	925F	03532	C3C0
86CB	4341	9260	03533	C3C1
8A95	4342	9261	03534	C3C2
935B	4343	9262	03535	C3C3
56E3	4344	9263	03536	C3C4
58C7	4345	9264	03537	C3C5
5F3E	4346	9265	03538	C3C6
65AD	4347	9266	03539	C3C7
6696	4348	9267	03540	C3C8
6A80	4349	9268	03541	C3C9
6BB5	434A	9269	03542	C3CA
7537	434B	926A	03543	C3CB
E7AE				C3CC
5024	434D	926C	03545	C3CD
77E5	434E	926D	03546	C3CE
5730	434F	926E	03547	C3CF
5F1B	4350	926F	03548	C3D0
6065	4351	9270	03549	C3D1
667A	4352	9271	03550	C3D2
6C60	4353	9272	03551	C3D3
75F4	4354	9273	03552	C3D4
7A1A	4355	9274	03553	C3D5
E7A3				C3D6
81F4	4357	9276	03555	C3D7
8718	4358	9277	03556	C3D8
9045	4359	9278	03557	C3D9
99B3	435A	9279	03558	C3DA
7BC9	435B	927A	03559	C3DB
755C	435C	927B	03560	C3DC
7AF9	435D	927C	03561	C3DD
7B51	435E	927D	03562	C3DE
84C4	435F	927E	03563	C3DF
9010	4360	9280	03564	C3E0
79E9	4361	9281	03565	C3E1
7A92	4362	9282	03566	C3E2
8336	4363	9283	03567	C3E3
5AE1	4364	9284	03568	C3E4
7740	4365	9285	03569	C3E5
4E2D	4366	9286	03570	C3E6
4EF2	4367	9287	03571	C3E7
5B99	4368	9288	03572	C3E8
5FE0	4369	9289	03573	C3E9
62BD	436A	928A	03574	C3EA
663C	436B	928B	03575	C3EB
67F1	436C	928C	03576	C3EC
6CE8	436D	928D	03577	C3ED
866B	436E	928E	03578	C3EE
8877	436F	928F	03579	C3EF
E3E6				C3F0
914E	4371	9291	03581	C3F1
92F3	4372	9292	03582	C3F2
99D0	4373	9293	03583	C3F3
6A17	4374	9294	03584	C3F4
E2DB				C3F5
732A	4376	9296	03586	C3F6
82E7	4377	9297	03587	C3F7
8457	4378	9298	03588	C3F8
8CAF	4379	9299	03589	C3F9
4E01	437A	929A	03590	C3FA
5146	437B	929B	03591	C3FB
E16A				C3FC
558B	437D	929D	03593	C3FD
5BF5	437E	929E	03594	C3FE
5E16	4421	929F	03601	C4A1
5E33	4422	92A0	03602	C4A2
5E81	4423	92A1	03603	C4A3
5F14	4424	92A2	03604	C4A4
5F35	4425	92A3	03605	C4A5
5F6B	4426	92A4	03606	C4A6
5FB4	4427	92A5	03607	C4A7
61F2	4428	92A6	03608	C4A8
6311	4429	92A7	03609	C4A9
66A2	442A	92A8	03610	C4AA
671D	442B	92A9	03611	C4AB
6F6E	442C	92AA	03612	C4AC
7252	442D	92AB	03613	C4AD
753A	442E	92AC	03614	C4AE
773A	442F	92AD	03615	C4AF
8074	4430	92AE	03616	C4B0
8139	4431	92AF	03617	C4B1
8178	4432	92B0	03618	C4B2
8776	4433	92B1	03619	C4B3
8ABF	4434	92B2	03620	C4B4
8ADC	4435	92B3	03621	C4B5
8D85	4436	92B4	03622	C4B6
8DF3	4437	92B5	03623	C4B7
929A	4438	92B6	03624	C4B8
9577	4439	92B7	03625	C4B9
9802	443A	92B8	03626	C4BA
9CE5	443B	92B9	03627	C4BB
52C5	443C	92BA	03628	C4BC
E243				C4BD
E79E				C4BE
6715	443F	92BD	03631	C4BF
6C88	4440	92BE	03632	C4C0
73CD	4441	92BF	03633	C4C1
8CC3	4442	92C0	03634	C4C2
93AE	4443	92C1	03635	C4C3
9673	4444	92C2	03636	C4C4
6D25	4445	92C3	03637	C4C5
E1C4				C4C6
690E	4447	92C5	03639	C4C7
E29C				C4C8
8FFD	4449	92C7	03641	C4C9
E47F				C4CA
75DB	444B	92C9	03643	C4CB
901A	444C	92CA	03644	C4CC
E1BE				C4CD
6802	444E	92CC	03646	C4CE
6451				C4CF
69FB	4450	92CE	03648	C4D0
4F43	4451	92CF	03649	C4D1
6F2C	4452	92D0	03650	C4D2
67D8	4453	92D1	03651	C4D3
8FBB	4454	92D2	03652	C4D4
8526	4455	92D3	03653	C4D5
7DB4	4456	92D4	03654	C4D6
9354	4457	92D5	03655	C4D7
693F	4458	92D6	03656	C4D8
6F70	4459	92D7	03657	C4D9
576A	445A	92D8	03658	C4DA
58FA	5464	9AE2	05268	C4DB
5B2C	445C	92DA	03660	C4DC
7D2C	445D	92DB	03661	C4DD
722A	445E	92DC	03662	C4DE
540A	445F	92DD	03663	C4DF
91E3	4460	92DE	03664	C4E0
9DB4	4461	92DF	03665	C4E1
4EAD	4462	92E0	03666	C4E2
4F4E	4463	92E1	03667	C4E3
505C	4464	92E2	03668	C4E4
5075	4465	92E3	03669	C4E5
5243	4466	92E4	03670	C4E6
8C9E	4467	92E5	03671	C4E7
5448	4468	92E6	03672	C4E8
5824	4469	92E7	03673	C4E9
5B9A	446A	92E8	03674	C4EA
5E1D	446B	92E9	03675	C4EB
5E95	446C	92EA	03676	C4EC
5EAD	446D	92EB	03677	C4ED
5EF7	446E	92EC	03678	C4EE
5F1F	446F	92ED	03679	C4EF
608C	4470	92EE	03680	C4F0
62B5	4471	92EF	03681	C4F1
E244				C4F2
63D0	4473	92F1	03683	C4F3
68AF	4474	92F2	03684	C4F4
6C40	4475	92F3	03685	C4F5
7887	4476	92F4	03686	C4F6
798E	4477	92F5	03687	C4F7
7A0B	4478	92F6	03688	C4F8
7DE0	4479	92F7	03689	C4F9
8247	447A	92F8	03690	C4FA
8A02	447B	92F9	03691	C4FB
8AE6	447C	92FA	03692	C4FC
8E44	447D	92FB	03693	C4FD
9013	447E	92FC	03694	C4FE
90B8	4521	9340	03701	C5A1
E465				C5A2
91D8	4523	9342	03703	C5A3
9F0E	4524	9343	03704	C5A4
6CE5	4525	9344	03705	C5A5
6458	4526	9345	03706	C5A6
E259				C5A7
6575	4528	9347	03708	C5A8
6EF4	4529	9348	03709	C5A9
7684	452A	9349	03710	C5AA
7B1B	452B	934A	03711	C5AB
9069	452C	934B	03712	C5AC
93D1	452D	934C	03713	C5AD
E2C7				C5AE
54F2	452F	934E	03715	C5AF
5FB9	4530	934F	03716	C5B0
64A4	4531	9350	03717	C5B1
8F4D	4532	9351	03718	C5B2
8FED	4533	9352	03719	C5B3
9244	4534	9353	03720	C5B4
5178	4535	9354	03721	C5B5
5861				C5B6
5929	4537	9356	03723	C5B7
5C55	4538	9357	03724	C5B8
5E97	4539	9358	03725	C5B9
6DFB	453A	9359	03726	C5BA
7E8F	453B	935A	03727	C5BB
751C	453C	935B	03728	C5BC
8CBC	453D	935C	03729	C5BD
8EE2	453E	935D	03730	C5BE
985A				C5BF
70B9	4540	935F	03732	C5C0
4F1D	4541	9360	03733	C5C1
6BBF	4542	9361	03734	C5C2
6FB1	4543	9362	03735	C5C3
7530	4544	9363	03736	C5C4
96FB	4545	9364	03737	C5C5
E15E				C5C6
5410	4547	9366	03739	C5C7
E1BB				C5C8
5857	4549	9368	03741	C5C9
59AC	454A	9369	03742	C5CA
E1ED				C5CB
5F92	454C	936B	03744	C5CC
6597	454D	936C	03745	C5CD
675C	454E	936D	03746	C5CE
6E21	454F	936E	03747	C5CF
767B	4550	936F	03748	C5D0
E3A7				C5D1
E406				C5D2
9014	4553	9372	03751	C5D3
90FD	4554	9373	03752	C5D4
934D	4555	9374	03753	C5D5
7825	4556	9375	03754	C5D6
792A	626A	E1E8	06674	C5D7
52AA	4558	9377	03756	C5D8
5EA6	4559	9378	03757	C5D9
571F	455A	9379	03758	C5DA
5974	455B	937A	03759	C5DB
6012	455C	937B	03760	C5DC
5012	455D	937C	03761	C5DD
515A	455E	937D	03762	C5DE
51AC	455F	937E	03763	C5DF
51CD	4560	9380	03764	C5E0
5200	4561	9381	03765	C5E1
5510	4562	9382	03766	C5E2
5854	4563	9383	03767	C5E3
E1BF				C5E4
5957	4565	9385	03769	C5E5
5B95	4566	9386	03770	C5E6
5CF6	4567	9387	03771	C5E7
5D8B	4568	9388	03772	C5E8
60BC	4569	9389	03773	C5E9
6295	456A	938A	03774	C5EA
642D	456B	938B	03775	C5EB
6771	456C	938C	03776	C5EC
6843	456D	938D	03777	C5ED
6AAE	5B6D	9E8D	05977	C5EE
68DF	456F	938F	03779	C5EF
76D7	4570	9390	03780	C5F0
6DD8	4571	9391	03781	C5F1
6E6F	4572	9392	03782	C5F2
6FE4	5E39	9FB7	06225	C5F3
706F	4574	9394	03784	C5F4
71C8	4575	9395	03785	C5F5
5F53	4576	9396	03786	C5F6
75D8	4577	9397	03787	C5F7
79B1				C5F8
7B49	4579	9399	03789	C5F9
7B54	457A	939A	03790	C5FA
7B52	457B	939B	03791	C5FB
7CD6	457C	939C	03792	C5FC
7D71	457D	939D	03793	C5FD
5230	457E	939E	03794	C5FE
8463	4621	939F	03801	C6A1
8569	4622	93A0	03802	C6A2
85E4	4623	93A1	03803	C6A3
8A0E	4624	93A2	03804	C6A4
8B04	4625	93A3	03805	C6A5
8C46	4626	93A4	03806	C6A6
8E0F	4627	93A5	03807	C6A7
9003	4628	93A6	03808	C6A8
900F	4629	93A7	03809	C6A9
9419	462A	93A8	03810	C6AA
9676	462B	93A9	03811	C6AB
982D	462C	93AA	03812	C6AC
9A30	462D	93AB	03813	C6AD
95D8	462E	93AC	03814	C6AE
50CD	462F	93AD	03815	C6AF
52D5	4630	93AE	03816	C6B0
540C	4631	93AF	03817	C6B1
5802	4632	93B0	03818	C6B2
5C0E	4633	93B1	03819	C6B3
61A7	4634	93B2	03820	C6B4
649E	4635	93B3	03821	C6B5
6D1E	4636	93B4	03822	C6B6
77B3	4637	93B5	03823	C6B7
7AE5	4638	93B6	03824	C6B8
80F4	4639	93B7	03825	C6B9
8404	463A	93B8	03826	C6BA
9053	463B	93B9	03827	C6BB
9285	463C	93BA	03828	C6BC
5CE0	463D	93BB	03829	C6BD
E4DA				C6BE
533F	463F	93BD	03831	C6BF
5F97	4640	93BE	03832	C6C0
5FB3	4641	93BF	03833	C6C1
7006				C6C2
7279	4643	93C1	03835	C6C3
7763	4644	93C2	03836	C6C4
79BF	4645	93C3	03837	C6C5
7BE4	4646	93C4	03838	C6C6
6BD2	4647	93C5	03839	C6C7
72EC	4648	93C6	03840	C6C8
8AAD	4649	93C7	03841	C6C9
6803	464A	93C8	03842	C6CA
E2A5				C6CB
51F8	464C	93CA	03844	C6CC
7A81	464D	93CB	03845	C6CD
6934	464E	93CC	03846	C6CE
5C4A	464F	93CD	03847	C6CF
9CF6	4650	93CE	03848	C6D0
82EB	4651	93CF	03849	C6D1
5BC5	4652	93D0	03850	C6D2
9149	4653	93D1	03851	C6D3
E2DA				C6D4
5678	4655	93D3	03853	C6D5
5C6F	4656	93D4	03854	C6D6
60C7	4657	93D5	03855	C6D7
6566	4658	93D6	03856	C6D8
6C8C	4659	93D7	03857	C6D9
E38C				C6DA
E444				C6DB
9813	465C	93DA	03860	C6DC
5451	465D	93DB	03861	C6DD
66C7	465E	93DC	03862	C6DE
920D	465F	93DD	03863	C6DF
5948	4660	93DE	03864	C6E0
E45E				C6E1
5185	4662	93E0	03866	C6E2
4E4D	4663	93E1	03867	C6E3
51EA	4664	93E2	03868	C6E4
8599	4665	93E3	03869	C6E5
E3F4				C6E6
E2DE				C6E7
637A	4668	93E6	03872	C6E8
934B	4669	93E7	03873	C6E9
E296				C6EA
99B4	466B	93E9	03875	C6EB
7E04	466C	93EA	03876	C6EC
7577	466D	93EB	03877	C6ED
5357	466E	93EC	03878	C6EE
6960	466F	93ED	03879	C6EF
8EDF	4670	93EE	03880	C6F0
96E3	4671	93EF	03881	C6F1
6C5D	4672	93F0	03882	C6F2
4E8C	4673	93F1	03883	C6F3
5C3C	4674	93F2	03884	C6F4
5F10	4675	93F3	03885	C6F5
9087	6D6E	E78E	07778	C6F6
5302	4677	93F5	03887	C6F7
8CD1	4678	93F6	03888	C6F8
8089	4679	93F7	03889	C6F9
8679	467A	93F8	03890	C6FA
5EFF	467B	93F9	03891	C6FB
65E5	467C	93FA	03892	C6FC
4E73	467D	93FB	03893	C6FD
5165	467E	93FC	03894	C6FE
5982	4721	9440	03901	C7A1
5C3F	4722	9441	03902	C7A2
97EE	4723	9442	03903	C7A3
4EFB	4724	9443	03904	C7A4
598A	4725	9444	03905	C7A5
5FCD	4726	9445	03906	C7A6
8A8D	4727	9446	03907	C7A7
6FE1	4728	9447	03908	C7A8
E333				C7A9
E32E				C7AA
5BE7	472B	944A	03911	C7AB
8471	472C	944B	03912	C7AC
732B	472D	944C	03913	C7AD
E798				C7AE
5E74	472F	944E	03915	C7AF
5FF5	4730	944F	03916	C7B0
637B	4731	9450	03917	C7B1
649A	4732	9451	03918	C7B2
71C3	4733	9452	03919	C7B3
7C98	4734	9453	03920	C7B4
4E43	4735	9454	03921	C7B5
E20A				C7B6
4E4B	4737	9456	03923	C7B7
57DC	4738	9457	03924	C7B8
56CA				C7B9
60A9	473A	9459	03926	C7BA
6FC3	473B	945A	03927	C7BB
7D0D	473C	945B	03928	C7BC
80FD	473D	945C	03929	C7BD
8133	473E	945D	03930	C7BE
81BF	473F	945E	03931	C7BF
8FB2	4740	945F	03932	C7C0
8997	4741	9460	03933	C7C1
86A4	4742	9461	03934	C7C2
5DF4	4743	9462	03935	C7C3
628A	4744	9463	03936	C7C4
64AD	4745	9464	03937	C7C5
8987	4746	9465	03938	C7C6
6777	4747	9466	03939	C7C7
6CE2	4748	9467	03940	C7C8
6D3E	4749	9468	03941	C7C9
7436	474A	9469	03942	C7CA
7834	474B	946A	03943	C7CB
5A46	474C	946B	03944	C7CC
7F75	474D	946C	03945	C7CD
82AD	474E	946D	03946	C7CE
99AC	474F	946E	03947	C7CF
4FF3	4750	946F	03948	C7D0
5EC3	4751	9470	03949	C7D1
62DD	4752	9471	03950	C7D2
6392	4753	9472	03951	C7D3
6557	4754	9473	03952	C7D4
676F	4755	9474	03953	C7D5
76C3	4756	9475	03954	C7D6
E2F3				C7D7
80CC	4758	9477	03956	C7D8
80BA	4759	9478	03957	C7D9
8F29	475A	9479	03958	C7DA
914D	475B	947A	03959	C7DB
500D	475C	947B	03960	C7DC
57F9	475D	947C	03961	C7DD
5A92	475E	947D	03962	C7DE
6885	475F	947E	03963	C7DF
6973	4760	9480	03964	C7E0
7164	4761	9481	03965	C7E1
72FD	4762	9482	03966	C7E2
8CB7	4763	9483	03967	C7E3
58F2	4764	9484	03968	C7E4
8CE0	4765	9485	03969	C7E5
966A	4766	9486	03970	C7E6
E433				C7E7
877F	4768	9488	03972	C7E8
E334				C7E9
77E7	476A	948A	03974	C7EA
8429	476B	948B	03975	C7EB
4F2F	476C	948C	03976	C7EC
E63D				C7ED
535A	476E	948E	03978	C7EE
62CD	476F	948F	03979	C7EF
67CF	4770	9490	03980	C7F0
6CCA	4771	9491	03981	C7F1
767D	4772	9492	03982	C7F2
7B94	4773	9493	03983	C7F3
7C95	4774	9494	03984	C7F4
8236	4775	9495	03985	C7F5
8584	4776	9496	03986	C7F6
8FEB	4777	9497	03987	C7F7
66DD	4778	9498	03988	C7F8
6F20	4779	9499	03989	C7F9
7206	477A	949A	03990	C7FA
7E1B	477B	949B	03991	C7FB
83AB	477C	949C	03992	C7FC
E4BF				C7FD
9EA6	477E	949E	03994	C7FE
51FD	4821	949F	04001	C8A1
7BB1	4822	94A0	04002	C8A2
7872	4823	94A1	04003	C8A3
E345				C8A4
8087	4825	94A3	04005	C8A5
7B48	4826	94A4	04006	C8A6
6AE8	4827	94A5	04007	C8A7
5E61	4828	94A6	04008	C8A8
808C	4829	94A7	04009	C8A9
7551	482A	94A8	04010	C8AA
7560	482B	94A9	04011	C8AB
516B	482C	94AA	04012	C8AC
9262	482D	94AB	04013	C8AD
E795				C8AE
767A	482F	94AD	04015	C8AF
91B1				C8B0
9AEA	4831	94AF	04017	C8B1
4F10	4832	94B0	04018	C8B2
7F70	4833	94B1	04019	C8B3
629C	4834	94B2	04020	C8B4
7B4F	4835	94B3	04021	C8B5
95A5	4836	94B4	04022	C8B6
9CE9	4837	94B5	04023	C8B7
567A	4838	94B6	04024	C8B8
5859	4839	94B7	04025	C8B9
86E4	483A	94B8	04026	C8BA
96BC	483B	94B9	04027	C8BB
4F34	483C	94BA	04028	C8BC
5224	483D	94BB	04029	C8BD
534A	483E	94BC	04030	C8BE
53CD	483F	94BD	04031	C8BF
E18E				C8C0
5E06	4841	94BF	04033	C8C1
642C	4842	94C0	04034	C8C2
6591	4843	94C1	04035	C8C3
677F	4844	94C2	04036	C8C4
6C3E	4845	94C3	04037	C8C5
6C4E	4846	94C4	04038	C8C6
7248	4847	94C5	04039	C8C7
72AF	4848	94C6	04040	C8C8
E300				C8C9
7554	484A	94C8	04042	C8CA
7E41	484B	94C9	04043	C8CB
822C	484C	94CA	04044	C8CC
85E9	484D	94CB	04045	C8CD
8CA9	484E	94CC	04046	C8CE
7BC4	484F	94CD	04047	C8CF
91C6	4850	94CE	04048	C8D0
7169	4851	94CF	04049	C8D1
E4AC				C8D2
98EF	4853	94D1	04051	C8D3
E245				C8D4
6669	4855	94D3	04053	C8D5
756A	4856	94D4	04054	C8D6
76E4	4857	94D5	04055	C8D7
78D0	4858	94D6	04056	C8D8
8543	4859	94D7	04057	C8D9
86EE	485A	94D8	04058	C8DA
E784				C8DB
5351	485C	94DA	04060	C8DC
5426	485D	94DB	04061	C8DD
5983	485E	94DC	04062	C8DE
5E87	485F	94DD	04063	C8DF
5F7C	4860	94DE	04064	C8E0
60B2	4861	94DF	04065	C8E1
E239				C8E2
6279	4863	94E1	04067	C8E3
62AB	4864	94E2	04068	C8E4
E78E				C8E5
6BD4	4866	94E4	04070	C8E6
6CCC	4867	94E5	04071	C8E7
75B2	4868	94E6	04072	C8E8
76AE	4869	94E7	04073	C8E9
7891	486A	94E8	04074	C8EA
79D8	486B	94E9	04075	C8EB
E368				C8EC
7F77	486D	94EB	04077	C8ED
80A5	486E	94EC	04078	C8EE
88AB	486F	94ED	04079	C8EF
E3EC				C8F0
8CBB	4871	94EF	04081	C8F1
907F	4872	94F0	04082	C8F2
975E	4873	94F1	04083	C8F3
98DB	4874	94F2	04084	C8F4
E29F				C8F5
7C38	4876	94F4	04086	C8F6
5099	4877	94F5	04087	C8F7
5C3E	4878	94F6	04088	C8F8
5FAE	4879	94F7	04089	C8F9
6787	487A	94F8	04090	C8FA
6BD8	487B	94F9	04091	C8FB
7435	487C	94FA	04092	C8FC
7709	487D	94FB	04093	C8FD
7F8E	487E	94FC	04094	C8FE
9F3B	4921	9540	04101	C9A1
E288				C9A2
E336				C9A3
5339	4924	9543	04104	C9A4
758B	4925	9544	04105	C9A5
9AED	4926	9545	04106	C9A6
5F66	4927	9546	04107	C9A7
819D	4928	9547	04108	C9A8
83F1	4929	9548	04109	C9A9
8098	492A	9549	04110	C9AA
5F3C	492B	954A	04111	C9AB
5FC5	492C	954B	04112	C9AC
7562	492D	954C	04113	C9AD
7B46	492E	954D	04114	C9AE
E445				C9AF
6A9C	5B58	9E77	05956	C9B0
59EB	4931	9550	04117	C9B1
E1D5				C9B2
7D10	4933	9552	04119	C9B3
767E	4934	9553	04120	C9B4
E3F7				C9B5
4FF5	4936	9555	04122	C9B6
5F6A	4937	9556	04123	C9B7
6A19	4938	9557	04124	C9B8
6C37	4939	9558	04125	C9B9
6F02	493A	9559	04126	C9BA
74E2	493B	955A	04127	C9BB
7968	493C	955B	04128	C9BC
8868	493D	955C	04129	C9BD
8A55	493E	955D	04130	C9BE
8C79	493F	955E	04131	C9BF
E202				C9C0
63CF	4941	9560	04133	C9C1
75C5	4942	9561	04134	C9C2
79D2	4943	9562	04135	C9C3
82D7	4944	9563	04136	C9C4
9328	4945	9564	04137	C9C5
92F2	4946	9565	04138	C9C6
849C	4947	9566	04139	C9C7
86ED	4948	9567	04140	C9C8
9C2D	4949	9568	04141	C9C9
54C1	494A	9569	04142	C9CA
5F6C	494B	956A	04143	C9CB
658C	494C	956B	04144	C9CC
6D5C	494D	956C	04145	C9CD
E2DC				C9CE
E402				C9CF
8CD3	4950	956F	04148	C9D0
FA6A				C9D1
654F	4952	9571	04150	C9D2
E79A				C9D3
4E0D	4954	9573	04152	C9D4
4ED8	4955	9574	04153	C9D5
57E0	4956	9575	04154	C9D6
592B	4957	9576	04155	C9D7
5A66	4958	9577	04156	C9D8
5BCC	4959	9578	04157	C9D9
51A8	495A	9579	04158	C9DA
5E03	495B	957A	04159	C9DB
5E9C	495C	957B	04160	C9DC
6016	495D	957C	04161	C9DD
6276	495E	957D	04162	C9DE
6577	495F	957E	04163	C9DF
E2ED				C9E0
666E	4961	9581	04165	C9E1
6D6E	4962	9582	04166	C9E2
7236	4963	9583	04167	C9E3
7B26	4964	9584	04168	C9E4
8150	4965	9585	04169	C9E5
819A	4966	9586	04170	C9E6
8299	4967	9587	04171	C9E7
8B5C	4968	9588	04172	C9E8
8CA0	4969	9589	04173	C9E9
8CE6	496A	958A	04174	C9EA
8D74	496B	958B	04175	C9EB
961C	496C	958C	04176	C9EC
9644	496D	958D	04177	C9ED
4FAE	496E	958E	04178	C9EE
64AB	496F	958F	04179	C9EF
6B66	4970	9590	04180	C9F0
821E	4971	9591	04181	C9F1
8461	4972	9592	04182	C9F2
856A	4973	9593	04183	C9F3
90E8	4974	9594	04184	C9F4
5C01	4975	9595	04185	C9F5
6953	4976	9596	04186	C9F6
98A8	4977	9597	04187	C9F7
847A	4978	9598	04188	C9F8
8557	4979	9599	04189	C9F9
4F0F	497A	959A	04190	C9FA
526F	497B	959B	04191	C9FB
5FA9	497C	959C	04192	C9FC
5E45	497D	959D	04193	C9FD
670D	497E	959E	04194	C9FE
798F	4A21	959F	04201	CAA1
8179	4A22	95A0	04202	CAA2
8907	4A23	95A1	04203	CAA3
8986	4A24	95A2	04204	CAA4
6DF5	4A25	95A3	04205	CAA5
5F17	4A26	95A4	04206	CAA6
6255	4A27	95A5	04207	CAA7
6CB8	4A28	95A6	04208	CAA8
4ECF	4A29	95A7	04209	CAA9
7269	4A2A	95A8	04210	CAAA
9B92	4A2B	95A9	04211	CAAB
E16E				CAAC
543B	4A2D	95AB	04213	CAAD
E1AD				CAAE
E788				CAAF
E22B				CAB0
626E	4A31	95AF	04217	CAB1
711A	4A32	95B0	04218	CAB2
596E	4A33	95B1	04219	CAB3
E353				CAB4
7CDE	4A35	95B3	04221	CAB5
E35B				CAB6
E49A				CAB7
6587	4A38	95B6	04224	CAB8
805E	4A39	95B7	04225	CAB9
4E19	4A3A	95B8	04226	CABA
4F75	4A3B	95B9	04227	CABB
5175	4A3C	95BA	04228	CABC
5840	4A3D	95BB	04229	CABD
5E63	4A3E	95BC	04230	CABE
5E73	4A3F	95BD	04231	CABF
5F0A	4A40	95BE	04232	CAC0
67C4	4A41	95BF	04233	CAC1
4E26	4A42	95C0	04234	CAC2
E3AF				CAC3
9589	4A44	95C2	04236	CAC4
965B	4A45	95C3	04237	CAC5
7C73	4A46	95C4	04238	CAC6
9801	4A47	95C5	04239	CAC7
50FB	4A48	95C6	04240	CAC8
58C1	4A49	95C7	04241	CAC9
7656	4A4A	95C8	04242	CACA
78A7	4A4B	95C9	04243	CACB
5225	4A4C	95CA	04244	CACC
E31E				CACD
8511	4A4E	95CC	04246	CACE
7B86	4A4F	95CD	04247	CACF
504F	4A50	95CE	04248	CAD0
5909	4A51	95CF	04249	CAD1
7247	4A52	95D0	04250	CAD2
E347				CAD3
7DE8	4A54	95D2	04252	CAD4
8FBA	4A55	95D3	04253	CAD5
E7B1				CAD6
904D	4A57	95D5	04255	CAD7
4FBF	4A58	95D6	04256	CAD8
52C9	4A59	95D7	04257	CAD9
E1D2				CADA
5F01	4A5B	95D9	04259	CADB
E4A4				CADC
4FDD	4A5D	95DB	04261	CADD
8217	4A5E	95DC	04262	CADE
92EA	4A5F	95DD	04263	CADF
5703	4A60	95DE	04264	CAE0
6355	4A61	95DF	04265	CAE1
6B69	4A62	95E0	04266	CAE2
752B	4A63	95E1	04267	CAE3
88DC	4A64	95E2	04268	CAE4
8F14	4A65	95E3	04269	CAE5
7A42	4A66	95E4	04270	CAE6
52DF	4A67	95E5	04271	CAE7
5893	4A68	95E6	04272	CAE8
6155	4A69	95E7	04273	CAE9
620A	4A6A	95E8	04274	CAEA
66AE	4A6B	95E9	04275	CAEB
6BCD	4A6C	95EA	04276	CAEC
7C3F	4A6D	95EB	04277	CAED
83E9	4A6E	95EC	04278	CAEE
5023	4A6F	95ED	04279	CAEF
4FF8	4A70	95EE	04280	CAF0
5305	4A71	95EF	04281	CAF1
5446	4A72	95F0	04282	CAF2
5831	4A73	95F1	04283	CAF3
5949	4A74	95F2	04284	CAF4
5B9D	4A75	95F3	04285	CAF5
5CF0	4A76	95F4	04286	CAF6
5CEF	4A77	95F5	04287	CAF7
5D29	4A78	95F6	04288	CAF8
E1FC				CAF9
62B1	4A7A	95F8	04290	CAFA
6367	4A7B	95F9	04291	CAFB
653E	4A7C	95FA	04292	CAFC
65B9	4A7D	95FB	04293	CAFD
670B	4A7E	95FC	04294	CAFE
6CD5	4B21	9640	04301	CBA1
E2B7				CBA2
70F9	4B23	9642	04303	CBA3
7832	4B24	9643	04304	CBA4
7E2B	4B25	9644	04305	CBA5
80DE	4B26	9645	04306	CBA6
82B3	4B27	9646	04307	CBA7
840C	4B28	9647	04308	CBA8
E3AE				CBA9
8702	4B2A	9649	04310	CBAA
8912	4B2B	964A	04311	CBAB
8A2A	4B2C	964B	04312	CBAC
8C4A	4B2D	964C	04313	CBAD
90A6	4B2E	964D	04314	CBAE
92D2	4B2F	964E	04315	CBAF
98FD	4B30	964F	04316	CBB0
9CF3	4B31	9650	04317	CBB1
9D6C	4B32	9651	04318	CBB2
4E4F	4B33	9652	04319	CBB3
4EA1	4B34	9653	04320	CBB4
508D	4B35	9654	04321	CBB5
5256	4B36	9655	04322	CBB6
574A	4B37	9656	04323	CBB7
59A8	4B38	9657	04324	CBB8
5E3D	4B39	9658	04325	CBB9
5FD8	4B3A	9659	04326	CBBA
5FD9	4B3B	965A	04327	CBBB
623F	4B3C	965B	04328	CBBC
66B4	4B3D	965C	04329	CBBD
671B	4B3E	965D	04330	CBBE
67D0	4B3F	965E	04331	CBBF
68D2	4B40	965F	04332	CBC0
5192	4B41	9660	04333	CBC1
7D21	4B42	9661	04334	CBC2
80AA	4B43	9662	04335	CBC3
81A8	4B44	9663	04336	CBC4
8B00	4B45	9664	04337	CBC5
8C8C	4B46	9665	04338	CBC6
8CBF	4B47	9666	04339	CBC7
927E	4B48	9667	04340	CBC8
9632	4B49	9668	04341	CBC9
5420	4B4A	9669	04342	CBCA
9830				CBCB
5317	4B4C	966B	04344	CBCC
50D5	4B4D	966C	04345	CBCD
535C	4B4E	966D	04346	CBCE
58A8	4B4F	966E	04347	CBCF
64B2	4B50	966F	04348	CBD0
6734	4B51	9670	04349	CBD1
7267	4B52	9671	04350	CBD2
7766	4B53	9672	04351	CBD3
7A46	4B54	9673	04352	CBD4
91E6	4B55	9674	04353	CBD5
52C3	4B56	9675	04354	CBD6
6CA1	4B57	9676	04355	CBD7
6B86	4B58	9677	04356	CBD8
5800	4B59	9678	04357	CBD9
5E4C	4B5A	9679	04358	CBDA
5954	4B5B	967A	04359	CBDB
672C	4B5C	967B	04360	CBDC
7FFB	4B5D	967C	04361	CBDD
51E1	4B5E	967D	04362	CBDE
E31B				CBDF
6469	4B60	9680	04364	CBE0
78E8	4B61	9681	04365	CBE1
9B54	4B62	9682	04366	CBE2
9EBB	4B63	9683	04367	CBE3
57CB	4B64	9684	04368	CBE4
59B9	4B65	9685	04369	CBE5
6627	4B66	9686	04370	CBE6
679A	4B67	9687	04371	CBE7
6BCE	4B68	9688	04372	CBE8
54E9	4B69	9689	04373	CBE9
69C7	7422	EAA0	08402	CBEA
5E55	4B6B	968B	04375	CBEB
819C	4B6C	968C	04376	CBEC
6795	4B6D	968D	04377	CBED
9BAA	4B6E	968E	04378	CBEE
67FE	4B6F	968F	04379	CBEF
E4D8				CBF0
685D	4B71	9691	04381	CBF1
4EA6	4B72	9692	04382	CBF2
4FE3	4B73	9693	04383	CBF3
53C8	4B74	9694	04384	CBF4
62B9	4B75	9695	04385	CBF5
672B	4B76	9696	04386	CBF6
6CAB	4B77	9697	04387	CBF7
E41B				CBF8
5118	5056	98D4	04854	CBF9
E3BB				CBFA
9EBF	4B7B	969B	04391	CBFB
4E07	4B7C	969C	04392	CBFC
6162	4B7D	969D	04393	CBFD
6E80	4B7E	969E	04394	CBFE
6F2B	4C21	969F	04401	CCA1
8513	4C22	96A0	04402	CCA2
5473	4C23	96A1	04403	CCA3
672A	4C24	96A2	04404	CCA4
9B45	4C25	96A3	04405	CCA5
5DF3	4C26	96A4	04406	CCA6
7B95	4C27	96A5	04407	CCA7
E65F				CCA8
5BC6	4C29	96A7	04409	CCA9
871C	4C2A	96A8	04410	CCAA
6E4A	4C2B	96A9	04411	CCAB
84D1	4C2C	96AA	04412	CCAC
7A14	4C2D	96AB	04413	CCAD
8108	4C2E	96AC	04414	CCAE
5999	4C2F	96AD	04415	CCAF
7C8D	4C30	96AE	04416	CCB0
6C11	4C31	96AF	04417	CCB1
7720	4C32	96B0	04418	CCB2
52D9	4C33	96B1	04419	CCB3
5922	4C34	96B2	04420	CCB4
7121	4C35	96B3	04421	CCB5
725F	4C36	96B4	04422	CCB6
77DB	4C37	96B5	04423	CCB7
9727	4C38	96B6	04424	CCB8
9D61	4C39	96B7	04425	CCB9
690B	4C3A	96B8	04426	CCBA
5A7F	4C3B	96B9	04427	CCBB
5A18	4C3C	96BA	04428	CCBC
51A5	4C3D	96BB	04429	CCBD
540D	4C3E	96BC	04430	CCBE
547D	4C3F	96BD	04431	CCBF
660E	4C40	96BE	04432	CCC0
76DF	4C41	96BF	04433	CCC1
8FF7	4C42	96C0	04434	CCC2
9298	4C43	96C1	04435	CCC3
9CF4	4C44	96C2	04436	CCC4
59EA	4C45	96C3	04437	CCC5
725D	4C46	96C4	04438	CCC6
6EC5	4C47	96C5	04439	CCC7
514D	4C48	96C6	04440	CCC8
68C9	4C49	96C7	04441	CCC9
7DBF	4C4A	96C8	04442	CCCA
7DEC	4C4B	96C9	04443	CCCB
9762	4C4C	96CA	04444	CCCC
9EB5				CCCD
6478	4C4E	96CC	04446	CCCE
6A21	4C4F	96CD	04447	CCCF
8302	4C50	96CE	04448	CCD0
5984	4C51	96CF	04449	CCD1
5B5F	4C52	96D0	04450	CCD2
6BDB	4C53	96D1	04451	CCD3
731B	4C54	96D2	04452	CCD4
76F2	4C55	96D3	04453	CCD5
7DB2	4C56	96D4	04454	CCD6
8017	4C57	96D5	04455	CCD7
E3AB				CCD8
E15C				CCD9
6728	4C5A	96D8	04458	CCDA
9ED9	4C5B	96D9	04459	CCDB
76EE	4C5C	96DA	04460	CCDC
6762	4C5D	96DB	04461	CCDD
52FF	4C5E	96DC	04462	CCDE
E4BA				CCDF
5C24	4C60	96DE	04464	CCE0
623B	4C61	96DF	04465	CCE1
E34F				CCE2
8CB0	4C63	96E1	04467	CCE3
554F	4C64	96E2	04468	CCE4
60B6	4C65	96E3	04469	CCE5
7D0B	4C66	96E4	04470	CCE6
9580	4C67	96E5	04471	CCE7
5301	4C68	96E6	04472	CCE8
4E5F	4C69	96E7	04473	CCE9
51B6	4C6A	96E8	04474	CCEA
591C	4C6B	96E9	04475	CCEB
E2F1				CCEC
8036	4C6D	96EB	04477	CCED
91CE	4C6E	96EC	04478	CCEE
5F25	4C6F	96ED	04479	CCEF
77E2	4C70	96EE	04480	CCF0
5384	4C71	96EF	04481	CCF1
5F79	4C72	96F0	04482	CCF2
7D04	4C73	96F1	04483	CCF3
85AC	4C74	96F2	04484	CCF4
8A33	4C75	96F3	04485	CCF5
8E8D	4C76	96F4	04486	CCF6
9756	4C77	96F5	04487	CCF7
67F3	4C78	96F6	04488	CCF8
85EA	692E	E54D	07314	CCF9
E485				CCFA
6109	4C7B	96F9	04491	CCFB
E228				CCFC
6CB9	4C7D	96FB	04493	CCFD
E316				CCFE
8AED	4D21	9740	04501	CDA1
8F38	4D22	9741	04502	CDA2
552F	4D23	9742	04503	CDA3
4F51	4D24	9743	04504	CDA4
512A	4D25	9744	04505	CDA5
52C7	4D26	9745	04506	CDA6
53CB	4D27	9746	04507	CDA7
5BA5	4D28	9747	04508	CDA8
5E7D	4D29	9748	04509	CDA9
60A0	4D2A	9749	04510	CDAA
6182	4D2B	974A	04511	CDAB
63D6	4D2C	974B	04512	CDAC
6709	4D2D	974C	04513	CDAD
67DA	4D2E	974D	04514	CDAE
6E67	4D2F	974E	04515	CDAF
6D8C	4D30	974F	04516	CDB0
7336	4D31	9750	04517	CDB1
E2F7				CDB2
7531	4D33	9752	04519	CDB3
7950	4D34	9753	04520	CDB4
88D5	4D35	9754	04521	CDB5
8A98	4D36	9755	04522	CDB6
904A	4D37	9756	04523	CDB7
9091	4D38	9757	04524	CDB8
90F5	4D39	9758	04525	CDB9
96C4	4D3A	9759	04526	CDBA
878D	4D3B	975A	04527	CDBB
5915	4D3C	975B	04528	CDBC
4E88	4D3D	975C	04529	CDBD
4F59	4D3E	975D	04530	CDBE
4E0E	4D3F	975E	04531	CDBF
8A89	4D40	975F	04532	CDC0
8F3F	4D41	9760	04533	CDC1
9810	4D42	9761	04534	CDC2
50AD	4D43	9762	04535	CDC3
5E7C	4D44	9763	04536	CDC4
5996	4D45	9764	04537	CDC5
5BB9	4D46	9765	04538	CDC6
5EB8	4D47	9766	04539	CDC7
63DA	4D48	9767	04540	CDC8
63FA	4D49	9768	04541	CDC9
64C1	4D4A	9769	04542	CDCA
66DC	4D4B	976A	04543	CDCB
694A	4D4C	976B	04544	CDCC
69D8	4D4D	976C	04545	CDCD
6D0B	4D4E	976D	04546	CDCE
6EB6	4D4F	976E	04547	CDCF
7194	4D50	976F	04548	CDD0
7528	4D51	9770	04549	CDD1
7AAF	4D52	9771	04550	CDD2
7F8A	4D53	9772	04551	CDD3
E382				CDD4
8449	4D55	9774	04553	CDD5
84C9	4D56	9775	04554	CDD6
8981	4D57	9776	04555	CDD7
8B21	4D58	9777	04556	CDD8
8E0A	4D59	9778	04557	CDD9
9059	7423	EAA1	08403	CDDA
967D	4D5B	977A	04559	CDDB
990A	4D5C	977B	04560	CDDC
617E	4D5D	977C	04561	CDDD
6291	4D5E	977D	04562	CDDE
6B32	4D5F	977E	04563	CDDF
6C83	4D60	9780	04564	CDE0
6D74	4D61	9781	04565	CDE1
7FCC	4D62	9782	04566	CDE2
7FFC	4D63	9783	04567	CDE3
6DC0	4D64	9784	04568	CDE4
7F85	4D65	9785	04569	CDE5
87BA	4D66	9786	04570	CDE6
88F8	4D67	9787	04571	CDE7
6765	4D68	9788	04572	CDE8
840A				CDE9
983C	4D6A	978A	04574	CDEA
96F7	4D6B	978B	04575	CDEB
6D1B	4D6C	978C	04576	CDEC
7D61	4D6D	978D	04577	CDED
843D	4D6E	978E	04578	CDEE
916A	4D6F	978F	04579	CDEF
4E71	4D70	9790	04580	CDF0
5375	4D71	9791	04581	CDF1
5D50	4D72	9792	04582	CDF2
6B04	4D73	9793	04583	CDF3
6FEB	4D74	9794	04584	CDF4
85CD	4D75	9795	04585	CDF5
862D	4D76	9796	04586	CDF6
89A7	4D77	9797	04587	CDF7
5229	4D78	9798	04588	CDF8
540F	4D79	9799	04589	CDF9
5C65	4D7A	979A	04590	CDFA
674E	4D7B	979B	04591	CDFB
68A8	4D7C	979C	04592	CDFC
7406	4D7D	979D	04593	CDFD
7483	4D7E	979E	04594	CDFE
75E2	4E21	979F	04601	CEA1
88CF	4E22	97A0	04602	CEA2
88E1	4E23	97A1	04603	CEA3
91CC	4E24	97A2	04604	CEA4
96E2	4E25	97A3	04605	CEA5
9678	4E26	97A4	04606	CEA6
5F8B	4E27	97A5	04607	CEA7
7387	4E28	97A6	04608	CEA8
7ACB	4E29	97A7	04609	CEA9
844E	4E2A	97A8	04610	CEAA
63A0	4E2B	97A9	04611	CEAB
7565	4E2C	97AA	04612	CEAC
5289	4E2D	97AB	04613	CEAD
6D41	4E2E	97AC	04614	CEAE
6E9C	4E2F	97AD	04615	CEAF
7409	4E30	97AE	04616	CEB0
7559	4E31	97AF	04617	CEB1
786B	4E32	97B0	04618	CEB2
7C92	4E33	97B1	04619	CEB3
9686	4E34	97B2	04620	CEB4
7ADC	4E35	97B3	04621	CEB5
9F8D	4E36	97B4	04622	CEB6
4FB6	4E37	97B5	04623	CEB7
616E	4E38	97B6	04624	CEB8
65C5	4E39	97B7	04625	CEB9
865C	4E3A	97B8	04626	CEBA
4E86	4E3B	97B9	04627	CEBB
4EAE	4E3C	97BA	04628	CEBC
50DA	4E3D	97BB	04629	CEBD
4E21	4E3E	97BC	04630	CEBE
51CC	4E3F	97BD	04631	CEBF
5BEE	4E40	97BE	04632	CEC0
6599	4E41	97BF	04633	CEC1
6881	4E42	97C0	04634	CEC2
6DBC	4E43	97C1	04635	CEC3
731F	4E44	97C2	04636	CEC4
7642	4E45	97C3	04637	CEC5
77AD	4E46	97C4	04638	CEC6
7A1C	4E47	97C5	04639	CEC7
7CE7	4E48	97C6	04640	CEC8
826F	4E49	97C7	04641	CEC9
8AD2	4E4A	97C8	04642	CECA
E457				CECB
91CF	4E4C	97CA	04644	CECC
9675	4E4D	97CB	04645	CECD
9818	4E4E	97CC	04646	CECE
529B	4E4F	97CD	04647	CECF
7DD1	4E50	97CE	04648	CED0
502B	4E51	97CF	04649	CED1
5398	4E52	97D0	04650	CED2
6797	4E53	97D1	04651	CED3
6DCB	4E54	97D2	04652	CED4
71D0	4E55	97D3	04653	CED5
7433	4E56	97D4	04654	CED6
81E8	4E57	97D5	04655	CED7
8F2A	4E58	97D6	04656	CED8
96A3	4E59	97D7	04657	CED9
9C57	4E5A	97D8	04658	CEDA
9E9F	4E5B	97D9	04659	CEDB
7460	4E5C	97DA	04660	CEDC
5841	4E5D	97DB	04661	CEDD
6D99	4E5E	97DC	04662	CEDE
7D2F	4E5F	97DD	04663	CEDF
985E	4E60	97DE	04664	CEE0
4EE4	4E61	97DF	04665	CEE1
4F36	4E62	97E0	04666	CEE2
4F8B	4E63	97E1	04667	CEE3
51B7	4E64	97E2	04668	CEE4
52B1	4E65	97E3	04669	CEE5
5DBA	4E66	97E4	04670	CEE6
601C	4E67	97E5	04671	CEE7
73B2	4E68	97E6	04672	CEE8
793C	4E69	97E7	04673	CEE9
82D3	4E6A	97E8	04674	CEEA
9234	4E6B	97E9	04675	CEEB
96B7	4E6C	97EA	04676	CEEC
96F6	4E6D	97EB	04677	CEED
970A	4E6E	97EC	04678	CEEE
9E97	4E6F	97ED	04679	CEEF
9F62	4E70	97EE	04680	CEF0
66A6	4E71	97EF	04681	CEF1
6B74	4E72	97F0	04682	CEF2
5217	4E73	97F1	04683	CEF3
52A3	4E74	97F2	04684	CEF4
70C8	4E75	97F3	04685	CEF5
88C2	4E76	97F4	04686	CEF6
5EC9	4E77	97F5	04687	CEF7
604B	4E78	97F6	04688	CEF8
6190	4E79	97F7	04689	CEF9
E2CB				CEFA
E2E1				CEFB
E34B				CEFC
7DF4	4E7D	97FB	04693	CEFD
806F	4E7E	97FC	04694	CEFE
E3B0				CFA1
9023	4F22	9841	04702	CFA2
932C	4F23	9842	04703	CFA3
5442	4F24	9843	04704	CFA4
9B6F	4F25	9844	04705	CFA5
6AD3	4F26	9845	04706	CFA6
7089	4F27	9846	04707	CFA7
8CC2	4F28	9847	04708	CFA8
8DEF	4F29	9848	04709	CFA9
9732	4F2A	9849	04710	CFAA
52B4	4F2B	984A	04711	CFAB
5A41	4F2C	984B	04712	CFAC
5ECA	4F2D	984C	04713	CFAD
5F04	4F2E	984D	04714	CFAE
6717	4F2F	984E	04715	CFAF
697C	4F30	984F	04716	CFB0
E29D				CFB1
6D6A	4F32	9851	04718	CFB2
6F0F	4F33	9852	04719	CFB3
7262	4F34	9853	04720	CFB4
72FC	4F35	9854	04721	CFB5
7C60	6446	E2C4	06838	CFB6
8001	4F37	9856	04723	CFB7
807E	4F38	9857	04724	CFB8
881F				CFB9
90CE	4F3A	9859	04726	CFBA
516D	4F3B	985A	04727	CFBB
9E93	4F3C	985B	04728	CFBC
7984	4F3D	985C	04729	CFBD
808B	4F3E	985D	04730	CFBE
9332	4F3F	985E	04731	CFBF
8AD6	4F40	985F	04732	CFC0
502D	4F41	9860	04733	CFC1
548C	4F42	9861	04734	CFC2
8A71	4F43	9862	04735	CFC3
6B6A	4F44	9863	04736	CFC4
8CC4	4F45	9864	04737	CFC5
8107	4F46	9865	04738	CFC6
60D1	4F47	9866	04739	CFC7
67A0	4F48	9867	04740	CFC8
9DF2	4F49	9868	04741	CFC9
4E99	4F4A	9869	04742	CFCA
4E98	4F4B	986A	04743	CFCB
9C10	4F4C	986B	04744	CFCC
8A6B	4F4D	986C	04745	CFCD
85C1	4F4E	986D	04746	CFCE
8568	4F4F	986E	04747	CFCF
6900	4F50	986F	04748	CFD0
6E7E	4F51	9870	04749	CFD1
7897	4F52	9871	04750	CFD2
8155	4F53	9872	04751	CFD3
5F0C	5021	989F	04801	D0A1
4E10	5022	98A0	04802	D0A2
4E15	5023	98A1	04803	D0A3
4E2A	5024	98A2	04804	D0A4
4E31	5025	98A3	04805	D0A5
4E36	5026	98A4	04806	D0A6
4E3C	5027	98A5	04807	D0A7
4E3F	5028	98A6	04808	D0A8
E136				D0A9
4E56	502A	98A8	04810	D0AA
4E58	502B	98A9	04811	D0AB
4E82	502C	98AA	04812	D0AC
4E85	502D	98AB	04813	D0AD
E13B				D0AE
4E8A	502F	98AD	04815	D0AF
8212	5030	98AE	04816	D0B0
E20C				D0B1
4E8E	5032	98B0	04818	D0B2
4E9E	5033	98B1	04819	D0B3
E133				D0B4
4EA0	5035	98B3	04821	D0B5
4EA2	5036	98B4	04822	D0B6
4EB0	5037	98B5	04823	D0B7
4EB3	5038	98B6	04824	D0B8
4EB6	5039	98B7	04825	D0B9
4ECE	503A	98B8	04826	D0BA
4ECD	503B	98B9	04827	D0BB
4EC4	503C	98BA	04828	D0BC
4EC6	503D	98BB	04829	D0BD
4EC2	503E	98BC	04830	D0BE
E143				D0BF
4EDE	5040	98BE	04832	D0C0
E142				D0C1
4EDF	5042	98C0	04834	D0C2
4EF7	5043	98C1	04835	D0C3
4F09	5044	98C2	04836	D0C4
4F5A	5045	98C3	04837	D0C5
4F30	5046	98C4	04838	D0C6
4F5B	5047	98C5	04839	D0C7
4F5D	5048	98C6	04840	D0C8
4F57	5049	98C7	04841	D0C9
4F47	504A	98C8	04842	D0CA
4F76	504B	98C9	04843	D0CB
4F88	504C	98CA	04844	D0CC
4F8F	504D	98CB	04845	D0CD
4F98	504E	98CC	04846	D0CE
4F7B	504F	98CD	04847	D0CF
4F69	5050	98CE	04848	D0D0
4F70	5051	98CF	04849	D0D1
4F91	5052	98D0	04850	D0D2
4F6F	5053	98D1	04851	D0D3
4F86	5054	98D2	04852	D0D4
4F96	5055	98D3	04853	D0D5
4FAD	4B79	9699	04389	D0D6
4FD4	5057	98D5	04855	D0D7
4FDF	5058	98D6	04856	D0D8
4FCE	5059	98D7	04857	D0D9
4FD8	505A	98D8	04858	D0DA
4FDB	505B	98D9	04859	D0DB
4FD1	505C	98DA	04860	D0DC
4FDA	505D	98DB	04861	D0DD
4FD0	505E	98DC	04862	D0DE
4FE4	505F	98DD	04863	D0DF
4FE5	5060	98DE	04864	D0E0
501A	5061	98DF	04865	D0E1
5028	5062	98E0	04866	D0E2
5014	5063	98E1	04867	D0E3
502A	5064	98E2	04868	D0E4
E151				D0E5
5005	5066	98E4	04870	D0E6
4F1C	5067	98E5	04871	D0E7
4FF6	5068	98E6	04872	D0E8
5021	5069	98E7	04873	D0E9
5029	506A	98E8	04874	D0EA
502C	506B	98E9	04875	D0EB
4FFE	506C	98EA	04876	D0EC
4FEF	506D	98EB	04877	D0ED
5011	506E	98EC	04878	D0EE
5006	506F	98ED	04879	D0EF
5043	5070	98EE	04880	D0F0
5047	5071	98EF	04881	D0F1
6703	5072	98F0	04882	D0F2
5055	5073	98F1	04883	D0F3
5050	5074	98F2	04884	D0F4
5048	5075	98F3	04885	D0F5
505A	5076	98F4	04886	D0F6
5056	5077	98F5	04887	D0F7
506C	5078	98F6	04888	D0F8
5078	5079	98F7	04889	D0F9
5080	507A	98F8	04890	D0FA
509A	507B	98F9	04891	D0FB
5085	507C	98FA	04892	D0FC
50B4	507D	98FB	04893	D0FD
50B2	507E	98FC	04894	D0FE
50C9	5121	9940	04901	D1A1
E158				D1A2
50B3	5123	9942	04903	D1A3
50C2	5124	9943	04904	D1A4
50D6	5125	9944	04905	D1A5
50DE	5126	9945	04906	D1A6
50E5	5127	9946	04907	D1A7
50ED	5128	9947	04908	D1A8
50E3	5129	9948	04909	D1A9
50EE	512A	9949	04910	D1AA
50F9	512B	994A	04911	D1AB
50F5	512C	994B	04912	D1AC
5109	512D	994C	04913	D1AD
5101	512E	994D	04914	D1AE
5102	512F	994E	04915	D1AF
5116	5130	994F	04916	D1B0
5115	5131	9950	04917	D1B1
5114	5132	9951	04918	D1B2
511A	5133	9952	04919	D1B3
5121	5134	9953	04920	D1B4
513A	5135	9954	04921	D1B5
5137	5136	9955	04922	D1B6
513C	5137	9956	04923	D1B7
513B	5138	9957	04924	D1B8
513F	5139	9958	04925	D1B9
5140	513A	9959	04926	D1BA
5152	513B	995A	04927	D1BB
514C	513C	995B	04928	D1BC
E156				D1BD
5162	513E	995D	04930	D1BE
7AF8	513F	995E	04931	D1BF
5169	5140	995F	04932	D1C0
516A	5141	9960	04933	D1C1
516E	5142	9961	04934	D1C2
5180	5143	9962	04935	D1C3
5182	5144	9963	04936	D1C4
56D8	5145	9964	04937	D1C5
518C	5146	9965	04938	D1C6
5189	5147	9966	04939	D1C7
518F	5148	9967	04940	D1C8
5191	5149	9968	04941	D1C9
5193	514A	9969	04942	D1CA
E165				D1CB
5196	514C	996B	04944	D1CC
E166				D1CD
51A6	514E	996D	04946	D1CE
E167				D1CF
51A9	5150	996F	04948	D1D0
51AA	5151	9970	04949	D1D1
51AB	5152	9971	04950	D1D2
51B3	5153	9972	04951	D1D3
51B1	5154	9973	04952	D1D4
51B2	5155	9974	04953	D1D5
51B0	5156	9975	04954	D1D6
51B5	5157	9976	04955	D1D7
51BD	5158	9977	04956	D1D8
51C5	5159	9978	04957	D1D9
51C9	515A	9979	04958	D1DA
51DB	515B	997A	04959	D1DB
51E0	515C	997B	04960	D1DC
8655	515D	997C	04961	D1DD
51E9	515E	997D	04962	D1DE
51ED	515F	997E	04963	D1DF
51F0	5160	9980	04964	D1E0
51F5	5161	9981	04965	D1E1
51FE	5162	9982	04966	D1E2
E16C				D1E3
520B	5164	9984	04968	D1E4
5214	5165	9985	04969	D1E5
520E	5166	9986	04970	D1E6
5227	5167	9987	04971	D1E7
522A	5168	9988	04972	D1E8
522E	5169	9989	04973	D1E9
5233	516A	998A	04974	D1EA
5239	516B	998B	04975	D1EB
E174				D1EC
5244	516D	998D	04977	D1ED
524B	516E	998E	04978	D1EE
524C	516F	998F	04979	D1EF
525E	5170	9990	04980	D1F0
5254	5171	9991	04981	D1F1
526A	5172	9992	04982	D1F2
5274	5173	9993	04983	D1F3
5269	5174	9994	04984	D1F4
5273	5175	9995	04985	D1F5
527F	5176	9996	04986	D1F6
527D	5177	9997	04987	D1F7
528D	5178	9998	04988	D1F8
E17A				D1F9
5292	517A	999A	04990	D1FA
E472				D1FB
5288	517C	999C	04992	D1FC
5291	517D	999D	04993	D1FD
8FA8	517E	999E	04994	D1FE
8FA7	5221	999F	05001	D2A1
52AC	5222	99A0	05002	D2A2
52AD	5223	99A1	05003	D2A3
52BC	5224	99A2	05004	D2A4
52B5	5225	99A3	05005	D2A5
52C1	5226	99A4	05006	D2A6
52CD	5227	99A5	05007	D2A7
52D7	5228	99A6	05008	D2A8
52DE	5229	99A7	05009	D2A9
52E3	522A	99A8	05010	D2AA
52E6	522B	99A9	05011	D2AB
98ED	522C	99AA	05012	D2AC
52E0	522D	99AB	05013	D2AD
52F3	522E	99AC	05014	D2AE
52F5	522F	99AD	05015	D2AF
52F8	5230	99AE	05016	D2B0
52F9	5231	99AF	05017	D2B1
5306	5232	99B0	05018	D2B2
5308	5233	99B1	05019	D2B3
7538	5234	99B2	05020	D2B4
530D	5235	99B3	05021	D2B5
5310	5236	99B4	05022	D2B6
530F	5237	99B5	05023	D2B7
5315	5238	99B6	05024	D2B8
531A	5239	99B7	05025	D2B9
5323	523A	99B8	05026	D2BA
532F	523B	99B9	05027	D2BB
5331	523C	99BA	05028	D2BC
5333	523D	99BB	05029	D2BD
5338	523E	99BC	05030	D2BE
5340	523F	99BD	05031	D2BF
5346	5240	99BE	05032	D2C0
5345	5241	99BF	05033	D2C1
4E17	5242	99C0	05034	D2C2
5349	5243	99C1	05035	D2C3
534D	5244	99C2	05036	D2C4
51D6	5245	99C3	05037	D2C5
535E	5246	99C4	05038	D2C6
5369	5247	99C5	05039	D2C7
536E	5248	99C6	05040	D2C8
5918	5249	99C7	05041	D2C9
537B	524A	99C8	05042	D2CA
5377	524B	99C9	05043	D2CB
5382	524C	99CA	05044	D2CC
5396	524D	99CB	05045	D2CD
53A0	524E	99CC	05046	D2CE
53A6	524F	99CD	05047	D2CF
53A5	5250	99CE	05048	D2D0
53AE	5251	99CF	05049	D2D1
53B0	5252	99D0	05050	D2D2
53B6	5253	99D1	05051	D2D3
53C3	5254	99D2	05052	D2D4
7C12	5255	99D3	05053	D2D5
96D9	5256	99D4	05054	D2D6
53DF	5257	99D5	05055	D2D7
66FC	5258	99D6	05056	D2D8
71EE	5259	99D7	05057	D2D9
53EE	525A	99D8	05058	D2DA
53E8	525B	99D9	05059	D2DB
53ED	525C	99DA	05060	D2DC
53FA	525D	99DB	05061	D2DD
5401	525E	99DC	05062	D2DE
543D	525F	99DD	05063	D2DF
5440	5260	99DE	05064	D2E0
542C	5261	99DF	05065	D2E1
542D	5262	99E0	05066	D2E2
543C	5263	99E1	05067	D2E3
542E	5264	99E2	05068	D2E4
5436	5265	99E3	05069	D2E5
5429	5266	99E4	05070	D2E6
E194				D2E7
544E	5268	99E6	05072	D2E8
548F	5269	99E7	05073	D2E9
5475	526A	99E8	05074	D2EA
548E	526B	99E9	05075	D2EB
545F	526C	99EA	05076	D2EC
5471	526D	99EB	05077	D2ED
5477	526E	99EC	05078	D2EE
5470	526F	99ED	05079	D2EF
5492	5270	99EE	05080	D2F0
547B	5271	99EF	05081	D2F1
5480	5272	99F0	05082	D2F2
5476	5273	99F1	05083	D2F3
5484	5274	99F2	05084	D2F4
5490	5275	99F3	05085	D2F5
5486	5276	99F4	05086	D2F6
54C7	5277	99F5	05087	D2F7
54A2	5278	99F6	05088	D2F8
54B8	5279	99F7	05089	D2F9
54A5	527A	99F8	05090	D2FA
E197				D2FB
54C4	527C	99FA	05092	D2FC
54C8	527D	99FB	05093	D2FD
54A8	527E	99FC	05094	D2FE
54AB	5321	9A40	05101	D3A1
54C2	5322	9A41	05102	D3A2
54A4	5323	9A42	05103	D3A3
54BE	5324	9A43	05104	D3A4
54BC	5325	9A44	05105	D3A5
54D8	5326	9A45	05106	D3A6
54E5	5327	9A46	05107	D3A7
54E6	5328	9A47	05108	D3A8
550F	5329	9A48	05109	D3A9
5514	532A	9A49	05110	D3AA
E19B				D3AB
54EE	532C	9A4B	05112	D3AC
54ED	532D	9A4C	05113	D3AD
54FA	532E	9A4D	05114	D3AE
54E2	532F	9A4E	05115	D3AF
E1A1				D3B0
5540	5331	9A50	05117	D3B1
5563	5332	9A51	05118	D3B2
E1A2				D3B3
552E	5334	9A53	05120	D3B4
555C	5335	9A54	05121	D3B5
5545	5336	9A55	05122	D3B6
5556	5337	9A56	05123	D3B7
5557	5338	9A57	05124	D3B8
5538	5339	9A58	05125	D3B9
E19F				D3BA
555D	533B	9A5A	05127	D3BB
E1A6				D3BC
5580	533D	9A5C	05129	D3BD
54AF	533E	9A5D	05130	D3BE
558A	533F	9A5E	05131	D3BF
559F	5340	9A5F	05132	D3C0
557B	5341	9A60	05133	D3C1
557E	5342	9A61	05134	D3C2
5598	5343	9A62	05135	D3C3
559E	5344	9A63	05136	D3C4
55AE	5345	9A64	05137	D3C5
557C	5346	9A65	05138	D3C6
5583	5347	9A66	05139	D3C7
55A9	5348	9A67	05140	D3C8
5587	5349	9A68	05141	D3C9
55A8	534A	9A69	05142	D3CA
55DA	534B	9A6A	05143	D3CB
55C5	534C	9A6B	05144	D3CC
55DF	534D	9A6C	05145	D3CD
55C4	534E	9A6D	05146	D3CE
55DC	534F	9A6E	05147	D3CF
55E4	5350	9A6F	05148	D3D0
55D4	5351	9A70	05149	D3D1
5614	5352	9A71	05150	D3D2
55F7	5353	9A72	05151	D3D3
5616	5354	9A73	05152	D3D4
55FE	5355	9A74	05153	D3D5
55FD	5356	9A75	05154	D3D6
561B	5357	9A76	05155	D3D7
55F9	5358	9A77	05156	D3D8
564E	5359	9A78	05157	D3D9
5650	535A	9A79	05158	D3DA
71DF	535B	9A7A	05159	D3DB
5634	535C	9A7B	05160	D3DC
5636	535D	9A7C	05161	D3DD
E1AA				D3DE
5638	535F	9A7E	05163	D3DF
566B	5360	9A80	05164	D3E0
5664	5361	9A81	05165	D3E1
562F	5362	9A82	05166	D3E2
566C	5363	9A83	05167	D3E3
566A	5364	9A84	05168	D3E4
5686	5365	9A85	05169	D3E5
5680	5366	9A86	05170	D3E6
568A	5367	9A87	05171	D3E7
56A0	5368	9A88	05172	D3E8
5694	5369	9A89	05173	D3E9
568F	536A	9A8A	05174	D3EA
56A5	536B	9A8B	05175	D3EB
E1AF				D3EC
56B6	536D	9A8D	05177	D3ED
56B4	536E	9A8E	05178	D3EE
56C2	536F	9A8F	05179	D3EF
56BC	5370	9A90	05180	D3F0
56C1	5371	9A91	05181	D3F1
56C3	5372	9A92	05182	D3F2
56C0	5373	9A93	05183	D3F3
56C8	5374	9A94	05184	D3F4
56CE	5375	9A95	05185	D3F5
56D1	5376	9A96	05186	D3F6
E1B0				D3F7
56D7	5378	9A98	05188	D3F8
56EE	5379	9A99	05189	D3F9
56F9	537A	9A9A	05190	D3FA
5700	537B	9A9B	05191	D3FB
56FF	537C	9A9C	05192	D3FC
5704	537D	9A9D	05193	D3FD
5709	537E	9A9E	05194	D3FE
5708	5421	9A9F	05201	D4A1
570B	5422	9AA0	05202	D4A2
E1B2				D4A3
5713	5424	9AA2	05204	D4A4
5718	5425	9AA3	05205	D4A5
5716	5426	9AA4	05206	D4A6
55C7	5427	9AA5	05207	D4A7
571C	5428	9AA6	05208	D4A8
5726	5429	9AA7	05209	D4A9
5737	542A	9AA8	05210	D4AA
5738	542B	9AA9	05211	D4AB
574E	542C	9AAA	05212	D4AC
573B	542D	9AAB	05213	D4AD
5740	542E	9AAC	05214	D4AE
574F	542F	9AAD	05215	D4AF
5769	5430	9AAE	05216	D4B0
57C0	5431	9AAF	05217	D4B1
5788	5432	9AB0	05218	D4B2
5761	5433	9AB1	05219	D4B3
577F	5434	9AB2	05220	D4B4
5789	5435	9AB3	05221	D4B5
5793	5436	9AB4	05222	D4B6
57A0	5437	9AB5	05223	D4B7
57B3	5438	9AB6	05224	D4B8
57A4	5439	9AB7	05225	D4B9
57AA	543A	9AB8	05226	D4BA
57B0	543B	9AB9	05227	D4BB
57C3	543C	9ABA	05228	D4BC
57C6	543D	9ABB	05229	D4BD
57D4	543E	9ABC	05230	D4BE
57D2	543F	9ABD	05231	D4BF
57D3	5440	9ABE	05232	D4C0
580A	5441	9ABF	05233	D4C1
57D6	5442	9AC0	05234	D4C2
57E3	5443	9AC1	05235	D4C3
E1BA				D4C4
E1BD				D4C5
581D	5446	9AC4	05238	D4C6
5872	5447	9AC5	05239	D4C7
5821	5448	9AC6	05240	D4C8
5862	5449	9AC7	05241	D4C9
584B	544A	9AC8	05242	D4CA
5870	544B	9AC9	05243	D4CB
6BC0	544C	9ACA	05244	D4CC
5852	544D	9ACB	05245	D4CD
583D	544E	9ACC	05246	D4CE
5879	544F	9ACD	05247	D4CF
5885	5450	9ACE	05248	D4D0
58B9	5451	9ACF	05249	D4D1
589F	5452	9AD0	05250	D4D2
58AB	5453	9AD1	05251	D4D3
58BA	5454	9AD2	05252	D4D4
58DE	5455	9AD3	05253	D4D5
58BB	5456	9AD4	05254	D4D6
58B8	5457	9AD5	05255	D4D7
58AE	5458	9AD6	05256	D4D8
58C5	5459	9AD7	05257	D4D9
58D3	545A	9AD8	05258	D4DA
58D1	545B	9AD9	05259	D4DB
58D7	545C	9ADA	05260	D4DC
58D9	545D	9ADB	05261	D4DD
58D8	545E	9ADC	05262	D4DE
58E5	545F	9ADD	05263	D4DF
58DC	5460	9ADE	05264	D4E0
58E4	5461	9ADF	05265	D4E1
58DF	5462	9AE0	05266	D4E2
58EF	5463	9AE1	05267	D4E3
58F7	445B	92D9	03659	D4E4
58F9	5465	9AE3	05269	D4E5
58FB	5466	9AE4	05270	D4E6
58FC	5467	9AE5	05271	D4E7
58FD	5468	9AE6	05272	D4E8
5902	5469	9AE7	05273	D4E9
590A	546A	9AE8	05274	D4EA
5910	546B	9AE9	05275	D4EB
E1C8				D4EC
68A6	546D	9AEB	05277	D4ED
5925	546E	9AEC	05278	D4EE
592C	546F	9AED	05279	D4EF
592D	5470	9AEE	05280	D4F0
5932	5471	9AEF	05281	D4F1
5938	5472	9AF0	05282	D4F2
593E	5473	9AF1	05283	D4F3
7AD2	5474	9AF2	05284	D4F4
5955	5475	9AF3	05285	D4F5
5950	5476	9AF4	05286	D4F6
594E	5477	9AF5	05287	D4F7
595A	5478	9AF6	05288	D4F8
5958	5479	9AF7	05289	D4F9
5962	547A	9AF8	05290	D4FA
5960	547B	9AF9	05291	D4FB
5967	547C	9AFA	05292	D4FC
596C	547D	9AFB	05293	D4FD
5969	547E	9AFC	05294	D4FE
5978	5521	9B40	05301	D5A1
E1CD				D5A2
599D	5523	9B42	05303	D5A3
4F5E	5524	9B43	05304	D5A4
4FAB	5525	9B44	05305	D5A5
59A3	5526	9B45	05306	D5A6
59B2	5527	9B46	05307	D5A7
59C6	5528	9B47	05308	D5A8
59E8	5529	9B48	05309	D5A9
59DC	552A	9B49	05310	D5AA
598D	552B	9B4A	05311	D5AB
59D9	552C	9B4B	05312	D5AC
59DA	552D	9B4C	05313	D5AD
5A25	552E	9B4D	05314	D5AE
5A1F	552F	9B4E	05315	D5AF
5A11	5530	9B4F	05316	D5B0
5A1C	5531	9B50	05317	D5B1
5A09	5532	9B51	05318	D5B2
5A1A	5533	9B52	05319	D5B3
5A40	5534	9B53	05320	D5B4
5A6C	5535	9B54	05321	D5B5
5A49	5536	9B55	05322	D5B6
5A35	5537	9B56	05323	D5B7
5A36	5538	9B57	05324	D5B8
5A62	5539	9B58	05325	D5B9
5A6A	553A	9B59	05326	D5BA
5A9A	553B	9B5A	05327	D5BB
5ABC	553C	9B5B	05328	D5BC
E1D8				D5BD
5ACB	553E	9B5D	05330	D5BE
5AC2	553F	9B5E	05331	D5BF
5ABD	5540	9B5F	05332	D5C0
5AE3	5541	9B60	05333	D5C1
5AD7	5542	9B61	05334	D5C2
5AE6	5543	9B62	05335	D5C3
5AE9	5544	9B63	05336	D5C4
5AD6	5545	9B64	05337	D5C5
5AFA	5546	9B65	05338	D5C6
5AFB	5547	9B66	05339	D5C7
5B0C	5548	9B67	05340	D5C8
5B0B	5549	9B68	05341	D5C9
5B16	554A	9B69	05342	D5CA
5B32	554B	9B6A	05343	D5CB
5AD0	554C	9B6B	05344	D5CC
5B2A	554D	9B6C	05345	D5CD
5B36	554E	9B6D	05346	D5CE
5B3E	554F	9B6E	05347	D5CF
5B43	5550	9B6F	05348	D5D0
5B45	5551	9B70	05349	D5D1
5B40	5552	9B71	05350	D5D2
5B51	5553	9B72	05351	D5D3
5B55	5554	9B73	05352	D5D4
5B5A	5555	9B74	05353	D5D5
5B5B	5556	9B75	05354	D5D6
5B65	5557	9B76	05355	D5D7
5B69	5558	9B77	05356	D5D8
E789				D5D9
5B73	555A	9B79	05358	D5DA
5B75	555B	9B7A	05359	D5DB
5B78	555C	9B7B	05360	D5DC
E1DC				D5DD
5B7A	555E	9B7D	05362	D5DE
5B80	555F	9B7E	05363	D5DF
5B83	5560	9B80	05364	D5E0
5BA6	5561	9B81	05365	D5E1
5BB8	5562	9B82	05366	D5E2
E1E2				D5E3
5BC7	5564	9B84	05368	D5E4
5BC9	5565	9B85	05369	D5E5
5BD4	5566	9B86	05370	D5E6
5BD0	5567	9B87	05371	D5E7
5BE4	5568	9B88	05372	D5E8
5BE6	5569	9B89	05373	D5E9
5BE2	556A	9B8A	05374	D5EA
5BDE	556B	9B8B	05375	D5EB
5BE5	556C	9B8C	05376	D5EC
5BEB	556D	9B8D	05377	D5ED
5BF0	556E	9B8E	05378	D5EE
5BF6	556F	9B8F	05379	D5EF
5BF3	5570	9B90	05380	D5F0
5C05	5571	9B91	05381	D5F1
5C07	5572	9B92	05382	D5F2
5C08	5573	9B93	05383	D5F3
5C0D	5574	9B94	05384	D5F4
E65A				D5F5
5C20	5576	9B96	05386	D5F6
5C22	5577	9B97	05387	D5F7
E1EB				D5F8
E78A				D5F9
5C39	557A	9B9A	05390	D5FA
5C41	557B	9B9B	05391	D5FB
5C46	557C	9B9C	05392	D5FC
5C4E	557D	9B9D	05393	D5FD
5C53	557E	9B9E	05394	D5FE
5C50	5621	9B9F	05401	D6A1
E65C				D6A2
5B71	5623	9BA1	05403	D6A3
5C6C	5624	9BA2	05404	D6A4
5C6E	5625	9BA3	05405	D6A5
4E62	5626	9BA4	05406	D6A6
5C76	5627	9BA5	05407	D6A7
5C79	5628	9BA6	05408	D6A8
5C8C	5629	9BA7	05409	D6A9
5C91	562A	9BA8	05410	D6AA
5C94	562B	9BA9	05411	D6AB
599B	562C	9BAA	05412	D6AC
5CAB	562D	9BAB	05413	D6AD
5CBB	562E	9BAC	05414	D6AE
5CB6	562F	9BAD	05415	D6AF
5CBC	5630	9BAE	05416	D6B0
5CB7	5631	9BAF	05417	D6B1
5CC5	5632	9BB0	05418	D6B2
5CBE	5633	9BB1	05419	D6B3
5CC7	5634	9BB2	05420	D6B4
5CD9	5635	9BB3	05421	D6B5
5CE9	5636	9BB4	05422	D6B6
5CFD	5637	9BB5	05423	D6B7
E1EF				D6B8
5CED	5639	9BB7	05425	D6B9
5D8C	563A	9BB8	05426	D6BA
5CEA	563B	9BB9	05427	D6BB
5D0B	563C	9BBA	05428	D6BC
5D15	563D	9BBB	05429	D6BD
5D17	563E	9BBC	05430	D6BE
5D5C	563F	9BBD	05431	D6BF
5D1F	5640	9BBE	05432	D6C0
5D1B	5641	9BBF	05433	D6C1
5D11	5642	9BC0	05434	D6C2
5D14	5643	9BC1	05435	D6C3
5D22	5644	9BC2	05436	D6C4
5D1A	5645	9BC3	05437	D6C5
5D19	5646	9BC4	05438	D6C6
5D18	5647	9BC5	05439	D6C7
5D4C	5648	9BC6	05440	D6C8
5D52	5649	9BC7	05441	D6C9
5D4E	564A	9BC8	05442	D6CA
5D4B	564B	9BC9	05443	D6CB
5D6C	564C	9BCA	05444	D6CC
5D73	564D	9BCB	05445	D6CD
5D76	564E	9BCC	05446	D6CE
5D87	564F	9BCD	05447	D6CF
5D84	5650	9BCE	05448	D6D0
5D82	5651	9BCF	05449	D6D1
5DA2	5652	9BD0	05450	D6D2
5D9D	5653	9BD1	05451	D6D3
5DAC	5654	9BD2	05452	D6D4
5DAE	5655	9BD3	05453	D6D5
5DBD	5656	9BD4	05454	D6D6
5D90	5657	9BD5	05455	D6D7
5DB7	5658	9BD6	05456	D6D8
5DBC	5659	9BD7	05457	D6D9
5DC9	565A	9BD8	05458	D6DA
5DCD	565B	9BD9	05459	D6DB
5DD3	565C	9BDA	05460	D6DC
5DD2	565D	9BDB	05461	D6DD
5DD6	565E	9BDC	05462	D6DE
5DDB	565F	9BDD	05463	D6DF
5DEB	5660	9BDE	05464	D6E0
5DF2	5661	9BDF	05465	D6E1
5DF5	5662	9BE0	05466	D6E2
5E0B	5663	9BE1	05467	D6E3
5E1A	5664	9BE2	05468	D6E4
5E19	5665	9BE3	05469	D6E5
5E11	5666	9BE4	05470	D6E6
5E1B	5667	9BE5	05471	D6E7
5E36	5668	9BE6	05472	D6E8
5E37	5669	9BE7	05473	D6E9
5E44	566A	9BE8	05474	D6EA
E1F7				D6EB
5E40	566C	9BEA	05476	D6EC
5E4E	566D	9BEB	05477	D6ED
5E57	566E	9BEC	05478	D6EE
5E54	566F	9BED	05479	D6EF
5E5F	5670	9BEE	05480	D6F0
5E62	5671	9BEF	05481	D6F1
5E64	5672	9BF0	05482	D6F2
5E47	5673	9BF1	05483	D6F3
5E75	5674	9BF2	05484	D6F4
5E76	5675	9BF3	05485	D6F5
5E7A	5676	9BF4	05486	D6F6
9EBC	5677	9BF5	05487	D6F7
5E7F	5678	9BF6	05488	D6F8
5EA0	5679	9BF7	05489	D6F9
5EC1	567A	9BF8	05490	D6FA
5EC2	567B	9BF9	05491	D6FB
5EC8	567C	9BFA	05492	D6FC
E203				D6FD
E200				D6FE
5ED6	5721	9C40	05501	D7A1
5EE3	5722	9C41	05502	D7A2
5EDD	5723	9C42	05503	D7A3
5EDA	5724	9C43	05504	D7A4
5EDB	5725	9C44	05505	D7A5
E78B				D7A6
5EE1	5727	9C46	05507	D7A7
5EE8	5728	9C47	05508	D7A8
5EE9	5729	9C48	05509	D7A9
5EEC	572A	9C49	05510	D7AA
5EF1	572B	9C4A	05511	D7AB
5EF3	572C	9C4B	05512	D7AC
5EF0	572D	9C4C	05513	D7AD
E204				D7AE
E207				D7AF
5EFE	5730	9C4F	05516	D7B0
5F03	5731	9C50	05517	D7B1
5F09	5732	9C51	05518	D7B2
5F5D	5733	9C52	05519	D7B3
5F5C	5734	9C53	05520	D7B4
5F0B	5735	9C54	05521	D7B5
5F11	5736	9C55	05522	D7B6
5F16	5737	9C56	05523	D7B7
5F29	5738	9C57	05524	D7B8
E20E				D7B9
E210				D7BA
5F41	573B	9C5A	05527	D7BB
5F48	573C	9C5B	05528	D7BC
5F4C	573D	9C5C	05529	D7BD
5F4E	573E	9C5D	05530	D7BE
5F2F	573F	9C5E	05531	D7BF
5F51	5740	9C5F	05532	D7C0
E212				D7C1
5F57	5742	9C61	05534	D7C2
5F59	5743	9C62	05535	D7C3
5F61	5744	9C63	05536	D7C4
5F6D	5745	9C64	05537	D7C5
5F73	5746	9C65	05538	D7C6
5F77	5747	9C66	05539	D7C7
5F83	5748	9C67	05540	D7C8
5F82	5749	9C68	05541	D7C9
5F7F	574A	9C69	05542	D7CA
5F8A	574B	9C6A	05543	D7CB
5F88	574C	9C6B	05544	D7CC
5F91	574D	9C6C	05545	D7CD
5F87	574E	9C6D	05546	D7CE
5F9E	574F	9C6E	05547	D7CF
5F99	5750	9C6F	05548	D7D0
E217				D7D1
5FA0	5752	9C71	05550	D7D2
5FA8	5753	9C72	05551	D7D3
5FAD	5754	9C73	05552	D7D4
5FBC	5755	9C74	05553	D7D5
5FD6	5756	9C75	05554	D7D6
5FFB	5757	9C76	05555	D7D7
5FE4	5758	9C77	05556	D7D8
5FF8	5759	9C78	05557	D7D9
5FF1	575A	9C79	05558	D7DA
5FDD	575B	9C7A	05559	D7DB
60B3	575C	9C7B	05560	D7DC
5FFF	575D	9C7C	05561	D7DD
6021	575E	9C7D	05562	D7DE
6060	575F	9C7E	05563	D7DF
6019	5760	9C80	05564	D7E0
6010	5761	9C81	05565	D7E1
6029	5762	9C82	05566	D7E2
600E	5763	9C83	05567	D7E3
6031	5764	9C84	05568	D7E4
601B	5765	9C85	05569	D7E5
6015	5766	9C86	05570	D7E6
602B	5767	9C87	05571	D7E7
6026	5768	9C88	05572	D7E8
600F	5769	9C89	05573	D7E9
603A	576A	9C8A	05574	D7EA
605A	576B	9C8B	05575	D7EB
6041	576C	9C8C	05576	D7EC
606A	576D	9C8D	05577	D7ED
6077	576E	9C8E	05578	D7EE
605F	576F	9C8F	05579	D7EF
604A	5770	9C90	05580	D7F0
6046	5771	9C91	05581	D7F1
604D	5772	9C92	05582	D7F2
6063	5773	9C93	05583	D7F3
6043	5774	9C94	05584	D7F4
6064	5775	9C95	05585	D7F5
6042	5776	9C96	05586	D7F6
606C	5777	9C97	05587	D7F7
606B	5778	9C98	05588	D7F8
6059	5779	9C99	05589	D7F9
6081	577A	9C9A	05590	D7FA
608D	577B	9C9B	05591	D7FB
60E7	577C	9C9C	05592	D7FC
6083	577D	9C9D	05593	D7FD
609A	577E	9C9E	05594	D7FE
6084	5821	9C9F	05601	D8A1
609B	5822	9CA0	05602	D8A2
6096	5823	9CA1	05603	D8A3
E223				D8A4
6092	5825	9CA3	05605	D8A5
60A7	5826	9CA4	05606	D8A6
E221				D8A7
60E1	5828	9CA6	05608	D8A8
60B8	5829	9CA7	05609	D8A9
60E0	582A	9CA8	05610	D8AA
60D3	582B	9CA9	05611	D8AB
60B4	582C	9CAA	05612	D8AC
5FF0	582D	9CAB	05613	D8AD
60BD	582E	9CAC	05614	D8AE
60C6	582F	9CAD	05615	D8AF
60B5	5830	9CAE	05616	D8B0
E225				D8B1
614D	5832	9CB0	05618	D8B2
6115	5833	9CB1	05619	D8B3
6106	5834	9CB2	05620	D8B4
60F6	5835	9CB3	05621	D8B5
60F7	5836	9CB4	05622	D8B6
6100	5837	9CB5	05623	D8B7
60F4	5838	9CB6	05624	D8B8
60FA	5839	9CB7	05625	D8B9
6103	583A	9CB8	05626	D8BA
6121	583B	9CB9	05627	D8BB
60FB	583C	9CBA	05628	D8BC
60F1	583D	9CBB	05629	D8BD
610D	583E	9CBC	05630	D8BE
610E	583F	9CBD	05631	D8BF
6147	5840	9CBE	05632	D8C0
613E	5841	9CBF	05633	D8C1
6128	5842	9CC0	05634	D8C2
6127	5843	9CC1	05635	D8C3
614A	5844	9CC2	05636	D8C4
613F	5845	9CC3	05637	D8C5
613C	5846	9CC4	05638	D8C6
612C	5847	9CC5	05639	D8C7
6134	5848	9CC6	05640	D8C8
613D	5849	9CC7	05641	D8C9
6142	584A	9CC8	05642	D8CA
6144	584B	9CC9	05643	D8CB
6173	584C	9CCA	05644	D8CC
6177	584D	9CCB	05645	D8CD
6158	584E	9CCC	05646	D8CE
6159	584F	9CCD	05647	D8CF
615A	5850	9CCE	05648	D8D0
616B	5851	9CCF	05649	D8D1
6174	5852	9CD0	05650	D8D2
616F	5853	9CD1	05651	D8D3
6165	5854	9CD2	05652	D8D4
6171	5855	9CD3	05653	D8D5
615F	5856	9CD4	05654	D8D6
615D	5857	9CD5	05655	D8D7
6153	5858	9CD6	05656	D8D8
6175	5859	9CD7	05657	D8D9
6199	585A	9CD8	05658	D8DA
6196	585B	9CD9	05659	D8DB
6187	585C	9CDA	05660	D8DC
61AC	585D	9CDB	05661	D8DD
6194	585E	9CDC	05662	D8DE
619A	585F	9CDD	05663	D8DF
618A	5860	9CDE	05664	D8E0
6191	5861	9CDF	05665	D8E1
61AB	5862	9CE0	05666	D8E2
61AE	5863	9CE1	05667	D8E3
61CC	5864	9CE2	05668	D8E4
61CA	5865	9CE3	05669	D8E5
61C9	5866	9CE4	05670	D8E6
61F7	5867	9CE5	05671	D8E7
61C8	5868	9CE6	05672	D8E8
61C3	5869	9CE7	05673	D8E9
61C6	586A	9CE8	05674	D8EA
61BA	586B	9CE9	05675	D8EB
61CB	586C	9CEA	05676	D8EC
7F79	586D	9CEB	05677	D8ED
61CD	586E	9CEC	05678	D8EE
61E6	586F	9CED	05679	D8EF
61E3	5870	9CEE	05680	D8F0
61F6	5871	9CEF	05681	D8F1
61FA	5872	9CF0	05682	D8F2
61F4	5873	9CF1	05683	D8F3
61FF	5874	9CF2	05684	D8F4
61FD	5875	9CF3	05685	D8F5
61FC	5876	9CF4	05686	D8F6
E233				D8F7
6200	5878	9CF6	05688	D8F8
6208	5879	9CF7	05689	D8F9
6209	587A	9CF8	05690	D8FA
620D	587B	9CF9	05691	D8FB
620C	587C	9CFA	05692	D8FC
6214	587D	9CFB	05693	D8FD
621B	587E	9CFC	05694	D8FE
621E	5921	9D40	05701	D9A1
6221	5922	9D41	05702	D9A2
622A	5923	9D42	05703	D9A3
622E	5924	9D43	05704	D9A4
6230	5925	9D44	05705	D9A5
6232	5926	9D45	05706	D9A6
6233	5927	9D46	05707	D9A7
E237				D9A8
624E	5929	9D48	05709	D9A9
625E	592A	9D49	05710	D9AA
6263	592B	9D4A	05711	D9AB
625B	592C	9D4B	05712	D9AC
E23B				D9AD
E23A				D9AE
627C	592F	9D4E	05715	D9AF
6282	5930	9D4F	05716	D9B0
6289	5931	9D50	05717	D9B1
627E	5932	9D51	05718	D9B2
6292	5933	9D52	05719	D9B3
6293	5934	9D53	05720	D9B4
6296	5935	9D54	05721	D9B5
62D4	5936	9D55	05722	D9B6
6283	5937	9D56	05723	D9B7
6294	5938	9D57	05724	D9B8
62D7	5939	9D58	05725	D9B9
62D1	593A	9D59	05726	D9BA
62BB	593B	9D5A	05727	D9BB
62CF	593C	9D5B	05728	D9BC
62FF	593D	9D5C	05729	D9BD
62C6	593E	9D5D	05730	D9BE
64D4	593F	9D5E	05731	D9BF
62C8	5940	9D5F	05732	D9C0
62DC	5941	9D60	05733	D9C1
62CC	5942	9D61	05734	D9C2
62CA	5943	9D62	05735	D9C3
62C2	5944	9D63	05736	D9C4
62C7	5945	9D64	05737	D9C5
629B	5946	9D65	05738	D9C6
62C9	5947	9D66	05739	D9C7
630C	5948	9D67	05740	D9C8
62EE	5949	9D68	05741	D9C9
62F1	594A	9D69	05742	D9CA
6327	594B	9D6A	05743	D9CB
6302	594C	9D6B	05744	D9CC
E242				D9CD
62EF	594E	9D6D	05746	D9CE
62F5	594F	9D6E	05747	D9CF
6350	5950	9D6F	05748	D9D0
633E	5951	9D70	05749	D9D1
634D	5952	9D71	05750	D9D2
641C	5953	9D72	05751	D9D3
634F	5954	9D73	05752	D9D4
6396	5955	9D74	05753	D9D5
638E	5956	9D75	05754	D9D6
6380	5957	9D76	05755	D9D7
63AB	5958	9D77	05756	D9D8
6376	5959	9D78	05757	D9D9
63A3	595A	9D79	05758	D9DA
638F	595B	9D7A	05759	D9DB
6389	595C	9D7B	05760	D9DC
639F	595D	9D7C	05761	D9DD
63B5	595E	9D7D	05762	D9DE
636B	595F	9D7E	05763	D9DF
E24C				D9E0
E24F				D9E1
63E9	5962	9D82	05766	D9E2
63C0	5963	9D83	05767	D9E3
63C6	5964	9D84	05768	D9E4
63E3	5965	9D85	05769	D9E5
63C9	5966	9D86	05770	D9E6
63D2	5967	9D87	05771	D9E7
63F6	5968	9D88	05772	D9E8
63C4	5969	9D89	05773	D9E9
6416	596A	9D8A	05774	D9EA
6434	596B	9D8B	05775	D9EB
E251				D9EC
6413	596D	9D8D	05777	D9ED
6426	596E	9D8E	05778	D9EE
6436	596F	9D8F	05779	D9EF
651D	5970	9D90	05780	D9F0
6417	5971	9D91	05781	D9F1
6428	5972	9D92	05782	D9F2
640F	5973	9D93	05783	D9F3
6467	5974	9D94	05784	D9F4
E78C				D9F5
6476	5976	9D96	05786	D9F6
644E	5977	9D97	05787	D9F7
64B9	3349	8A68	01941	D9F8
6495	5979	9D99	05789	D9F9
6493	597A	9D9A	05790	D9FA
64A5	597B	9D9B	05791	D9FB
64A9	597C	9D9C	05792	D9FC
6488	597D	9D9D	05793	D9FD
64BC	597E	9D9E	05794	D9FE
64DA	5A21	9D9F	05801	DAA1
64D2	5A22	9DA0	05802	DAA2
64C5	5A23	9DA1	05803	DAA3
64C7	5A24	9DA2	05804	DAA4
64BB	5A25	9DA3	05805	DAA5
64D8	5A26	9DA4	05806	DAA6
64C2	5A27	9DA5	05807	DAA7
64F1	5A28	9DA6	05808	DAA8
64E7	5A29	9DA7	05809	DAA9
8209	5A2A	9DA8	05810	DAAA
64E0	5A2B	9DA9	05811	DAAB
64E1	5A2C	9DAA	05812	DAAC
62AC	5A2D	9DAB	05813	DAAD
64E3	5A2E	9DAC	05814	DAAE
64EF	5A2F	9DAD	05815	DAAF
652C	5A30	9DAE	05816	DAB0
E25A				DAB1
64F4	5A32	9DB0	05818	DAB2
64F2	5A33	9DB1	05819	DAB3
64FA	5A34	9DB2	05820	DAB4
6500	5A35	9DB3	05821	DAB5
64FD	5A36	9DB4	05822	DAB6
6518	5A37	9DB5	05823	DAB7
651C	5A38	9DB6	05824	DAB8
6522				DAB9
6524	5A3A	9DB8	05826	DABA
6523	5A3B	9DB9	05827	DABB
652B	5A3C	9DBA	05828	DABC
6534	5A3D	9DBB	05829	DABD
6535	5A3E	9DBC	05830	DABE
6537	5A3F	9DBD	05831	DABF
E78D				DAC0
6538	5A41	9DBF	05833	DAC1
754B	5A42	9DC0	05834	DAC2
6548	5A43	9DC1	05835	DAC3
6556	5A44	9DC2	05836	DAC4
6555	5A45	9DC3	05837	DAC5
654D	5A46	9DC4	05838	DAC6
6558	5A47	9DC5	05839	DAC7
655E	5A48	9DC6	05840	DAC8
655D	5A49	9DC7	05841	DAC9
6572	5A4A	9DC8	05842	DACA
E25D				DACB
6582	5A4C	9DCA	05844	DACC
6583	5A4D	9DCB	05845	DACD
8B8A	5A4E	9DCC	05846	DACE
659B	5A4F	9DCD	05847	DACF
659F	5A50	9DCE	05848	DAD0
65AB	5A51	9DCF	05849	DAD1
65B7	5A52	9DD0	05850	DAD2
65C3	5A53	9DD1	05851	DAD3
65C6	5A54	9DD2	05852	DAD4
65C1	5A55	9DD3	05853	DAD5
65C4	5A56	9DD4	05854	DAD6
65CC	5A57	9DD5	05855	DAD7
65D2	5A58	9DD6	05856	DAD8
65DB	5A59	9DD7	05857	DAD9
65D9	5A5A	9DD8	05858	DADA
65E0	5A5B	9DD9	05859	DADB
65E1	5A5C	9DDA	05860	DADC
65F1	5A5D	9DDB	05861	DADD
6772	5A5E	9DDC	05862	DADE
660A	5A5F	9DDD	05863	DADF
6603	5A60	9DDE	05864	DAE0
E268				DAE1
6773	5A62	9DE0	05866	DAE2
6635	5A63	9DE1	05867	DAE3
6636	5A64	9DE2	05868	DAE4
6634	5A65	9DE3	05869	DAE5
661C	5A66	9DE4	05870	DAE6
664F	5A67	9DE5	05871	DAE7
6644	5A68	9DE6	05872	DAE8
6649	5A69	9DE7	05873	DAE9
6641	5A6A	9DE8	05874	DAEA
665E	5A6B	9DE9	05875	DAEB
665D	5A6C	9DEA	05876	DAEC
6664	5A6D	9DEB	05877	DAED
6667	5A6E	9DEC	05878	DAEE
6668	5A6F	9DED	05879	DAEF
665F	5A70	9DEE	05880	DAF0
6662	5A71	9DEF	05881	DAF1
6670	5A72	9DF0	05882	DAF2
6683	5A73	9DF1	05883	DAF3
6688	5A74	9DF2	05884	DAF4
668E	5A75	9DF3	05885	DAF5
6689	5A76	9DF4	05886	DAF6
6684	5A77	9DF5	05887	DAF7
6698	5A78	9DF6	05888	DAF8
669D	5A79	9DF7	05889	DAF9
E271				DAFA
66B9	5A7B	9DF9	05891	DAFB
66C9	5A7C	9DFA	05892	DAFC
66BE	5A7D	9DFB	05893	DAFD
66BC	5A7E	9DFC	05894	DAFE
66C4	5B21	9E40	05901	DBA1
66B8	5B22	9E41	05902	DBA2
66D6	5B23	9E42	05903	DBA3
E272				DBA4
66E0	5B25	9E44	05905	DBA5
663F	5B26	9E45	05906	DBA6
66E6	5B27	9E46	05907	DBA7
66E9	5B28	9E47	05908	DBA8
66F0	5B29	9E48	05909	DBA9
E275				DBAA
66F7	5B2B	9E4A	05911	DBAB
670F	5B2C	9E4B	05912	DBAC
6716	5B2D	9E4C	05913	DBAD
E27F				DBAE
E280				DBAF
6727	5B30	9E4F	05916	DBB0
9738	5B31	9E50	05917	DBB1
672E	5B32	9E51	05918	DBB2
673F	5B33	9E52	05919	DBB3
6736	5B34	9E53	05920	DBB4
6741	5B35	9E54	05921	DBB5
6738	5B36	9E55	05922	DBB6
6737	5B37	9E56	05923	DBB7
6746	5B38	9E57	05924	DBB8
675E	5B39	9E58	05925	DBB9
6760	5B3A	9E59	05926	DBBA
6759	5B3B	9E5A	05927	DBBB
6763	5B3C	9E5B	05928	DBBC
6764	5B3D	9E5C	05929	DBBD
6789	5B3E	9E5D	05930	DBBE
6770	5B3F	9E5E	05931	DBBF
67A9	5B40	9E5F	05932	DBC0
677C	5B41	9E60	05933	DBC1
676A	5B42	9E61	05934	DBC2
678C	5B43	9E62	05935	DBC3
678B	5B44	9E63	05936	DBC4
E283				DBC5
67A1	5B46	9E65	05938	DBC6
6785	5B47	9E66	05939	DBC7
67B7	5B48	9E67	05940	DBC8
67EF	5B49	9E68	05941	DBC9
E289				DBCA
67EC	5B4B	9E6A	05943	DBCB
67B3	5B4C	9E6B	05944	DBCC
67E9	5B4D	9E6C	05945	DBCD
67B8	5B4E	9E6D	05946	DBCE
67E4	5B4F	9E6E	05947	DBCF
67DE	5B50	9E6F	05948	DBD0
67DD	5B51	9E70	05949	DBD1
67E2	5B52	9E71	05950	DBD2
67EE	5B53	9E72	05951	DBD3
67B9	5B54	9E73	05952	DBD4
67CE	5B55	9E74	05953	DBD5
67C6	5B56	9E75	05954	DBD6
67E7	5B57	9E76	05955	DBD7
6867	4930	954F	04116	DBD8
681E	5B59	9E78	05957	DBD9
6846	5B5A	9E79	05958	DBDA
6829	5B5B	9E7A	05959	DBDB
6840	5B5C	9E7B	05960	DBDC
684D	5B5D	9E7C	05961	DBDD
6832	5B5E	9E7D	05962	DBDE
684E	5B5F	9E7E	05963	DBDF
68B3	5B60	9E80	05964	DBE0
682B	5B61	9E81	05965	DBE1
6859	5B62	9E82	05966	DBE2
6863	5B63	9E83	05967	DBE3
6877	5B64	9E84	05968	DBE4
687F	5B65	9E85	05969	DBE5
689F	5B66	9E86	05970	DBE6
688F	5B67	9E87	05971	DBE7
68AD	5B68	9E88	05972	DBE8
6894	5B69	9E89	05973	DBE9
689D	5B6A	9E8A	05974	DBEA
E291				DBEB
E292				DBEC
68BC	456E	938E	03778	DBED
68B9	5B6E	9E8E	05978	DBEE
6874	5B6F	9E8F	05979	DBEF
68B5	5B70	9E90	05980	DBF0
68A0	5B71	9E91	05981	DBF1
68BA	5B72	9E92	05982	DBF2
690F	5B73	9E93	05983	DBF3
E6A6				DBF4
687E	5B75	9E95	05985	DBF5
6901	5B76	9E96	05986	DBF6
68CA	5B77	9E97	05987	DBF7
6908	5B78	9E98	05988	DBF8
68D8	5B79	9E99	05989	DBF9
6922	5B7A	9E9A	05990	DBFA
6926	5B7B	9E9B	05991	DBFB
68E1	5B7C	9E9C	05992	DBFC
E295				DBFD
68CD	5B7E	9E9E	05994	DBFE
68D4	5C21	9E9F	06001	DCA1
68E7	5C22	9EA0	06002	DCA2
68D5	5C23	9EA1	06003	DCA3
6936	5C24	9EA2	06004	DCA4
6912	5C25	9EA3	06005	DCA5
6904	5C26	9EA4	06006	DCA6
68D7	5C27	9EA5	06007	DCA7
68E3	5C28	9EA6	06008	DCA8
6925	5C29	9EA7	06009	DCA9
68F9	5C2A	9EA8	06010	DCAA
68E0	5C2B	9EA9	06011	DCAB
68EF	5C2C	9EAA	06012	DCAC
6928	5C2D	9EAB	06013	DCAD
692A	5C2E	9EAC	06014	DCAE
691A	5C2F	9EAD	06015	DCAF
6923	5C30	9EAE	06016	DCB0
6921	5C31	9EAF	06017	DCB1
68C6	5C32	9EB0	06018	DCB2
6979	5C33	9EB1	06019	DCB3
6977	5C34	9EB2	06020	DCB4
695C	5C35	9EB3	06021	DCB5
6978	5C36	9EB4	06022	DCB6
696B	5C37	9EB5	06023	DCB7
6954	5C38	9EB6	06024	DCB8
697E	5C39	9EB7	06025	DCB9
696E	5C3A	9EB8	06026	DCBA
6939	5C3B	9EB9	06027	DCBB
6974	5C3C	9EBA	06028	DCBC
E298				DCBD
6959	5C3E	9EBC	06030	DCBE
6930	5C3F	9EBD	06031	DCBF
6961	5C40	9EBE	06032	DCC0
695E	5C41	9EBF	06033	DCC1
695D	5C42	9EC0	06034	DCC2
6981	5C43	9EC1	06035	DCC3
696A	5C44	9EC2	06036	DCC4
69B2	5C45	9EC3	06037	DCC5
69AE	5C46	9EC4	06038	DCC6
69D0	5C47	9EC5	06039	DCC7
69BF	5C48	9EC6	06040	DCC8
69C1	5C49	9EC7	06041	DCC9
69D3	5C4A	9EC8	06042	DCCA
69BE	5C4B	9EC9	06043	DCCB
69CE	5C4C	9ECA	06044	DCCC
5BE8	5C4D	9ECB	06045	DCCD
69CA	5C4E	9ECC	06046	DCCE
69DD	5C4F	9ECD	06047	DCCF
69BB	5C50	9ECE	06048	DCD0
69C3	5C51	9ECF	06049	DCD1
69A7	5C52	9ED0	06050	DCD2
E2A1				DCD3
6991	5C54	9ED2	06052	DCD4
69A0	5C55	9ED3	06053	DCD5
699C	5C56	9ED4	06054	DCD6
6995	5C57	9ED5	06055	DCD7
69B4	5C58	9ED6	06056	DCD8
69DE	5C59	9ED7	06057	DCD9
69E8	5C5A	9ED8	06058	DCDA
6A02	5C5B	9ED9	06059	DCDB
6A1B	5C5C	9EDA	06060	DCDC
69FF	5C5D	9EDB	06061	DCDD
6B0A	5C5E	9EDC	06062	DCDE
69F9	5C5F	9EDD	06063	DCDF
69F2	5C60	9EDE	06064	DCE0
69E7	5C61	9EDF	06065	DCE1
6A05	5C62	9EE0	06066	DCE2
69B1	5C63	9EE1	06067	DCE3
6A1E	5C64	9EE2	06068	DCE4
69ED	5C65	9EE3	06069	DCE5
6A14	5C66	9EE4	06070	DCE6
69EB	5C67	9EE5	06071	DCE7
6A0A	5C68	9EE6	06072	DCE8
6A12	5C69	9EE7	06073	DCE9
6AC1	5C6A	9EE8	06074	DCEA
6A23	5C6B	9EE9	06075	DCEB
6A13	5C6C	9EEA	06076	DCEC
6A44	5C6D	9EEB	06077	DCED
6A0C	5C6E	9EEC	06078	DCEE
6A72	5C6F	9EED	06079	DCEF
6A36	5C70	9EEE	06080	DCF0
6A78	5C71	9EEF	06081	DCF1
6A47	5C72	9EF0	06082	DCF2
6A62	5C73	9EF1	06083	DCF3
6A59	5C74	9EF2	06084	DCF4
6A66	5C75	9EF3	06085	DCF5
6A48	5C76	9EF4	06086	DCF6
6A38	5C77	9EF5	06087	DCF7
6A22	5C78	9EF6	06088	DCF8
6A90	5C79	9EF7	06089	DCF9
6A8D	5C7A	9EF8	06090	DCFA
6AA0	5C7B	9EF9	06091	DCFB
6A84	5C7C	9EFA	06092	DCFC
6AA2	5C7D	9EFB	06093	DCFD
6AA3	5C7E	9EFC	06094	DCFE
6A97	5D21	9F40	06101	DDA1
8617	5D22	9F41	06102	DDA2
6ABB	5D23	9F42	06103	DDA3
6AC3	5D24	9F43	06104	DDA4
6AC2	5D25	9F44	06105	DDA5
6AB8	5D26	9F45	06106	DDA6
6AB3	5D27	9F46	06107	DDA7
E2A6				DDA8
E2AB				DDA9
6AD1	5D2A	9F49	06110	DDAA
6ADF	5D2B	9F4A	06111	DDAB
6AAA	5D2C	9F4B	06112	DDAC
6ADA	5D2D	9F4C	06113	DDAD
6AEA	5D2E	9F4D	06114	DDAE
6AFB	5D2F	9F4E	06115	DDAF
6B05	5D30	9F4F	06116	DDB0
8616	5D31	9F50	06117	DDB1
6AFA	5D32	9F51	06118	DDB2
6B12	5D33	9F52	06119	DDB3
6B16	5D34	9F53	06120	DDB4
9B31	5D35	9F54	06121	DDB5
6B1F	5D36	9F55	06122	DDB6
6B38	5D37	9F56	06123	DDB7
6B37	5D38	9F57	06124	DDB8
76DC	5D39	9F58	06125	DDB9
6B39	5D3A	9F59	06126	DDBA
98EE	5D3B	9F5A	06127	DDBB
6B47	5D3C	9F5B	06128	DDBC
6B43	5D3D	9F5C	06129	DDBD
6B49	5D3E	9F5D	06130	DDBE
6B50	5D3F	9F5E	06131	DDBF
6B59	5D40	9F5F	06132	DDC0
6B54	5D41	9F60	06133	DDC1
6B5B	5D42	9F61	06134	DDC2
6B5F	5D43	9F62	06135	DDC3
6B61	5D44	9F63	06136	DDC4
6B78	5D45	9F64	06137	DDC5
6B79	5D46	9F65	06138	DDC6
6B7F	5D47	9F66	06139	DDC7
6B80	5D48	9F67	06140	DDC8
6B84	5D49	9F68	06141	DDC9
6B83	5D4A	9F69	06142	DDCA
6B8D	5D4B	9F6A	06143	DDCB
6B98	5D4C	9F6B	06144	DDCC
6B95	5D4D	9F6C	06145	DDCD
6B9E	5D4E	9F6D	06146	DDCE
6BA4	5D4F	9F6E	06147	DDCF
6BAA	5D50	9F6F	06148	DDD0
6BAB	5D51	9F70	06149	DDD1
6BAF	5D52	9F71	06150	DDD2
6BB2	5D53	9F72	06151	DDD3
6BB1	5D54	9F73	06152	DDD4
6BB3	5D55	9F74	06153	DDD5
6BB7	5D56	9F75	06154	DDD6
6BBC	5D57	9F76	06155	DDD7
6BC6	5D58	9F77	06156	DDD8
6BCB	5D59	9F78	06157	DDD9
6BD3	5D5A	9F79	06158	DDDA
6BDF	5D5B	9F7A	06159	DDDB
6BEC	5D5C	9F7B	06160	DDDC
6BEB	5D5D	9F7C	06161	DDDD
6BF3	5D5E	9F7D	06162	DDDE
6BEF	5D5F	9F7E	06163	DDDF
9EBE	5D60	9F80	06164	DDE0
E2B0				DDE1
6C13	5D62	9F82	06166	DDE2
6C14	5D63	9F83	06167	DDE3
6C1B	5D64	9F84	06168	DDE4
6C24	5D65	9F85	06169	DDE5
6C23	5D66	9F86	06170	DDE6
6C5E	5D67	9F87	06171	DDE7
6C55	5D68	9F88	06172	DDE8
6C62	5D69	9F89	06173	DDE9
6C6A	5D6A	9F8A	06174	DDEA
6C82	5D6B	9F8B	06175	DDEB
6C8D	5D6C	9F8C	06176	DDEC
6C9A	5D6D	9F8D	06177	DDED
6C81	5D6E	9F8E	06178	DDEE
6C9B	5D6F	9F8F	06179	DDEF
6C7E	5D70	9F90	06180	DDF0
6C68	5D71	9F91	06181	DDF1
6C73	5D72	9F92	06182	DDF2
6C92	5D73	9F93	06183	DDF3
6C90	5D74	9F94	06184	DDF4
6CC4	5D75	9F95	06185	DDF5
6CF1	5D76	9F96	06186	DDF6
6CD3	5D77	9F97	06187	DDF7
6CBD	5D78	9F98	06188	DDF8
6CD7	5D79	9F99	06189	DDF9
6CC5	5D7A	9F9A	06190	DDFA
6CDD	5D7B	9F9B	06191	DDFB
6CAE	5D7C	9F9C	06192	DDFC
6CB1	5D7D	9F9D	06193	DDFD
6CBE	5D7E	9F9E	06194	DDFE
6CBA	5E21	9F9F	06201	DEA1
6CDB	5E22	9FA0	06202	DEA2
6CEF	5E23	9FA1	06203	DEA3
6CD9	5E24	9FA2	06204	DEA4
6CEA	5E25	9FA3	06205	DEA5
6D1F	5E26	9FA4	06206	DEA6
884D	5E27	9FA5	06207	DEA7
6D36	5E28	9FA6	06208	DEA8
6D2B	5E29	9FA7	06209	DEA9
6D3D	5E2A	9FA8	06210	DEAA
6D38	5E2B	9FA9	06211	DEAB
6D19	5E2C	9FAA	06212	DEAC
6D35	5E2D	9FAB	06213	DEAD
6D33	5E2E	9FAC	06214	DEAE
6D12	5E2F	9FAD	06215	DEAF
6D0C	5E30	9FAE	06216	DEB0
6D63	5E31	9FAF	06217	DEB1
6D93	5E32	9FB0	06218	DEB2
6D64	5E33	9FB1	06219	DEB3
6D5A	5E34	9FB2	06220	DEB4
6D79	5E35	9FB3	06221	DEB5
6D59	5E36	9FB4	06222	DEB6
E2C3				DEB7
6D95	5E38	9FB6	06224	DEB8
6D9B	4573	9393	03783	DEB9
6D85	5E3A	9FB8	06226	DEBA
6DF9	5E3B	9FB9	06227	DEBB
6E15	5E3C	9FBA	06228	DEBC
6E0A	5E3D	9FBB	06229	DEBD
6DB5	5E3E	9FBC	06230	DEBE
6DC7	5E3F	9FBD	06231	DEBF
6DE6	5E40	9FBE	06232	DEC0
6DB8	5E41	9FBF	06233	DEC1
6DC6	5E42	9FC0	06234	DEC2
6DEC	5E43	9FC1	06235	DEC3
6DDE	5E44	9FC2	06236	DEC4
6DCC	5E45	9FC3	06237	DEC5
6DE8	5E46	9FC4	06238	DEC6
6DD2	5E47	9FC5	06239	DEC7
6DC5	5E48	9FC6	06240	DEC8
6DFA	5E49	9FC7	06241	DEC9
6DD9	5E4A	9FC8	06242	DECA
6DE4	5E4B	9FC9	06243	DECB
6DD5	5E4C	9FCA	06244	DECC
6DEA	5E4D	9FCB	06245	DECD
6DEE	5E4E	9FCC	06246	DECE
6E2D	5E4F	9FCD	06247	DECF
E2C5				DED0
6E2E	5E51	9FCF	06249	DED1
6E19	5E52	9FD0	06250	DED2
6E72	5E53	9FD1	06251	DED3
6E5F	5E54	9FD2	06252	DED4
6E3E	5E55	9FD3	06253	DED5
6E23	5E56	9FD4	06254	DED6
6E6B	5E57	9FD5	06255	DED7
6E2B	5E58	9FD6	06256	DED8
6E76	5E59	9FD7	06257	DED9
6E4D	5E5A	9FD8	06258	DEDA
6E1F	5E5B	9FD9	06259	DEDB
6E43	5E5C	9FDA	06260	DEDC
6E3A	5E5D	9FDB	06261	DEDD
6E4E	5E5E	9FDC	06262	DEDE
6E24	5E5F	9FDD	06263	DEDF
6EFF	5E60	9FDE	06264	DEE0
6E1D	5E61	9FDF	06265	DEE1
6E38	5E62	9FE0	06266	DEE2
6E82	5E63	9FE1	06267	DEE3
6EAA	5E64	9FE2	06268	DEE4
6E98	5E65	9FE3	06269	DEE5
6EC9	5E66	9FE4	06270	DEE6
6EB7	5E67	9FE5	06271	DEE7
6ED3	5E68	9FE6	06272	DEE8
6EBD	5E69	9FE7	06273	DEE9
6EAF	5E6A	9FE8	06274	DEEA
6EC4	5E6B	9FE9	06275	DEEB
6EB2	5E6C	9FEA	06276	DEEC
6ED4	5E6D	9FEB	06277	DEED
E2CA				DEEE
6E8F	5E6F	9FED	06279	DEEF
6EA5	5E70	9FEE	06280	DEF0
6EC2	5E71	9FEF	06281	DEF1
6E9F	5E72	9FF0	06282	DEF2
6F41	5E73	9FF1	06283	DEF3
6F11	5E74	9FF2	06284	DEF4
E794				DEF5
E2CD				DEF6
6EF8	5E77	9FF5	06287	DEF7
E2CE				DEF8
6F3F	5E79	9FF7	06289	DEF9
6EF2	5E7A	9FF8	06290	DEFA
6F31	5E7B	9FF9	06291	DEFB
6EEF	5E7C	9FFA	06292	DEFC
6F32	5E7D	9FFB	06293	DEFD
6ECC	5E7E	9FFC	06294	DEFE
6F3E	5F21	E040	06301	DFA1
6F13	5F22	E041	06302	DFA2
6EF7	5F23	E042	06303	DFA3
6F86	5F24	E043	06304	DFA4
6F7A	5F25	E044	06305	DFA5
6F78	5F26	E045	06306	DFA6
6F81	5F27	E046	06307	DFA7
6F80	5F28	E047	06308	DFA8
6F6F	5F29	E048	06309	DFA9
6F5B	5F2A	E049	06310	DFAA
6FF3	5F2B	E04A	06311	DFAB
6F6D	5F2C	E04B	06312	DFAC
6F82	5F2D	E04C	06313	DFAD
6F7C	5F2E	E04D	06314	DFAE
6F58	5F2F	E04E	06315	DFAF
6F8E	5F30	E04F	06316	DFB0
6F91	5F31	E050	06317	DFB1
6FC2	5F32	E051	06318	DFB2
6F66	5F33	E052	06319	DFB3
6FB3	5F34	E053	06320	DFB4
6FA3	5F35	E054	06321	DFB5
6FA1	5F36	E055	06322	DFB6
6FA4	5F37	E056	06323	DFB7
6FB9	5F38	E057	06324	DFB8
6FC6	5F39	E058	06325	DFB9
6FAA	5F3A	E059	06326	DFBA
6FDF	5F3B	E05A	06327	DFBB
6FD5	5F3C	E05B	06328	DFBC
6FEC	5F3D	E05C	06329	DFBD
6FD4	5F3E	E05D	06330	DFBE
6FD8	5F3F	E05E	06331	DFBF
6FF1	5F40	E05F	06332	DFC0
6FEE	5F41	E060	06333	DFC1
E2D5				DFC2
7009	5F43	E062	06335	DFC3
700B	5F44	E063	06336	DFC4
6FFA	5F45	E064	06337	DFC5
7011	5F46	E065	06338	DFC6
7001	5F47	E066	06339	DFC7
700F	5F48	E067	06340	DFC8
6FFE	5F49	E068	06341	DFC9
701B	5F4A	E069	06342	DFCA
701A	5F4B	E06A	06343	DFCB
6F74	5F4C	E06B	06344	DFCC
701D	5F4D	E06C	06345	DFCD
7018	5F4E	E06D	06346	DFCE
701F	5F4F	E06E	06347	DFCF
7030	5F50	E06F	06348	DFD0
703E	5F51	E070	06349	DFD1
7032	5F52	E071	06350	DFD2
7051	5F53	E072	06351	DFD3
7063	5F54	E073	06352	DFD4
7099	5F55	E074	06353	DFD5
7092	5F56	E075	06354	DFD6
70AF	5F57	E076	06355	DFD7
70F1	5F58	E077	06356	DFD8
E2E0				DFD9
70B8	5F5A	E079	06358	DFDA
70B3	5F5B	E07A	06359	DFDB
70AE	5F5C	E07B	06360	DFDC
70DF	5F5D	E07C	06361	DFDD
70CB	5F5E	E07D	06362	DFDE
70DD	5F5F	E07E	06363	DFDF
70D9	5F60	E080	06364	DFE0
7109	5F61	E081	06365	DFE1
70FD	5F62	E082	06366	DFE2
711C	5F63	E083	06367	DFE3
7119	5F64	E084	06368	DFE4
7165	5F65	E085	06369	DFE5
E797				DFE6
7188	5F67	E087	06371	DFE7
7166	5F68	E088	06372	DFE8
E2E4				DFE9
714C	5F6A	E08A	06374	DFEA
7156	5F6B	E08B	06375	DFEB
716C	5F6C	E08C	06376	DFEC
718F	5F6D	E08D	06377	DFED
71FB	5F6E	E08E	06378	DFEE
7184	5F6F	E08F	06379	DFEF
7195	5F70	E090	06380	DFF0
71A8	5F71	E091	06381	DFF1
71AC	5F72	E092	06382	DFF2
E2E7				DFF3
71B9	5F74	E094	06384	DFF4
71BE	5F75	E095	06385	DFF5
71D2	5F76	E096	06386	DFF6
71C9	5F77	E097	06387	DFF7
71D4	5F78	E098	06388	DFF8
71CE	5F79	E099	06389	DFF9
71E0	5F7A	E09A	06390	DFFA
71EC	5F7B	E09B	06391	DFFB
71E7	5F7C	E09C	06392	DFFC
71F5	5F7D	E09D	06393	DFFD
71FC	5F7E	E09E	06394	DFFE
71F9	6021	E09F	06401	E0A1
E2E8				E0A2
720D	6023	E0A1	06403	E0A3
7210	6024	E0A2	06404	E0A4
721B	6025	E0A3	06405	E0A5
7228	6026	E0A4	06406	E0A6
722D	6027	E0A5	06407	E0A7
722C	6028	E0A6	06408	E0A8
7230	6029	E0A7	06409	E0A9
7232	602A	E0A8	06410	E0AA
E2F2				E0AB
723C	602C	E0AA	06412	E0AC
723F	602D	E0AB	06413	E0AD
7240	602E	E0AC	06414	E0AE
7246	602F	E0AD	06415	E0AF
724B	6030	E0AE	06416	E0B0
7258	6031	E0AF	06417	E0B1
7274	6032	E0B0	06418	E0B2
727E	6033	E0B1	06419	E0B3
7282	6034	E0B2	06420	E0B4
7281	6035	E0B3	06421	E0B5
7287	6036	E0B4	06422	E0B6
7292	6037	E0B5	06423	E0B7
7296	6038	E0B6	06424	E0B8
72A2	6039	E0B7	06425	E0B9
72A7	603A	E0B8	06426	E0BA
72B9	603B	E0B9	06427	E0BB
72B2	603C	E0BA	06428	E0BC
72C3	603D	E0BB	06429	E0BD
72C6	603E	E0BC	06430	E0BE
72C4	603F	E0BD	06431	E0BF
72CE	6040	E0BE	06432	E0C0
72D2	6041	E0BF	06433	E0C1
72E2	6042	E0C0	06434	E0C2
72E0	6043	E0C1	06435	E0C3
E2F5				E0C4
72F9	6045	E0C3	06437	E0C5
72F7	6046	E0C4	06438	E0C6
500F	6047	E0C5	06439	E0C7
7317	6048	E0C6	06440	E0C8
730A	6049	E0C7	06441	E0C9
731C	604A	E0C8	06442	E0CA
7316	604B	E0C9	06443	E0CB
731D	604C	E0CA	06444	E0CC
7334	604D	E0CB	06445	E0CD
732F	604E	E0CC	06446	E0CE
7329	604F	E0CD	06447	E0CF
7325	6050	E0CE	06448	E0D0
733E	6051	E0CF	06449	E0D1
734E	6052	E0D0	06450	E0D2
734F	6053	E0D1	06451	E0D3
9ED8	6054	E0D2	06452	E0D4
7357	6055	E0D3	06453	E0D5
736A	6056	E0D4	06454	E0D6
7368	6057	E0D5	06455	E0D7
7370	6058	E0D6	06456	E0D8
7378	6059	E0D7	06457	E0D9
7375	605A	E0D8	06458	E0DA
737B	605B	E0D9	06459	E0DB
737A	605C	E0DA	06460	E0DC
73C8	605D	E0DB	06461	E0DD
73B3	605E	E0DC	06462	E0DE
E2FD				E0DF
73BB	6060	E0DE	06464	E0E0
73C0	6061	E0DF	06465	E0E1
E301				E0E2
73EE	6063	E0E1	06467	E0E3
73DE	6064	E0E2	06468	E0E4
74A2	6065	E0E3	06469	E0E5
7405	6066	E0E4	06470	E0E6
746F	6067	E0E5	06471	E0E7
7425	6068	E0E6	06472	E0E8
73F8	6069	E0E7	06473	E0E9
E305				E0EA
743A	606B	E0E9	06475	E0EB
7455	606C	E0EA	06476	E0EC
743F	606D	E0EB	06477	E0ED
745F	606E	E0EC	06478	E0EE
7459	606F	E0ED	06479	E0EF
7441	6070	E0EE	06480	E0F0
745C	6071	E0EF	06481	E0F1
7469	6072	E0F0	06482	E0F2
7470	6073	E0F1	06483	E0F3
7463	6074	E0F2	06484	E0F4
746A	6075	E0F3	06485	E0F5
7464	7424	EAA2	08404	E0F6
747E	6077	E0F5	06487	E0F7
748B	6078	E0F6	06488	E0F8
749E	6079	E0F7	06489	E0F9
74A7	607A	E0F8	06490	E0FA
74CA	607B	E0F9	06491	E0FB
74CF	607C	E0FA	06492	E0FC
74D4	607D	E0FB	06493	E0FD
73F1	607E	E0FC	06494	E0FE
74E0	6121	E140	06501	E1A1
74E3	6122	E141	06502	E1A2
74E7	6123	E142	06503	E1A3
74E9	6124	E143	06504	E1A4
74EE	6125	E144	06505	E1A5
74F2	6126	E145	06506	E1A6
74F0	6127	E146	06507	E1A7
74F1	6128	E147	06508	E1A8
74F8	6129	E148	06509	E1A9
74F7	612A	E149	06510	E1AA
E30C				E1AB
7503	612C	E14B	06512	E1AC
7505	612D	E14C	06513	E1AD
750C	612E	E14D	06514	E1AE
E79B				E1AF
E30D				E1B0
E30F				E1B1
E79C				E1B2
751E	6133	E152	06519	E1B3
E310				E1B4
752C	6135	E154	06521	E1B5
753C	6136	E155	06522	E1B6
7544	6137	E156	06523	E1B7
754D	6138	E157	06524	E1B8
754A	6139	E158	06525	E1B9
7549	613A	E159	06526	E1BA
755B	613B	E15A	06527	E1BB
7546	613C	E15B	06528	E1BC
755A	613D	E15C	06529	E1BD
7569	613E	E15D	06530	E1BE
7564	613F	E15E	06531	E1BF
7567	6140	E15F	06532	E1C0
756B	6141	E160	06533	E1C1
756D	6142	E161	06534	E1C2
7578	6143	E162	06535	E1C3
7576	6144	E163	06536	E1C4
7586	6145	E164	06537	E1C5
7587	6146	E165	06538	E1C6
7574	6147	E166	06539	E1C7
758A	6148	E167	06540	E1C8
7589	6149	E168	06541	E1C9
7582	614A	E169	06542	E1CA
7594	614B	E16A	06543	E1CB
759A	614C	E16B	06544	E1CC
759D	614D	E16C	06545	E1CD
75A5	614E	E16D	06546	E1CE
75A3	614F	E16E	06547	E1CF
75C2	6150	E16F	06548	E1D0
75B3	6151	E170	06549	E1D1
75C3	6152	E171	06550	E1D2
75B5	6153	E172	06551	E1D3
75BD	6154	E173	06552	E1D4
75B8	6155	E174	06553	E1D5
E313				E1D6
75B1	6157	E176	06555	E1D7
75CD	6158	E177	06556	E1D8
75CA	6159	E178	06557	E1D9
75D2	615A	E179	06558	E1DA
75D9	615B	E17A	06559	E1DB
75E3	615C	E17B	06560	E1DC
75DE	615D	E17C	06561	E1DD
75FE	615E	E17D	06562	E1DE
75FF	615F	E17E	06563	E1DF
75FC	6160	E180	06564	E1E0
7601	6161	E181	06565	E1E1
75F0	6162	E182	06566	E1E2
75FA	6163	E183	06567	E1E3
75F2	6164	E184	06568	E1E4
75F3	6165	E185	06569	E1E5
760B	6166	E186	06570	E1E6
760D	6167	E187	06571	E1E7
7609	6168	E188	06572	E1E8
761F	6169	E189	06573	E1E9
7627	616A	E18A	06574	E1EA
7620	616B	E18B	06575	E1EB
7621	616C	E18C	06576	E1EC
7622	616D	E18D	06577	E1ED
7624	616E	E18E	06578	E1EE
7634	616F	E18F	06579	E1EF
7630	6170	E190	06580	E1F0
763B	6171	E191	06581	E1F1
7647	6172	E192	06582	E1F2
7648	6173	E193	06583	E1F3
7646	6174	E194	06584	E1F4
765C	6175	E195	06585	E1F5
7658	6176	E196	06586	E1F6
7661	6177	E197	06587	E1F7
7662	6178	E198	06588	E1F8
7668	6179	E199	06589	E1F9
7669	617A	E19A	06590	E1FA
766A	617B	E19B	06591	E1FB
7667	617C	E19C	06592	E1FC
766C	617D	E19D	06593	E1FD
7670	617E	E19E	06594	E1FE
7672	6221	E19F	06601	E2A1
7676	6222	E1A0	06602	E2A2
E79D				E2A3
767C	6224	E1A2	06604	E2A4
7680	6225	E1A3	06605	E2A5
7683	6226	E1A4	06606	E2A6
7688	6227	E1A5	06607	E2A7
768B	6228	E1A6	06608	E2A8
E318				E2A9
7696	622A	E1A8	06610	E2AA
E319				E2AB
7699	622C	E1AA	06612	E2AC
769A	622D	E1AB	06613	E2AD
76B0	622E	E1AC	06614	E2AE
76B4	622F	E1AD	06615	E2AF
76B8	6230	E1AE	06616	E2B0
76B9	6231	E1AF	06617	E2B1
76BA	6232	E1B0	06618	E2B2
76C2	6233	E1B1	06619	E2B3
76CD	6234	E1B2	06620	E2B4
76D6	6235	E1B3	06621	E2B5
76D2	6236	E1B4	06622	E2B6
76DE	6237	E1B5	06623	E2B7
76E1	6238	E1B6	06624	E2B8
76E5	6239	E1B7	06625	E2B9
76E7	623A	E1B8	06626	E2BA
76EA	623B	E1B9	06627	E2BB
862F	623C	E1BA	06628	E2BC
76FB	623D	E1BB	06629	E2BD
7708	623E	E1BC	06630	E2BE
7707	623F	E1BD	06631	E2BF
7704	6240	E1BE	06632	E2C0
7729	6241	E1BF	06633	E2C1
7724	6242	E1C0	06634	E2C2
771E	6243	E1C1	06635	E2C3
7725	6244	E1C2	06636	E2C4
7726	6245	E1C3	06637	E2C5
771B	6246	E1C4	06638	E2C6
7737	6247	E1C5	06639	E2C7
7738	6248	E1C6	06640	E2C8
7747	6249	E1C7	06641	E2C9
775A	624A	E1C8	06642	E2CA
7768	624B	E1C9	06643	E2CB
776B	624C	E1CA	06644	E2CC
775B	624D	E1CB	06645	E2CD
7765	624E	E1CC	06646	E2CE
777F	624F	E1CD	06647	E2CF
777E	6250	E1CE	06648	E2D0
7779	6251	E1CF	06649	E2D1
778E	6252	E1D0	06650	E2D2
778B	6253	E1D1	06651	E2D3
7791	6254	E1D2	06652	E2D4
77A0	6255	E1D3	06653	E2D5
779E	6256	E1D4	06654	E2D6
77B0	6257	E1D5	06655	E2D7
77B6	6258	E1D6	06656	E2D8
77B9	6259	E1D7	06657	E2D9
77BF	625A	E1D8	06658	E2DA
77BC	625B	E1D9	06659	E2DB
77BD	625C	E1DA	06660	E2DC
77BB	625D	E1DB	06661	E2DD
E320				E2DE
77CD	625F	E1DD	06663	E2DF
77D7	6260	E1DE	06664	E2E0
77DA	6261	E1DF	06665	E2E1
77DC	6262	E1E0	06666	E2E2
77E3	6263	E1E1	06667	E2E3
77EE	6264	E1E2	06668	E2E4
77FC	6265	E1E3	06669	E2E5
780C	6266	E1E4	06670	E2E6
7812	6267	E1E5	06671	E2E7
783F	395C	8D7B	02560	E2E8
7820	6269	E1E7	06673	E2E9
783A	4557	9376	03755	E2EA
7845	626B	E1E9	06675	E2EB
788E	626C	E1EA	06676	E2EC
E7A0				E2ED
7886	626E	E1EC	06678	E2EE
E328				E2EF
789A	6270	E1EE	06680	E2F0
788C	6271	E1EF	06681	E2F1
78A3	6272	E1F0	06682	E2F2
78B5	6273	E1F1	06683	E2F3
78AA	6274	E1F2	06684	E2F4
78AF	6275	E1F3	06685	E2F5
78D1	6276	E1F4	06686	E2F6
78C6	6277	E1F5	06687	E2F7
78CB	6278	E1F6	06688	E2F8
78D4	6279	E1F7	06689	E2F9
78BE	627A	E1F8	06690	E2FA
78BC	627B	E1F9	06691	E2FB
78C5	627C	E1FA	06692	E2FC
78CA	627D	E1FB	06693	E2FD
78EC	627E	E1FC	06694	E2FE
78E7	6321	E240	06701	E3A1
78DA	6322	E241	06702	E3A2
78FD	6323	E242	06703	E3A3
78F4	6324	E243	06704	E3A4
7907	6325	E244	06705	E3A5
7912	6326	E245	06706	E3A6
7911	6327	E246	06707	E3A7
7919	6328	E247	06708	E3A8
792C	6329	E248	06709	E3A9
792B	632A	E249	06710	E3AA
7940	632B	E24A	06711	E3AB
7960	632C	E24B	06712	E3AC
7957	632D	E24C	06713	E3AD
795F	632E	E24D	06714	E3AE
795A	632F	E24E	06715	E3AF
7955	6330	E24F	06716	E3B0
7953	6331	E250	06717	E3B1
797A	6332	E251	06718	E3B2
797F	6333	E252	06719	E3B3
798A	6334	E253	06720	E3B4
799D	6335	E254	06721	E3B5
79A7	6336	E255	06722	E3B6
9F4B	6337	E256	06723	E3B7
79AA	6338	E257	06724	E3B8
79AE	6339	E258	06725	E3B9
79B3	633A	E259	06726	E3BA
79B9	633B	E25A	06727	E3BB
79BA	633C	E25B	06728	E3BC
79C9	633D	E25C	06729	E3BD
79D5	633E	E25D	06730	E3BE
79E7	633F	E25E	06731	E3BF
79EC	6340	E25F	06732	E3C0
79E1	6341	E260	06733	E3C1
79E3	6342	E261	06734	E3C2
7A08	6343	E262	06735	E3C3
7A0D	6344	E263	06736	E3C4
7A18	6345	E264	06737	E3C5
7A19	6346	E265	06738	E3C6
7A20	6347	E266	06739	E3C7
7A1F	6348	E267	06740	E3C8
7980	6349	E268	06741	E3C9
E337				E3CA
7A3B	634B	E26A	06743	E3CB
7A3E	634C	E26B	06744	E3CC
7A37	634D	E26C	06745	E3CD
7A43	634E	E26D	06746	E3CE
7A57	634F	E26E	06747	E3CF
7A49	6350	E26F	06748	E3D0
7A61	6351	E270	06749	E3D1
7A62	6352	E271	06750	E3D2
7A69	6353	E272	06751	E3D3
9F9D	6354	E273	06752	E3D4
7A70	6355	E274	06753	E3D5
7A79	6356	E275	06754	E3D6
7A7D	6357	E276	06755	E3D7
7A88	6358	E277	06756	E3D8
E33E				E3D9
7A95	635A	E279	06758	E3DA
7A98	635B	E27A	06759	E3DB
7A96	635C	E27B	06760	E3DC
7AA9	635D	E27C	06761	E3DD
7AC8	635E	E27D	06762	E3DE
7AB0	635F	E27E	06763	E3DF
7AB6	6360	E280	06764	E3E0
7AC5	6361	E281	06765	E3E1
7AC4	6362	E282	06766	E3E2
7ABF	6363	E283	06767	E3E3
9083	6364	E284	06768	E3E4
7AC7	6365	E285	06769	E3E5
7ACA	6366	E286	06770	E3E6
7ACD	6367	E287	06771	E3E7
7ACF	6368	E288	06772	E3E8
7AD5	6369	E289	06773	E3E9
7AD3	636A	E28A	06774	E3EA
7AD9	636B	E28B	06775	E3EB
7ADA	636C	E28C	06776	E3EC
7ADD	636D	E28D	06777	E3ED
7AE1	636E	E28E	06778	E3EE
7AE2	636F	E28F	06779	E3EF
7AE6	6370	E290	06780	E3F0
7AED	6371	E291	06781	E3F1
7AF0	6372	E292	06782	E3F2
7B02	6373	E293	06783	E3F3
7B0F	6374	E294	06784	E3F4
7B0A	6375	E295	06785	E3F5
7B06	6376	E296	06786	E3F6
7B33	6377	E297	06787	E3F7
7B18	6378	E298	06788	E3F8
7B19	6379	E299	06789	E3F9
7B1E	637A	E29A	06790	E3FA
7B35	637B	E29B	06791	E3FB
7B28	637C	E29C	06792	E3FC
7B36	637D	E29D	06793	E3FD
7B50	637E	E29E	06794	E3FE
7B7A	6421	E29F	06801	E4A1
7B04	6422	E2A0	06802	E4A2
7B4D	6423	E2A1	06803	E4A3
7B0B	6424	E2A2	06804	E4A4
7B4C	6425	E2A3	06805	E4A5
7B45	6426	E2A4	06806	E4A6
E342				E4A7
7B65	6428	E2A6	06808	E4A8
7B74	6429	E2A7	06809	E4A9
7B67	642A	E2A8	06810	E4AA
7B70	642B	E2A9	06811	E4AB
7B71	642C	E2AA	06812	E4AC
7B6C	642D	E2AB	06813	E4AD
7B6E	642E	E2AC	06814	E4AE
7B9D	642F	E2AD	06815	E4AF
7B98	6430	E2AE	06816	E4B0
7B9F	6431	E2AF	06817	E4B1
7B8D	6432	E2B0	06818	E4B2
7B9C	6433	E2B1	06819	E4B3
7B9A	6434	E2B2	06820	E4B4
7B8B	6435	E2B3	06821	E4B5
7B92	6436	E2B4	06822	E4B6
7B8F	6437	E2B5	06823	E4B7
7B5D	6438	E2B6	06824	E4B8
E343				E4B9
7BCB	643A	E2B8	06826	E4BA
7BC1	643B	E2B9	06827	E4BB
7BCC	643C	E2BA	06828	E4BC
7BCF	643D	E2BB	06829	E4BD
7BB4	643E	E2BC	06830	E4BE
E348				E4BF
E349				E4C0
7BE9	6441	E2BF	06833	E4C1
7C11	6442	E2C0	06834	E4C2
7C14	6443	E2C1	06835	E4C3
7BE6	6444	E2C2	06836	E4C4
7BE5	6445	E2C3	06837	E4C5
7BED	4F36	9855	04722	E4C6
7C00	6447	E2C5	06839	E4C7
7C07	6448	E2C6	06840	E4C8
E7A1				E4C9
7BF3	644A	E2C8	06842	E4CA
7BF7	644B	E2C9	06843	E4CB
7C17	644C	E2CA	06844	E4CC
7C0D	644D	E2CB	06845	E4CD
7BF6	644E	E2CC	06846	E4CE
7C23	644F	E2CD	06847	E4CF
7C27	6450	E2CE	06848	E4D0
7C2A	6451	E2CF	06849	E4D1
7C1F	6452	E2D0	06850	E4D2
7C37	6453	E2D1	06851	E4D3
7C2B	6454	E2D2	06852	E4D4
7C3D	6455	E2D3	06853	E4D5
7C4C	6456	E2D4	06854	E4D6
7C43	6457	E2D5	06855	E4D7
7C54	6458	E2D6	06856	E4D8
7C4F	6459	E2D7	06857	E4D9
7C40	645A	E2D8	06858	E4DA
E34D				E4DB
E34E				E4DC
7C5F	645D	E2DB	06861	E4DD
7C64	645E	E2DC	06862	E4DE
7C56	645F	E2DD	06863	E4DF
7C65	6460	E2DE	06864	E4E0
7C6C	6461	E2DF	06865	E4E1
7C75	6462	E2E0	06866	E4E2
7C83	6463	E2E1	06867	E4E3
E351				E4E4
7CA4	6465	E2E3	06869	E4E5
7CAD	6466	E2E4	06870	E4E6
7CA2	6467	E2E5	06871	E4E7
7CAB	6468	E2E6	06872	E4E8
7CA1	6469	E2E7	06873	E4E9
7CA8	646A	E2E8	06874	E4EA
E354				E4EB
7CB2	646C	E2EA	06876	E4EC
7CB1	646D	E2EB	06877	E4ED
7CAE	646E	E2EC	06878	E4EE
7CB9	646F	E2ED	06879	E4EF
7CBD	6470	E2EE	06880	E4F0
7CC0	6471	E2EF	06881	E4F1
7CC5	6472	E2F0	06882	E4F2
7CC2	6473	E2F1	06883	E4F3
E356				E4F4
7CD2	6475	E2F3	06885	E4F5
7CDC	6476	E2F4	06886	E4F6
7CE2	6477	E2F5	06887	E4F7
9B3B	6478	E2F6	06888	E4F8
7CEF	6479	E2F7	06889	E4F9
7CF2	647A	E2F8	06890	E4FA
7CF4	647B	E2F9	06891	E4FB
7CF6	647C	E2FA	06892	E4FC
7CFA	647D	E2FB	06893	E4FD
7D06	647E	E2FC	06894	E4FE
7D02	6521	E340	06901	E5A1
7D1C	6522	E341	06902	E5A2
7D15	6523	E342	06903	E5A3
E35C				E5A4
7D45	6525	E344	06905	E5A5
7D4B	6526	E345	06906	E5A6
7D2E	6527	E346	06907	E5A7
7D32	6528	E347	06908	E5A8
7D3F	6529	E348	06909	E5A9
7D35	652A	E349	06910	E5AA
7D46	652B	E34A	06911	E5AB
E364				E5AC
7D56	652D	E34C	06913	E5AD
7D4E	652E	E34D	06914	E5AE
7D72	652F	E34E	06915	E5AF
7D68	6530	E34F	06916	E5B0
7D6E	6531	E350	06917	E5B1
7D4F	6532	E351	06918	E5B2
7D63	6533	E352	06919	E5B3
7D93	6534	E353	06920	E5B4
7D89	6535	E354	06921	E5B5
7D5B	6536	E355	06922	E5B6
7D8F	6537	E356	06923	E5B7
7D7D	6538	E357	06924	E5B8
E362				E5B9
7DBA	653A	E359	06926	E5BA
E365				E5BB
7DA3	653C	E35B	06928	E5BC
7DB5	653D	E35C	06929	E5BD
7DC7	653E	E35D	06930	E5BE
7DBD	653F	E35E	06931	E5BF
7DAB	6540	E35F	06932	E5C0
7E3D	6541	E360	06933	E5C1
7DA2	6542	E361	06934	E5C2
7DAF	6543	E362	06935	E5C3
7DDC	6544	E363	06936	E5C4
7DB8	6545	E364	06937	E5C5
E366				E5C6
7DB0	6547	E366	06939	E5C7
7DD8	6548	E367	06940	E5C8
E36F				E5C9
7DE4	654A	E369	06942	E5CA
7DDE	654B	E36A	06943	E5CB
7DFB	654C	E36B	06944	E5CC
7DF2	654D	E36C	06945	E5CD
7DE1	654E	E36D	06946	E5CE
7E05	654F	E36E	06947	E5CF
7E0A	6550	E36F	06948	E5D0
7E23	6551	E370	06949	E5D1
7E21	6552	E371	06950	E5D2
7E12	6553	E372	06951	E5D3
7E31	6554	E373	06952	E5D4
7E1F	6555	E374	06953	E5D5
7E09	6556	E375	06954	E5D6
7E0B	6557	E376	06955	E5D7
E371				E5D8
7E46	6559	E378	06957	E5D9
7E66	655A	E379	06958	E5DA
7E3B	655B	E37A	06959	E5DB
7E35	655C	E37B	06960	E5DC
7E39	655D	E37C	06961	E5DD
E373				E5DE
7E37	655F	E37E	06963	E5DF
7E32	6560	E380	06964	E5E0
7E3A	6561	E381	06965	E5E1
7E67	6562	E382	06966	E5E2
7E5D	6563	E383	06967	E5E3
7E56	6564	E384	06968	E5E4
7E5E	6565	E385	06969	E5E5
7E59	6566	E386	06970	E5E6
7E5A	6567	E387	06971	E5E7
7E79	6568	E388	06972	E5E8
7E6A	6569	E389	06973	E5E9
7E69	656A	E38A	06974	E5EA
7E7C	656B	E38B	06975	E5EB
7E7B	656C	E38C	06976	E5EC
7E83	656D	E38D	06977	E5ED
7DD5	656E	E38E	06978	E5EE
7E7D	656F	E38F	06979	E5EF
8FAE	6570	E390	06980	E5F0
7E7F	6571	E391	06981	E5F1
7E88	6572	E392	06982	E5F2
7E89	6573	E393	06983	E5F3
7E8C	6574	E394	06984	E5F4
7E92	6575	E395	06985	E5F5
7E90	6576	E396	06986	E5F6
7E93	6577	E397	06987	E5F7
7E94	6578	E398	06988	E5F8
7E96	6579	E399	06989	E5F9
7E8E	657A	E39A	06990	E5FA
E7A2				E5FB
7E9C	657C	E39C	06992	E5FC
7F38	657D	E39D	06993	E5FD
7F3A	657E	E39E	06994	E5FE
7F45	6621	E39F	07001	E6A1
7F4C	6622	E3A0	07002	E6A2
7F4D	6623	E3A1	07003	E6A3
7F4E	6624	E3A2	07004	E6A4
7F50	6625	E3A3	07005	E6A5
7F51	6626	E3A4	07006	E6A6
7F55	6627	E3A5	07007	E6A7
7F54	6628	E3A6	07008	E6A8
7F58	6629	E3A7	07009	E6A9
7F5F	662A	E3A8	07010	E6AA
7F60	662B	E3A9	07011	E6AB
7F68	662C	E3AA	07012	E6AC
7F69	662D	E3AB	07013	E6AD
7F67	662E	E3AC	07014	E6AE
7F78	662F	E3AD	07015	E6AF
7F82	6630	E3AE	07016	E6B0
7F86	6631	E3AF	07017	E6B1
7F83	6632	E3B0	07018	E6B2
7F88	6633	E3B1	07019	E6B3
7F87	6634	E3B2	07020	E6B4
7F8C	6635	E3B3	07021	E6B5
7F94	6636	E3B4	07022	E6B6
7F9E	6637	E3B5	07023	E6B7
7F9D	6638	E3B6	07024	E6B8
7F9A	6639	E3B7	07025	E6B9
7FA3	663A	E3B8	07026	E6BA
7FAF	663B	E3B9	07027	E6BB
7FB2	663C	E3BA	07028	E6BC
7FB9	663D	E3BB	07029	E6BD
7FAE	663E	E3BC	07030	E6BE
7FB6	663F	E3BD	07031	E6BF
7FB8	6640	E3BE	07032	E6C0
8B71	6641	E3BF	07033	E6C1
7FC5	6642	E3C0	07034	E6C2
7FC6	6643	E3C1	07035	E6C3
7FCA	6644	E3C2	07036	E6C4
7FD5	6645	E3C3	07037	E6C5
E37C				E6C6
7FE1	6647	E3C5	07039	E6C7
7FE6	6648	E3C6	07040	E6C8
E37F				E6C9
7FF3	664A	E3C8	07042	E6CA
7FF9	664B	E3C9	07043	E6CB
98DC	664C	E3CA	07044	E6CC
8006	664D	E3CB	07045	E6CD
8004	664E	E3CC	07046	E6CE
800B	664F	E3CD	07047	E6CF
8012	6650	E3CE	07048	E6D0
8018	6651	E3CF	07049	E6D1
8019	6652	E3D0	07050	E6D2
801C	6653	E3D1	07051	E6D3
8021	6654	E3D2	07052	E6D4
8028	6655	E3D3	07053	E6D5
803F	6656	E3D4	07054	E6D6
803B	6657	E3D5	07055	E6D7
804A	6658	E3D6	07056	E6D8
8046	6659	E3D7	07057	E6D9
8052	665A	E3D8	07058	E6DA
8058	665B	E3D9	07059	E6DB
805A	665C	E3DA	07060	E6DC
805F	665D	E3DB	07061	E6DD
8062	665E	E3DC	07062	E6DE
8068	665F	E3DD	07063	E6DF
8073	6660	E3DE	07064	E6E0
8072	6661	E3DF	07065	E6E1
8070	6662	E3E0	07066	E6E2
E385				E6E3
8079	6664	E3E2	07068	E6E4
807D	6665	E3E3	07069	E6E5
807F	6666	E3E4	07070	E6E6
8084	6667	E3E5	07071	E6E7
8086	6668	E3E6	07072	E6E8
8085	6669	E3E7	07073	E6E9
809B	666A	E3E8	07074	E6EA
8093	666B	E3E9	07075	E6EB
809A	666C	E3EA	07076	E6EC
80AD	666D	E3EB	07077	E6ED
5190	666E	E3EC	07078	E6EE
80AC	666F	E3ED	07079	E6EF
80DB	6670	E3EE	07080	E6F0
80E5	6671	E3EF	07081	E6F1
80D9	6672	E3F0	07082	E6F2
80DD	6673	E3F1	07083	E6F3
80C4	6674	E3F2	07084	E6F4
80DA	6675	E3F3	07085	E6F5
80D6	6676	E3F4	07086	E6F6
8109	6677	E3F5	07087	E6F7
80EF	6678	E3F6	07088	E6F8
80F1	6679	E3F7	07089	E6F9
811B	667A	E3F8	07090	E6FA
8129	667B	E3F9	07091	E6FB
8123	667C	E3FA	07092	E6FC
812F	667D	E3FB	07093	E6FD
814B	667E	E3FC	07094	E6FE
968B	6721	E440	07101	E7A1
8146	6722	E441	07102	E7A2
813E	6723	E442	07103	E7A3
E38D				E7A4
8151	6725	E444	07105	E7A5
80FC	6726	E445	07106	E7A6
E38E				E7A7
816E	6728	E447	07108	E7A8
8165	6729	E448	07109	E7A9
8166	672A	E449	07110	E7AA
8174	672B	E44A	07111	E7AB
8183	672C	E44B	07112	E7AC
8188	672D	E44C	07113	E7AD
818A	672E	E44D	07114	E7AE
8180	672F	E44E	07115	E7AF
8182	6730	E44F	07116	E7B0
81A0	6731	E450	07117	E7B1
8195	6732	E451	07118	E7B2
81A4	6733	E452	07119	E7B3
81A3	6734	E453	07120	E7B4
815F	6735	E454	07121	E7B5
8193	6736	E455	07122	E7B6
81A9	6737	E456	07123	E7B7
81B0	6738	E457	07124	E7B8
81B5	6739	E458	07125	E7B9
81BE	673A	E459	07126	E7BA
81B8	673B	E45A	07127	E7BB
81BD	673C	E45B	07128	E7BC
81C0	673D	E45C	07129	E7BD
81C2	673E	E45D	07130	E7BE
81BA	673F	E45E	07131	E7BF
81C9	6740	E45F	07132	E7C0
81CD	6741	E460	07133	E7C1
81D1	6742	E461	07134	E7C2
81D9	6743	E462	07135	E7C3
81D8	6744	E463	07136	E7C4
81C8	6745	E464	07137	E7C5
81DA	6746	E465	07138	E7C6
81DF	6747	E466	07139	E7C7
81E0	6748	E467	07140	E7C8
81E7	6749	E468	07141	E7C9
81FA	674A	E469	07142	E7CA
81FB	674B	E46A	07143	E7CB
81FE	674C	E46B	07144	E7CC
8201	674D	E46C	07145	E7CD
8202	674E	E46D	07146	E7CE
8205	674F	E46E	07147	E7CF
8207	6750	E46F	07148	E7D0
820A	6751	E470	07149	E7D1
820D	6752	E471	07150	E7D2
8210	6753	E472	07151	E7D3
8216	6754	E473	07152	E7D4
8229	6755	E474	07153	E7D5
822B	6756	E475	07154	E7D6
8238	6757	E476	07155	E7D7
8233	6758	E477	07156	E7D8
8240	6759	E478	07157	E7D9
8259	675A	E479	07158	E7DA
8258	675B	E47A	07159	E7DB
825D	675C	E47B	07160	E7DC
825A	675D	E47C	07161	E7DD
825F	675E	E47D	07162	E7DE
8264	675F	E47E	07163	E7DF
8262	6760	E480	07164	E7E0
E398				E7E1
826A	6762	E482	07166	E7E2
826B	6763	E483	07167	E7E3
E395				E7E4
8271	6765	E485	07169	E7E5
8277	6766	E486	07170	E7E6
8278	6767	E487	07171	E7E7
E399				E7E8
E39A				E7E9
8292	676A	E48A	07174	E7EA
82AB	676B	E48B	07175	E7EB
829F	676C	E48C	07176	E7EC
82BB	676D	E48D	07177	E7ED
82AC	676E	E48E	07178	E7EE
82E1	676F	E48F	07179	E7EF
E39F				E7F0
82DF	6771	E491	07181	E7F1
E39E				E7F2
82F4	6773	E493	07183	E7F3
82F3	6774	E494	07184	E7F4
82FA	6775	E495	07185	E7F5
8393	6776	E496	07186	E7F6
8303	6777	E497	07187	E7F7
82FB	6778	E498	07188	E7F8
82F9	6779	E499	07189	E7F9
82DE	677A	E49A	07190	E7FA
8306	677B	E49B	07191	E7FB
82DC	677C	E49C	07192	E7FC
8309	677D	E49D	07193	E7FD
82D9	677E	E49E	07194	E7FE
8335	6821	E49F	07201	E8A1
8334	6822	E4A0	07202	E8A2
8316	6823	E4A1	07203	E8A3
8332	6824	E4A2	07204	E8A4
8331	6825	E4A3	07205	E8A5
8340	6826	E4A4	07206	E8A6
8339	6827	E4A5	07207	E8A7
8350	6828	E4A6	07208	E8A8
8345	6829	E4A7	07209	E8A9
832F	682A	E4A8	07210	E8AA
E3A2				E8AB
8317	682C	E4AA	07212	E8AC
8318	682D	E4AB	07213	E8AD
8385	682E	E4AC	07214	E8AE
E7A6				E8AF
83AA	6830	E4AE	07216	E8B0
839F	6831	E4AF	07217	E8B1
83A2	6832	E4B0	07218	E8B2
8396	6833	E4B1	07219	E8B3
8323	6834	E4B2	07220	E8B4
838E	6835	E4B3	07221	E8B5
8387	6836	E4B4	07222	E8B6
838A	6837	E4B5	07223	E8B7
837C	6838	E4B6	07224	E8B8
83B5	6839	E4B7	07225	E8B9
8373	683A	E4B8	07226	E8BA
E3A3				E8BB
83A0	683C	E4BA	07228	E8BC
8389	683D	E4BB	07229	E8BD
83A8	683E	E4BC	07230	E8BE
83F4	683F	E4BD	07231	E8BF
8413	6840	E4BE	07232	E8C0
83EB	6841	E4BF	07233	E8C1
83CE	6842	E4C0	07234	E8C2
83FD	6843	E4C1	07235	E8C3
8403	6844	E4C2	07236	E8C4
83D8	6845	E4C3	07237	E8C5
840B	6846	E4C4	07238	E8C6
83C1	6847	E4C5	07239	E8C7
83F7	6848	E4C6	07240	E8C8
8407	6849	E4C7	07241	E8C9
83E0	684A	E4C8	07242	E8CA
E3A9				E8CB
840D	684C	E4CA	07244	E8CC
E3A8				E8CD
8420	684E	E4CC	07246	E8CE
83BD	684F	E4CD	07247	E8CF
8438	6850	E4CE	07248	E8D0
8506	6851	E4CF	07249	E8D1
83FB	6852	E4D0	07250	E8D2
846D	6853	E4D1	07251	E8D3
842A	6854	E4D2	07252	E8D4
843C	6855	E4D3	07253	E8D5
855A	6856	E4D4	07254	E8D6
8484	6857	E4D5	07255	E8D7
8477	6858	E4D6	07256	E8D8
846B	6859	E4D7	07257	E8D9
84AD	685A	E4D8	07258	E8DA
846E	685B	E4D9	07259	E8DB
8482	685C	E4DA	07260	E8DC
8469	685D	E4DB	07261	E8DD
8446	685E	E4DC	07262	E8DE
842C	685F	E4DD	07263	E8DF
846F	6860	E4DE	07264	E8E0
8479	6861	E4DF	07265	E8E1
8435	6862	E4E0	07266	E8E2
84CA	6863	E4E1	07267	E8E3
8462	6864	E4E2	07268	E8E4
84B9	6865	E4E3	07269	E8E5
84BF	6866	E4E4	07270	E8E6
849F	6867	E4E5	07271	E8E7
84D9	6868	E4E6	07272	E8E8
84CD	6869	E4E7	07273	E8E9
84BB	686A	E4E8	07274	E8EA
84DA	686B	E4E9	07275	E8EB
84D0	686C	E4EA	07276	E8EC
84C1	686D	E4EB	07277	E8ED
84C6	686E	E4EC	07278	E8EE
84D6	686F	E4ED	07279	E8EF
84A1	6870	E4EE	07280	E8F0
8521	6871	E4EF	07281	E8F1
84FF	6872	E4F0	07282	E8F2
84F4	6873	E4F1	07283	E8F3
8517	6874	E4F2	07284	E8F4
8518	6875	E4F3	07285	E8F5
852C	6876	E4F4	07286	E8F6
851F	6877	E4F5	07287	E8F7
8515	6878	E4F6	07288	E8F8
8514	6879	E4F7	07289	E8F9
84FC	687A	E4F8	07290	E8FA
8540	687B	E4F9	07291	E8FB
8563	687C	E4FA	07292	E8FC
8558	687D	E4FB	07293	E8FD
8548	687E	E4FC	07294	E8FE
8541	6921	E540	07301	E9A1
854A	3C49	8EC7	02841	E9A2
854B	6923	E542	07303	E9A3
8555	6924	E543	07304	E9A4
8580	6925	E544	07305	E9A5
85A4	6926	E545	07306	E9A6
8588	6927	E546	07307	E9A7
8591	6928	E547	07308	E9A8
858A	6929	E548	07309	E9A9
85A8	692A	E549	07310	E9AA
856D	692B	E54A	07311	E9AB
8594	692C	E54B	07312	E9AC
859B	692D	E54C	07313	E9AD
85AE	4C79	96F7	04489	E9AE
E3B6				E9AF
859C	6930	E54F	07316	E9B0
8577	6931	E550	07317	E9B1
857E	6932	E551	07318	E9B2
8590	6933	E552	07319	E9B3
85C9	6934	E553	07320	E9B4
85BA	6935	E554	07321	E9B5
85CF	6936	E555	07322	E9B6
85B9	6937	E556	07323	E9B7
85D0	6938	E557	07324	E9B8
85D5	6939	E558	07325	E9B9
E7A9				E9BA
85E5	693B	E55A	07327	E9BB
85DC	693C	E55B	07328	E9BC
85F9	693D	E55C	07329	E9BD
860A	693E	E55D	07330	E9BE
8613	693F	E55E	07331	E9BF
860B	6940	E55F	07332	E9C0
85FE	6941	E560	07333	E9C1
85FA	6942	E561	07334	E9C2
8606	6943	E562	07335	E9C3
8622	6944	E563	07336	E9C4
861A	6945	E564	07337	E9C5
8630	6946	E565	07338	E9C6
863F	6947	E566	07339	E9C7
864D	6948	E567	07340	E9C8
4E55	6949	E568	07341	E9C9
E3C2				E9CA
865F	694B	E56A	07343	E9CB
8667	694C	E56B	07344	E9CC
8671	694D	E56C	07345	E9CD
8693	694E	E56D	07346	E9CE
86A3	694F	E56E	07347	E9CF
86A9	6950	E56F	07348	E9D0
86AA	6951	E570	07349	E9D1
868B	6952	E571	07350	E9D2
868C	6953	E572	07351	E9D3
86B6	6954	E573	07352	E9D4
86AF	6955	E574	07353	E9D5
86C4	6956	E575	07354	E9D6
86C6	6957	E576	07355	E9D7
86B0	6958	E577	07356	E9D8
86C9	6959	E578	07357	E9D9
86CE	3342	8A61	01934	E9DA
86AB	695B	E57A	07359	E9DB
86D4	695C	E57B	07360	E9DC
86DE	695D	E57C	07361	E9DD
E3C8				E9DE
86EC	695F	E57E	07363	E9DF
E3C9				E9E0
86DB	6961	E581	07365	E9E1
86EF	6962	E582	07366	E9E2
E7AB				E9E3
8706	6964	E584	07368	E9E4
8708	6965	E585	07369	E9E5
8700	6966	E586	07370	E9E6
8703	6967	E587	07371	E9E7
86FB	6968	E588	07372	E9E8
E7AA				E9E9
8709	696A	E58A	07374	E9EA
870D	696B	E58B	07375	E9EB
86F9	696C	E58C	07376	E9EC
870A	696D	E58D	07377	E9ED
8734	696E	E58E	07378	E9EE
873F	696F	E58F	07379	E9EF
8737	6970	E590	07380	E9F0
873B	6971	E591	07381	E9F1
8725	6972	E592	07382	E9F2
8729	6973	E593	07383	E9F3
E3CB				E9F4
8760	6975	E595	07385	E9F5
875F	6976	E596	07386	E9F6
8778	6977	E597	07387	E9F7
874C	6978	E598	07388	E9F8
874E	6979	E599	07389	E9F9
8774	697A	E59A	07390	E9FA
8757	697B	E59B	07391	E9FB
8768	697C	E59C	07392	E9FC
876E	697D	E59D	07393	E9FD
E3CF				E9FE
8753	6A21	E59F	07401	EAA1
8763	6A22	E5A0	07402	EAA2
876A	6A23	E5A1	07403	EAA3
8805	6A24	E5A2	07404	EAA4
87A2	6A25	E5A3	07405	EAA5
879F	6A26	E5A4	07406	EAA6
8782	6A27	E5A5	07407	EAA7
87AF	6A28	E5A6	07408	EAA8
87CB	6A29	E5A7	07409	EAA9
87BD	6A2A	E5A8	07410	EAAA
87C0	6A2B	E5A9	07411	EAAB
87D0	6A2C	E5AA	07412	EAAC
96D6	6A2D	E5AB	07413	EAAD
87AB	6A2E	E5AC	07414	EAAE
E3D1				EAAF
87B3	6A30	E5AE	07416	EAB0
87C7	6A31	E5AF	07417	EAB1
87C6	6A32	E5B0	07418	EAB2
87BB	6A33	E5B1	07419	EAB3
87EF	6A34	E5B2	07420	EAB4
87F2	6A35	E5B3	07421	EAB5
87E0	6A36	E5B4	07422	EAB6
880F	6A37	E5B5	07423	EAB7
880D	6A38	E5B6	07424	EAB8
87FE	6A39	E5B7	07425	EAB9
87F6	6A3A	E5B8	07426	EABA
87F7	6A3B	E5B9	07427	EABB
880E	6A3C	E5BA	07428	EABC
E3D2				EABD
8811	6A3E	E5BC	07430	EABE
8816	6A3F	E5BD	07431	EABF
8815	6A40	E5BE	07432	EAC0
8822	6A41	E5BF	07433	EAC1
E3D4				EAC2
8831	6A43	E5C1	07435	EAC3
8836	6A44	E5C2	07436	EAC4
8839	6A45	E5C3	07437	EAC5
8827	6A46	E5C4	07438	EAC6
883B	6A47	E5C5	07439	EAC7
8844	6A48	E5C6	07440	EAC8
E3D5				EAC9
8852	6A4A	E5C8	07442	EACA
8859	6A4B	E5C9	07443	EACB
885E	6A4C	E5CA	07444	EACC
8862	6A4D	E5CB	07445	EACD
886B	6A4E	E5CC	07446	EACE
8881	6A4F	E5CD	07447	EACF
887E	6A50	E5CE	07448	EAD0
889E	6A51	E5CF	07449	EAD1
8875	6A52	E5D0	07450	EAD2
887D	6A53	E5D1	07451	EAD3
88B5	6A54	E5D2	07452	EAD4
8872	6A55	E5D3	07453	EAD5
8882	6A56	E5D4	07454	EAD6
8897	6A57	E5D5	07455	EAD7
8892	6A58	E5D6	07456	EAD8
88AE	6A59	E5D7	07457	EAD9
8899	6A5A	E5D8	07458	EADA
88A2	6A5B	E5D9	07459	EADB
888D	6A5C	E5DA	07460	EADC
88A4	6A5D	E5DB	07461	EADD
88B0	6A5E	E5DC	07462	EADE
88BF	6A5F	E5DD	07463	EADF
88B1	6A60	E5DE	07464	EAE0
88C3	6A61	E5DF	07465	EAE1
88C4	6A62	E5E0	07466	EAE2
88D4	6A63	E5E1	07467	EAE3
E7AC				EAE4
88D9	6A65	E5E3	07469	EAE5
88DD	6A66	E5E4	07470	EAE6
88F9	6A67	E5E5	07471	EAE7
8902	6A68	E5E6	07472	EAE8
88FC	6A69	E5E7	07473	EAE9
E3D7				EAEA
88E8	6A6B	E5E9	07475	EAEB
88F2	6A6C	E5EA	07476	EAEC
8904	6A6D	E5EB	07477	EAED
890C	6A6E	E5EC	07478	EAEE
E3D8				EAEF
8913	6A70	E5EE	07480	EAF0
8943	6A71	E5EF	07481	EAF1
891E	6A72	E5F0	07482	EAF2
8925	6A73	E5F1	07483	EAF3
892A	6A74	E5F2	07484	EAF4
892B	6A75	E5F3	07485	EAF5
8941	6A76	E5F4	07486	EAF6
8944	6A77	E5F5	07487	EAF7
893B	6A78	E5F6	07488	EAF8
8936	6A79	E5F7	07489	EAF9
8938	6A7A	E5F8	07490	EAFA
894C	6A7B	E5F9	07491	EAFB
891D	6A7C	E5FA	07492	EAFC
8960	6A7D	E5FB	07493	EAFD
895E	6A7E	E5FC	07494	EAFE
8966	6B21	E640	07501	EBA1
8964	6B22	E641	07502	EBA2
896D	6B23	E642	07503	EBA3
896A	6B24	E643	07504	EBA4
896F	6B25	E644	07505	EBA5
8974	6B26	E645	07506	EBA6
8977	6B27	E646	07507	EBA7
897E	6B28	E647	07508	EBA8
8983	6B29	E648	07509	EBA9
E3DF				EBAA
898A	6B2B	E64A	07511	EBAB
8993	6B2C	E64B	07512	EBAC
8998	6B2D	E64C	07513	EBAD
89A1	6B2E	E64D	07514	EBAE
89A9	6B2F	E64E	07515	EBAF
89A6	6B30	E64F	07516	EBB0
89AC	6B31	E650	07517	EBB1
E3E2				EBB2
89B2	6B33	E652	07519	EBB3
89BA	6B34	E653	07520	EBB4
89BD	6B35	E654	07521	EBB5
89BF	6B36	E655	07522	EBB6
89C0	6B37	E656	07523	EBB7
89DA	6B38	E657	07524	EBB8
89DC	6B39	E658	07525	EBB9
89DD	6B3A	E659	07526	EBBA
89E7	6B3B	E65A	07527	EBBB
89F4	6B3C	E65B	07528	EBBC
89F8	6B3D	E65C	07529	EBBD
8A03	6B3E	E65D	07530	EBBE
8A16	6B3F	E65E	07531	EBBF
8A10	6B40	E65F	07532	EBC0
8A0C	6B41	E660	07533	EBC1
8A1B	6B42	E661	07534	EBC2
E3E4				EBC3
8A25	6B44	E663	07536	EBC4
8A36	6B45	E664	07537	EBC5
8A41	6B46	E665	07538	EBC6
8A5B	6B47	E666	07539	EBC7
8A52	6B48	E667	07540	EBC8
8A46	6B49	E668	07541	EBC9
8A48	6B4A	E669	07542	EBCA
E7AD				EBCB
8A6D	6B4C	E66B	07544	EBCC
8A6C	6B4D	E66C	07545	EBCD
8A62	6B4E	E66D	07546	EBCE
8A85	6B4F	E66E	07547	EBCF
8A82	6B50	E66F	07548	EBD0
8A84	6B51	E670	07549	EBD1
8AA8	6B52	E671	07550	EBD2
8AA1	6B53	E672	07551	EBD3
8A91	6B54	E673	07552	EBD4
8AA5	6B55	E674	07553	EBD5
8AA6	6B56	E675	07554	EBD6
8A9A	6B57	E676	07555	EBD7
8AA3	6B58	E677	07556	EBD8
8AC4	6B59	E678	07557	EBD9
8ACD	6B5A	E679	07558	EBDA
8AC2	6B5B	E67A	07559	EBDB
8ADA	6B5C	E67B	07560	EBDC
E735				EBDD
8AF3	6B5E	E67D	07562	EBDE
8AE7	6B5F	E67E	07563	EBDF
8AE4	6B60	E680	07564	EBE0
8AF1	6B61	E681	07565	EBE1
8B14	6B62	E682	07566	EBE2
8AE0	6B63	E683	07567	EBE3
8AE2	6B64	E684	07568	EBE4
8AF7	6B65	E685	07569	EBE5
E3F2				EBE6
8ADB	6B67	E687	07571	EBE7
8B0C	6B68	E688	07572	EBE8
8B07	6B69	E689	07573	EBE9
8B1A	6B6A	E68A	07574	EBEA
8AE1	6B6B	E68B	07575	EBEB
8B16	6B6C	E68C	07576	EBEC
8B10	6B6D	E68D	07577	EBED
8B17	6B6E	E68E	07578	EBEE
8B20	6B6F	E68F	07579	EBEF
8B33	6B70	E690	07580	EBF0
97AB	6B71	E691	07581	EBF1
8B26	6B72	E692	07582	EBF2
8B2B	6B73	E693	07583	EBF3
8B3E	6B74	E694	07584	EBF4
8B28	6B75	E695	07585	EBF5
8B41	6B76	E696	07586	EBF6
8B4C	6B77	E697	07587	EBF7
8B4F	6B78	E698	07588	EBF8
8B4E	6B79	E699	07589	EBF9
8B49	6B7A	E69A	07590	EBFA
8B56	6B7B	E69B	07591	EBFB
8B5B	6B7C	E69C	07592	EBFC
8B5A	6B7D	E69D	07593	EBFD
E737				EBFE
8B5F	6C21	E69F	07601	ECA1
8B6C	6C22	E6A0	07602	ECA2
8B6F	6C23	E6A1	07603	ECA3
8B74	6C24	E6A2	07604	ECA4
8B7D	6C25	E6A3	07605	ECA5
8B80	6C26	E6A4	07606	ECA6
8B8C	6C27	E6A5	07607	ECA7
8B8E	6C28	E6A6	07608	ECA8
8B92	6C29	E6A7	07609	ECA9
8B93	6C2A	E6A8	07610	ECAA
8B96	6C2B	E6A9	07611	ECAB
8B99	6C2C	E6AA	07612	ECAC
8B9A	6C2D	E6AB	07613	ECAD
E3FA				ECAE
E3FB				ECAF
8C3F	6C30	E6AE	07616	ECB0
8C48	6C31	E6AF	07617	ECB1
8C4C	6C32	E6B0	07618	ECB2
8C4E	6C33	E6B1	07619	ECB3
8C50	6C34	E6B2	07620	ECB4
E3FC				ECB5
8C62	6C36	E6B4	07622	ECB6
8C6C	6C37	E6B5	07623	ECB7
8C78	6C38	E6B6	07624	ECB8
8C7A	6C39	E6B7	07625	ECB9
8C82	6C3A	E6B8	07626	ECBA
8C89	6C3B	E6B9	07627	ECBB
8C85	6C3C	E6BA	07628	ECBC
8C8A	6C3D	E6BB	07629	ECBD
8C8D	6C3E	E6BC	07630	ECBE
8C8E	6C3F	E6BD	07631	ECBF
8C94	6C40	E6BE	07632	ECC0
8C7C	6C41	E6BF	07633	ECC1
8C98	6C42	E6C0	07634	ECC2
621D	6C43	E6C1	07635	ECC3
8CAD	6C44	E6C2	07636	ECC4
8CAA	6C45	E6C3	07637	ECC5
8CBD	6C46	E6C4	07638	ECC6
8CB2	6C47	E6C5	07639	ECC7
8CB3	6C48	E6C6	07640	ECC8
8CAE	6C49	E6C7	07641	ECC9
8CB6	6C4A	E6C8	07642	ECCA
8CC8	6C4B	E6C9	07643	ECCB
8CC1	6C4C	E6CA	07644	ECCC
8CCE	4128	9147	03308	ECCD
8CE3	6C4E	E6CC	07646	ECCE
8CDA	6C4F	E6CD	07647	ECCF
8CFD	6C50	E6CE	07648	ECD0
8CFA	6C51	E6CF	07649	ECD1
8CFB	6C52	E6D0	07650	ECD2
8D04	6C53	E6D1	07651	ECD3
8D05	6C54	E6D2	07652	ECD4
8D0A	6C55	E6D3	07653	ECD5
8D07	6C56	E6D4	07654	ECD6
E409				ECD7
8D0D	6C58	E6D6	07656	ECD8
8D10	6C59	E6D7	07657	ECD9
9F4E	6C5A	E6D8	07658	ECDA
8D13	6C5B	E6D9	07659	ECDB
8CCD	6C5C	E6DA	07660	ECDC
8D14	6C5D	E6DB	07661	ECDD
8D16	6C5E	E6DC	07662	ECDE
8D67	6C5F	E6DD	07663	ECDF
8D6D	6C60	E6DE	07664	ECE0
8D71	6C61	E6DF	07665	ECE1
E7AF				ECE2
8D81	6C63	E6E1	07667	ECE3
8D99	6C64	E6E2	07668	ECE4
8DC2	6C65	E6E3	07669	ECE5
8DBE	6C66	E6E4	07670	ECE6
8DBA	6C67	E6E5	07671	ECE7
8DCF	6C68	E6E6	07672	ECE8
E40D				ECE9
8DD6	6C6A	E6E8	07674	ECEA
8DCC	6C6B	E6E9	07675	ECEB
8DDB	6C6C	E6EA	07676	ECEC
8DCB	6C6D	E6EB	07677	ECED
8DEA	6C6E	E6EC	07678	ECEE
8DEB	6C6F	E6ED	07679	ECEF
8DDF	6C70	E6EE	07680	ECF0
8DE3	6C71	E6EF	07681	ECF1
8DFC	6C72	E6F0	07682	ECF2
8E08	6C73	E6F1	07683	ECF3
8E09	6C74	E6F2	07684	ECF4
8DFF	6C75	E6F3	07685	ECF5
8E1D	6C76	E6F4	07686	ECF6
8E1E	6C77	E6F5	07687	ECF7
8E10	6C78	E6F6	07688	ECF8
8E1F	6C79	E6F7	07689	ECF9
8E42	6C7A	E6F8	07690	ECFA
8E35	6C7B	E6F9	07691	ECFB
8E30	6C7C	E6FA	07692	ECFC
8E34	6C7D	E6FB	07693	ECFD
8E4A	6C7E	E6FC	07694	ECFE
8E47	6D21	E740	07701	EDA1
8E49	6D22	E741	07702	EDA2
8E4C	6D23	E742	07703	EDA3
8E50	6D24	E743	07704	EDA4
8E48	6D25	E744	07705	EDA5
8E59	6D26	E745	07706	EDA6
8E64	6D27	E746	07707	EDA7
8E60	6D28	E747	07708	EDA8
8E2A	6D29	E748	07709	EDA9
8E63	6D2A	E749	07710	EDAA
8E55	6D2B	E74A	07711	EDAB
8E76	6D2C	E74B	07712	EDAC
8E72	6D2D	E74C	07713	EDAD
8E7C	6D2E	E74D	07714	EDAE
8E81	6D2F	E74E	07715	EDAF
8E87	6D30	E74F	07716	EDB0
8E85	6D31	E750	07717	EDB1
8E84	6D32	E751	07718	EDB2
8E8B	6D33	E752	07719	EDB3
8E8A	6D34	E753	07720	EDB4
8E93	6D35	E754	07721	EDB5
8E91	6D36	E755	07722	EDB6
8E94	6D37	E756	07723	EDB7
8E99	6D38	E757	07724	EDB8
8EAA	6D39	E758	07725	EDB9
E40F				EDBA
E410				EDBB
E412				EDBC
E414				EDBD
E7B0				EDBE
E413				EDBF
8EC5	6D40	E75F	07732	EDC0
8EC8	6D41	E760	07733	EDC1
8ECB	6D42	E761	07734	EDC2
8EDB	6D43	E762	07735	EDC3
8EE3	6D44	E763	07736	EDC4
8EFC	6D45	E764	07737	EDC5
8EFB	6D46	E765	07738	EDC6
8EEB	6D47	E766	07739	EDC7
8EFE	6D48	E767	07740	EDC8
8F0A	6D49	E768	07741	EDC9
8F05	6D4A	E769	07742	EDCA
8F15	6D4B	E76A	07743	EDCB
8F12	6D4C	E76B	07744	EDCC
8F19	6D4D	E76C	07745	EDCD
8F13	6D4E	E76D	07746	EDCE
8F1C	6D4F	E76E	07747	EDCF
8F1F	6D50	E76F	07748	EDD0
8F1B	6D51	E770	07749	EDD1
8F0C	6D52	E771	07750	EDD2
8F26	6D53	E772	07751	EDD3
8F33	6D54	E773	07752	EDD4
8F3B	6D55	E774	07753	EDD5
8F39	6D56	E775	07754	EDD6
8F45	6D57	E776	07755	EDD7
8F42	6D58	E777	07756	EDD8
8F3E	6D59	E778	07757	EDD9
8F4C	6D5A	E779	07758	EDDA
8F49	6D5B	E77A	07759	EDDB
8F46	6D5C	E77B	07760	EDDC
8F4E	6D5D	E77C	07761	EDDD
8F57	6D5E	E77D	07762	EDDE
8F5C	6D5F	E77E	07763	EDDF
8F62	6D60	E780	07764	EDE0
8F63	6D61	E781	07765	EDE1
8F64	6D62	E782	07766	EDE2
8F9C	6D63	E783	07767	EDE3
8F9F	6D64	E784	07768	EDE4
8FA3	6D65	E785	07769	EDE5
8FAD	6D66	E786	07770	EDE6
8FAF	6D67	E787	07771	EDE7
8FB7	6D68	E788	07772	EDE8
8FDA	6D69	E789	07773	EDE9
8FE5	6D6A	E78A	07774	EDEA
8FE2	6D6B	E78B	07775	EDEB
E425				EDEC
8FEF	6D6D	E78D	07777	EDED
E429				EDEE
8FF4	6D6F	E78F	07779	EDEF
9005	6D70	E790	07780	EDF0
8FF9	6D71	E791	07781	EDF1
8FFA	6D72	E792	07782	EDF2
9011	6D73	E793	07783	EDF3
9015	6D74	E794	07784	EDF4
E7B2				EDF5
900D	6D76	E796	07786	EDF6
901E	6D77	E797	07787	EDF7
9016	6D78	E798	07788	EDF8
900B	6D79	E799	07789	EDF9
9027	6D7A	E79A	07790	EDFA
9036	6D7B	E79B	07791	EDFB
9035	6D7C	E79C	07792	EDFC
9039	6D7D	E79D	07793	EDFD
8FF8	6D7E	E79E	07794	EDFE
904F	6E21	E79F	07801	EEA1
E7B3				EEA2
9051	6E23	E7A1	07803	EEA3
9052	6E24	E7A2	07804	EEA4
900E	6E25	E7A3	07805	EEA5
9049	6E26	E7A4	07806	EEA6
903E	6E27	E7A5	07807	EEA7
9056	6E28	E7A6	07808	EEA8
E44A				EEA9
905E	6E2A	E7A8	07810	EEAA
9068	6E2B	E7A9	07811	EEAB
E44F				EEAC
9076	6E2D	E7AB	07813	EEAD
96A8	6E2E	E7AC	07814	EEAE
9072	6E2F	E7AD	07815	EEAF
9082	6E30	E7AE	07816	EEB0
E458				EEB1
9081	6E32	E7B0	07818	EEB2
9080	6E33	E7B1	07819	EEB3
908A	6E34	E7B2	07820	EEB4
9089	6E35	E7B3	07821	EEB5
908F	6E36	E7B4	07822	EEB6
E461				EEB7
90AF	6E38	E7B6	07824	EEB8
90B1	6E39	E7B7	07825	EEB9
90B5	6E3A	E7B8	07826	EEBA
90E2	6E3B	E7B9	07827	EEBB
E463				EEBC
E462				EEBD
90DB	6E3E	E7BC	07830	EEBE
9102	6E3F	E7BD	07831	EEBF
9112	6E40	E7BE	07832	EEC0
9119	6E41	E7BF	07833	EEC1
9132	6E42	E7C0	07834	EEC2
9130	6E43	E7C1	07835	EEC3
914A	6E44	E7C2	07836	EEC4
9156	6E45	E7C3	07837	EEC5
9158	6E46	E7C4	07838	EEC6
9163	6E47	E7C5	07839	EEC7
9165	6E48	E7C6	07840	EEC8
9169	6E49	E7C7	07841	EEC9
9173	6E4A	E7C8	07842	EECA
9172	6E4B	E7C9	07843	EECB
918B	6E4C	E7CA	07844	EECC
9189	6E4D	E7CB	07845	EECD
9182	6E4E	E7CC	07846	EECE
91A2	6E4F	E7CD	07847	EECF
91AB	6E50	E7CE	07848	EED0
91AF	6E51	E7CF	07849	EED1
91AA	6E52	E7D0	07850	EED2
E470				EED3
91B4	6E54	E7D2	07852	EED4
91BA	6E55	E7D3	07853	EED5
91C0	6E56	E7D4	07854	EED6
91C1	6E57	E7D5	07855	EED7
91C9	6E58	E7D6	07856	EED8
91CB	6E59	E7D7	07857	EED9
91D0	6E5A	E7D8	07858	EEDA
91D6	6E5B	E7D9	07859	EEDB
91DF	6E5C	E7DA	07860	EEDC
91E1	6E5D	E7DB	07861	EEDD
91DB	6E5E	E7DC	07862	EEDE
E176				EEDF
E475				EEE0
91F6	6E61	E7DF	07865	EEE1
921E	6E62	E7E0	07866	EEE2
91FF	6E63	E7E1	07867	EEE3
9214	6E64	E7E2	07868	EEE4
922C	6E65	E7E3	07869	EEE5
9215	6E66	E7E4	07870	EEE6
9211	6E67	E7E5	07871	EEE7
925E	6E68	E7E6	07872	EEE8
9257	6E69	E7E7	07873	EEE9
9245	6E6A	E7E8	07874	EEEA
9249	6E6B	E7E9	07875	EEEB
9264	6E6C	E7EA	07876	EEEC
9248	6E6D	E7EB	07877	EEED
9295	6E6E	E7EC	07878	EEEE
923F	6E6F	E7ED	07879	EEEF
924B	6E70	E7EE	07880	EEF0
9250	6E71	E7EF	07881	EEF1
929C	6E72	E7F0	07882	EEF2
9296	6E73	E7F1	07883	EEF3
9293	6E74	E7F2	07884	EEF4
929B	6E75	E7F3	07885	EEF5
925A	6E76	E7F4	07886	EEF6
92CF	6E77	E7F5	07887	EEF7
92B9	6E78	E7F6	07888	EEF8
92B7	6E79	E7F7	07889	EEF9
92E9	6E7A	E7F8	07890	EEFA
930F	6E7B	E7F9	07891	EEFB
92FA	6E7C	E7FA	07892	EEFC
9344	6E7D	E7FB	07893	EEFD
932E	6E7E	E7FC	07894	EEFE
9319	6F21	E840	07901	EFA1
9322	6F22	E841	07902	EFA2
931A	6F23	E842	07903	EFA3
9323	6F24	E843	07904	EFA4
933A	6F25	E844	07905	EFA5
9335	6F26	E845	07906	EFA6
933B	6F27	E846	07907	EFA7
935C	6F28	E847	07908	EFA8
9360	6F29	E848	07909	EFA9
937C	6F2A	E849	07910	EFAA
936E	6F2B	E84A	07911	EFAB
9356	6F2C	E84B	07912	EFAC
93B0	6F2D	E84C	07913	EFAD
93AC	6F2E	E84D	07914	EFAE
93AD	6F2F	E84E	07915	EFAF
9394	6F30	E84F	07916	EFB0
93B9	6F31	E850	07917	EFB1
93D6	6F32	E851	07918	EFB2
93D7	6F33	E852	07919	EFB3
93E8	6F34	E853	07920	EFB4
93E5	6F35	E854	07921	EFB5
93D8	6F36	E855	07922	EFB6
93C3	6F37	E856	07923	EFB7
93DD	6F38	E857	07924	EFB8
93D0	6F39	E858	07925	EFB9
93C8	6F3A	E859	07926	EFBA
93E4	6F3B	E85A	07927	EFBB
941A	6F3C	E85B	07928	EFBC
9414	6F3D	E85C	07929	EFBD
9413	6F3E	E85D	07930	EFBE
9403	6F3F	E85E	07931	EFBF
9407	6F40	E85F	07932	EFC0
9410	6F41	E860	07933	EFC1
9436	6F42	E861	07934	EFC2
942B	6F43	E862	07935	EFC3
9435	6F44	E863	07936	EFC4
9421	6F45	E864	07937	EFC5
943A	6F46	E865	07938	EFC6
9441	6F47	E866	07939	EFC7
9452	6F48	E867	07940	EFC8
9444	6F49	E868	07941	EFC9
945B	6F4A	E869	07942	EFCA
9460	6F4B	E86A	07943	EFCB
9462	6F4C	E86B	07944	EFCC
945E	6F4D	E86C	07945	EFCD
946A	6F4E	E86D	07946	EFCE
9229	6F4F	E86E	07947	EFCF
9470	6F50	E86F	07948	EFD0
9475	6F51	E870	07949	EFD1
E489				EFD2
947D	6F53	E872	07951	EFD3
945A	6F54	E873	07952	EFD4
947C	6F55	E874	07953	EFD5
947E	6F56	E875	07954	EFD6
9481	6F57	E876	07955	EFD7
947F	6F58	E877	07956	EFD8
9582	6F59	E878	07957	EFD9
9587	6F5A	E879	07958	EFDA
958A	6F5B	E87A	07959	EFDB
E48D				EFDC
9596	6F5D	E87C	07961	EFDD
9598	6F5E	E87D	07962	EFDE
9599	6F5F	E87E	07963	EFDF
95A0	6F60	E880	07964	EFE0
95A8	6F61	E881	07965	EFE1
95A7	6F62	E882	07966	EFE2
95AD	6F63	E883	07967	EFE3
95BC	6F64	E884	07968	EFE4
95BB	6F65	E885	07969	EFE5
95B9	6F66	E886	07970	EFE6
95BE	6F67	E887	07971	EFE7
95CA	6F68	E888	07972	EFE8
6FF6	6F69	E889	07973	EFE9
95C3	6F6A	E88A	07974	EFEA
95CD	6F6B	E88B	07975	EFEB
95CC	6F6C	E88C	07976	EFEC
95D5	6F6D	E88D	07977	EFED
95D4	6F6E	E88E	07978	EFEE
95D6	6F6F	E88F	07979	EFEF
95DC	6F70	E890	07980	EFF0
95E1	6F71	E891	07981	EFF1
95E5	6F72	E892	07982	EFF2
95E2	6F73	E893	07983	EFF3
9621	6F74	E894	07984	EFF4
9628	6F75	E895	07985	EFF5
962E	6F76	E896	07986	EFF6
962F	6F77	E897	07987	EFF7
9642	6F78	E898	07988	EFF8
964C	6F79	E899	07989	EFF9
964F	6F7A	E89A	07990	EFFA
964B	6F7B	E89B	07991	EFFB
9677	6F7C	E89C	07992	EFFC
965C	6F7D	E89D	07993	EFFD
965E	6F7E	E89E	07994	EFFE
965D	7021	E89F	08001	F0A1
965F	7022	E8A0	08002	F0A2
9666	7023	E8A1	08003	F0A3
9672	7024	E8A2	08004	F0A4
966C	7025	E8A3	08005	F0A5
968D	7026	E8A4	08006	F0A6
9698	7027	E8A5	08007	F0A7
9695	7028	E8A6	08008	F0A8
9697	7029	E8A7	08009	F0A9
96AA	702A	E8A8	08010	F0AA
96A7	702B	E8A9	08011	F0AB
96B1	702C	E8AA	08012	F0AC
96B2	702D	E8AB	08013	F0AD
96B0	702E	E8AC	08014	F0AE
96B4	702F	E8AD	08015	F0AF
96B6	7030	E8AE	08016	F0B0
96B8	7031	E8AF	08017	F0B1
96B9	7032	E8B0	08018	F0B2
96CE	7033	E8B1	08019	F0B3
96CB	7034	E8B2	08020	F0B4
96C9	7035	E8B3	08021	F0B5
96CD	7036	E8B4	08022	F0B6
894D	7037	E8B5	08023	F0B7
96DC	7038	E8B6	08024	F0B8
970D	7039	E8B7	08025	F0B9
96D5	703A	E8B8	08026	F0BA
96F9	703B	E8B9	08027	F0BB
9704	703C	E8BA	08028	F0BC
E49C				F0BD
9708	703E	E8BC	08030	F0BE
9713	703F	E8BD	08031	F0BF
970E	7040	E8BE	08032	F0C0
9711	7041	E8BF	08033	F0C1
E7B4				F0C2
9716	7043	E8C1	08035	F0C3
9719	7044	E8C2	08036	F0C4
9724	7045	E8C3	08037	F0C5
972A	7046	E8C4	08038	F0C6
9730	7047	E8C5	08039	F0C7
9739	7048	E8C6	08040	F0C8
973D	7049	E8C7	08041	F0C9
973E	704A	E8C8	08042	F0CA
9744	704B	E8C9	08043	F0CB
9746	704C	E8CA	08044	F0CC
9748	704D	E8CB	08045	F0CD
9742	704E	E8CC	08046	F0CE
9749	704F	E8CD	08047	F0CF
975C	7050	E8CE	08048	F0D0
E49D				F0D1
9764	7052	E8D0	08050	F0D2
9766	7053	E8D1	08051	F0D3
9768	7054	E8D2	08052	F0D4
52D2	7055	E8D3	08053	F0D5
E4A0				F0D6
E49E				F0D7
9779	7058	E8D6	08056	F0D8
9785	7059	E8D7	08057	F0D9
977C	705A	E8D8	08058	F0DA
9781	705B	E8D9	08059	F0DB
977A	705C	E8DA	08060	F0DC
9786	705D	E8DB	08061	F0DD
978B	705E	E8DC	08062	F0DE
978F	705F	E8DD	08063	F0DF
9790	7060	E8DE	08064	F0E0
979C	7061	E8DF	08065	F0E1
97A8	7062	E8E0	08066	F0E2
97A6	7063	E8E1	08067	F0E3
97A3	7064	E8E2	08068	F0E4
97B3	7065	E8E3	08069	F0E5
97B4	7066	E8E4	08070	F0E6
97C3	7067	E8E5	08071	F0E7
97C6	7068	E8E6	08072	F0E8
97C8	7069	E8E7	08073	F0E9
97CB	706A	E8E8	08074	F0EA
97DC	706B	E8E9	08075	F0EB
97ED	706C	E8EA	08076	F0EC
9F4F	706D	E8EB	08077	F0ED
97F2	706E	E8EC	08078	F0EE
7ADF	706F	E8ED	08079	F0EF
97F6	7070	E8EE	08080	F0F0
97F5	7071	E8EF	08081	F0F1
980F	7072	E8F0	08082	F0F2
E4AD				F0F3
E7B5				F0F4
E7B6				F0F5
9821	7076	E8F4	08086	F0F6
9837	7077	E8F5	08087	F0F7
E770				F0F8
9846	7079	E8F7	08089	F0F9
984F	707A	E8F8	08090	F0FA
984B	707B	E8F9	08091	F0FB
986B	707C	E8FA	08092	F0FC
986F	707D	E8FB	08093	F0FD
9870	707E	E8FC	08094	F0FE
9871	7121	E940	08101	F1A1
9874	7122	E941	08102	F1A2
9873	7123	E942	08103	F1A3
98AA	7124	E943	08104	F1A4
98AF	7125	E944	08105	F1A5
98B1	7126	E945	08106	F1A6
98B6	7127	E946	08107	F1A7
98C4	7128	E947	08108	F1A8
98C3	7129	E948	08109	F1A9
98C6	712A	E949	08110	F1AA
98E9	712B	E94A	08111	F1AB
98EB	712C	E94B	08112	F1AC
E4B8				F1AD
9909	712E	E94D	08114	F1AE
9912	712F	E94E	08115	F1AF
9914	7130	E94F	08116	F1B0
9918	7131	E950	08117	F1B1
9921	7132	E951	08118	F1B2
991D	7133	E952	08119	F1B3
991E	7134	E953	08120	F1B4
9924	7135	E954	08121	F1B5
9920	7136	E955	08122	F1B6
992C	7137	E956	08123	F1B7
992E	7138	E957	08124	F1B8
993D	7139	E958	08125	F1B9
993E	713A	E959	08126	F1BA
9942	713B	E95A	08127	F1BB
9949	713C	E95B	08128	F1BC
9945	713D	E95C	08129	F1BD
9950	713E	E95D	08130	F1BE
994B	713F	E95E	08131	F1BF
9951	7140	E95F	08132	F1C0
9952	7141	E960	08133	F1C1
994C	7142	E961	08134	F1C2
9955	7143	E962	08135	F1C3
9997	7144	E963	08136	F1C4
9998	7145	E964	08137	F1C5
99A5	7146	E965	08138	F1C6
99AD	7147	E966	08139	F1C7
99AE	7148	E967	08140	F1C8
E4C0				F1C9
99DF	714A	E969	08142	F1CA
E4C2				F1CB
99DD	714C	E96B	08144	F1CC
99D8	714D	E96C	08145	F1CD
99D1	714E	E96D	08146	F1CE
99ED	714F	E96E	08147	F1CF
E4C3				F1D0
99F1	7151	E970	08149	F1D1
99F2	7152	E971	08150	F1D2
99FB	7153	E972	08151	F1D3
99F8	7154	E973	08152	F1D4
9A01	7155	E974	08153	F1D5
9A0F	7156	E975	08154	F1D6
9A05	7157	E976	08155	F1D7
99E2	7158	E977	08156	F1D8
E4C4				F1D9
9A2B	715A	E979	08158	F1DA
9A37	715B	E97A	08159	F1DB
9A45	715C	E97B	08160	F1DC
9A42	715D	E97C	08161	F1DD
9A40	715E	E97D	08162	F1DE
9A43	715F	E97E	08163	F1DF
9A3E	7160	E980	08164	F1E0
9A55	7161	E981	08165	F1E1
9A4D	7162	E982	08166	F1E2
9A5B	7163	E983	08167	F1E3
9A57	7164	E984	08168	F1E4
9A5F	7165	E985	08169	F1E5
9A62	7166	E986	08170	F1E6
9A65	7167	E987	08171	F1E7
9A64	7168	E988	08172	F1E8
9A69	7169	E989	08173	F1E9
9A6B	716A	E98A	08174	F1EA
9A6A	716B	E98B	08175	F1EB
9AAD	716C	E98C	08176	F1EC
9AB0	716D	E98D	08177	F1ED
9ABC	716E	E98E	08178	F1EE
9AC0	716F	E98F	08179	F1EF
9ACF	7170	E990	08180	F1F0
9AD1	7171	E991	08181	F1F1
9AD3	7172	E992	08182	F1F2
9AD4	7173	E993	08183	F1F3
9ADE	7174	E994	08184	F1F4
9ADF	7175	E995	08185	F1F5
9AE2	7176	E996	08186	F1F6
9AE3	7177	E997	08187	F1F7
9AE6	7178	E998	08188	F1F8
9AEF	7179	E999	08189	F1F9
9AEB	717A	E99A	08190	F1FA
9AEE	717B	E99B	08191	F1FB
9AF4	717C	E99C	08192	F1FC
9AF1	717D	E99D	08193	F1FD
9AF7	717E	E99E	08194	F1FE
9AFB	7221	E99F	08201	F2A1
9B06	7222	E9A0	08202	F2A2
9B18	7223	E9A1	08203	F2A3
9B1A	7224	E9A2	08204	F2A4
9B1F	7225	E9A3	08205	F2A5
9B22	7226	E9A4	08206	F2A6
9B23	7227	E9A5	08207	F2A7
9B25	7228	E9A6	08208	F2A8
9B27	7229	E9A7	08209	F2A9
9B28	722A	E9A8	08210	F2AA
9B29	722B	E9A9	08211	F2AB
9B2A	722C	E9AA	08212	F2AC
9B2E	722D	E9AB	08213	F2AD
9B2F	722E	E9AC	08214	F2AE
9B32	722F	E9AD	08215	F2AF
9B44	7230	E9AE	08216	F2B0
9B43	7231	E9AF	08217	F2B1
9B4F	7232	E9B0	08218	F2B2
E4C9				F2B3
9B4E	7234	E9B2	08220	F2B4
9B51	7235	E9B3	08221	F2B5
9B58	7236	E9B4	08222	F2B6
9B74	7237	E9B5	08223	F2B7
9B93	7238	E9B6	08224	F2B8
9B83	7239	E9B7	08225	F2B9
9B91	723A	E9B8	08226	F2BA
9B96	723B	E9B9	08227	F2BB
E4CB				F2BC
9B9F	723D	E9BB	08229	F2BD
E4CD				F2BE
9BA8	723F	E9BD	08231	F2BF
9BB4	7240	E9BE	08232	F2C0
9BC0	7241	E9BF	08233	F2C1
9BCA	7242	E9C0	08234	F2C2
9BB9	7243	E9C1	08235	F2C3
9BC6	7244	E9C2	08236	F2C4
9BCF	7245	E9C3	08237	F2C5
9BD1	7246	E9C4	08238	F2C6
9BD2	7247	E9C5	08239	F2C7
9BE3	7248	E9C6	08240	F2C8
9BE2	7249	E9C7	08241	F2C9
9BE4	724A	E9C8	08242	F2CA
9BD4	724B	E9C9	08243	F2CB
E4D3				F2CC
9BF5	3033	88B1	01619	F2CD
E4D2				F2CE
9BF1	724F	E9CD	08247	F2CF
9BF0	7250	E9CE	08248	F2D0
9C15	7251	E9CF	08249	F2D1
9C14	7252	E9D0	08250	F2D2
9C09	7253	E9D1	08251	F2D3
9C13	7254	E9D2	08252	F2D4
9C0C	7255	E9D3	08253	F2D5
9C06	7256	E9D4	08254	F2D6
9C08	7257	E9D5	08255	F2D7
9C12	7258	E9D6	08256	F2D8
9C0A	7259	E9D7	08257	F2D9
9C04	725A	E9D8	08258	F2DA
9C2E	725B	E9D9	08259	F2DB
9C1B	725C	E9DA	08260	F2DC
9C25	725D	E9DB	08261	F2DD
9C24	725E	E9DC	08262	F2DE
9C21	725F	E9DD	08263	F2DF
9C30	7260	E9DE	08264	F2E0
9C47	7261	E9DF	08265	F2E1
9C32	7262	E9E0	08266	F2E2
9C46	7263	E9E1	08267	F2E3
9C3E	7264	E9E2	08268	F2E4
9C5A	7265	E9E3	08269	F2E5
9C60	7266	E9E4	08270	F2E6
9C67	7267	E9E5	08271	F2E7
9C76	7268	E9E6	08272	F2E8
9C78	7269	E9E7	08273	F2E9
9CE7	726A	E9E8	08274	F2EA
9CEC	726B	E9E9	08275	F2EB
9CF0	726C	E9EA	08276	F2EC
E4D9				F2ED
9D08	726E	E9EC	08278	F2EE
9CEB	726F	E9ED	08279	F2EF
9D03	7270	E9EE	08280	F2F0
9D06	7271	E9EF	08281	F2F1
9D2A	7272	E9F0	08282	F2F2
9D26	7273	E9F1	08283	F2F3
9D2C	3229	89A7	01809	F2F4
9D23	7275	E9F3	08285	F2F5
9D1F	7276	E9F4	08286	F2F6
9D44	7277	E9F5	08287	F2F7
9D15	7278	E9F6	08288	F2F8
9D12	7279	E9F7	08289	F2F9
9D41	727A	E9F8	08290	F2FA
9D3F	727B	E9F9	08291	F2FB
9D3E	727C	E9FA	08292	F2FC
9D46	727D	E9FB	08293	F2FD
9D48	727E	E9FC	08294	F2FE
9D5D	7321	EA40	08301	F3A1
9D5E	7322	EA41	08302	F3A2
9D64	7323	EA42	08303	F3A3
9D51	7324	EA43	08304	F3A4
9D50	7325	EA44	08305	F3A5
9D59	7326	EA45	08306	F3A6
9D72	7327	EA46	08307	F3A7
9D89	7328	EA47	08308	F3A8
9D87	7329	EA48	08309	F3A9
9DAB	732A	EA49	08310	F3AA
9D6F	732B	EA4A	08311	F3AB
9D7A	732C	EA4B	08312	F3AC
9D9A	732D	EA4C	08313	F3AD
9DA4	732E	EA4D	08314	F3AE
9DA9	732F	EA4E	08315	F3AF
9DB2	7330	EA4F	08316	F3B0
9DC4	7331	EA50	08317	F3B1
9DC1	7332	EA51	08318	F3B2
9DBB	7333	EA52	08319	F3B3
9DB8	7334	EA53	08320	F3B4
9DBA	7335	EA54	08321	F3B5
9DC6	7336	EA55	08322	F3B6
9DCF	7337	EA56	08323	F3B7
9DC2	7338	EA57	08324	F3B8
E7B7				F3B9
9DD3	733A	EA59	08326	F3BA
9DF8	733B	EA5A	08327	F3BB
9DE6	733C	EA5B	08328	F3BC
9DED	733D	EA5C	08329	F3BD
9DEF	733E	EA5D	08330	F3BE
9DFD	733F	EA5E	08331	F3BF
9E1A	7340	EA5F	08332	F3C0
9E1B	7341	EA60	08333	F3C1
9E1E	7342	EA61	08334	F3C2
9E75	7343	EA62	08335	F3C3
9E79	7344	EA63	08336	F3C4
9E7D	7345	EA64	08337	F3C5
9E81	7346	EA65	08338	F3C6
9E88	7347	EA66	08339	F3C7
9E8B	7348	EA67	08340	F3C8
9E8C	7349	EA68	08341	F3C9
9E92	734A	EA69	08342	F3CA
9E95	734B	EA6A	08343	F3CB
9E91	734C	EA6B	08344	F3CC
9E9D	734D	EA6C	08345	F3CD
9EA5	734E	EA6D	08346	F3CE
9EA9	734F	EA6E	08347	F3CF
E7B8				F3D0
E4DF				F3D1
E4E0				F3D2
9761	7353	EA72	08351	F3D3
9ECC	7354	EA73	08352	F3D4
9ECE	7355	EA74	08353	F3D5
9ECF	7356	EA75	08354	F3D6
9ED0	7357	EA76	08355	F3D7
9ED4	7358	EA77	08356	F3D8
9EDC	7359	EA78	08357	F3D9
9EDE	735A	EA79	08358	F3DA
9EDD	735B	EA7A	08359	F3DB
9EE0	735C	EA7B	08360	F3DC
9EE5	735D	EA7C	08361	F3DD
9EE8	735E	EA7D	08362	F3DE
9EEF	735F	EA7E	08363	F3DF
9EF4	7360	EA80	08364	F3E0
9EF6	7361	EA81	08365	F3E1
9EF7	7362	EA82	08366	F3E2
9EF9	7363	EA83	08367	F3E3
9EFB	7364	EA84	08368	F3E4
9EFC	7365	EA85	08369	F3E5
9EFD	7366	EA86	08370	F3E6
9F07	7367	EA87	08371	F3E7
9F08	7368	EA88	08372	F3E8
76B7	7369	EA89	08373	F3E9
9F15	736A	EA8A	08374	F3EA
9F21	736B	EA8B	08375	F3EB
9F2C	736C	EA8C	08376	F3EC
9F3E	736D	EA8D	08377	F3ED
9F4A	736E	EA8E	08378	F3EE
9F52	736F	EA8F	08379	F3EF
9F54	7370	EA90	08380	F3F0
9F63	7371	EA91	08381	F3F1
9F5F	7372	EA92	08382	F3F2
9F60	7373	EA93	08383	F3F3
9F61	7374	EA94	08384	F3F4
9F66	7375	EA95	08385	F3F5
9F67	7376	EA96	08386	F3F6
9F6C	7377	EA97	08387	F3F7
9F6A	7378	EA98	08388	F3F8
9F77	7379	EA99	08389	F3F9
9F72	737A	EA9A	08390	F3FA
9F76	737B	EA9B	08391	F3FB
9F95	737C	EA9C	08392	F3FC
9F9C	737D	EA9D	08393	F3FD
9FA0	737E	EA9E	08394	F3FE
