/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2012  Intel Corporation. All rights reserved.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <unistd.h>
#include <stdlib.h>
#include <stdbool.h>
#include <sys/socket.h>

#include <glib.h>

#include "lib/bluetooth.h"
#include "lib/sdp.h"
#include "lib/sdp_lib.h"

#include "src/shared/util.h"
#include "src/sdpd.h"

struct sdp_pdu {
	bool valid;
	const void *raw_data;
	size_t raw_size;
	uint8_t cont_len;
};

struct test_data {
	int mtu;
	struct sdp_pdu *pdu_list;
};

#define raw_data(args...) ((const unsigned char[]) { args })
#define build_u128(args...) ((const uint128_t) { .data = { args } })

#define raw_pdu(args...) \
	{							\
		.valid = true,					\
		.raw_data = raw_data(args),			\
		.raw_size = sizeof(raw_data(args)),		\
	}

#define raw_pdu_cont(cont, args...) \
	{							\
		.valid = true,					\
		.raw_data = raw_data(args),			\
		.raw_size = sizeof(raw_data(args)),		\
		.cont_len = cont,				\
	}

#define define_test(name, _mtu, args...) \
	do {								\
		const struct sdp_pdu pdus[] = {				\
			args, { }, { }					\
		};							\
		static struct test_data data;				\
		data.mtu = _mtu;					\
		data.pdu_list = g_malloc(sizeof(pdus));			\
		memcpy(data.pdu_list, pdus, sizeof(pdus));		\
		g_test_add_data_func(name, &data, test_sdp);		\
	} while (0)

#define define_ss(name, args...) define_test("/TP/SERVER/SS/" name, 48, args)
#define define_sa(name, args...) define_test("/TP/SERVER/SA/" name, 48, args)
#define define_ssa(name, args...) define_test("/TP/SERVER/SSA/" name, 48, args)
#define define_brw(name, args...) define_test("/TP/SERVER/BRW/" name, 672, args)

/* SDP Data Element (DE) tests */
struct test_data_de {
	const void *input_data;
	size_t input_size;
	sdp_data_t expected;
};

#define exp_data(_dtd, val_type, val_data) \
	((const sdp_data_t) {			\
		.dtd = _dtd,			\
		.val.val_type = val_data,	\
	})

#define define_test_de_attr(name, input, exp) \
	do {								\
		static struct test_data_de data;			\
		data.input_data = input;				\
		data.input_size = sizeof(input);			\
		data.expected = exp;					\
		g_test_add_data_func("/sdp/DE/ATTR/" name, &data,	\
						test_sdp_de_attr);	\
	} while (0)

struct context {
	GMainLoop *main_loop;
	guint server_source;
	guint client_source;
	int fd;
	int mtu;
	uint8_t cont_data[16];
	uint8_t cont_size;
	unsigned int pdu_offset;
	const struct sdp_pdu *pdu_list;
};

static void sdp_debug(const char *str, void *user_data)
{
	const char *prefix = user_data;

	g_print("%s%s\n", prefix, str);
}

void btd_debug(const char *format, ...);

void btd_debug(const char *format, ...)
{
}

static void context_quit(struct context *context)
{
	g_main_loop_quit(context->main_loop);
}

static gboolean server_handler(GIOChannel *channel, GIOCondition cond,
							gpointer user_data)
{
	struct context *context = user_data;
	sdp_pdu_hdr_t hdr;
	void *buf;
	size_t size;
	ssize_t len;
	int fd;

	fd = g_io_channel_unix_get_fd(channel);

	if (cond & (G_IO_NVAL | G_IO_ERR | G_IO_HUP)) {
		sdp_svcdb_collect_all(fd);
		return FALSE;
	}

	len = recv(fd, &hdr, sizeof(sdp_pdu_hdr_t), MSG_PEEK);
	if (len != sizeof(sdp_pdu_hdr_t)) {
		sdp_svcdb_collect_all(fd);
		return FALSE;
	}

	size = sizeof(sdp_pdu_hdr_t) + ntohs(hdr.plen);

	buf = malloc(size);
	if (!buf)
		return TRUE;

	len = recv(fd, buf, size, 0);
	if (len <= 0) {
		sdp_svcdb_collect_all(fd);
		free(buf);
		return FALSE;
	}

	if (g_test_verbose() == TRUE)
		util_hexdump('<', buf, len, sdp_debug, "SDP: ");

	handle_internal_request(fd, context->mtu, buf, len);

	return TRUE;
}

static gboolean send_pdu(gpointer user_data)
{
	struct context *context = user_data;
	const struct sdp_pdu *req_pdu;
	uint16_t pdu_len;
	unsigned char *buf;
	ssize_t len;

	req_pdu = &context->pdu_list[context->pdu_offset];

	pdu_len = req_pdu->raw_size + context->cont_size;

	buf = g_malloc0(pdu_len);

	memcpy(buf, req_pdu->raw_data, req_pdu->raw_size);

	if (context->cont_size > 0)
		memcpy(buf + req_pdu->raw_size, context->cont_data,
							context->cont_size);

	len = write(context->fd, buf, pdu_len);

	g_free(buf);

	g_assert(len == pdu_len);

	return FALSE;
}

static void context_increment(struct context *context)
{
	context->pdu_offset += 2;

	if (!context->pdu_list[context->pdu_offset].valid) {
		context_quit(context);
		return;
	}

	g_idle_add(send_pdu, context);
}

static gboolean client_handler(GIOChannel *channel, GIOCondition cond,
							gpointer user_data)
{
	struct context *context = user_data;
	const struct sdp_pdu *rsp_pdu;
	unsigned char buf[512];
	ssize_t len;
	int fd;

	rsp_pdu = &context->pdu_list[context->pdu_offset + 1];

	if (cond & (G_IO_NVAL | G_IO_ERR | G_IO_HUP))
		return FALSE;

	fd = g_io_channel_unix_get_fd(channel);

	len = read(fd, buf, sizeof(buf));
	if (len < 0)
		return FALSE;

	if (g_test_verbose() == TRUE)
		util_hexdump('>', buf, len, sdp_debug, "SDP: ");

	g_assert(len > 0);
	g_assert((size_t) len == rsp_pdu->raw_size + rsp_pdu->cont_len);

	g_assert(memcmp(buf, rsp_pdu->raw_data,	rsp_pdu->raw_size) == 0);

	if (rsp_pdu->cont_len > 0)
		memcpy(context->cont_data, buf + rsp_pdu->raw_size,
							rsp_pdu->cont_len);

	context->cont_size = rsp_pdu->cont_len;

	context_increment(context);

	return TRUE;
}

static void update_db_timestamp(void)
{
}

static void register_serial_port(void)
{
	sdp_list_t *svclass_id, *apseq, *proto[2], *profiles, *root, *aproto;
	uuid_t root_uuid, sp_uuid, l2cap, rfcomm;
	sdp_profile_desc_t profile;
	uint8_t u8 = 1;
	sdp_data_t *sdp_data, *channel;
	sdp_record_t *record = sdp_record_alloc();

	record->handle = sdp_next_handle();

	sdp_record_add(BDADDR_ANY, record);
	sdp_data = sdp_data_alloc(SDP_UINT32, &record->handle);
	sdp_attr_add(record, SDP_ATTR_RECORD_HANDLE, sdp_data);

	sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
	root = sdp_list_append(0, &root_uuid);
	sdp_set_browse_groups(record, root);
	sdp_list_free(root, 0);

	sdp_uuid16_create(&sp_uuid, SERIAL_PORT_SVCLASS_ID);
	svclass_id = sdp_list_append(0, &sp_uuid);
	sdp_set_service_classes(record, svclass_id);
	sdp_list_free(svclass_id, 0);

	sdp_uuid16_create(&profile.uuid, SERIAL_PORT_PROFILE_ID);
	profile.version = 0x0100;
	profiles = sdp_list_append(0, &profile);
	sdp_set_profile_descs(record, profiles);
	sdp_list_free(profiles, 0);

	sdp_uuid16_create(&l2cap, L2CAP_UUID);
	proto[0] = sdp_list_append(0, &l2cap);
	apseq = sdp_list_append(0, proto[0]);

	sdp_uuid16_create(&rfcomm, RFCOMM_UUID);
	proto[1] = sdp_list_append(0, &rfcomm);
	channel = sdp_data_alloc(SDP_UINT8, &u8);
	proto[1] = sdp_list_append(proto[1], channel);
	apseq = sdp_list_append(apseq, proto[1]);

	aproto = sdp_list_append(0, apseq);
	sdp_set_access_protos(record, aproto);

	sdp_add_lang_attr(record);

	sdp_set_info_attr(record, "Serial Port", "BlueZ", "COM Port");

	sdp_set_url_attr(record, "http://www.bluez.org/",
			"http://www.bluez.org/", "http://www.bluez.org/");

	sdp_set_service_id(record, sp_uuid);
	sdp_set_service_ttl(record, 0xffff);
	sdp_set_service_avail(record, 0xff);
	sdp_set_record_state(record, 0x00001234);

	sdp_data_free(channel);
	sdp_list_free(proto[0], 0);
	sdp_list_free(proto[1], 0);
	sdp_list_free(apseq, 0);
	sdp_list_free(aproto, 0);

	update_db_timestamp();
}

static void register_object_push(void)
{
	sdp_list_t *svclass_id, *pfseq, *apseq, *root;
	uuid_t root_uuid, opush_uuid, l2cap_uuid, rfcomm_uuid, obex_uuid;
	sdp_profile_desc_t profile[1];
	sdp_list_t *aproto, *proto[3];
	uint8_t chan = 9;
	sdp_data_t *channel;
	uint8_t formats[] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0xff };
	void *dtds[sizeof(formats)], *values[sizeof(formats)];
	unsigned int i;
	uint8_t dtd = SDP_UINT8;
	sdp_data_t *sdp_data, *sflist;
	sdp_record_t *record = sdp_record_alloc();

	record->handle = sdp_next_handle();

	sdp_record_add(BDADDR_ANY, record);
	sdp_data = sdp_data_alloc(SDP_UINT32, &record->handle);
	sdp_attr_add(record, SDP_ATTR_RECORD_HANDLE, sdp_data);

	sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
	root = sdp_list_append(0, &root_uuid);
	sdp_set_browse_groups(record, root);

	sdp_uuid16_create(&opush_uuid, OBEX_OBJPUSH_SVCLASS_ID);
	svclass_id = sdp_list_append(0, &opush_uuid);
	sdp_set_service_classes(record, svclass_id);

	sdp_uuid16_create(&profile[0].uuid, OBEX_OBJPUSH_PROFILE_ID);
	profile[0].version = 0x0100;
	pfseq = sdp_list_append(0, profile);
	sdp_set_profile_descs(record, pfseq);

	sdp_uuid16_create(&l2cap_uuid, L2CAP_UUID);
	proto[0] = sdp_list_append(0, &l2cap_uuid);
	apseq = sdp_list_append(0, proto[0]);

	sdp_uuid16_create(&rfcomm_uuid, RFCOMM_UUID);
	proto[1] = sdp_list_append(0, &rfcomm_uuid);
	channel = sdp_data_alloc(SDP_UINT8, &chan);
	proto[1] = sdp_list_append(proto[1], channel);
	apseq = sdp_list_append(apseq, proto[1]);

	sdp_uuid16_create(&obex_uuid, OBEX_UUID);
	proto[2] = sdp_list_append(0, &obex_uuid);
	apseq = sdp_list_append(apseq, proto[2]);

	aproto = sdp_list_append(0, apseq);
	sdp_set_access_protos(record, aproto);

	for (i = 0; i < sizeof(formats); i++) {
		dtds[i] = &dtd;
		values[i] = &formats[i];
	}
	sflist = sdp_seq_alloc(dtds, values, sizeof(formats));
	sdp_attr_add(record, SDP_ATTR_SUPPORTED_FORMATS_LIST, sflist);

	sdp_set_info_attr(record, "OBEX Object Push", 0, 0);

	sdp_data_free(channel);
	sdp_list_free(root, 0);
	sdp_list_free(svclass_id, 0);
	sdp_list_free(pfseq, 0);
	sdp_list_free(proto[0], 0);
	sdp_list_free(proto[1], 0);
	sdp_list_free(proto[2], 0);
	sdp_list_free(apseq, 0);
	sdp_list_free(aproto, 0);

	update_db_timestamp();
}

static void register_hid_keyboard(void)
{
	sdp_list_t *svclass_id, *pfseq, *apseq, *root;
	uuid_t root_uuid, hidkb_uuid, l2cap_uuid, hidp_uuid;
	sdp_profile_desc_t profile[1];
	sdp_list_t *aproto, *proto[3];
	sdp_data_t *psm, *lang_lst, *lang_lst2, *hid_spec_lst, *hid_spec_lst2;
	unsigned int i;
	uint8_t dtd = SDP_UINT16;
	uint8_t dtd2 = SDP_UINT8;
	uint8_t dtd_data = SDP_TEXT_STR8;
	void *dtds[2];
	void *values[2];
	void *dtds2[2];
	void *values2[2];
	int leng[2];
	uint8_t hid_spec_type = 0x22;
	uint16_t hid_attr_lang[] = { 0x409, 0x100 };
	static const uint16_t ctrl = 0x11;
	static const uint16_t intr = 0x13;
	static const uint16_t hid_attr[] = { 0x100, 0x111, 0x40, 0x0d,
								0x01, 0x01 };
	static const uint16_t hid_attr2[] = { 0x0, 0x01, 0x100, 0x1f40,
								0x01, 0x01 };
	const uint8_t hid_spec[] = {
		0x05, 0x01, // usage page
		0x09, 0x06, // keyboard
		0xa1, 0x01, // key codes
		0x85, 0x01, // minimum
		0x05, 0x07, // max
		0x19, 0xe0, // logical min
		0x29, 0xe7, // logical max
		0x15, 0x00, // report size
		0x25, 0x01, // report count
		0x75, 0x01, // input data variable absolute
		0x95, 0x08, // report count
		0x81, 0x02, // report size
		0x75, 0x08,
		0x95, 0x01,
		0x81, 0x01,
		0x75, 0x01,
		0x95, 0x05,
		0x05, 0x08,
		0x19, 0x01,
		0x29, 0x05,
		0x91, 0x02,
		0x75, 0x03,
		0x95, 0x01,
		0x91, 0x01,
		0x75, 0x08,
		0x95, 0x06,
		0x15, 0x00,
		0x26, 0xff,
		0x00, 0x05,
		0x07, 0x19,
		0x00, 0x2a,
		0xff, 0x00,
		0x81, 0x00,
		0x75, 0x01,
		0x95, 0x01,
		0x15, 0x00,
		0x25, 0x01,
		0x05, 0x0c,
		0x09, 0xb8,
		0x81, 0x06,
		0x09, 0xe2,
		0x81, 0x06,
		0x09, 0xe9,
		0x81, 0x02,
		0x09, 0xea,
		0x81, 0x02,
		0x75, 0x01,
		0x95, 0x04,
		0x81, 0x01,
		0xc0         // end tag
	};
	sdp_data_t *sdp_data;
	sdp_record_t *record = sdp_record_alloc();

	record->handle = sdp_next_handle();

	sdp_record_add(BDADDR_ANY, record);
	sdp_data = sdp_data_alloc(SDP_UINT32, &record->handle);
	sdp_attr_add(record, SDP_ATTR_RECORD_HANDLE, sdp_data);

	sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
	root = sdp_list_append(0, &root_uuid);
	sdp_set_browse_groups(record, root);
	sdp_list_free(root, 0);

	sdp_add_lang_attr(record);

	sdp_uuid16_create(&hidkb_uuid, HID_SVCLASS_ID);
	svclass_id = sdp_list_append(0, &hidkb_uuid);
	sdp_set_service_classes(record, svclass_id);
	sdp_list_free(svclass_id, 0);

	sdp_uuid16_create(&profile[0].uuid, HID_PROFILE_ID);
	profile[0].version = 0x0100;
	pfseq = sdp_list_append(0, profile);
	sdp_set_profile_descs(record, pfseq);
	sdp_list_free(pfseq, 0);

	/* protocols */
	sdp_uuid16_create(&l2cap_uuid, L2CAP_UUID);
	proto[1] = sdp_list_append(0, &l2cap_uuid);
	psm = sdp_data_alloc(SDP_UINT16, &ctrl);
	proto[1] = sdp_list_append(proto[1], psm);
	apseq = sdp_list_append(0, proto[1]);

	sdp_uuid16_create(&hidp_uuid, HIDP_UUID);
	proto[2] = sdp_list_append(0, &hidp_uuid);
	apseq = sdp_list_append(apseq, proto[2]);

	aproto = sdp_list_append(0, apseq);
	sdp_set_access_protos(record, aproto);

	sdp_data_free(psm);
	sdp_list_free(proto[1], 0);
	sdp_list_free(proto[2], 0);
	sdp_list_free(apseq, 0);
	sdp_list_free(aproto, 0);

	/* additional protocols */
	proto[1] = sdp_list_append(0, &l2cap_uuid);
	psm = sdp_data_alloc(SDP_UINT16, &intr);
	proto[1] = sdp_list_append(proto[1], psm);
	apseq = sdp_list_append(0, proto[1]);

	sdp_uuid16_create(&hidp_uuid, HIDP_UUID);
	proto[2] = sdp_list_append(0, &hidp_uuid);
	apseq = sdp_list_append(apseq, proto[2]);

	aproto = sdp_list_append(0, apseq);
	sdp_set_add_access_protos(record, aproto);

	sdp_data_free(psm);
	sdp_list_free(proto[1], 0);
	sdp_list_free(proto[2], 0);
	sdp_list_free(apseq, 0);
	sdp_list_free(aproto, 0);

	sdp_set_info_attr(record, "HID Keyboard", NULL, NULL);

	for (i = 0; i < sizeof(hid_attr) / 2; i++)
		sdp_attr_add_new(record,
				SDP_ATTR_HID_DEVICE_RELEASE_NUMBER + i,
				SDP_UINT16, &hid_attr[i]);

	dtds[0] = &dtd2;
	values[0] = &hid_spec_type;
	dtds[1] = &dtd_data;
	values[1] = (uint8_t *) hid_spec;
	leng[0] = 0;
	leng[1] = sizeof(hid_spec);
	hid_spec_lst = sdp_seq_alloc_with_length(dtds, values, leng, 2);
	hid_spec_lst2 = sdp_data_alloc(SDP_SEQ8, hid_spec_lst);
	sdp_attr_add(record, SDP_ATTR_HID_DESCRIPTOR_LIST, hid_spec_lst2);

	for (i = 0; i < sizeof(hid_attr_lang) / 2; i++) {
		dtds2[i] = &dtd;
		values2[i] = &hid_attr_lang[i];
	}

	lang_lst = sdp_seq_alloc(dtds2, values2, sizeof(hid_attr_lang) / 2);
	lang_lst2 = sdp_data_alloc(SDP_SEQ8, lang_lst);
	sdp_attr_add(record, SDP_ATTR_HID_LANG_ID_BASE_LIST, lang_lst2);

	sdp_attr_add_new(record, SDP_ATTR_HID_SDP_DISABLE,
						SDP_UINT16, &hid_attr2[0]);

	for (i = 0; i < sizeof(hid_attr2) / 2 - 1; i++)
		sdp_attr_add_new(record, SDP_ATTR_HID_REMOTE_WAKEUP + i,
						SDP_UINT16, &hid_attr2[i + 1]);

	update_db_timestamp();
}

static void register_file_transfer(void)
{
	sdp_list_t *svclass_id, *pfseq, *apseq, *root;
	uuid_t root_uuid, ftrn_uuid, l2cap_uuid, rfcomm_uuid, obex_uuid;
	sdp_profile_desc_t profile[1];
	sdp_list_t *aproto, *proto[3];
	uint8_t u8 = 10;
	sdp_data_t *sdp_data, *channel;
	sdp_record_t *record = sdp_record_alloc();

	record->handle = sdp_next_handle();

	sdp_record_add(BDADDR_ANY, record);
	sdp_data = sdp_data_alloc(SDP_UINT32, &record->handle);
	sdp_attr_add(record, SDP_ATTR_RECORD_HANDLE, sdp_data);

	sdp_uuid16_create(&root_uuid, PUBLIC_BROWSE_GROUP);
	root = sdp_list_append(0, &root_uuid);
	sdp_set_browse_groups(record, root);

	sdp_uuid16_create(&ftrn_uuid, OBEX_FILETRANS_SVCLASS_ID);
	svclass_id = sdp_list_append(0, &ftrn_uuid);
	sdp_set_service_classes(record, svclass_id);

	sdp_uuid16_create(&profile[0].uuid, OBEX_FILETRANS_PROFILE_ID);
	profile[0].version = 0x0100;
	pfseq = sdp_list_append(0, &profile[0]);
	sdp_set_profile_descs(record, pfseq);

	sdp_uuid16_create(&l2cap_uuid, L2CAP_UUID);
	proto[0] = sdp_list_append(0, &l2cap_uuid);
	apseq = sdp_list_append(0, proto[0]);

	sdp_uuid16_create(&rfcomm_uuid, RFCOMM_UUID);
	proto[1] = sdp_list_append(0, &rfcomm_uuid);
	channel = sdp_data_alloc(SDP_UINT8, &u8);
	proto[1] = sdp_list_append(proto[1], channel);
	apseq = sdp_list_append(apseq, proto[1]);

	sdp_uuid16_create(&obex_uuid, OBEX_UUID);
	proto[2] = sdp_list_append(0, &obex_uuid);
	apseq = sdp_list_append(apseq, proto[2]);

	aproto = sdp_list_append(0, apseq);
	sdp_set_access_protos(record, aproto);

	sdp_set_info_attr(record, "OBEX File Transfer", 0, 0);

	sdp_data_free(channel);
	sdp_list_free(root, 0);
	sdp_list_free(svclass_id, 0);
	sdp_list_free(pfseq, 0);
	sdp_list_free(proto[0], 0);
	sdp_list_free(proto[1], 0);
	sdp_list_free(proto[2], 0);
	sdp_list_free(apseq, 0);
	sdp_list_free(aproto, 0);

	update_db_timestamp();
}

static struct context *create_context(void)
{
	struct context *context = g_new0(struct context, 1);
	GIOChannel *channel;
	int err, sv[2];

	context->main_loop = g_main_loop_new(NULL, FALSE);
	g_assert(context->main_loop);

	err = socketpair(AF_UNIX, SOCK_SEQPACKET | SOCK_CLOEXEC, 0, sv);
	g_assert(err == 0);

	channel = g_io_channel_unix_new(sv[0]);

	g_io_channel_set_close_on_unref(channel, TRUE);
	g_io_channel_set_encoding(channel, NULL, NULL);
	g_io_channel_set_buffered(channel, FALSE);

	context->server_source = g_io_add_watch(channel,
				G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
				server_handler, context);
	g_assert(context->server_source > 0);

	g_io_channel_unref(channel);

	channel = g_io_channel_unix_new(sv[1]);

	g_io_channel_set_close_on_unref(channel, TRUE);
	g_io_channel_set_encoding(channel, NULL, NULL);
	g_io_channel_set_buffered(channel, FALSE);

	context->client_source = g_io_add_watch(channel,
				G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
				client_handler, context);
	g_assert(context->client_source > 0);

	g_io_channel_unref(channel);

	context->fd = sv[1];

	set_fixed_db_timestamp(0x496f0654);

	register_public_browse_group();
	register_server_service();

	register_serial_port();
	register_object_push();
	register_hid_keyboard();
	register_file_transfer();
	register_file_transfer();
	register_file_transfer();
	register_file_transfer();
	register_file_transfer();

	return context;
}

static void execute_context(struct context *context)
{
	g_main_loop_run(context->main_loop);

	sdp_svcdb_collect_all(context->fd);
	sdp_svcdb_reset();

	g_source_remove(context->server_source);
	g_source_remove(context->client_source);

	g_main_loop_unref(context->main_loop);

	g_free(context);
}

static void test_sdp(gconstpointer data)
{
	const struct test_data *test = data;
	struct context *context = create_context();

	context->mtu = test->mtu;
	context->pdu_list = test->pdu_list;

	g_idle_add(send_pdu, context);

	execute_context(context);

	g_free(test->pdu_list);
}

static void test_sdp_de_attr(gconstpointer data)
{
	const struct test_data_de *test = data;
	uint128_t u128;
	sdp_data_t *d;
	int size = 0;

	d = sdp_extract_attr(test->input_data, test->input_size, &size, NULL);
	g_assert(d != NULL);
	g_assert_cmpuint(test->input_size, ==, size);
	g_assert_cmpuint(test->expected.dtd, ==, d->dtd);

	if (g_test_verbose() == TRUE)
		g_print("DTD=0x%02x\n", d->dtd);

	switch (d->dtd) {
	case SDP_TEXT_STR8:
	case SDP_TEXT_STR16:
	case SDP_URL_STR8:
	case SDP_URL_STR16:
		g_assert_cmpstr(test->expected.val.str, ==, d->val.str);
		break;
	case SDP_DATA_NIL:
	case SDP_UINT8:
		g_assert_cmpuint(test->expected.val.uint8, ==, d->val.uint8);
		break;
	case SDP_UINT16:
		g_assert_cmpuint(test->expected.val.uint16, ==, d->val.uint16);
		break;
	case SDP_UINT32:
		g_assert_cmpuint(test->expected.val.uint32, ==, d->val.uint32);
		break;
	case SDP_UINT64:
		g_assert_cmpuint(test->expected.val.uint64, ==, d->val.uint64);
		break;
	case SDP_BOOL:
	case SDP_INT8:
		g_assert_cmpuint(test->expected.val.int8, ==, d->val.int8);
		break;
	case SDP_INT16:
		g_assert_cmpuint(test->expected.val.int16, ==, d->val.int16);
		break;
	case SDP_INT32:
		g_assert_cmpuint(test->expected.val.int32, ==, d->val.int32);
		break;
	case SDP_INT64:
		g_assert_cmpuint(test->expected.val.int64, ==, d->val.int64);
		break;
	case SDP_UINT128:
	case SDP_INT128:
		/* Expected bytes are in network order */
		hton128(&d->val.uint128, &u128);
		g_assert(memcmp(&test->expected.val.uint128, &u128,
						sizeof(uint128_t)) == 0);
		break;
	default:
		g_assert_not_reached();
	}

	sdp_data_free(d);
}

int main(int argc, char *argv[])
{
	g_test_init(&argc, &argv, NULL);

	/*
	 * Service Search Request
	 *
	 * Verify the correct behaviour of the IUT when searching for
	 * existing service(s).
	 */
	define_ss("BV-01-C/UUID-16",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x08, 0x35, 0x03, 0x19,
			0x11, 0x05, 0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x01, 0x00));
	define_ss("BV-01-C/UUID-32",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x0a, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x00, 0x03, 0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00));
	define_ss("BV-01-C/UUID-128",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x01, 0x00));

	/*
	 * Service Search Request
	 *
	 * Verify the correct behaviour of the IUT when searching for
	 * existing service(s), using continuation state.
	 */
	define_ss("BV-03-C/UUID-16",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x08, 0x35, 0x03, 0x19,
			0x01, 0x00, 0xff, 0xff, 0x00),
		raw_pdu_cont(8, 0x03, 0x00, 0x01, 0x00, 0x29, 0x00, 0x08, 0x00,
				0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
				0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
				0x05, 0x00, 0x01, 0x00, 0x06, 0x08),
		raw_pdu_cont(8, 0x02, 0x00, 0x02, 0x00, 0x10, 0x35, 0x03, 0x19,
				0x01, 0x00, 0xff, 0xff, 0x08),
		raw_pdu(0x03, 0x00, 0x02, 0x00, 0x09, 0x00, 0x08, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x07, 0x00));
	define_ss("BV-03-C/UUID-32",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x0a, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0x00),
		raw_pdu_cont(8, 0x03, 0x00, 0x01, 0x00, 0x29, 0x00, 0x08, 0x00,
				0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
				0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
				0x05, 0x00, 0x01, 0x00, 0x06, 0x08),
		raw_pdu_cont(8, 0x02, 0x00, 0x02, 0x00, 0x12, 0x35, 0x05, 0x1a,
				0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0x08),
		raw_pdu(0x03, 0x00, 0x02, 0x00, 0x09, 0x00, 0x08, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x07, 0x00));
	define_ss("BV-03-C/UUID-128",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x00),
		raw_pdu_cont(8, 0x03, 0x00, 0x01, 0x00, 0x29, 0x00, 0x08, 0x00,
				0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
				0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
				0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
				0x05, 0x00, 0x01, 0x00, 0x06, 0x08),
		raw_pdu_cont(8, 0x02, 0x00, 0x02, 0x00, 0x1e, 0x35, 0x11, 0x1c,
				0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
				0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
				0xff, 0xff, 0x08),
		raw_pdu(0x03, 0x00, 0x02, 0x00, 0x09, 0x00, 0x08, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x07, 0x00));

	/*
	 * Service Search Request
	 *
	 * Verify the correct behaviour of the IUT when searching for
	 * no existing service(s).
	 */
	define_ss("BV-04-C/UUID-16",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x08, 0x35, 0x03, 0x19,
			0xff, 0xff, 0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00,
			0x00, 0x00));
	define_ss("BV-04-C/UUID-128",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00,
			0x00, 0x00));
	define_ss("BV-04-C/UUID-32",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x0a, 0x35, 0x05, 0x1a,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00,
			0x00, 0x00));

	/*
	 * Service Search Request
	 *
	 * Verify the correct behaviour of the IUT when searching for
	 * existing service(s), using invalid PDU size.
	 */
	define_ss("BI-01-C/UUID-16",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x01, 0x00, 0x00, 0x05, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x04));
	define_ss("BI-01-C/UUID-32",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x05, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x04));
	define_ss("BI-01-C/UUID-128",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x05, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x04));

	/*
	 * Service Search Request
	 *
	 * Verify the correct behaviour of the IUT when searching for
	 * existing service(s), using invalid request syntax.
	 */
	define_ss("BI-02-C/UUID-16",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x06, 0x35, 0x03, 0x19,
			0x01, 0x00, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03));
	define_ss("BI-02-C/UUID-32",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x08, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x01, 0x00, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03));
	define_ss("BI-02-C/UUID-128",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x14, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with attribute(s).
	 */
	define_sa("BV-01-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x0a, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x05, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with the existing
	 * Attribute(s) using ContinuationState.
	 */
	define_sa("BV-03-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x07, 0x35, 0x06, 0x09, 0x00, 0x00,
			0x09, 0x00, 0x01, 0x00),
		raw_pdu_cont(8, 0x05, 0x00, 0x01, 0x00, 0x12, 0x00, 0x07, 0x35,
				0x10, 0x09, 0x00, 0x00, 0x0a, 0x00, 0x08),
		raw_pdu_cont(8, 0x04, 0x00, 0x02, 0x00, 0x17, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x07, 0x35, 0x06, 0x09, 0x00, 0x00,
				0x09, 0x00, 0x01, 0x08),
		raw_pdu_cont(8, 0x05, 0x00, 0x02, 0x00, 0x12, 0x00, 0x07, 0x01,
				0x00, 0x00, 0x09, 0x00, 0x01, 0x35, 0x08),
		raw_pdu_cont(8, 0x04, 0x00, 0x03, 0x00, 0x17, 0x00, 0x01, 0x00,
				0x00, 0x00, 0x07, 0x35, 0x06, 0x09, 0x00, 0x00,
				0x09, 0x00, 0x01, 0x08),
		raw_pdu(0x05, 0x00, 0x03, 0x00, 0x07, 0x00, 0x04, 0x03,
			0x19, 0x11, 0x01, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * ServiceID attribute.
	 */
	define_sa("BV-04-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x19, 0x35, 0x03, 0x09, 0x00, 0x03,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0b, 0x00, 0x08, 0x35,
			0x06, 0x09, 0x00, 0x03, 0x19, 0x11, 0x01, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * ProtocolDescriptorList attribute.
	 */
	define_sa("BV-05-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x04,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x1b, 0x00, 0x18, 0x35,
			0x16, 0x09, 0x00, 0x04, 0x35, 0x11, 0x35, 0x03,
			0x19, 0x01, 0x00, 0x35, 0x05, 0x19, 0x00, 0x03,
			0x08, 0x09, 0x35, 0x03, 0x19, 0x00, 0x08, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with the
	 * ServiceRecordState attribute.
	 */
	define_sa("BV-06-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x0d, 0x35, 0x03, 0x09, 0x00, 0x02,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x02, 0x0a, 0x00, 0x00, 0x12,
			0x34, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with the
	 * ServiceInfoTime attribute.
	 */
	define_sa("BV-07-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x0d, 0x35, 0x03, 0x09, 0x00, 0x07,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x07, 0x0a, 0x00, 0x00, 0xff,
			0xff, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * BrowseGroupList attribute.
	 */
	define_sa("BV-08-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x0a, 0x35, 0x03, 0x09, 0x00, 0x05,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x05, 0x35, 0x03, 0x19, 0x10,
			0x02, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * LanguageBaseAttributeIdList attribute.
	 */
	define_sa("BV-09-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x13, 0x35, 0x03, 0x09, 0x00, 0x06,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x13, 0x00, 0x10, 0x35,
			0x0e, 0x09, 0x00, 0x06, 0x35, 0x09, 0x09, 0x65,
			0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01, 0x00, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * ServiceAvailability attribute.
	 */
	define_sa("BV-10-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x0a, 0x35, 0x03, 0x09, 0x00, 0x08,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x07, 0x35,
			0x05, 0x09, 0x00, 0x08, 0x08, 0xff, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * IconURL attribute.
	 */
	define_sa("BV-11-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x1f, 0x35, 0x03, 0x09, 0x00, 0x0c,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x1f, 0x00, 0x1c, 0x35,
			0x1a, 0x09, 0x00, 0x0c, 0x45, 0x15, 0x68, 0x74,
			0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77,
			0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a, 0x2e, 0x6f,
			0x72, 0x67, 0x2f, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * ServiceName attribute.
	 */
	define_sa("BV-12-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x16, 0x35, 0x03, 0x09, 0x00, 0x06,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x13, 0x00, 0x10, 0x35,
			0x0e, 0x09, 0x00, 0x06, 0x35, 0x09, 0x09, 0x65,
			0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x15, 0x35, 0x03, 0x09, 0x01, 0x00,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x09, 0x01, 0x00, 0x25, 0x0b, 0x53, 0x65,
			0x72, 0x69, 0x61, 0x6c, 0x20, 0x50, 0x6f, 0x72,
			0x74, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * ServiceDescription attribute.
	 */
	define_sa("BV-13-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x16, 0x35, 0x03, 0x09, 0x00, 0x06,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x13, 0x00, 0x10, 0x35,
			0x0e, 0x09, 0x00, 0x06, 0x35, 0x09, 0x09, 0x65,
			0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x12, 0x35, 0x03, 0x09, 0x01, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x12, 0x00, 0x0f, 0x35,
			0x0d, 0x09, 0x01, 0x01, 0x25, 0x08, 0x43, 0x4f,
			0x4d, 0x20, 0x50, 0x6f, 0x72, 0x74, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * ProviderName attribute.
	 */
	define_sa("BV-14-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x16, 0x35, 0x03, 0x09, 0x00, 0x06,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x13, 0x00, 0x10, 0x35,
			0x0e, 0x09, 0x00, 0x06, 0x35, 0x09, 0x09, 0x65,
			0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x0f, 0x35, 0x03, 0x09, 0x01, 0x02,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x09, 0x01, 0x02, 0x25, 0x05, 0x42, 0x6c,
			0x75, 0x65, 0x5a, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * VersionNumberList attribute.
	 */
	define_sa("BV-15-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x0d, 0x35, 0x03, 0x09, 0x02, 0x00,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x02, 0x00, 0x35, 0x03, 0x09, 0x01,
			0x00, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with the
	 * ServiceDatabaseState attribute.
	 */
	define_sa("BV-16-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x0d, 0x35, 0x03, 0x09, 0x02, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x02, 0x01, 0x0a, 0x49, 0x6f, 0x06,
			0x54, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with the
	 * BluetoothProfileDescriptorList attribute.
	 */
	define_sa("BV-17-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x0f, 0x35, 0x03, 0x09, 0x00, 0x09,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x12, 0x00, 0x0f, 0x35,
			0x0d, 0x09, 0x00, 0x09, 0x35, 0x08, 0x35, 0x06,
			0x19, 0x11, 0x05, 0x09, 0x01, 0x00, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with the
	 * DocumentationURL attribute.
	 */
	define_sa("BV-18-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x1f, 0x35, 0x03, 0x09, 0x00, 0x0a,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x1f, 0x00, 0x1c, 0x35,
			0x1a, 0x09, 0x00, 0x0a, 0x45, 0x15, 0x68, 0x74,
			0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77,
			0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a, 0x2e, 0x6f,
			0x72, 0x67, 0x2f, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with the
	 * ClientExecutableURL attribute.
	 */
	define_sa("BV-19-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x1f, 0x35, 0x03, 0x09, 0x00, 0x0b,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x1f, 0x00, 0x1c, 0x35,
			0x1a, 0x09, 0x00, 0x0b, 0x45, 0x15, 0x68, 0x74,
			0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77,
			0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a, 0x2e, 0x6f,
			0x72, 0x67, 0x2f, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for non-existing Attribute.
	 */
	define_sa("BV-20-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x07, 0x35, 0x03, 0x09, 0xff, 0xff,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify that the IUT is able to respond with an
	 * AdditionalProtocolDescriptorList attribute.
	 */
	define_sa("BV-21-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x24, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x02, 0x00),
		raw_pdu(0x04, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x0d,
			0x00),
		raw_pdu(0x05, 0x00, 0x02, 0x00, 0x19, 0x00, 0x16, 0x35,
			0x14, 0x09, 0x00, 0x0d, 0x35, 0x0f, 0x35, 0x0d,
			0x35, 0x06, 0x19, 0x01, 0x00, 0x09, 0x00, 0x13,
			0x35, 0x03, 0x19, 0x00, 0x11, 0x00));

	/*
	 * Service Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Attribute, using invalid ServiceRecordHandle.
	 */
	define_sa("BI-01-C",
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0xff, 0xff, 0xff,
			0xff, 0x00, 0x07, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x02));

	/*
	 * Service Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Attribute, using invalid request syntax.
	 */
	define_sa("BI-02-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0a, 0x00, 0x01, 0x00,
			0x00, 0x35, 0x03, 0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03));

	/*
	 * Service Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Attribute, using invalid PDU-Size.
	 */
	define_sa("BI-03-C",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x01, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x00, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x11, 0x00, 0x01, 0x00,
			0x00, 0x00, 0x07, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x04));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for non-existing Service, existing Attribute using
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-01-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0xff, 0xff, 0x00, 0x0a, 0x35, 0x03, 0x09, 0x00,
			0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));
	define_ssa("BV-01-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x0a, 0x35, 0x03,
			0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));
	define_ssa("BV-01-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0x00, 0x0a, 0x35, 0x03, 0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service, non-existing Attribute using
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-02-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x05, 0x00, 0x0a, 0x35, 0x03, 0x09, 0xff,
			0xff, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));
	define_ssa("BV-02-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x0a, 0x35, 0x03,
			0x09, 0xff, 0xff, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));
	define_ssa("BV-02-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x0a, 0x35, 0x03, 0x09, 0xff, 0xff, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for non-existing Service, non-existing Attribute using
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-03-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0xff, 0xff, 0x00, 0x0a, 0x35, 0x03, 0x09, 0xff,
			0xff, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));
	define_ssa("BV-03-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0xff, 0xff, 0xff, 0xff, 0x00, 0x0a, 0x35, 0x03,
			0x09, 0xff, 0xff, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));
	define_ssa("BV-03-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
			0x00, 0x0a, 0x35, 0x03, 0x09, 0xff, 0xff, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x05, 0x00, 0x02, 0x35,
			0x00, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute(s) using
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-04-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x05, 0x00, 0x11, 0x35, 0x03, 0x09, 0x00,
			0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x05, 0x00));
	define_ssa("BV-04-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x11, 0x35, 0x03,
			0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x05, 0x00));
	define_ssa("BV-04-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x11, 0x35, 0x03, 0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x05, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Attributes, using Continuation State and
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-06-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x10, 0x35, 0x03, 0x19,
			0x11, 0x05, 0x00, 0x07, 0x35, 0x06, 0x09, 0x00,
			0x00, 0x09, 0x00, 0x01, 0x00),
		raw_pdu_cont(8, 0x07, 0x00, 0x01, 0x00, 0x12, 0x00, 0x07, 0x35,
				0x12, 0x35, 0x10, 0x09, 0x00, 0x00, 0x08),
		raw_pdu_cont(8, 0x06, 0x00, 0x02, 0x00, 0x18, 0x35, 0x03, 0x19,
				0x11, 0x05, 0x00, 0x07, 0x35, 0x06, 0x09, 0x00,
				0x00, 0x09, 0x00, 0x01, 0x08),
		raw_pdu_cont(8, 0x07, 0x00, 0x02, 0x00, 0x12, 0x00, 0x07, 0x0a,
				0x00, 0x01, 0x00, 0x01, 0x09, 0x00, 0x08),
		raw_pdu_cont(8, 0x06, 0x00, 0x03, 0x00, 0x18, 0x35, 0x03, 0x19,
				0x11, 0x05, 0x00, 0x07, 0x35, 0x06, 0x09, 0x00,
				0x00, 0x09, 0x00, 0x01, 0x08),
		raw_pdu(0x07, 0x00, 0x03, 0x00, 0x09, 0x00, 0x06, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x00));
	define_ssa("BV-06-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x12, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x07, 0x35, 0x06,
			0x09, 0x00, 0x00, 0x09, 0x00, 0x01, 0x00),
		raw_pdu_cont(8, 0x07, 0x00, 0x01, 0x00, 0x12, 0x00, 0x07, 0x35,
				0x12, 0x35, 0x10, 0x09, 0x00, 0x00, 0x08),
		raw_pdu_cont(8, 0x06, 0x00, 0x02, 0x00, 0x1a, 0x35, 0x05, 0x1a,
				0x00, 0x00, 0x11, 0x05, 0x00, 0x07, 0x35, 0x06,
				0x09, 0x00, 0x00, 0x09, 0x00, 0x01, 0x08),
		raw_pdu_cont(8, 0x07, 0x00, 0x02, 0x00, 0x12, 0x00, 0x07, 0x0a,
				0x00, 0x01, 0x00, 0x01, 0x09, 0x00, 0x08),
		raw_pdu_cont(8, 0x06, 0x00, 0x03, 0x00, 0x1a, 0x35, 0x05, 0x1a,
				0x00, 0x00, 0x11, 0x05, 0x00, 0x07, 0x35, 0x06,
				0x09, 0x00, 0x00, 0x09, 0x00, 0x01, 0x08),
		raw_pdu(0x07, 0x00, 0x03, 0x00, 0x09, 0x00, 0x06, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x00));
	define_ssa("BV-06-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1e, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x07, 0x35, 0x06, 0x09, 0x00, 0x00, 0x09,
			0x00, 0x01, 0x00),
		raw_pdu_cont(8, 0x07, 0x00, 0x01, 0x00, 0x12, 0x00, 0x07, 0x35,
				0x12, 0x35, 0x10, 0x09, 0x00, 0x00, 0x08),
		raw_pdu_cont(8, 0x06, 0x00, 0x02, 0x00, 0x26, 0x35, 0x11, 0x1c,
				0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
				0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
				0x00, 0x07, 0x35, 0x06, 0x09, 0x00, 0x00, 0x09,
				0x00, 0x01, 0x08),
		raw_pdu_cont(8, 0x07, 0x00, 0x02, 0x00, 0x12, 0x00, 0x07, 0x0a,
				0x00, 0x01, 0x00, 0x01, 0x09, 0x00, 0x08),
		raw_pdu_cont(8, 0x06, 0x00, 0x03, 0x00, 0x26, 0x35, 0x11, 0x1c,
				0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
				0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
				0x00, 0x07, 0x35, 0x06, 0x09, 0x00, 0x00, 0x09,
				0x00, 0x01, 0x08),
		raw_pdu(0x07, 0x00, 0x03, 0x00, 0x09, 0x00, 0x06, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ServiceRecordState
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-07-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x13, 0x35, 0x03, 0x09, 0x00,
			0x02, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x02, 0x0a, 0x00,
			0x00, 0x12, 0x34, 0x00));
	define_ssa("BV-07-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x13, 0x35, 0x03,
			0x09, 0x00, 0x02, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x02, 0x0a, 0x00,
			0x00, 0x12, 0x34, 0x00));
	define_ssa("BV-07-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x13, 0x35, 0x03, 0x09, 0x00, 0x02, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x02, 0x0a, 0x00,
			0x00, 0x12, 0x34, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ServiceDataBaseState
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-08-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x10, 0x00, 0x00, 0x13, 0x35, 0x03, 0x09, 0x02,
			0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x02, 0x01, 0x0a, 0x49,
			0x6f, 0x06, 0x54, 0x00));
	define_ssa("BV-08-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x10, 0x00, 0x00, 0x13, 0x35, 0x03,
			0x09, 0x02, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x02, 0x01, 0x0a, 0x49,
			0x6f, 0x06, 0x54, 0x00));
	define_ssa("BV-08-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x13, 0x35, 0x03, 0x09, 0x02, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x02, 0x01, 0x0a, 0x49,
			0x6f, 0x06, 0x54, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ServiceInfoTimeToLive
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-09-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x13, 0x35, 0x03, 0x09, 0x00,
			0x07, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x07, 0x0a, 0x00,
			0x00, 0xff, 0xff, 0x00));
	define_ssa("BV-09-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x13, 0x35, 0x03,
			0x09, 0x00, 0x07, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x07, 0x0a, 0x00,
			0x00, 0xff, 0xff, 0x00));
	define_ssa("BV-09-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x13, 0x35, 0x03, 0x09, 0x00, 0x07, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x07, 0x0a, 0x00,
			0x00, 0xff, 0xff, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ServiceID using
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-10-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x1e, 0x35, 0x03, 0x09, 0x00,
			0x03, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x35, 0x06, 0x09, 0x00, 0x03, 0x19, 0x11,
			0x01, 0x00));
	define_ssa("BV-10-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x1e, 0x35, 0x03,
			0x09, 0x00, 0x03, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x35, 0x06, 0x09, 0x00, 0x03, 0x19, 0x11,
			0x01, 0x00));
	define_ssa("BV-10-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x1e, 0x35, 0x03, 0x09, 0x00, 0x03, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x35, 0x06, 0x09, 0x00, 0x03, 0x19, 0x11,
			0x01, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ProtocolDescriptorList
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-11-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00,
			0x04, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x1d, 0x00, 0x1a, 0x35,
			0x18, 0x35, 0x16, 0x09, 0x00, 0x04, 0x35, 0x11,
			0x35, 0x03, 0x19, 0x01, 0x00, 0x35, 0x05, 0x19,
			0x00, 0x03, 0x08, 0x09, 0x35, 0x03, 0x19, 0x00,
			0x08, 0x00));
	define_ssa("BV-11-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x05, 0xff, 0xff, 0x35, 0x03,
			0x09, 0x00, 0x04, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x1d, 0x00, 0x1a, 0x35,
			0x18, 0x35, 0x16, 0x09, 0x00, 0x04, 0x35, 0x11,
			0x35, 0x03, 0x19, 0x01, 0x00, 0x35, 0x05, 0x19,
			0x00, 0x03, 0x08, 0x09, 0x35, 0x03, 0x19, 0x00,
			0x08, 0x00));
	define_ssa("BV-11-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x04, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x1d, 0x00, 0x1a, 0x35,
			0x18, 0x35, 0x16, 0x09, 0x00, 0x04, 0x35, 0x11,
			0x35, 0x03, 0x19, 0x01, 0x00, 0x35, 0x05, 0x19,
			0x00, 0x03, 0x08, 0x09, 0x35, 0x03, 0x19, 0x00,
			0x08, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute BrowseGroupList
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-12-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00,
			0x05, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x05, 0x35, 0x03,
			0x19, 0x10, 0x02, 0x00));
	define_ssa("BV-12-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x05, 0xff, 0xff, 0x35, 0x03,
			0x09, 0x00, 0x05, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x05, 0x35, 0x03,
			0x19, 0x10, 0x02, 0x00));
	define_ssa("BV-12-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x05, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x00, 0x05, 0x35, 0x03,
			0x19, 0x10, 0x02, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute LanguageBaseAttributeIdList
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-13-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x18, 0x35, 0x03, 0x09, 0x00,
			0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00));
	define_ssa("BV-13-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x18, 0x35, 0x03,
			0x09, 0x00, 0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00));
	define_ssa("BV-13-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x18, 0x35, 0x03, 0x09, 0x00, 0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ServiceAvailability
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-14-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x0f, 0x35, 0x03, 0x09, 0x00,
			0x08, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x09, 0x35,
			0x07, 0x35, 0x05, 0x09, 0x00, 0x08, 0x08, 0xff,
			0x00));
	define_ssa("BV-14-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x0f, 0x35, 0x03,
			0x09, 0x00, 0x08, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x09, 0x35,
			0x07, 0x35, 0x05, 0x09, 0x00, 0x08, 0x08, 0xff,
			0x00));
	define_ssa("BV-14-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x0f, 0x35, 0x03, 0x09, 0x00, 0x08, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x09, 0x35,
			0x07, 0x35, 0x05, 0x09, 0x00, 0x08, 0x08, 0xff,
			0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute IconURL using
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-15-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x24, 0x35, 0x03, 0x09, 0x00,
			0x0c, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0c, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));
	define_ssa("BV-15-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x24, 0x35, 0x03,
			0x09, 0x00, 0x0c, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0c, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));
	define_ssa("BV-15-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x24, 0x35, 0x03, 0x09, 0x00, 0x0c, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0c, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ServiceName using
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-16-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x1b, 0x35, 0x03, 0x09, 0x00,
			0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x1d, 0x35, 0x03, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x17, 0x00, 0x14, 0x35,
			0x12, 0x35, 0x10, 0x09, 0x01, 0x00, 0x25, 0x0b,
			0x53, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x50,
			0x6f, 0x72, 0x74, 0x00));
	define_ssa("BV-16-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x1b, 0x35, 0x03,
			0x09, 0x00, 0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x1d, 0x35, 0x03,
			0x09, 0x01, 0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x17, 0x00, 0x14, 0x35,
			0x12, 0x35, 0x10, 0x09, 0x01, 0x00, 0x25, 0x0b,
			0x53, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x50,
			0x6f, 0x72, 0x74, 0x00));
	define_ssa("BV-16-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x1b, 0x35, 0x03, 0x09, 0x00, 0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x1d, 0x35, 0x03, 0x09, 0x01, 0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x17, 0x00, 0x14, 0x35,
			0x12, 0x35, 0x10, 0x09, 0x01, 0x00, 0x25, 0x0b,
			0x53, 0x65, 0x72, 0x69, 0x61, 0x6c, 0x20, 0x50,
			0x6f, 0x72, 0x74, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ServiceDescription
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-17-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x1b, 0x35, 0x03, 0x09, 0x00,
			0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x1a, 0x35, 0x03, 0x09, 0x01,
			0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x14, 0x00, 0x11, 0x35,
			0x0f, 0x35, 0x0d, 0x09, 0x01, 0x01, 0x25, 0x08,
			0x43, 0x4f, 0x4d, 0x20, 0x50, 0x6f, 0x72, 0x74,
			0x00));
	define_ssa("BV-17-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x1b, 0x35, 0x03,
			0x09, 0x00, 0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x1a, 0x35, 0x03,
			0x09, 0x01, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x14, 0x00, 0x11, 0x35,
			0x0f, 0x35, 0x0d, 0x09, 0x01, 0x01, 0x25, 0x08,
			0x43, 0x4f, 0x4d, 0x20, 0x50, 0x6f, 0x72, 0x74,
			0x00));
	define_ssa("BV-17-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x1b, 0x35, 0x03, 0x09, 0x00, 0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x1a, 0x35, 0x03, 0x09, 0x01, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x14, 0x00, 0x11, 0x35,
			0x0f, 0x35, 0x0d, 0x09, 0x01, 0x01, 0x25, 0x08,
			0x43, 0x4f, 0x4d, 0x20, 0x50, 0x6f, 0x72, 0x74,
			0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ProviderName using
	 * ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-18-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x1b, 0x35, 0x03, 0x09, 0x00,
			0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x17, 0x35, 0x03, 0x09, 0x01,
			0x02, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x11, 0x00, 0x0e, 0x35,
			0x0c, 0x35, 0x0a, 0x09, 0x01, 0x02, 0x25, 0x05,
			0x42, 0x6c, 0x75, 0x65, 0x5a, 0x00));
	define_ssa("BV-18-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x1b, 0x35, 0x03,
			0x09, 0x00, 0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x17, 0x35, 0x03,
			0x09, 0x01, 0x02, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x11, 0x00, 0x0e, 0x35,
			0x0c, 0x35, 0x0a, 0x09, 0x01, 0x02, 0x25, 0x05,
			0x42, 0x6c, 0x75, 0x65, 0x5a, 0x00));
	define_ssa("BV-18-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x1b, 0x35, 0x03, 0x09, 0x00, 0x06, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x15, 0x00, 0x12, 0x35,
			0x10, 0x35, 0x0e, 0x09, 0x00, 0x06, 0x35, 0x09,
			0x09, 0x65, 0x6e, 0x09, 0x00, 0x6a, 0x09, 0x01,
			0x00, 0x00),
		raw_pdu(0x06, 0x00, 0x02, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x17, 0x35, 0x03, 0x09, 0x01, 0x02, 0x00),
		raw_pdu(0x07, 0x00, 0x02, 0x00, 0x11, 0x00, 0x0e, 0x35,
			0x0c, 0x35, 0x0a, 0x09, 0x01, 0x02, 0x25, 0x05,
			0x42, 0x6c, 0x75, 0x65, 0x5a, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute VersionNumberList
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-19-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x10, 0x00, 0x00, 0x12, 0x35, 0x03, 0x09, 0x02,
			0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x02, 0x00, 0x35, 0x03,
			0x09, 0x01, 0x00, 0x00));
	define_ssa("BV-19-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x10, 0x00, 0x00, 0x12, 0x35, 0x03,
			0x09, 0x02, 0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x02, 0x00, 0x35, 0x03,
			0x09, 0x01, 0x00, 0x00));
	define_ssa("BV-19-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x12, 0x35, 0x03, 0x09, 0x02, 0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0f, 0x00, 0x0c, 0x35,
			0x0a, 0x35, 0x08, 0x09, 0x02, 0x00, 0x35, 0x03,
			0x09, 0x01, 0x00, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching for
	 * existing Service(s) and Attribute BluetoothProfileDescriptorList
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-20-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00,
			0x09, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x14, 0x00, 0x11, 0x35,
			0x0f, 0x35, 0x0d, 0x09, 0x00, 0x09, 0x35, 0x08,
			0x35, 0x06, 0x19, 0x11, 0x05, 0x09, 0x01, 0x00,
			0x00));
	define_ssa("BV-20-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x05, 0xff, 0xff, 0x35, 0x03,
			0x09, 0x00, 0x09, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x14, 0x00, 0x11, 0x35,
			0x0f, 0x35, 0x0d, 0x09, 0x00, 0x09, 0x35, 0x08,
			0x35, 0x06, 0x19, 0x11, 0x05, 0x09, 0x01, 0x00,
			0x00));
	define_ssa("BV-20-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x05, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x09, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x14, 0x00, 0x11, 0x35,
			0x0f, 0x35, 0x0d, 0x09, 0x00, 0x09, 0x35, 0x08,
			0x35, 0x06, 0x19, 0x11, 0x05, 0x09, 0x01, 0x00,
			0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute DocumentationURL
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-21-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x24, 0x35, 0x03, 0x09, 0x00,
			0x0a, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0a, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));
	define_ssa("BV-21-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x24, 0x35, 0x03,
			0x09, 0x00, 0x0a, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0a, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));
	define_ssa("BV-21-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x24, 0x35, 0x03, 0x09, 0x00, 0x0a, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0a, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Service(s) and Attribute ClientExecutableURL
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-22-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x01, 0x00, 0x24, 0x35, 0x03, 0x09, 0x00,
			0x0b, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0b, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));
	define_ssa("BV-22-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x24, 0x35, 0x03,
			0x09, 0x00, 0x0b, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0b, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));
	define_ssa("BV-22-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x00, 0x24, 0x35, 0x03, 0x09, 0x00, 0x0b, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x21, 0x00, 0x1e, 0x35,
			0x1c, 0x35, 0x1a, 0x09, 0x00, 0x0b, 0x45, 0x15,
			0x68, 0x74, 0x74, 0x70, 0x3a, 0x2f, 0x2f, 0x77,
			0x77, 0x77, 0x2e, 0x62, 0x6c, 0x75, 0x65, 0x7a,
			0x2e, 0x6f, 0x72, 0x67, 0x2f, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching for
	 * existing Service(s) and Attribute AdditionalProtocolDescriptorList
	 * using ServiceSearchAttributeRequest.
	 */
	define_ssa("BV-23-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x11, 0x24, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00,
			0x0d, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x1b, 0x00, 0x18, 0x35,
			0x16, 0x35, 0x14, 0x09, 0x00, 0x0d, 0x35, 0x0f,
			0x35, 0x0d, 0x35, 0x06, 0x19, 0x01, 0x00, 0x09,
			0x00, 0x13, 0x35, 0x03, 0x19, 0x00, 0x11, 0x00));
	define_ssa("BV-23-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x11, 0x24, 0xff, 0xff, 0x35, 0x03,
			0x09, 0x00, 0x0d, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x1b, 0x00, 0x18, 0x35,
			0x16, 0x35, 0x14, 0x09, 0x00, 0x0d, 0x35, 0x0f,
			0x35, 0x0d, 0x35, 0x06, 0x19, 0x01, 0x00, 0x09,
			0x00, 0x13, 0x35, 0x03, 0x19, 0x00, 0x11, 0x00));
	define_ssa("BV-23-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x11, 0x24, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x0d, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x1b, 0x00, 0x18, 0x35,
			0x16, 0x35, 0x14, 0x09, 0x00, 0x0d, 0x35, 0x0f,
			0x35, 0x0d, 0x35, 0x06, 0x19, 0x01, 0x00, 0x09,
			0x00, 0x13, 0x35, 0x03, 0x19, 0x00, 0x11, 0x00));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Attribute, using invalid request syntax,
	 * using the ServiceSearchAttributeRequest PDU.
	 */
	define_ssa("BI-01-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0b, 0x35, 0x03, 0x19,
			0x01, 0x00, 0x35, 0x03, 0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03));
	define_ssa("BI-01-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x01, 0x00, 0x35, 0x03, 0x09, 0x00,
			0x01, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03));
	define_ssa("BI-01-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x19, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0x35, 0x03, 0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x03));

	/*
	 * Service Search Attribute Request
	 *
	 * Verify the correct behaviour of the IUT when searching
	 * for existing Attribute, using invalid PDU-size, using the
	 * ServiceSearchAttributeRequest PDU.
	 */
	define_ssa("BI-02-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x12, 0x35, 0x03, 0x19,
			0x01, 0x00, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00,
			0x01, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x04));
	define_ssa("BI-02-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x14, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x01, 0x00, 0xff, 0xff, 0x35, 0x03,
			0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x04));
	define_ssa("BI-02-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x20, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x04));

	/*
	 * Service Browse
	 *
	 * Verify that the IUT behave correct using SDP_ServiceSearchRequest
	 * and SDP_ServiceAttributeRequest for Service Browse.
	 */
	define_brw("BV-01-C/UUID-16",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x08, 0x35, 0x03, 0x19,
			0x10, 0x02, 0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x25, 0x00, 0x08, 0x00,
			0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
			0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
			0x05, 0x00, 0x01, 0x00, 0x06, 0x00, 0x01, 0x00,
			0x07, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x05, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x24, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x03, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x04, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x06, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x07, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x08, 0x35, 0x03, 0x19,
			0x10, 0x01, 0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x00, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x02, 0x00,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0b, 0x00, 0x08, 0x35,
			0x06, 0x09, 0x02, 0x00, 0x19, 0x10, 0x02, 0x00),
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x08, 0x35, 0x03, 0x19,
			0x10, 0x02, 0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x25, 0x00, 0x08, 0x00,
			0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
			0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
			0x05, 0x00, 0x01, 0x00, 0x06, 0x00, 0x01, 0x00,
			0x07, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x05, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x24, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x03, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x04, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x06, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x07, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00));
	define_brw("BV-01-C/UUID-32",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x0a, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x10, 0x02, 0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x25, 0x00, 0x08, 0x00,
			0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
			0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
			0x05, 0x00, 0x01, 0x00, 0x06, 0x00, 0x01, 0x00,
			0x07, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x05, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x24, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x03, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x04, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x06, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x07, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x0a, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x10, 0x01, 0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x00, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x02, 0x00,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0b, 0x00, 0x08, 0x35,
			0x06, 0x09, 0x02, 0x00, 0x19, 0x10, 0x02, 0x00),
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x0a, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x10, 0x02, 0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x25, 0x00, 0x08, 0x00,
			0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
			0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
			0x05, 0x00, 0x01, 0x00, 0x06, 0x00, 0x01, 0x00,
			0x07, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x05, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x24, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x03, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x04, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x06, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x07, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00));
	define_brw("BV-01-C/UUID-128",
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x02, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x25, 0x00, 0x08, 0x00,
			0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
			0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
			0x05, 0x00, 0x01, 0x00, 0x06, 0x00, 0x01, 0x00,
			0x07, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x05, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x24, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x03, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x04, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x06, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x07, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x09, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x00, 0x00, 0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x00, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x02, 0x00,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0b, 0x00, 0x08, 0x35,
			0x06, 0x09, 0x02, 0x00, 0x19, 0x10, 0x02, 0x00),
		raw_pdu(0x02, 0x00, 0x01, 0x00, 0x16, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x02, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x00),
		raw_pdu(0x03, 0x00, 0x01, 0x00, 0x25, 0x00, 0x08, 0x00,
			0x08, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00,
			0x01, 0x00, 0x01, 0x00, 0x02, 0x00, 0x01, 0x00,
			0x03, 0x00, 0x01, 0x00, 0x04, 0x00, 0x01, 0x00,
			0x05, 0x00, 0x01, 0x00, 0x06, 0x00, 0x01, 0x00,
			0x07, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x00, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x01, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x05, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x24, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x03, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x04, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x05, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x06, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x04, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x01, 0x00,
			0x07, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01,
			0x00),
		raw_pdu(0x05, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00));

	/*
	 * Service Browse
	 *
	 * Verify that the IUT behave correct using
	 * SDP_ServiceSearchAttributeRequest for Service Browse.
	 */
	define_brw("BV-02-C/UUID-16",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x10, 0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00,
			0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x55, 0x00, 0x52, 0x35,
			0x50, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x01, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x24, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x06, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x06, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x06, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x10, 0x01, 0xff, 0xff, 0x35, 0x03, 0x09, 0x02,
			0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x35, 0x06, 0x09, 0x02, 0x00, 0x19, 0x10,
			0x02, 0x00),
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0d, 0x35, 0x03, 0x19,
			0x10, 0x02, 0xff, 0xff, 0x35, 0x03, 0x09, 0x00,
			0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x55, 0x00, 0x52, 0x35,
			0x50, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x01, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x24, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x06, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x06, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x06, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00));
	define_brw("BV-02-C/UUID-32",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x10, 0x02, 0xff, 0xff, 0x35, 0x03,
			0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x55, 0x00, 0x52, 0x35,
			0x50, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x01, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x24, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x06, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x06, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x06, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x10, 0x01, 0xff, 0xff, 0x35, 0x03,
			0x09, 0x02, 0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x35, 0x06, 0x09, 0x02, 0x00, 0x19, 0x10,
			0x02, 0x00),
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x0f, 0x35, 0x05, 0x1a,
			0x00, 0x00, 0x10, 0x02, 0xff, 0xff, 0x35, 0x03,
			0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x55, 0x00, 0x52, 0x35,
			0x50, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x01, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x24, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x06, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x06, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x06, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00));
	define_brw("BV-02-C/UUID-128",
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x02, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x55, 0x00, 0x52, 0x35,
			0x50, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x01, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x24, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x06, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x06, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x06, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00),
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x01, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x35, 0x03, 0x09, 0x02, 0x00, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x0d, 0x00, 0x0a, 0x35,
			0x08, 0x35, 0x06, 0x09, 0x02, 0x00, 0x19, 0x10,
			0x02, 0x00),
		raw_pdu(0x06, 0x00, 0x01, 0x00, 0x1b, 0x35, 0x11, 0x1c,
			0x00, 0x00, 0x10, 0x02, 0x00, 0x00, 0x10, 0x00,
			0x80, 0x00, 0x00, 0x80, 0x5f, 0x9b, 0x34, 0xfb,
			0xff, 0xff, 0x35, 0x03, 0x09, 0x00, 0x01, 0x00),
		raw_pdu(0x07, 0x00, 0x01, 0x00, 0x55, 0x00, 0x52, 0x35,
			0x50, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x01, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x05, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x24, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x35, 0x08, 0x09, 0x00, 0x01, 0x35, 0x03,
			0x19, 0x11, 0x06, 0x35, 0x08, 0x09, 0x00, 0x01,
			0x35, 0x03, 0x19, 0x11, 0x06, 0x35, 0x08, 0x09,
			0x00, 0x01, 0x35, 0x03, 0x19, 0x11, 0x06, 0x35,
			0x08, 0x09, 0x00, 0x01, 0x35, 0x03, 0x19, 0x11,
			0x06, 0x00));

	/*
	 * SDP Data Element (DE) tests
	 *
	 * Test extraction of valid DEs supported by sdp_extract_attr().
	 */
	define_test_de_attr("TEXT_STR8/empty",
			raw_data(0x25, 0x00),
			exp_data(SDP_TEXT_STR8, str, ""));
	define_test_de_attr("TEXT_STR8",
			raw_data(0x25, 0x04, 0x41, 0x42, 0x43, 0x44),
			exp_data(SDP_TEXT_STR8, str, "ABCD"));
	define_test_de_attr("TEXT_STR16",
			raw_data(0x26, 0x00, 0x04, 0x41, 0x42, 0x43, 0x44),
			exp_data(SDP_TEXT_STR16, str, "ABCD"));
	define_test_de_attr("URL_STR8",
			raw_data(0x45, 0x15, 0x68, 0x74, 0x74, 0x70, 0x3a,
				0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x62, 0x6c,
				0x75, 0x65, 0x7a, 0x2e, 0x6f, 0x72, 0x67,
				0x2f),
			exp_data(SDP_URL_STR8, str, "http://www.bluez.org/"));
	define_test_de_attr("URL_STR16",
			raw_data(0x46, 0x00, 0x15, 0x68, 0x74, 0x74, 0x70,
				0x3a, 0x2f, 0x2f, 0x77, 0x77, 0x77, 0x2e, 0x62,
				0x6c, 0x75, 0x65, 0x7a, 0x2e, 0x6f, 0x72, 0x67,
				0x2f),
			exp_data(SDP_URL_STR16, str, "http://www.bluez.org/"));
	define_test_de_attr("NIL",
			raw_data(0x00),
			exp_data(SDP_DATA_NIL, uint8, 0));
	define_test_de_attr("UINT8",
			raw_data(0x08, 0xff),
			exp_data(SDP_UINT8, uint8, UINT8_MAX));
	define_test_de_attr("INT8",
			raw_data(0x10, 0x80),
			exp_data(SDP_INT8, int8, INT8_MIN));
	define_test_de_attr("BOOL",
			raw_data(0x28, 0x01),
			exp_data(SDP_BOOL, int8, 1));
	define_test_de_attr("UINT16",
			raw_data(0x09, 0xff, 0xff),
			exp_data(SDP_UINT16, uint16, UINT16_MAX));
	define_test_de_attr("INT16",
			raw_data(0x11, 0x80, 0x00),
			exp_data(SDP_INT16, int16, INT16_MIN));
	define_test_de_attr("UINT32",
			raw_data(0x0A, 0xff, 0xff, 0xff, 0xff),
			exp_data(SDP_UINT32, uint32, UINT32_MAX));
	define_test_de_attr("INT32",
			raw_data(0x12, 0x80, 0x00, 0x00, 0x00),
			exp_data(SDP_INT32, int32, INT32_MIN));
	define_test_de_attr("UINT64",
			raw_data(0x0B, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
									0xff),
			exp_data(SDP_UINT64, uint64, UINT64_MAX));
	define_test_de_attr("INT64",
			raw_data(0x13, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
									0x00),
			exp_data(SDP_INT64, int64, INT64_MIN));
	/* UINT128/INT128 are just byte arrays parsed as uint128_t */
	define_test_de_attr("UINT128",
			raw_data(0x0C, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
					0xff, 0xff, 0xff),
			exp_data(SDP_UINT128, uint128,
				build_u128(0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff,
						0xff, 0xff, 0xff, 0xff, 0xff)));
	define_test_de_attr("INT128",
			raw_data(0x14, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00),
			exp_data(SDP_INT128, uint128,
				build_u128(0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00,
						0x00, 0x00, 0x00, 0x00, 0x00)));

	return g_test_run();
}
