/* 
        pd.c    (c) 1997-8  Grant R. Guenther <grant@torque.net>
                            Under the terms of the GNU General Public License.

        This is the high-level driver for parallel port IDE hard
        drives based on chips supported by the paride module.

	By default, the driver will autoprobe for a single parallel
	port IDE drive, but if their individual parameters are
        specified, the driver can handle up to 4 drives.

        The behaviour of the pd driver can be altered by setting
        some parameters from the insmod command line.  The following
        parameters are adjustable:
 
	    drive0  	These four arguments can be arrays of	    
	    drive1	1-8 integers as follows:
	    drive2
	    drive3	<prt>,<pro>,<uni>,<mod>,<geo>,<sby>,<dly>,<slv>

			Where,

		<prt>	is the base of the parallel port address for
			the corresponding drive.  (required)

		<pro>   is the protocol number for the adapter that
			supports this drive.  These numbers are
                        logged by 'paride' when the protocol modules
			are initialised.  (0 if not given)

		<uni>   for those adapters that support chained
			devices, this is the unit selector for the
		        chain of devices on the given port.  It should
			be zero for devices that don't support chaining.
			(0 if not given)

		<mod>   this can be -1 to choose the best mode, or one
		        of the mode numbers supported by the adapter.
			(-1 if not given)

		<geo>   this defaults to 0 to indicate that the driver
			should use the CHS geometry provided by the drive
			itself.  If set to 1, the driver will provide
			a logical geometry with 64 heads and 32 sectors
			per track, to be consistent with most SCSI
		        drivers.  (0 if not given)

		<sby>   set this to zero to disable the power saving
			standby mode, if needed.  (1 if not given)

		<dly>   some parallel ports require the driver to 
			go more slowly.  -1 sets a default value that
			should work with the chosen protocol.  Otherwise,
			set this to a small integer, the larger it is
			the slower the port i/o.  In some cases, setting
			this to zero will speed up the device. (default -1)

		<slv>   IDE disks can be jumpered to master or slave.
                        Set this to 0 to choose the master drive, 1 to
                        choose the slave, -1 (the default) to choose the
                        first drive found.
			

            major       You may use this parameter to overide the
                        default major number (45) that this driver
                        will use.  Be sure to change the device
                        name as well.

            name        This parameter is a character string that
                        contains the name the kernel will use for this
                        device (in /proc output, for instance).
			(default "pd")

	    cluster	The driver will attempt to aggregate requests
			for adjacent blocks into larger multi-block
			clusters.  The maximum cluster size (in 512
			byte sectors) is set with this parameter.
			(default 64)

	    verbose	This parameter controls the amount of logging
			that the driver will do.  Set it to 0 for 
			normal operation, 1 to see autoprobe progress
			messages, or 2 to see additional debugging
			output.  (default 0)

            nice        This parameter controls the driver's use of
                        idle CPU time, at the expense of some speed.

        If this driver is built into the kernel, you can use kernel
        the following command line parameters, with the same values
        as the corresponding module parameters listed above:

            pd.drive0
            pd.drive1
            pd.drive2
            pd.drive3
            pd.cluster
            pd.nice

        In addition, you can use the parameter pd.disable to disable
        the driver entirely.
 
*/

/* Changes:

	1.01	GRG 1997.01.24	Restored pd_reset()
				Added eject ioctl
	1.02    GRG 1998.05.06  SMP spinlock changes, 
				Added slave support
	1.03    GRG 1998.06.16  Eliminate an Ugh.
	1.04	GRG 1998.08.15  Extra debugging, use HZ in loop timing
	1.05    GRG 1998.09.24  Added jumbo support

*/

#define PD_VERSION      "1.05"
#define PD_MAJOR	45
#define PD_NAME		"pd"
#define PD_UNITS	4

/* Here are things one can override from the insmod command.
   Most are autoprobed by paride unless set here.  Verbose is off
   by default.

*/

static int verbose = 0;
static int major = PD_MAJOR;
static char *name = PD_NAME;
static int cluster = 64;
static int nice = 0;
static int disable = 0;

static int drive0[8] = { 0, 0, 0, -1, 0, 1, -1, -1 };
static int drive1[8] = { 0, 0, 0, -1, 0, 1, -1, -1 };
static int drive2[8] = { 0, 0, 0, -1, 0, 1, -1, -1 };
static int drive3[8] = { 0, 0, 0, -1, 0, 1, -1, -1 };

static int (*drives[4])[8] = {&drive0, &drive1, &drive2, &drive3};
static int pd_drive_count;

enum {D_PRT, D_PRO, D_UNI, D_MOD, D_GEO, D_SBY, D_DLY, D_SLV};

/* end of parameters */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/hdreg.h>
#include <linux/cdrom.h>	/* for the eject ioctl */

#include <asm/uaccess.h>

static spinlock_t pd_lock = SPIN_LOCK_UNLOCKED;

#ifndef MODULE

#include "setup.h"

static STT pd_stt[7] = {
	{"drive0", 8, drive0},
	{"drive1", 8, drive1},
	{"drive2", 8, drive2},
	{"drive3", 8, drive3},
	{"disable", 1, &disable},
	{"cluster", 1, &cluster},
	{"nice", 1, &nice}
};

void pd_setup(char *str, int *ints)
{
	generic_setup(pd_stt, 7, str);
}

#endif

MODULE_PARM(verbose, "i");
MODULE_PARM(major, "i");
MODULE_PARM(name, "s");
MODULE_PARM(cluster, "i");
MODULE_PARM(nice, "i");
MODULE_PARM(drive0, "1-8i");
MODULE_PARM(drive1, "1-8i");
MODULE_PARM(drive2, "1-8i");
MODULE_PARM(drive3, "1-8i");

#include "paride.h"

#define MAJOR_NR   major

#include <linux/blk.h>
#include <linux/blkpg.h>

#include "pseudo.h"

#define PD_BITS    4
#define DEVICE_NR(device) (minor(device)>>PD_BITS)
#define PD_PARTNS  	(1<<PD_BITS)
#define PD_DEVS		PD_PARTNS*PD_UNITS

/* numbers for "SCSI" geometry */

#define PD_LOG_HEADS    64
#define PD_LOG_SECTS    32

#define PD_ID_OFF       54
#define PD_ID_LEN       14

#define PD_MAX_RETRIES  5
#define PD_TMO          800	/* interrupt timeout in jiffies */
#define PD_SPIN_DEL     50	/* spin delay in micro-seconds  */

#define PD_SPIN         (1000000*PD_TMO)/(HZ*PD_SPIN_DEL)

#define STAT_ERR        0x00001
#define STAT_INDEX      0x00002
#define STAT_ECC        0x00004
#define STAT_DRQ        0x00008
#define STAT_SEEK       0x00010
#define STAT_WRERR      0x00020
#define STAT_READY      0x00040
#define STAT_BUSY       0x00080

#define ERR_AMNF        0x00100
#define ERR_TK0NF       0x00200
#define ERR_ABRT        0x00400
#define ERR_MCR         0x00800
#define ERR_IDNF        0x01000
#define ERR_MC          0x02000
#define ERR_UNC         0x04000
#define ERR_TMO         0x10000

#define IDE_READ        	0x20
#define IDE_WRITE       	0x30
#define IDE_READ_VRFY		0x40
#define IDE_INIT_DEV_PARMS	0x91
#define IDE_STANDBY     	0x96
#define IDE_ACKCHANGE   	0xdb
#define IDE_DOORLOCK    	0xde
#define IDE_DOORUNLOCK  	0xdf
#define IDE_IDENTIFY    	0xec
#define IDE_EJECT		0xed

void pd_setup(char *str, int *ints);
static int pd_open(struct inode *inode, struct file *file);
static void do_pd_request(request_queue_t * q);
static int pd_ioctl(struct inode *inode, struct file *file,
		    unsigned int cmd, unsigned long arg);
static int pd_release(struct inode *inode, struct file *file);
static int pd_revalidate(kdev_t dev);
static int pd_detect(void);
static void do_pd_read(void);
static void do_pd_read_start(void);
static void do_pd_write(void);
static void do_pd_write_start(void);
static void do_pd_read_drq(void);
static void do_pd_write_done(void);

#define PD_NAMELEN	8

struct pd_unit {
	struct pi_adapter pia;	/* interface to paride layer */
	struct pi_adapter *pi;
	int access;		/* count of active opens ... */
	int capacity;		/* Size of this volume in sectors */
	int heads;		/* physical geometry */
	int sectors;
	int cylinders;
	int can_lba;
	int drive;		/* master=0 slave=1 */
	int changed;		/* Have we seen a disk change ? */
	int removable;		/* removable media device  ?  */
	int standby;
	int alt_geom;
	int present;
	char name[PD_NAMELEN];	/* pda, pdb, etc ... */
	struct gendisk *gd;
};

struct pd_unit pd[PD_UNITS];

static int pd_identify(struct pd_unit *disk);
static void pd_media_check(struct pd_unit *disk);
static void pd_doorlock(struct pd_unit *disk, int func);
static int pd_check_media(kdev_t dev);
static void pd_eject(struct pd_unit *disk);

static char pd_scratch[512];	/* scratch block buffer */

/* the variables below are used mainly in the I/O request engine, which
   processes only one request at a time.
*/

static struct pd_unit *pd_current; /* current request's drive */
static int pd_retries = 0;	/* i/o error retry count */
static int pd_busy = 0;		/* request being processed ? */
static struct request *pd_req;	/* current request */
static int pd_block;		/* address of next requested block */
static int pd_count;		/* number of blocks still to do */
static int pd_run;		/* sectors in current cluster */
static int pd_cmd;		/* current command READ/WRITE */
static char *pd_buf;		/* buffer for request in progress */

static DECLARE_WAIT_QUEUE_HEAD(pd_wait_open);

static char *pd_errs[17] = { "ERR", "INDEX", "ECC", "DRQ", "SEEK", "WRERR",
	"READY", "BUSY", "AMNF", "TK0NF", "ABRT", "MCR",
	"IDNF", "MC", "UNC", "???", "TMO"
};

/* kernel glue structures */

extern struct block_device_operations pd_fops;

static struct block_device_operations pd_fops = {
	.owner			= THIS_MODULE,
	.open			= pd_open,
	.release		= pd_release,
	.ioctl			= pd_ioctl,
	.check_media_change	= pd_check_media,
	.revalidate		= pd_revalidate
};

static void pd_init_units(void)
{
	int unit;

	pd_drive_count = 0;
	for (unit = 0; unit < PD_UNITS; unit++) {
		int *parm = *drives[unit];
		struct pd_unit *disk = pd + unit;
		disk->pi = &disk->pia;
		disk->access = 0;
		disk->changed = 1;
		disk->capacity = 0;
		disk->drive = parm[D_SLV];
		disk->present = 0;
		snprintf(disk->name, PD_NAMELEN, "%s%c", name, 'a'+unit);
		disk->alt_geom = parm[D_GEO];
		disk->standby = parm[D_SBY];
		if (parm[D_PRT])
			pd_drive_count++;
	}
}

static int pd_open(struct inode *inode, struct file *file)
{
	int unit = DEVICE_NR(inode->i_rdev);
	struct pd_unit *disk = pd + unit;

	if (unit >= PD_UNITS || !disk->present)
		return -ENODEV;

	disk->access++;

	if (disk->removable) {
		pd_media_check(disk);
		pd_doorlock(disk, IDE_DOORLOCK);
	}
	return 0;
}

static int pd_ioctl(struct inode *inode, struct file *file,
	 unsigned int cmd, unsigned long arg)
{
	struct hd_geometry *geo = (struct hd_geometry *) arg;
	struct hd_geometry g;
	struct pd_unit *disk = pd + DEVICE_NR(inode->i_rdev);

	switch (cmd) {
	case CDROMEJECT:
		if (disk->access == 1)
			pd_eject(disk);
		return 0;
	case HDIO_GETGEO:
		if (disk->alt_geom) {
			g.heads = PD_LOG_HEADS;
			g.sectors = PD_LOG_SECTS;
			g.cylinders = disk->capacity / (g.heads * g.sectors);
		} else {
			g.heads = disk->heads;
			g.sectors = disk->sectors;
			g.cylinders = disk->cylinders;
		}
		g.start = get_start_sect(inode->i_bdev);
		if (copy_to_user(geo, &g, sizeof(struct hd_geometry)))
			return -EFAULT;
		return 0;
	default:
		return -EINVAL;
	}
}

static int pd_release(struct inode *inode, struct file *file)
{
	int unit = DEVICE_NR(inode->i_rdev);
	struct pd_unit *disk = pd + unit;

	if (!--disk->access && disk->removable)
		pd_doorlock(disk, IDE_DOORUNLOCK);

	return 0;
}

static int pd_check_media(kdev_t dev)
{
	int r, unit = DEVICE_NR(dev);
	struct pd_unit *disk = pd + unit;
	if (!disk->removable)
		return 0;
	pd_media_check(disk);
	r = disk->changed;
	disk->changed = 0;
	return r;
}

static int pd_revalidate(kdev_t dev)
{
	int unit = DEVICE_NR(dev);
	struct pd_unit *disk = pd + unit;
	if (pd_identify(disk))
		set_capacity(disk->gd, disk->capacity);
	else
		set_capacity(disk->gd, 0);
	return 0;
}

static inline int status_reg(struct pd_unit *disk)
{
	return pi_read_regr(disk->pi, 1, 6);
}

static inline int read_reg(struct pd_unit *disk, int reg)
{
	return pi_read_regr(disk->pi, 0, reg);
}

static inline void write_status(struct pd_unit *disk, int val)
{
	pi_write_regr(disk->pi, 1, 6, val);
}

static inline void write_reg(struct pd_unit *disk, int reg, int val)
{
	pi_write_regr(disk->pi, 0, reg, val);
}

static inline u8 DRIVE(struct pd_unit *disk)
{
	return 0xa0+0x10*disk->drive;
}

/*  ide command interface */

static void pd_print_error(struct pd_unit *disk, char *msg, int status)
{
	int i;

	printk("%s: %s: status = 0x%x =", disk->name, msg, status);
	for (i = 0; i < 18; i++)
		if (status & (1 << i))
			printk(" %s", pd_errs[i]);
	printk("\n");
}

static void pd_reset(struct pd_unit *disk)
{				/* called only for MASTER drive */
	pi_connect(disk->pi);
	write_status(disk, 4);
	udelay(50);
	write_status(disk, 0);
	pi_disconnect(disk->pi);
	udelay(250);
}

#define DBMSG(msg)	((verbose>1)?(msg):NULL)

static int pd_wait_for(struct pd_unit *disk, int w, char *msg)
{				/* polled wait */
	int k, r, e;

	k = 0;
	while (k < PD_SPIN) {
		r = status_reg(disk);
		k++;
		if (((r & w) == w) && !(r & STAT_BUSY))
			break;
		udelay(PD_SPIN_DEL);
	}
	e = (read_reg(disk, 1) << 8) + read_reg(disk, 7);
	if (k >= PD_SPIN)
		e |= ERR_TMO;
	if ((e & (STAT_ERR | ERR_TMO)) && (msg != NULL))
		pd_print_error(disk, msg, e);
	return e;
}

static void pd_send_command(struct pd_unit *disk, int n, int s, int h, int c0, int c1, int func)
{
	write_reg(disk, 6, DRIVE(disk) + h);
	write_reg(disk, 1, 0);		/* the IDE task file */
	write_reg(disk, 2, n);
	write_reg(disk, 3, s);
	write_reg(disk, 4, c0);
	write_reg(disk, 5, c1);
	write_reg(disk, 7, func);

	udelay(1);
}

static void pd_ide_command(struct pd_unit *disk, int func, int block, int count)
{
	int c1, c0, h, s;

	if (disk->can_lba) {
		s = block & 255;
		c0 = (block >>= 8) & 255;
		c1 = (block >>= 8) & 255;
		h = ((block >>= 8) & 15) + 0x40;
	} else {
		s = (block % disk->sectors) + 1;
		h = (block /= disk->sectors) % disk->heads;
		c0 = (block /= disk->heads) % 256;
		c1 = (block >>= 8);
	}
	pd_send_command(disk, count, s, h, c0, c1, func);
}

/* According to the ATA standard, the default CHS geometry should be
   available following a reset.  Some Western Digital drives come up
   in a mode where only LBA addresses are accepted until the device
   parameters are initialised.
*/

static void pd_init_dev_parms(struct pd_unit *disk)
{
	pi_connect(disk->pi);
	pd_wait_for(disk, 0, DBMSG("before init_dev_parms"));
	pd_send_command(disk, disk->sectors, 0, disk->heads - 1, 0, 0,
			IDE_INIT_DEV_PARMS);
	udelay(300);
	pd_wait_for(disk, 0, "Initialise device parameters");
	pi_disconnect(disk->pi);
}

static void pd_doorlock(struct pd_unit *disk, int func)
{
	pi_connect(disk->pi);
	if (!(pd_wait_for(disk, STAT_READY, "Lock") & STAT_ERR)) {
		pd_send_command(disk, 1, 0, 0, 0, 0, func);
		pd_wait_for(disk, STAT_READY, "Lock done");
	}
	pi_disconnect(disk->pi);
}

static void pd_eject(struct pd_unit *disk)
{
	pi_connect(disk->pi);
	pd_wait_for(disk, 0, DBMSG("before unlock on eject"));
	pd_send_command(disk, 1, 0, 0, 0, 0, IDE_DOORUNLOCK);
	pd_wait_for(disk, 0, DBMSG("after unlock on eject"));
	pd_wait_for(disk, 0, DBMSG("before eject"));
	pd_send_command(disk, 0, 0, 0, 0, 0, IDE_EJECT);
	pd_wait_for(disk, 0, DBMSG("after eject"));
	pi_disconnect(disk->pi);
}

static void pd_media_check(struct pd_unit *disk)
{
	int r;

	pi_connect(disk->pi);
	r = pd_wait_for(disk, STAT_READY, DBMSG("before media_check"));
	if (!(r & STAT_ERR)) {
		pd_send_command(disk, 1, 1, 0, 0, 0, IDE_READ_VRFY);
		r = pd_wait_for(disk, STAT_READY, DBMSG("RDY after READ_VRFY"));
	} else
		disk->changed = 1;	/* say changed if other error */
	if (r & ERR_MC) {
		disk->changed = 1;
		pd_send_command(disk, 1, 0, 0, 0, 0, IDE_ACKCHANGE);
		pd_wait_for(disk, STAT_READY, DBMSG("RDY after ACKCHANGE"));
		pd_send_command(disk, 1, 1, 0, 0, 0, IDE_READ_VRFY);
		r = pd_wait_for(disk, STAT_READY, DBMSG("RDY after VRFY"));
	}
	pi_disconnect(disk->pi);

}

static void pd_standby_off(struct pd_unit *disk)
{
	pi_connect(disk->pi);
	pd_wait_for(disk, 0, DBMSG("before STANDBY"));
	pd_send_command(disk, 0, 0, 0, 0, 0, IDE_STANDBY);
	pd_wait_for(disk, 0, DBMSG("after STANDBY"));
	pi_disconnect(disk->pi);
}

static int pd_identify(struct pd_unit *disk)
{
	int j;
	char id[PD_ID_LEN + 1];

/* WARNING:  here there may be dragons.  reset() applies to both drives,
   but we call it only on probing the MASTER. This should allow most
   common configurations to work, but be warned that a reset can clear
   settings on the SLAVE drive.
*/

	if (disk->drive == 0)
		pd_reset(disk);

	pi_connect(disk->pi);
	write_reg(disk, 6, DRIVE(disk));
	pd_wait_for(disk, 0, DBMSG("before IDENT"));
	pd_send_command(disk, 1, 0, 0, 0, 0, IDE_IDENTIFY);

	if (pd_wait_for(disk, STAT_DRQ, DBMSG("IDENT DRQ")) & STAT_ERR) {
		pi_disconnect(disk->pi);
		return 0;
	}
	pi_read_block(disk->pi, pd_scratch, 512);
	pi_disconnect(disk->pi);
	disk->can_lba = pd_scratch[99] & 2;
	disk->sectors = le16_to_cpu(*(u16 *) (pd_scratch + 12));
	disk->heads = le16_to_cpu(*(u16 *) (pd_scratch + 6));
	disk->cylinders = le16_to_cpu(*(u16 *) (pd_scratch + 2));
	if (disk->can_lba)
		disk->capacity = le32_to_cpu(*(u32 *) (pd_scratch + 120));
	else
		disk->capacity = disk->sectors * disk->heads * disk->cylinders;

	for (j = 0; j < PD_ID_LEN; j++)
		id[j ^ 1] = pd_scratch[j + PD_ID_OFF];
	j = PD_ID_LEN - 1;
	while ((j >= 0) && (id[j] <= 0x20))
		j--;
	j++;
	id[j] = 0;

	disk->removable = pd_scratch[0] & 0x80;

	printk("%s: %s, %s, %d blocks [%dM], (%d/%d/%d), %s media\n",
	       disk->name, id,
	       disk->drive ? "slave" : "master",
	       disk->capacity, disk->capacity / 2048,
	       disk->cylinders, disk->heads, disk->sectors,
	       disk->removable ? "removable" : "fixed");

	if (disk->capacity)
		pd_init_dev_parms(disk);
	if (!disk->standby)
		pd_standby_off(disk);

	return 1;
}

static int pd_probe_drive(struct pd_unit *disk)
{
	if (disk->drive == -1) {
		for (disk->drive = 0; disk->drive <= 1; disk->drive++)
			if (pd_identify(disk))
				return 1;
		return 0;
	}
	return pd_identify(disk);
}

static int pd_detect(void)
{
	int k, unit;
	struct pd_unit *disk;

	k = 0;
	if (pd_drive_count == 0) { /* nothing spec'd - so autoprobe for 1 */
		disk = pd;
		if (pi_init(disk->pi, 1, -1, -1, -1, -1, -1, pd_scratch,
			    PI_PD, verbose, disk->name)) {
			if (pd_probe_drive(disk)) {
				disk->present = 1;
				k = 1;
			} else
				pi_release(disk->pi);
		}

	} else {
		for (unit = 0, disk = pd; unit < PD_UNITS; unit++, disk++) {
			int *parm = *drives[unit];
			if (!parm[D_PRT])
				continue;
			if (pi_init(disk->pi, 0, parm[D_PRT], parm[D_MOD],
				     parm[D_UNI], parm[D_PRO], parm[D_DLY],
				     pd_scratch, PI_PD, verbose, disk->name)) {
				if (pd_probe_drive(disk)) {
					disk->present = 1;
					k = unit + 1;
				} else
					pi_release(disk->pi);
			}
		}
	}
	for (unit = 0, disk = pd; unit < PD_UNITS; unit++, disk++) {
		if (disk->present) {
			struct gendisk *p = alloc_disk();
			if (!p) {
				disk->present = 0;
				k--;
				continue;
			}
			strcpy(p->disk_name, disk->name);
			p->minor_shift = PD_BITS;
			p->fops = &pd_fops;
			p->major = major;
			p->first_minor = unit << PD_BITS;
			set_capacity(p, disk->capacity);
			disk->gd = p;
			add_disk(p);
		}
	}
	if (k)
		return 1;
	printk("%s: no valid drive found\n", name);
	return 0;
}

/* The i/o request engine */

static int pd_ready(void)
{
	return !(status_reg(pd_current) & STAT_BUSY);
}

static void do_pd_request(request_queue_t * q)
{
	int unit;

	if (pd_busy)
		return;
repeat:
	if (blk_queue_empty(QUEUE))
		return;

	pd_req = elv_next_request(QUEUE);
	unit = DEVICE_NR(pd_req->rq_dev);
	if (unit >= PD_UNITS) {
		end_request(pd_req, 0);
		goto repeat;
	}

	pd_block = pd_req->sector;
	pd_run = pd_req->nr_sectors;
	pd_count = pd_req->current_nr_sectors;
	pd_current = pd + unit;
	if (pd_block + pd_count > get_capacity(pd_current->gd)) {
		end_request(pd_req, 0);
		goto repeat;
	}

	pd_cmd = rq_data_dir(pd_req);
	pd_buf = pd_req->buffer;
	pd_retries = 0;

	pd_busy = 1;
	if (pd_cmd == READ)
		pi_do_claimed(pd_current->pi, do_pd_read);
	else if (pd_cmd == WRITE)
		pi_do_claimed(pd_current->pi, do_pd_write);
	else {
		pd_busy = 0;
		end_request(pd_req, 0);
		goto repeat;
	}
}

static int pd_next_buf(void)
{
	long saved_flags;

	pd_count--;
	pd_run--;
	pd_buf += 512;
	pd_block++;
	if (!pd_run)
		return 1;
	if (pd_count)
		return 0;
	spin_lock_irqsave(&pd_lock, saved_flags);
	end_request(pd_req, 1);
	pd_count = pd_req->current_nr_sectors;
	pd_buf = pd_req->buffer;
	spin_unlock_irqrestore(&pd_lock, saved_flags);
	return 0;
}

static inline void next_request(int success)
{
	long saved_flags;

	spin_lock_irqsave(&pd_lock, saved_flags);
	end_request(pd_req, success);
	pd_busy = 0;
	do_pd_request(NULL);
	spin_unlock_irqrestore(&pd_lock, saved_flags);
}

static void do_pd_read(void)
{
	ps_set_intr(do_pd_read_start, 0, 0, nice);
}

static void do_pd_read_start(void)
{
	pd_busy = 1;

	pi_connect(pd_current->pi);
	if (pd_wait_for(pd_current, STAT_READY, "do_pd_read") & STAT_ERR) {
		pi_disconnect(pd_current->pi);
		if (pd_retries < PD_MAX_RETRIES) {
			pd_retries++;
			pi_do_claimed(pd_current->pi, do_pd_read_start);
			return;
		}
		next_request(0);
		return;
	}
	pd_ide_command(pd_current, IDE_READ, pd_block, pd_run);
	ps_set_intr(do_pd_read_drq, pd_ready, PD_TMO, nice);
}

static void do_pd_read_drq(void)
{
	while (1) {
		if (pd_wait_for(pd_current, STAT_DRQ, "do_pd_read_drq") & STAT_ERR) {
			pi_disconnect(pd_current->pi);
			if (pd_retries < PD_MAX_RETRIES) {
				pd_retries++;
				pi_do_claimed(pd_current->pi, do_pd_read_start);
				return;
			}
			next_request(0);
			return;
		}
		pi_read_block(pd_current->pi, pd_buf, 512);
		if (pd_next_buf())
			break;
	}
	pi_disconnect(pd_current->pi);
	next_request(1);
}

static void do_pd_write(void)
{
	ps_set_intr(do_pd_write_start, 0, 0, nice);
}

static void do_pd_write_start(void)
{
	pd_busy = 1;

	pi_connect(pd_current->pi);
	if (pd_wait_for(pd_current, STAT_READY, "do_pd_write") & STAT_ERR) {
		pi_disconnect(pd_current->pi);
		if (pd_retries < PD_MAX_RETRIES) {
			pd_retries++;
			pi_do_claimed(pd_current->pi, do_pd_write_start);
			return;
		}
		next_request(0);
		return;
	}
	pd_ide_command(pd_current, IDE_WRITE, pd_block, pd_run);
	while (1) {
		if (pd_wait_for(pd_current, STAT_DRQ, "do_pd_write_drq") & STAT_ERR) {
			pi_disconnect(pd_current->pi);
			if (pd_retries < PD_MAX_RETRIES) {
				pd_retries++;
				pi_do_claimed(pd_current->pi, do_pd_write_start);
				return;
			}
			next_request(0);
			return;
		}
		pi_write_block(pd_current->pi, pd_buf, 512);
		if (pd_next_buf())
			break;
	}
	ps_set_intr(do_pd_write_done, pd_ready, PD_TMO, nice);
}

static void do_pd_write_done(void)
{
	if (pd_wait_for(pd_current, STAT_READY, "do_pd_write_done") & STAT_ERR) {
		pi_disconnect(pd_current->pi);
		if (pd_retries < PD_MAX_RETRIES) {
			pd_retries++;
			pi_do_claimed(pd_current->pi, do_pd_write_start);
			return;
		}
		next_request(0);
		return;
	}
	pi_disconnect(pd_current->pi);
	next_request(1);
}

static int __init pd_init(void)
{
	request_queue_t *q;
	if (disable)
		return -1;
	if (register_blkdev(MAJOR_NR, name, &pd_fops)) {
		printk("%s: unable to get major number %d\n", name, major);
		return -1;
	}
	q = BLK_DEFAULT_QUEUE(MAJOR_NR);
	blk_init_queue(q, do_pd_request, &pd_lock);
	blk_queue_max_sectors(q, cluster);

	printk("%s: %s version %s, major %d, cluster %d, nice %d\n",
	       name, name, PD_VERSION, major, cluster, nice);
	pd_init_units();
	if (!pd_detect()) {
		unregister_blkdev(MAJOR_NR, name);
		return -1;
	}
	return 0;
}

static void __exit pd_exit(void)
{
	struct pd_unit *disk;
	int unit;
	unregister_blkdev(MAJOR_NR, name);
	for (unit = 0, disk = pd; unit < PD_UNITS; unit++, disk++) {
		if (disk->present) {
			struct gendisk *p = disk->gd;
			disk->gd = NULL;
			del_gendisk(p);
			put_disk(p);
			pi_release(disk->pi);
		}
	}
}

MODULE_LICENSE("GPL");
module_init(pd_init)
module_exit(pd_exit)
