/*
 * Copyright (c) 1983, 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
static char sccsid[] = "@(#)syslog.c	8.4 (Berkeley) 3/18/94";
#endif /* LIBC_SCCS and not lint */

/*
 * SYSLOG -- print message on log file
 *
 * This routine looks a lot like printf, except that it outputs to the
 * log file instead of the standard output.  Also:
 *	adds a timestamp,
 *	prints the module name in front of the message,
 *	has some other formatting types (or will sometime),
 *	adds a newline on the end of the message.
 *
 * The output of this routine is intended to be read by syslogd(8).
 *
 * Author: Eric Allman
 * Modified to use UNIX domain IPC by Ralph Campbell
 * Patched March 12, 1996 by A. Ian Vogelesang <vogelesang@hdshq.com>
 *  - to correct the handling of message & format string truncation,
 *  - to visibly tag truncated records to facilitate
 *    investigation of such Bad Things with grep, and,
 *  - to correct the handling of case where "write"
 *    returns after writing only part of the message.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/signal.h>
#include <sys/syslog.h>
#if 0
#include "syslog.h"
#include "pathnames.h"
#endif

#include <sys/uio.h>
#include <sys/wait.h>
#include <netdb.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include <paths.h>
#include <stdio.h>
#include <ctype.h>
#include <signal.h>


#if defined(_REENTRENT) || defined(_THREAD_SAFE)
# include <pthread.h>

extern int __writev( int, const struct iovec *, size_t);

/* We need to initialize the mutex.  For this we use a method provided
   by pthread function 'pthread_once'.  For this we need a once block.  */
static pthread_once__t _once_block = pthread_once_init;

/* This is the mutex which protects the global environment of simultaneous
   modifications.  */
static pthread_mutex_t _syslog_mutex;

static void
DEFUN_VOID(_init_syslog_mutex)
{
  pthread_mutex_init(&_syslog_mutex, pthread_mutexattr_default);
}

# define LOCK() \
   do { pthread_once(&_once_block, _init_syslog_mutex);
        pthread_mutex_lock(&_syslog_mutex); } while (0)
# define UNLOCK() pthread_mutex_unlock(&_syslog_mutex)

#else /* !_REENTRENT && !_THREAD_SAFE */

# define LOCK()
# define UNLOCK()

#endif /* _REENTRENT || _THREAD_SAFE */


static int	LogFile = -1;		/* fd for log */
static int	connected;		/* have done connect */
static int	LogStat = 0;		/* status bits, set by openlog() */
static const char *LogTag = "syslog";	/* string to tag the entry with */
static int	LogFacility = LOG_USER;	/* default facility code */
static int	LogMask = 0xff;		/* mask of priorities to be logged */

/* #define TRUNCATED "[truncated by syslog.c] "
*/
#define TRUNCATED "[truncated] " /* AIV */

static void closelog_intern( int );
void syslog( int, const char *, ...);
void vsyslog( int, const char *, va_list );
void openlog( const char *, int, int );
void closelog( void );
int setlogmask( int );
static void prepend(char *buf, int bufsize, char *tag);

static void 
closelog_intern(int to_default)
{
	LOCK();
	(void) close(LogFile);
	LogFile = -1;
	connected = 0;
	if (to_default)
	{
		LogStat = 0;
		LogTag = "syslog";
		LogFacility = LOG_USER;
		LogMask = 0xff;
	}
	UNLOCK();
}

static void
sigpipe_handler (int sig)
{
  closelog_intern (0);
}

/*
 * syslog, vsyslog --
 *     print message on log file; output is intended for syslogd(8).
 */
void
syslog(int pri, const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	vsyslog(pri, fmt, ap);
	va_end(ap);
}

void
vsyslog( int pri, const char *fmt, va_list ap )
{
	register int cnt;
	register char *p;
	char *last_chr;
	time_t now;
	int fd, saved_errno;
	int sz, rc;
	char tbuf[2048], fmt_cpy[1024], *stdp;

	struct sigaction action, oldaction;
	int sigpipe;
	memset (&action, 0, sizeof (action));
	action.sa_handler = sigpipe_handler;
	sigemptyset (&action.sa_mask);
	sigpipe = sigaction (SIGPIPE, &action, &oldaction);

	saved_errno = errno;

	LOCK();

	/* See if we should just throw out this message. */
	if (!(LogMask & LOG_MASK(LOG_PRI(pri))) || (pri &~ (LOG_PRIMASK|LOG_FACMASK)))
	{
		goto getout;
	}
	if (LogFile < 0 || !connected)
		openlog(LogTag, LogStat | LOG_NDELAY, 0);

	/* Set default facility if none specified. */
	if ((pri & LOG_FACMASK) == 0)
		pri |= LogFacility;

	/* Build the message. */
	(void)time(&now);
	(void)snprintf(tbuf, sizeof(tbuf), "<%d>%.15s ", pri, ctime(&now) + 4);
	stdp = "";
	for (p = tbuf; *p; ++p);
	if (LogStat & LOG_PERROR)
		stdp = p;
	if (LogTag) {
		(void)strcpy(p, LogTag);
		for (; *p; ++p);
	}
	if (LogStat & LOG_PID) {
		(void)snprintf(p, sizeof(tbuf)-(p-tbuf), "[%d]", getpid());
		for (; *p; ++p);
	}
	if (LogTag) {
		*p++ = ':';
		*p++ = ' ';
		*p = '\0';
	}

	/* Substitute error message for %m. */
	{
		register char ch, *t1;

		/* We have to make sure we don't overrun fmt_cpy. */
		for (t1 = fmt_cpy; ((ch=*fmt)!='\0') && (t1<(fmt_cpy+(sizeof(fmt_cpy)-1))); ++fmt) {
			if (ch == '%' && fmt[1] == '%') {
				if (t1<(fmt_cpy+(sizeof(fmt_cpy)-2))) {
					*t1++='%'; *t1++='%'; fmt++;
					continue;
				} else {
					/* flag message as truncated to alert the good guys */
					prepend(tbuf, sizeof(tbuf), TRUNCATED); 
					for (; *p; ++p);
					break;
				}
			} 
			if (ch == '%' && fmt[1] == 'm') {
				++fmt;
				sz=sizeof(fmt_cpy)-(t1-fmt_cpy);
				rc=snprintf(t1, sz, "%s", strerror(saved_errno));
				if ((-1==rc) || (rc>sz)) {
					/* flag message as truncated to alert the good guys */
					prepend(tbuf, sizeof(tbuf), TRUNCATED); 
					for (; *p; ++p);
					for (; *t1; t1++);
					break;
				}
				for (; *t1; t1++);
			}
			else
				*t1++ = ch;
		}
		*t1 = '\0';
		if (ch!=0) {
			/* flag message as truncated to alert the good guys */
			prepend(tbuf, sizeof(tbuf), TRUNCATED); 
			for (; *p; ++p);
		}
	}

	sz=sizeof(tbuf)-(p-tbuf);
	rc=vsnprintf(p, sz, fmt_cpy, ap);
	if ((-1==rc) || (rc>sz)) {
		/* flag message as truncated to alert the good guys */
		prepend(tbuf, sizeof(tbuf), TRUNCATED); 
	}
	for(; *p; p++);
	cnt = p - tbuf;

	/* Output to stderr if requested. */
	if (LogStat & LOG_PERROR) {
		struct iovec iov[2];
		register struct iovec *v = iov;

		v->iov_base = stdp;
		v->iov_len = cnt - (stdp - tbuf);
		++v;
		v->iov_base = "\n";
		v->iov_len = 1;
		(void)writev(STDERR_FILENO, iov, 2);
	}

	/* Output the message to the local logger. */
#if 1
	/* Use NUL as a message delimiter. */
	p=tbuf; last_chr=tbuf+cnt;
	do {
		rc=write(LogFile, p, ((last_chr-p)+1));
		if (rc<0) {
			if ((errno==EAGAIN) || (errno==EINTR))
				rc=0;
			else {
				closelog_intern(0);
				break;
			}
		}
		p+=rc;
	} while (p<=last_chr);
	if (p>last_chr) 
	{
		goto getout;
	}
#else
	if (write(LogFile, tbuf, cnt) >= 0) {
		goto getout;
	}
#endif
	/*
	 * Output the message to the console; don't worry about blocking,
	 * if console blocks everything will.  Make sure the error reported
	 * is the one from the syslogd failure.
	 */
	/* should mode be `O_WRONLY | O_NOCTTY' ? -- Uli */
	if (LogStat & LOG_CONS &&
	    (fd = open(_PATH_CONSOLE, O_WRONLY, 0)) >= 0) {
		(void)strcat(tbuf, "\r\n");
		cnt += 2;
		p = index(tbuf, '>') + 1;
		(void)write(fd, p, cnt - (p - tbuf));
		(void)close(fd);
	}

getout:
	UNLOCK();
	if (sigpipe == 0)
		sigaction (SIGPIPE, &oldaction,
			(struct sigaction *) NULL);
}

static struct sockaddr SyslogAddr;	/* AF_UNIX address of local logger */
/*
 * OPENLOG -- open system log
 */
void
openlog( const char *ident, int logstat, int logfac )
{
        LOCK();

	if (ident != NULL)
		LogTag = ident;
	LogStat = logstat;
	if (logfac != 0 && (logfac &~ LOG_FACMASK) == 0)
		LogFacility = logfac;
	if (LogFile == -1) {
		SyslogAddr.sa_family = AF_UNIX;
		(void)strncpy(SyslogAddr.sa_data, _PATH_LOG,
		    sizeof(SyslogAddr.sa_data));
		if (LogStat & LOG_NDELAY) {
		        if ((LogFile = socket(AF_UNIX, SOCK_STREAM, 0)) == -1){
			        UNLOCK();
				return;
			}
/*			fcntl(LogFile, F_SETFD, 1); */
		}
	}
	if (LogFile != -1 && !connected &&
#if 0
	    connect(LogFile, &SyslogAddr, sizeof(SyslogAddr.sa_family)+
			strlen(SyslogAddr.sa_data)) != -1)
#else
	    connect(LogFile, &SyslogAddr, sizeof(SyslogAddr) -
			sizeof(SyslogAddr.sa_data) +
			strlen(SyslogAddr.sa_data)) != -1)
#endif
		connected = 1;

        UNLOCK();
}

/*
 * CLOSELOG -- close the system log
 */
void
closelog( void )
{
	closelog_intern(1);
}

/*
 * SETLOGMASK -- set the log mask level
 */
int
setlogmask( int pmask )
{
	int omask;

	omask = LogMask;
	LOCK();
	if (pmask != 0)
		LogMask = pmask;
	UNLOCK();
	return (omask);
}

static void prepend(char *buf, int bufsize, char *tag)
{

	/* Prepends "tag" to the beginning of the data in "buf" 
	   right after the priority (if present) and the timestamp
	   (if present) shifting existing contents to the right 
	   to make room, and truncating any characters which may 
	   get shifted off the end of "buf". If the tag is already
	   there, we return without doing anything.
	   A. Ian Vogelesang <vogelesang@hdshq.com> March 7/96 */

	/* bufsize is total size of buf, including terminating null */
	/* tag must be null-terminated string */

	int taglen,bufleft;
	char *p;
	register char *q;

	if ((NULL==buf) || (NULL==tag)) return;
	taglen=strlen(tag);
	if (0==taglen) return;

	p=buf;

	if ((p[0]=='<') && isdigit(p[1])) {
		if (p[2]=='>') p+=3;
		else if (isdigit(p[2]))
			if (p[3]=='>') p+=4;
			else if (isdigit(p[3]) && (p[4]=='>')) p+=5;
	}
	if (isalpha(p[0]) && isalpha(p[1]) && isalpha(p[2]) && \
	    (' '==p[3]) && ((' '==p[4]) || isdigit(p[4])) && isdigit(p[5]) && \
	    (' '==p[6]) && isdigit(p[7]) && isdigit(p[8]) && \
	    (':'==p[9]) && isdigit(p[10]) && isdigit(p[11]) && \
	    (':'==p[12]) && isdigit(p[13]) && isdigit(p[14]) && \
	    (' '==p[15]) ) p+=16;
	bufleft=bufsize-(p-buf);
	if (taglen>=(bufleft-1)) {
		bcopy(tag,p,(bufleft-1));
		*(p+(bufleft-1))='\0';
		return;
	}
	if (0==bcmp(p,tag,taglen)) return;
	*(p+(bufleft-1)-taglen)='\0';
	for (q=index(p,'\0'); q>=p; q--) *(q+taglen)=*q;
	bcopy(tag,p,taglen);
	return;
}
