/*
 * $Id$
 *
 * Copyright (c) 1997 Andrew G. Morgan <morgan@parc.power.net>
 *
 * This file contains the memory management code for the user-space
 * audit subsystem
 */

#include <sys/audit.h>
#include "audit.h"

/*
 * This is an internal function used to provide memory for
 * other functions in this library.  New memory is labelled.
 * Note that errno is set by malloc() if this fails.
 */

#ifdef DEBUG
static int blks_out=0;
#endif

void *__aud_malloc(int size, int type)
{
    struct purpose *new_mem;

    if (size <= 0)
	return NULL;

    new_mem = malloc(sizeof(*new_mem) + size);
    if (new_mem) {
	new_mem->magic = __AUD_MAGIC_PURPOSE_;
	new_mem->flavor = __AUD_FLAVOR_DYNAMIC | type;
	new_mem->length = size;
	new_mem->data = ++new_mem;
#ifdef DEBUG
	fprintf(stderr, "[=> allocated: %d blocks ]\n", ++blks_out);
#endif
    }

    return new_mem;
}

/*
 * This, internal function can be used to verify that this memory
 * was provided by the library.  It is also able to lose the
 * 'const' cast (unlocking the memory from the compiler)
 */

void *__aud_lookup(const void *old)
{
    const struct purpose *this = (const struct purpose *) old;

    if (this && (--this)->magic == __AUD_MAGIC_PURPOSE_) {
	void *data = this->data;

	if (old == data)
	    return data;
    }

    return NULL;
}

/*
 * This function performs three tasks.
 * - liberates memory allocated by the library
 * - differentiates between static and dynamic memory
 * - scrubs dynamic memory before properly free()'ing it
 */

int __aud_free(const void *old)
{
    struct purpose *old_p = __aud_lookup(old);

    if (old_p == NULL) {
	return 0;
    } else if ((--old_p)->magic == __AUD_MAGIC_PURPOSE_) {
	if ((old_p)->flavor & __AUD_FLAVOR_DYNAMIC) {
	    memset(old_p, 0, sizeof(*old_p) + old_p->length);	   /* Scrub */
	    free(old_p);	                                  /* Delete */
#ifdef DEBUG
	    fprintf(stderr, "[freed a block: %d remaining]\n", --blks_out);
#endif
	    return 1;      /* free()'d something */
	}
	return 0;      /* memory was static */
    } else {
	/* this memory was not allocated by this library */
	errno = EINVAL;
	return -1;
    }
}

/*
 * This is used to delete the section lists of the audit record
 */

static void __aud_free_section(struct __aud_section_s *section)
{
    while (section != NULL) {
	struct __aud_info_s *info;
	struct __aud_section_s *this = section;

	/* identify any subsequent section */
	section = section->after;

	/* is there a list of info items to delete? */
	for (info=this->data; info; info=this->data) {
	    this->data = info->next;

	    /* was there some data? */
	    if (info->datum.aud_info_p) {
		__aud_free(info->datum.aud_info_p);
	    }

	    /* free the information item carrier */
	    __aud_free(info);
	}

	/* liberate this section */
	__aud_free(this);
    }
}

/* p126 - liberate memory used to store an audit object */

/* XXX - our memory model is a little more fussy than the one defined
   by POSIX. Basically, the library malloc()s objects with a header
   indicating the size and location (and type?) of the block.  In this
   way we can self-index objects and also ensure that they are
   scrubbed before being liberated.  It also means that all
   audit-generated objects are typed as 'const' to make it (slightly)
   difficult for programmers to accidentally change them outside the
   official API. Another side effect is that the library can
   transparently deal with objects that are actually malloc()'d and
   also objects that are in static memory. */

int aud_free(const void *obj_p)
{
    struct purpose *full_p = __aud_lookup(obj_p);

    if (full_p != obj_p) {
	errno = EINVAL;
	return -1;
    }

    /* look up the flavor of the object and deal with its children
       then return here to finally deal with this object. */

    if (full_p) {
	const struct __aud_rec_s *record = (struct __aud_rec_s *) obj_p;

	switch ((full_p-1)->flavor & ~__AUD_FLAVOR_DYNAMIC) {
	case __AUD_MAGIC_RECORD:
	    /* liberate the sections */
	    __aud_free_section(record->event);
	    __aud_free_section(record->header);
	    __aud_free_section(record->object);
	    __aud_free_section(record->subject);
	    break;
	case __AUD_MAGIC_DATA:
	    break;
	default:
	    errno = EINVAL;
	    return -1;
	}
    }

    switch (__aud_free(obj_p)) {
    case 0:
    case 1:
	errno = 0;
	return 0;
    case -1:
	errno = EINVAL;
	return -1;
    deafult:
	fprintf(stderr, "internal error: " __FILE__ " line %d\n", __LINE__);
	errno = EINVAL;
	return -1;
    }
}

/*
 * $Log$
 */
