/*
 * $Id$
 *
 * Copyright (c) 1997 Andrew G. Morgan <morgan@parc.power.net>
 *
 * This file contains the functions that manipulate entries of
 * an audit record.  It is a generic abstraction of the POSIX functions
 * and is designed for internal use only.
 */

#include "audit.h"

/* This function is a front end to aud_delete_*() */

int __aud_delete_section(int magic, const void *section_d)
{
    struct __aud_info_s *info;
    struct __aud_section_s *sect = __aud_lookup(section_d);

    /* verify that this is a known structure and it is what we think
       it is */

    if (sect == NULL || sect->magic != magic) {
	errno = EINVAL;
	return -1;
    }

    /* verify that it is attatched to something */

    if (sect->before == NULL) {
	fprintf(stderr, "internal error: " __FILE__ " line %d: %d\n",
		__LINE__, magic);
	exit(-1);
    }

    /* link previous to next */

    if (sect->after) {
	sect->after->before = sect->before;
    }
    *(sect->before) = sect->after;

    /* we are ready to delete the linked list of data items now */

    while ((info = sect->data)) {
	if (info->magic != __AUD_MAGIC_INFO) {
	    fprintf(stderr, "internal error " __FILE__ " line %d\n", __LINE__);
	    exit(1);
	}
	sect->data = info->next;
	(void) __aud_free(info->datum.aud_info_p);
	(void) __aud_free(info);
    }

    /* finally delete sect */

    --*(sect->owner_count);
    (void) __aud_free(sect);
    
    /* we're done */

    return 0;
}

/* This function is a front end to aud_delete_*_info() */

int __aud_delete_section_info(int magic, const void *section_d, int item_id)
{
    struct __aud_info_s **parent_holder, *info;
    struct __aud_section_s *sect = __aud_lookup(section_d);

    /* first verify that we know the sect pointer */
    if (sect == NULL || sect->magic != magic) {
	errno = EINVAL;
	return -1;
    }

    /* next locate the info item to be deleted (see item_id in
       <sys/audit.h>) */

    for (parent_holder = &(sect->data);
	 (info = *parent_holder); parent_holder = &(info->next)) {

	if (info->magic != __AUD_MAGIC_INFO) {
	    fprintf(stderr, "internal error " __FILE__ " line %d\n", __LINE__);
	    exit(1);
	}
	if (info->item_id == item_id) {
	    /* link up the next item to preserve any linked list (at
               this point we are committed to return success - eg.
               24.4.4.3) */
	    *parent_holder = info->next;
	    /* delete item */
	    sect->count--;
	    (void) __aud_free(info->datum.aud_info_p);
	    (void) __aud_free(info);

	    return 0;
	}
    }

    /* we didn't find it */

    errno = EINVAL;
    return -1;
}

static int ___locate_sect(int magic, struct __aud_rec_s *record,
			  struct __aud_section_s ***sect_p, int **owner_count)
{
    /* which section are we interested in? */

    switch (magic) {
    case __AUD_MAGIC_EVINFO:
	*sect_p = &(record->event);
	*owner_count = &(record->event_count);
	break;
    case __AUD_MAGIC_HDR:
	*sect_p = &(record->header);
	*owner_count = &(record->header_count);
	break;
    case __AUD_MAGIC_OBJ:
	*sect_p = &(record->object);
	*owner_count = &(record->object_count);
	break;
    case __AUD_MAGIC_SUBJ:
	*sect_p = &(record->subject);
	*owner_count = &(record->subject_count);
	break;
    default:
	fprintf(stderr, __FILE__ " unknown request (%d) on line %d\n",
		magic, __LINE__);
	errno = EINVAL;
	return 0;
    }

    return 1;
}

/* this is a generic backend to the aud_get_*() functions */

int __aud_get_section(int magic, aud_rec_t ar, int index,
		      const void **section_p)
{
    int i, *counter;
    struct __aud_rec_s *record = __aud_lookup(ar);
    struct __aud_section_s **sect_holder, *sect;

    /* check if we recognize the record */
    if (record == NULL || record->magic != __AUD_MAGIC_RECORD) {
	errno = EINVAL;
	return -1;
    }

    /* obtain the first section identifier */
    if (!___locate_sect(magic, record, &sect_holder, &counter)) {
	return -1;
    }

    /* return the number of sections */
    if (section_p == NULL) {
	return *counter;
    }

    /* locate required event entry */
    for (i=index, sect=*sect_holder; sect && i-- > 0; sect=sect->after) {
	if (sect->magic != magic) {
	    fprintf(stderr, "internal error " __FILE__ " line %d\n", __LINE__);
	    exit(1);
	}
	if (i==0) {
	    sect->prev_item_id = 0;         /* no previous info requests */
	    *section_p = sect;
	    return *counter;    /* number of sects in record */
	}
    }

    /* not found */
    errno = EINVAL;
    return -1;
}

/* p130 - get information from an event - returns non-negative number
   of items in the event specific data. */

int __aud_get_section_info(int magic, const void *section_d, int item_id,
			   const aud_info_t **aud_sect_info_p)
{
    /* check that we recognize the section */
    struct __aud_info_s *info;
    struct __aud_section_s *sect = __aud_lookup(section_d);

    if (sect == NULL || sect->magic != magic) {
	errno = EINVAL;
	return -1;
    }

    /* special case */
    if (AUD_FIRST_ITEM == item_id) {
	if (sect->data) {
	    item_id = sect->data->item_id;
	} else {
	    errno = EINVAL;
	    return -1;
	}
    }

    /* now search sect for the requested item */
    for (info = sect->data; info; info = info->next) {
	if (info->magic != __AUD_MAGIC_INFO) {
	    fprintf(stderr, "internal error " __FILE__ " line %d\n", __LINE__);
	    exit(1);
	}

	/* is this the desired entry? */
	if (info->item_id == item_id) {
	    sect->prev_item_id = item_id;
	    if (aud_sect_info_p)
		*aud_sect_info_p = &(info->datum);
	    return sect->count;                  /* number of known items */
	}

	/* should we be looking for the next item? */
	if (AUD_NEXT_ITEM == item_id && info->item_id == sect->prev_item_id) {
	    /* XXX - this is not strictly what POSIX requires
	       (24.4.17.2).  What we do is look for the next one as
	       they were previously set.  POSIX specifies an order for
	       each sect type => library-bloat.  In general, the
	       items should have been entered in the correct order and
	       thus there will be little problem.  However, note bene
	       - we fail to satisfy the written spec here. */
	    if (info->next) {
		item_id = info->next->item_id;
	    } else {
		errno = EINVAL;
		return -1;
	    }
	}
    }

    /* not found */

    errno = EINVAL;
    return -1;
}

/* p146 - create an empty set of event specific data in an audit
   record - returns 0(success) -1(failure) */

/* XXX - should next_p be a pointer to a descriptor or just a descriptor? */

int __aud_put_section(int magic, aud_rec_t ar,
		      const struct __aud_section_s **next_p,
		      const struct __aud_section_s **new_p)
{
    int *counter;
    struct __aud_rec_s *record = __aud_lookup(ar);
    struct __aud_section_s **sect_holder, *nsection, *section;

    /* verify that record is valid */
    if (record == NULL || record->magic != __AUD_MAGIC_RECORD) {
	errno = EINVAL;
	return -1;
    }
	
    /* locate first section of required flavor */
    if (!___locate_sect(magic, record, &sect_holder, &counter)) {
	return -1;
    }

    /* find where we are going to insert the section - verify that the
       'next_p' is valid. */
    for (; (nsection = *sect_holder); sect_holder=&(nsection->after)) {
	if (nsection->magic != magic) {
	    fprintf(stderr, "internal error " __FILE__ " line %d\n", __LINE__);
	    exit(1);
	}
	if (next_p && *next_p == nsection) {
	    /* found section */
	    break;
	}
    }

    /* check that the next section is the one we were seeking */
    if (next_p && *next_p != nsection) {
	errno = EINVAL;
	return -1;
    }

    /* allocate a new section */
    section = __aud_malloc(sizeof(struct __aud_section_s), magic);
    if (section == NULL) {
	errno = ENOMEM;
	return -1;
    }

    /* initialize the section */
    section->magic = magic;
    section->count = 0;
    section->prev_item_id = 0;
    section->owner_count = counter;
    section->data = NULL;

    /* link the section into the section list */
    *sect_holder = section;
    section->before = sect_holder;
    section->after = nsection;
    if (nsection)
	nsection->before = &(section->after);
    ++*(section->owner_count);

    /* return new section */
    *new_p = section;
    return 0;
}

/* p147 - add a data item to an section specific set - returns
   0(success) -1(failure). */

int __aud_put_section_info(int magic, const void *section_d,
			   int position, int item_id,
			   const aud_info_t *aud_sect_info_p)
{
    struct __aud_info_s *after, **before, *info;
    struct __aud_section_s *sect = __aud_lookup(section_d);

    /* verify that the sect is valid */
    if (sect == NULL || sect->magic != magic) {
	errno = EINVAL;
	return -1;
    }

    before = &(sect->data);
    after = NULL;

    /* scan list of current info items */
    for (info=*before; info; info = info->next) {
	if (info->magic != __AUD_MAGIC_INFO) {
	    fprintf(stderr, "internal error in " __FILE__ " line %d\n",
		    __LINE__);
	    exit(1);
	}
	/* found an item_id => error! */
	if (info->item_id == item_id) {
	    errno = EINVAL;
	    return -1;
	}
	/* locate the 'position' item */
	if (info->item_id == position) {
	    if (after != NULL) {    /* this possibility is not covered
				       by POSIX - it is hard to see how
				       it would actually occur. */
		errno = EINVAL;
		return -1;
	    }
	    after = info;
	}
    
	if (after == NULL)
	    before = &(info->next);
    }

    if (after == NULL && position != AUD_LAST_ITEM) {
	/* item was not found */
	errno = EINVAL;
	return -1;
    }

    /* here, before points to where we need to supply an info pointer
       and after points to the next one (after may be NULL) */

    /* duplicate the info item */
    info = __aud_malloc(sizeof(struct __aud_info_s), __AUD_MAGIC_INFO);
    if (info == NULL) {
	errno = ENOMEM;
	return -1;
    }

    /* initialize info data */
    info->datum = *aud_sect_info_p;
    info->datum.aud_info_p = __aud_malloc(info->datum.aud_info_length,
					  __AUD_MAGIC_DATA);
    info->magic = __AUD_MAGIC_INFO;
    info->item_id = item_id;

    /* XXX - this will fail _really_badly_ for AUD_TYPE_STRING_ARRAY
       its a pity but we have to reconstruct the array in memory: a
       special case :( It will also require some special handling when
       we convert the record to something we can write to the audit
       log.  [much care needed here.] */
    memcpy(info->datum.aud_info_p, aud_sect_info_p->aud_info_p,
	   info->datum.aud_info_length);

    /* insert info structure into the list of items */
    *before = info;
    info->next = after;

    /* one more item added */
    sect->count++;

    return 0;
}

/*
 * $Log$
 */
