/*
 * $Id$
 *
 * Copyright (c) 1997 Andrew G. Morgan <morgan@parc.power.net>
 *
 * This file contains the functions that know about text representations
 * of things in audit records.
 */

#include <sys/audit.h>
#include <stdarg.h>
#include "audit.h"

#define TEXTBLOCK 512

#define __AUD_FORMAT_RECORD_BEGIN  "[#E=%d,#H=%d,#O=%d,#S=%d]\n"
#define __AUD_FORMAT_RECORD_END    ""
#define __AUD_FORMAT_EVENT_BEGIN   "[Event(s):"
#define __AUD_FORMAT_EVENT_END     "]\n"
#define __AUD_FORMAT_HEADER_BEGIN  "[Header(s):"
#define __AUD_FORMAT_HEADER_END    "]\n"
#define __AUD_FORMAT_OBJECT_BEGIN  "[Object(s):"
#define __AUD_FORMAT_OBJECT_END    "]\n"
#define __AUD_FORMAT_SUBJECT_BEGIN "[Subject(s):"
#define __AUD_FORMAT_SUBJECT_END   "]\n"
#define __AUD_FORMAT_SECTION_BEGIN "<"
#define __AUD_FORMAT_SECTION_END   ">"

static char *___append_text(char *text, int *from, int *buflen,
			    int max_args_size, const char *format, ... )
{
    va_list arglist_p;
    int needed_length;

    /* need more memory? */
    needed_length = *from + (1+strlen(format)+max_args_size);
    if (text == NULL || needed_length > (*buflen-*from)) {
	char *new_text;

	/* allocate some space */
	needed_length = TEXTBLOCK*(((TEXTBLOCK-1) + needed_length)/TEXTBLOCK);
	new_text = __aud_malloc(needed_length, __AUD_MAGIC_DATA);
	if (new_text != NULL) {
	    /* copy old text to new_text */
	    *buflen = needed_length;
	    memcpy(new_text, text, *from);
	}

	/* liberate old text */
	__aud_free(text);

	if ((text = new_text) == NULL) {
	    *buflen = 0;
	    *from = 0;
	    return NULL;        /* tidy up and go home */
	} else {
	    new_text = NULL;	/* paranoia */
	}
    }

    /* XXX - by getting here, we have enough memory but we use
       vsnprintf anyway ;) */

    va_start(arglist_p, format);
    *from += vsnprintf(text+*from, (1+strlen(format)+max_args_size),
		       format, arglist_p);
    va_end(arglist_p);

    /* return the lengthened text */

    return text;
}

/* This function lists the contents of the given list of sections */

#include "_aud_info_type.c"

static char *expand_sections(char *text, int *from, int *length,
			     struct __aud_section_s *sect, int count)
{
    int j;

    /* spin through list of sections */
    for (j=0; j<count; ++j, sect=sect->after) {
	struct __aud_info_s *info;

	/* this means that count is bad */
	if (sect == NULL) {
	    __aud_free(text);
	    errno = EINVAL;
	    return (text = NULL);
	}

	text = ___append_text(text, from, length, 0,
			      __AUD_FORMAT_SECTION_BEGIN);
	if (text == NULL) {
	    return NULL;
	}

	/* now list all the info items */

	for (info=sect->data; info; info=info->next) {
	    int i,item_id;

	    /* identify the item_id */
	    item_id = info->item_id;
	    if (item_id < 0 || item_id > MAX_ITEM_ID) {
		item_id = 0;
	    }

	    /* prefix */
	    text = ___append_text(text, from, length,
				  strlen(___aud_item_id[item_id]),
				  "[%s", ___aud_item_id[item_id]);

	    switch (info->datum.aud_info_type) {
		/* treat data as a series of integer (32-bit) values */
	    case AUD_TYPE_AUD_ID:
	    case AUD_TYPE_GID:
	    case AUD_TYPE_INT:
	    case AUD_TYPE_PID:
	    case AUD_TYPE_UID:
		if (info->datum.aud_info_length % sizeof(int)) {
		    __aud_free(text);
		    errno = EINVAL;
		    return (text = NULL);
		}
		for (i=0; i<info->datum.aud_info_length/sizeof(int); ++i) {
		    text = ___append_text(text, from, length, 11,
					  ___aud_info_type
					  [info->datum.aud_info_type],
					  ((int *)info->datum.aud_info_p)[i]
			);
		}
		break;
		/* treat as a series of short's */
	    case AUD_TYPE_SHORT:
		if (info->datum.aud_info_length % sizeof(short)) {
		    __aud_free(text);
		    errno = EINVAL;
		    return (text = NULL);
		}
		for (i=0; i<info->datum.aud_info_length/sizeof(short); ++i) {
		    text = ___append_text(text, from, length, 6,
					  ___aud_info_type
					  [info->datum.aud_info_type],
					  ((short *)info->datum.aud_info_p)[i]
			);
		}
		break;
		/* a character array */
	    case AUD_TYPE_STRING:
		text = ___append_text(text, from, length,
				      strlen(info->datum.aud_info_p),
				      ___aud_info_type[
					  info->datum.aud_info_type],
				      info->datum.aud_info_p);
		break;
	    case AUD_TYPE_STRING_ARRAY:
		fprintf(stderr,
			"XXX - if you are planning to use string arrays,\n"
			"you must fix the problem in `aud_generic.c' of\n"
			"libaudit.\n");
		exit(1);
		break;
	    case AUD_TYPE_AUD_STATUS:
		if (info->datum.aud_info_length % sizeof(aud_status_t)) {
		    __aud_free(text);
		    errno = EINVAL;
		    return (text = NULL);
		}
		for (i=0; i<info->datum.aud_info_length/sizeof(aud_status_t);
		     ++i) {
		    text = ___append_text(text, from, length, 11,
					  ___aud_info_type
					  [info->datum.aud_info_type],
					  ((aud_status_t *)
					   info->datum.aud_info_p)[i]
			);
		}
		break;
	    default:
		/* Don't know yet */
		fprintf(stderr, "? found an unknown aud_info_t ? (%d)\n",
			info->datum.aud_info_type);
	    }
	    text = ___append_text(text, from, length, 0, "]");
	}

	/* finally close this section */

	text = ___append_text(text, from, length, 0,
			      __AUD_FORMAT_SECTION_END);
	if (text == NULL) {
	    return NULL;
	}
    }

    return text;
}

/* p159 - obtain a text version of the audit record */

const char *aud_rec_to_text(aud_rec_t ar, ssize_t *len_p)
{
    char *output=NULL;
    int from=0, length=0;
    int i, j;
    struct __aud_rec_s *record = __aud_lookup(ar);
    struct __aud_section_s *sect;

    /* verify that this audit record is valid */
    if (record == NULL || record->magic != __AUD_MAGIC_RECORD) {
	errno = EINVAL;
	return NULL;
    }

    /* begin the title */
    output = ___append_text(output, &from, &length, 4*21 /* 4x64 bit length */,
			    __AUD_FORMAT_RECORD_BEGIN,
			    record->event_count, record->header_count,
			    record->object_count, record->subject_count);

    /* second list the header(s) */
    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_HEADER_BEGIN);
    }
    if (output != NULL) {
	output = expand_sections(output, &from, &length, record->header,
				 record->header_count);
    }
    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_HEADER_END);
    }
    /* third list the subject(s) */
    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_SUBJECT_BEGIN);
    }
    if (output != NULL) {
	output = expand_sections(output, &from, &length, record->subject,
				 record->subject_count);
    }
    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_SUBJECT_END);
    }
    /* fourth list the event(s) */    
    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_EVENT_BEGIN);
    }
    if (output != NULL) {
	output = expand_sections(output, &from, &length, record->event,
				 record->event_count);
    }
    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_EVENT_END);
    }
    /* last list the object(s) */
    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_OBJECT_BEGIN);
    }
    if (output != NULL) {
	output = expand_sections(output, &from, &length, record->object,
				 record->object_count);
    }
    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_OBJECT_END);
    }

    if (output != NULL) {
	output = ___append_text(output, &from, &length, 0,
				__AUD_FORMAT_RECORD_END);
    }

    /* indicate the length */
    if (output != NULL && len_p != NULL) {
	*len_p = from;
    }

    return output;
}

/* p126 - return a string describing the numeric audit event type */

const char *aud_evid_to_text(int event_type, ssize_t *aud_info_length)
{
    errno = ENOSYS;
    return NULL;
}

/* p125 - return audit event type for argument string */

int aud_evid_from_text(const char *text)
{
    errno = ENOSYS;
    return -1;
}

/* p144 - map an id to a text name */

const char *aud_id_to_text(aud_id_t audit_id, ssize_t *len_p)
{
    errno = ENOSYS;
    return NULL;
}

/* p143 - map a name to an id */

aud_id_t aud_id_from_text(const char *name)
{
    errno = ENOSYS;
    return (aud_id_t) -1;
}

/*
 * $Log$
 */
