/*--------------------------------------------------------------------
 * FILE:
 *     rlog.c
 *
 * NOTE:
 *     This file is composed of the functions to call with the source
 *     at pgreplicate for replicate ahead log.
 *
 * Portions Copyright (c) 2004, Atsushi Mitani
 *--------------------------------------------------------------------
 */
#include "postgres.h"
#include "postgres_fe.h"

#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <signal.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netdb.h>
#include <dirent.h>
#include <arpa/inet.h>

#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif

#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

#include "libpq-fe.h"
#include "libpq-int.h"
#include "fe-auth.h"
#include "access/xact.h"
#include "replicate_com.h"
#include "pgreplicate.h"

/*--------------------------------------
 * PROTOTYPE DECLARATION
 *--------------------------------------
 */
void PGRwrite_rlog(ReplicateHeader * header, char * query);
void PGRreconfirm_commit(int sock, ReplicateHeader * header);
void PGRset_rlog(ReplicateHeader * header, char * query);
void PGRunset_rlog(ReplicateHeader * header, char * query);

static int set_query_log(ReplicateHeader * header, char * query);
static void delete_query_log(void);
static int set_commit_log(ReplicateHeader * header);
static CommitLogInf * get_commit_log(ReplicateHeader * header);
static void delete_commit_log(ReplicateHeader * header);
static bool was_committed_transaction(ReplicateHeader * header);

void
PGRwrite_rlog(ReplicateHeader * header, char * query)
{
 	if (header == NULL)
	{
		return;
	}
	switch (header->cmdSts)
	{
		case CMD_STS_QUERY:
			set_query_log(header,query);
			break;
		case CMD_STS_DELETE_QUERY:
			delete_query_log();
			break;
		case CMD_STS_TRANSACTION:
			if (PGR_Log_Header->cmdType == CMD_TYPE_COMMIT)
			{
				set_commit_log(header);
			}
		case CMD_STS_DELETE_TRANSACTION:
			if (PGR_Log_Header->cmdType == CMD_TYPE_COMMIT)
			{
				delete_commit_log(header);
			}
			break;
	}
}

static int
set_query_log(ReplicateHeader * header, char * query)
{
	char * func = "set_query_log()";
	int size = 0;

	delete_query_log();
	Replicateion_Log->header = (ReplicateHeader *)malloc(sizeof(ReplicateHeader));
	if (Replicateion_Log->header == (ReplicateHeader *)NULL)
	{
		show_error("%s:malloc failed: (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	size = ntohl(header->query_size);
	Replicateion_Log->query = (char *)malloc(size);
	if (Replicateion_Log->query == (char *)NULL)
	{
		show_error("%s:malloc failed: (%s)",func,strerror(errno));
		return STATUS_ERROR;
	}
	memcpy(Replicateion_Log->header,header,sizeof(ReplicateHeader));
	memcpy(Replicateion_Log->query,query,size);

	return STATUS_OK;
}

static void
delete_query_log(void)
{

	if (Replicateion_Log->header != NULL)
	{
		free(Replicateion_Log->header);
	}
	if (Replicateion_Log->query != NULL)
	{
		free(Replicateion_Log->query);
	}

}

static int
set_commit_log(ReplicateHeader * header)
{
	CommitLogInf * commit_log = NULL;
	ReplicateHeader * c_header;

	if (Commit_Log_Tbl == NULL)
	{
		return STATUS_ERROR;
	}
	commit_log = Commit_Log_Tbl + 1;
	while (	commit_log->inf.useFlag != DB_TBL_END )
	{
		if (commit_log->inf.useFlag != DB_TBL_USE)
		{
			commit_log->inf.useFlag = DB_TBL_USE;
			c_header = &(commit_log->header);
			c_header->port = ntohs(header->port);
			c_header->pid = ntohs(header->pid);
			memcpy(c_header->from_host,header->from_host,sizeof(c_header->from_host));
			memcpy(c_header->dbName,header->dbName,sizeof(c_header->dbName));
			memcpy(c_header->userName,header->userName,sizeof(c_header->userName));
			c_header->tv.tv_sec = ntohl(header->tv.tv_sec);
			c_header->tv.tv_usec = ntohl(header->tv.tv_usec);

			Commit_Log_Tbl->inf.commit_log_num ++;
			return STATUS_OK;
		}
		commit_log ++;
	}
	return STATUS_ERROR;
}

static CommitLogInf *
get_commit_log(ReplicateHeader * header)
{
	CommitLogInf * commit_log = NULL;
	ReplicateHeader * c_header;
	int cnt = 0;

	if (Commit_Log_Tbl == NULL)
	{
		return (CommitLogInf *)NULL;
	}
	commit_log = Commit_Log_Tbl;
	while (	commit_log->inf.useFlag != DB_TBL_END )
	{
		if (commit_log->inf.useFlag == DB_TBL_USE)
		{
			cnt ++;
			c_header = &(commit_log->header);
			if (c_header == NULL)
			{
				commit_log ++;
				continue;
			}
			if ( (c_header->port == ntohs(header->port)) &&
				(c_header->pid == ntohs(header->pid))	&&
				(strncmp(c_header->from_host,header->from_host,sizeof(c_header->from_host)) == 0)	&&
				(strncmp(c_header->dbName,header->dbName,sizeof(c_header->dbName)) == 0)	&&
				(strncmp(c_header->userName,header->userName,sizeof(c_header->userName)) == 0) &&
				(c_header->tv.tv_sec == ntohl(header->tv.tv_sec))	&&
				(c_header->tv.tv_usec == ntohl(header->tv.tv_usec)))
			{
				return commit_log;	
			}
		}
		if (cnt >= Commit_Log_Tbl->inf.commit_log_num)
		{
			break;
		}
		commit_log ++;
	}
	return (CommitLogInf *)NULL;
}

static void
delete_commit_log(ReplicateHeader * header)
{
	CommitLogInf * commit_log = NULL;

	commit_log = get_commit_log(header);
	if (commit_log != NULL)
	{
		memset(&(commit_log->header),0,sizeof(commit_log->header));
		commit_log->inf.useFlag = DB_TBL_INIT;
		Commit_Log_Tbl->inf.commit_log_num --;
	}
}

static bool
was_committed_transaction(ReplicateHeader * header)
{
	CommitLogInf * commit_log = NULL;

	commit_log = get_commit_log(header);
	if (commit_log != NULL)
	{
		return true;
	}
	return false;
}

void 
PGRreconfirm_commit(int sock, ReplicateHeader * header)
{
	int status = PGR_NOT_YET_COMMIT;
	char result[PGR_MESSAGE_BUFSIZE];

	/* check the transaction was committed */
	if (was_committed_transaction(header) == true)
	{
		status = PGR_ALREADY_COMMITTED;
	}
	snprintf(result,PGR_MESSAGE_BUFSIZE,"%d,%d", PGR_TRANSACTION_CONFIRM_ANSWER_FUNC_NO,status);

	PGRreturn_result(sock, result, PGR_NOWAIT_ANSWER);
}

void
PGRset_rlog(ReplicateHeader * header, char * query)
{
	int status = STATUS_OK;
	bool send_flag = false;

 	if (PGR_Log_Header == NULL)
	{
		return;
	}
	switch (PGR_Log_Header->cmdSts)
	{
		case CMD_STS_QUERY:
			send_flag = true;
			break;
		case CMD_STS_TRANSACTION:
			if (PGR_Log_Header->cmdType == CMD_TYPE_COMMIT)
			{
				send_flag = true;
			}
			break;
	}
	if (send_flag != true)
	{
		return;
	}
	PGR_Log_Header->query_size = htonl(strlen(query));
	if (PGR_Cascade == true)
	{
		/* save log data in remote server */
		PGR_Log_Header->cmdSys = CMD_SYS_LOG;

		status = PGRsend_lower_cascade(PGR_Log_Header, query);	
		if (status != STATUS_OK)
		{
			PGRwrite_rlog(PGR_Log_Header, query);
		}
	}
	else
	{
		/* save log data in local server */
		 PGRwrite_rlog(PGR_Log_Header, query);
	}
}

void
PGRunset_rlog(ReplicateHeader * header, char * query)
{
	int status = STATUS_OK;
	bool send_flag = false;

 	if (PGR_Log_Header == NULL)
	{
		return;
	}
	switch (PGR_Log_Header->cmdSts)
	{
		case CMD_STS_QUERY:
			send_flag = true;
			PGR_Log_Header->cmdSts = CMD_STS_DELETE_QUERY;
			break;
		case CMD_STS_TRANSACTION:
			if (PGR_Log_Header->cmdType == CMD_TYPE_COMMIT)
			{
				PGR_Log_Header->cmdSts = CMD_STS_DELETE_TRANSACTION;
				send_flag = true;
			}
			break;
	}
	if (send_flag != true)
	{
		return;
	}
	PGR_Log_Header->query_size = htonl(strlen(query));
	if (PGR_Cascade == true)
	{
		/* save log data in remote server */
		PGR_Log_Header->cmdSys = CMD_SYS_LOG;

		status = PGRsend_lower_cascade(PGR_Log_Header, query);	
		if (status != STATUS_OK)
		{
			PGRwrite_rlog(PGR_Log_Header, query);
		}
	}
	else
	{
		/* save log data in local server */
		 PGRwrite_rlog(PGR_Log_Header, query);
	}
}

