/*--------------------------------------------------------------------
 * FILE:
 *     admin_main.c
 *
 * NOTE:
 *     This file is composed of the admin process 
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */

/*--------------------------------------
 * INTERFACE ROUTINES
 *
 * I/O call:
 *      
 *-------------------------------------
 */
#include "postgres.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <ctype.h>
#include <time.h>
#include <pwd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <arpa/inet.h>
#include <sys/param.h>
#include <sys/select.h>
#include <sys/file.h>
#include <dirent.h>
#include <getopt.h>

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

/*
#include "replicate_com.h"
*/
#include "libpq/pqsignal.h"
#include "pgc_admin.h"

char * ServerTbl = NULL;
int ServerTblSize = 0;
Physical_Server_Info * PhysicalServerTbl = NULL;
SSL_Server_Info * AdminTbl = NULL;
SSL_Server_Info * ProbeTbl = NULL;
Pglb_Info * PglbTbl = NULL;
Cluster_Info * ClusterDbTbl = NULL;
Pgrp_Info * PgrpTbl = NULL;
Partial_Info * PartialTbl = NULL;
Record_Num_Info * RecordNumTbl = NULL;
Shmid_Info ShmidTbl;
int Fork_Delay_Time = 3;

int exit_signo = SIGTERM;
volatile bool exit_processing = false;
volatile bool restart_processing = false;
char * PGR_Data_Path = NULL;
char * PGR_Write_Path = NULL;

/* for replicate_com.h */
ConfDataType * ConfData_Top = (ConfDataType *)NULL;
ConfDataType * ConfData_End = (ConfDataType *)NULL;
int PGR_Lifecheck_Interval = 15;
int Debug_Print = 0;
int Log_Print = 0;

int AdminSock = -1;

SSL_Info SSL_Tbl;

static void set_exit_processing(int signo);
static void set_restart_processing(int signo);
static int admin_main(void);
static void admin_loop(void * arg);
static void admin_exit(int exit_status);
static void usage(void);

int
main(int argc, char * argv[])
{
	char * func = "main()";
	int opt = 0;
	char * r_path = NULL;
	char * w_path = NULL;
	bool detach = true;

	r_path = getenv("PGDATA");
	if (r_path == NULL)
		r_path = ".";
	while ((opt = getopt(argc, argv, "d:D:W:i:lvnh")) != -1)
	{
		switch (opt)
		{
			case 'd':
				if (!optarg)
				{
					usage();
					exit(1);
				}
				Fork_Delay_Time = atoi(optarg);
				break;
			case 'D':
				if (!optarg)
				{
					usage();
					exit(1);
				}
				r_path = optarg;
				break;
			case 'W':
				if (!optarg)
				{
					usage();
					exit(1);
				}
				w_path = optarg;
				break;
			case 'i':
				if (!optarg)
				{
					usage();
					exit(1);
				}
				PGR_Lifecheck_Interval = atoi(optarg);
				break;
			case 'l':
				Log_Print = 1;
				break;
			case 'v':
				Debug_Print = 1;
				break;
			case 'n':
				detach = false;
				break;
			case 'h':
				usage();
				exit(0);
				break;
			default:
				usage();
				exit(1);
		}
	}
	PGR_Data_Path = r_path;
	if (w_path == NULL)
	{
		PGR_Write_Path = PGR_Data_Path;
	}
	else
	{
		PGR_Write_Path = w_path;
	}

	if (optind == (argc-1) && !strncasecmp(argv[optind],"stop",4))
	{
		PGC_Stop_Process(PGR_Write_Path,PGC_ADMIN_PID_FILE);
		admin_exit(0);
	}
	else if (optind == (argc-1) && !strncasecmp(argv[optind],"status",6))
	{
		PGC_Status_Process(PGR_Write_Path,PGC_ADMIN_PID_FILE);
		admin_exit(0);
	}
	else if (optind == (argc-1) && !strncasecmp(argv[optind],"shutdown",8))
	{
		PGC_Admin_Shutdown(PGR_Data_Path);
		PGC_Stop_Process(PGR_Write_Path,PGC_ADMIN_PID_FILE);
		admin_exit(0);
	}
	else if (optind == argc)
	{
		if (PGC_Is_Exist_Pid_File(PGR_Write_Path, PGC_ADMIN_PID_FILE))
		{
			show_error("pid file %s/%s found. is another admin running?", PGR_Write_Path, PGC_ADMIN_PID_FILE);
			exit(1);
		}
	}
	else if (optind < argc)
	{
		usage();
		exit(1);
	}

	PGRinitmask();

	if (detach)
	{
		PGC_Daemonize();
	}

	if (PGC_Write_Pid_File(PGR_Write_Path, PGC_ADMIN_PID_FILE) != STATUS_OK)
	{
		exit(1);
	}

	PGRsignal(SIGHUP, set_restart_processing);
	PGRsignal(SIGINT, set_exit_processing);
	PGRsignal(SIGQUIT, set_exit_processing);
	PGRsignal(SIGTERM, set_exit_processing);
	PGRsignal(SIGCHLD, PGC_Child_Wait);
	PGRsignal(SIGPIPE, SIG_IGN);
	PG_SETMASK(&UnBlockSig);

	if (PGRinit_Admin(PGR_Data_Path) != STATUS_OK)
	{
		show_error("%s:PGRinit_Admin error",func);
		admin_exit(0);
	}
	if (PGRget_Admin_Conf_Data(PGR_Data_Path, PGC_ADMIN_CONF_FILE) != STATUS_OK)
	{
		show_error("%s:PGRget_Admin_Conf_Data error",func);
		admin_exit(0);
	}

	if (PGC_Admin_Status_Monitor(Fork_Delay_Time) != STATUS_OK)
	{
		show_error("%s:PGC_Admin_Status_Monitor error",func);
		admin_exit(0);
	}

	admin_main();

	admin_exit(0);
	return STATUS_OK;
}

static void
set_exit_processing(int signo)
{
	exit_signo = signo;
	exit_processing = true;
	PGRsignal(signo, SIG_IGN);
	admin_exit(0);
}

static void
set_restart_processing(int signo)
{
	restart_processing = true;
	PGRsignal(SIGHUP, set_restart_processing);
}

static int
admin_main(void)
{
	char * func = "admin_main()";
    BIO     *acc, *client;
    pthread_t tid;
	char port[8];

	if (AdminTbl == NULL)
	{
		show_error("%s:AdminTbl is NULL",func);
		return STATUS_ERROR;
	}
	memset(&SSL_Tbl, 0, sizeof(SSL_Info));
    PGC_SSL_Init(  );
    PGC_SSL_Seed_Prng(  );
 
    SSL_Tbl.ctx = PGC_SSL_Server_Setup_Ctx(AdminTbl);
	if (SSL_Tbl.ctx == NULL)
	{
		show_error("%s:PGC_SSL_Server_Setup_Ctx failed",func);
		return STATUS_ERROR;
	}
 
	memset(port, 0, sizeof(port));
 	sprintf(port,"%d",AdminTbl->portNumber);
    acc = BIO_new_accept(port);
    if (!acc)
	{
		show_error("%s:BIO_new_accept failed. Port [%d] can not be used",func,ProbeTbl->portNumber);
		ERR_print_errors_fp(stderr);
		return STATUS_ERROR;
	}
 
    if (BIO_do_accept(acc) <= 0)
	{
        show_error("%s: BIO_do_accept failed", func); 
		ERR_print_errors_fp(stderr);
		BIO_free(acc);
		return STATUS_ERROR;
	}

	for (;;)
	{
		if (BIO_do_accept(acc) <= 0)
		{
        	show_error("%s: BIO_do_accept failed", func); 
			ERR_print_errors_fp(stderr);
			continue;
		}
 
        client = BIO_pop(acc);
        if (!(SSL_Tbl.ssl = SSL_new(SSL_Tbl.ctx)))
		{
        	show_error("%s: SSL_new failed", func); 
			ERR_print_errors_fp(stderr);
			continue;
		}
        SSL_set_accept_state(SSL_Tbl.ssl);
        SSL_set_bio(SSL_Tbl.ssl, client, client);
        pthread_create(&(tid), NULL, (void *)admin_loop, &SSL_Tbl);
	}
	PGC_Close_SSL(&SSL_Tbl);
    BIO_free(acc);

	return STATUS_OK;
}

static void
admin_loop(void * arg)
{
	char * func ="admin_loop()";
	SSL_Info *ssl_tbl = (SSL_Info *)arg;
	bool loop_end = false;
	Probe_Header header;
	char * packet = NULL;
	int status = 0;

	sleep(Fork_Delay_Time);

    pthread_detach(pthread_self());
    if (SSL_accept(ssl_tbl->ssl) <= 0)
	{
        show_error("Error accepting SSL connection");
		return;
	}
	show_debug("SSL_accept OK");
	for(;;)
	{
		memset(&header,0,sizeof(Probe_Header));
		packet = PGC_Read_Probe_Packet(ssl_tbl, &header);
		if ((packet == NULL) && (header.body_length != 0))
		{
			break;
		}
		switch (ntohs(header.packet_no))
		{
			case GET_STS_ANS_PKT:
				/*
				 * server status information 
				 */
				show_debug("%s:GET_STS_ANS_PKT recv\n",func);
				PGC_Admin_Set_Status(ssl_tbl, &header, packet);
				loop_end = true;
				break;
			case USER_RECOVERY_REQ_PKT:
				show_debug("%s:USER_RECOVERY_REQ_PKT recv\n",func);
				status = PGC_Admin_User_Exec_Req(RECOVERY_REQ_PKT, ntohs(header.serverType), ntohs(header.serverNo));
				PGC_Response_Send(ssl_tbl, USER_RECOVERY_ANS_PKT, status);
				loop_end = true;
				break;
			case USER_START_REQ_PKT:
				show_debug("%s:USER_START_REQ_PKT recv\n",func);
				status = PGC_Admin_User_Exec_Req(START_REQ_PKT, ntohs(header.serverType), ntohs(header.serverNo));
				PGC_Response_Send(ssl_tbl, USER_START_ANS_PKT, status);
				loop_end = true;
				break;
			case USER_STOP_REQ_PKT:
				show_debug("%s:USER_STOP_REQ_PKT recv\n",func);
				status = PGC_Admin_User_Exec_Req(STOP_REQ_PKT, ntohs(header.serverType), ntohs(header.serverNo));
				PGC_Response_Send(ssl_tbl, USER_STOP_ANS_PKT, status);
				loop_end = true;
				break;
			case USER_STATUS_REQ_PKT:
				show_debug("%s:USER_STATUS_REQ_PKT recv\n",func);
				PGC_Admin_User_Status_Req(ssl_tbl, ntohs(header.serverType), ntohs(header.serverNo));
				loop_end = true;
				break;
			default:
				show_error("%s:others\n",func);
				loop_end = true;
				break;
		}
		if (packet != NULL)
		{
			free(packet);
			packet = NULL;
		}
		if (loop_end)
		{
			break;
		}
	}
}

static void
admin_exit(int exit_status)
{
	char fname[256];

	/* clear shared memory */
	PGR_Clear_Shm();
	/* delete pid file */
	snprintf(fname, sizeof(fname), "%s/%s", PGR_Write_Path, PGC_ADMIN_PID_FILE);
	unlink(fname);
	exit(exit_status);
}

/*--------------------------------------------------------------------
 * SYMBOL
 *    usage()
 * NOTES
 *    show usage of pglb
 * ARGS
 *    void
 * RETURN
 *    none
 *--------------------------------------------------------------------
 */
static void
usage(void)
{
	char * path;

	path = getenv("PGDATA");
	if (path == NULL)
		path = ".";
	fprintf(stderr,"PGReplicate version [%s]\n",PGREPLICATE_VERSION);
	fprintf(stderr,"A replication server for cluster DB servers (based on PostgreSQL)\n\n");
	fprintf(stderr,"usage: pgc_admin [-D path_of_config_file] [-W path_of_work_files] [-n][-v][-h][stop][shutdown][status]\n");
	fprintf(stderr,"    config file default path: %s/%s\n",path, PGREPLICATE_CONF_FILE);
	fprintf(stderr,"    -n: don't run in daemon mode.\n");
	fprintf(stderr,"    -v: debug mode. need '-n' flag\n");
	fprintf(stderr,"    -h: print this help\n");
	fprintf(stderr,"    stop: stop pgc_admin\n");
	fprintf(stderr,"    shutdown: stop all PGCluster services\n");
	fprintf(stderr,"    status: show status of pgc_admin\n");
}
