/*--------------------------------------------------------------------
 * FILE:
 *     probe.c
 *
 * NOTE:
 *     This file is composed of the probe process 
 *     Low level I/O functions that called by in these functions are 
 *     contained in 'replicate_com.c'.
 *
 *--------------------------------------------------------------------
 */

/*--------------------------------------
 * INTERFACE ROUTINES
 *
 * I/O call:
 *      
 *-------------------------------------
 */
#include "postgres.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <ctype.h>
#include <time.h>
#include <pwd.h>
#include <arpa/inet.h>
#include <sys/file.h>
#include <sys/ipc.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <getopt.h>

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

#include "libpq/pqsignal.h"
#include "pgc_admin.h"

char * ServerTbl = NULL;
int ServerTblSize = 0;
Physical_Server_Info * PhysicalServerTbl = NULL;
SSL_Server_Info * AdminTbl = NULL;
SSL_Server_Info * ProbeTbl = NULL;
Pglb_Info * PglbTbl = NULL;
Cluster_Info * ClusterDbTbl = NULL;
Pgrp_Info * PgrpTbl = NULL;
Partial_Info * PartialTbl = NULL;
Record_Num_Info * RecordNumTbl = NULL;
Shmid_Info ShmidTbl;
int Fork_Delay_Time = 3;

int exit_signo = SIGTERM;
volatile bool exit_processing = false;
volatile bool restart_processing = false;
char * PGR_Data_Path = NULL;
char * PGR_Write_Path = NULL;

/* for replicate_com.h */
ConfDataType * ConfData_Top = (ConfDataType *)NULL;
ConfDataType * ConfData_End = (ConfDataType *)NULL;
int PGR_Lifecheck_Interval = 15;
int Debug_Print = 0;
int Log_Print = 0;

int ProbeSock = -1;
SSL_Info SSL_Tbl;

static void set_exit_processing(int signo);
static void set_restart_processing(int signo);
static int probe_main(void);
static void probe_loop(void * arg);
static void probe_exit(int exit_status);
static void restart_process(void);
static void usage(void);

int
main(int argc, char * argv[])
{
	char * func = "main()";
	int opt = 0;
	char * r_path = NULL;
	char * w_path = NULL;
	bool detach = true;

	r_path = getenv("PGDATA");
	if (r_path == NULL)
		r_path = ".";
	while ((opt = getopt(argc, argv, "d:D:W:lvnh")) != -1)
	{
		switch (opt)
		{
			case 'd':
				if (!optarg)
				{
					usage();
					exit(1);
				}
				Fork_Delay_Time = atoi(optarg);
				break;
			case 'D':
				if (!optarg)
				{
					usage();
					exit(1);
				}
				r_path = optarg;
				break;
			case 'W':
				if (!optarg)
				{
					usage();
					exit(1);
				}
				w_path = optarg;
				break;
			case 'l':
				Log_Print = 1;
				break;
			case 'v':
				Debug_Print = 1;
				break;
			case 'n':
				detach = false;
				break;
			case 'h':
				usage();
				exit(0);
				break;
			default:
				usage();
				exit(1);
		}
	}
	PGR_Data_Path = r_path;
	if (w_path == NULL)
	{
		PGR_Write_Path = PGR_Data_Path;
	}
	else
	{
		PGR_Write_Path = w_path;
	}

	if (optind == (argc-1) && !strncasecmp(argv[optind],"stop",4))
	{
		PGC_Stop_Process(PGR_Write_Path,PGC_PROBE_PID_FILE);
		probe_exit(0);
	}
	else if (optind == (argc-1) && !strncasecmp(argv[optind],"status",6))
	{
		PGC_Status_Process(PGR_Write_Path,PGC_PROBE_PID_FILE);
		probe_exit(0);
	}
	else if (optind == argc)
	{
		if (PGC_Is_Exist_Pid_File(PGR_Write_Path, PGC_PROBE_PID_FILE))
		{
			show_error("pid file %s/%s found. is another probe running?", PGR_Write_Path, PGC_PROBE_PID_FILE);
			exit(1);
		}
	}
	else if (optind < argc)
	{
		usage();
		exit(1);
	}

	PGRinitmask();

	if (detach)
	{
		PGC_Daemonize();
	}

	if (PGC_Write_Pid_File(PGR_Write_Path, PGC_PROBE_PID_FILE) != STATUS_OK)
	{
		exit(1);
	}

	PGRsignal(SIGHUP, set_restart_processing);
	PGRsignal(SIGINT, set_exit_processing);
	PGRsignal(SIGQUIT, set_exit_processing);
	PGRsignal(SIGTERM, set_exit_processing);
	PGRsignal(SIGCHLD, PGC_Child_Wait);
	PGRsignal(SIGPIPE, SIG_IGN);
	PG_SETMASK(&UnBlockSig);

	if (PGRinit_Probe(PGR_Data_Path) != STATUS_OK)
	{
		show_error("%s:PGRinit_Probe error",func);
		probe_exit(0);
	}
	if (PGRget_Admin_Conf_Data(PGR_Data_Path, PGC_PROBE_CONF_FILE ) != STATUS_OK)
	{
		show_error("%s:PGRget_Admin_Conf_Data error",func);
		probe_exit(0);
	}

	probe_main();

	probe_exit(0);
	return STATUS_OK;
}

static void
set_exit_processing(int signo)
{
	exit_signo = signo;
	exit_processing = true;
	PGRsignal(signo, SIG_IGN);
	probe_exit(0);
}

static void
set_restart_processing(int signo)
{
	restart_processing = true;
	PGRsignal(SIGHUP, set_restart_processing);
}

static int
probe_main(void)
{
	char * func = "probe_main()";
    BIO     *acc, *client;
    pthread_t tid;
	char port[8];

	if (ProbeTbl == NULL)
	{
		show_error("%s:ProbeTbl is NULL",func);
		return STATUS_ERROR;
	}
	memset(&SSL_Tbl, 0, sizeof(SSL_Info));

    PGC_SSL_Init();
    PGC_SSL_Seed_Prng();
 
    SSL_Tbl.ctx = PGC_SSL_Server_Setup_Ctx(ProbeTbl);
	if (SSL_Tbl.ctx == NULL)
	{
		show_error("%s:PGC_SSL_Server_Setup_Ctx failed",func);
		ERR_print_errors_fp(stderr);
		return STATUS_ERROR;
	}
 
	memset(port, 0, sizeof(port));
	sprintf(port,"%d",ProbeTbl->portNumber);
    acc = BIO_new_accept(port);
	show_debug("%s:BIO_new_accept(port:%s)",func,port);
    if (!acc)
	{
		show_error("%s:BIO_new_accept failed. Port [%d] can not be used",func,ProbeTbl->portNumber);
		ERR_print_errors_fp(stderr);
		return STATUS_ERROR;
	}
 
    if (BIO_do_accept(acc) <= 0)
	{
        show_error("%s: BIO_do_accept failed", func); 
		ERR_print_errors_fp(stderr);
		sleep(1);
	}

	for (;;)
	{ 
		if (restart_processing)
		{
			restart_process();
			restart_processing = false;
		}
		if (BIO_do_accept(acc) <= 0)
		{
        	show_debug("%s: BIO_do_accept failed", func); 
			ERR_print_errors_fp(stderr);
			sleep(1);
			continue;
		}
 
        client = BIO_pop(acc);
        if (!(SSL_Tbl.ssl = SSL_new(SSL_Tbl.ctx)))
		{
        	show_debug("%s: SSL_new failed", func); 
			ERR_print_errors_fp(stderr);
			sleep(1);
			continue;
		}
        SSL_set_accept_state(SSL_Tbl.ssl);
        SSL_set_bio(SSL_Tbl.ssl, client, client);
        pthread_create(&(tid), NULL, (void *)probe_loop, &SSL_Tbl);
    }
 
	PGC_Close_SSL(&SSL_Tbl);

	if (SSL_Tbl.ctx != NULL)
	{
    	SSL_CTX_free(SSL_Tbl.ctx);
		SSL_Tbl.ctx = NULL;
	}

    BIO_free(acc);

	return STATUS_OK;
}

static void
probe_loop(void * arg)
{
	char * func = "probe_loop()";
	SSL_Info *ssl_tbl = (SSL_Info *)arg;
	bool loop_end = false;
	Probe_Header header;
	char * packet = NULL;
	SSL_Server_Info * admin = AdminTbl;
	bool authorized_server = false;

	sleep(Fork_Delay_Time);

    pthread_detach(pthread_self());
    if (SSL_accept(ssl_tbl->ssl) <= 0)
	{
		ERR_print_errors_fp(stderr);
        show_error("Error accepting SSL connection");
		return;
	}
	show_debug("SSL_accept OK");
	/* check authorized server  */
	while (admin->portNumber != 0)
	{
		if (PGC_SSL_Post_Connection_Check(ssl_tbl, admin->hostName) == X509_V_OK)
		{
			authorized_server = true;
			break;
		}
		admin ++;
	}
	if (authorized_server == false)
	{
		show_error("Error checking SSL object after connection");
		return;
	}
	for(;;)
	{
		memset(&header,0,sizeof(Probe_Header));
		packet = PGC_Read_Probe_Packet(ssl_tbl, &header);
		if ((packet == NULL) && (header.body_length != 0))
		{
			show_error("%s:PGC_Read_Probe_Packet failed\n",func);
			/*
			SSL_clear(ssl_tbl->ssl);
			*/
			break;

		}
		switch (ntohs(header.packet_no))
		{
			case INIT_NOTICE_PKT:
				/*
				 * Admin server startup notice
				 */
				show_debug("%s:INIT_NOTICE_PKT: received\n",func);
				PGC_Probe_Init(ssl_tbl, &header, (SSL_Server_Info *)packet);
				break;
			case INFO_NOTICE_PKT:
				/*
				 * Server setup information notice
				 */
				show_debug("INFO_NOTICE_PKT: received\n");
				PGC_SSL_Server_Info(ssl_tbl, &header, packet);
				break;
			case GET_STS_REQ_PKT:
				/*
				 * Server status acquisition request
				 */
				show_debug("GET_STS_REQ_PKT: received\n");
				PGC_Probe_Get_Status(&header, packet);
				loop_end = true;
				break;
			case SET_STS_REQ_PKT:
				/*
				 * Server status store request
				 */
				show_debug("SET_STS_REQ_PKT: received\n");
				PGC_Probe_Set_Status(ssl_tbl, &header, packet);
				loop_end = true;
				break;
			case START_REQ_PKT:
			case STOP_REQ_PKT:
			case RECOVERY_REQ_PKT:
			case STATUS_REQ_PKT:
				/*
				 * Server start/stop/restart request
				 */
				show_debug("EXEC PKTS: received\n");
				PGC_Probe_Exec(ssl_tbl, &header, packet);
				loop_end = true;
				break;
			default:
				loop_end = true;
				break;
		}
		if (packet != NULL)
		{
			free(packet);
			packet = NULL;
		}
		if (loop_end)
		{
			break;
		}
	}
	/*
	show_debug("%s: PGC_Close_SSL",func);
	PGC_Close_SSL(ssl_tbl);
	show_debug("%s: ERR_remove_state",func);
	ERR_remove_state(0);
	*/
}

static void
probe_exit(int exit_status)
{
	char fname[256];
	sigset_t mask;

	sigemptyset(&mask);
	sigaddset(&mask, SIGTERM);
	sigaddset(&mask, SIGINT);
	sigaddset(&mask, SIGQUIT);
	sigaddset(&mask, SIGCHLD);
	sigprocmask(SIG_BLOCK, &mask, NULL);

	kill (0, SIGTERM);

	PGC_Child_Wait(0);

	/* clear shared memory */
	PGR_Clear_Shm();
	/* delete pid file */
	snprintf(fname, sizeof(fname), "%s/%s", PGR_Write_Path, PGC_PROBE_PID_FILE);
	unlink(fname);

	exit(exit_status);
}

static void
restart_process(void)
{
	char * func = "restart_process()";

	PGR_Clear_Shm();
	if (PGRinit_Probe(PGR_Data_Path) != STATUS_OK)
	{
		show_error("%s:PGRinit_Probe error",func);
		probe_exit(0);
	}
	if (PGRget_Admin_Conf_Data(PGR_Data_Path, PGC_PROBE_CONF_FILE) != STATUS_OK)
	{
		show_error("%s:PGRget_Admin_Conf_Data error",func);
		probe_exit(0);
	}
}

/*--------------------------------------------------------------------
 * SYMBOL
 *    usage()
 * NOTES
 *    show usage of pglb
 * ARGS
 *    void
 * RETURN
 *    none
 *--------------------------------------------------------------------
 */
static void
usage(void)
{
	char * path;

	path = getenv("PGDATA");
	if (path == NULL)
		path = ".";
	fprintf(stderr,"PGReplicate version [%s]\n",PGREPLICATE_VERSION);
	fprintf(stderr,"A replication server for cluster DB servers (based on PostgreSQL)\n\n");
	fprintf(stderr,"usage: pgc_probe [-D path_of_config_file] [-W path_of_work_files] [-n][-v][-h][stop][status]\n");
	fprintf(stderr,"    config file default path: %s/%s\n",path, PGREPLICATE_CONF_FILE);
	fprintf(stderr,"    -n: don't run in daemon mode.\n");
	fprintf(stderr,"    -v: debug mode. need '-n' flag\n");
	fprintf(stderr,"    -h: print this help\n");
	fprintf(stderr,"    stop: stop pgc_probe\n");
	fprintf(stderr,"    status: show status of pgc_probe\n");
}
