/*
 * httpd.c: httpd server mode.
 *
 * Copyright (c) 2009-2010, NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 */

#include "pg_reporter.h"
#include "pgut/pgut-httpd.h"

#include <dirent.h>
#include <sys/stat.h>
#include <time.h>

#include <libxslt/transform.h>

static void cgi_error(const char *message);
static const char *httpd_callback(StringInfo out, const http_request *request);
static const char *makeDatabaseReport(StringInfo out, List *vars, const char *xml);
static xmlDocPtr makeDatabaseList(void);
static char *decode(const char *str, int length);


/*-------------------------------------------------------------------------
 * cgi-mode
 *-------------------------------------------------------------------------
 */

int
cgi_main(void)
{
	char		   *query_string;
	StringInfoData 	body;			/* body text of the report */
	const char	   *result;			/* content-type or error code */
	const char	   *xml;			/* template name */
	List		   *vars;

	pgut_abort_level = FATAL;

	if ((query_string = getenv("QUERY_STRING")) == NULL)
	{
		cgi_error("QUERY_STRING is not defined");
		return 1;
	}

	/* parse QUERY_STRING and make a variable table */
	vars = makeVariables(query_string);
	if (vars == NULL)
	{
		cgi_error("invalid URL request");
		return 1;
	}

	/* retrieve template name from URL parameter */
	if ((xml = getVariable(vars, "template")) == NULL)
	{
		elog(WARNING, "\"template\" option was not found in url parameter");
		result = "*";
	}
	else
	{
		/* Make a database report. */
		initStringInfo(&body);
		result = makeDatabaseReport(&body, vars, xml);
	}

	/* content-type starts with '*' means an error. */
	if (result[0] == '*')
	{
		cgi_error(result);
		return 1;
	}

	printf("Content-Type: %s\r\n", result);
	printf("\r\n");
	fwrite(body.data, 1, body.len, stdout);

	list_destroy(vars, freeVariable);
	termStringInfo(&body);
	return 0;
}

/* write error messages for web browsers */
static void
cgi_error(const char *message)
{
	printf("Content-Type: text/html\r\n");
	printf("\n");
	printf("<html>\n");
	printf("<head>\n");
	printf("<title>Error</title>\n");
	printf("</head>\n");
	printf("<body>\n");
	printf("%s error:\n", PROGRAM_NAME);
	printf("<br />\n");
	printf("%s<br />\n", message);
	printf("</body>\n");
	printf("</html>\n");
}

/*-------------------------------------------------------------------------
 * httpd-mode
 *-------------------------------------------------------------------------
 */

int
httpd_main(int port, const char *listen_addresses)
{

	if (listen_addresses == NULL)
		listen_addresses = "localhost";

	elog(INFO, "httpd: start (port=%d)", port);
	pgut_abort_level = FATAL;
	pgut_httpd(port, listen_addresses, httpd_callback);

	return 0;
}

static const char *
httpd_callback(StringInfo out, const http_request *request)
{
	const char *path;
	char	   *url = NULL;
	const char *xml;
	List	   *vars = NIL;
	const char *result = "*500";

	if (request->url[0] != '/')
	{
		elog(WARNING, "URL parameter is invalid");
		return "*400";
	}

	path = request->url;
	while (*path == '/') { path++; }
	url = pgut_strdup(path);

	elog(DEBUG2, "request: URL=%s, path=%s, params=%s",
		 request->url, url, (request->params ? request->params : "(null)"));

	/* return a database list if root directory is required. */
	if (url[0] == '\0')
	{
		xmlDocPtr	xml = makeDatabaseList();
		Database *db = getFirstDatabase();

		if (xml != NULL)
		{
			/* change directory */
			if (chdir(db->report_dir) < 0)
				return "*500*";
			result = applyStylesheet(out, xml, XSL_DBNAMES, NIL);
			xmlFreeDoc(xml);
		}
		else
		{
			result = "*404";
			goto done;
		}
		goto done;
	}
	else
	{
		char *cginame = strrchr(url, '/');

		if (cginame == NULL)
			cginame = url;
		else
			cginame ++;

		if (strcmp(cginame, URL_PATH) != 0)
		{
			result = "*404";
			goto done;
		}
	}


	/* make a variable table from URL parameters, id and template */
	vars = makeVariables(request->params);
	xml = getVariable(vars, "template");

	/* Make a database report. */
	result = makeDatabaseReport(out, vars, xml);

done:
	free(url);
	list_destroy(vars, freeVariable);
	xsltCleanupGlobals();
	xmlCleanupParser();
	return result;
}

/*-------------------------------------------------------------------------
 * common routines for httpd and cgi modes
 *-------------------------------------------------------------------------
 */

static const char *
makeDatabaseReport(StringInfo out, List *vars, const char *xml)
{
	const char	   *result;
	struct stat		st;
	char			path[MAXPGPATH];
	const char	   *id;
	Database	   *db;

	/* return a database list if template-xml is "dbnames". */
	if (strcmp(xml, XML_DBNAMES) == 0)
	{
		xmlDocPtr	xml = makeDatabaseList();
		Database *db = getFirstDatabase();

		if (xml != NULL)
		{
			/* change directory */
			if (chdir(db->report_dir) < 0)
				return "*500*";
			result = applyStylesheet(out, xml, XSL_DBNAMES, NIL);
			xmlFreeDoc(xml);
		}
		else
		{
			return "*404";
		}

		return result;
	}

	/* Find a database connection information by "id" parameter. */
	if ((id = getVariable(vars, "id")) == NULL)
		return "*500";
	if ((db = getDatabase(id)) == NULL)
		return "*500";

	/* change directory */
	if (chdir(db->report_dir) < 0)
		return "*500*";

	/* Make a report if a XML file is found. */
	snprintf(path, sizeof(path), "template/%s.xml", xml);
	if (stat(&path[0], &st) != 0)
	{
		elog(WARNING, "template not found: %s", path);
		return "*404";
	}

	/* Setup connection information. */
	host = db->host;
	port = db->port;
	dbname = db->database;
	username = db->username;
	password = db->password;
	prompt_password = NO;

	/* Make a report. */
	vars = addVariable(vars, "host", host ? host : "");
	vars = addVariable(vars, "port", port ? port : "");
	vars = addVariable(vars, "database", dbname ? dbname : "");
	result = makeReport(out, path, vars);

	/* Clear connection information.*/
	host = NULL;
	port = NULL;
	dbname = NULL;
	username = NULL;
	password = NULL;

	return result;
}

/*
 * <dbnames>
 *   <dbname>
 *     <id>id</id>
 *     <host>host</host>
 *     <port>port</port>
 *     <database>database</database>
 *   </dbname>
 *   ...
 * </dbnames>
 */
static xmlDocPtr
makeDatabaseList(void)
{
	xmlDocPtr		doc;
	xmlNodePtr		root;
	ListCell	   *cell;
	List		   *dbs;

	dbs = getDatabases(NULL);

	/* make a XML document */
	doc = xmlNewDoc((const xmlChar *) "1.0");

	/* make a root node named "dbnames" */
	root = xmlNewDocNode(doc, NULL, (const xmlChar *) "dbnames", NULL);
	xmlDocSetRootElement(doc, root);

	/* add databases */
	foreach(cell, dbs)
	{
		Database   *db = lfirst(cell);
		xmlNodePtr	dbname;

		dbname = xmlNewChild(root, NULL, (const xmlChar *) "dbname", NULL);
		xmlNewTextChild(dbname, NULL,
			(const xmlChar *) "id",
			(const xmlChar *) db->id);
		xmlNewTextChild(dbname, NULL,
			(const xmlChar *) "host",
			(const xmlChar *) db->host);
		xmlNewTextChild(dbname, NULL,
			(const xmlChar *) "port",
			(const xmlChar *) db->port);
		xmlNewTextChild(dbname, NULL,
			(const xmlChar *) "database",
			(const xmlChar *) db->database);
	}

	/* list_destroy(dbs, freeDatabase); */

	return doc;
}

/* decode uuencoded string */
static char *
decode(const char *str, int length)
{
    int				i, pos;
	char		   *buf;
	unsigned char	c;
	char			wk[3];

	buf = pgut_malloc(length + 1);
	for (i = 0, pos = 0 ; i < length ; i++)
	{
		c = str[i];

		if (c == '%')
		{
			i++;
			memcpy(wk, str+i, 2);
			wk[2] = 0x00;

			c = (char) strtol(wk, NULL, 16);
			i++;
		}

		buf[pos++] = c;
	}

	buf[pos] = 0x00;

	return buf;
}

/*-------------------------------------------------------------------------
 * variable managements
 *-------------------------------------------------------------------------
 */

/* make a variable table */
List *
makeVariables(const char *parameters)
{
	List	   *vars = NIL;
	const char *name;

	/* fast path for no vars */
	if (parameters == NULL || parameters[0] == '\0')
		return NIL;

	name = parameters;
	for (;;)
	{
		Variable	   *var;
		const char	   *value;

		/* variable format is 'name=value' */
		if ((value = strchr(name, '=')) == NULL || name == value)
		{
			elog(WARNING, "bad url format \"%s\"", parameters);
			break;
		}

		var = pgut_new(Variable);
		var->name = decode(name, value - name);
		value++;

		vars = lappend(vars, var);

		/* split with '&' */
		name = strchr(value, '&');
		if (name == NULL)
			name = value + strlen(value);
		if (name == value)
		{
			/* format is "name=&..." */
			var->value = pgut_strdup("");
		}
		else
		{
			/* format is "name=value&..." */
			var->value = decode(value, name - value);
		}

		/* goto next var or exit */
		if (*name == '&')
			name++;
		else
			break;
	}

	return vars;
}

/* add a variable to the variable table */
List *
addVariable(List *vars, const char *name, const char *value)
{
	Variable *var;

	var = pgut_new(Variable);
	var->name = pgut_strdup(name);
	var->value = pgut_strdup(value);

	/* XXX: unique checks needed? */
	return lappend(vars, var);
}

/* find a variable from variable table */
const char *
getVariable(List *vars, const char *name)
{
	ListCell *cell;

	foreach(cell, vars)
	{
		Variable *var = lfirst(cell);
		if (strcmp(var->name, name) == 0)
			return var->value;
	}

	return NULL;
}

/* free Variable object */
void
freeVariable(Variable *var)
{
	if (var == NULL)
		return;

	free(var->name);
	free(var->value);
	free(var);
}
