#############################################################################
#  Copyright (C) 2009 Nippon Telegraph and Telephone Corporation
#############################################################################

#####################################################################
# Function: pg_report_config.pm
#
#
# summary:
# Define actual procudure as query to generate report. These queries
# are executed in "pg_make_report.pl".
#
#
# recital:
#
#
#####################################################################

use warnings;
use strict;
use Carp;
use utf8;
use base qw(Exporter);

our @EXPORT_OK = qw($report_resources);

#-------------------------------------------------------------------------------
# Query for getting snapshot information from snapshot manage DB
#
# Edit follwoing query or add new query, then you can add new monitoring item by 
# using taken snapshot info. 
#
# The concrete way is described following. See [explanation] tag. 
#
#-------------------------------------------------------------------------------
our $report_resources = {

    #---------------------------------------------------------------------------
    # Level 1: Host information and DB Information
    # 
    # Define getting unique information for each host or DB.
    #---------------------------------------------------------------------------
    
    #---------------------------------------------------------------------------
    #  [explanation]
    #  Define getting information way from snapshot manage DB.
    #  Define format like following..
    #
    #     Information stored array => {target information definition}
    #
    #   "Information stored array" is referable on template file.
    #    ex) host_info, db_info...
    #   "target information definition" contains following factors.
    #
    #   - query
    #     Query for getting information from snapshot manage DB.
    #     It use some "bind" variables.
    #
    #   - bind
    #     Actual values set to bind variables. We also can set following keywords
    #     
    #        START_ID .... Start snapshot id
    #        END_ID ...... End snapshot od
    #        START_DATE .. Start timestamp
    #        END_DATE .... End timestamp
    #        DBID ........ target DBID
    #        HOSTID  ..... target Host id
    #
    #  - variables
    #    Define mapping variables on template-file and variables in information
    #    taken by query.
    #
    #    "variables" is define as hash which using name of variables as key.
    #    The hash contains following..
    #
    #      * colname => column name of query
    #              We can refer the information through variables (in template file).
    #              It is omissible, and in that case we set "name of variables" as 
    #              column name.
    #
    #      * stats   => value of data trend
    #              Define keyword of what kind of trend we calculate.
    #              We can use following keyword.
    #
    #                 = diff ...... latest - oldest
    #                 = mean ...... average (sum of all value / num of snapshot)
    #                 = diffmean .. diff / duration (sec)
    #                 = variance .. variance
    #                 = max ....... max value in specified snapshots
    #                 = min ....... min value in specified snasphots
    #                 = latest .... latest value
    #                 = oldest .... oldest value
    #                 = allvalues . all value
    #
    #              It is omissible, and in that case we set stats to "all keyword".
    #
    #      * default => acquirable value through "name of varibales"
    #              It is omissible, and in that case we set stats to "latest".
    #---------------------------------------------------------------------------


    #
    # Host basic information
    #
    # [explanation]
    # Host basic information are referable in template file as following format
    # $hostinfo[0]->{kernel_version}
    #
    hostinfo => {

        query => q(
            SELECT kernel_version, processors, mem_total, swap_total,
                   pg_version, maxwritten_clean 
            FROM t_hostinfo
            WHERE host_id = ? AND host_snapshot_date > ? AND host_snapshot_date <= ?
            ORDER BY host_snapshot_id
        ),

        bind => [qw(
            HOSTID
            START_DATE
            END_DATE
        )],

        variables => {
            #
            # [explanation]
            # variables => {define}
            # Describe above format.
            # please see following example..
            #
            kernel_version => {
                colname => 'kernel_version', 
                stats   => [qw(allvalues latest)],
                default  => 'latest'
            },
            processors     => { stats   => [qw(allvalues latest)], },
            
            #
            # [explanation]
            # Follwoing expample is all definitoin is abbreviated.
            #           colname => name of varibales
            #           stats   => all value
            #           default => latest
            #
            mem_total      => {},
            swap_total     => {},
            pg_version     => { stats    => [qw(allvalues latest)], },
            maxwritten_clean => {},
        },
    },
    
    #
    # DB basic information
    #
    dbinfo => {
        query => q(
            SELECT dbsize, conn_number, datfrozenxid_age, num_user_rel,
				db_xact_commit, db_xact_rollback, db_blks_read, db_blks_hit,
				db_tup_returned, db_tup_fetched, db_tup_inserted, db_tup_updated, db_tup_deleted
            FROM t_dbinfo
            WHERE snapshot_id BETWEEN ? and ? and dbid = ?
            ORDER BY snapshot_id
        ),

        bind => [qw(
            START_ID
            END_ID
            DBID
        )],

        variables => {
            dbsize           => {},
            conn_number      => {},
            datfrozenxid_age  => {},
            num_user_rel     => {},
			db_xact_commit   => {},
			db_xact_rollback => {},
			db_blks_read     => {},
			db_blks_hit      => {},
			db_tup_returned  => {},
			db_tup_fetched   => {},
			db_tup_inserted  => {},
			db_tup_updated   => {},
			db_tup_deleted   => {},
        },
    },


    #
    # checkpoint log information
    #
    checkpointinfo => {
        query => q(
            SELECT c.checkpoint_act_time AS checkpoint_act_time,
                c.checkpoint_start_trig AS checkpoint_start_trig,
                c.checkpoint_num_buffers AS checkpoint_num_buffers,
                c.checkpoint_create_wal AS checkpoint_create_wal,
                c.checkpoint_delete_wal AS checkpoint_delete_wal,
                c.checkpoint_recycle_wal AS checkpoint_recycle_wal,
                c.checkpoint_write_duration AS checkpoint_write_duration,
                c.checkpoint_sync_duration AS checkpoint_sync_duration,
                c.checkpoint_total_duration AS checkpoint_total_duration
            FROM t_checkpointinfo c LEFT JOIN t_hostinfo h ON (h.host_snapshot_id = c.host_snapshot_id)
            WHERE h.host_id = ? AND h.host_snapshot_date > ? AND h.host_snapshot_date <= ?
            ORDER BY c.host_snapshot_id
        ),
        
        bind => [qw(
            HOSTID
            START_DATE
            END_DATE
        )],

        variables => {
            checkpoint_act_time => {stats => [qw(allvalues)], default => 'allvalues'},
            checkpoint_start_trig => {stats => [qw(latest allvalues)]},
            checkpoint_num_buffers => {},
            checkpoint_create_wal => {},
            checkpoint_delete_wal => {},
            checkpoint_recycle_wal => {},
            checkpoint_write_duration => {},
            checkpoint_sync_duration => {},
            checkpoint_total_duration => {},
        },
    },
    
    #
    # autovacuum log information
    #
    autovacuuminfo => {
        query => q(
            SELECT autovacuum_table, autovacuum_num_page_remove,
                   autovacuum_num_page_remain, autovacuum_num_tup_remove,
                   autovacuum_num_tup_remain, autovacuum_duration
            FROM t_autovacuuminfo
            WHERE snapshot_id BETWEEN ? and ? and dbid = ?
            ORDER BY snapshot_id
        ),    

        bind => [qw(
            START_ID
            END_ID
            DBID
        )],
        
        variables => {
            autovacuum_table  => {stats => [qw(allvalues), qw(latest)]},
            autovacuum_num_page_remove => {},
            autovacuum_num_page_remain => {},
            autovacuum_num_tup_remove => {},
            autovacuum_num_tup_remain => {},
            autovacuum_duration => {},
        },
        
        primarykey => [qw(autovacuum_table)],

    },

	#
	# autovacuum start infomation
	#
    all_autovacuuminfo => {
        query => q(
            SELECT autovacuum_act_time - (autovacuum_duration::bigint::text)::interval as autovacuum_start_time,
				   autovacuum_table, 
                   autovacuum_duration
            FROM t_autovacuuminfo
            WHERE snapshot_id BETWEEN ? and ? and dbid = ?
        ),    

        bind => [qw(
            START_ID
            END_ID
            DBID
        )],
        
        variables => {
            autovacuum_start_time => {stats => [qw(allvalues)]},
            autovacuum_table      => {stats => [qw(allvalues)]},
            autovacuum_duration   => {},
        },
		primarykey => [qw(autovacuum_start_time autovacuum_table)],

    },

	#
	# device usage information
	#

	fs_size => {
		query => q(
			SELECT tblspc_oid as tblspc, name, device_id, device_name, location,
			  pg_size_pretty(used) as used_size,
			  pg_size_pretty(avail) as avail_size,
			  pg_size_pretty(total) as total_size,
			  (used::float / total::float ) * 100 as use_ratio
			FROM t_fs_size f LEFT JOIN t_hostinfo h ON (h.host_snapshot_id = f.host_snapshot_id)
            WHERE h.host_id = ? AND h.host_snapshot_date > ? AND h.host_snapshot_date <= ?
		),

		bind => [qw(
            HOSTID
            START_DATE
            END_DATE
        )],

        variables => {
            tblspc          => {stats =>[qw(latest)]},
            name            => {stats =>[qw(latest)]},
            device_id       => {stats =>[qw(latest)]},
            device_name     => {stats =>[qw(latest)]},
            location        => {stats =>[qw(latest)]},
            used_size       => {stats =>[qw(latest)]},
            avail_size      => {stats =>[qw(latest)]},
            total_size      => {stats =>[qw(latest)]},
            use_ratio       => {stats =>[qw(latest)]},
        },

		primarykey => [qw(device_id name location)],

	},


	#
	# long transaction information
	#

	host_act => {
		query => q(
			SELECT dbname, procpid, usename, client_addr, waiting, max(duration) as elapsed_time,check_time, 
			  backend_start, current_query
            FROM t_hostactivity a LEFT JOIN t_hostinfo h ON (h.host_snapshot_id = a.host_snapshot_id)
            WHERE h.host_id = ? AND h.host_snapshot_date >= ? AND h.host_snapshot_date <= ?
			GROUP BY dbname, procpid, usename, waiting, client_addr, check_time, backend_start, current_query
			ORDER BY elapsed_time DESC
		),

		bind => [qw(
			HOSTID
			START_DATE
			END_DATE
		)],

		variables => {
			dbname        => {stats =>[qw(latest)]},
			procpid       => {stats =>[qw(latest)]},
			usename       => {stats =>[qw(latest)]},
			client_addr   => {stats =>[qw(latest)]},
			waiting       => {stats =>[qw(latest)]},
			elapsed_time  => {stats =>[qw(latest)]},
			check_time    => {stats =>[qw(latest)]},
			backend_start => {stats =>[qw(latest)]},
			current_query => {stats =>[qw(latest)]},
		},

		primarykey => [qw(dbname procpid current_query)],

	},


	#
	# many access table (many seq_scan occured table) Top 10
	#

    seq_scan_10 => {
        query => q(
            SELECT e.schemaname, e.tablename, (e.seq_scan - s.seq_scan) as seq_scan,
			(e.seq_tup_read - s.seq_tup_read) as seq_tup_read,
			  CASE (e.seq_scan - s.seq_scan) 
			    WHEN 0 THEN 0
			  ELSE
			    ((e.seq_tup_read::float - s.seq_tup_read::float) / (e.seq_scan::float - s.seq_scan::float))
			  END as tup_per_scan
            FROM
			  (SELECT schemaname, tablename, seq_scan, seq_tup_read
			   FROM t_tableinfo WHERE snapshot_id = ? and dbid = ? and seq_scan > 0) s
			 JOIN
			  (SELECT schemaname, tablename, seq_scan, seq_tup_read
			   FROM t_tableinfo WHERE snapshot_id = ? and dbid = ? and seq_scan > 0) e
			 ON s.tablename = e.tablename
            ORDER BY seq_tup_read DESC
            LIMIT 10
        ),

		bind => [qw(
			START_ID
			DBID
			END_ID
			DBID
		)],

		variables => {
			schemaname => { stats => [qw(latest)]},
			tablename => { stats => [qw(latest)]},
			seq_scan => { stats => [qw(latest)]},
			seq_tup_read => { stats => [qw(latest)]},
			tup_per_scan => { stats => [qw(latest)]},
		},

		primarykey => [qw(schemaname tablename)],

	},
    

	#
	# fragmented table (contains many dead redord table) Top 10 
	#

    dead_rec_10 => {
        query => q(
            SELECT schemaname, tablename, n_dead_tuple, n_live_tuple, (n_dead_tuple::float / n_live_tuple::float * 100) as dead_ratio
            FROM t_tableinfo
            WHERE snapshot_id = ? and dbid = ? and n_live_tuple > 0 and n_dead_tuple > 0
            ORDER BY n_dead_tuple DESC
            LIMIT 10
        ),

        bind => [qw(
            END_ID
            DBID
        )],

        variables => {
            schemaname => { stats => [qw(latest)]},
            tablename  => { stats => [qw(latest)]},
            n_dead_tuple => { stats => [qw(latest)]},
            n_live_tuple => { stats => [qw(latest)]},
	    dead_ratio   => { stats => [qw(latest)]},
        },

        primarykey => [qw(schemaname tablename)],

    },

    #
    # low density in user table Top 10
    #

    low_density_10 => {
        query => q(
            SELECT schemaname, relname, reltuples, logical_pages, physical_pages, tratio
            FROM t_tablefill
            WHERE snapshot_id = ? and dbid = ? 
            ORDER BY tratio
            LIMIT 10
        ),

        bind => [qw(
            END_ID
            DBID
        )],

        variables => {
            schemaname     => { stats => [qw(latest)]},
            relname        => { stats => [qw(latest)]},
            reltuples      => { stats => [qw(latest)]},
            logical_pages  => { stats => [qw(latest)]},
            physical_pages => { stats => [qw(latest)]},
            tratio        => { stats => [qw(latest)]},
        },

        primarykey => [qw(schemaname relname)],

    },


    #
    # cluster-key column correlation worst Top 10
    #

    column_corr_10 => {
        query => q(
            SELECT schemaname, relname, column_name, correlation
            FROM t_columncorr
            WHERE snapshot_id = ? and dbid = ?
            ORDER BY correlation
            LIMIT 10
        ),

        bind => [qw(
            END_ID
            DBID
        )],

        variables => {
            schemaname     => { stats => [qw(latest)]},
            relname        => { stats => [qw(latest)]},
            column_name   => { stats => [qw(latest)]},
            correlation   => { stats => [qw(latest)]},
        },

        primarykey => [qw(schemaname relname)],

    },

    
    #
    # DB setting information
    #
    dbsetting => {
        query => q(
            SELECT name, setting, source FROM t_dbsetting
            WHERE snapshot_id BETWEEN ? and ? and dbid = ?
            ORDER BY name, snapshot_id
        ),
        
        bind => [qw(
            START_ID
            END_ID
            DBID
        )],
                    
        variables => {
            name    => { stats => [qw(allvalues latest)], },
            setting => { stats => [qw(allvalues latest)], },
            source  => { stats => [qw(allvalues latest)], },
        },
        
        primarykey => [qw(name)],
    },


	#
	# pg_stat_statements information
	#
	statstatements => {
		query => q(
		SELECT n.usename as usename , n.datname as datname , n.query as query,
               CASE 
					WHEN o.calls IS NULL THEN n.calls
                	ELSE (n.calls - o.calls)
               END as calls,
               CASE 
					WHEN o.total_time IS NULL THEN n.total_time
               		ELSE (n.total_time - o.total_time)
               END as total_time,
               CASE 
					WHEN o.rows IS NULL THEN n.rows
               		ELSE (n.rows - o.rows)
               END as rows
		FROM
			(SELECT * FROM t_statstatements st,
				(SELECT max(host_snapshot_id) as id FROM t_hostinfo WHERE host_snapshot_date <= ? AND host_id = ?) sp_new
				 WHERE st.host_snapshot_id = sp_new.id ) n
			LEFT JOIN
			(SELECT * FROM t_statstatements st,
				(SELECT min(host_snapshot_id) as id FROM t_hostinfo WHERE host_snapshot_date >= ? AND host_id = ?) sp_old
				 WHERE st.host_snapshot_id = sp_old.id ) o
		ON n.usename = o.usename AND n.datname = o.datname AND n.query = o.query
		ORDER BY total_time DESC LIMIT 10
		),	

		bind => [qw(
			END_DATE
			HOSTID
			START_DATE
			HOSTID
		)],

		variables => {
			usename => {stats => [qw(allvalues latest)],},
			datname => {stats => [qw(allvalues latest)],},
			query   => {stats => [qw(allvalues latest)],},
			calls      => {},
			total_time => {},
			rows       => {},
		},

		primarykey => [qw(usename datname query)],

	},

	#
	# pg_stat_user_functions information
	#
    statfunc => {
        query => q(
            SELECT n.schemaname as schemaname , n.funcname as funcname,
               CASE
					WHEN o.calls IS NULL THEN n.calls
               		ELSE (n.calls - o.calls)
               END as calls,
               CASE 
					WHEN o.total_time IS NULL THEN n.total_time
               		ELSE (n.total_time - o.total_time)
               END as total_time,
               CASE
					WHEN o.self_time IS NULL THEN n.self_time
               		ELSE (n.self_time - o.self_time)
               END as self_time
            FROM
              (SELECT schemaname, funcname, calls, total_time, self_time
               FROM t_statfunc WHERE snapshot_id = ? and dbid = ?) n
			LEFT JOIN
              (SELECT schemaname, funcname, calls, total_time, self_time
               FROM t_statfunc WHERE snapshot_id = ? and dbid = ?) o
            ON n.schemaname = o.schemaname AND n.funcname = o.funcname
            ORDER BY total_time DESC
            LIMIT 10
		),	

		bind => [qw(
			END_ID
			DBID
			START_ID
			DBID
		)],

		variables => {
			schemaname => {stats => [qw(allvalues latest)],},
			funcname   => {stats => [qw(allvalues latest)],},
			calls      => {},
			total_time => {},
			self_time  => {},
		},

		primarykey => [qw(schemaname funcname)],

	},
        
    #
    # Schema information (contains table, index information as children)
    #
    schemas => {
        query => q(
            SELECT schemaname FROM t_tableinfo
            WHERE snapshot_id BETWEEN ? and ? and dbid = ?
            ORDER BY schemaname, snapshot_id
        ),

        bind => [qw(
            START_ID
            END_ID
            DBID
        )],
            
        variables => {
            schemaname => { stats => [qw(latest)], },
        },

        primarykey => [qw(schemaname)],
        

        #-----------------------------------------------------------------------
        # Level 2：Table and Index information ("table" contains column information as children)
        #-----------------------------------------------------------------------
        children => {
            #
            # Table statistics information
            #
            tables => {
                query => q(
                    SELECT schemaname, tablename, table_oid, relpages, reltuples, n_tup_ins, n_tup_upd, n_tup_del,
                           n_tup_hot_upd, seq_scan, idx_scan, seq_tup_read,
                           idx_tup_fetch, heap_blks_hit,
                           idx_blks_hit, toast_blks_hit, tidx_blks_hit,
                           heap_blks_read, idx_blks_read, toast_blks_read,
                           tidx_blks_read, n_live_tuple, n_dead_tuple,
                           relation_size,
			   last_vacuum, last_autovacuum, last_analyze, last_autoanalyze
                    FROM   t_tableinfo 
                    WHERE snapshot_id BETWEEN ? and ? and dbid = ?
                    ORDER BY schemaname, tablename, snapshot_id
                ),

                bind => [qw(
                    START_ID
                    END_ID
                    DBID
                )],
            
                variables => {
                    tablename => { stats => [qw(latest)], },
                    table_oid  => { stats => [qw(latest)], },
                    relpages => {},
                    reltuples => {},
                    n_tup_ins => {},
                    n_tup_upd => {},
                    n_tup_del => {},
                    n_tup_hot_upd => {},
                    seq_scan => {},
                    idx_scan => {},
                    seq_tup_read => {},
                    idx_tup_fetch => {},
                    heap_blks_hit => {},
                    idx_blks_hit => {},
                    toast_blks_hit => {},
                    tidx_blks_hit => {},
                    heap_blks_read => {},
                    idx_blks_read => {},
                    toast_blks_read => {},
                    tidx_blks_read => {},
                    n_live_tuple => {},
                    n_dead_tuple => {},
                    relation_size => {},
		    last_vacuum => { stats => [qw(latest)], },
		    last_autovacuum => { stats => [qw(latest)], },
		    last_analyze => { stats => [qw(latest)], },
		    last_autoanalyze => { stats => [qw(latest)], },
                },
                
                primarykey => [qw(schemaname tablename)],
                

                #
                # Level3 : column information
                #
                children => {
                    #
                    # Column statistics information 
                    #
                    columns => {
                        query => q(
                            SELECT schemaname, tablename, column_name,
                                   n_distinct, correlation
                            FROM t_columninfo
                            WHERE snapshot_id BETWEEN ? and ? and dbid = ?
                            ORDER BY schemaname, tablename, column_name, snapshot_id
                        ),
                        
                        bind => [qw(
                            START_ID
                            END_ID
                            DBID
                        )],
                        
                        variables => {
                            column_name => { stats => [qw(latest)], },
                            n_distinct => {},
                            correlation => {},
                        },
                        
                        primarykey => [qw(schemaname tablename column_name)],
                    }, 
                },
            },
                
            #
            # Index statistics information
            #
            indexes => {
                query => q(
                    SELECT schemaname, index_oid, indexname, tablename,
                           indexdef, cluster_key, index_size, idx_scan, bmp_tup_fetch,
                           idx_tup_fetch, idx_blks_read, idx_blks_hit
                    FROM   t_indexinfo 
                    WHERE snapshot_id BETWEEN ? and ? and dbid = ?
                    ORDER BY schemaname, tablename, snapshot_id
                ),

                bind => [qw(
                    START_ID
                    END_ID
                    DBID
                )],
                
                variables => {
                    index_oid   => { stats => [qw(latest)], },
                    indexname   => { stats => [qw(latest)], },
                    tablename   => { stats => [qw(latest)], },
                    indexdef    => { stats => [qw(latest)], },
                    cluster_key => { stats => [qw(latest)], },
                    index_size  => {},
                    idx_scan    => {},
                    bmp_tup_fetch  => {},
                    idx_tup_fetch => {},
                    idx_blks_read => {},
                    idx_blks_hit  => {},
                },

                primarykey => [qw(schemaname indexname)],
            },
        },
    },
};

1;




