/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.collections4.bag;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;

import org.apache.commons.collections4.Bag;
import org.apache.commons.collections4.SortedBag;
import org.junit.jupiter.api.Test;

/**
 * Extension of {@link AbstractBagTest} for exercising the {@link TreeBag}
 * implementation.
 */
public class TreeBagTest<T> extends AbstractSortedBagTest<T> {

    @Override
    public String getCompatibilityVersion() {
        return "4";
    }

    @Override
    public SortedBag<T> makeObject() {
        return new TreeBag<>();
    }

    @SuppressWarnings("unchecked")
    public SortedBag<T> setupBag() {
        final SortedBag<T> bag = makeObject();
        bag.add((T) "C");
        bag.add((T) "A");
        bag.add((T) "B");
        bag.add((T) "D");
        return bag;
    }

    @Test
    public void testCollections265() {
        final Bag<Object> bag = new TreeBag<>();

        assertThrows(IllegalArgumentException.class, () -> bag.add(new Object()));
    }

    @Test
    public void testCollections555() {
        final Bag<Object> bag = new TreeBag<>();

        assertThrows(NullPointerException.class, () -> bag.add(null));

        final Bag<String> bag2 = new TreeBag<>(String::compareTo);
        // jdk bug: adding null to an empty TreeMap works
        // thus ensure that the bag is not empty before adding null
        bag2.add("a");

        assertThrows(NullPointerException.class, () -> bag2.add(null));
    }

    @Test
    public void testOrdering() {
        final Bag<T> bag = setupBag();
        assertEquals("A", bag.toArray()[0], "Should get elements in correct order");
        assertEquals("B", bag.toArray()[1], "Should get elements in correct order");
        assertEquals("C", bag.toArray()[2], "Should get elements in correct order");
        assertEquals("A", ((SortedBag<T>) bag).first(), "Should get first key");
        assertEquals("D", ((SortedBag<T>) bag).last(), "Should get last key");
    }

//    public void testCreate() throws Exception {
//        Bag<T> bag = makeObject();
//        writeExternalFormToDisk((java.io.Serializable) bag, "src/test/resources/data/test/TreeBag.emptyCollection.version4.obj");
//        bag = makeFullCollection();
//        writeExternalFormToDisk((java.io.Serializable) bag, "src/test/resources/data/test/TreeBag.fullCollection.version4.obj");
//    }

}
