/**
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#pragma once

#include <memory>
#include <string>
#include <utility>
#include <vector>
#include <sstream>
#include <map>

#include "../nodes/MetricsBase.h"
#include "Connection.h"

namespace org::apache::nifi::minifi::state::response {

/**
 * Justification and Purpose: Provides repository metrics. Provides critical information to the
 * C2 server.
 *
 */
class RepositoryMetrics : public ResponseNode {
 public:
  RepositoryMetrics(std::string name, const utils::Identifier &uuid)
      : ResponseNode(std::move(name), uuid) {
  }

  explicit RepositoryMetrics(std::string name)
      : ResponseNode(std::move(name)) {
  }

  RepositoryMetrics()
      : ResponseNode("RepositoryMetrics") {
  }

  MINIFIAPI static constexpr const char* Description = "Metric node that defines repository metric information";

  std::string getName() const override {
    return "RepositoryMetrics";
  }

  void addRepository(const std::shared_ptr<core::Repository> &repo) {
    if (nullptr != repo) {
      repositories_.insert(std::make_pair(repo->getName(), repo));
    }
  }

  std::vector<SerializedResponseNode> serialize() override {
    std::vector<SerializedResponseNode> serialized;
    for (auto conn : repositories_) {
      auto repo = conn.second;
      SerializedResponseNode parent;
      parent.name = repo->getName();
      SerializedResponseNode datasize;
      datasize.name = "running";
      datasize.value = repo->isRunning();

      SerializedResponseNode datasizemax;
      datasizemax.name = "full";
      datasizemax.value = repo->isFull();

      SerializedResponseNode queuesize;
      queuesize.name = "size";
      queuesize.value = std::to_string(repo->getRepoSize());

      parent.children.push_back(datasize);
      parent.children.push_back(datasizemax);
      parent.children.push_back(queuesize);

      serialized.push_back(parent);
    }
    return serialized;
  }

  std::vector<PublishedMetric> calculateMetrics() override {
    std::vector<PublishedMetric> metrics;
    for (const auto& [_, repo] : repositories_) {
      metrics.push_back({"is_running", (repo->isRunning() ? 1.0 : 0.0), {{"metric_class", getName()}, {"repository_name", repo->getName()}}});
      metrics.push_back({"is_full", (repo->isFull() ? 1.0 : 0.0), {{"metric_class", getName()}, {"repository_name", repo->getName()}}});
      metrics.push_back({"repository_size", static_cast<double>(repo->getRepoSize()), {{"metric_class", getName()}, {"repository_name", repo->getName()}}});
    }
    return metrics;
  }

 protected:
  std::map<std::string, std::shared_ptr<core::Repository>> repositories_;
};

}  // namespace org::apache::nifi::minifi::state::response
