/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   Directory handling routines
   Copyright (C) Andrew Tridgell 1992-1998
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

extern int DEBUGLEVEL;

/*
   This module implements directory related functions for Samba.
*/

typedef struct _dptr_struct {
	struct _dptr_struct *next, *prev;
	int dnum;
	uint16 spid;
	connection_struct *conn;
	void *ptr;
	BOOL expect_close;
	char *wcard; /* Field only used for trans2_ searches */
	uint16 attr; /* Field only used for trans2_ searches */
	char *path;
} dptr_struct;

static struct bitmap *dptr_bmap;
static dptr_struct *dirptrs;

static int dptrs_open = 0;

#define INVALID_DPTR_KEY (-3)

/* OKKY:MAX_DIRECTORY_HANDLES == 2048 */

/****************************************************************************
 Initialise the dir bitmap.
****************************************************************************/

void init_dptrs(void)
{
  static BOOL dptrs_init=False;

  if (dptrs_init)
    return;

  dirptrs	= NULL;
  dptr_bmap = bitmap_allocate(MAX_DIRECTORY_HANDLES);

  if (!dptr_bmap)
    exit_server("out of memory in init_dptrs\n");

  dptrs_init = True;
}

/****************************************************************************
 Idle a dptr - the directory is closed but the control info is kept.
****************************************************************************/

static void dptr_idle(dptr_struct *dptr)
{
  if (dptr->ptr) {
    DEBUG(4,("Idling dptr dnum %d\n",dptr->dnum));
    dptrs_open--;
    CloseDir(dptr->ptr);
    dptr->ptr = NULL;
  }
}

/****************************************************************************
 Idle the oldest dptr.
****************************************************************************/

static void dptr_idleoldest(void)
{
  dptr_struct *dptr;

  /*
   * Go to the end of the list.
   */
  for(dptr = dirptrs; dptr && dptr->next; dptr = dptr->next)
    ;

  if(!dptr) {
    DEBUG(0,("No dptrs available to idle ?\n"));
    return;
  }

  /*
   * Idle the oldest pointer.
   */

  for(; dptr; dptr = dptr->prev) {
    if (dptr->ptr) {
      dptr_idle(dptr);
      return;
    }
  }
}

/****************************************************************************
 Get the dptr_struct for a dir index.
****************************************************************************/

static dptr_struct *dptr_get(int key, BOOL forclose)
{
  dptr_struct *dptr;

  for(dptr = dirptrs; dptr; dptr = dptr->next) {
    if(dptr->dnum == key) {
      if (!forclose && !dptr->ptr) {
        if (dptrs_open >= MAX_OPEN_DIRECTORIES)
          dptr_idleoldest();
        DEBUG(4,("Reopening dptr key %d\n",key));
        if ((dptr->ptr = OpenDir(dptr->conn, dptr->path, True)))
          dptrs_open++;
      }
      DLIST_PROMOTE(dirptrs,dptr);
      return dptr;
    }
  }
  return(NULL);
}

/****************************************************************************
 Get the dptr ptr for a dir index.
****************************************************************************/

static void *dptr_ptr(int key)
{
  dptr_struct *dptr = dptr_get(key, False);

  if (dptr)
    return(dptr->ptr);
  return(NULL);
}

/****************************************************************************
 Get the dir path for a dir index.
****************************************************************************/

char *dptr_path(int key)
{
  dptr_struct *dptr = dptr_get(key, False);

  if (dptr)
    return(dptr->path);
  return(NULL);
}

/****************************************************************************
 Get the dir wcard for a dir index (lanman2 specific).
****************************************************************************/

char *dptr_wcard(int key)
{
  dptr_struct *dptr = dptr_get(key, False);

  if (dptr)
    return(dptr->wcard);
  return(NULL);
}

/****************************************************************************
 Set the dir wcard for a dir index (lanman2 specific).
 Returns 0 on ok, 1 on fail.
****************************************************************************/

BOOL dptr_set_wcard(int key, char *wcard)
{
  dptr_struct *dptr = dptr_get(key, False);

  if (dptr) {
    dptr->wcard = wcard;
    return True;
  }
  return False;
}

/****************************************************************************
 Set the dir attrib for a dir index (lanman2 specific).
 Returns 0 on ok, 1 on fail.
****************************************************************************/

BOOL dptr_set_attr(int key, uint16 attr)
{
  dptr_struct *dptr = dptr_get(key, False);

  if (dptr) {
    dptr->attr = attr;
    return True;
  }
  return False;
}

/****************************************************************************
 Get the dir attrib for a dir index (lanman2 specific)
****************************************************************************/

uint16 dptr_attr(int key)
{
  dptr_struct *dptr = dptr_get(key, False);

  if (dptr)
    return(dptr->attr);
  return(0);
}

/****************************************************************************
 Close a dptr (internal func).
****************************************************************************/

static void dptr_close_internal(dptr_struct *dptr)
{
  DEBUG(4,("closing dptr key %d\n",dptr->dnum));

  DLIST_REMOVE(dirptrs, dptr);

  /* 
   * Free the dnum in the bitmap. Remember the dnum value is always 
   * biased by one with respect to the bitmap.
   */

  if(bitmap_query( dptr_bmap, dptr->dnum - 1) != True) {
    DEBUG(0,("dptr_close_internal : Error - closing dnum = %d and bitmap not set !\n",
			dptr->dnum ));
  }

  bitmap_clear(dptr_bmap, dptr->dnum - 1);

  if (dptr->ptr) {
    CloseDir(dptr->ptr);
    dptrs_open--;
  }

  /* Lanman 2 specific code */
  if (dptr->wcard)
    free(dptr->wcard);
  string_set(&dptr->path,"");
  free((char *)dptr);
}

/****************************************************************************
 Close a dptr given a key.
****************************************************************************/

void dptr_close(int *key)
{
  dptr_struct *dptr;

  if(*key == INVALID_DPTR_KEY)
    return;

  /* OS/2 seems to use -1 to indicate "close all directories" */
  if (*key == -1) {
    dptr_struct *next;
    for(dptr = dirptrs; dptr; dptr = next) {
      next = dptr->next;
      dptr_close_internal(dptr);
    }
    *key = INVALID_DPTR_KEY;
    return;
  }

  dptr = dptr_get(*key, True);

  if (!dptr) {
    DEBUG(0,("Invalid key %d given to dptr_close\n", *key));
    return;
  }

  dptr_close_internal(dptr);

  *key = INVALID_DPTR_KEY;
}

/****************************************************************************
 Close all dptrs for a cnum.
****************************************************************************/

void dptr_closecnum(connection_struct *conn)
{
  dptr_struct *dptr, *next;
  for(dptr = dirptrs; dptr; dptr = next) {
    next = dptr->next;
    if (dptr->conn == conn)
      dptr_close_internal(dptr);
  }
}

/****************************************************************************
 Idle all dptrs for a cnum.
****************************************************************************/

void dptr_idlecnum(connection_struct *conn)
{
  dptr_struct *dptr;
  for(dptr = dirptrs; dptr; dptr = dptr->next) {
    if (dptr->conn == conn && dptr->ptr)
      dptr_idle(dptr);
  }
}

/****************************************************************************
 Close a dptr that matches a given path, only if it matches the spid also.
****************************************************************************/

void dptr_closepath(char *path,uint16 spid)
{
  dptr_struct *dptr, *next;
  for(dptr = dirptrs; dptr; dptr = next) {
    next = dptr->next;
    if (spid == dptr->spid && strequal(dptr->path,path))
      dptr_close_internal(dptr);
  }
}

/****************************************************************************
 Start a directory listing.
****************************************************************************/

static BOOL start_dir(connection_struct *conn,char *directory)
{
  DEBUG(5,("start_dir dir=%s\n",directory));

  if (!check_name(directory,conn))
    return(False);
  
  if (! *directory)
    directory = ".";

  conn->dirptr = OpenDir(conn, directory, True);
  if (conn->dirptr) {    
    dptrs_open++;
    string_set(&conn->dirpath,directory);
    return(True);
  }
  
  return(False);
}

/****************************************************************************
 Try and close the oldest handle not marked for
 expect close in the hope that the client has
 finished with that one.
****************************************************************************/

static void dptr_close_oldest(BOOL old)
{
  dptr_struct *dptr;

  /*
   * Go to the end of the list.
   */
  for(dptr = dirptrs; dptr && dptr->next; dptr = dptr->next)
    ;

  if(!dptr) {
    DEBUG(0,("No old dptrs available to close oldest ?\n"));
    return;
  }

  /*
   * If 'old' is true, close the oldest oldhandle dnum (ie. 1 < dnum < 256) that
   * does not have expect_close set. If 'old' is false, close
   * one of the new dnum handles.
   */

  for(; dptr; dptr = dptr->prev) {
    if ((old && (dptr->dnum < 256) && !dptr->expect_close) ||
        (!old && (dptr->dnum > 255))) {
      dptr_close_internal(dptr);
      return;
    }
  }
}

/****************************************************************************
 Create a new dir ptr. If the flag old_handle is true then we must allocate
 from the bitmap range 0 - 255 as old SMBsearch directory handles are only
 one byte long. If old_handle is false we allocate from the range
 256 - MAX_DIRECTORY_HANDLES. We bias the number we return by 1 to ensure
 a directory handle is never zero. All the above is folklore taught to
 me at Andrew's knee.... :-) :-). JRA.
****************************************************************************/

int dptr_create(connection_struct *conn,char *path, BOOL old_handle, BOOL expect_close,uint16 spid)
{
  dptr_struct *dptr;

  if (!start_dir(conn,path))
    return(-2); /* Code to say use a unix error return code. */

  if (dptrs_open >= MAX_OPEN_DIRECTORIES)
    dptr_idleoldest();

  dptr = (dptr_struct *)malloc(sizeof(dptr_struct));
  if(!dptr) {
    DEBUG(0,("malloc fail in dptr_create.\n"));
    return -1;
  }

  ZERO_STRUCTP(dptr);

  if(old_handle) {

    /*
     * This is an old-style SMBsearch request. Ensure the
     * value we return will fit in the range 1-255.
     */

    dptr->dnum = bitmap_find(dptr_bmap, 0);

    if(dptr->dnum == -1 || dptr->dnum > 254) {

      /*
       * Try and close the oldest handle not marked for
       * expect close in the hope that the client has
       * finished with that one.
       */

      dptr_close_oldest(True);

      /* Now try again... */
      dptr->dnum = bitmap_find(dptr_bmap, 0);

      if(dptr->dnum == -1 || dptr->dnum > 254) {
        DEBUG(0,("dptr_create: returned %d: Error - all old dirptrs in use ?\n", dptr->dnum));
        free((char *)dptr);
        return -1;
      }
    }
  } else {

    /*
     * This is a new-style trans2 request. Allocate from
     * a range that will return 256 - MAX_DIRECTORY_HANDLES.
     */

    dptr->dnum = bitmap_find(dptr_bmap, 255);

    if(dptr->dnum == -1 || dptr->dnum < 255) {

      /*
       * Try and close the oldest handle close in the hope that
       * the client has finished with that one. This will only
       * happen in the case of the Win98 client bug where it leaks
       * directory handles.
       */

      dptr_close_oldest(False);

      /* Now try again... */
      dptr->dnum = bitmap_find(dptr_bmap, 255);

      if(dptr->dnum == -1 || dptr->dnum < 255) {
        DEBUG(0,("dptr_create: returned %d: Error - all new dirptrs in use ?\n", dptr->dnum));
        free((char *)dptr);
        return -1;
      }
    }
  }

  bitmap_set(dptr_bmap, dptr->dnum);

  dptr->dnum += 1; /* Always bias the dnum by one - no zero dnums allowed. */

  dptr->ptr = conn->dirptr;
  string_set(&dptr->path,path);
  dptr->conn = conn;
  dptr->spid = spid;
  dptr->expect_close = expect_close;
  dptr->wcard = NULL; /* Only used in lanman2 searches */
  dptr->attr = 0; /* Only used in lanman2 searches */

  DLIST_ADD(dirptrs, dptr);

  DEBUG(3,("creating new dirptr %d for path %s, expect_close = %d\n",
	   dptr->dnum,path,expect_close));  

  return(dptr->dnum);
}

/****************************************************************************
 Fill the 5 byte server reserved dptr field.
****************************************************************************/

BOOL dptr_fill(char *buf1,unsigned int key)
{
  unsigned char *buf = (unsigned char *)buf1;
  void *p = dptr_ptr(key);
  uint32 offset;
  if (!p) {
    DEBUG(1,("filling null dirptr %d\n",key));
    return(False);
  }
  offset = TellDir(p);
  DEBUG(6,("fill on key %u dirptr 0x%lx now at %d\n",key,
	   (long)p,(int)offset));
  buf[0] = key;
  SIVAL(buf,1,offset | DPTR_MASK);
  return(True);
}

/****************************************************************************
 Fetch the dir ptr and seek it given the 5 byte server field.
****************************************************************************/

void *dptr_fetch(char *buf,int *num)
{
  unsigned int key = *(unsigned char *)buf;
  void *p = dptr_ptr(key);
  uint32 offset;
  if (!p) {
    DEBUG(3,("fetched null dirptr %d\n",key));
    return(NULL);
  }
  *num = key;
  offset = IVAL(buf,1)&~DPTR_MASK;
  SeekDir(p,offset);
  DEBUG(3,("fetching dirptr %d for path %s at offset %d\n",
	   key,dptr_path(key),offset));
  return(p);
}

/****************************************************************************
 Fetch the dir ptr.
****************************************************************************/

void *dptr_fetch_lanman2(int dptr_num)
{
  void *p = dptr_ptr(dptr_num);

  if (!p) {
    DEBUG(3,("fetched null dirptr %d\n",dptr_num));
    return(NULL);
  }
  DEBUG(3,("fetching dirptr %d for path %s\n",dptr_num,dptr_path(dptr_num)));
  return(p);
}

/****************************************************************************
 Check a filetype for being valid.
****************************************************************************/

BOOL dir_check_ftype(connection_struct *conn,int mode,SMB_STRUCT_STAT *st,int dirtype)
{
  if (((mode & ~dirtype) & (aHIDDEN | aSYSTEM | aDIR)) != 0)
    return False;
  return True;
}

/****************************************************************************
 Get an 8.3 directory entry.
****************************************************************************/

BOOL get_dir_entry(connection_struct *conn,char *mask,int dirtype,char *fname,
                   SMB_OFF_T *size,int *mode,time_t *date,BOOL check_descend)
{
  char *dname;
  BOOL found = False;
  SMB_STRUCT_STAT sbuf;
  pstring path;
  pstring pathreal;
  BOOL isrootdir;
  pstring filename;
  BOOL needslash;

  *path = *pathreal = *filename = 0;

  isrootdir = (strequal(conn->dirpath,"./") ||
	       strequal(conn->dirpath,".") ||
	       strequal(conn->dirpath,"/"));
  
  needslash = ( conn->dirpath[strlen(conn->dirpath) -1] != '/');

  if (!conn->dirptr)
    return(False);
  
  while (!found)
  {
    BOOL filename_is_mask = False;
    dname = ReadDirName(conn->dirptr);

    DEBUG(6,("readdir on dirptr 0x%lx now at offset %d\n",
          (long)conn->dirptr,TellDir(conn->dirptr)));
      
    if (dname == NULL) 
      return(False);
      
    pstrcpy(filename,dname);      

    if ((filename_is_mask = (strcmp(filename,mask) == 0)) ||
        (name_map_mangle(filename,True,False,SNUM(conn)) &&
         mask_match(filename,mask,False,False)))
    {
      if (isrootdir && (strequal(filename,"..") || strequal(filename,".")))
        continue;

      pstrcpy(fname,filename);
      *path = 0;
      pstrcpy(path,conn->dirpath);
      if(needslash)
        pstrcat(path,"/");
      pstrcpy(pathreal,path);
      pstrcat(path,fname);
      pstrcat(pathreal,dname);
      if (dos_stat(pathreal,&sbuf) != 0) 
      {
        DEBUG(5,("Couldn't stat 1 [%s]. Error = %s\n",path, strerror(errno) ));
        continue;
      }

      if (check_descend && !strequal(fname,".") && !strequal(fname,".."))
        continue;
	  
      *mode = dos_mode(conn,pathreal,&sbuf);

      if (!dir_check_ftype(conn,*mode,&sbuf,dirtype)) 
      {
        DEBUG(5,("[%s] attribs didn't match %x\n",filename,dirtype));
        continue;
      }

      if (!filename_is_mask)
      {
        /* Now we can allow the mangled cache to be updated */
        pstrcpy(filename,dname);
        name_map_mangle(filename,True,True,SNUM(conn));
      }

      *size = sbuf.st_size;
      *date = sbuf.st_mtime;

      DEBUG(5,("get_dir_entry found %s fname=%s\n",pathreal,fname));
	  
      found = True;
    }
  }

  return(found);
}





#define	BASIC_DIRCHUNK_SIZE	4096
#define	DIRENTRY_INCREMENT	256

typedef struct _chunk {
    struct _chunk	*next;
    char                *curtop;
    int			free;
} chunk;

typedef struct _DirEntry {
    size_t	off_from_top;
    size_t	strlen;
    char	*name;
} DirEntry;

typedef struct _Dir {
  int pos;
  int numentries;
    int		numallocated;
    char	*dirname;
    DirEntry	*deP;

    /* below here is chunk manager for allocating, removing */
    chunk	*chunkP;
} Dir;

/*******************************************************************
 Open a directory.
********************************************************************/
void *OpenDir( connection_struct *conn, char *name, BOOL use_veto )
{
    Dir *dirp;
    char *n;
    DIR *p = dos_opendir(name);
    size_t namelen;

    if ( !p ) {
        return NULL;
    }
    namelen	= strlen( name ) + 1;
    dirp	= (Dir *)malloc( sizeof( Dir ) + namelen );
    if ( !dirp ) {
        closedir( p );
        return NULL;
    }

    dirp->pos	        = 0;
    dirp->numentries	= 0;
    dirp->numallocated	= 0;
    dirp->deP		= NULL;
    dirp->chunkP	= NULL;
    dirp->dirname	= (char *)(dirp+1);
    memmove( dirp->dirname, name, namelen );

    while ( n = dos_readdirname( p )) {
        size_t	len	= strlen( n ) + 1;

    /* If it's a vetoed file, pretend it doesn't even exist */
        if ( use_veto && conn && IS_VETO_PATH( conn, n )) {
            continue;
        }

        if ( !( dirp->numallocated - dirp->numentries )) {
            DirEntry	*newdeP;
            size_t	newsize;

            newsize	= ( dirp->numallocated + DIRENTRY_INCREMENT )
                * sizeof( DirEntry );
            newdeP      = (DirEntry *)Realloc( dirp->deP, newsize );
            if ( !newdeP ) {
                DEBUG(0,("Out of memory in OpenDir\n"));
                break;
            }
            dirp->deP	= newdeP;
            dirp->numallocated	+= DIRENTRY_INCREMENT;
        }

        dirp->deP[dirp->numentries].strlen = len;

        if ( dirp->numentries == 0 ) {
            dirp->deP[dirp->numentries].off_from_top	= 0;
        } else {
            dirp->deP[dirp->numentries].off_from_top
                = dirp->deP[dirp->numentries - 1].off_from_top
                + dirp->deP[dirp->numentries - 1].strlen;
        }

        /* ... Well ... I know it's silly to check for file name */
        /* longer then 4096, Still I beleave I should do this */
        if ( len > BASIC_DIRCHUNK_SIZE - sizeof( chunk ) ) {
            /* in this case, you'll allocate new chunk just for this */
            chunk	*newp;

            newp	= (chunk *)malloc( sizeof( chunk ) + len );
            if ( !newp ) {
                DEBUG(0,("Out of memory in OpenDir\n"));
                break;
            }

            newp->curtop	= (char *)(newp+1);
            newp->free		= 0;
            dirp->deP[dirp->numentries].name	= newp->curtop;

            /* insert new chunk to "one after" chunkP */
            /* so that current chunkP's free area will not be waste. */
            if ( !dirp->chunkP ) {
                newp->next	= NULL;
                dirp->chunkP	= newp;
            } else {
                newp->next      = dirp->chunkP->next;
                dirp->chunkP->next	= newp;
            }

        } else {
            if (( dirp->chunkP == NULL )||( dirp->chunkP->free < len )) {
                chunk	*newp;
                newp	= (chunk *)malloc( BASIC_DIRCHUNK_SIZE );
                if ( !newp ) {
		  DEBUG(0,("Out of memory in OpenDir\n"));
		  break;
		}

                newp->curtop	= (char *)(newp+1);
                newp->free	= BASIC_DIRCHUNK_SIZE - sizeof( chunk );
                newp->next	= dirp->chunkP;
                dirp->chunkP	= newp;
            }

            dirp->deP[dirp->numentries].name	= dirp->chunkP->curtop;
            dirp->chunkP->curtop               += len;
            dirp->chunkP->free		       -= len;
	}

        /* since we know the length of 'n', we only should use memmove() */
        /* it's still equivalent or faster then pstrcpy() */
        memmove( dirp->deP[dirp->numentries].name, n, len );
	dirp->numentries++;
    }

    closedir(p);
    return (void *)(dirp);
}


/*******************************************************************
 Close a directory.
********************************************************************/
void CloseDir(void *p)
{
  Dir *dirp = (Dir *)p;

    if ( !p ) {
        return;
    }

    /* first, release chunkP */
    while ( dirp->chunkP ) {
        chunk	*next	= dirp->chunkP->next;
        free( dirp->chunkP );
        dirp->chunkP	= next;
    }

    /* release deP */
    if ( dirp->deP ) {
        free( dirp->deP );
    }
    free( dirp );
}


/*******************************************************************
 Read from a directory.
********************************************************************/
char *ReadDirName( void *p )
{
    char	*retval;
  Dir *dirp = (Dir *)p;

    if (( !dirp )||( dirp->pos >= dirp->numentries )) {
        return NULL;
    }

    retval	= dirp->deP[dirp->pos].name;
  dirp->pos++;

    return retval;
}

/*******************************************************************
 Seek a dir.
********************************************************************/
BOOL SeekDir( void *p, int pos )
{
  Dir *dirp = (Dir *)p;

    if ( !dirp ) {
        return False;
  }

    if (( pos >= dirp->numentries )||( pos < 0 )) {
        return False;
    }

    dirp->pos	= pos;
    return True;
}


/*******************************************************************
 Tell a dir position.
********************************************************************/
int TellDir( void *p )
{
  Dir *dirp = (Dir *)p;

    if (!dirp) {
        return(-1);
    }
  
  return(dirp->pos);
}

/* ----------------------------------------------------------------------- **
 * This section manages a global directory cache.
 * (It should probably be split into a separate module.  crh)
 * ----------------------------------------------------------------------- **
 */
/*
 * Totally changed by Kenichi Okuyama.
 * Now it uses hash/length of path/name to speedup the lookup.
 *
 * It's nice to use Fermat number for HASH_BASE_VALUE. Especially
 * Fermat number that is smaller then HASH_TABLE_SIZE.
 *
 * Also, HASH_TABLE_SIZE is assumed to be power of 2 ( like, 4,8,16,32,64...).
 * so that we only need to take modular only once at end.
 */

#define	HASH_TABLE_SIZE	64
#define	HASH_BASE_VALUE	17

#define	DIRNAME_ALIGN	4
#define FIX_ALIGN(len)	(((len)+DIRNAME_ALIGN-1)&(size_t)(~(DIRNAME_ALIGN-1)))

typedef struct _dir_cache_entry{
    char			*p;
    char			*pathP;
    char        		*nameP;
    char        		*dnameP;
    unsigned long		path_hash;
    unsigned long       	name_hash;
    unsigned long		dname_hash;
    size_t			path_len;
    size_t			name_len;
    size_t			dname_len;
    int                         snum;
    int				inuse;
} dir_cache_entry;

static int	initialized_dir_cache	= 0;
static dir_cache_entry	entry[HASH_TABLE_SIZE][HASH_TABLE_SIZE];


static unsigned long dirCacheHash( char *str )
{
    unsigned long       hash	= 0UL;

    for (; *str; str++ ) {
        hash	= ( hash * HASH_BASE_VALUE ) + ( *str );
    }
    return hash % HASH_TABLE_SIZE;
}

static void initialize_dirCache( void )
{
    int	i, j;
    for ( i = 0; i < HASH_TABLE_SIZE; i++ ) {
        for ( j = 0; j < HASH_TABLE_SIZE; j++ ) {
            entry[i][j].inuse	= 0;
        }
    }
    initialized_dir_cache	= 1;
}

/*
   * Add an entry to the directory cache.
 * only one directory per same [path_hash][name_hash].
   */
void DirCacheAdd( char *path, char *name, char *dname, int snum )
{
    size_t		path_len, name_len, dname_len;
    unsigned long	path_hash, name_hash, dname_hash;
    dir_cache_entry	*entryP;

    if ( !initialized_dir_cache ) {
        initialize_dirCache();
    }

    if ( !path ) {
        path_len	= 0;
        path_hash	= 0UL;
    } else {
        path_len	= strlen( path ) +1;
        path_hash	= dirCacheHash( path );
    }

    if ( !name ) {
        name_len	= 0;
        name_hash	= 0UL;
    } else {
        name_len	= strlen( name ) + 1;
        name_hash	= dirCacheHash( name );
    }

    if ( !dname ) {
        dname_len	= 0;
        dname_hash	= 0UL;
    } else {
        dname_len	= strlen( dname ) + 1;
        dname_hash	= dirCacheHash( dname );
    }

    entryP	= &(entry[path_hash][name_hash]);
    if ( entryP->inuse ) {
        if ( entryP->p ) {
            free( entryP->p );
            entryP->p	= NULL;
        }
        entryP->inuse	= 0;
    }

    if ( FIX_ALIGN( path_len )+FIX_ALIGN( name_len )+FIX_ALIGN( dname_len ) ) {
        entryP->p       = malloc( FIX_ALIGN( path_len )
                                  + FIX_ALIGN( name_len )
                                  + FIX_ALIGN( dname_len ));
        if ( !( entryP->p )) {
            return;
        }
    }

    if ( path ) {
        entryP->pathP	= (char *)(entryP->p);
        memmove( entryP->pathP,  path,  path_len );
    } else {
        entryP->pathP	= NULL;
    }

    if ( name ) {
        entryP->nameP	= (char *)(entryP->p) + FIX_ALIGN( path_len );
        memmove( entryP->nameP,  name,  name_len );
    } else {
        entryP->nameP	= NULL;
    }

    if ( dname ) {
        entryP->dnameP
            = (char *)(entryP->p)
            + FIX_ALIGN( path_len ) + FIX_ALIGN( name_len );
        memmove( entryP->dnameP, dname, dname_len );
    } else {
        entryP->dnameP	= NULL;
    }

    entryP->path_hash	= path_hash;
    entryP->name_hash	= name_hash;
    entryP->dname_hash	= dname_hash;

    entryP->path_len	= path_len;
    entryP->name_len	= name_len;
    entryP->dname_len	= dname_len;

    entryP->snum	= snum;
    entryP->inuse	= 1;
}

/* ------------------------------------------------------------------------ **
   * Search for an entry to the directory cache.
   *
   *  Input:  path  -
   *          name  -
   *          snum  -
   *
   *  Output: The dname string of the located entry, or NULL if the entry was
   *          not found.
   *
   * ------------------------------------------------------------------------ **
   */
char *DirCacheCheck( char *path, char *name, int snum )
{
    size_t		path_len, name_len;
    unsigned long	path_hash, name_hash;

    if ( !initialized_dir_cache ) {
        initialize_dirCache();
        return NULL;
    }

    if ( path ) {
        path_len	= strlen( path ) + 1;
        path_hash	= dirCacheHash( path );
    } else {
        path_len	= 0;
        path_hash	= 0UL;
      }

    if ( name ) {
        name_len	= strlen( name ) + 1;
        name_hash	= dirCacheHash( name );
    } else {
        name_len	= 0;
        name_hash	= 0UL;
    }

    if (( entry[path_hash][name_hash].inuse )&&
        ( entry[path_hash][name_hash].snum == snum )&&
        ( strcmp( entry[path_hash][name_hash].pathP, path ) == 0 )&&
        ( strcmp( entry[path_hash][name_hash].nameP, name ) == 0 )) {
        return entry[path_hash][name_hash].dnameP;
    }

    return NULL;
}


/* ------------------------------------------------------------------------ **
   * Remove all cache entries which have an snum that matches the input.
   *
   *  Input:  snum  -
   *
   *  Output: None.
   *
   * ------------------------------------------------------------------------ **
   */
void DirCacheFlush(int snum)
{
    int	i, j;

    if ( !initialized_dir_cache ) {
        initialize_dirCache();
        return;
    }

    for ( i = 0; i < HASH_TABLE_SIZE; i++ ) {
        for ( j = 0; j < HASH_TABLE_SIZE; j++ ) {
            if (( entry[i][j].inuse )&&
                ( entry[i][j].snum == snum )) {
	        entry[i][j].inuse       = 0;
                if ( entry[i][j].p ) {
                    free( entry[i][j].p );
                    entry[i][j].p	= NULL;
                }
            }
        }
    }
}
