/*                                                                
**  Copyright (C) 2019  Smithsonian Astrophysical Observatory 
*/                                                                

/*                                                                          */
/*  This program is free software; you can redistribute it and/or modify    */
/*  it under the terms of the GNU General Public License as published by    */
/*  the Free Software Foundation; either version 3 of the License, or       */
/*  (at your option) any later version.                                     */
/*                                                                          */
/*  This program is distributed in the hope that it will be useful,         */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/*  GNU General Public License for more details.                            */
/*                                                                          */
/*  You should have received a copy of the GNU General Public License along */
/*  with this program; if not, write to the Free Software Foundation, Inc., */
/*  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.             */
/*                                                                          */

/* pfile.h
**/

#include <stdio.h>
#include <float.h>
#include <limits.h>

#define PARAMPATH	"$PDIRS $PFILES $UPARM ."
#define PARAMEXTN	".par .rdb $PFEXTN"

#define INDEFS          (SHRT_MIN)
#define INDEFL          (LONG_MIN)
#define INDEFI          (INT_MIN)
#define INDEFR          (FLT_MAX)
#define INDEFD          (DBL_MAX)
#define INDEF           INDEFR

typedef double real;

/* The Value structure    */
typedef struct _Value
{
  int type;
  void *value;
  char *indir;
} Value;

typedef int VType;

#define ConverterType void *(*)(void*,void*)
typedef struct _Converter
{
  int from;
  int to;
  void *(*convert) (void *, void *);
} Converter;


#define PARAM_ALREADY_READ 1
#define PARAM_NOT_READ     0


/* The parameter structure */

struct _ParamFile;

typedef struct _Parameter
{
  int ptype;
  int pflag;
  int pquery;
  int redirect;			/* flag to check cyclic redirects */
  char *pname;
  int pmode;
  Value pvalue;
  Value pmin;
  Value pmax;
  char *pprompt;
  char *pdefault;		/* default value of param */
  void *extension;
  Value *(*phook) (struct _ParamFile *, struct _Parameter *);
} Parameter;


typedef struct _ParamList
{
  FILE *file;
  char *filename;
  char *psetname;
  char *mode;

  /* Methods                      */
  void (*clos) (void);
  struct _ParamList *(*read) (FILE *, char *);
  void (*writ) (FILE *, struct _ParamList *);

  int nparam;
  int nalloc;
  Parameter **param;
} ParamList;

typedef struct _ParamFile
{
  struct _ParamFile *next;
  ParamList *psets;
  ParamList *alist;

  int mode;
  unsigned long refcount;
  void *extension;
} *ParamFile;




typedef struct _ParamFormat
{
  ParamList *(*open) (const char *, const char *, const char *);
  char *extn;
  char *(*find) (const char *, char *, char *, char *);
} ParamFormat;


#define vtype_compar_func  int (*)(void *, void *, int )

typedef struct _vtype
{
  char *name;
  struct _Converter *converters;
  int (*size) (void *);
  Value *(*access) (ParamFile, Parameter *, VType, Value *);
  int (*compar) (void *, void *, int);
} vtype;



/* ParamFile operations 	*/

ParamFile ParamFileLookup (char *fname);
ParamFile ParamPSetLookup (const char *psetn);
ParamList *ParamFileArgs (ParamList * plist, char *argv[], int argc);

void ParamFree (Parameter * param);
void PFree (ParamList * plist);
extern ParamList *ParamFileCant;
char *ParamMatch (ParamList * plist, char *pname);

#include "ipar.h"
#include "iarg.h"

int ParamFileMode (ParamFile pf);

VType String2Type (char *string);
void *Type2String (VType type, char *string);
void *Mode2String (int *value, char *copy);

void *StringCopy (char *value, char *copy);
int SSSize (void *str);

int SSComp (char *value1, char *value2, int type);

ParamList *PListCreate (void);


ParamList *PListAdd (ParamList * plist, Parameter * param);

ParamList *PListDel (ParamList * plist, Parameter * param);


/* Parameter Construction Utilities */

Parameter *ParamStruct (Parameter * param, char *line);

Parameter *ParamLookup (ParamList * plist, const char *pname);

Parameter *ParamAlloc (void);

Parameter *ParamName (Parameter * p, char *name);

Parameter *ParamMode (Parameter * p, int *mode);
Parameter *ParamType (Parameter * p, VType type);
Parameter *ParamValue (Parameter * p, VType type, char *value);
Parameter *ParamDefault (Parameter * p);
Parameter *ParamMin (Parameter * p, VType type, void *min);
Parameter *ParamMax (Parameter * p, VType type, char *max);
Parameter *ParamPrompt (Parameter * p, char *prompt);

Value *VIsIndirect (Value * value);
Value *cxcValidate (ParamFile pfile,
		    Parameter * p, Value * value, int interactive);

char *QuoteValue (char *in, char *out);

void *Convert (void *value, VType from, void *copy, VType to);
void *VConvert (Value * value, void *copy, VType to);

Value *VNewValue (Value * pvalue, void *value, VType type);

char *NewString (char *str);


int IndirectClassing (ParamFile pf, Parameter * p, Value * value,
		      char *class);
int GetIndirectPFiles (char **pfiles, int maxpfiles);
int UsingIndirectClassing (void);
int LookupInClass (char *pfile, char *param, char **ipfile, char **iparam);

int PParseName (const char *fname, char *xname);

/* Parameter method access macros (operations on values) */

#define TypeBits	ListType
#define TypeIndex(x) ( x & ~TypeBits)

#define VAccess(pfile, p, type, value)					\
	( (p && VTypeList[TypeIndex(type)]->access) ?		 	\
	  (*(VTypeList[TypeIndex(type)]->access))(pfile, p, type, value)\
	: (Value *) NULL )

#define VCompar(type, value1, value2, op)				\
	  ( VTypeList[TypeIndex(type)]->compar ?			\
	     (*(VTypeList[TypeIndex(type)]->compar))(value1, value2, op) : 0 )


#define VSize(type, value)						\
	  ( VTypeList[TypeIndex(type)]->size ?				\
	     (*(VTypeList[TypeIndex(type)]->size))(value) : 0 )


#define PHooker(p)		( (p)->phook 				 \
					? (*((p)->phook))(p, value, mode)\
					: NULL )


/* Middle level interface */

int ParamSetX (ParamFile pfile, const char *name, VType type, void *value);
void *ParamGetX (ParamFile pfile, const char *name, VType type, int *xmode);


void VDirect (ParamFile pfile, Parameter * param, Value * value);

int VXSize (void *value);
Value *VXAcce (ParamFile pfile, Parameter * p, VType type, Value * value);


/* Parameter Types */

extern vtype *VTypeList[];

#define ListType	0x1000

#define BooleanType	 1
#define IntegerType	 2
#define RealType	 3
#define StringType	 4
#define FileType	 5
#define GType	 	 6
#define StructType	 7
#define GCurType	 8
#define ImCurType	 9
#define ModeType	10
#define PSetType	11
#define RangeType	12
#define CommentType	13

#define MaxTypes	13

extern vtype booleanRec;
extern vtype integerRec;
extern vtype realRec;
extern vtype stringRec;
extern vtype fileRec;
extern vtype gRec;
extern vtype structRec;
extern vtype modeRec;
extern vtype gcurRec;
extern vtype imcurRec;
extern vtype psetRec;
extern vtype isetRec;
extern vtype rangeRec;
extern vtype commentRec;


/* Size of things */

#define PSTAK_SZ	10
#define PFILE_SZ	30
#define FIELD_SZ	132

/* Parameter Fields */

#define PMODE	1
/*
#define PTYPE	7
*/
#define PVALUE	2
#define PMIN	3
#define PMAX	4
#define PPROMPT	5
#define PWORTH	6


/* Modes */

#define AMODE		1
#define HMODE		2
#define LMODE		4
#define QMODE		8
#define HHMODE		16
#define APMODE          32
#define LLMODE          64

#ifndef SZ_PFLINE
#define SZ_PFLINE	2048
#endif

/* ERRORS */
extern int parerr;
extern char *parammessage[];

#define PARNOERROR	0
#define	PARNOTFOUND	1
#define	PARNOTUNIQ	2
#define	PARNOFILE	3
#define PARARGS		4
#define PARPOSITIONAL	5
#define PARBADTYPE	6
#define PARBADMIN	7
#define PARBADMAX	8
#define PARLIMITMIN	9
#define PARLIMITMAX	10
#define PARNOPOINTER	11
#define PARCANTCONVERT	12
#define PARNOVALUE	13
#define PARBADARGV	14
#define PARBADPFILE	15
#define PARNOCONVERTER	16
#define PARBADMODE	17
#define PARSUBFIELD	18
#define PARRANGEMAX	19
#define PARBADRANGE	20
#define PARCANTRANGE	21
#define PARLIMITRANGE	22
#define PARTOOMANYPOS	23
#define PARCANTPARSE	24
#define PARENUMUNIQ	25
#define PARBADINDIRECT	26
#define PARNOLISTFILE	27
#define PARBADENUM	28
#define PARBACK2FRONT	29
#define PARNOREF	30
#define PARNOPSET	31
#define PARNOREFR	32
#define PARCYCLICERR    33
#define PARSELFDIRECT   34
#define PARMAXGTMIN     35

#include "listhook.h"

typedef struct _listrecord
{
  FILE *file;
  int n;
} listrecord;


#define IsInList(x)	((x)->phook != NULL )
