package Acme::CPANModules::OrderedHash;

use strict;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2023-10-05'; # DATE
our $DIST = 'Acme-CPANModules-OrderedHash'; # DIST
our $VERSION = '0.002'; # VERSION

our $LIST = {
    summary => "List of modules that provide ordered hash data type",
    description => <<'_',

When you ask a Perl's hash for the list of keys, the answer comes back
unordered. In fact, Perl explicitly randomizes the order of keys it returns
everytime. The random ordering is a (security) feature, not a bug. However,
sometimes you want to know the order of insertion. These modules provide you
with an ordered hash; most of them implement it by recording the order of
insertion of keys in an additional array.

Other related modules:

<pm:Tie::SortHash> - will automatically sort keys when you call `keys()`,
`values()`, `each()`. But this module does not maintain insertion order.

_
    entries => [

        {
            module => 'Tie::IxHash',
            bench_code => sub {
                my ($op, $numkeys, $numrep) = @_;

                tie my %hash, "Tie::IxHash";
                for (1..$numkeys) { $hash{"key$_"} = $_ }

                if ($op eq 'delete') {
                    for (1..$numkeys) { delete $hash{"key$_"} }
                } elsif ($op eq 'keys') {
                    for (1..$numrep) { my @keys = keys %hash }
                }
            },
        },

        {
            module => 'Hash::Ordered',
            bench_code => sub {
                my ($op, $numkeys, $numrep) = @_;

                my $hash = Hash::Ordered->new;
                for (1..$numkeys) { $hash->set("key$_" => $_) }

                if ($op eq 'delete') {
                    for (1..$numkeys) { $hash->delete("key$_") }
                } elsif ($op eq 'keys') {
                    for (1..$numrep) { my @keys = $hash->keys }
                }
            },
        },

        {
            module => 'Tie::Hash::Indexed',
            description => <<'MARKDOWN',

Provides two interfaces: tied hash and OO.

MARKDOWN
            bench_code => sub {
                my ($op, $numkeys, $numrep) = @_;

                tie my %hash, "Tie::Hash::Indexed";
                for (1..$numkeys) { $hash{"key$_"} = $_ }

                if ($op eq 'delete') {
                    for (1..$numkeys) { delete $hash{"key$_"} }
                } elsif ($op eq 'keys') {
                    for (1..$numrep) { my @keys = keys %hash }
                }
            },
        },

        {
            module => 'Tie::LLHash',
            bench_code => sub {
                my ($op, $numkeys, $numrep) = @_;

                tie my %hash, "Tie::LLHash";
                for (1..$numkeys) { (tied %hash)->insert("key$_" => $_) }

                if ($op eq 'delete') {
                    for (1..$numkeys) { delete $hash{"key$_"} }
                } elsif ($op eq 'keys') {
                    for (1..$numrep) { my @keys = keys %hash }
                }
            },
        },

        {
            module => 'Tie::StoredOrderHash',
            bench_code => sub {
                my ($op, $numkeys, $numrep) = @_;

                tie my %hash, "Tie::StoredOrderHash";
                for (1..$numkeys) { $hash{"key$_"} = $_ }

                if ($op eq 'delete') {
                    for (1..$numkeys) { delete $hash{"key$_"} }
                } elsif ($op eq 'keys') {
                    for (1..$numrep) { my @keys = keys %hash }
                }
            },
        },

        {
            module => 'Array::OrdHash',
            description => <<'_',

Provide something closest to PHP's associative array, where you can refer
elements by key or by numeric index, and insertion order is remembered.

_
            bench_code => sub {
                my ($op, $numkeys, $numrep) = @_;

                my $hash = Array::OrdHash->new;
                for (1..$numkeys) { $hash->{"key$_"} = $_ }

                if ($op eq 'delete') {
                    for (1..$numkeys) { delete $hash->{"key$_"} }
                } elsif ($op eq 'keys') {
                    for (1..$numrep) { my @keys = keys %$hash }
                }
            },
        },

        {
            module => 'List::Unique::DeterministicOrder',
            description => <<'MARKDOWN',

Provide a list, not hash.

MARKDOWN
            bench_code => sub {
                my ($op, $numkeys, $numrep) = @_;

                my $hash = List::Unique::DeterministicOrder->new(data=>[]);
                for (1..$numkeys) { $hash->push("key$_") }

                if ($op eq 'delete') {
                    for (1..$numkeys) { $hash->delete("key$_") }
                } elsif ($op eq 'keys') {
                    for (1..$numrep) { my @keys = $hash->keys }
                }
            },
        },
    ],

    bench_datasets => [
        {name=>'insert 1000 pairs', argv => ['insert', 1000]},
        {name=>'insert 1000 pairs + delete', argv => ['delete', 1000]},
        {name=>'insert 1000 pairs + return keys 100 times', argv => ['keys', 1000, 100]},
        # TODO: iterating
    ],
};

1;
# ABSTRACT: List of modules that provide ordered hash data type

__END__

=pod

=encoding UTF-8

=head1 NAME

Acme::CPANModules::OrderedHash - List of modules that provide ordered hash data type

=head1 VERSION

This document describes version 0.002 of Acme::CPANModules::OrderedHash (from Perl distribution Acme-CPANModules-OrderedHash), released on 2023-10-05.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher --cpanmodules-module OrderedHash

To run module startup overhead benchmark:

 % bencher --module-startup --cpanmodules-module OrderedHash

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

When you ask a Perl's hash for the list of keys, the answer comes back
unordered. In fact, Perl explicitly randomizes the order of keys it returns
everytime. The random ordering is a (security) feature, not a bug. However,
sometimes you want to know the order of insertion. These modules provide you
with an ordered hash; most of them implement it by recording the order of
insertion of keys in an additional array.

Other related modules:

L<Tie::SortHash> - will automatically sort keys when you call C<keys()>,
C<values()>, C<each()>. But this module does not maintain insertion order.

=head1 ACME::CPANMODULES ENTRIES

=over

=item L<Tie::IxHash>

Author: L<CHORNY|https://metacpan.org/author/CHORNY>

=item L<Hash::Ordered>

Author: L<DAGOLDEN|https://metacpan.org/author/DAGOLDEN>

=item L<Tie::Hash::Indexed>

Author: L<MHX|https://metacpan.org/author/MHX>

Provides two interfaces: tied hash and OO.


=item L<Tie::LLHash>

Author: L<XAERXESS|https://metacpan.org/author/XAERXESS>

=item L<Tie::StoredOrderHash>

Author: L<TFM|https://metacpan.org/author/TFM>

=item L<Array::OrdHash>

Author: L<WOWASURIN|https://metacpan.org/author/WOWASURIN>

Provide something closest to PHP's associative array, where you can refer
elements by key or by numeric index, and insertion order is remembered.


=item L<List::Unique::DeterministicOrder>

Author: L<SLAFFAN|https://metacpan.org/author/SLAFFAN>

Provide a list, not hash.


=back

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Tie::IxHash> 1.23

L<Hash::Ordered> 0.014

L<Tie::Hash::Indexed> 0.08

L<Tie::LLHash> 1.004

L<Tie::StoredOrderHash> 0.22

L<Array::OrdHash> 1.03

L<List::Unique::DeterministicOrder> 0.004

=head1 BENCHMARK PARTICIPANTS

=over

=item * Tie::IxHash (perl_code)

L<Tie::IxHash>



=item * Hash::Ordered (perl_code)

L<Hash::Ordered>



=item * Tie::Hash::Indexed (perl_code)

L<Tie::Hash::Indexed>



=item * Tie::LLHash (perl_code)

L<Tie::LLHash>



=item * Tie::StoredOrderHash (perl_code)

L<Tie::StoredOrderHash>



=item * Array::OrdHash (perl_code)

L<Array::OrdHash>



=item * List::Unique::DeterministicOrder (perl_code)

L<List::Unique::DeterministicOrder>



=back

=head1 BENCHMARK DATASETS

=over

=item * insert 1000 pairs

=item * insert 1000 pairs + delete

=item * insert 1000 pairs + return keys 100 times

=back

=head1 BENCHMARK SAMPLE RESULTS

=head2 Sample benchmark #1

Run on: perl: I<< v5.38.0 >>, CPU: I<< Intel(R) Core(TM) i5-7200U CPU @ 2.50GHz (2 cores) >>, OS: I<< GNU/Linux Ubuntu version 20.04 >>, OS kernel: I<< Linux version 5.4.0-91-generic >>.

Benchmark command (default options):

 % bencher --cpanmodules-module OrderedHash

Result formatted as table (split, part 1 of 3):

 #table1#
 {dataset=>"insert 1000 pairs"}
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                      | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Tie::StoredOrderHash             |       360 |      2.78 |                 0.00% |               199.08% | 2.6e-06 |      20 |
 | Tie::LLHash                      |       370 |      2.7  |                 4.20% |               187.04% | 1.5e-05 |      20 |
 | Array::OrdHash                   |       540 |      1.8  |                51.22% |                97.78% | 2.5e-06 |      20 |
 | Tie::IxHash                      |       670 |      1.5  |                87.47% |                59.53% | 1.7e-06 |      20 |
 | Hash::Ordered                    |       889 |      1.13 |               147.19% |                20.99% | 8.6e-07 |      20 |
 | Tie::Hash::Indexed               |       960 |      1    |               168.05% |                11.58% | 2.2e-06 |      21 |
 | List::Unique::DeterministicOrder |      1100 |      0.93 |               199.08% |                 0.00% | 1.1e-06 |      20 |
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

          Rate   T:S   T:L   A:O   T:I   H:O  TH:I  LU:D 
  T:S    360/s    --   -2%  -35%  -46%  -59%  -64%  -66% 
  T:L    370/s    2%    --  -33%  -44%  -58%  -62%  -65% 
  A:O    540/s   54%   50%    --  -16%  -37%  -44%  -48% 
  T:I    670/s   85%   80%   19%    --  -24%  -33%  -38% 
  H:O    889/s  146%  138%   59%   32%    --  -11%  -17% 
  TH:I   960/s  177%  170%   80%   50%   12%    --   -6% 
  LU:D  1100/s  198%  190%   93%   61%   21%    7%    -- 
 
 Legends:
   A:O: participant=Array::OrdHash
   H:O: participant=Hash::Ordered
   LU:D: participant=List::Unique::DeterministicOrder
   T:I: participant=Tie::IxHash
   T:L: participant=Tie::LLHash
   T:S: participant=Tie::StoredOrderHash
   TH:I: participant=Tie::Hash::Indexed

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 2 of 3):

 #table2#
 {dataset=>"insert 1000 pairs + delete"}
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                      | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Tie::IxHash                      |      16.5 |      60.5 |                 0.00% |              3719.19% |   4e-05 |      20 |
 | Tie::StoredOrderHash             |     200   |       5.1 |              1081.74% |               223.18% |   1e-05 |      20 |
 | Tie::LLHash                      |     220   |       4.6 |              1219.38% |               189.47% | 9.5e-06 |      20 |
 | Array::OrdHash                   |     280   |       3.6 |              1577.30% |               127.70% | 6.4e-06 |      21 |
 | Hash::Ordered                    |     370   |       2.7 |              2110.87% |                72.75% | 8.4e-06 |      20 |
 | List::Unique::DeterministicOrder |     610   |       1.6 |              3607.28% |                 3.02% | 2.2e-06 |      20 |
 | Tie::Hash::Indexed               |     630   |       1.6 |              3719.19% |                 0.00% | 9.3e-06 |      20 |
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

          Rate    T:I   T:S   T:L   A:O   H:O  LU:D  TH:I 
  T:I   16.5/s     --  -91%  -92%  -94%  -95%  -97%  -97% 
  T:S    200/s  1086%    --   -9%  -29%  -47%  -68%  -68% 
  T:L    220/s  1215%   10%    --  -21%  -41%  -65%  -65% 
  A:O    280/s  1580%   41%   27%    --  -25%  -55%  -55% 
  H:O    370/s  2140%   88%   70%   33%    --  -40%  -40% 
  LU:D   610/s  3681%  218%  187%  125%   68%    --    0% 
  TH:I   630/s  3681%  218%  187%  125%   68%    0%    -- 
 
 Legends:
   A:O: participant=Array::OrdHash
   H:O: participant=Hash::Ordered
   LU:D: participant=List::Unique::DeterministicOrder
   T:I: participant=Tie::IxHash
   T:L: participant=Tie::LLHash
   T:S: participant=Tie::StoredOrderHash
   TH:I: participant=Tie::Hash::Indexed

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


Result formatted as table (split, part 3 of 3):

 #table3#
 {dataset=>"insert 1000 pairs + return keys 100 times"}
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                      | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | Tie::StoredOrderHash             |        10 |      96   |                 0.00% |               885.35% |   0.00017 |      21 |
 | Tie::LLHash                      |        12 |      86   |                11.19% |               786.17% | 9.1e-05   |      20 |
 | Array::OrdHash                   |        13 |      75   |                28.17% |               668.78% |   0.00055 |      20 |
 | Tie::IxHash                      |        16 |      63   |                51.87% |               548.80% | 7.8e-05   |      20 |
 | Tie::Hash::Indexed               |        24 |      41   |               131.40% |               325.82% |   0.00013 |      20 |
 | Hash::Ordered                    |        69 |      15   |               559.03% |                49.52% | 6.1e-05   |      20 |
 | List::Unique::DeterministicOrder |       100 |       9.7 |               885.35% |                 0.00% | 1.1e-05   |      20 |
 +----------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

The above result formatted in L<Benchmark.pm|Benchmark> style:

         Rate   T:S   T:L   A:O   T:I  TH:I   H:O  LU:D 
  T:S    10/s    --  -10%  -21%  -34%  -57%  -84%  -89% 
  T:L    12/s   11%    --  -12%  -26%  -52%  -82%  -88% 
  A:O    13/s   28%   14%    --  -16%  -45%  -80%  -87% 
  T:I    16/s   52%   36%   19%    --  -34%  -76%  -84% 
  TH:I   24/s  134%  109%   82%   53%    --  -63%  -76% 
  H:O    69/s  540%  473%  400%  320%  173%    --  -35% 
  LU:D  100/s  889%  786%  673%  549%  322%   54%    -- 
 
 Legends:
   A:O: participant=Array::OrdHash
   H:O: participant=Hash::Ordered
   LU:D: participant=List::Unique::DeterministicOrder
   T:I: participant=Tie::IxHash
   T:L: participant=Tie::LLHash
   T:S: participant=Tie::StoredOrderHash
   TH:I: participant=Tie::Hash::Indexed

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


=head2 Sample benchmark #2

Benchmark command (benchmarking module startup overhead):

 % bencher --cpanmodules-module OrderedHash --module-startup

Result formatted as table:

 #table4#
 +----------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant                      | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +----------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | List::Unique::DeterministicOrder |      18   |              10   |                 0.00% |               108.56% | 4.5e-05   |      20 |
 | Hash::Ordered                    |      16   |               8   |                10.05% |                89.52% | 4.4e-05   |      20 |
 | Tie::Hash::Indexed               |      15   |               7   |                14.63% |                81.94% | 2.6e-05   |      21 |
 | Array::OrdHash                   |      15   |               7   |                18.44% |                76.10% | 1.7e-05   |      21 |
 | Tie::IxHash                      |      14.8 |               6.8 |                19.12% |                75.09% | 1.5e-05   |      20 |
 | Tie::LLHash                      |      14   |               6   |                29.30% |                61.30% | 2.5e-05   |      20 |
 | Tie::StoredOrderHash             |      11   |               3   |                67.04% |                24.86% | 2.9e-05   |      20 |
 | perl -e1 (baseline)              |       8   |               0   |               108.56% |                 0.00% |   0.00015 |      20 |
 +----------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


The above result formatted in L<Benchmark.pm|Benchmark> style:

                          Rate  LU:D   H:O  TH:I   A:O   T:I   T:L   T:S  perl -e1 (baseline) 
  LU:D                  55.6/s    --  -11%  -16%  -16%  -17%  -22%  -38%                 -55% 
  H:O                   62.5/s   12%    --   -6%   -6%   -7%  -12%  -31%                 -50% 
  TH:I                  66.7/s   19%    6%    --    0%   -1%   -6%  -26%                 -46% 
  A:O                   66.7/s   19%    6%    0%    --   -1%   -6%  -26%                 -46% 
  T:I                   67.6/s   21%    8%    1%    1%    --   -5%  -25%                 -45% 
  T:L                   71.4/s   28%   14%    7%    7%    5%    --  -21%                 -42% 
  T:S                   90.9/s   63%   45%   36%   36%   34%   27%    --                 -27% 
  perl -e1 (baseline)  125.0/s  125%  100%   87%   87%   85%   75%   37%                   -- 
 
 Legends:
   A:O: mod_overhead_time=7 participant=Array::OrdHash
   H:O: mod_overhead_time=8 participant=Hash::Ordered
   LU:D: mod_overhead_time=10 participant=List::Unique::DeterministicOrder
   T:I: mod_overhead_time=6.8 participant=Tie::IxHash
   T:L: mod_overhead_time=6 participant=Tie::LLHash
   T:S: mod_overhead_time=3 participant=Tie::StoredOrderHash
   TH:I: mod_overhead_time=7 participant=Tie::Hash::Indexed
   perl -e1 (baseline): mod_overhead_time=0 participant=perl -e1 (baseline)

The above result presented as chart:

=begin html

<img src="data:image/png;base64,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" />

=end html


To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 FAQ

=head2 What is an Acme::CPANModules::* module?

An Acme::CPANModules::* module, like this module, contains just a list of module
names that share a common characteristics. It is a way to categorize modules and
document CPAN. See L<Acme::CPANModules> for more details.

=head2 What are ways to use this Acme::CPANModules module?

Aside from reading this Acme::CPANModules module's POD documentation, you can
install all the listed modules (entries) using L<cpanm-cpanmodules> script (from
L<App::cpanm::cpanmodules> distribution):

 % cpanm-cpanmodules -n OrderedHash

Alternatively you can use the L<cpanmodules> CLI (from L<App::cpanmodules>
distribution):

    % cpanmodules ls-entries OrderedHash | cpanm -n

or L<Acme::CM::Get>:

    % perl -MAcme::CM::Get=OrderedHash -E'say $_->{module} for @{ $LIST->{entries} }' | cpanm -n

or directly:

    % perl -MAcme::CPANModules::OrderedHash -E'say $_->{module} for @{ $Acme::CPANModules::OrderedHash::LIST->{entries} }' | cpanm -n

This Acme::CPANModules module contains benchmark instructions. You can run a
benchmark for some/all the modules listed in this Acme::CPANModules module using
the L<bencher> CLI (from L<Bencher> distribution):

    % bencher --cpanmodules-module OrderedHash

This Acme::CPANModules module also helps L<lcpan> produce a more meaningful
result for C<lcpan related-mods> command when it comes to finding related
modules for the modules listed in this Acme::CPANModules module.
See L<App::lcpan::Cmd::related_mods> for more details on how "related modules"
are found.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Acme-CPANModules-OrderedHash>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Acme-CPANModules-OrderedHash>.

=head1 SEE ALSO

L<Acme::CPANModules::HashUtilities>

L<Acme::CPANModules> - about the Acme::CPANModules namespace

L<cpanmodules> - CLI tool to let you browse/view the lists

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>,
L<Pod::Weaver::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla- and/or Pod::Weaver plugins. Any additional steps required beyond
that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2023 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Acme-CPANModules-OrderedHash>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
