;;; mew-mark.el --- Marking for Mew Summary and Virtual mode

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Mar  2, 1997
;; Revised: Jul  8, 2001

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; undo-func
;;;

(defun mew-mark-unrefile (fld msg)
  "Delete refile state and delete the mark."
  (mew-refile-reset fld msg))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; exec-func
;;;

(defun mew-mark-exec-refile (src msgs)
  "Refile MSGs from the SRC folder."
  (let (dsts tmp msg msg-dsts dsts-msgses)
    (while msgs
      (setq msg (car msgs))
      (setq msgs (cdr msgs))
      (setq msg-dsts (assoc msg (mew-sinfo-get-refile)))
      (setq dsts (cdr msg-dsts))
      (if dsts ;; sanity check
	  (if (setq tmp (assoc dsts dsts-msgses))
	      (nconc tmp (list msg))	      
	    (setq dsts-msgses (cons (list dsts msg) dsts-msgses))))
      (mew-sinfo-set-refile (delq msg-dsts (mew-sinfo-get-refile))))
    ;; refile at once
    (while dsts-msgses
      (mew-summary-mv-msgs src (car dsts-msgses))
      (setq dsts-msgses (cdr dsts-msgses)))))

(defun mew-mark-exec-unlink (src dels)
  "Unlink DELS from the SRC folder.
DELS represents the messages to be deleted."
  (let ((mew-msg-rm-policy 'always))
    (mew-mark-exec-delete src dels)))

(defun mew-mark-exec-delete (src dels)
  "Delete messages from the SRC folder according to mew-msg-rm-policy.
DELS represents the messages to be deleted."
  (when dels
    (let ((rm-it nil))
      (cond
       ((eq mew-msg-rm-policy 'always)
	(setq rm-it t))
       ((eq mew-msg-rm-policy 'trashonly)
	(if (string= src mew-trash-folder)
	    (setq rm-it t)))
       ((eq mew-msg-rm-policy 'uselist)
	(if (mew-member-match src mew-msg-rm-folder-list)
	    (setq rm-it t))))
      ;; (t ;; 'totrash)
      (if rm-it
	  (mew-summary-clean-msgs src dels)
	(if (string= src mew-trash-folder)
	    (progn
	      (goto-char (point-min))
	      (while (re-search-forward mew-regex-msg-delete nil t)
		(mew-summary-undo-one)))
	  (mew-summary-rm-msgs src dels))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; sanity-func
;;;

(defun mew-mark-sanity-refile (msgs)
  "Check the destination folder for MSGS."
  (let (msg dst dsts uniq-dsts udst err-dsts dir)
    (while msgs
      (setq msg (car msgs))
      (setq msgs (cdr msgs))
      (setq dsts (cdr (assoc msg (mew-sinfo-get-refile))))
      (while dsts
	(setq dst (car dsts))
	(setq dsts (cdr dsts))
	(unless (member dst uniq-dsts)
	  (setq uniq-dsts (cons dst uniq-dsts)))))
    (unless (member mew-trash-folder uniq-dsts)
      (setq uniq-dsts (cons mew-trash-folder uniq-dsts)))
    (while uniq-dsts
      (setq udst (car uniq-dsts))
      (setq uniq-dsts (cdr uniq-dsts))
      (setq dir (mew-expand-folder udst))
      (if (file-exists-p dir)
	  (if (file-directory-p dir)
	      (unless (file-writable-p dir)
		(set-file-modes dir mew-folder-mode))
	    (setq err-dsts (cons udst err-dsts))) ;; NG
	(mew-make-directory dir)))
    err-dsts))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Basic functions and macros for mark
;;;

(defun mew-summary-get-mark ()
  "Get a mark on the current message."
  (save-excursion
    (beginning-of-line)
    (if (looking-at mew-regex-msg-mark)
	(string-to-char (mew-match 2))
      nil)))

(defalias 'mew-summary-marked-p 'mew-summary-get-mark)

(defsubst mew-summary-mark-as (mark &optional force)
  "Mark this message with MARK if possible."
   (when (or force (not (mew-summary-marked-p)))
     (save-excursion
       (beginning-of-line)
       (when (re-search-forward mew-regex-msg nil t)
	 (mew-mark-put-here mark)))))

(defsubst mew-mark-put-here (mark &optional after-mark)
  (mew-elet
   (if after-mark (forward-char -1))
   (delete-char 1)
   (insert-and-inherit (char-to-string mark)) ;; inherit highlight
   (mew-highlight-mark-line mark)))

(defsubst mew-mark-delete-here (&optional after-mark)
  (mew-elet
   (if after-mark (forward-char -1))
   (delete-char 1)
   (insert-and-inherit " ") ;; inherit highlight
   (mew-highlight-unmark-line)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Entire buffer
;;;

(defun mew-summary-mark-exist-p (mark-list)
  "See if this Summary mode has one or more marked messages."
  (let ((regex (mew-mark-list-regex mark-list)))
    (save-excursion
      (goto-char (point-min))
      (re-search-forward regex nil t))))

(defun mew-summary-mark-collect (mark &optional begin end)
  "This function returns a list of message number."
  (save-excursion
    (let ((regex (mew-mark-regex mark))
	  (msglist nil))
      (goto-char (if begin begin (point-min)))
      (while (re-search-forward regex end t)
	(setq msglist (cons (mew-summary-message-number) msglist)))
      (nreverse msglist))))

(defun mew-summary-mark-collect2 (mark)
  "For virtual mode, this function returns a list of
cons pairs of folder name and message number."
  (save-excursion
    (let ((regex (mew-mark-regex mark))
          (msglist nil))
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
        (setq msglist (cons 
                       (cons
                        (mew-summary-folder-name)
                        (mew-summary-message-number))
                       msglist)))
      (nreverse msglist))))

(defun mew-summary-mark-collect3 (mark)
  "This function returns a list of line number."
  (save-excursion
    (let ((regex (mew-mark-regex mark))
	  (i 1)
	  ret)
      (goto-char (point-min))
      (while (not (eobp))
	(if (looking-at regex)
	    (setq ret (cons i ret)))
	(forward-line)
	(setq i (1+ i)))
      (nreverse ret))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Base function
;;;

(defun mew-mark-afterstep (mark case)
  "Move the cursor after marking according to MARK's CASE.
See also mew-mark-afterstep-spec."
  (let ((action (mew-markas-nth mark case)))
    (cond
     ((eq action 0)
      ()) ;; stay
     ((eq action 1)
      (mew-summary-goto-message)
      (mew-decode-syntax-delete)
      (cond
       ((eq mew-summary-mark-direction 'up)
	(forward-line -1))
       ((eq mew-summary-mark-direction 'down)
	(forward-line))
       ((eq mew-summary-mark-direction 'next)
	(if (eq (mew-sinfo-get-direction) 'up)
	    (forward-line -1)
	  (forward-line)))))
     ((eq action 2)
      (mew-summary-goto-message)
      (mew-decode-syntax-delete)
      ;; for C-x C-x
      (beginning-of-line)
      (let ((zmacs-regions nil))
	(push-mark (point) t t))
      (mew-summary-display-after mew-summary-mark-direction)))))

(defun mew-mark-put-mark (newmark &optional no-msg)
  "Put the NEWMARK on the current line if possible.
If NO-MSG is non-nil, no message is displayed.
NO-MSG also means that this function is being called in loop."
  (mew-summary-msg-or-part
   (let (oldmark oldlevel oldname newlevel newname case zmacs-regions
	 msg fld marked)
     (save-excursion
       (mew-summary-goto-message)
       (when (mew-virtual-p)
	 (setq msg (mew-summary-message-number))
	 (setq fld (mew-summary-folder-name)))
       (setq oldmark (mew-summary-get-mark))
       (setq oldlevel (mew-markdb-level oldmark))
       (setq oldname (mew-markdb-name oldmark))
       (setq newlevel (mew-markdb-level newmark))
       (setq newname (mew-markdb-name newmark))
       (cond
	((null oldmark);; no mark
	 (setq case 1)
	 (mew-summary-mark-as newmark)
	 (setq marked t))
	((eq oldmark newmark)
	 (setq case 2)
	 (or no-msg
	     (mew-markdb-statefullp oldmark)
	     (message "Already marked as '%s'" oldname)))
	((< oldlevel newlevel)
	 (setq case 3)
	 (mew-summary-undo-one no-msg)
	 (mew-summary-mark-as newmark)
	 (setq marked t))
	((= oldlevel newlevel)
	 (cond
	  ((mew-markdb-statefullp oldmark)
	   (if (or no-msg
		   (y-or-n-p (format "Already marked as '%s'. %s it? "
				     oldname (mew-capitalize newname))))
	       (progn
		 (setq case 4)
		 (mew-summary-undo-one no-msg)
		 (mew-summary-mark-as newmark)
		 (setq marked t))
	     (setq case 5)))
	  (t
	   (setq case 6)
	   (mew-summary-undo-one no-msg)
	   (mew-summary-mark-as newmark)
	   (setq marked t))))
	(t;; > oldlevel newlevel
	 (setq case 7)
	 (message "Can't mark here because '%s' is stronger than '%s'"
		  oldname newname))))
     (if (and msg fld marked)
	 (mew-summary-mark-in-physical fld msg newmark))
     (or no-msg (mew-mark-afterstep newmark case))
     (set-buffer-modified-p nil))))

(defun mew-mark-put-mark-loop (func count stayp)
  "Unless COUNT is numeric, just call FUNC once. 
The cursor moves forward. STAYP has no effect.
If COUNT is positive, call FUNC in COUNT times moving the cursor forward.
If COUNT is negative, call FUNC in COUNT times moving the cursor backward.
If COUNT is numeric and STAYP is non-nil, the cursor stays in the
original position."
  (if (and func (fboundp func))
      (mew-summary-msg-or-part
       (if (numberp count)
	   (let ((start (point)))
	     (mew-decode-syntax-delete)
	     ;; positive loop
	     (while (and (> count 0) (not (eobp)))
	       (setq count (1- count))
	       (funcall func 'no-msg)
	       (forward-line))
	     ;; negative loop
	     (while (< count 0)
	       (if (bobp)
		   ;; need to call the func
		   (setq count 0)
		 (setq count (1+ count)))
	       (funcall func 'no-msg)
	       (forward-line -1))
	     (and stayp (goto-char start)))
	 ;; just one
	 (funcall func))
       (set-buffer-modified-p nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Physical
;;;

(defun mew-summary-mark-in-physical (fld msg new-mark)
  (save-excursion
    (set-buffer fld)
    (save-excursion
      (goto-char (point-min))
      (when (re-search-forward (mew-regex-jmp-msg msg) nil t)
	(mew-mark-put-here new-mark t)
	(set-buffer-modified-p nil)))))

(defun mew-summary-unmark-in-physical (fld msg &optional func)
  (save-excursion
    (set-buffer fld)
    (save-excursion
      (goto-char (point-min))
      (when (re-search-forward (mew-regex-jmp-msg msg) nil t)
	(mew-mark-delete-here t)
	(if func (funcall func fld msg))
	(set-buffer-modified-p nil)))))

(defmacro mew-mark-alist-set (alist fld msg)
  `(let ((imsg (string-to-int ,msg))
	 (fld-msgs (assoc ,fld ,alist)))
     (if fld-msgs
	 (nconc fld-msgs (list imsg))
       (setq ,alist (cons (list ,fld imsg) ,alist)))))

(defun mew-summary-mark-in-physical-alist (alist mark)
  (let (ent fld msg msgs)
    (while alist
      (setq ent (car alist))
      (setq alist (cdr alist))
      (setq fld (car ent))
      (setq msgs (sort (cdr ent) (function <)))
      (when (get-buffer fld)
	(set-buffer fld)
	(save-excursion
	  (goto-char (point-min))
	  (while msgs
	    (setq msg (int-to-string (car msgs)))
	    (setq msgs (cdr msgs))
	    (when (re-search-forward (mew-regex-jmp-msg msg) nil t)
	      (mew-mark-put-here mark t)
	      (forward-line)))
	  (set-buffer-modified-p nil))))))

(defun mew-summary-unmark-in-physical-alist (alist func)
  (let (ent fld msg msgs)
    (while alist
      (setq ent (car alist))
      (setq alist (cdr alist))
      (setq fld (car ent))
      (setq msgs (sort (cdr ent) (function <)))
      (when (get-buffer fld)
	(set-buffer fld)
	(save-excursion
	  (goto-char (point-min))
	  (while msgs
	    (setq msg (int-to-string (car msgs)))
	    (setq msgs (cdr msgs))
	    (when (re-search-forward (mew-regex-jmp-msg msg) nil t)
	      (mew-mark-delete-here t)
	      (if func (funcall func fld msg))
	      (forward-line)))
	  (set-buffer-modified-p nil))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Review: "*" in Summary mode
;;;

(defun mew-summary-review (&optional count)
  "\\<mew-summary-mode-map>
Put the review mark (default is '*') in COUNT times.
Use '\\[mew-summary-display-review-down]' or '\\[mew-summary-display-review-up]' to jump to a message marked with '*'.
See also '\\[mew-summary-mark-refile]', '\\[mew-summary-mark-delete]', '\\[mew-summary-mark-regexp]', and '\\[mew-summary-mark-all]'."
  (interactive "P")
  (mew-mark-put-mark-loop (function mew-summary-review-one) count nil))

(defun mew-summary-review-one (&optional no-msg)
  "Put the review mark (default is '*') on this message."
  (mew-mark-put-mark mew-mark-review no-msg))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Multi: "@" in Summary mode
;;;

(defun mew-summary-multi (&optional count)
  "\\<mew-summary-mode-map>
Put the multi mark (default is '@') in COUNT times
for '\\[mew-summary-multi-forward]', '\\[mew-summary-unshar]', '\\[mew-summary-uudecode]', '\\[mew-summary-burst-multi]'. "
  (interactive "P")
  (mew-mark-put-mark-loop (function mew-summary-multi-one) count nil))

(defun mew-summary-multi-one (&optional no-msg)
  "Put the multi mark (default is '@') on this message."
  (mew-mark-put-mark mew-mark-multi no-msg))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Delete: "D" in Summary mode
;;;

(defun mew-summary-delete (&optional count)
  "Put the delete mark (default is 'D') in COUNT times."
  (interactive "P")
  (mew-mark-put-mark-loop (function mew-summary-delete-one) count nil))

(defun mew-summary-delete-one (&optional no-msg)
  "Put the delete mark (default is 'D') on this message."
  (mew-mark-put-mark mew-mark-delete no-msg))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Unlink: "X" in Summary mode
;;;

(defun mew-summary-unlink (&optional count)
  "Put the unlink mark (default is 'X') in COUNT times."
  (interactive "P")
  (mew-mark-put-mark-loop (function mew-summary-unlink-one) count nil))

(defun mew-summary-unlink-one (&optional no-msg)
  "Put the unlink mark (default is 'X') on this message."
  (mew-mark-put-mark mew-mark-unlink no-msg))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; All and regex
;;;

(defun mew-summary-mark-all (&optional arg)
  "Put the '*' mark onto all messages which are not marked."
  (interactive "P")
  (if arg
      (let ((begend (mew-summary-get-region)))
	(mew-summary-mark-region (car begend) (cdr begend)))
    (mew-summary-mark-region (point-min) (point-max))))

(defun mew-summary-mark-region (beg end)
  "Put the '*' mark onto all messages which are not marked between
BEG and END."
  (interactive "r")
  (mew-decode-syntax-delete)
  (let ((regex (concat mew-regex-msg " "))
	(mark mew-mark-review) ;; someday ...
	fld msg alist)
    (save-excursion
      (goto-char beg)
      (while (re-search-forward regex end t)
	(mew-summary-mark-as mark)
	(when (mew-virtual-p)
	  (setq fld (mew-summary-folder-name))
	  (setq msg (mew-summary-message-number))
	  (mew-mark-alist-set alist fld msg))
	(forward-line)) ;; make search faster
      (set-buffer-modified-p nil))
    (mew-summary-mark-in-physical-alist alist mark)))

(defun mew-summary-mark-regexp ()
  "Put the '*' mark onto all messages matched to a regular expression."
  (interactive)
  (mew-decode-syntax-delete)
  (let ((regex (read-string "Regexp: "))
	(mark mew-mark-review) ;; someday ...
	fld msg alist)
    (while (string= regex "")
      (setq regex (read-string "Regexp: ")))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
	(when (looking-at "[^\n]*\r") ;; before thread info
	  (mew-summary-mark-as mark)
	  (when (mew-virtual-p)
	    (setq fld (mew-summary-folder-name))
	    (setq msg (mew-summary-message-number))
	    (mew-mark-alist-set alist fld msg)))
	(forward-line)) ;; make search faster
      (set-buffer-modified-p nil))
    (mew-summary-mark-in-physical-alist alist mark)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Undo
;;;

(defsubst mew-mark-unmark ()
  (save-excursion
    (beginning-of-line)
    (if (re-search-forward mew-regex-msg nil t)
	(mew-mark-delete-here))))

(defun mew-summary-undo (&optional count)
  "Cancel the mark in COUNT times."
  (interactive "P")
  (mew-mark-put-mark-loop (function mew-summary-undo-one) count 'stayp))

(defun mew-summary-undo-one (&optional no-msg)
  "Cancel the mark on this message."
  (if (eobp)
      (or no-msg (message "No message"))
    (let (mark func fld msg)
      (save-excursion
	(mew-summary-goto-message)
	(setq mark (mew-summary-get-mark))
	(if (null mark)
	    (or no-msg (message "No mark"))
	  (setq func (mew-markdb-func-undo mark))
	  (or (fboundp func) (setq func nil))
	  (setq fld (mew-summary-folder-name))
	  (setq msg (mew-summary-message-number))
	  (cond
	   ((mew-virtual-p)
	    (mew-mark-unmark)
	    (mew-summary-unmark-in-physical fld msg func))
	   (t ;; Summary mode
	    (mew-mark-unmark)
	    (if func (funcall func fld msg)))))))))

;;

(defun mew-summary-undo-all ()
  "Cancel all marks according to what you input."
  (interactive)
  (let ((char (mew-input-mark)))
    (if (and char (not (char-equal char ?\r)))
	(mew-mark-undo-mark char))))

(defun mew-mark-undo-mark (mark &optional no-msg)
  "Undo MARK on the entire buffer.
If optional argument NO-MSG is non-nil, no message is displayed."
  (or no-msg (message "Unmarking ... "))
  (mew-decode-syntax-delete)
  (let* ((regex (mew-mark-regex mark))
	 (func (mew-markdb-func-undo mark))
	 alist fld msg)
    (or (fboundp func) (setq func nil))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regex nil t)
	(setq fld (mew-summary-folder-name))
	(setq msg (mew-summary-message-number))
	(mew-mark-delete-here t)
	(if (mew-virtual-p)
	    (mew-mark-alist-set alist fld msg)
	  (if func (funcall func fld msg))) ;; in physical only
	(forward-line))  ;; make search faster
      (set-buffer-modified-p nil)
      (mew-summary-unmark-in-physical-alist alist func)))
  (or no-msg (message "Unmarking ... done")))

;;

(defun mew-summary-mark-undo-all ()
  "Unmark all message marked with 'o' or 'D' or 'X'."
  (interactive)
  (let ((marks mew-summary-mark-undo-marks))
    (message "Unmarking ... ")
    (while marks
      (mew-mark-undo-mark (car marks) 'nomsg)
      (setq marks (cdr marks)))
    (message "Unmarking ... done")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Exchange
;;;

(defun mew-summary-exchange-mark (oldmark newmark)
  (let ((regex (mew-mark-regex oldmark))
	fld msg alist)
    (save-excursion
      (goto-char (point-min))
      (if (not (re-search-forward regex nil t))
	  (message "No marked messages")
	(beginning-of-line)
	(while (re-search-forward regex nil t)
	  (mew-mark-put-here newmark t)
	  (when (mew-virtual-p)
	    (setq fld (mew-summary-folder-name))
	    (setq msg (mew-summary-message-number))
	    (mew-mark-alist-set alist fld msg))
	  (forward-line)) ;; make search faster
	(set-buffer-modified-p nil)
	(mew-summary-mark-in-physical-alist alist newmark)))))
   
(defun mew-summary-mark-delete ()	;; * -> D
  "Put the delete mark onto all messages marked with '*'."
  (interactive)
  (mew-summary-exchange-mark mew-mark-review mew-mark-delete))

(defun mew-summary-mark-unlink ()	;; * -> X
  "Put the delete mark onto all messages marked with '*'."
  (interactive)
  (mew-summary-exchange-mark mew-mark-review mew-mark-unlink))

(defun mew-summary-mark-multi ()	;; * -> @
  "Change the '*' mark into the '@' mark."
  (interactive)
  (mew-summary-exchange-mark mew-mark-review mew-mark-multi))

(defun mew-summary-mark-review ()	;; @ -> *
  "Change the '@' mark into the '*' mark."
  (interactive)
  (mew-summary-exchange-mark mew-mark-multi mew-mark-review))

(defun mew-summary-mark-swap ()		;; @ <-> *
  "Swap the '@' mark and the '*' mark."
  (interactive)
  (mew-summary-exchange-mark mew-mark-multi mew-mark-tmp)
  (mew-summary-exchange-mark mew-mark-review mew-mark-multi)
  (mew-summary-exchange-mark mew-mark-tmp mew-mark-review))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Processing marks
;;;

(defun mew-summary-exec (&optional arg)
  "\\<mew-summary-mode-map> Process marked messages. When called with
'\\[universal-argument]', process marked messages in the specified
region. To cancel the '*' mark, use '\\[mew-summary-undo]' or
'\\[mew-summary-undo-all]'."
  (interactive "P")
  (cond
   ((mew-virtual-p)
    (if (not (mew-thread-p))
	(message "Move to a physical folder first")
      (let ((folder (substring (mew-summary-folder-name 'ext) 1))
	    (msg (mew-summary-message-number)))
	(if (not (and folder (get-buffer folder)))
	    (message "No physical folder")
	  (mew-summary-goto-folder nil folder 'no-ls)
	  (if msg
	      (mew-summary-jump-message msg)
	    (goto-char (point-max)))
	  (message "Now in %s. Type '%s' again"
		   folder
		   (substitute-command-keys
		    "\\<mew-summary-mode-map>\\[mew-summary-exec]"))))))
   (arg
    (mew-summary-exec-region (mark) (point)))
   (t
    (mew-summary-exec-region (point-min) (point-max)))))

(defun mew-summary-exec-one (&optional arg)
  "Process the current marked messages.
When called with '\\[universal-argument]', process 
all marked messages before the current message."
  (interactive "P")
  (mew-summary-only
   (mew-summary-goto-message)
   (mew-decode-syntax-delete)
   (save-window-excursion
     (let (beg end)
       (save-excursion
	 (beginning-of-line)
	 (setq beg (if arg (point-min) (point)))
	 (end-of-line)
	 (setq end (1+ (point))))
       (if (> end (point-max))
	   (message "No message")
	 (mew-summary-exec-region beg end))))))

(defun mew-summary-exec-delete ()
  "Process messages marked with 'D'."
  (interactive)
  (mew-summary-only
   (let* ((ent (assoc mew-mark-delete mew-mark-spec))
	  (mew-mark-spec (list ent)))
     (mew-summary-exec-region (point-min) (point-max)))))

(defun mew-summary-exec-unlink ()
  "Process messages marked with 'X'."
  (interactive)
  (mew-summary-only
   (let* ((ent (assoc mew-mark-unlink mew-mark-spec))
	  (mew-mark-spec (list ent)))
     (mew-summary-exec-region (point-min) (point-max)))))

(defun mew-summary-exec-refile ()
  "Process messages marked with 'o'."
  (interactive)
  (mew-summary-only
   (let* ((ent (assoc mew-mark-refile mew-mark-spec))
	  (mew-mark-spec (list ent)))
     (mew-summary-exec-region (point-min) (point-max)))))

(defvar mew-summary-exec-error-msg nil)

(defun mew-summary-exec-region (beg end)
  "Process marked messages between BEG and END."
  (interactive "r")
  (mew-summary-only
   (when (mew-summary-exclusive-p)
     (save-excursion
       (save-restriction
	 (narrow-to-region beg end)
	 (goto-char (point-min))
	 (message "Collecting marks ... ")
	 (condition-case nil
	     (let ((marks (mew-mark-get-all-marks))
		   (src (mew-summary-folder-name 'ext))
		   (cnt 0)
		   msgs err-folders mark
		   func-exec func-sanity)
	       (while marks
		 (setq mark (car marks))
		 (setq marks (cdr marks))
		 (setq func-exec (mew-markdb-func-exec mark))
		 (when func-exec
		   (setq msgs (mew-summary-mark-collect 
			       mark (point-min) (point-max)))
		   (when msgs
		     (when (= cnt 0)
		       ;; opening...
		       (setq mew-summary-buffer-process t)
		       (message "Refiling and deleting ... ")
		       (mew-window-configure 'summary)
		       (mew-current-set nil nil nil)
		       (mew-decode-syntax-delete))
		     (setq cnt (1+ cnt))
		     ;; refiling and deleting...
		     (setq func-sanity (mew-markdb-func-sanity mark))
		     (when (and func-sanity
				(setq err-folders (funcall func-sanity msgs)))
		       (setq mew-summary-buffer-process nil)
		       (mew-warn
			"Nothing proceeded. Folder(s) MUST be a file!: %s"
			(mew-join "," err-folders)))
		     (funcall func-exec src msgs))))
	       (if (= cnt 0)
		   (message "No marks")
		 ;; ending...
		 (mew-mark-kill-line-region (point-min) (point-max))
		 (mew-summary-folder-cache-save)
                 (mew-touch-folder (mew-summary-folder-name 'ext))
		 (setq mew-summary-buffer-process nil)
		 (run-hooks 'mew-summary-exec-hook)
		 (set-buffer-modified-p nil)
		 (mew-summary-reset-mode-line (current-buffer))
		 (mew-cache-clean-up)
		 (message "Refiling and deleting ... done")))
	   (quit
	    (set-buffer-modified-p nil)
	    (setq mew-summary-buffer-process nil))))))))

(defsubst mew-mark-kill-line ()
  (let (start)
    (beginning-of-line)
    (setq start (point))
    (forward-line)
    (mew-elet (delete-region start (point)))))

(defun mew-mark-kill-line-region (beg end)
  "Kill lines marked with mark to be killed."
  (let ((m (make-marker)) mark)
    (set-marker m end)
    (goto-char beg)
    (while (re-search-forward mew-regex-msg-mark m t)
      (setq mark (string-to-char (mew-match 2)))
      (if (mew-markdb-killp mark)
	  (mew-mark-kill-line)))
    (set-buffer-modified-p nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Moving and removing
;;;

(defun mew-summary-mv-msgs (src dsts-msgs)
  (let* ((dsts (car dsts-msgs)) ;; (+foo +bar)
	 (msgs (cdr dsts-msgs)) ;; (1 2 3)
	 (myselfp (member src dsts))
	 (refile-mark (regexp-quote (char-to-string mew-mark-refile)))
	 msgs- msg srcfile dstfile dst num)
    (if (mew-folder-remotep src)
	() ;; IMAP
      (if myselfp
	  ;; msg stays in the src folder with the same number
	  (progn
	    (setq dsts (delete src dsts))
	    (while msgs
	      (setq msg (car msgs))
	      (setq msgs (cdr msgs))
	      (if (file-regular-p (mew-expand-folder src msg))
		  (setq msgs- (cons msg msgs-))))
	    (setq msgs- (nreverse msgs-))
	    (setq msgs msgs-))
	(setq dst (car dsts)) 
	(setq dsts (cdr dsts))
	(setq num (string-to-int (mew-folder-new-message dst 'num-only)))
	(while msgs
	  (setq srcfile (mew-expand-folder src (car msgs)))
	  (setq msgs (cdr msgs))
	  (when (and (file-exists-p srcfile) (file-writable-p srcfile))
	    (setq msgs- (cons (int-to-string num) msgs-))
	    (setq dstfile (mew-expand-folder dst (int-to-string num)))
	    (setq num (1+ num))
	    (rename-file srcfile dstfile 'override)))
	(mew-touch-folder dst)
	(setq msgs- (nreverse msgs-))
	(setq src dst)
	) ;; myselfp
      (while dsts
	(setq dst (car dsts)) 
	(setq dsts (cdr dsts))
	(setq num (string-to-int (mew-folder-new-message dst 'num-only)))
	(setq msgs msgs-)
	(while msgs
	  (setq srcfile (mew-expand-folder src (car msgs)))
	  (setq msgs (cdr msgs))
	  (setq dstfile (mew-expand-folder dst (int-to-string num)))
	  (setq num (1+ num))
	  (mew-link srcfile dstfile))
	(mew-touch-folder dst)))
    (when myselfp
      (goto-char (point-min))
      (setq msgs msgs-) ;; anyway
      (while msgs
	;; illegal marks remain and the lines will be deleted.
	(if (re-search-forward
	     (format "^ *%s%s" (car msgs) refile-mark) nil t)
	    (mew-mark-delete-here t))
	(setq msgs (cdr msgs))))))

(defun mew-summary-rm-msgs (src dels)
  (if (mew-folder-remotep src)
      () ;; IMAP
    (let* ((queue-p (mew-folder-queuep src))
	   num srcfile dstfile mqifile)
      (setq num (string-to-int
		 (mew-folder-new-message mew-trash-folder 'num-only)))
      ;; must be here after ensuring that +trash exists.
      (while dels
	(setq srcfile (mew-expand-folder src (car dels)))
	(setq dels (cdr dels))
	(setq dstfile (mew-expand-folder mew-trash-folder
					 (int-to-string num)))
	(setq num (1+ num))
	(if (file-regular-p srcfile)
	    ;; if not, the marked line will be deleted anyway.
	    (rename-file srcfile dstfile 'override))
	(when queue-p
	  (setq mqifile (concat srcfile mew-queue-info-suffix))
	  (if (file-regular-p mqifile)
	      (delete-file mqifile))))
      (mew-touch-folder mew-trash-folder))))

(defun mew-summary-clean-msgs (src dels)
  (if (mew-folder-remotep src)
      () ;; IMAP
    (let (file)
      (while dels
	(setq file (mew-expand-folder src (car dels)))
	(setq dels (cdr dels))
	(if (file-regular-p file)
	    ;; if not, the marked line will be deleted anyway.
	    (delete-file file))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Reviewing
;;;

(defun mew-summary-set-walk-mark ()
  (let ((char (mew-input-mark)))
    (cond
     ((char-equal char ?\r)
      (setq mew-mark-walk mew-mark-default-walk)
      (message "Target mark was set to '%s'." (char-to-string mew-mark-walk)))
     (char (setq mew-mark-walk char)))))

(defun mew-summary-down-mark (mark)
  (forward-line)
  (cond 
   ((re-search-forward (mew-mark-regex mark) nil t)
    (beginning-of-line)
    t)
   (t 
    (forward-line -1)
    (message "No more marked message")
    nil)))

(defun mew-summary-display-review-down (&optional arg)
  "Jump to the message marked with '*' below."
  (interactive "P")
  (if arg
      (mew-summary-set-walk-mark)
    (if (mew-summary-down-mark mew-mark-walk)
	(mew-summary-display nil))))

(defun mew-summary-up-mark (mark)
  (cond 
   ((re-search-backward (mew-mark-regex mark) nil t)
    t)
   (t 
    (message "No more marked message")
    nil)))

(defun mew-summary-display-review-up (&optional arg)
  "Jump to the message marked with '*' above."
  (interactive "P")
  (if arg
      (mew-summary-set-walk-mark)
    (if (mew-summary-up-mark mew-mark-walk)
	(mew-summary-display nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Cleaning +trash
;;;

(defun mew-summary-clean-folder-cache (folder)
  "Erase Summary mode then remove and touch the cache file."
  (if (get-buffer folder)
      (save-excursion
	(set-buffer folder)
	(mew-erase-buffer)
	(set-buffer-modified-p nil)))
  (let ((cfile (mew-expand-folder folder mew-summary-cache-file)))
    (if (file-exists-p cfile)
	(write-region "" nil cfile nil 'no-msg))))

(defun mew-summary-clean-trash ()
  "Remove all messages in +trash."
  (interactive)
  (let* ((trashdir (mew-expand-folder mew-trash-folder))
	 (msgs (mew-dir-messages trashdir)))
    (if (null msgs)
	(message "No messages to be removed in %s" mew-trash-folder)
      (when (yes-or-no-p
	     (format "Remove all messages in %s? " mew-trash-folder))
	(message "Removing all messages in %s ... " mew-trash-folder)
	(mew-summary-clean-msgs mew-trash-folder msgs)
	(mew-summary-clean-folder-cache mew-trash-folder)
	(message "Removing all messages in %s ... done" mew-trash-folder)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Clean up marks!
;;;

(defun mew-mark-clean (&optional sum-buf)
  "Process marked messages for this folder."
  (if sum-buf
      (set-buffer sum-buf)
    (setq sum-buf (mew-summary-folder-name 'ext)))
  (if (and (mew-summary-p)
           (mew-summary-mark-exist-p
            (list mew-mark-delete mew-mark-refile mew-mark-unlink))) ;; xxx
      (if (y-or-n-p (format "Marks exist in %s. Process them? " sum-buf))
	  (mew-summary-exec))))

(defun mew-mark-init ()
  (add-hook 'kill-emacs-hook (function mew-mark-clean-up)))

(defun mew-mark-clean-up ()
  "Process marked messages for all Summary modes.
Typically called by kill-emacs-hook."
  (remove-hook 'kill-emacs-hook (function mew-mark-clean-up))
  (mew-decode-syntax-delete)
  (let ((bufs mew-buffers) buf)
    (save-excursion
      (while bufs
	(setq buf (car bufs))
	(setq bufs (cdr bufs))
	(if (bufferp (get-buffer buf))
	    (mew-mark-clean buf))))))

;; Saving marks is a really bad idea.
;; First because there is no way to fill the gap if the folder is newer
;; than the cache at quitting.
;; Even if the cache is newer, saving marks faces dilemma if 
;; multiple Emacses run.

(provide 'mew-mark)

;;; Copyright Notice:

;; Copyright (C) 1997-2001 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-mark.el ends here
