;;; mew-smtp.el

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Dec  3, 1999
;; Revised: Jul  8, 2001

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; SMTP info
;;;

(defvar mew-smtp-info-list
  '("status" "recipients" "orig-recipients"
    "messages" "error" "lock" "ssh-process" "auth"
    "string" "cont" "timer" "server" "ssh-server" "port"
    "raw-header" "from" "bcc" "dcc" "fcc" "logtime" "msgid"
    "helo-domain" "user" "auth" "auth-list" "queue" "case"
    "sender"))

(mew-info-defun "mew-smtp-" mew-smtp-info-list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Process name
;;;

(defconst mew-smtp-info-prefix "mew-smtp-info-")

(defsubst mew-smtp-info-name (server &optional sshsrv)
  (if sshsrv
      (concat mew-smtp-info-prefix sshsrv "-" server)
    (concat mew-smtp-info-prefix server)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; FSM
;;;

(defvar mew-smtp-fsm
  '(("greeting"     ("220" . "ehlo"))
    ("ehlo"         ("250" . "auth") (t . "helo"))
    ;;
    ("auth"         ("250" . "mail-from"))
    ("cram-md5"     ("334" . "pwd-cram-md5") (t . "wpwd"))
    ("pwd-cram-md5" ("235" . "mail-from") (t . "wpwd"))
    ("login"        ("334" . "pwd-login") (t . "wpwd"))
    ("pwd-login"    ("235" . "mail-from") (t . "wpwd"))
    ("plain"        ("235" . "mail-from") (t . "wpwd"))
    ;;
    ("helo"         ("250" . "mail-from"))
    ("mail-from"    ("250" . "rcpt-to"))
    ("rcpt-to"      ("250" . "data"))
    ("data"         ("354" . "content"))
    ("content"      ("250" . "done"))
    ("done"         ("250" . "noop"))))

(defsubst mew-smtp-fsm-by-status (status)
  (assoc status mew-smtp-fsm))

(defsubst mew-smtp-fsm-next (status code)
  (cdr (mew-assoc-match code (cdr (mew-smtp-fsm-by-status status)) 0)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; SMTP AUTH
;;;

(defun mew-smtp-passtag (pnm auth)
  (concat (downcase auth)
	  "/" (mew-smtp-get-user pnm)
	  "@" (mew-smtp-get-server pnm)
	  ":" (mew-smtp-get-port pnm)))

(defun mew-smtp-input-passwd (prompt pnm auth)
  (let ((tag (mew-smtp-passtag pnm auth)))
    (mew-input-passwd prompt tag)))

(defvar mew-smtp-auth-alist
  '(("CRAM-MD5" mew-smtp-command-auth-cram-md5)
    ("PLAIN"    mew-smtp-command-auth-plain)
    ("LOGIN"    mew-smtp-command-auth-login)))

(defsubst mew-smtp-auth-get-func (auth)
  (nth 1 (mew-assoc-case-equal auth mew-smtp-auth-alist 0)))

;;

(defun mew-smtp-command-auth-cram-md5 (pro pnm)
  (process-send-string
   pro (format "AUTH CRAM-MD5%s" mew-cs-eol))
  (mew-smtp-set-status pnm "cram-md5"))

(defun mew-smtp-command-pwd-cram-md5 (pro pnm)
  (let ((str (mew-smtp-get-string pnm))
	(user (mew-smtp-get-user pnm))
	challenge passwd cram-md5)
    (if (string-match " \\([A-Za-z0-9+/]+=*\\)" str) ;; xxx
	(setq challenge (mew-match 1 str)))
    (setq passwd (mew-smtp-input-passwd "CRAM-MD5 password: " pnm "cram-md5"))
    (setq cram-md5 (mew-cram-md5 user passwd challenge))
    (process-send-string pro (format "%s%s" cram-md5 mew-cs-eol))))

(defun mew-smtp-command-auth-login (pro pnm)
  (let* ((user (mew-smtp-get-user pnm))
	 (euser (mew-base64-encode-string user)))
    (process-send-string pro (format "AUTH LOGIN %s%s" euser mew-cs-eol))
    (mew-smtp-set-status pnm "login")))

(defun mew-smtp-command-pwd-login (pro pnm)
  (let* ((passwd (mew-smtp-input-passwd "LOGIN password: " pnm "login"))
	 (login (mew-base64-encode-string passwd)))
    (process-send-string pro (format "%s%s" login mew-cs-eol))))

(defun mew-smtp-command-auth-plain (pro pnm)
  (let* ((passwd (mew-smtp-input-passwd "PLAIN password: " pnm "plain"))
	 (user (mew-smtp-get-user pnm))
	 (plain (mew-base64-encode-string (format "\0%s\0%s" user passwd))))
    (process-send-string pro (format "AUTH PLAIN %s%s" plain mew-cs-eol))
    (mew-smtp-set-status pnm "plain")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Commands
;;;

(defun mew-smtp-send-message (pnm &optional unlock)
  (let* ((process (get-process pnm))
	 (server (mew-smtp-get-server pnm))
	 (port (mew-smtp-get-port pnm))
	 (ssh-server (mew-smtp-get-ssh-server pnm))
	 (ret t)
	 sshpro sshname errmsg opened lport)
    (cond
     ((null (mew-smtp-get-recipients pnm))
      (setq ret nil)
      (message "No recipient!"))
     ((and (processp process) (not unlock) (mew-smtp-get-lock pnm))
      (setq errmsg "SMTP connection is locked.")
      (mew-smtp-set-error pnm errmsg)
      (mew-smtp-queue pnm errmsg))
     (t
      (if (and (processp process)
	       (memq (process-status process) '(open)))
	  (setq opened t))
      (mew-smtp-set-string pnm nil)
      (cond
       (opened
	(set-process-buffer process (current-buffer))
	(set-process-sentinel process 'mew-smtp-sentinel)
	(set-process-filter process 'mew-smtp-filter)
	(mew-smtp-command-mail-from process pnm)
	(mew-smtp-set-status pnm "mail-from")
	(message "Sending in background ... "))
       (t
	(if (null ssh-server)
	    (setq process (mew-smtp-open pnm server port))
	  (setq sshpro (mew-open-ssh-stream server port ssh-server))
	  (mew-smtp-set-ssh-process pnm sshpro)
	  (if (null sshpro)
	      (setq errmsg (concat "Can't connect to " ssh-server))
	    (setq sshname (process-name sshpro))
	    (setq lport (mew-ssh-pnm-to-lport sshname))
	    (if lport (setq process (mew-smtp-open pnm "localhost" lport)))))
	(cond
	 (process
	  (mew-smtp-set-status pnm "greeting")
	  (set-process-buffer process (current-buffer))
	  (set-process-sentinel process 'mew-smtp-sentinel)
	  (set-process-filter process 'mew-smtp-filter)
	  (message "Sending in background ... "))
	 (t
	  (mew-smtp-queue pnm errmsg)
	  (mew-smtp-tear-down pnm)))))))
    ret))

(defun mew-smtp-flush-queue (qfld case)
  (let* ((server (mew-smtp-server case))
	 (ssh-server (mew-smtp-ssh-server case))
	 (pnm (mew-smtp-info-name server ssh-server))
	 msgs flushp-msgs flushp)
    (cond
     ((mew-smtp-get-messages pnm)
      (message "%s is being flushed" qfld))
     ((mew-smtp-get-lock pnm)
      (message "SMTP connection for %s is locked" qfld))
     (t
      ;; can't lock here
      (mew-summary-clean-folder-cache qfld)
      (setq msgs (directory-files (mew-expand-folder qfld)
				  'full mew-regex-message-files))
      (setq flushp-msgs (mew-smtp-get-next pnm msgs)) ;; a file inserted
      (setq flushp (car flushp-msgs))
      (mew-smtp-set-case pnm case)
      (mew-smtp-set-server pnm server)
      (mew-smtp-set-port pnm (mew-smtp-port case))
      (mew-smtp-set-ssh-server pnm ssh-server)
      (mew-smtp-set-queue pnm qfld)
      (mew-smtp-set-messages pnm (cdr flushp-msgs))
      (mew-smtp-set-auth pnm (mew-smtp-auth case))
      (mew-smtp-set-auth-list pnm (mew-smtp-auth-list case))
      (mew-smtp-set-user pnm (mew-smtp-user case))
      (mew-smtp-set-helo-domain pnm (mew-smtp-helo-domain case))
      (mew-smtp-set-sender pnm nil)
      ;; If Mail-From is specified for the case, override the old one.
      (if (mew-smtp-mail-from case)
	  (mew-smtp-set-from pnm (mew-smtp-mail-from case)))
      ;; in msg's buffer
      (when flushp
	(message "Flushing %s ... " qfld)
	(mew-smtp-send-message pnm))))
    (run-hooks 'mew-smtp-flush-hook))) ;; xxx

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Opening SMTP
;;;

(defun mew-smtp-open (pnm server port)
  (let ((sprt (mew-port-sanity-check port))
	errmsg process tm)
    (mew-smtp-set-error pnm nil)
    (condition-case emsg
	(progn
	  (setq tm (mew-timer mew-smtp-timeout-time 'mew-smtp-timeout))
	  (message "Connecting to %s with %s port ... " server port)
	  (setq process (open-network-stream pnm nil server sprt))
	  (process-kill-without-query process)
	  (mew-set-process-cs process mew-cs-text-for-net mew-cs-text-for-net)
	  (message "Connecting to %s with %s port ... done" server port))
      (quit
       (setq errmsg (format "can't make a %s connection to %s" port server))
       (mew-smtp-set-error pnm errmsg)
       (setq process nil)
       (message errmsg))
      (error
       (setq errmsg (nth 1 emsg))
       (mew-smtp-set-error pnm errmsg)
       (setq process nil)
       (message errmsg)))
    (if tm (cancel-timer tm))
    process))

(defun mew-smtp-timeout ()
  (signal 'quit nil))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Closing SMTP
;;;

(defun mew-smtp-close (pnm)
  (interactive)
  (let ((process (get-process pnm)))
    (if (and (processp process) (memq (process-status process) '(open)))
	(progn
	  (set-process-buffer process nil)
	  (set-process-filter process nil)
	  (process-send-string process (format "QUIT%s" mew-cs-eol)))
      (mew-smtp-tear-down pnm))))

(defun mew-smtp-tear-down (pnm)
  (let ((process (get-process pnm))
	(sshpro (mew-smtp-get-ssh-process pnm)))
    (if (processp process) (delete-process process))
    (if (processp sshpro) (delete-process sshpro))
    (mew-info-clean-up pnm)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Filter and sentinel
;;;

(defun mew-smtp-debug (label string)
  (when (mew-debug 'net)
    (save-excursion
      (set-buffer (get-buffer-create mew-buffer-debug))
      (insert (format "\n<%s>\n%s\n" label string)))))

(defun mew-smtp-filter (process string)
  (let* ((pnm (process-name process))
	 (status (mew-smtp-get-status pnm))
	 (str (concat (mew-smtp-get-string pnm) string))
	 next func code)
    (mew-smtp-debug (upcase status) string)
    (mew-filter
     ;; SMTP server's strings should be short enough.
     (mew-smtp-set-string pnm str)
     (cond
      ((and (string-match "\n$" str)
	    (string-match "^\\([1-5][0-7][0-9]\\) " str))
       (setq code (mew-match 1 str))
       (setq next (mew-smtp-fsm-next status code))
       (cond
	(next
	 (mew-smtp-set-status pnm next)
	 (setq func (intern-soft (concat "mew-smtp-command-" next)))
	 (and func (funcall func process pnm))
	 (mew-smtp-set-string pnm nil))
	(t
	 ;; (mew-passwd-set-passwd mew-draft-smtp-server nil) ;; xxx
	 (if (string-match "^pwd-" status)
	     (message "Password is wrong."))
	 (mew-smtp-recover pnm str))))
      (t ()))))) ;; stay

(defun mew-smtp-sentinel (process event)
  (let ((pnm (process-name process)))
    (mew-smtp-debug "SMTP SENTINEL" event)
    (mew-smtp-tear-down pnm)
    (run-hooks 'mew-smtp-sentinel-hook)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Filters
;;;

(defun mew-smtp-command-ehlo (pro pnm)
  (let ((helo-domain (mew-smtp-get-helo-domain pnm)))
    (process-send-string pro (format "EHLO %s%s" helo-domain mew-cs-eol))))

(defun mew-smtp-command-helo (pro pnm)
  (let ((helo-domain (mew-smtp-get-helo-domain pnm)))
    (process-send-string pro (format "HELO %s%s" helo-domain mew-cs-eol))))

(defun mew-smtp-command-auth (pro pnm)
  (cond
   ((mew-smtp-get-auth pnm)
    (let ((str (mew-smtp-get-string pnm)) auth func)
      (if (and (string-match "AUTH \\([^\n\r]+\\)\r?\n" str)
	       (setq auth (mew-auth-select
			   (mew-match 1 str) mew-smtp-auth-list))
	       (setq func (mew-smtp-auth-get-func auth))
	       (fboundp func))
	  (progn
	    (mew-smtp-set-auth pnm auth)
	    (funcall func pro pnm))
	(mew-smtp-debug "<AUTH>" "No preferred SMTP AUTH.\n")
	;; xxx how should we act?
	(mew-smtp-set-status pnm "mail-from")
	(mew-smtp-command-mail-from pro pnm))))
   (t
    (mew-smtp-set-status pnm "mail-from")
    (mew-smtp-command-mail-from pro pnm))))

(defun mew-smtp-command-wpwd (pro pnm)
  (let ((auth (mew-smtp-get-auth pnm)))
    (mew-passwd-set-passwd (mew-smtp-passtag pnm auth) nil)
    (mew-smtp-recover pnm (format "Wrong password for %s" auth))))

(defun mew-smtp-command-mail-from (pro pnm)
  ;; Session is about to start. Let's lock the connection first.
  (let ((from (mew-smtp-get-from pnm)))
    (mew-smtp-set-lock pnm t)
    (process-send-string pro (format "MAIL FROM:<%s>%s" from mew-cs-eol))))

(defun mew-smtp-command-rcpt-to (pro pnm)
  (let* ((recipients (mew-smtp-get-recipients pnm))
	 (recipient (car recipients)))
    (setq recipients (cdr recipients))
    (mew-smtp-set-recipients pnm recipients)
    (if recipients (mew-smtp-set-status pnm "mail-from"))
    (process-send-string pro (format "RCPT TO:<%s>%s" recipient mew-cs-eol))))

(defun mew-smtp-command-data (pro pnm)
  (widen)
  (clear-visited-file-modtime)
  ;;
  (let* ((resentp (mew-header-existp mew-resent-from:))
	 (sender: (if resentp mew-resent-sender: mew-sender:))
	 (from: (if resentp mew-resent-from: mew-from:)))
    (unless (mew-header-existp sender:)
      (let* ((from (mew-addrstr-parse-address-list 
		    (mew-header-get-value from:)))
	     (nfrom (length from))
	     (mail-from (mew-smtp-get-from pnm)))
	(if (= nfrom 1)
	    (if (and mew-use-sender (not (string= mail-from (car from))))
		(mew-smtp-set-sender pnm (cons sender: mail-from)))
	  (mew-smtp-set-sender pnm (cons sender: mail-from))))))
  ;;
  (goto-char (point-max))
  (unless (bolp) (insert "\n"))
  (goto-char (point-min))
  (while (re-search-forward "^\\." nil t)
    (insert ".")
    (forward-line))
  (when (string= mew-cs-eol "\r\n")
    (goto-char (point-min))
    (while (search-forward "\n" nil t) (replace-match "\r\n" nil t)))
  (set-buffer-modified-p nil)
  (mew-smtp-set-cont pnm (point-min))
  (mew-smtp-set-timer pnm nil)
  (process-send-string pro (format "DATA%s" mew-cs-eol)))

(defun mew-smtp-command-content (pro pnm)
  (save-excursion
    (let ((cont (mew-smtp-get-cont pnm))
	  (sender (mew-smtp-get-sender pnm))
          (inc 1000) (i 0) (N 10))
      (set-buffer (process-buffer pro))
      ;; Sender:
      (when (and sender (= cont (point-min)))
	(process-send-string
	 pro (format "%s %s%s" (car sender) (cdr sender) mew-cs-eol))
	;; Ensuring cont is not (point-min) in the next time
        (let ((next (min (point-max) (+ cont inc))))
          (process-send-region pro cont next)
          (setq cont next)
          (setq i (1+ i))))
      ;;
      (while (and (< cont (point-max)) (not (input-pending-p)) (< i N))
        (let ((next (min (point-max) (+ cont inc))))
          (process-send-region pro cont next)
          (setq cont next)
          (setq i (1+ i))))
      (mew-smtp-set-cont pnm cont)
      (if (< cont (point-max))
          (let ((timer
                 (if (input-pending-p)
                     (run-with-idle-timer
		      0.01 nil 'mew-smtp-command-content pro pnm)
                   (run-at-time 0.01 nil 'mew-smtp-command-content pro pnm))))
            (mew-smtp-set-timer pnm timer))
        (mew-smtp-set-cont pnm nil)
        (mew-smtp-set-timer pnm nil)
        (process-send-string pro (format ".%s" mew-cs-eol))))))

(defun mew-queue-backup (orig)
  (let* ((norg (file-name-sans-extension orig))
	 (back (mew-prepend-prefix norg mew-backup-prefix))
	 (info (concat norg mew-queue-info-suffix))
	 (ibck (mew-prepend-prefix info mew-backup-prefix)))
    (if (file-exists-p orig) (rename-file orig back 'override))
    (if (file-exists-p info) (rename-file info ibck 'override))
    back))

(defun mew-smtp-command-done (pro pnm)
  (let* ((back (mew-queue-backup (buffer-file-name)))
	 (fcc (mew-smtp-get-fcc pnm))
	 fcc-file link fld buf)
    ;; mew-folder-new-message may be slow if the folder contains
    ;; a lot of messages. So, let's Fcc in background.
    (catch 'loop
      (while fcc
	(setq fld (car fcc))
	(setq fcc (cdr fcc))
	(setq fcc-file (mew-folder-new-message fld))
	(when fcc-file
	  (copy-file back fcc-file)
	  (mew-touch-folder fld)
	  (throw 'loop nil))))
    (while fcc
      (setq fld (car fcc))
      (setq fcc (cdr fcc))
      (setq link (mew-folder-new-message fld))
      (when link
	(mew-link fcc-file link)
	(mew-touch-folder fld)))
    (mew-smtp-log pnm)
    (if (mew-smtp-get-bcc pnm)
	(mew-smtp-bcc pnm back)
      ;; killing buffer
      (setq buf (process-buffer pro))
      (set-process-buffer pro nil)
      (mew-remove-buffer buf)
      (mew-smtp-send-next-message pnm))))

(defun mew-smtp-send-next-message (pnm &optional no-msg)
  (if (mew-smtp-get-messages pnm)
      (let* ((msgs (mew-smtp-get-messages pnm))
	     (flushp-msgs (mew-smtp-get-next pnm msgs)) ;; a file inserted
	     (flushp (car flushp-msgs)))
	;; in msg's buffer
	(mew-smtp-set-messages pnm (cdr flushp-msgs))
	(if flushp (mew-smtp-send-message pnm 'unlock)))
    (mew-smtp-set-status pnm "noop")
    (mew-smtp-set-lock pnm nil)
    (unless mew-smtp-keep-connection (mew-smtp-close pnm))
    (or no-msg (message "Sending in background ... done"))))

(defun mew-smtp-command-noop (pro pnm)
  ())

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Bcc:
;;;

(defun mew-smtp-bcc (pnm back)
  (let* ((dir (file-name-directory back))
	 (msg (file-name-nondirectory back)))
    (mew-elet
     (erase-buffer)
     (mew-set-buffer-multibyte t)
     (mew-smtp-set-recipients pnm (mew-smtp-get-bcc pnm))
     (mew-smtp-set-orig-recipients pnm (mew-smtp-get-bcc pnm))
     (mew-smtp-set-bcc pnm nil)
     (mew-draft-header-insert mew-to: "Bcc-Receiver:;")
     (mew-draft-header-insert mew-subj: mew-bcc-subject)
     (mew-draft-header-insert mew-from: (mew-smtp-get-from pnm))
     (mew-header-set (concat mew-header-separator "\n"))
     (insert mew-bcc-body)
     (goto-char (mew-header-end))
     (forward-line)
     (setq mew-encode-syntax (mew-encode-syntax-initial dir))
     (setq mew-encode-syntax
	   (mew-syntax-insert-entry
	    mew-encode-syntax
	    '(2)
	    (mew-encode-syntax-single msg mew-type-msg nil nil nil)))
     (mew-encode-multipart mew-encode-syntax dir 0 'buffered)
     (mew-encode-make-header)
     (mew-encode-save-draft)
     (mew-overlay-delete-buffer)
     (mew-smtp-send-message pnm 'unlock))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Queuing
;;;

;; qdir -> qfld

(defun mew-smtp-queue (pnm err)
  ;; Must be in a buffer where a message is contained.
  (let* ((oname (buffer-name))
	 (orig (buffer-file-name))
	 (qfld (mew-smtp-get-queue pnm))
	 file file-info nname)
    (mew-folder-check qfld 'force-to-create)
    (if (and (string-match (concat mew-queue-work-suffix "$") orig)
	     ;; another process has already queued?
	     (not (file-exists-p (file-name-sans-extension orig))))
	;; +queue/1.wrk
	(setq file (file-name-sans-extension orig))
      ;; +draft/1
      (setq file (mew-folder-new-message qfld)))
    (rename-file orig file 'override)
    (setq file-info (concat file mew-queue-info-suffix))
    (setq nname (concat (file-name-as-directory qfld)
			(file-name-nondirectory file)))
    (mew-lisp-save file-info (mew-info pnm))
    (mew-remove-buffer (current-buffer))
    (message "%s has been queued to %s (%s)"
	     oname nname (or (mew-smtp-get-error pnm) err))
    (mew-touch-folder qfld)
    (mew-smtp-set-error pnm nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Next message
;;;

(defun mew-smtp-insert-file (pnm file)
  (when (file-readable-p file)
    (let* ((work (concat file mew-queue-work-suffix))
	   (info (concat file mew-queue-info-suffix))
	   (server (mew-smtp-get-server pnm))
	   (port (mew-smtp-get-port pnm))
	   (ssh-server (mew-smtp-get-ssh-server pnm)))
      (rename-file file work 'override)
      (mew-frwlet
       mew-cs-text-for-read mew-cs-dummy
       (set-buffer (find-file-noselect work)))
      (set (intern pnm) (mew-lisp-load info))
      (mew-smtp-set-server pnm server)
      (mew-smtp-set-port pnm port)
      (mew-smtp-set-ssh-server pnm ssh-server)
      (mew-smtp-set-recipients pnm (mew-smtp-get-orig-recipients pnm))
      t)))

(defun mew-smtp-get-next (pnm msgs)
  (when msgs
    (let* ((msg (car msgs))
	   flushp)
      (setq msgs (cdr msgs))
      (if (mew-smtp-insert-file pnm msg) (setq flushp t))
      (catch 'loop
	(while (not flushp)
	  (setq msg (car msgs))
	  (setq msgs (cdr msgs))
	  (if (mew-smtp-insert-file pnm msg)
	      (throw 'loop (setq flushp t)))))
      (cons flushp msgs))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Recovery
;;;

(defun mew-smtp-recover (pnm err)
  (set-buffer (process-buffer (get-process pnm)))
  (mew-smtp-log pnm err)
  (mew-smtp-queue pnm err)
  (let ((back (mew-smtp-get-messages pnm)))
    (mew-smtp-close pnm)
    (mew-smtp-set-messages pnm back)
    (mew-smtp-send-next-message pnm 'no-msg))) ;; xxx

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Logging
;;;

(defun mew-smtp-log (pnm &optional err)
  (let ((logtime (mew-smtp-get-logtime pnm))
	(msgid (mew-smtp-get-msgid pnm))
	(recipients (mew-smtp-get-orig-recipients pnm))
	(server (mew-smtp-get-server pnm))
	(buf (generate-new-buffer mew-buffer-prefix)))
    (save-excursion
      (set-buffer buf)
      (mew-erase-buffer)
      (and logtime (insert logtime))
      (and msgid (insert " id=" msgid))
      (and server (insert " server=" server))
      (and recipients
	   (setq recipients (mapconcat (function identity) recipients ",")))
      (and recipients (insert " recipients=" recipients))
      (if err
	  (insert " status=" "("
                  (substring err 0 (string-match "\n+$" err))
                  ")")
	(insert " status=sent"))
      (insert "\n")
      (write-region (point-min) (point-max)
		    (expand-file-name mew-smtp-log-file mew-conf-path)
		    'append 'no-msg))
    (mew-remove-buffer buf)))

(provide 'mew-smtp)

;;; Copyright Notice:

;; Copyright (C) 1999-2001 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-smtp.el ends here
