package Net::Async::Slack::Commands;

use strict;
use warnings;

our $VERSION = '0.014'; # VERSION
our $AUTHORITY = 'cpan:TEAM'; # AUTHORITY

use utf8;

=encoding utf8

=head1 NAME

Net::Async::Slack::Commands - supported API methods for L<Net::Async::Slack>

=head1 DESCRIPTION

These are available on the top-level L<Net::Async::Slack> instance, and are autogenerated
from the API specification.

=cut

use Future::AsyncAwait;
use JSON::MaybeUTF8 qw(:v1);

=head2 admin_apps_approve

Approve an app for installation on a workspace.

L<API method documentation|https://api.slack.com/methods/admin.apps.approve>

Takes the following named parameters:

=over 4

=item * C<app_id> - The id of the app to approve. (string, optional)

=item * C<request_id> - The id of the request to approve. (string, optional)

=item * C<team_id> -  (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_apps_approve {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_apps_approve',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(app_id request_id team_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_apps_approved_list

List approved apps for an org or workspace.

L<API method documentation|https://api.slack.com/methods/admin.apps.approved.list>

Takes the following named parameters:

=over 4

=item * C<limit> - The maximum number of items to return. Must be between 1 - 1000 both inclusive. (integer, optional)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page (string, optional)

=item * C<team_id> -  (string, optional)

=item * C<enterprise_id> -  (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_apps_approved_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_apps_approved_list',
        %args{grep { exists $args{$_} } qw(limit cursor team_id enterprise_id)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_apps_requests_list

List app requests for a team/workspace.

L<API method documentation|https://api.slack.com/methods/admin.apps.requests.list>

Takes the following named parameters:

=over 4

=item * C<limit> - The maximum number of items to return. Must be between 1 - 1000 both inclusive. (integer, optional)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page (string, optional)

=item * C<team_id> -  (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_apps_requests_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_apps_requests_list',
        %args{grep { exists $args{$_} } qw(limit cursor team_id)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_apps_restrict

Restrict an app for installation on a workspace.

L<API method documentation|https://api.slack.com/methods/admin.apps.restrict>

Takes the following named parameters:

=over 4

=item * C<app_id> - The id of the app to restrict. (string, optional)

=item * C<request_id> - The id of the request to restrict. (string, optional)

=item * C<team_id> -  (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_apps_restrict {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_apps_restrict',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(app_id request_id team_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_apps_restricted_list

List restricted apps for an org or workspace.

L<API method documentation|https://api.slack.com/methods/admin.apps.restricted.list>

Takes the following named parameters:

=over 4

=item * C<limit> - The maximum number of items to return. Must be between 1 - 1000 both inclusive. (integer, optional)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page (string, optional)

=item * C<team_id> -  (string, optional)

=item * C<enterprise_id> -  (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_apps_restricted_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_apps_restricted_list',
        %args{grep { exists $args{$_} } qw(limit cursor team_id enterprise_id)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_archive

Archive a public or private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.archive>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to archive. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_archive {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_archive',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_convert_to_private

Convert a public channel to a private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.convertToPrivate>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to convert to private. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_convert_to_private {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_convert_to_private',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_create

Create a public or private channel-based conversation.

L<API method documentation|https://api.slack.com/methods/admin.conversations.create>

Takes the following named parameters:

=over 4

=item * C<name> - Name of the public or private channel to create. (string, required)

=item * C<description> - Description of the public or private channel to create. (string, optional)

=item * C<is_private> - When `true`, creates a private channel instead of a public channel (boolean, required)

=item * C<org_wide> - When `true`, the channel will be available org-wide. Note: if the channel is not `org_wide=true`, you must specify a `team_id` for this channel (boolean, optional)

=item * C<team_id> - The workspace to create the channel in. Note: this argument is required unless you set `org_wide=true`. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_create {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_create',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(name description is_private org_wide team_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_delete

Delete a public or private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.delete>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to delete. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_delete {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_delete',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_disconnect_shared

Disconnect a connected channel from one or more workspaces.

L<API method documentation|https://api.slack.com/methods/admin.conversations.disconnectShared>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to be disconnected from some workspaces. (string, required)

=item * C<leaving_team_ids> - The team to be removed from the channel. Currently only a single team id can be specified. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_disconnect_shared {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_disconnect_shared',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id leaving_team_ids)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_ekm_list_original_connected_channel_info

List all disconnected channels—i.e., channels that were once connected to other workspaces and then disconnected—and the corresponding original channel IDs for key revocation with EKM.

L<API method documentation|https://api.slack.com/methods/admin.conversations.ekm.listOriginalConnectedChannelInfo>

Takes the following named parameters:

=over 4

=item * C<channel_ids> - A comma-separated list of channels to filter to. (string, optional)

=item * C<team_ids> - A comma-separated list of the workspaces to which the channels you would like returned belong. (string, optional)

=item * C<limit> - The maximum number of items to return. Must be between 1 - 1000 both inclusive. (integer, optional)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_ekm_list_original_connected_channel_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_ekm_list_original_connected_channel_info',
        %args{grep { exists $args{$_} } qw(channel_ids team_ids limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_get_conversation_prefs

Get conversation preferences for a public or private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.getConversationPrefs>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to get preferences for. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_get_conversation_prefs {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_get_conversation_prefs',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_get_teams

Get all the workspaces a given public or private channel is connected to within this Enterprise org.

L<API method documentation|https://api.slack.com/methods/admin.conversations.getTeams>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to determine connected workspaces within the organization for. (string, required)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page (string, optional)

=item * C<limit> - The maximum number of items to return. Must be between 1 - 1000 both inclusive. (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_get_teams {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_get_teams',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id cursor limit)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_invite

Invite a user to a public or private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.invite>

Takes the following named parameters:

=over 4

=item * C<user_ids> - The users to invite. (string, required)

=item * C<channel_id> - The channel that the users will be invited to. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_invite {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_invite',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(user_ids channel_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_rename

Rename a public or private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.rename>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to rename. (string, required)

=item * C<name> -  (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_rename {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_rename',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id name)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_restrict_access_add_group

Add an allowlist of IDP groups for accessing a channel

L<API method documentation|https://api.slack.com/methods/admin.conversations.restrictAccess.addGroup>

Takes the following named parameters:

=over 4

=item * C<team_id> - The workspace where the channel exists. This argument is required for channels only tied to one workspace, and optional for channels that are shared across an organization. (string, optional)

=item * C<group_id> - The [IDP Group](https://slack.com/help/articles/115001435788-Connect-identity-provider-groups-to-your-Enterprise-Grid-org) ID to be an allowlist for the private channel. (string, required)

=item * C<channel_id> - The channel to link this group to. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_restrict_access_add_group {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_restrict_access_add_group',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id group_id channel_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_restrict_access_list_groups

List all IDP Groups linked to a channel

L<API method documentation|https://api.slack.com/methods/admin.conversations.restrictAccess.listGroups>

Takes the following named parameters:

=over 4

=item * C<channel_id> -  (string, required)

=item * C<team_id> - The workspace where the channel exists. This argument is required for channels only tied to one workspace, and optional for channels that are shared across an organization. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_restrict_access_list_groups {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_restrict_access_list_groups',
        %args{grep { exists $args{$_} } qw(channel_id team_id)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_restrict_access_remove_group

Remove a linked IDP group linked from a private channel

L<API method documentation|https://api.slack.com/methods/admin.conversations.restrictAccess.removeGroup>

Takes the following named parameters:

=over 4

=item * C<team_id> - The workspace where the channel exists. This argument is required for channels only tied to one workspace, and optional for channels that are shared across an organization. (string, required)

=item * C<group_id> - The [IDP Group](https://slack.com/help/articles/115001435788-Connect-identity-provider-groups-to-your-Enterprise-Grid-org) ID to remove from the private channel. (string, required)

=item * C<channel_id> - The channel to remove the linked group from. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_restrict_access_remove_group {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_restrict_access_remove_group',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id group_id channel_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_search

Search for public or private channels in an Enterprise organization.

L<API method documentation|https://api.slack.com/methods/admin.conversations.search>

Takes the following named parameters:

=over 4

=item * C<team_ids> - Comma separated string of team IDs, signifying the workspaces to search through. (string, optional)

=item * C<query> - Name of the the channel to query by. (string, optional)

=item * C<limit> - Maximum number of items to be returned. Must be between 1 - 20 both inclusive. Default is 10. (integer, optional)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page. (string, optional)

=item * C<search_channel_types> - The type of channel to include or exclude in the search. For example `private` will search private channels, while `private_exclude` will exclude them. For a full list of types, check the [Types section](#types). (string, optional)

=item * C<sort> - Possible values are `relevant` (search ranking based on what we think is closest), `name` (alphabetical), `member_count` (number of users in the channel), and `created` (date channel was created). You can optionally pair this with the `sort_dir` arg to change how it is sorted  (string, optional)

=item * C<sort_dir> - Sort direction. Possible values are `asc` for ascending order like (1, 2, 3) or (a, b, c), and `desc` for descending order like (3, 2, 1) or (c, b, a) (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_search {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_search',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_ids query limit cursor search_channel_types sort sort_dir)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    return $res;
}


=head2 admin_conversations_set_conversation_prefs

Set the posting permissions for a public or private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.setConversationPrefs>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to set the prefs for (string, required)

=item * C<prefs> - The prefs for this channel in a stringified JSON format. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_set_conversation_prefs {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_set_conversation_prefs',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id prefs)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_set_teams

Set the workspaces in an Enterprise grid org that connect to a public or private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.setTeams>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The encoded `channel_id` to add or remove to workspaces. (string, required)

=item * C<team_id> - The workspace to which the channel belongs. Omit this argument if the channel is a cross-workspace shared channel. (string, optional)

=item * C<target_team_ids> - A comma-separated list of workspaces to which the channel should be shared. Not required if the channel is being shared org-wide. (string, optional)

=item * C<org_channel> - True if channel has to be converted to an org channel (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_set_teams {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_set_teams',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id team_id target_team_ids org_channel)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_conversations_unarchive

Unarchive a public or private channel.

L<API method documentation|https://api.slack.com/methods/admin.conversations.unarchive>

Takes the following named parameters:

=over 4

=item * C<channel_id> - The channel to unarchive. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_conversations_unarchive {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_conversations_unarchive',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_emoji_add

Add an emoji.

L<API method documentation|https://api.slack.com/methods/admin.emoji.add>

Takes the following named parameters:

=over 4

=item * C<name> - The name of the emoji to be removed. Colons (`:myemoji:`) around the value are not required, although they may be included. (string, required)

=item * C<url> - The URL of a file to use as an image for the emoji. Square images under 128KB and with transparent backgrounds work best. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_emoji_add {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_emoji_add',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(name url)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_emoji_add_alias

Add an emoji alias.

L<API method documentation|https://api.slack.com/methods/admin.emoji.addAlias>

Takes the following named parameters:

=over 4

=item * C<name> - The name of the emoji to be aliased. Colons (`:myemoji:`) around the value are not required, although they may be included. (string, required)

=item * C<alias_for> - The alias of the emoji. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_emoji_add_alias {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_emoji_add_alias',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(name alias_for)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_emoji_list

List emoji for an Enterprise Grid organization.

L<API method documentation|https://api.slack.com/methods/admin.emoji.list>

Takes the following named parameters:

=over 4

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page (string, optional)

=item * C<limit> - The maximum number of items to return. Must be between 1 - 1000 both inclusive. (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_emoji_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_emoji_list',
        %args{grep { exists $args{$_} } qw(cursor limit)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_emoji_remove

Remove an emoji across an Enterprise Grid organization

L<API method documentation|https://api.slack.com/methods/admin.emoji.remove>

Takes the following named parameters:

=over 4

=item * C<name> - The name of the emoji to be removed. Colons (`:myemoji:`) around the value are not required, although they may be included. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_emoji_remove {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_emoji_remove',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(name)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_emoji_rename

Rename an emoji.

L<API method documentation|https://api.slack.com/methods/admin.emoji.rename>

Takes the following named parameters:

=over 4

=item * C<name> - The name of the emoji to be renamed. Colons (`:myemoji:`) around the value are not required, although they may be included. (string, required)

=item * C<new_name> - The new name of the emoji. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_emoji_rename {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_emoji_rename',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(name new_name)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_invite_requests_approve

Approve a workspace invite request.

L<API method documentation|https://api.slack.com/methods/admin.inviteRequests.approve>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID for the workspace where the invite request was made. (string, optional)

=item * C<invite_request_id> - ID of the request to invite. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_invite_requests_approve {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_invite_requests_approve',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id invite_request_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_invite_requests_approved_list

List all approved workspace invite requests.

L<API method documentation|https://api.slack.com/methods/admin.inviteRequests.approved.list>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID for the workspace where the invite requests were made. (string, optional)

=item * C<cursor> - Value of the `next_cursor` field sent as part of the previous API response (string, optional)

=item * C<limit> - The number of results that will be returned by the API on each invocation. Must be between 1 - 1000, both inclusive (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_invite_requests_approved_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_invite_requests_approved_list',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id cursor limit)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_invite_requests_denied_list

List all denied workspace invite requests.

L<API method documentation|https://api.slack.com/methods/admin.inviteRequests.denied.list>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID for the workspace where the invite requests were made. (string, optional)

=item * C<cursor> - Value of the `next_cursor` field sent as part of the previous API response (string, optional)

=item * C<limit> - The number of results that will be returned by the API on each invocation. Must be between 1 - 1000 both inclusive (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_invite_requests_denied_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_invite_requests_denied_list',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id cursor limit)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_invite_requests_deny

Deny a workspace invite request.

L<API method documentation|https://api.slack.com/methods/admin.inviteRequests.deny>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID for the workspace where the invite request was made. (string, optional)

=item * C<invite_request_id> - ID of the request to invite. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_invite_requests_deny {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_invite_requests_deny',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id invite_request_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_invite_requests_list

List all pending workspace invite requests.

L<API method documentation|https://api.slack.com/methods/admin.inviteRequests.list>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID for the workspace where the invite requests were made. (string, optional)

=item * C<cursor> - Value of the `next_cursor` field sent as part of the previous API response (string, optional)

=item * C<limit> - The number of results that will be returned by the API on each invocation. Must be between 1 - 1000, both inclusive (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_invite_requests_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_invite_requests_list',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id cursor limit)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_admins_list

List all of the admins on a given workspace.

L<API method documentation|https://api.slack.com/methods/admin.teams.admins.list>

Takes the following named parameters:

=over 4

=item * C<limit> - The maximum number of items to return. (integer, optional)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page. (string, optional)

=item * C<team_id> -  (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_admins_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_admins_list',
        %args{grep { exists $args{$_} } qw(limit cursor team_id)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_create

Create an Enterprise team.

L<API method documentation|https://api.slack.com/methods/admin.teams.create>

Takes the following named parameters:

=over 4

=item * C<team_domain> - Team domain (for example, slacksoftballteam). (string, required)

=item * C<team_name> - Team name (for example, Slack Softball Team). (string, required)

=item * C<team_description> - Description for the team. (string, optional)

=item * C<team_discoverability> - Who can join the team. A team's discoverability can be `open`, `closed`, `invite_only`, or `unlisted`. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_create {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_create',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_domain team_name team_description team_discoverability)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_list

List all teams on an Enterprise organization

L<API method documentation|https://api.slack.com/methods/admin.teams.list>

Takes the following named parameters:

=over 4

=item * C<limit> - The maximum number of items to return. Must be between 1 - 100 both inclusive. (integer, optional)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_list',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(limit cursor)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_owners_list

List all of the owners on a given workspace.

L<API method documentation|https://api.slack.com/methods/admin.teams.owners.list>

Takes the following named parameters:

=over 4

=item * C<team_id> -  (string, required)

=item * C<limit> - The maximum number of items to return. Must be between 1 - 1000 both inclusive. (integer, optional)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_owners_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_owners_list',
        %args{grep { exists $args{$_} } qw(team_id limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_settings_info

Fetch information about settings in a workspace

L<API method documentation|https://api.slack.com/methods/admin.teams.settings.info>

Takes the following named parameters:

=over 4

=item * C<team_id> -  (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_settings_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_settings_info',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_settings_set_default_channels

Set the default channels of a workspace.

L<API method documentation|https://api.slack.com/methods/admin.teams.settings.setDefaultChannels>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID for the workspace to set the default channel for. (string, required)

=item * C<channel_ids> - An array of channel IDs. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_settings_set_default_channels {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_settings_set_default_channels',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id channel_ids)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_settings_set_description

Set the description of a given workspace.

L<API method documentation|https://api.slack.com/methods/admin.teams.settings.setDescription>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID for the workspace to set the description for. (string, required)

=item * C<description> - The new description for the workspace. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_settings_set_description {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_settings_set_description',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id description)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_settings_set_discoverability

An API method that allows admins to set the discoverability of a given workspace

L<API method documentation|https://api.slack.com/methods/admin.teams.settings.setDiscoverability>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID of the workspace to set discoverability on. (string, required)

=item * C<discoverability> - This workspace's discovery setting. It must be set to one of `open`, `invite_only`, `closed`, or `unlisted`. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_settings_set_discoverability {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_settings_set_discoverability',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id discoverability)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_settings_set_icon

Sets the icon of a workspace.

L<API method documentation|https://api.slack.com/methods/admin.teams.settings.setIcon>

Takes the following named parameters:

=over 4

=item * C<image_url> - Image URL for the icon (string, required)

=item * C<team_id> - ID for the workspace to set the icon for. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_settings_set_icon {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_settings_set_icon',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(image_url team_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_teams_settings_set_name

Set the name of a given workspace.

L<API method documentation|https://api.slack.com/methods/admin.teams.settings.setName>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID for the workspace to set the name for. (string, required)

=item * C<name> - The new name of the workspace. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_teams_settings_set_name {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_teams_settings_set_name',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id name)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_usergroups_add_channels

Add one or more default channels to an IDP group.

L<API method documentation|https://api.slack.com/methods/admin.usergroups.addChannels>

Takes the following named parameters:

=over 4

=item * C<usergroup_id> - ID of the IDP group to add default channels for. (string, required)

=item * C<team_id> - The workspace to add default channels in. (string, optional)

=item * C<channel_ids> - Comma separated string of channel IDs. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_usergroups_add_channels {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_usergroups_add_channels',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(usergroup_id team_id channel_ids)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_usergroups_add_teams

Associate one or more default workspaces with an organization-wide IDP group.

L<API method documentation|https://api.slack.com/methods/admin.usergroups.addTeams>

Takes the following named parameters:

=over 4

=item * C<usergroup_id> - An encoded usergroup (IDP Group) ID. (string, required)

=item * C<team_ids> - A comma separated list of encoded team (workspace) IDs. Each workspace *MUST* belong to the organization associated with the token. (string, required)

=item * C<auto_provision> - When `true`, this method automatically creates new workspace accounts for the IDP group members. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_usergroups_add_teams {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_usergroups_add_teams',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(usergroup_id team_ids auto_provision)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_usergroups_list_channels

List the channels linked to an org-level IDP group (user group).

L<API method documentation|https://api.slack.com/methods/admin.usergroups.listChannels>

Takes the following named parameters:

=over 4

=item * C<usergroup_id> - ID of the IDP group to list default channels for. (string, required)

=item * C<team_id> - ID of the the workspace. (string, optional)

=item * C<include_num_members> - Flag to include or exclude the count of members per channel. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_usergroups_list_channels {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_usergroups_list_channels',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(usergroup_id team_id include_num_members)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_usergroups_remove_channels

Remove one or more default channels from an org-level IDP group (user group).

L<API method documentation|https://api.slack.com/methods/admin.usergroups.removeChannels>

Takes the following named parameters:

=over 4

=item * C<usergroup_id> - ID of the IDP Group (string, required)

=item * C<channel_ids> - Comma-separated string of channel IDs (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_usergroups_remove_channels {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_usergroups_remove_channels',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(usergroup_id channel_ids)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_assign

Add an Enterprise user to a workspace.

L<API method documentation|https://api.slack.com/methods/admin.users.assign>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID (`T1234`) of the workspace. (string, required)

=item * C<user_id> - The ID of the user to add to the workspace. (string, required)

=item * C<is_restricted> - True if user should be added to the workspace as a guest. (boolean, optional)

=item * C<is_ultra_restricted> - True if user should be added to the workspace as a single-channel guest. (boolean, optional)

=item * C<channel_ids> - Comma separated values of channel IDs to add user in the new workspace. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_assign {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_assign',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id user_id is_restricted is_ultra_restricted channel_ids)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_invite

Invite a user to a workspace.

L<API method documentation|https://api.slack.com/methods/admin.users.invite>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID (`T1234`) of the workspace. (string, required)

=item * C<email> - The email address of the person to invite. (string, required)

=item * C<channel_ids> - A comma-separated list of `channel_id`s for this user to join. At least one channel is required. (string, required)

=item * C<custom_message> - An optional message to send to the user in the invite email. (string, optional)

=item * C<real_name> - Full name of the user. (string, optional)

=item * C<resend> - Allow this invite to be resent in the future if a user has not signed up yet. (default: false) (boolean, optional)

=item * C<is_restricted> - Is this user a multi-channel guest user? (default: false) (boolean, optional)

=item * C<is_ultra_restricted> - Is this user a single channel guest user? (default: false) (boolean, optional)

=item * C<guest_expiration_ts> - Timestamp when guest account should be disabled. Only include this timestamp if you are inviting a guest user and you want their account to expire on a certain date. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_invite {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_invite',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id email channel_ids custom_message real_name resend is_restricted is_ultra_restricted guest_expiration_ts)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_list

List users on a workspace

L<API method documentation|https://api.slack.com/methods/admin.users.list>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID (`T1234`) of the workspace. (string, required)

=item * C<cursor> - Set `cursor` to `next_cursor` returned by the previous call to list items in the next page. (string, optional)

=item * C<limit> - Limit for how many users to be retrieved per page (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_list',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id cursor limit)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_remove

Remove a user from a workspace.

L<API method documentation|https://api.slack.com/methods/admin.users.remove>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID (`T1234`) of the workspace. (string, required)

=item * C<user_id> - The ID of the user to remove. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_remove {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_remove',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id user_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_session_invalidate

Invalidate a single session for a user by session_id

L<API method documentation|https://api.slack.com/methods/admin.users.session.invalidate>

Takes the following named parameters:

=over 4

=item * C<team_id> - ID of the team that the session belongs to (string, required)

=item * C<session_id> -  (integer, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_session_invalidate {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_session_invalidate',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id session_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_session_reset

Wipes all valid sessions on all devices for a given user

L<API method documentation|https://api.slack.com/methods/admin.users.session.reset>

Takes the following named parameters:

=over 4

=item * C<user_id> - The ID of the user to wipe sessions for (string, required)

=item * C<mobile_only> - Only expire mobile sessions (default: false) (boolean, optional)

=item * C<web_only> - Only expire web sessions (default: false) (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_session_reset {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_session_reset',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(user_id mobile_only web_only)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_set_admin

Set an existing guest, regular user, or owner to be an admin user.

L<API method documentation|https://api.slack.com/methods/admin.users.setAdmin>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID (`T1234`) of the workspace. (string, required)

=item * C<user_id> - The ID of the user to designate as an admin. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_set_admin {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_set_admin',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id user_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_set_expiration

Set an expiration for a guest user

L<API method documentation|https://api.slack.com/methods/admin.users.setExpiration>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID (`T1234`) of the workspace. (string, required)

=item * C<user_id> - The ID of the user to set an expiration for. (string, required)

=item * C<expiration_ts> - Timestamp when guest account should be disabled. (integer, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_set_expiration {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_set_expiration',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id user_id expiration_ts)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_set_owner

Set an existing guest, regular user, or admin user to be a workspace owner.

L<API method documentation|https://api.slack.com/methods/admin.users.setOwner>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID (`T1234`) of the workspace. (string, required)

=item * C<user_id> - Id of the user to promote to owner. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_set_owner {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_set_owner',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id user_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 admin_users_set_regular

Set an existing guest user, admin user, or owner to be a regular user.

L<API method documentation|https://api.slack.com/methods/admin.users.setRegular>

Takes the following named parameters:

=over 4

=item * C<team_id> - The ID (`T1234`) of the workspace. (string, required)

=item * C<user_id> - The ID of the user to designate as a regular user. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub admin_users_set_regular {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'admin_users_set_regular',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(team_id user_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 api_test

Checks API calling code.

L<API method documentation|https://api.slack.com/methods/api.test>

Takes the following named parameters:

=over 4

=item * C<error> - Error response to return (string, optional)

=item * C<foo> - example property to return (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub api_test {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'api_test',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(error foo)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 apps_event_authorizations_list

Get a list of authorizations for the given event context. Each authorization represents an app installation that the event is visible to.

L<API method documentation|https://api.slack.com/methods/apps.event.authorizations.list>

Takes the following named parameters:

=over 4

=item * C<event_context> -  (string, required)

=item * C<cursor> -  (string, optional)

=item * C<limit> -  (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub apps_event_authorizations_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'apps_event_authorizations_list',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(event_context cursor limit)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 apps_permissions_info

Returns list of permissions this app has on a team.

L<API method documentation|https://api.slack.com/methods/apps.permissions.info>

Resolves to a structure representing the response.

=cut

async sub apps_permissions_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'apps_permissions_info',
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 apps_permissions_request

Allows an app to request additional scopes

L<API method documentation|https://api.slack.com/methods/apps.permissions.request>

Takes the following named parameters:

=over 4

=item * C<scopes> - A comma separated list of scopes to request for (string, required)

=item * C<trigger_id> - Token used to trigger the permissions API (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub apps_permissions_request {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'apps_permissions_request',
        %args{grep { exists $args{$_} } qw(scopes trigger_id)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 apps_permissions_resources_list

Returns list of resource grants this app has on a team.

L<API method documentation|https://api.slack.com/methods/apps.permissions.resources.list>

Takes the following named parameters:

=over 4

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=item * C<limit> - The maximum number of items to return. (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub apps_permissions_resources_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'apps_permissions_resources_list',
        %args{grep { exists $args{$_} } qw(cursor limit)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 apps_permissions_scopes_list

Returns list of scopes this app has on a team.

L<API method documentation|https://api.slack.com/methods/apps.permissions.scopes.list>

Resolves to a structure representing the response.

=cut

async sub apps_permissions_scopes_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'apps_permissions_scopes_list',
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 apps_permissions_users_list

Returns list of user grants and corresponding scopes this app has on a team.

L<API method documentation|https://api.slack.com/methods/apps.permissions.users.list>

Takes the following named parameters:

=over 4

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=item * C<limit> - The maximum number of items to return. (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub apps_permissions_users_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'apps_permissions_users_list',
        %args{grep { exists $args{$_} } qw(cursor limit)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 apps_permissions_users_request

Enables an app to trigger a permissions modal to grant an app access to a user access scope.

L<API method documentation|https://api.slack.com/methods/apps.permissions.users.request>

Takes the following named parameters:

=over 4

=item * C<scopes> - A comma separated list of user scopes to request for (string, required)

=item * C<trigger_id> - Token used to trigger the request (string, required)

=item * C<user> - The user this scope is being requested for (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub apps_permissions_users_request {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'apps_permissions_users_request',
        %args{grep { exists $args{$_} } qw(scopes trigger_id user)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 apps_uninstall

Uninstalls your app from a workspace.

L<API method documentation|https://api.slack.com/methods/apps.uninstall>

Takes the following named parameters:

=over 4

=item * C<client_id> - Issued when you created your application. (string, optional)

=item * C<client_secret> - Issued when you created your application. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub apps_uninstall {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'apps_uninstall',
        %args{grep { exists $args{$_} } qw(client_id client_secret)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 auth_revoke

Revokes a token.

L<API method documentation|https://api.slack.com/methods/auth.revoke>

Takes the following named parameters:

=over 4

=item * C<test> - Setting this parameter to `1` triggers a _testing mode_ where the specified token will not actually be revoked. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub auth_revoke {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'auth_revoke',
        %args{grep { exists $args{$_} } qw(test)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 auth_test

Checks authentication & identity.

L<API method documentation|https://api.slack.com/methods/auth.test>

Resolves to a structure representing the response.

=cut

async sub auth_test {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'auth_test',
    );
    my ($res) = await $self->http_post(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 bots_info

Gets information about a bot user.

L<API method documentation|https://api.slack.com/methods/bots.info>

Takes the following named parameters:

=over 4

=item * C<bot> - Bot user to get info on (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub bots_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'bots_info',
        %args{grep { exists $args{$_} } qw(bot)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 calls_add

Registers a new Call.

L<API method documentation|https://api.slack.com/methods/calls.add>

Takes the following named parameters:

=over 4

=item * C<external_unique_id> - An ID supplied by the 3rd-party Call provider. It must be unique across all Calls from that service. (string, required)

=item * C<external_display_id> - An optional, human-readable ID supplied by the 3rd-party Call provider. If supplied, this ID will be displayed in the Call object. (string, optional)

=item * C<join_url> - The URL required for a client to join the Call. (string, required)

=item * C<desktop_app_join_url> - When supplied, available Slack clients will attempt to directly launch the 3rd-party Call with this URL. (string, optional)

=item * C<date_start> - Call start time in UTC UNIX timestamp format (integer, optional)

=item * C<title> - The name of the Call. (string, optional)

=item * C<created_by> - The valid Slack user ID of the user who created this Call. When this method is called with a user token, the `created_by` field is optional and defaults to the authed user of the token. Otherwise, the field is required. (string, optional)

=item * C<users> - The list of users to register as participants in the Call. [Read more on how to specify users here](/apis/calls#users). (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub calls_add {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'calls_add',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(external_unique_id external_display_id join_url desktop_app_join_url date_start title created_by users)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 calls_end

Ends a Call.

L<API method documentation|https://api.slack.com/methods/calls.end>

Takes the following named parameters:

=over 4

=item * C<id> - `id` returned when registering the call using the [`calls.add`](/methods/calls.add) method. (string, required)

=item * C<duration> - Call duration in seconds (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub calls_end {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'calls_end',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(id duration)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 calls_info

Returns information about a Call.

L<API method documentation|https://api.slack.com/methods/calls.info>

Takes the following named parameters:

=over 4

=item * C<id> - `id` of the Call returned by the [`calls.add`](/methods/calls.add) method. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub calls_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'calls_info',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 calls_participants_add

Registers new participants added to a Call.

L<API method documentation|https://api.slack.com/methods/calls.participants.add>

Takes the following named parameters:

=over 4

=item * C<id> - `id` returned by the [`calls.add`](/methods/calls.add) method. (string, required)

=item * C<users> - The list of users to add as participants in the Call. [Read more on how to specify users here](/apis/calls#users). (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub calls_participants_add {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'calls_participants_add',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(id users)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 calls_participants_remove

Registers participants removed from a Call.

L<API method documentation|https://api.slack.com/methods/calls.participants.remove>

Takes the following named parameters:

=over 4

=item * C<id> - `id` returned by the [`calls.add`](/methods/calls.add) method. (string, required)

=item * C<users> - The list of users to remove as participants in the Call. [Read more on how to specify users here](/apis/calls#users). (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub calls_participants_remove {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'calls_participants_remove',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(id users)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 calls_update

Updates information about a Call.

L<API method documentation|https://api.slack.com/methods/calls.update>

Takes the following named parameters:

=over 4

=item * C<id> - `id` returned by the [`calls.add`](/methods/calls.add) method. (string, required)

=item * C<title> - The name of the Call. (string, optional)

=item * C<join_url> - The URL required for a client to join the Call. (string, optional)

=item * C<desktop_app_join_url> - When supplied, available Slack clients will attempt to directly launch the 3rd-party Call with this URL. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub calls_update {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'calls_update',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(id title join_url desktop_app_join_url)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_delete

Deletes a message.

L<API method documentation|https://api.slack.com/methods/chat.delete>

Takes the following named parameters:

=over 4

=item * C<ts> - Timestamp of the message to be deleted. (number, optional)

=item * C<channel> - Channel containing the message to be deleted. (string, optional)

=item * C<as_user> - Pass true to delete the message as the authed user with `chat:write:user` scope. [Bot users](/bot-users) in this context are considered authed users. If unused or false, the message will be deleted with `chat:write:bot` scope. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub chat_delete {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_delete',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(ts channel as_user)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_delete_scheduled_message

Deletes a pending scheduled message from the queue.

L<API method documentation|https://api.slack.com/methods/chat.deleteScheduledMessage>

Takes the following named parameters:

=over 4

=item * C<as_user> - Pass true to delete the message as the authed user with `chat:write:user` scope. [Bot users](/bot-users) in this context are considered authed users. If unused or false, the message will be deleted with `chat:write:bot` scope. (boolean, optional)

=item * C<channel> - The channel the scheduled_message is posting to (string, required)

=item * C<scheduled_message_id> - `scheduled_message_id` returned from call to chat.scheduleMessage (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub chat_delete_scheduled_message {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_delete_scheduled_message',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(as_user channel scheduled_message_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_get_permalink

Retrieve a permalink URL for a specific extant message

L<API method documentation|https://api.slack.com/methods/chat.getPermalink>

Takes the following named parameters:

=over 4

=item * C<channel> - The ID of the conversation or channel containing the message (string, required)

=item * C<message_ts> - A message's `ts` value, uniquely identifying it within a channel (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub chat_get_permalink {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_get_permalink',
        %args{grep { exists $args{$_} } qw(channel message_ts)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_me_message

Share a me message into a channel.

L<API method documentation|https://api.slack.com/methods/chat.meMessage>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel to send message to. Can be a public channel, private group or IM channel. Can be an encoded ID, or a name. (string, optional)

=item * C<text> - Text of the message to send. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub chat_me_message {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_me_message',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel text)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_post_ephemeral

Sends an ephemeral message to a user in a channel.

L<API method documentation|https://api.slack.com/methods/chat.postEphemeral>

Takes the following named parameters:

=over 4

=item * C<as_user> - Pass true to post the message as the authed user. Defaults to true if the chat:write:bot scope is not included. Otherwise, defaults to false. (boolean, optional)

=item * C<attachments> - A JSON-based array of structured attachments, presented as a URL-encoded string. (string, optional)

=item * C<blocks> - A JSON-based array of structured blocks, presented as a URL-encoded string. (string, optional)

=item * C<channel> - Channel, private group, or IM channel to send message to. Can be an encoded ID, or a name. (string, required)

=item * C<icon_emoji> - Emoji to use as the icon for this message. Overrides `icon_url`. Must be used in conjunction with `as_user` set to `false`, otherwise ignored. See [authorship](#authorship) below. (string, optional)

=item * C<icon_url> - URL to an image to use as the icon for this message. Must be used in conjunction with `as_user` set to false, otherwise ignored. See [authorship](#authorship) below. (string, optional)

=item * C<link_names> - Find and link channel names and usernames. (boolean, optional)

=item * C<parse> - Change how messages are treated. Defaults to `none`. See [below](#formatting). (string, optional)

=item * C<text> - How this field works and whether it is required depends on other fields you use in your API call. [See below](#text_usage) for more detail. (string, optional)

=item * C<thread_ts> - Provide another message's `ts` value to post this message in a thread. Avoid using a reply's `ts` value; use its parent's value instead. Ephemeral messages in threads are only shown if there is already an active thread. (string, optional)

=item * C<user> - `id` of the user who will receive the ephemeral message. The user should be in the channel specified by the `channel` argument. (string, required)

=item * C<username> - Set your bot's user name. Must be used in conjunction with `as_user` set to false, otherwise ignored. See [authorship](#authorship) below. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub chat_post_ephemeral {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_post_ephemeral',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(as_user attachments blocks channel icon_emoji icon_url link_names parse text thread_ts user username)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_post_message

Sends a message to a channel.

L<API method documentation|https://api.slack.com/methods/chat.postMessage>

Takes the following named parameters:

=over 4

=item * C<as_user> - Pass true to post the message as the authed user, instead of as a bot. Defaults to false. See [authorship](#authorship) below. (string, optional)

=item * C<attachments> - A JSON-based array of structured attachments, presented as a URL-encoded string. (string, optional)

=item * C<blocks> - A JSON-based array of structured blocks, presented as a URL-encoded string. (string, optional)

=item * C<channel> - Channel, private group, or IM channel to send message to. Can be an encoded ID, or a name. See [below](#channels) for more details. (string, required)

=item * C<icon_emoji> - Emoji to use as the icon for this message. Overrides `icon_url`. Must be used in conjunction with `as_user` set to `false`, otherwise ignored. See [authorship](#authorship) below. (string, optional)

=item * C<icon_url> - URL to an image to use as the icon for this message. Must be used in conjunction with `as_user` set to false, otherwise ignored. See [authorship](#authorship) below. (string, optional)

=item * C<link_names> - Find and link channel names and usernames. (boolean, optional)

=item * C<mrkdwn> - Disable Slack markup parsing by setting to `false`. Enabled by default. (boolean, optional)

=item * C<parse> - Change how messages are treated. Defaults to `none`. See [below](#formatting). (string, optional)

=item * C<reply_broadcast> - Used in conjunction with `thread_ts` and indicates whether reply should be made visible to everyone in the channel or conversation. Defaults to `false`. (boolean, optional)

=item * C<text> - How this field works and whether it is required depends on other fields you use in your API call. [See below](#text_usage) for more detail. (string, optional)

=item * C<thread_ts> - Provide another message's `ts` value to make this message a reply. Avoid using a reply's `ts` value; use its parent instead. (string, optional)

=item * C<unfurl_links> - Pass true to enable unfurling of primarily text-based content. (boolean, optional)

=item * C<unfurl_media> - Pass false to disable unfurling of media content. (boolean, optional)

=item * C<username> - Set your bot's user name. Must be used in conjunction with `as_user` set to false, otherwise ignored. See [authorship](#authorship) below. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub chat_post_message {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_post_message',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(as_user attachments blocks channel icon_emoji icon_url link_names mrkdwn parse reply_broadcast text thread_ts unfurl_links unfurl_media username)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_schedule_message

Schedules a message to be sent to a channel.

L<API method documentation|https://api.slack.com/methods/chat.scheduleMessage>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel, private group, or DM channel to send message to. Can be an encoded ID, or a name. See [below](#channels) for more details. (string, optional)

=item * C<text> - How this field works and whether it is required depends on other fields you use in your API call. [See below](#text_usage) for more detail. (string, optional)

=item * C<post_at> - Unix EPOCH timestamp of time in future to send the message. (string, optional)

=item * C<parse> - Change how messages are treated. Defaults to `none`. See [chat.postMessage](chat.postMessage#formatting). (string, optional)

=item * C<as_user> - Pass true to post the message as the authed user, instead of as a bot. Defaults to false. See [chat.postMessage](chat.postMessage#authorship). (boolean, optional)

=item * C<link_names> - Find and link channel names and usernames. (boolean, optional)

=item * C<attachments> - A JSON-based array of structured attachments, presented as a URL-encoded string. (string, optional)

=item * C<blocks> - A JSON-based array of structured blocks, presented as a URL-encoded string. (string, optional)

=item * C<unfurl_links> - Pass true to enable unfurling of primarily text-based content. (boolean, optional)

=item * C<unfurl_media> - Pass false to disable unfurling of media content. (boolean, optional)

=item * C<thread_ts> - Provide another message's `ts` value to make this message a reply. Avoid using a reply's `ts` value; use its parent instead. (number, optional)

=item * C<reply_broadcast> - Used in conjunction with `thread_ts` and indicates whether reply should be made visible to everyone in the channel or conversation. Defaults to `false`. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub chat_schedule_message {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_schedule_message',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel text post_at parse as_user link_names attachments blocks unfurl_links unfurl_media thread_ts reply_broadcast)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_scheduled_messages_list

Returns a list of scheduled messages.

L<API method documentation|https://api.slack.com/methods/chat.scheduledMessages.list>

Takes the following named parameters:

=over 4

=item * C<channel> - The channel of the scheduled messages (string, optional)

=item * C<latest> - A UNIX timestamp of the latest value in the time range (number, optional)

=item * C<oldest> - A UNIX timestamp of the oldest value in the time range (number, optional)

=item * C<limit> - Maximum number of original entries to return. (integer, optional)

=item * C<cursor> - For pagination purposes, this is the `cursor` value returned from a previous call to `chat.scheduledmessages.list` indicating where you want to start this call from. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub chat_scheduled_messages_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_scheduled_messages_list',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel latest oldest limit cursor)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_unfurl

Provide custom unfurl behavior for user-posted URLs

L<API method documentation|https://api.slack.com/methods/chat.unfurl>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel ID of the message (string, required)

=item * C<ts> - Timestamp of the message to add unfurl behavior to. (string, required)

=item * C<unfurls> - URL-encoded JSON map with keys set to URLs featured in the the message, pointing to their unfurl blocks or message attachments. (string, optional)

=item * C<user_auth_message> - Provide a simply-formatted string to send as an ephemeral message to the user as invitation to authenticate further and enable full unfurling behavior (string, optional)

=item * C<user_auth_required> - Set to `true` or `1` to indicate the user must install your Slack app to trigger unfurls for this domain (boolean, optional)

=item * C<user_auth_url> - Send users to this custom URL where they will complete authentication in your app to fully trigger unfurling. Value should be properly URL-encoded. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub chat_unfurl {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_unfurl',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel ts unfurls user_auth_message user_auth_required user_auth_url)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 chat_update

Updates a message.

L<API method documentation|https://api.slack.com/methods/chat.update>

Takes the following named parameters:

=over 4

=item * C<as_user> - Pass true to update the message as the authed user. [Bot users](/bot-users) in this context are considered authed users. (string, optional)

=item * C<attachments> - A JSON-based array of structured attachments, presented as a URL-encoded string. This field is required when not presenting `text`. If you don't include this field, the message's previous `attachments` will be retained. To remove previous `attachments`, include an empty array for this field. (string, optional)

=item * C<blocks> - A JSON-based array of [structured blocks](/block-kit/building), presented as a URL-encoded string. If you don't include this field, the message's previous `blocks` will be retained. To remove previous `blocks`, include an empty array for this field. (string, optional)

=item * C<channel> - Channel containing the message to be updated. (string, required)

=item * C<link_names> - Find and link channel names and usernames. Defaults to `none`. If you do not specify a value for this field, the original value set for the message will be overwritten with the default, `none`. (string, optional)

=item * C<parse> - Change how messages are treated. Defaults to `client`, unlike `chat.postMessage`. Accepts either `none` or `full`. If you do not specify a value for this field, the original value set for the message will be overwritten with the default, `client`. (string, optional)

=item * C<text> - New text for the message, using the [default formatting rules](/reference/surfaces/formatting). It's not required when presenting `blocks` or `attachments`. (string, optional)

=item * C<ts> - Timestamp of the message to be updated. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub chat_update {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'chat_update',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(as_user attachments blocks channel link_names parse text ts)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_archive

Archives a conversation.

L<API method documentation|https://api.slack.com/methods/conversations.archive>

Takes the following named parameters:

=over 4

=item * C<channel> - ID of conversation to archive (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_archive {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_archive',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_close

Closes a direct message or multi-person direct message.

L<API method documentation|https://api.slack.com/methods/conversations.close>

Takes the following named parameters:

=over 4

=item * C<channel> - Conversation to close. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_close {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_close',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_create

Initiates a public or private channel-based conversation

L<API method documentation|https://api.slack.com/methods/conversations.create>

Takes the following named parameters:

=over 4

=item * C<name> - Name of the public or private channel to create (string, optional)

=item * C<is_private> - Create a private channel instead of a public one (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_create {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_create',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(name is_private)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_history

Fetches a conversation's history of messages and events.

L<API method documentation|https://api.slack.com/methods/conversations.history>

Takes the following named parameters:

=over 4

=item * C<channel> - Conversation ID to fetch history for. (string, optional)

=item * C<latest> - End of time range of messages to include in results. (number, optional)

=item * C<oldest> - Start of time range of messages to include in results. (number, optional)

=item * C<inclusive> - Include messages with latest or oldest timestamp in results only when either timestamp is specified. (boolean, optional)

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the users list hasn't been reached. (integer, optional)

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_history {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_history',
        %args{grep { exists $args{$_} } qw(channel latest oldest inclusive limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_info

Retrieve information about a conversation.

L<API method documentation|https://api.slack.com/methods/conversations.info>

Takes the following named parameters:

=over 4

=item * C<channel> - Conversation ID to learn more about (string, optional)

=item * C<include_locale> - Set this to `true` to receive the locale for this conversation. Defaults to `false` (boolean, optional)

=item * C<include_num_members> - Set to `true` to include the member count for the specified conversation. Defaults to `false` (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_info',
        %args{grep { exists $args{$_} } qw(channel include_locale include_num_members)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_invite

Invites users to a channel.

L<API method documentation|https://api.slack.com/methods/conversations.invite>

Takes the following named parameters:

=over 4

=item * C<channel> - The ID of the public or private channel to invite user(s) to. (string, optional)

=item * C<users> - A comma separated list of user IDs. Up to 1000 users may be listed. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_invite {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_invite',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel users)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_join

Joins an existing conversation.

L<API method documentation|https://api.slack.com/methods/conversations.join>

Takes the following named parameters:

=over 4

=item * C<channel> - ID of conversation to join (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_join {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_join',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_kick

Removes a user from a conversation.

L<API method documentation|https://api.slack.com/methods/conversations.kick>

Takes the following named parameters:

=over 4

=item * C<channel> - ID of conversation to remove user from. (string, optional)

=item * C<user> - User ID to be removed. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_kick {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_kick',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel user)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_leave

Leaves a conversation.

L<API method documentation|https://api.slack.com/methods/conversations.leave>

Takes the following named parameters:

=over 4

=item * C<channel> - Conversation to leave (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_leave {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_leave',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_list

Lists all channels in a Slack team.

L<API method documentation|https://api.slack.com/methods/conversations.list>

Takes the following named parameters:

=over 4

=item * C<exclude_archived> - Set to `true` to exclude archived channels from the list (boolean, optional)

=item * C<types> - Mix and match channel types by providing a comma-separated list of any combination of `public_channel`, `private_channel`, `mpim`, `im` (string, optional)

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the list hasn't been reached. Must be an integer no larger than 1000. (integer, optional)

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_list',
        %args{grep { exists $args{$_} } qw(exclude_archived types limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_mark

Sets the read cursor in a channel.

L<API method documentation|https://api.slack.com/methods/conversations.mark>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel or conversation to set the read cursor for. (string, optional)

=item * C<ts> - Unique identifier of message you want marked as most recently seen in this conversation. (number, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_mark {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_mark',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel ts)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_members

Retrieve members of a conversation.

L<API method documentation|https://api.slack.com/methods/conversations.members>

Takes the following named parameters:

=over 4

=item * C<channel> - ID of the conversation to retrieve members for (string, optional)

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the users list hasn't been reached. (integer, optional)

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_members {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_members',
        %args{grep { exists $args{$_} } qw(channel limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_open

Opens or resumes a direct message or multi-person direct message.

L<API method documentation|https://api.slack.com/methods/conversations.open>

Takes the following named parameters:

=over 4

=item * C<channel> - Resume a conversation by supplying an `im` or `mpim`'s ID. Or provide the `users` field instead. (string, optional)

=item * C<users> - Comma separated lists of users. If only one user is included, this creates a 1:1 DM.  The ordering of the users is preserved whenever a multi-person direct message is returned. Supply a `channel` when not supplying `users`. (string, optional)

=item * C<return_im> - Boolean, indicates you want the full IM channel definition in the response. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_open {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_open',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel users return_im)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_rename

Renames a conversation.

L<API method documentation|https://api.slack.com/methods/conversations.rename>

Takes the following named parameters:

=over 4

=item * C<channel> - ID of conversation to rename (string, optional)

=item * C<name> - New name for conversation. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_rename {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_rename',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel name)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_replies

Retrieve a thread of messages posted to a conversation

L<API method documentation|https://api.slack.com/methods/conversations.replies>

Takes the following named parameters:

=over 4

=item * C<channel> - Conversation ID to fetch thread from. (string, optional)

=item * C<ts> - Unique identifier of a thread's parent message. `ts` must be the timestamp of an existing message with 0 or more replies. If there are no replies then just the single message referenced by `ts` will return - it is just an ordinary, unthreaded message. (number, optional)

=item * C<latest> - End of time range of messages to include in results. (number, optional)

=item * C<oldest> - Start of time range of messages to include in results. (number, optional)

=item * C<inclusive> - Include messages with latest or oldest timestamp in results only when either timestamp is specified. (boolean, optional)

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the users list hasn't been reached. (integer, optional)

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_replies {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_replies',
        %args{grep { exists $args{$_} } qw(channel ts latest oldest inclusive limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_set_purpose

Sets the purpose for a conversation.

L<API method documentation|https://api.slack.com/methods/conversations.setPurpose>

Takes the following named parameters:

=over 4

=item * C<channel> - Conversation to set the purpose of (string, optional)

=item * C<purpose> - A new, specialer purpose (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_set_purpose {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_set_purpose',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel purpose)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_set_topic

Sets the topic for a conversation.

L<API method documentation|https://api.slack.com/methods/conversations.setTopic>

Takes the following named parameters:

=over 4

=item * C<channel> - Conversation to set the topic of (string, optional)

=item * C<topic> - The new topic string. Does not support formatting or linkification. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_set_topic {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_set_topic',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel topic)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 conversations_unarchive

Reverses conversation archival.

L<API method documentation|https://api.slack.com/methods/conversations.unarchive>

Takes the following named parameters:

=over 4

=item * C<channel> - ID of conversation to unarchive (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub conversations_unarchive {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'conversations_unarchive',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 dialog_open

Open a dialog with a user

L<API method documentation|https://api.slack.com/methods/dialog.open>

Takes the following named parameters:

=over 4

=item * C<dialog> - The dialog definition. This must be a JSON-encoded string. (string, required)

=item * C<trigger_id> - Exchange a trigger to post to the user. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub dialog_open {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'dialog_open',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(dialog trigger_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 dnd_end_dnd

Ends the current user's Do Not Disturb session immediately.

L<API method documentation|https://api.slack.com/methods/dnd.endDnd>

Resolves to a structure representing the response.

=cut

async sub dnd_end_dnd {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'dnd_end_dnd',
    );
    my ($res) = await $self->http_post(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 dnd_end_snooze

Ends the current user's snooze mode immediately.

L<API method documentation|https://api.slack.com/methods/dnd.endSnooze>

Resolves to a structure representing the response.

=cut

async sub dnd_end_snooze {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'dnd_end_snooze',
    );
    my ($res) = await $self->http_post(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 dnd_info

Retrieves a user's current Do Not Disturb status.

L<API method documentation|https://api.slack.com/methods/dnd.info>

Takes the following named parameters:

=over 4

=item * C<user> - User to fetch status for (defaults to current user) (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub dnd_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'dnd_info',
        %args{grep { exists $args{$_} } qw(user)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 dnd_set_snooze

Turns on Do Not Disturb mode for the current user, or changes its duration.

L<API method documentation|https://api.slack.com/methods/dnd.setSnooze>

Takes the following named parameters:

=over 4

=item * C<num_minutes> - Number of minutes, from now, to snooze until. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub dnd_set_snooze {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'dnd_set_snooze',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(num_minutes)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 dnd_team_info

Retrieves the Do Not Disturb status for up to 50 users on a team.

L<API method documentation|https://api.slack.com/methods/dnd.teamInfo>

Takes the following named parameters:

=over 4

=item * C<users> - Comma-separated list of users to fetch Do Not Disturb status for (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub dnd_team_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'dnd_team_info',
        %args{grep { exists $args{$_} } qw(users)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 emoji_list

Lists custom emoji for a team.

L<API method documentation|https://api.slack.com/methods/emoji.list>

Resolves to a structure representing the response.

=cut

async sub emoji_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'emoji_list',
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_comments_delete

Deletes an existing comment on a file.

L<API method documentation|https://api.slack.com/methods/files.comments.delete>

Takes the following named parameters:

=over 4

=item * C<file> - File to delete a comment from. (string, optional)

=item * C<id> - The comment to delete. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_comments_delete {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_comments_delete',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(file id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_delete

Deletes a file.

L<API method documentation|https://api.slack.com/methods/files.delete>

Takes the following named parameters:

=over 4

=item * C<file> - ID of file to delete. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_delete {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_delete',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(file)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_info

Gets information about a file.

L<API method documentation|https://api.slack.com/methods/files.info>

Takes the following named parameters:

=over 4

=item * C<file> - Specify a file by providing its ID. (string, optional)

=item * C<count> -  (string, optional)

=item * C<page> -  (string, optional)

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the list hasn't been reached. (integer, optional)

=item * C<cursor> - Parameter for pagination. File comments are paginated for a single file. Set `cursor` equal to the `next_cursor` attribute returned by the previous request's `response_metadata`. This parameter is optional, but pagination is mandatory: the default value simply fetches the first "page" of the collection of comments. See [pagination](/docs/pagination) for more details. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_info',
        %args{grep { exists $args{$_} } qw(file count page limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_list

List for a team, in a channel, or from a user with applied filters.

L<API method documentation|https://api.slack.com/methods/files.list>

Takes the following named parameters:

=over 4

=item * C<user> - Filter files created by a single user. (string, optional)

=item * C<channel> - Filter files appearing in a specific channel, indicated by its ID. (string, optional)

=item * C<ts_from> - Filter files created after this timestamp (inclusive). (number, optional)

=item * C<ts_to> - Filter files created before this timestamp (inclusive). (number, optional)

=item * C<types> - Filter files by type ([see below](#file_types)). You can pass multiple values in the types argument, like `types=spaces,snippets`.The default value is `all`, which does not filter the list. (string, optional)

=item * C<count> -  (string, optional)

=item * C<page> -  (string, optional)

=item * C<show_files_hidden_by_limit> - Show truncated file info for files hidden due to being too old, and the team who owns the file being over the file limit. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_list',
        %args{grep { exists $args{$_} } qw(user channel ts_from ts_to types count page show_files_hidden_by_limit)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_remote_add

Adds a file from a remote service

L<API method documentation|https://api.slack.com/methods/files.remote.add>

Takes the following named parameters:

=over 4

=item * C<external_id> - Creator defined GUID for the file. (string, optional)

=item * C<title> - Title of the file being shared. (string, optional)

=item * C<filetype> - type of file (string, optional)

=item * C<external_url> - URL of the remote file. (string, optional)

=item * C<preview_image> - Preview of the document via `multipart/form-data`. (string, optional)

=item * C<indexable_file_contents> - A text file (txt, pdf, doc, etc.) containing textual search terms that are used to improve discovery of the remote file. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_remote_add {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_remote_add',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(external_id title filetype external_url preview_image indexable_file_contents)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_remote_info

Retrieve information about a remote file added to Slack

L<API method documentation|https://api.slack.com/methods/files.remote.info>

Takes the following named parameters:

=over 4

=item * C<file> - Specify a file by providing its ID. (string, optional)

=item * C<external_id> - Creator defined GUID for the file. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_remote_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_remote_info',
        %args{grep { exists $args{$_} } qw(file external_id)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_remote_list

Retrieve information about a remote file added to Slack

L<API method documentation|https://api.slack.com/methods/files.remote.list>

Takes the following named parameters:

=over 4

=item * C<channel> - Filter files appearing in a specific channel, indicated by its ID. (string, optional)

=item * C<ts_from> - Filter files created after this timestamp (inclusive). (number, optional)

=item * C<ts_to> - Filter files created before this timestamp (inclusive). (number, optional)

=item * C<limit> - The maximum number of items to return. (integer, optional)

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_remote_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_remote_list',
        %args{grep { exists $args{$_} } qw(channel ts_from ts_to limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_remote_remove

Remove a remote file.

L<API method documentation|https://api.slack.com/methods/files.remote.remove>

Takes the following named parameters:

=over 4

=item * C<file> - Specify a file by providing its ID. (string, optional)

=item * C<external_id> - Creator defined GUID for the file. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_remote_remove {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_remote_remove',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(file external_id)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_remote_share

Share a remote file into a channel.

L<API method documentation|https://api.slack.com/methods/files.remote.share>

Takes the following named parameters:

=over 4

=item * C<file> - Specify a file registered with Slack by providing its ID. Either this field or `external_id` or both are required. (string, optional)

=item * C<external_id> - The globally unique identifier (GUID) for the file, as set by the app registering the file with Slack.  Either this field or `file` or both are required. (string, optional)

=item * C<channels> - Comma-separated list of channel IDs where the file will be shared. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_remote_share {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_remote_share',
        %args{grep { exists $args{$_} } qw(file external_id channels)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_remote_update

Updates an existing remote file.

L<API method documentation|https://api.slack.com/methods/files.remote.update>

Takes the following named parameters:

=over 4

=item * C<file> - Specify a file by providing its ID. (string, optional)

=item * C<external_id> - Creator defined GUID for the file. (string, optional)

=item * C<title> - Title of the file being shared. (string, optional)

=item * C<filetype> - type of file (string, optional)

=item * C<external_url> - URL of the remote file. (string, optional)

=item * C<preview_image> - Preview of the document via `multipart/form-data`. (string, optional)

=item * C<indexable_file_contents> - File containing contents that can be used to improve searchability for the remote file. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_remote_update {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_remote_update',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(file external_id title filetype external_url preview_image indexable_file_contents)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_revoke_public_url

Revokes public/external sharing access for a file

L<API method documentation|https://api.slack.com/methods/files.revokePublicURL>

Takes the following named parameters:

=over 4

=item * C<file> - File to revoke (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_revoke_public_url {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_revoke_public_url',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(file)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 files_shared_public_url

Enables a file for public/external sharing.

L<API method documentation|https://api.slack.com/methods/files.sharedPublicURL>

Takes the following named parameters:

=over 4

=item * C<file> - File to share (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub files_shared_public_url {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'files_shared_public_url',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(file)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}

=head2 migration_exchange

For Enterprise Grid workspaces, map local user IDs to global user IDs

L<API method documentation|https://api.slack.com/methods/migration.exchange>

Takes the following named parameters:

=over 4

=item * C<users> - A comma-separated list of user ids, up to 400 per request (string, required)

=item * C<team_id> - Specify team_id starts with `T` in case of Org Token (string, optional)

=item * C<to_old> - Specify `true` to convert `W` global user IDs to workspace-specific `U` IDs. Defaults to `false`. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub migration_exchange {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'migration_exchange',
        %args{grep { exists $args{$_} } qw(users team_id to_old)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 oauth_access

Exchanges a temporary OAuth verifier code for an access token.

L<API method documentation|https://api.slack.com/methods/oauth.access>

Takes the following named parameters:

=over 4

=item * C<client_id> - Issued when you created your application. (string, optional)

=item * C<client_secret> - Issued when you created your application. (string, optional)

=item * C<code> - The `code` param returned via the OAuth callback. (string, optional)

=item * C<redirect_uri> - This must match the originally submitted URI (if one was sent). (string, optional)

=item * C<single_channel> - Request the user to add your app only to a single channel. Only valid with a [legacy workspace app](https://api.slack.com/legacy-workspace-apps). (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub oauth_access {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'oauth_access',
        %args{grep { exists $args{$_} } qw(client_id client_secret code redirect_uri single_channel)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 oauth_token

Exchanges a temporary OAuth verifier code for a workspace token.

L<API method documentation|https://api.slack.com/methods/oauth.token>

Takes the following named parameters:

=over 4

=item * C<client_id> - Issued when you created your application. (string, optional)

=item * C<client_secret> - Issued when you created your application. (string, optional)

=item * C<code> - The `code` param returned via the OAuth callback. (string, optional)

=item * C<redirect_uri> - This must match the originally submitted URI (if one was sent). (string, optional)

=item * C<single_channel> - Request the user to add your app only to a single channel. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub oauth_token {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'oauth_token',
        %args{grep { exists $args{$_} } qw(client_id client_secret code redirect_uri single_channel)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 oauth_v2_access

Exchanges a temporary OAuth verifier code for an access token.

L<API method documentation|https://api.slack.com/methods/oauth.v2.access>

Takes the following named parameters:

=over 4

=item * C<client_id> - Issued when you created your application. (string, optional)

=item * C<client_secret> - Issued when you created your application. (string, optional)

=item * C<code> - The `code` param returned via the OAuth callback. (string, required)

=item * C<redirect_uri> - This must match the originally submitted URI (if one was sent). (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub oauth_v2_access {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'oauth_v2_access',
        %args{grep { exists $args{$_} } qw(client_id client_secret code redirect_uri)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 pins_add

Pins an item to a channel.

L<API method documentation|https://api.slack.com/methods/pins.add>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel to pin the item in. (string, required)

=item * C<timestamp> - Timestamp of the message to pin. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub pins_add {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'pins_add',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel timestamp)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 pins_list

Lists items pinned to a channel.

L<API method documentation|https://api.slack.com/methods/pins.list>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel to get pinned items for. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub pins_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'pins_list',
        %args{grep { exists $args{$_} } qw(channel)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    return $res;
}


=head2 pins_remove

Un-pins an item from a channel.

L<API method documentation|https://api.slack.com/methods/pins.remove>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel where the item is pinned to. (string, required)

=item * C<timestamp> - Timestamp of the message to un-pin. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub pins_remove {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'pins_remove',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel timestamp)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 reactions_add

Adds a reaction to an item.

L<API method documentation|https://api.slack.com/methods/reactions.add>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel where the message to add reaction to was posted. (string, required)

=item * C<name> - Reaction (emoji) name. (string, required)

=item * C<timestamp> - Timestamp of the message to add reaction to. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub reactions_add {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reactions_add',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel name timestamp)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 reactions_get

Gets reactions for an item.

L<API method documentation|https://api.slack.com/methods/reactions.get>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel where the message to get reactions for was posted. (string, optional)

=item * C<file> - File to get reactions for. (string, optional)

=item * C<file_comment> - File comment to get reactions for. (string, optional)

=item * C<full> - If true always return the complete reaction list. (boolean, optional)

=item * C<timestamp> - Timestamp of the message to get reactions for. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub reactions_get {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reactions_get',
        %args{grep { exists $args{$_} } qw(channel file file_comment full timestamp)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    return $res;
}


=head2 reactions_list

Lists reactions made by a user.

L<API method documentation|https://api.slack.com/methods/reactions.list>

Takes the following named parameters:

=over 4

=item * C<user> - Show reactions made by this user. Defaults to the authed user. (string, optional)

=item * C<full> - If true always return the complete reaction list. (boolean, optional)

=item * C<count> -  (integer, optional)

=item * C<page> -  (integer, optional)

=item * C<cursor> - Parameter for pagination. Set `cursor` equal to the `next_cursor` attribute returned by the previous request's `response_metadata`. This parameter is optional, but pagination is mandatory: the default value simply fetches the first "page" of the collection. See [pagination](/docs/pagination) for more details. (string, optional)

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the list hasn't been reached. (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub reactions_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reactions_list',
        %args{grep { exists $args{$_} } qw(user full count page cursor limit)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 reactions_remove

Removes a reaction from an item.

L<API method documentation|https://api.slack.com/methods/reactions.remove>

Takes the following named parameters:

=over 4

=item * C<name> - Reaction (emoji) name. (string, required)

=item * C<file> - File to remove reaction from. (string, optional)

=item * C<file_comment> - File comment to remove reaction from. (string, optional)

=item * C<channel> - Channel where the message to remove reaction from was posted. (string, optional)

=item * C<timestamp> - Timestamp of the message to remove reaction from. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub reactions_remove {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reactions_remove',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(name file file_comment channel timestamp)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 reminders_add

Creates a reminder.

L<API method documentation|https://api.slack.com/methods/reminders.add>

Takes the following named parameters:

=over 4

=item * C<text> - The content of the reminder (string, required)

=item * C<time> - When this reminder should happen: the Unix timestamp (up to five years from now), the number of seconds until the reminder (if within 24 hours), or a natural language description (Ex. "in 15 minutes," or "every Thursday") (string, required)

=item * C<user> - The user who will receive the reminder. If no user is specified, the reminder will go to user who created it. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub reminders_add {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reminders_add',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(text time user)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 reminders_complete

Marks a reminder as complete.

L<API method documentation|https://api.slack.com/methods/reminders.complete>

Takes the following named parameters:

=over 4

=item * C<reminder> - The ID of the reminder to be marked as complete (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub reminders_complete {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reminders_complete',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(reminder)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 reminders_delete

Deletes a reminder.

L<API method documentation|https://api.slack.com/methods/reminders.delete>

Takes the following named parameters:

=over 4

=item * C<reminder> - The ID of the reminder (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub reminders_delete {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reminders_delete',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(reminder)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 reminders_info

Gets information about a reminder.

L<API method documentation|https://api.slack.com/methods/reminders.info>

Takes the following named parameters:

=over 4

=item * C<reminder> - The ID of the reminder (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub reminders_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reminders_info',
        %args{grep { exists $args{$_} } qw(reminder)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 reminders_list

Lists all reminders created by or for a given user.

L<API method documentation|https://api.slack.com/methods/reminders.list>

Resolves to a structure representing the response.

=cut

async sub reminders_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'reminders_list',
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 rtm_connect

Starts a Real Time Messaging session.

L<API method documentation|https://api.slack.com/methods/rtm.connect>

Takes the following named parameters:

=over 4

=item * C<batch_presence_aware> - Batch presence deliveries via subscription. Enabling changes the shape of `presence_change` events. See [batch presence](/docs/presence-and-status#batching). (boolean, optional)

=item * C<presence_sub> - Only deliver presence events when requested by subscription. See [presence subscriptions](/docs/presence-and-status#subscriptions). (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub rtm_connect {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'rtm_connect',
        %args{grep { exists $args{$_} } qw(batch_presence_aware presence_sub)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 search_messages

Searches for messages matching a query.

L<API method documentation|https://api.slack.com/methods/search.messages>

Takes the following named parameters:

=over 4

=item * C<count> - Pass the number of results you want per "page". Maximum of `100`. (integer, optional)

=item * C<highlight> - Pass a value of `true` to enable query highlight markers (see below). (boolean, optional)

=item * C<page> -  (integer, optional)

=item * C<query> - Search query. (string, required)

=item * C<sort> - Return matches sorted by either `score` or `timestamp`. (string, optional)

=item * C<sort_dir> - Change sort direction to ascending (`asc`) or descending (`desc`). (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub search_messages {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'search_messages',
        %args{grep { exists $args{$_} } qw(count highlight page query sort sort_dir)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 stars_add

Adds a star to an item.

L<API method documentation|https://api.slack.com/methods/stars.add>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel to add star to, or channel where the message to add star to was posted (used with `timestamp`). (string, optional)

=item * C<file> - File to add star to. (string, optional)

=item * C<file_comment> - File comment to add star to. (string, optional)

=item * C<timestamp> - Timestamp of the message to add star to. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub stars_add {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'stars_add',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel file file_comment timestamp)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 stars_list

Lists stars for a user.

L<API method documentation|https://api.slack.com/methods/stars.list>

Takes the following named parameters:

=over 4

=item * C<count> -  (string, optional)

=item * C<page> -  (string, optional)

=item * C<cursor> - Parameter for pagination. Set `cursor` equal to the `next_cursor` attribute returned by the previous request's `response_metadata`. This parameter is optional, but pagination is mandatory: the default value simply fetches the first "page" of the collection. See [pagination](/docs/pagination) for more details. (string, optional)

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the list hasn't been reached. (integer, optional)

=back

Resolves to a structure representing the response.

=cut

async sub stars_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'stars_list',
        %args{grep { exists $args{$_} } qw(count page cursor limit)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 stars_remove

Removes a star from an item.

L<API method documentation|https://api.slack.com/methods/stars.remove>

Takes the following named parameters:

=over 4

=item * C<channel> - Channel to remove star from, or channel where the message to remove star from was posted (used with `timestamp`). (string, optional)

=item * C<file> - File to remove star from. (string, optional)

=item * C<file_comment> - File comment to remove star from. (string, optional)

=item * C<timestamp> - Timestamp of the message to remove star from. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub stars_remove {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'stars_remove',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channel file file_comment timestamp)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 team_access_logs

Gets the access logs for the current team.

L<API method documentation|https://api.slack.com/methods/team.accessLogs>

Takes the following named parameters:

=over 4

=item * C<before> - End of time range of logs to include in results (inclusive). (string, optional)

=item * C<count> -  (string, optional)

=item * C<page> -  (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub team_access_logs {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'team_access_logs',
        %args{grep { exists $args{$_} } qw(before count page)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 team_billable_info

Gets billable users information for the current team.

L<API method documentation|https://api.slack.com/methods/team.billableInfo>

Takes the following named parameters:

=over 4

=item * C<user> - A user to retrieve the billable information for. Defaults to all users. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub team_billable_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'team_billable_info',
        %args{grep { exists $args{$_} } qw(user)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 team_info

Gets information about the current team.

L<API method documentation|https://api.slack.com/methods/team.info>

Takes the following named parameters:

=over 4

=item * C<team> - Team to get info on, if omitted, will return information about the current team. Will only return team that the authenticated token is allowed to see through external shared channels (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub team_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'team_info',
        %args{grep { exists $args{$_} } qw(team)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 team_integration_logs

Gets the integration logs for the current team.

L<API method documentation|https://api.slack.com/methods/team.integrationLogs>

Takes the following named parameters:

=over 4

=item * C<app_id> - Filter logs to this Slack app. Defaults to all logs. (string, optional)

=item * C<change_type> - Filter logs with this change type. Defaults to all logs. (string, optional)

=item * C<count> -  (string, optional)

=item * C<page> -  (string, optional)

=item * C<service_id> - Filter logs to this service. Defaults to all logs. (string, optional)

=item * C<user> - Filter logs generated by this user’s actions. Defaults to all logs. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub team_integration_logs {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'team_integration_logs',
        %args{grep { exists $args{$_} } qw(app_id change_type count page service_id user)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 team_profile_get

Retrieve a team's profile.

L<API method documentation|https://api.slack.com/methods/team.profile.get>

Takes the following named parameters:

=over 4

=item * C<visibility> - Filter by visibility. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub team_profile_get {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'team_profile_get',
        %args{grep { exists $args{$_} } qw(visibility)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 usergroups_create

Create a User Group

L<API method documentation|https://api.slack.com/methods/usergroups.create>

Takes the following named parameters:

=over 4

=item * C<channels> - A comma separated string of encoded channel IDs for which the User Group uses as a default. (string, optional)

=item * C<description> - A short description of the User Group. (string, optional)

=item * C<handle> - A mention handle. Must be unique among channels, users and User Groups. (string, optional)

=item * C<include_count> - Include the number of users in each User Group. (boolean, optional)

=item * C<name> - A name for the User Group. Must be unique among User Groups. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub usergroups_create {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'usergroups_create',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(channels description handle include_count name)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 usergroups_disable

Disable an existing User Group

L<API method documentation|https://api.slack.com/methods/usergroups.disable>

Takes the following named parameters:

=over 4

=item * C<include_count> - Include the number of users in the User Group. (boolean, optional)

=item * C<usergroup> - The encoded ID of the User Group to disable. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub usergroups_disable {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'usergroups_disable',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(include_count usergroup)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 usergroups_enable

Enable a User Group

L<API method documentation|https://api.slack.com/methods/usergroups.enable>

Takes the following named parameters:

=over 4

=item * C<include_count> - Include the number of users in the User Group. (boolean, optional)

=item * C<usergroup> - The encoded ID of the User Group to enable. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub usergroups_enable {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'usergroups_enable',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(include_count usergroup)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 usergroups_list

List all User Groups for a team

L<API method documentation|https://api.slack.com/methods/usergroups.list>

Takes the following named parameters:

=over 4

=item * C<include_users> - Include the list of users for each User Group. (boolean, optional)

=item * C<include_count> - Include the number of users in each User Group. (boolean, optional)

=item * C<include_disabled> - Include disabled User Groups. (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub usergroups_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'usergroups_list',
        %args{grep { exists $args{$_} } qw(include_users include_count include_disabled)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 usergroups_update

Update an existing User Group

L<API method documentation|https://api.slack.com/methods/usergroups.update>

Takes the following named parameters:

=over 4

=item * C<handle> - A mention handle. Must be unique among channels, users and User Groups. (string, optional)

=item * C<description> - A short description of the User Group. (string, optional)

=item * C<channels> - A comma separated string of encoded channel IDs for which the User Group uses as a default. (string, optional)

=item * C<include_count> - Include the number of users in the User Group. (boolean, optional)

=item * C<usergroup> - The encoded ID of the User Group to update. (string, required)

=item * C<name> - A name for the User Group. Must be unique among User Groups. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub usergroups_update {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'usergroups_update',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(handle description channels include_count usergroup name)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 usergroups_users_list

List all users in a User Group

L<API method documentation|https://api.slack.com/methods/usergroups.users.list>

Takes the following named parameters:

=over 4

=item * C<include_disabled> - Allow results that involve disabled User Groups. (boolean, optional)

=item * C<usergroup> - The encoded ID of the User Group to update. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub usergroups_users_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'usergroups_users_list',
        %args{grep { exists $args{$_} } qw(include_disabled usergroup)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 usergroups_users_update

Update the list of users for a User Group

L<API method documentation|https://api.slack.com/methods/usergroups.users.update>

Takes the following named parameters:

=over 4

=item * C<include_count> - Include the number of users in the User Group. (boolean, optional)

=item * C<usergroup> - The encoded ID of the User Group to update. (string, required)

=item * C<users> - A comma separated string of encoded user IDs that represent the entire list of users for the User Group. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub usergroups_users_update {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'usergroups_users_update',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(include_count usergroup users)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_conversations

List conversations the calling user may access.

L<API method documentation|https://api.slack.com/methods/users.conversations>

Takes the following named parameters:

=over 4

=item * C<user> - Browse conversations by a specific user ID's membership. Non-public channels are restricted to those where the calling user shares membership. (string, optional)

=item * C<types> - Mix and match channel types by providing a comma-separated list of any combination of `public_channel`, `private_channel`, `mpim`, `im` (string, optional)

=item * C<exclude_archived> - Set to `true` to exclude archived channels from the list (boolean, optional)

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the list hasn't been reached. Must be an integer no larger than 1000. (integer, optional)

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub users_conversations {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_conversations',
        %args{grep { exists $args{$_} } qw(user types exclude_archived limit cursor)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_delete_photo

Delete the user profile photo

L<API method documentation|https://api.slack.com/methods/users.deletePhoto>

Resolves to a structure representing the response.

=cut

async sub users_delete_photo {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_delete_photo',
    );
    my ($res) = await $self->http_post(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_get_presence

Gets user presence information.

L<API method documentation|https://api.slack.com/methods/users.getPresence>

Takes the following named parameters:

=over 4

=item * C<user> - User to get presence info on. Defaults to the authed user. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub users_get_presence {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_get_presence',
        %args{grep { exists $args{$_} } qw(user)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_identity

Get a user's identity.

L<API method documentation|https://api.slack.com/methods/users.identity>

Resolves to a structure representing the response.

=cut

async sub users_identity {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_identity',
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    return $res;
}


=head2 users_info

Gets information about a user.

L<API method documentation|https://api.slack.com/methods/users.info>

Takes the following named parameters:

=over 4

=item * C<include_locale> - Set this to `true` to receive the locale for this user. Defaults to `false` (boolean, optional)

=item * C<user> - User to get info on (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub users_info {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_info',
        %args{grep { exists $args{$_} } qw(include_locale user)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_list

Lists all users in a Slack team.

L<API method documentation|https://api.slack.com/methods/users.list>

Takes the following named parameters:

=over 4

=item * C<limit> - The maximum number of items to return. Fewer than the requested number of items may be returned, even if the end of the users list hasn't been reached. Providing no `limit` value will result in Slack attempting to deliver you the entire result set. If the collection is too large you may experience `limit_required` or HTTP 500 errors. (integer, optional)

=item * C<cursor> - Paginate through collections of data by setting the `cursor` parameter to a `next_cursor` attribute returned by a previous request's `response_metadata`. Default value fetches the first "page" of the collection. See [pagination](/docs/pagination) for more detail. (string, optional)

=item * C<include_locale> - Set this to `true` to receive the locale for users. Defaults to `false` (boolean, optional)

=back

Resolves to a structure representing the response.

=cut

async sub users_list {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_list',
        %args{grep { exists $args{$_} } qw(limit cursor include_locale)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_lookup_by_email

Find a user with an email address.

L<API method documentation|https://api.slack.com/methods/users.lookupByEmail>

Takes the following named parameters:

=over 4

=item * C<email> - An email address belonging to a user in the workspace (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub users_lookup_by_email {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_lookup_by_email',
        %args{grep { exists $args{$_} } qw(email)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_profile_get

Retrieves a user's profile information.

L<API method documentation|https://api.slack.com/methods/users.profile.get>

Takes the following named parameters:

=over 4

=item * C<include_labels> - Include labels for each ID in custom profile fields (boolean, optional)

=item * C<user> - User to retrieve profile info for (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub users_profile_get {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_profile_get',
        %args{grep { exists $args{$_} } qw(include_labels user)}
    );
    my ($res) = await $self->http_get(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_profile_set

Set the profile information for a user.

L<API method documentation|https://api.slack.com/methods/users.profile.set>

Takes the following named parameters:

=over 4

=item * C<name> - Name of a single key to set. Usable only if `profile` is not passed. (string, optional)

=item * C<profile> - Collection of key:value pairs presented as a URL-encoded JSON hash. At most 50 fields may be set. Each field name is limited to 255 characters. (string, optional)

=item * C<user> - ID of user to change. This argument may only be specified by team admins on paid teams. (string, optional)

=item * C<value> - Value to set a single key to. Usable only if `profile` is not passed. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub users_profile_set {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_profile_set',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(name profile user value)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_set_active

Marked a user as active. Deprecated and non-functional.

L<API method documentation|https://api.slack.com/methods/users.setActive>

Resolves to a structure representing the response.

=cut

async sub users_set_active {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_set_active',
    );
    my ($res) = await $self->http_post(
        $uri,
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_set_photo

Set the user profile photo

L<API method documentation|https://api.slack.com/methods/users.setPhoto>

Takes the following named parameters:

=over 4

=item * C<crop_w> - Width/height of crop box (always square) (string, optional)

=item * C<crop_x> - X coordinate of top-left corner of crop box (string, optional)

=item * C<crop_y> - Y coordinate of top-left corner of crop box (string, optional)

=item * C<image> - File contents via `multipart/form-data`. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub users_set_photo {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_set_photo',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(crop_w crop_x crop_y image)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 users_set_presence

Manually sets user presence.

L<API method documentation|https://api.slack.com/methods/users.setPresence>

Takes the following named parameters:

=over 4

=item * C<presence> - Either `auto` or `away` (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub users_set_presence {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'users_set_presence',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(presence)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 views_open

Open a view for a user.

L<API method documentation|https://api.slack.com/methods/views.open>

Takes the following named parameters:

=over 4

=item * C<trigger_id> - Exchange a trigger to post to the user. (string, required)

=item * C<view> - A [view payload](/reference/surfaces/views). This must be a JSON-encoded string. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub views_open {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'views_open',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(trigger_id view)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 views_publish

Publish a static view for a User.

L<API method documentation|https://api.slack.com/methods/views.publish>

Takes the following named parameters:

=over 4

=item * C<user_id> - `id` of the user you want publish a view to. (string, required)

=item * C<view> - A [view payload](/reference/surfaces/views). This must be a JSON-encoded string. (string, required)

=item * C<hash> - A string that represents view state to protect against possible race conditions. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub views_publish {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'views_publish',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(user_id view hash)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 views_push

Push a view onto the stack of a root view.

L<API method documentation|https://api.slack.com/methods/views.push>

Takes the following named parameters:

=over 4

=item * C<trigger_id> - Exchange a trigger to post to the user. (string, required)

=item * C<view> - A [view payload](/reference/surfaces/views). This must be a JSON-encoded string. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub views_push {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'views_push',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(trigger_id view)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 views_update

Update an existing view.

L<API method documentation|https://api.slack.com/methods/views.update>

Takes the following named parameters:

=over 4

=item * C<view_id> - A unique identifier of the view to be updated. Either `view_id` or `external_id` is required. (string, optional)

=item * C<external_id> - A unique identifier of the view set by the developer. Must be unique for all views on a team. Max length of 255 characters. Either `view_id` or `external_id` is required. (string, optional)

=item * C<view> - A [view object](/reference/surfaces/views). This must be a JSON-encoded string. (string, optional)

=item * C<hash> - A string that represents view state to protect against possible race conditions. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub views_update {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'views_update',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(view_id external_id view hash)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 workflows_step_completed

Indicate that an app's step in a workflow completed execution.

L<API method documentation|https://api.slack.com/methods/workflows.stepCompleted>

Takes the following named parameters:

=over 4

=item * C<workflow_step_execute_id> - Context identifier that maps to the correct workflow step execution. (string, required)

=item * C<outputs> - Key-value object of outputs from your step. Keys of this object reflect the configured `key` properties of your [`outputs`](/reference/workflows/workflow_step#output) array from your `workflow_step` object. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub workflows_step_completed {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'workflows_step_completed',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(workflow_step_execute_id outputs)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 workflows_step_failed

Indicate that an app's step in a workflow failed to execute.

L<API method documentation|https://api.slack.com/methods/workflows.stepFailed>

Takes the following named parameters:

=over 4

=item * C<workflow_step_execute_id> - Context identifier that maps to the correct workflow step execution. (string, required)

=item * C<error> - A JSON-based object with a `message` property that should contain a human readable error message. (string, required)

=back

Resolves to a structure representing the response.

=cut

async sub workflows_step_failed {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'workflows_step_failed',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(workflow_step_execute_id error)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}


=head2 workflows_update_step

Update the configuration for a workflow extension step.

L<API method documentation|https://api.slack.com/methods/workflows.updateStep>

Takes the following named parameters:

=over 4

=item * C<workflow_step_edit_id> - A context identifier provided with `view_submission` payloads used to call back to `workflows.updateStep`. (string, required)

=item * C<inputs> - A JSON key-value map of inputs required from a user during configuration. This is the data your app expects to receive when the workflow step starts. **Please note**: the embedded variable format is set and replaced by the workflow system. You cannot create custom variables that will be replaced at runtime. [Read more about variables in workflow steps here](/workflows/steps#variables). (string, optional)

=item * C<outputs> - An JSON array of output objects used during step execution. This is the data your app agrees to provide when your workflow step was executed. (string, optional)

=item * C<step_name> - An optional field that can be used to override the step name that is shown in the Workflow Builder. (string, optional)

=item * C<step_image_url> - An optional field that can be used to override app image that is shown in the Workflow Builder. (string, optional)

=back

Resolves to a structure representing the response.

=cut

async sub workflows_update_step {
    my ($self, %args) = @_;
    my $uri = $self->endpoint(
        'workflows_update_step',
    );
    my $content = encode_json_utf8({
        %args{grep { exists $args{$_} } qw(workflow_step_edit_id inputs outputs step_name step_image_url)}
    });
    my ($res) = await $self->http_post(
        $uri,
        $content,
        content_type => 'application/json; charset=utf-8',
    );
    die $res unless $res->{ok};
    return $res;
}

1;
