package OpenInteract2::Brick::WhatsNew;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'whats_new-2.12.zip' => 'WHATS_NEW212ZIP',
);

sub get_name {
    return 'whats_new';
}

sub get_resources {
    return (
        'whats_new-2.12.zip' => [ 'pkg whats_new-2.12.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::WhatsNew - Base-64 encoded OI2 package 'whats_new-2.12.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "whats_new-2.12.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'whats_new' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<whats_new-2.12.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub WHATS_NEW212ZIP {
    return <<'SOMELONGSTRING';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SOMELONGSTRING
}

