package Workflow::Inotify::Handler;

use strict;
use warnings;

use parent qw(Class::Accessor::Fast);

__PACKAGE__->follow_best_practice;
__PACKAGE__->mk_accessors(qw(config));

our $VERSION = '1.0.2'; ## no critic (RequireInterpolationOfMetachars)

########################################################################
sub new {
########################################################################
  my ( $class, $config ) = @_;

  $class = ref($class) || $class;

  return $class->SUPER::new( { config => $config } );
}

########################################################################
sub handler {
########################################################################
  my ( $self, $event ) = @_;

  return print {*STDERR} sprintf "event: %s file: %s\n", $event->mask,
    $event->fullname;
}

1;

## no critic (RequirePodSections)

__END__

=pod

=head1 NAME

Workflow::Inotify::Handler - base class for creating
L<Linux::Inotify2> handlers

=head1 SYNOPSIS

 package MyHandler;

 use parent qw(Workflow::Inotify::Handler);

 sub handler {
   my ($self, $e) = @_;

   printf("event: %s name: %s\n", $e->mask, $e->fullname); 
 }

 1;

=head1 DESCRIPTION

Base class for creating C<Linux::Inotify2> event handlers.  You use
this base class to implement a handler that responds to events
generated by I<inotify> events. Your event handlers can do pretty much
anything they want, including C<fork()> a new process.  In general,
you want your handlers to be fast and lightweight.

A typical implementation will unload, possibly filter or interpret,
then queue the event for another process to handle.  This technique
encourages a high degree of decoupling of your architecture.

=head1 METHODS

=head2 new

 new( config )

The class is instantiated by the C<inotify.pl> script and is passed a
L<Config::IniFiles> object. Override the C<handler()> or the C<new()>
method if you choose.

I<HINT>: Add sections to the configuration file for your application
and access their values using the config object passed in the
constructor.

 my $user_name = $self->get_config()->val(application => 'user_name');

=head1 INOTIFY EVENTS

I<....from C<man 7 inotify>...>

The inotify_add_watch(2) mask argument and the mask
field of the inotify_event structure returned when read(2)ing an
inotify file descriptor are both bit masks identifying inotify events.
The following bits can be specified in mask when calling
inotify_add_watch(2) and may be returned in the mask field returned by
read(2):

=over 

=item * IN_ACCESS (+)

File was accessed (e.g., read(2), execve(2)).

=item * IN_ATTRIB (*)

Metadata changed for example, permissions (e.g., chmod(2)), timestamps
(e.g., utimensat(2)), extended attributes (setxattr(2)), link count
(since Linux 2.6.25; e.g., for the target of link(2) and for
unlink(2)), and user/group ID (e.g., chown(2)).

=item * IN_CLOSE_WRITE (+)

File opened for writing was closed.

=item * IN_CLOSE_NOWRITE (*)

File or directory not opened for writing was closed.

=item * IN_CREATE (+)

File/directory created in watched directory (e.g., open(2) O_CREAT,
mkdir(2), link(2), symlink(2), bind(2) on a UNIX domain socket).

=item * IN_DELETE (+)

File/directory deleted from watched directory.

=item * IN_DELETE_SELF

Watched file/directory was itself deleted.  (This event also occurs if
an object is moved to another filesystem, since mv(1) in effect copies
the file to the other filesystem and then deletes it from the original
filesystem.)  In addition, an=item * IN_IGNORED event will
subsequently begenerated for the watch descriptor.

=item * IN_MODIFY (+)

File was modified (e.g., write(2), truncate(2)).

=item * IN_MOVE_SELF

Watched file/directory was itself moved.

=item * IN_MOVED_FROM (+)

Generated for the directory containing the old filename when a file is renamed.

=item * IN_MOVED_TO (+)

Generated for the directory containing the new filename when a file is renamed.

=item * IN_OPEN (*)

File or directory was opened.

=back

When monitoring a directory:

=over

=item * the events marked above with an asterisk (*) can occur both
for the directory itself and for objects inside the directory; and

=item * the events marked with a plus sign (+) occur only for objects
inside the directory (not for the directory itself).

=back

=head2 handler

 handler( event )

An instance of L<Linux::Inotify::Event>.  See L<Linux::Inotify2>

=back

=head2 config

The method of your class that will handle the event.

=over 5

=item event

An instance of L<Linux::Inotify::Event>.  See L<Linux::Inotify2>

=back

=head2 get_config

 get_config()

Returns a L<Config::IniFiles> object initialized from your
configuration file.

=head1 AUTHOR

Rob Lauer - <rlauer6@comcast.net>

=head1 SEE ALSO

L</Linux::Inotify2>

=cut
