/* Copyright 2013-2016 Freescale Semiconductor Inc.
 * Copyright 2016 NXP
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * * Neither the name of the above-listed copyright holders nor the
 * names of any contributors may be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 *
 * ALTERNATIVELY, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") as published by the Free Software
 * Foundation, either version 2 of that License or (at your option) any
 * later version.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef __FSL_DPNI_H
#define __FSL_DPNI_H

#include "dpkg.h"

struct fsl_mc_io;

/**
 * Data Path Network Interface API
 * Contains initialization APIs and runtime control APIs for DPNI
 */

/** General DPNI macros */

/**
 * Maximum number of traffic classes
 */
#define DPNI_MAX_TC				8
/**
 * Maximum number of buffer pools per DPNI
 */
#define DPNI_MAX_DPBP				8

/**
 * All traffic classes considered; see dpni_set_queue()
 */
#define DPNI_ALL_TCS				(u8)(-1)
/**
 * All flows within traffic class considered; see dpni_set_queue()
 */
#define DPNI_ALL_TC_FLOWS			(u16)(-1)
/**
 * Generate new flow ID; see dpni_set_queue()
 */
#define DPNI_NEW_FLOW_ID			(u16)(-1)

/**
 * Tx traffic is always released to a buffer pool on transmit, there are no
 * resources allocated to have the frames confirmed back to the source after
 * transmission.
 */
#define DPNI_OPT_TX_FRM_RELEASE			0x000001
/**
 * Disables support for MAC address filtering for addresses other than primary
 * MAC address. This affects both unicast and multicast. Promiscuous mode can
 * still be enabled/disabled for both unicast and multicast. If promiscuous mode
 * is disabled, only traffic matching the primary MAC address will be accepted.
 */
#define DPNI_OPT_NO_MAC_FILTER			0x000002
/**
 * Allocate policers for this DPNI. They can be used to rate-limit traffic per
 * traffic class (TC) basis.
 */
#define DPNI_OPT_HAS_POLICING			0x000004
/**
 * Congestion can be managed in several ways, allowing the buffer pool to
 * deplete on ingress, taildrop on each queue or use congestion groups for sets
 * of queues. If set, it configures a single congestion groups across all TCs.
 * If reset, a congestion group is allocated for each TC. Only relevant if the
 * DPNI has multiple traffic classes.
 */
#define DPNI_OPT_SHARED_CONGESTION		0x000008
/**
 * Enables TCAM for Flow Steering and QoS look-ups. If not specified, all
 * look-ups are exact match. Note that TCAM is not available on LS1088 and its
 * variants. Setting this bit on these SoCs will trigger an error.
 */
#define DPNI_OPT_HAS_KEY_MASKING		0x000010
/**
 * Disables the flow steering table.
 */
#define DPNI_OPT_NO_FS				0x000020

int dpni_open(struct fsl_mc_io	*mc_io,
	      u32		cmd_flags,
	      int		dpni_id,
	      u16		*token);

int dpni_close(struct fsl_mc_io	*mc_io,
	       u32		cmd_flags,
	       u16		token);

/**
 * struct dpni_pools_cfg - Structure representing buffer pools configuration
 * @num_dpbp: Number of DPBPs
 * @pools: Array of buffer pools parameters; The number of valid entries
 *	must match 'num_dpbp' value
 */
struct dpni_pools_cfg {
	u8		num_dpbp;
	/**
	 * struct pools - Buffer pools parameters
	 * @dpbp_id: DPBP object ID
	 * @buffer_size: Buffer size
	 * @backup_pool: Backup pool
	 */
	struct {
		int	dpbp_id;
		u16	buffer_size;
		int	backup_pool;
	} pools[DPNI_MAX_DPBP];
};

int dpni_set_pools(struct fsl_mc_io		*mc_io,
		   u32				cmd_flags,
		   u16				token,
		   const struct dpni_pools_cfg	*cfg);

int dpni_enable(struct fsl_mc_io	*mc_io,
		u32			cmd_flags,
		u16			token);

int dpni_disable(struct fsl_mc_io	*mc_io,
		 u32			cmd_flags,
		 u16			token);

int dpni_is_enabled(struct fsl_mc_io	*mc_io,
		    u32			cmd_flags,
		    u16			token,
		    int			*en);

int dpni_reset(struct fsl_mc_io	*mc_io,
	       u32		cmd_flags,
	       u16		token);

/**
 * DPNI IRQ Index and Events
 */

/**
 * IRQ index
 */
#define DPNI_IRQ_INDEX				0
/**
 * IRQ event - indicates a change in link state
 */
#define DPNI_IRQ_EVENT_LINK_CHANGED		0x00000001

int dpni_set_irq_enable(struct fsl_mc_io	*mc_io,
			u32			cmd_flags,
			u16			token,
			u8			irq_index,
			u8			en);

int dpni_get_irq_enable(struct fsl_mc_io	*mc_io,
			u32			cmd_flags,
			u16			token,
			u8			irq_index,
			u8			*en);

int dpni_set_irq_mask(struct fsl_mc_io	*mc_io,
		      u32		cmd_flags,
		      u16		token,
		      u8		irq_index,
		      u32		mask);

int dpni_get_irq_mask(struct fsl_mc_io	*mc_io,
		      u32		cmd_flags,
		      u16		token,
		      u8		irq_index,
		      u32		*mask);

int dpni_get_irq_status(struct fsl_mc_io	*mc_io,
			u32			cmd_flags,
			u16			token,
			u8			irq_index,
			u32			*status);

int dpni_clear_irq_status(struct fsl_mc_io	*mc_io,
			  u32			cmd_flags,
			  u16			token,
			  u8			irq_index,
			  u32			status);

/**
 * struct dpni_attr - Structure representing DPNI attributes
 * @options: Any combination of the following options:
 *		DPNI_OPT_TX_FRM_RELEASE
 *		DPNI_OPT_NO_MAC_FILTER
 *		DPNI_OPT_HAS_POLICING
 *		DPNI_OPT_SHARED_CONGESTION
 *		DPNI_OPT_HAS_KEY_MASKING
 *		DPNI_OPT_NO_FS
 * @num_queues: Number of Tx and Rx queues used for traffic distribution.
 * @num_tcs: Number of traffic classes (TCs), reserved for the DPNI.
 * @mac_filter_entries: Number of entries in the MAC address filtering table.
 * @vlan_filter_entries: Number of entries in the VLAN address filtering table.
 * @qos_entries: Number of entries in the QoS classification table.
 * @fs_entries: Number of entries in the flow steering table.
 * @qos_key_size: Size, in bytes, of the QoS look-up key. Defining a key larger
 *		than this when adding QoS entries will result in an error.
 * @fs_key_size: Size, in bytes, of the flow steering look-up key. Defining a
 *		key larger than this when composing the hash + FS key will
 *		result in an error.
 * @wriop_version: Version of WRIOP HW block. The 3 version values are stored
 *		on 6, 5, 5 bits respectively.
 */
struct dpni_attr {
	u32 options;
	u8 num_queues;
	u8 num_tcs;
	u8 mac_filter_entries;
	u8 vlan_filter_entries;
	u8 qos_entries;
	u16 fs_entries;
	u8 qos_key_size;
	u8 fs_key_size;
	u16 wriop_version;
};

int dpni_get_attributes(struct fsl_mc_io	*mc_io,
			u32			cmd_flags,
			u16			token,
			struct dpni_attr	*attr);

/**
 * DPNI errors
 */

/**
 * Extract out of frame header error
 */
#define DPNI_ERROR_EOFHE	0x00020000
/**
 * Frame length error
 */
#define DPNI_ERROR_FLE		0x00002000
/**
 * Frame physical error
 */
#define DPNI_ERROR_FPE		0x00001000
/**
 * Parsing header error
 */
#define DPNI_ERROR_PHE		0x00000020
/**
 * Parser L3 checksum error
 */
#define DPNI_ERROR_L3CE		0x00000004
/**
 * Parser L3 checksum error
 */
#define DPNI_ERROR_L4CE		0x00000001

/**
 * enum dpni_error_action - Defines DPNI behavior for errors
 * @DPNI_ERROR_ACTION_DISCARD: Discard the frame
 * @DPNI_ERROR_ACTION_CONTINUE: Continue with the normal flow
 * @DPNI_ERROR_ACTION_SEND_TO_ERROR_QUEUE: Send the frame to the error queue
 */
enum dpni_error_action {
	DPNI_ERROR_ACTION_DISCARD = 0,
	DPNI_ERROR_ACTION_CONTINUE = 1,
	DPNI_ERROR_ACTION_SEND_TO_ERROR_QUEUE = 2
};

/**
 * struct dpni_error_cfg - Structure representing DPNI errors treatment
 * @errors: Errors mask; use 'DPNI_ERROR__<X>
 * @error_action: The desired action for the errors mask
 * @set_frame_annotation: Set to '1' to mark the errors in frame annotation
 *		status (FAS); relevant only for the non-discard action
 */
struct dpni_error_cfg {
	u32			errors;
	enum dpni_error_action	error_action;
	int			set_frame_annotation;
};

int dpni_set_errors_behavior(struct fsl_mc_io		*mc_io,
			     u32			cmd_flags,
			     u16			token,
			     struct dpni_error_cfg	*cfg);

/**
 * DPNI buffer layout modification options
 */

/**
 * Select to modify the time-stamp setting
 */
#define DPNI_BUF_LAYOUT_OPT_TIMESTAMP		0x00000001
/**
 * Select to modify the parser-result setting; not applicable for Tx
 */
#define DPNI_BUF_LAYOUT_OPT_PARSER_RESULT	0x00000002
/**
 * Select to modify the frame-status setting
 */
#define DPNI_BUF_LAYOUT_OPT_FRAME_STATUS	0x00000004
/**
 * Select to modify the private-data-size setting
 */
#define DPNI_BUF_LAYOUT_OPT_PRIVATE_DATA_SIZE	0x00000008
/**
 * Select to modify the data-alignment setting
 */
#define DPNI_BUF_LAYOUT_OPT_DATA_ALIGN		0x00000010
/**
 * Select to modify the data-head-room setting
 */
#define DPNI_BUF_LAYOUT_OPT_DATA_HEAD_ROOM	0x00000020
/**
 * Select to modify the data-tail-room setting
 */
#define DPNI_BUF_LAYOUT_OPT_DATA_TAIL_ROOM	0x00000040

/**
 * struct dpni_buffer_layout - Structure representing DPNI buffer layout
 * @options: Flags representing the suggested modifications to the buffer
 *		layout; Use any combination of 'DPNI_BUF_LAYOUT_OPT_<X>' flags
 * @pass_timestamp: Pass timestamp value
 * @pass_parser_result: Pass parser results
 * @pass_frame_status: Pass frame status
 * @private_data_size: Size kept for private data (in bytes)
 * @data_align: Data alignment
 * @data_head_room: Data head room
 * @data_tail_room: Data tail room
 */
struct dpni_buffer_layout {
	u32	options;
	int	pass_timestamp;
	int	pass_parser_result;
	int	pass_frame_status;
	u16	private_data_size;
	u16	data_align;
	u16	data_head_room;
	u16	data_tail_room;
};

/**
 * enum dpni_queue_type - Identifies a type of queue targeted by the command
 * @DPNI_QUEUE_RX: Rx queue
 * @DPNI_QUEUE_TX: Tx queue
 * @DPNI_QUEUE_TX_CONFIRM: Tx confirmation queue
 * @DPNI_QUEUE_RX_ERR: Rx error queue
 */enum dpni_queue_type {
	DPNI_QUEUE_RX,
	DPNI_QUEUE_TX,
	DPNI_QUEUE_TX_CONFIRM,
	DPNI_QUEUE_RX_ERR,
};

int dpni_get_buffer_layout(struct fsl_mc_io		*mc_io,
			   u32				cmd_flags,
			   u16				token,
			   enum dpni_queue_type		qtype,
			   struct dpni_buffer_layout	*layout);

int dpni_set_buffer_layout(struct fsl_mc_io		   *mc_io,
			   u32				   cmd_flags,
			   u16				   token,
			   enum dpni_queue_type		   qtype,
			   const struct dpni_buffer_layout *layout);

/**
 * enum dpni_offload - Identifies a type of offload targeted by the command
 * @DPNI_OFF_RX_L3_CSUM: Rx L3 checksum validation
 * @DPNI_OFF_RX_L4_CSUM: Rx L4 checksum validation
 * @DPNI_OFF_TX_L3_CSUM: Tx L3 checksum generation
 * @DPNI_OFF_TX_L4_CSUM: Tx L4 checksum generation
 */
enum dpni_offload {
	DPNI_OFF_RX_L3_CSUM,
	DPNI_OFF_RX_L4_CSUM,
	DPNI_OFF_TX_L3_CSUM,
	DPNI_OFF_TX_L4_CSUM,
};

int dpni_set_offload(struct fsl_mc_io	*mc_io,
		     u32		cmd_flags,
		     u16		token,
		     enum dpni_offload	type,
		     u32		config);

int dpni_get_offload(struct fsl_mc_io	*mc_io,
		     u32		cmd_flags,
		     u16		token,
		     enum dpni_offload	type,
		     u32		*config);

int dpni_get_qdid(struct fsl_mc_io	*mc_io,
		  u32			cmd_flags,
		  u16			token,
		  enum dpni_queue_type	qtype,
		  u16			*qdid);

int dpni_get_tx_data_offset(struct fsl_mc_io	*mc_io,
			    u32			cmd_flags,
			    u16			token,
			    u16			*data_offset);

#define DPNI_STATISTICS_CNT		7

union dpni_statistics {
	/**
	 * struct page_0 - Page_0 statistics structure
	 * @ingress_all_frames: Ingress frame count
	 * @ingress_all_bytes: Ingress byte count
	 * @ingress_multicast_frames: Ingress multicast frame count
	 * @ingress_multicast_bytes: Ingress multicast byte count
	 * @ingress_broadcast_frames: Ingress broadcast frame count
	 * @ingress_broadcast_bytes: Ingress broadcast byte count
	 */
	struct {
		u64 ingress_all_frames;
		u64 ingress_all_bytes;
		u64 ingress_multicast_frames;
		u64 ingress_multicast_bytes;
		u64 ingress_broadcast_frames;
		u64 ingress_broadcast_bytes;
	} page_0;
	/**
	 * struct page_1 - Page_1 statistics structure
	 * @egress_all_frames: Egress frame count
	 * @egress_all_bytes: Egress byte count
	 * @egress_multicast_frames: Egress multicast frame count
	 * @egress_multicast_bytes: Egress multicast byte count
	 * @egress_broadcast_frames: Egress broadcast frame count
	 * @egress_broadcast_bytes: Egress broadcast byte count
	 */
	struct {
		u64 egress_all_frames;
		u64 egress_all_bytes;
		u64 egress_multicast_frames;
		u64 egress_multicast_bytes;
		u64 egress_broadcast_frames;
		u64 egress_broadcast_bytes;
	} page_1;
	/**
	 * struct page_2 - Page_2 statistics structure
	 * @ingress_filtered_frames: Ingress filtered frame count
	 * @ingress_discarded_frames: Ingress discarded frame count
	 * @ingress_nobuffer_discards: Ingress discarded frame count
	 *				due to lack of buffers
	 * @egress_discarded_frames: Egress discarded frame count
	 * @egress_confirmed_frames: Egress confirmed frame count
	 */
	struct {
		u64 ingress_filtered_frames;
		u64 ingress_discarded_frames;
		u64 ingress_nobuffer_discards;
		u64 egress_discarded_frames;
		u64 egress_confirmed_frames;
	} page_2;
	/**
	 * struct raw - raw statistics structure
	 */
	struct {
		u64 counter[DPNI_STATISTICS_CNT];
	} raw;
};

int dpni_get_statistics(struct fsl_mc_io	*mc_io,
			u32			cmd_flags,
			u16			token,
			u8			page,
			union dpni_statistics	*stat);

/**
 * Enable auto-negotiation
 */
#define DPNI_LINK_OPT_AUTONEG		0x0000000000000001ULL
/**
 * Enable half-duplex mode
 */
#define DPNI_LINK_OPT_HALF_DUPLEX	0x0000000000000002ULL
/**
 * Enable pause frames
 */
#define DPNI_LINK_OPT_PAUSE		0x0000000000000004ULL
/**
 * Enable a-symmetric pause frames
 */
#define DPNI_LINK_OPT_ASYM_PAUSE	0x0000000000000008ULL

/**
 * struct - Structure representing DPNI link configuration
 * @rate: Rate
 * @options: Mask of available options; use 'DPNI_LINK_OPT_<X>' values
 */
struct dpni_link_cfg {
	u32 rate;
	u64 options;
};

int dpni_set_link_cfg(struct fsl_mc_io			*mc_io,
		      u32				cmd_flags,
		      u16				token,
		      const struct dpni_link_cfg	*cfg);

/**
 * struct dpni_link_state - Structure representing DPNI link state
 * @rate: Rate
 * @options: Mask of available options; use 'DPNI_LINK_OPT_<X>' values
 * @up: Link state; '0' for down, '1' for up
 */
struct dpni_link_state {
	u32	rate;
	u64	options;
	int	up;
};

int dpni_get_link_state(struct fsl_mc_io	*mc_io,
			u32			cmd_flags,
			u16			token,
			struct dpni_link_state	*state);

int dpni_set_max_frame_length(struct fsl_mc_io	*mc_io,
			      u32		cmd_flags,
			      u16		token,
			      u16		max_frame_length);

int dpni_get_max_frame_length(struct fsl_mc_io	*mc_io,
			      u32		cmd_flags,
			      u16		token,
			      u16		*max_frame_length);

int dpni_set_multicast_promisc(struct fsl_mc_io *mc_io,
			       u32		cmd_flags,
			       u16		token,
			       int		en);

int dpni_get_multicast_promisc(struct fsl_mc_io *mc_io,
			       u32		cmd_flags,
			       u16		token,
			       int		*en);

int dpni_set_unicast_promisc(struct fsl_mc_io	*mc_io,
			     u32		cmd_flags,
			     u16		token,
			     int		en);

int dpni_get_unicast_promisc(struct fsl_mc_io	*mc_io,
			     u32		cmd_flags,
			     u16		token,
			     int		*en);

int dpni_set_primary_mac_addr(struct fsl_mc_io *mc_io,
			      u32		cmd_flags,
			      u16		token,
			      const u8		mac_addr[6]);

int dpni_get_primary_mac_addr(struct fsl_mc_io	*mc_io,
			      u32		cmd_flags,
			      u16		token,
			      u8		mac_addr[6]);

int dpni_get_port_mac_addr(struct fsl_mc_io	*mc_io,
			   u32			cm_flags,
			   u16			token,
			   u8			mac_addr[6]);

int dpni_add_mac_addr(struct fsl_mc_io	*mc_io,
		      u32		cmd_flags,
		      u16		token,
		      const u8		mac_addr[6]);

int dpni_remove_mac_addr(struct fsl_mc_io	*mc_io,
			 u32			cmd_flags,
			 u16			token,
			 const u8		mac_addr[6]);

int dpni_clear_mac_filters(struct fsl_mc_io	*mc_io,
			   u32			cmd_flags,
			   u16			token,
			   int			unicast,
			   int			multicast);

/**
 * enum dpni_dist_mode - DPNI distribution mode
 * @DPNI_DIST_MODE_NONE: No distribution
 * @DPNI_DIST_MODE_HASH: Use hash distribution; only relevant if
 *		the 'DPNI_OPT_DIST_HASH' option was set at DPNI creation
 * @DPNI_DIST_MODE_FS:  Use explicit flow steering; only relevant if
 *	 the 'DPNI_OPT_DIST_FS' option was set at DPNI creation
 */
enum dpni_dist_mode {
	DPNI_DIST_MODE_NONE = 0,
	DPNI_DIST_MODE_HASH = 1,
	DPNI_DIST_MODE_FS = 2
};

/**
 * enum dpni_fs_miss_action -   DPNI Flow Steering miss action
 * @DPNI_FS_MISS_DROP: In case of no-match, drop the frame
 * @DPNI_FS_MISS_EXPLICIT_FLOWID: In case of no-match, use explicit flow-id
 * @DPNI_FS_MISS_HASH: In case of no-match, distribute using hash
 */
enum dpni_fs_miss_action {
	DPNI_FS_MISS_DROP = 0,
	DPNI_FS_MISS_EXPLICIT_FLOWID = 1,
	DPNI_FS_MISS_HASH = 2
};

/**
 * struct dpni_fs_tbl_cfg - Flow Steering table configuration
 * @miss_action: Miss action selection
 * @default_flow_id: Used when 'miss_action = DPNI_FS_MISS_EXPLICIT_FLOWID'
 */
struct dpni_fs_tbl_cfg {
	enum dpni_fs_miss_action	miss_action;
	u16				default_flow_id;
};

int dpni_prepare_key_cfg(const struct dpkg_profile_cfg *cfg,
			 u8 *key_cfg_buf);

/**
 * struct dpni_rx_tc_dist_cfg - Rx traffic class distribution configuration
 * @dist_size: Set the distribution size;
 *	supported values: 1,2,3,4,6,7,8,12,14,16,24,28,32,48,56,64,96,
 *	112,128,192,224,256,384,448,512,768,896,1024
 * @dist_mode: Distribution mode
 * @key_cfg_iova: I/O virtual address of 256 bytes DMA-able memory filled with
 *		the extractions to be used for the distribution key by calling
 *		dpni_prepare_key_cfg() relevant only when
 *		'dist_mode != DPNI_DIST_MODE_NONE', otherwise it can be '0'
 * @fs_cfg: Flow Steering table configuration; only relevant if
 *		'dist_mode = DPNI_DIST_MODE_FS'
 */
struct dpni_rx_tc_dist_cfg {
	u16			dist_size;
	enum dpni_dist_mode	dist_mode;
	u64			key_cfg_iova;
	struct dpni_fs_tbl_cfg	fs_cfg;
};

int dpni_set_rx_tc_dist(struct fsl_mc_io			*mc_io,
			u32					cmd_flags,
			u16					token,
			u8					tc_id,
			const struct dpni_rx_tc_dist_cfg	*cfg);

/**
 * enum dpni_dest - DPNI destination types
 * @DPNI_DEST_NONE: Unassigned destination; The queue is set in parked mode and
 *		does not generate FQDAN notifications; user is expected to
 *		dequeue from the queue based on polling or other user-defined
 *		method
 * @DPNI_DEST_DPIO: The queue is set in schedule mode and generates FQDAN
 *		notifications to the specified DPIO; user is expected to dequeue
 *		from the queue only after notification is received
 * @DPNI_DEST_DPCON: The queue is set in schedule mode and does not generate
 *		FQDAN notifications, but is connected to the specified DPCON
 *		object; user is expected to dequeue from the DPCON channel
 */
enum dpni_dest {
	DPNI_DEST_NONE = 0,
	DPNI_DEST_DPIO = 1,
	DPNI_DEST_DPCON = 2
};

/**
 * struct dpni_queue - Queue structure
 * @user_context: User data, presented to the user along with any frames from
 *		this queue. Not relevant for Tx queues.
 */
struct dpni_queue {
/**
 * struct destination - Destination structure
 * @id: ID of the destination, only relevant if DEST_TYPE is > 0.
 *		Identifies either a DPIO or a DPCON object. Not relevant for
 *		Tx queues.
 * @type: May be one of the following:
 *	0 - No destination, queue can be manually queried, but will not
 *		push traffic or notifications to a DPIO;
 *	1 - The destination is a DPIO. When traffic becomes available in
 *		the queue a FQDAN (FQ data available notification) will be
 *		generated to selected DPIO;
 *	2 - The destination is a DPCON. The queue is associated with a
 *		DPCON object for the purpose of scheduling between multiple
 *		queues. The DPCON may be independently configured to
 *		generate notifications. Not relevant for Tx queues.
 * @hold_active: Hold active, maintains a queue scheduled for longer
 *		in a DPIO during dequeue to reduce spread of traffic.
 *		Only relevant if queues are not affined to a single DPIO.
 */
	struct {
		u16 id;
		enum dpni_dest type;
		char hold_active;
		u8 priority;
	} destination;
	u64 user_context;
	struct {
		u64 value;
		char stash_control;
	} flc;
};

/**
 * struct dpni_queue_id - Queue identification, used for enqueue commands
 *			or queue control
 * @fqid: FQID used for enqueueing to and/or configuration of this specific FQ
 * @qdbin: Queueing bin, used to enqueue using QDID, DQBIN, QPRI. Only relevant
 *		for Tx queues.
 */
struct dpni_queue_id {
	u32 fqid;
	u16 qdbin;
};

/**
 * Set User Context
 */
#define DPNI_QUEUE_OPT_USER_CTX		0x00000001
#define DPNI_QUEUE_OPT_DEST		0x00000002
#define DPNI_QUEUE_OPT_FLC		0x00000004
#define DPNI_QUEUE_OPT_HOLD_ACTIVE	0x00000008

int dpni_set_queue(struct fsl_mc_io	*mc_io,
		   u32			cmd_flags,
		   u16			token,
		   enum dpni_queue_type	qtype,
		   u8			tc,
		   u8			index,
		   u8			options,
		   const struct dpni_queue *queue);

int dpni_get_queue(struct fsl_mc_io	*mc_io,
		   u32			cmd_flags,
		   u16			token,
		   enum dpni_queue_type	qtype,
		   u8			tc,
		   u8			index,
		   struct dpni_queue	*queue,
		   struct dpni_queue_id	*qid);

/**
 * enum dpni_congestion_unit - DPNI congestion units
 * @DPNI_CONGESTION_UNIT_BYTES: bytes units
 * @DPNI_CONGESTION_UNIT_FRAMES: frames units
 */
enum dpni_congestion_unit {
	DPNI_CONGESTION_UNIT_BYTES = 0,
	DPNI_CONGESTION_UNIT_FRAMES
};

/**
 * enum dpni_congestion_point - Structure representing congestion point
 * @DPNI_CP_QUEUE: Set taildrop per queue, identified by QUEUE_TYPE, TC and
 *		QUEUE_INDEX
 * @DPNI_CP_GROUP: Set taildrop per queue group. Depending on options used to
 *		define the DPNI this can be either per TC (default) or per
 *		interface (DPNI_OPT_SHARED_CONGESTION set at DPNI create).
 *		QUEUE_INDEX is ignored if this type is used.
 */
enum dpni_congestion_point {
	DPNI_CP_QUEUE,
	DPNI_CP_GROUP,
};

/**
 * struct dpni_taildrop - Structure representing the taildrop
 * @enable:	Indicates whether the taildrop is active or not.
 * @units:	Indicates the unit of THRESHOLD. Queue taildrop only supports
 *		byte units, this field is ignored and assumed = 0 if
 *		CONGESTION_POINT is 0.
 * @threshold:	Threshold value, in units identified by UNITS field. Value 0
 *		cannot be used as a valid taildrop threshold, THRESHOLD must
 *		be > 0 if the taildrop is enabled.
 */
struct dpni_taildrop {
	char enable;
	enum dpni_congestion_unit units;
	u32 threshold;
};

int dpni_set_taildrop(struct fsl_mc_io *mc_io,
		      u32 cmd_flags,
		      u16 token,
		      enum dpni_congestion_point cg_point,
		      enum dpni_queue_type q_type,
		      u8 tc,
		      u8 q_index,
		      struct dpni_taildrop *taildrop);

int dpni_get_taildrop(struct fsl_mc_io *mc_io,
		      u32 cmd_flags,
		      u16 token,
		      enum dpni_congestion_point cg_point,
		      enum dpni_queue_type q_type,
		      u8 tc,
		      u8 q_index,
		      struct dpni_taildrop *taildrop);

/**
 * struct dpni_rule_cfg - Rule configuration for table lookup
 * @key_iova: I/O virtual address of the key (must be in DMA-able memory)
 * @mask_iova: I/O virtual address of the mask (must be in DMA-able memory)
 * @key_size: key and mask size (in bytes)
 */
struct dpni_rule_cfg {
	u64	key_iova;
	u64	mask_iova;
	u8	key_size;
};

#endif /* __FSL_DPNI_H */
