/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.util.net;

import com.google.common.base.Optional;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.net.HostAndPort;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.brooklyn.util.exceptions.RuntimeInterruptedException;
import org.apache.brooklyn.util.net.Networking;
import org.apache.brooklyn.util.repeat.Repeater;
import org.apache.brooklyn.util.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReachableSocketFinder {
    private static final Logger LOG = LoggerFactory.getLogger(ReachableSocketFinder.class);
    private final Predicate<? super HostAndPort> socketTester;
    private final Duration gracePeriod;

    public ReachableSocketFinder() {
        this(Networking.isReachablePredicate());
    }

    public ReachableSocketFinder(Predicate<? super HostAndPort> socketTester) {
        this(socketTester, Duration.FIVE_SECONDS);
    }

    public ReachableSocketFinder(Predicate<? super HostAndPort> socketTester, Duration gracePeriod) {
        this.socketTester = (Predicate)Preconditions.checkNotNull(socketTester, (Object)"socketTester");
        this.gracePeriod = (Duration)Preconditions.checkNotNull((Object)gracePeriod, (Object)"gracePeriod");
    }

    public HostAndPort findOpenSocketOnNode(Iterable<? extends HostAndPort> sockets, Duration timeout) {
        Preconditions.checkNotNull(sockets, (Object)"sockets");
        Preconditions.checkState((!Iterables.isEmpty(sockets) ? 1 : 0) != 0, (Object)"No hostAndPort sockets supplied");
        Preconditions.checkNotNull((Object)timeout, (Object)"timeout");
        Iterable reachable = Optional.presentInstances(this.tryReachable(sockets, timeout, false));
        Iterator iter = reachable.iterator();
        if (iter.hasNext()) {
            return (HostAndPort)iter.next();
        }
        LOG.warn("No sockets in {} reachable after {}", sockets, (Object)timeout);
        throw new NoSuchElementException("Could not connect to any socket in " + sockets);
    }

    public Iterable<HostAndPort> findOpenSocketsOnNode(Iterable<? extends HostAndPort> sockets, Duration timeout) {
        Preconditions.checkNotNull(sockets, (Object)"sockets");
        Preconditions.checkState((!Iterables.isEmpty(sockets) ? 1 : 0) != 0, (Object)"No hostAndPort sockets supplied");
        Preconditions.checkNotNull((Object)timeout, (Object)"timeout");
        return Optional.presentInstances(this.tryReachable(sockets, timeout, true));
    }

    private Iterable<Optional<HostAndPort>> tryReachable(Iterable<? extends HostAndPort> sockets, final Duration timeout, final boolean useGracePeriod) {
        LOG.debug("Blocking on reachable sockets in {} for {}", sockets, (Object)timeout);
        ArrayList futures = Lists.newArrayList();
        final AtomicReference sinceFirstCompleted = new AtomicReference();
        final ListeningExecutorService executor = MoreExecutors.listeningDecorator((ExecutorService)Executors.newFixedThreadPool(10));
        for (final HostAndPort hostAndPort : sockets) {
            futures.add(executor.submit((Callable)new Callable<Optional<HostAndPort>>(){
                Future<Boolean> checker;

                private void reschedule() {
                    this.checker = executor.submit((Callable)new SocketChecker(hostAndPort, ReachableSocketFinder.this.socketTester));
                }

                private boolean gracePeriodExpired() {
                    Stopwatch firstCompleted = (Stopwatch)sinceFirstCompleted.get();
                    return firstCompleted != null && (!useGracePeriod || ReachableSocketFinder.this.gracePeriod.subtract(Duration.of(firstCompleted)).isNegative());
                }

                private boolean isComplete() throws ExecutionException, InterruptedException {
                    boolean currentCheckComplete = this.checker.isDone();
                    if (currentCheckComplete && this.checker.get().booleanValue()) {
                        LOG.trace("{} determined that {} is reachable", (Object)this, (Object)hostAndPort);
                        sinceFirstCompleted.compareAndSet(null, Stopwatch.createStarted());
                        return true;
                    }
                    if (currentCheckComplete) {
                        LOG.trace("{} unsure if {} is reachable, scheduling another check", (Object)this, (Object)hostAndPort);
                        this.reschedule();
                    }
                    return false;
                }

                @Override
                public Optional<HostAndPort> call() throws Exception {
                    LOG.trace("Checking reachability of {}", (Object)hostAndPort);
                    this.reschedule();
                    Repeater.create().limitTimeTo(timeout).backoffTo(Duration.FIVE_SECONDS).until(new Callable<Boolean>(){

                        @Override
                        public Boolean call() throws Exception {
                            return this.isComplete() || this.gracePeriodExpired();
                        }
                    }).run();
                    if (this.checker.isDone() && this.checker.get().booleanValue()) {
                        LOG.trace("Finished checking reachability of {}: success", (Object)hostAndPort);
                        return Optional.of((Object)hostAndPort);
                    }
                    LOG.trace("Finished checking reachability of {}: failure", (Object)hostAndPort);
                    this.checker.cancel(true);
                    return Optional.absent();
                }
            }));
        }
        ImmutableList.Builder results = ImmutableList.builder();
        for (ListenableFuture f : futures) {
            try {
                results.add(f.get());
            }
            catch (InterruptedException e) {
                throw new RuntimeInterruptedException(e);
            }
            catch (ExecutionException e) {
                LOG.trace("Suppressed exception waiting for " + f, (Throwable)e);
            }
        }
        executor.shutdownNow();
        ImmutableList immutableList = results.build();
        LOG.debug("Determined reachability of sockets {}: {}", sockets, (Object)immutableList);
        return immutableList;
    }

    private static class SocketChecker
    implements Callable<Boolean> {
        final HostAndPort socket;
        final Predicate<? super HostAndPort> predicate;

        private SocketChecker(HostAndPort socket, Predicate<? super HostAndPort> predicate) {
            this.socket = socket;
            this.predicate = predicate;
        }

        @Override
        public Boolean call() {
            return this.predicate.apply((Object)this.socket);
        }
    }
}

