/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.datastax.driver.core;

import static com.datastax.driver.core.TestUtils.nonQuietClusterCloseOptions;
import static org.assertj.core.api.Assertions.assertThat;
import static org.scassandra.http.client.PrimingRequest.then;

import org.scassandra.Scassandra;
import org.scassandra.http.client.PrimingRequest;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

public class MetricsInFlightTest {
  private ScassandraCluster sCluster;

  @BeforeMethod(groups = "short")
  public void setUp() {
    sCluster = ScassandraCluster.builder().withNodes(1).build();
    sCluster.init();
  }

  @AfterMethod(groups = "short")
  public void tearDown() {
    clearActivityLog();
    sCluster.stop();
  }

  public void clearActivityLog() {
    for (Scassandra node : sCluster.nodes()) {
      node.activityClient().clearAllRecordedActivity();
    }
  }

  public Cluster.Builder builder() {
    // Note: nonQuietClusterCloseOptions is used to speed up tests
    return Cluster.builder()
        .addContactPoints(sCluster.address(1).getAddress())
        .withPort(sCluster.getBinaryPort())
        .withNettyOptions(nonQuietClusterCloseOptions);
  }

  @Test(groups = "short")
  public void should_count_inflight_requests_metrics() {
    sCluster
        .node(1)
        .primingClient()
        .prime(
            PrimingRequest.queryBuilder()
                .withQuery("mock query")
                .withThen(then().withFixedDelay(100000L))
                .build());

    Cluster cluster = null;
    try {
      cluster = builder().build();
      Session session = cluster.connect();

      assertThat(cluster.getMetrics().getInFlightRequests().getValue()).isEqualTo(0);
      session.executeAsync("mock query");
      session.executeAsync("mock query");
      assertThat(cluster.getMetrics().getInFlightRequests().getValue()).isEqualTo(2);

    } finally {
      if (cluster != null) {
        cluster.close();
      }
    }
  }

  @Test(groups = "short")
  public void should_countdown_inflight_requests_metrics() {
    sCluster
        .node(1)
        .primingClient()
        .prime(PrimingRequest.queryBuilder().withQuery("mock query").withThen(then()).build());

    Cluster cluster = null;
    try {
      cluster = builder().build();
      Session session = cluster.connect();

      assertThat(cluster.getMetrics().getInFlightRequests().getValue()).isEqualTo(0);
      session.executeAsync("mock query").getUninterruptibly();
      session.executeAsync("mock query").getUninterruptibly();
      assertThat(cluster.getMetrics().getInFlightRequests().getValue()).isEqualTo(0);

    } finally {
      if (cluster != null) {
        cluster.close();
      }
    }
  }
}
