package org.apache.torque.sql.whereclausebuilder;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.torque.TorqueException;
import org.apache.torque.adapter.Adapter;
import org.apache.torque.criteria.PreparedStatementPart;
import org.apache.torque.criteria.SqlEnum;
import org.apache.torque.sql.WhereClauseExpression;

/**
 * Builds a PreparedStatementPart from a WhereClauseExpression containing
 * a Like operator.
 *
 * @version $Id: LikeBuilder.java 1448414 2013-02-20 21:06:35Z tfischer $
 */
public class LikeBuilder extends AbstractWhereClausePsPartBuilder
{
    /** The backslash character*/
    private static final char BACKSLASH = '\\';

    /**
     * Builds the PS part for a WhereClauseExpression with a LIKE operator.
     * Multicharacter wildcards % and * may be used
     * as well as single character wildcards, _ and ?.  These
     * characters can be escaped with \.
     *
     * e.g. criteria = "fre%" -> columnName LIKE 'fre%'
     *                        -> UPPER(columnName) LIKE UPPER('fre%')
     *      criteria = "50\%" -> columnName = '50%'
     *
     * @param whereClausePart the part of the where clause to build.
     *        Can be modified in this method.
     * @param ignoreCase If true and columns represent Strings, the appropriate
     *        function defined for the database will be used to ignore
     *        differences in case.
     * @param adapter The adapter for the database for which the SQL
     *        should be created, not null.
     *
     * @return the rendered SQL for the WhereClauseExpression
     */
    public PreparedStatementPart buildPs(
                WhereClauseExpression whereClausePart,
                boolean ignoreCase,
                Adapter adapter)
            throws TorqueException
    {
        if (!(whereClausePart.getRValue() instanceof String))
        {
            throw new TorqueException(
                "rValue must be a String for the operator "
                    + whereClausePart.getOperator());
        }
        String value = (String) whereClausePart.getRValue();
        // If selection criteria contains wildcards use LIKE otherwise
        // use = (equals).  Wildcards can be escaped by prepending
        // them with \ (backslash). However, if we switch from
        // like to equals, we need to remove the escape characters.
        // from the wildcards.
        // So we need two passes: The first replaces * and ? by % and _,
        // and checks whether we switch to equals,
        // the second removes escapes if we have switched to equals.
        int position = 0;
        StringBuffer sb = new StringBuffer();
        boolean replaceWithEquals = true;
        while (position < value.length())
        {
            char checkWildcard = value.charAt(position);

            switch (checkWildcard)
            {
            case BACKSLASH:
                if (position + 1 >= value.length())
                {
                    // ignore backslashes at end
                    break;
                }
                position++;
                char escapedChar = value.charAt(position);
                if (escapedChar != '*' && escapedChar != '?')
                {
                    sb.append(checkWildcard);
                }
                // code below copies escaped character into sb
                checkWildcard = escapedChar;
                break;
            case '%':
            case '_':
                replaceWithEquals = false;
                break;
            case '*':
                replaceWithEquals = false;
                checkWildcard = '%';
                break;
            case '?':
                replaceWithEquals = false;
                checkWildcard = '_';
                break;
            default:
                break;
            }

            sb.append(checkWildcard);
            position++;
        }
        value = sb.toString();

        PreparedStatementPart result;
        if (ignoreCase)
        {
            if (adapter.useIlike() && !replaceWithEquals)
            {
                if (SqlEnum.LIKE.equals(whereClausePart.getOperator()))
                {
                    whereClausePart.setOperator(SqlEnum.ILIKE);
                }
                else if (SqlEnum.NOT_LIKE.equals(whereClausePart.getOperator()))
                {
                    whereClausePart.setOperator(SqlEnum.NOT_ILIKE);
                }
                result = getObjectOrColumnPsPartBuilder().buildPs(
                        whereClausePart.getLValue(), false, adapter);
            }
            else
            {
                // no native case insensitive like is offered by the DB,
                // or the LIKE was replaced with equals.
                // need to ignore case manually.
                result = getObjectOrColumnPsPartBuilder().buildPs(
                        whereClausePart.getLValue(), true, adapter);
            }
        }
        else
        {
            result = getObjectOrColumnPsPartBuilder().buildPs(
                    whereClausePart.getLValue(), ignoreCase, adapter);
        }

        if (replaceWithEquals)
        {
            if (whereClausePart.getOperator().equals(SqlEnum.NOT_LIKE)
                    || whereClausePart.getOperator().equals(SqlEnum.NOT_ILIKE))
            {
                result.getSql().append(SqlEnum.NOT_EQUAL);
            }
            else
            {
                result.getSql().append(SqlEnum.EQUAL);
            }

            // remove escape backslashes from String
            position = 0;
            sb = new StringBuffer();
            while (position < value.length())
            {
                char checkWildcard = value.charAt(position);

                if (checkWildcard == BACKSLASH
                        && position + 1 < value.length())
                {
                    position++;
                    // code below copies escaped character into sb
                    checkWildcard = value.charAt(position);
                }
                sb.append(checkWildcard);
                position++;
            }
            value = sb.toString();
        }
        else
        {
            result.getSql().append(whereClausePart.getOperator());
        }

        String rValueSql = "?";
        // handle ignoreCase if necessary
        if (ignoreCase && (!(adapter.useIlike()) || replaceWithEquals))
        {
            rValueSql = adapter.ignoreCase(rValueSql);
        }
        // handle escape clause if necessary
        if (!replaceWithEquals && adapter.useEscapeClauseForLike())
        {
            rValueSql = rValueSql + SqlEnum.ESCAPE + "'\\'";
        }

        result.getPreparedStatementReplacements().add(value);
        result.getSql().append(rValueSql);
        return result;
    }

    /**
     * {@inheritDoc}
     */
    public boolean isApplicable(
            WhereClauseExpression whereClauseExpression,
            Adapter adapter)
    {
        if (whereClauseExpression.getOperator().equals(SqlEnum.LIKE)
            || whereClauseExpression.getOperator().equals(SqlEnum.NOT_LIKE)
            || whereClauseExpression.getOperator().equals(SqlEnum.ILIKE)
            || whereClauseExpression.getOperator().equals(SqlEnum.NOT_ILIKE))
        {
            return true;
        }
        return false;
    }
}
