/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.log4j.db;

import org.apache.log4j.spi.Component;
import org.apache.log4j.spi.OptionHandler;

import java.sql.Connection;
import java.sql.SQLException;


/**
 * The {@code ConnectionSource} interface provides a pluggable means of
 * transparently obtaining JDBC {@link java.sql.Connection}s for log4j classes
 * that require the use of a {@link java.sql.Connection}.
 *
 * @author <a href="mailto:rdecampo@twcny.rr.com">Ray DeCampo</a>
 */
public interface ConnectionSource extends Component, OptionHandler {

    int UNKNOWN_DIALECT = 0;
    int POSTGRES_DIALECT = 1;
    int MYSQL_DIALECT = 2;
    int ORACLE_DIALECT = 3;
    int MSSQL_DIALECT = 4;
    int HSQL_DIALECT = 5;

    /**
     * Obtain a {@link java.sql.Connection} for use.  The client is
     * responsible for closing the {@link java.sql.Connection} when it is no
     * longer required.
     *
     * @throws SQLException if a {@link java.sql.Connection} could not be
     *                      obtained
     */
    Connection getConnection() throws SQLException;

    /**
     * Get the SQL dialect that should be used for this connection. Note that the
     * dialect is not needed if the JDBC driver supports the getGeneratedKeys
     * method.
     */
    int getSQLDialectCode();

    /**
     * If the connection supports the JDBC 3.0 getGeneratedKeys method, then
     * we do not need any specific dialect support.
     */
    boolean supportsGetGeneratedKeys();

    /**
     * If the connection does not support batch updates, we will avoid using them.
     */
    boolean supportsBatchUpdates();
}
