package org.apache.jetspeed.portlet;

import org.apache.jetspeed.portlet.event.WindowListener;
import java.io.InvalidObjectException;
import java.io.ObjectStreamException;

/**
 ** The <CODE>PortletWindow</CODE> represents the window that encloses
 ** a portlet. The portlet window can send events on manipulation of
 ** its various window controls, like the "minimize" or "close" buttons.
 ** But a portlet can also interrogate the portlet window about its
 ** current visibility state. For example, a portlet may render its
 ** content differently depending on whether its window is maximized or
 ** not.
 **
 ** @author   <A HREF="mailto:tboehme@apache.org">Thomas F. Boehme</A>
 **/

public interface PortletWindow
{
    /**
     ** The <CODE>State</CODE> class is a finite enumeration of
     ** the possible states that a window can assume.
     **/

    public static class State implements java.io.Serializable
    {
        /**
         ** The standard "one-of many" window state on a page.
         **/

        public final static State NORMAL = new State ("Normal", 0);

        /**
         ** In this window state the portlet is displayed maximized
         ** which means that it is the only portlet shown on the page.
         **/

        public final static State MAXIMIZED = new State ("Maximized", 1);

        /**
         ** In this window state the portlet is displayed minimzed
         ** which means that only the portlet title is shown
         **/

        public final static State MINIMIZED = new State ("Minimized", 2);

        /**
         ** Allows the portlet window to be detached of the normal
         ** content view of the portal and thus be shown in a separate
         ** window.
         **/

        public final static State DETACHED = new State ("Detached", 3);

        /**
         ** Allows the portlet window to be moved
         **/

        public final static State MOVING  = new State ("Moving", 4);

        /**
         ** Allows the portlet window to be resized
         **/

        public final static State RESIZING = new State ("Resizing", 5);

        /**
         ** The window is or will be closed and thus is not shown on the
         ** portal page anymore.
         **/

        public final static State CLOSED = new State ("Closed", 6);

        private final static PortletWindow.State[] STATES = 
                                        {
                                         NORMAL, MAXIMIZED, MINIMIZED,
                                         DETACHED, MOVING, RESIZING,
                                         CLOSED
                                        };
        
        private String identifier;

        private int value;

        private State (String identifier, int value)
        {
            this.identifier = identifier;
            this.value = value;
        }

        public int getId()
        {
            return value;
        }

        public String toString ()
        {
            return (identifier);
        }

        public Object readResolve() throws ObjectStreamException {
            try {
                return STATES[value];
            }
            catch (IndexOutOfBoundsException e) {
                throw new InvalidObjectException("Unknown Portlet State");
            }
        }
    }

    /**
     ** Returns whether this portlet window is currently detached.
     **
     ** @return   <CODE>true</CODE> if this window is detached, <BR>
     **           <CODE>false</CODE> otherwise
     * @deprecated use getWindowState() since version 1.1
     **/

    public boolean isDetached ();

    /**
     ** Returns whether this portlet window is currently maximized.
     **
     ** @return   <CODE>true</CODE> if this window is maximized, <BR>
     **           <CODE>false</CODE> otherwise
     * @deprecated use getWindowState() since version 1.1
     **/

    public boolean isMaximized ();

    /**
     * Returns whether this portlet window is currently minimized.
     * 
     * @return <CODE>true</CODE> if this window is minimized, <BR>
     *         <CODE>false</CODE> otherwise
     * @deprecated use getWindowState() since version 1.1
     */
    public boolean isMinimized ();

    /**
     ** Returns the current window state of this portlet
     **
     ** @return   the window state
     **/

    public State getWindowState ();

    /**
     ** Defines which portlet window state is shown next.
     **
     ** <P>
     ** This function may only be used during event processing,
     ** in any other case the call has no effect.
     **
     ** @param   state
     **          the portlet window state to be shown next
     **
     ** @exception   AccessDeniedException
     **              if the portlet tries to access this function
     **              outside of the event processing
     **/

    public void setWindowState (State state) throws AccessDeniedException;
}
