/*
 * Decompiled with CFR 0.152.
 */
package reactor.netty.resources;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.pool.ChannelHealthChecker;
import io.netty.channel.pool.ChannelPool;
import io.netty.channel.pool.ChannelPoolHandler;
import io.netty.util.AttributeKey;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.Promise;
import io.netty.util.internal.PlatformDependent;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import reactor.core.Disposable;
import reactor.core.publisher.Mono;
import reactor.core.publisher.MonoProcessor;
import reactor.core.publisher.MonoSink;
import reactor.netty.Connection;
import reactor.netty.ConnectionObserver;
import reactor.netty.ReactorNetty;
import reactor.netty.channel.BootstrapHandlers;
import reactor.netty.channel.ChannelOperations;
import reactor.netty.resources.ConnectionProvider;
import reactor.netty.resources.NewConnectionProvider;
import reactor.util.Logger;
import reactor.util.Loggers;
import reactor.util.annotation.NonNull;
import reactor.util.concurrent.Queues;
import reactor.util.context.Context;

final class PooledConnectionProvider
implements ConnectionProvider {
    final ConcurrentMap<PoolKey, Pool> channelPools;
    final String name;
    final PoolFactory poolFactory;
    static final Logger log = Loggers.getLogger(PooledConnectionProvider.class);
    static final AttributeKey<ConnectionObserver> OWNER = AttributeKey.valueOf((String)"connectionOwner");

    PooledConnectionProvider(String name, PoolFactory poolFactory) {
        this.name = name;
        this.poolFactory = poolFactory;
        this.channelPools = PlatformDependent.newConcurrentHashMap();
    }

    @Override
    public void disposeWhen(@NonNull SocketAddress address) {
        List<Map.Entry> toDispose = this.channelPools.entrySet().stream().filter(p -> this.compareAddresses(((PoolKey)p.getKey()).holder, address)).collect(Collectors.toList());
        toDispose.forEach(e -> {
            if (this.channelPools.remove(e.getKey(), e.getValue())) {
                if (log.isDebugEnabled()) {
                    log.debug("Disposing pool for {}", new Object[]{((PoolKey)e.getKey()).fqdn});
                }
                ((Pool)e.getValue()).pool.close();
            }
        });
    }

    private boolean compareAddresses(SocketAddress origin, SocketAddress target) {
        if (origin.equals(target)) {
            return true;
        }
        if (origin instanceof InetSocketAddress && target instanceof InetSocketAddress) {
            InetSocketAddress isaOrigin = (InetSocketAddress)origin;
            InetSocketAddress isaTarget = (InetSocketAddress)target;
            InetAddress iaTarget = isaTarget.getAddress();
            if (iaTarget != null && iaTarget.isAnyLocalAddress() && isaOrigin.getPort() == isaTarget.getPort()) {
                return true;
            }
        }
        return false;
    }

    public Mono<Connection> acquire(Bootstrap b) {
        return Mono.create(sink -> {
            Pool pool;
            Bootstrap bootstrap = b.clone();
            ChannelOperations.OnSetup opsFactory = BootstrapHandlers.channelOperationFactory(bootstrap);
            ConnectionObserver obs = BootstrapHandlers.connectionObserver(bootstrap);
            NewConnectionProvider.convertLazyRemoteAddress(bootstrap);
            ChannelHandler handler = bootstrap.config().handler();
            PoolKey holder = new PoolKey(bootstrap.config().remoteAddress(), handler != null ? handler.hashCode() : -1);
            while ((pool = (Pool)this.channelPools.get(holder)) == null) {
                if (log.isDebugEnabled()) {
                    log.debug("Creating new client pool [{}] for {}", new Object[]{this.name, bootstrap.config().remoteAddress()});
                }
                if (this.channelPools.putIfAbsent(holder, pool = new Pool(bootstrap, this.poolFactory, opsFactory)) == null) break;
                pool.close();
            }
            PooledConnectionProvider.disposableAcquire((MonoSink<Connection>)sink, obs, pool);
        });
    }

    @Override
    public void dispose() {
        this.disposeLater().subscribe();
    }

    @Override
    public Mono<Void> disposeLater() {
        return Mono.fromRunnable(() -> {
            for (PoolKey key : this.channelPools.keySet()) {
                Pool pool = (Pool)this.channelPools.remove(key);
                if (pool == null) continue;
                pool.close();
            }
        });
    }

    public boolean isDisposed() {
        return this.channelPools.isEmpty() || this.channelPools.values().stream().allMatch(AtomicBoolean::get);
    }

    public String toString() {
        return "PooledConnectionProvider {name=" + this.name + ", poolFactory=" + this.poolFactory + '}';
    }

    static void disposableAcquire(MonoSink<Connection> sink, ConnectionObserver obs, Pool pool) {
        Future<Channel> f = pool.acquire();
        DisposableAcquire disposableAcquire = new DisposableAcquire(sink, f, pool, obs);
        f.addListener((GenericFutureListener)disposableAcquire);
        sink.onCancel((Disposable)disposableAcquire);
    }

    static final class PoolKey {
        final SocketAddress holder;
        final int pipelineKey;
        final String fqdn;

        PoolKey(SocketAddress holder, int pipelineKey) {
            this.holder = holder;
            this.fqdn = holder instanceof InetSocketAddress ? holder.toString() : null;
            this.pipelineKey = pipelineKey;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            PoolKey poolKey = (PoolKey)o;
            return this.pipelineKey == poolKey.pipelineKey && Objects.equals(this.holder, poolKey.holder) && Objects.equals(this.fqdn, poolKey.fqdn);
        }

        public int hashCode() {
            return Objects.hash(this.holder, this.pipelineKey, this.fqdn);
        }
    }

    static final class DisposableAcquire
    implements Disposable,
    GenericFutureListener<Future<Channel>>,
    ConnectionObserver,
    Runnable {
        final Future<Channel> f;
        final MonoSink<Connection> sink;
        final Pool pool;
        final ConnectionObserver obs;

        DisposableAcquire(MonoSink<Connection> sink, Future<Channel> future, Pool pool, ConnectionObserver obs) {
            this.f = future;
            this.pool = pool;
            this.sink = sink;
            this.obs = obs;
        }

        public final void dispose() {
            if (this.isDisposed()) {
                return;
            }
            this.f.removeListener((GenericFutureListener)this);
            if (!this.f.isDone()) {
                this.f.cancel(true);
            }
        }

        @Override
        public Context currentContext() {
            return this.sink.currentContext();
        }

        @Override
        public void onUncaughtException(Connection connection, Throwable error) {
            this.sink.error(error);
            this.obs.onUncaughtException(connection, error);
        }

        @Override
        public void onStateChange(Connection connection, ConnectionObserver.State newState) {
            if (newState == ConnectionObserver.State.CONFIGURED) {
                this.sink.success((Object)connection);
            }
            this.obs.onStateChange(connection, newState);
        }

        public boolean isDisposed() {
            return this.f.isCancelled() || this.f.isDone();
        }

        @Override
        public void run() {
            Channel c = (Channel)this.f.getNow();
            this.pool.activeConnections.incrementAndGet();
            this.pool.inactiveConnections.decrementAndGet();
            ConnectionObserver current = (ConnectionObserver)c.attr(OWNER).getAndSet((Object)this);
            if (current instanceof PendingConnectionObserver) {
                PendingConnectionObserver.Pending p;
                PendingConnectionObserver pending = (PendingConnectionObserver)current;
                current = null;
                this.registerClose(c);
                while ((p = pending.pendingQueue.poll()) != null) {
                    if (p.error != null) {
                        this.onUncaughtException(p.connection, p.error);
                        continue;
                    }
                    if (p.state == null) continue;
                    this.onStateChange(p.connection, p.state);
                }
            } else if (current == null) {
                this.registerClose(c);
            }
            if (current != null) {
                Connection conn = Connection.from(c);
                if (log.isDebugEnabled()) {
                    log.debug(ReactorNetty.format(c, "Channel acquired, now {} active connections and {} inactive connections"), new Object[]{this.pool.activeConnections, this.pool.inactiveConnections});
                }
                this.obs.onStateChange(conn, ConnectionObserver.State.ACQUIRED);
                PooledConnection con = conn.as(PooledConnection.class);
                if (con != null) {
                    ChannelOperations<?, ?> ops = this.pool.opsFactory.create(con, con, null);
                    if (ops != null) {
                        ops.bind();
                        this.obs.onStateChange(ops, ConnectionObserver.State.CONFIGURED);
                        this.sink.success(ops);
                    } else {
                        this.sink.success((Object)con);
                    }
                } else {
                    this.sink.success((Object)conn);
                }
                return;
            }
            if (log.isDebugEnabled()) {
                log.debug(ReactorNetty.format(c, "Channel connected, now {} active connections and {} inactive connections"), new Object[]{this.pool.activeConnections, this.pool.inactiveConnections});
            }
            if (this.pool.opsFactory == ChannelOperations.OnSetup.empty()) {
                this.sink.success((Object)Connection.from(c));
            }
        }

        void registerClose(Channel c) {
            if (log.isDebugEnabled()) {
                log.debug(ReactorNetty.format(c, "Registering pool release on close event for channel"));
            }
            c.closeFuture().addListener(ff -> this.pool.release(c));
        }

        public final void operationComplete(Future<Channel> f) throws Exception {
            if (!f.isSuccess()) {
                this.pool.inactiveConnections.decrementAndGet();
                if (f.isCancelled()) {
                    if (log.isDebugEnabled()) {
                        log.debug("Cancelled acquiring from pool {}", new Object[]{this.pool});
                    }
                    return;
                }
                if (f.cause() != null) {
                    this.sink.error(f.cause());
                } else {
                    this.sink.error((Throwable)new IOException("error while acquiring from " + this.pool));
                }
            } else {
                Channel c = (Channel)f.get();
                if (!c.isActive()) {
                    if (log.isDebugEnabled()) {
                        log.debug(ReactorNetty.format(c, "Immediately aborted pooled channel, re-acquiring new channel"));
                    }
                    PooledConnectionProvider.disposableAcquire(this.sink, this.obs, this.pool);
                    return;
                }
                if (c.eventLoop().inEventLoop()) {
                    this.run();
                } else {
                    c.eventLoop().execute((Runnable)this);
                }
            }
        }
    }

    static final class PooledConnection
    implements Connection,
    ConnectionObserver {
        final Channel channel;
        final Pool pool;
        final MonoProcessor<Void> onTerminate;

        PooledConnection(Channel channel, Pool pool) {
            this.channel = channel;
            this.pool = pool;
            this.onTerminate = MonoProcessor.create();
        }

        ConnectionObserver owner() {
            ConnectionObserver obs;
            do {
                if ((obs = (ConnectionObserver)this.channel.attr(OWNER).get()) != null) {
                    return obs;
                }
                obs = new PendingConnectionObserver();
            } while (!this.channel.attr(OWNER).compareAndSet(null, (Object)obs));
            return obs;
        }

        @Override
        public Mono<Void> onTerminate() {
            return this.onTerminate.or(this.onDispose());
        }

        @Override
        public Channel channel() {
            return this.channel;
        }

        @Override
        public Context currentContext() {
            return this.owner().currentContext();
        }

        @Override
        public void onUncaughtException(Connection connection, Throwable error) {
            this.owner().onUncaughtException(connection, error);
        }

        @Override
        public void onStateChange(Connection connection, ConnectionObserver.State newState) {
            if (log.isDebugEnabled()) {
                log.debug(ReactorNetty.format(connection.channel(), "onStateChange({}, {})"), new Object[]{connection, newState});
            }
            if (newState == ConnectionObserver.State.DISCONNECTING) {
                if (!this.isPersistent() && this.channel.isActive()) {
                    this.channel.close();
                    this.owner().onStateChange(connection, ConnectionObserver.State.DISCONNECTING);
                    return;
                }
                if (!this.channel.isActive()) {
                    this.owner().onStateChange(connection, ConnectionObserver.State.DISCONNECTING);
                    return;
                }
                if (log.isDebugEnabled()) {
                    log.debug(ReactorNetty.format(connection.channel(), "Releasing channel"));
                }
                ConnectionObserver obs = (ConnectionObserver)this.channel.attr(OWNER).getAndSet((Object)ConnectionObserver.emptyListener());
                this.pool.release(this.channel).addListener(f -> {
                    if (log.isDebugEnabled() && !f.isSuccess()) {
                        log.debug("Failed cleaning the channel from pool", f.cause());
                    }
                    this.onTerminate.onComplete();
                    obs.onStateChange(connection, ConnectionObserver.State.RELEASED);
                });
                return;
            }
            this.owner().onStateChange(connection, newState);
        }

        public String toString() {
            return "PooledConnection{channel=" + this.channel + '}';
        }
    }

    static final class PendingConnectionObserver
    implements ConnectionObserver {
        final Queue<Pending> pendingQueue = (Queue)Queues.unbounded((int)4).get();

        PendingConnectionObserver() {
        }

        @Override
        public void onUncaughtException(Connection connection, Throwable error) {
            this.pendingQueue.add(new Pending(connection, error, null));
        }

        @Override
        public void onStateChange(Connection connection, ConnectionObserver.State newState) {
            this.pendingQueue.add(new Pending(connection, null, newState));
        }

        static class Pending {
            final Connection connection;
            final Throwable error;
            final ConnectionObserver.State state;

            Pending(Connection connection, @Nullable Throwable error, @Nullable ConnectionObserver.State state) {
                this.connection = connection;
                this.error = error;
                this.state = state;
            }
        }
    }

    static final class Pool
    extends AtomicBoolean
    implements ChannelPoolHandler,
    ChannelPool,
    ChannelHealthChecker {
        final ChannelPool pool;
        final EventLoopGroup defaultGroup;
        final Bootstrap bootstrap;
        final ChannelOperations.OnSetup opsFactory;
        final AtomicInteger activeConnections = new AtomicInteger();
        final AtomicInteger inactiveConnections = new AtomicInteger();
        final Future<Boolean> HEALTHY;
        final Future<Boolean> UNHEALTHY;

        Pool(Bootstrap bootstrap, PoolFactory provider, ChannelOperations.OnSetup opsFactory) {
            this.bootstrap = bootstrap;
            this.opsFactory = opsFactory;
            this.pool = provider.newPool(bootstrap, this, this);
            this.defaultGroup = bootstrap.config().group();
            this.HEALTHY = this.defaultGroup.next().newSucceededFuture((Object)true);
            this.UNHEALTHY = this.defaultGroup.next().newSucceededFuture((Object)false);
        }

        public Future<Boolean> isHealthy(Channel channel) {
            return channel.isActive() ? this.HEALTHY : this.UNHEALTHY;
        }

        public Future<Channel> acquire() {
            return this.acquire((Promise<Channel>)this.defaultGroup.next().newPromise());
        }

        public Future<Channel> acquire(Promise<Channel> promise) {
            return this.pool.acquire(promise);
        }

        public Future<Void> release(Channel channel) {
            return this.pool.release(channel);
        }

        public Future<Void> release(Channel channel, Promise<Void> promise) {
            return this.pool.release(channel, promise);
        }

        public void close() {
            if (this.compareAndSet(false, true)) {
                this.pool.close();
            }
        }

        public void channelReleased(Channel ch) {
            this.activeConnections.decrementAndGet();
            this.inactiveConnections.incrementAndGet();
            if (log.isDebugEnabled()) {
                log.debug(ReactorNetty.format(ch, "Channel cleaned, now {} active connections and {} inactive connections"), new Object[]{this.activeConnections, this.inactiveConnections});
            }
        }

        public void channelAcquired(Channel ch) {
        }

        public void channelCreated(Channel ch) {
            this.inactiveConnections.incrementAndGet();
            if (log.isDebugEnabled()) {
                log.debug(ReactorNetty.format(ch, "Created new pooled channel, now {} active connections and {} inactive connections"), new Object[]{this.activeConnections, this.inactiveConnections});
            }
            PooledConnection pooledConnection = new PooledConnection(ch, this);
            pooledConnection.bind();
            Bootstrap bootstrap = this.bootstrap.clone();
            BootstrapHandlers.finalizeHandler(bootstrap, this.opsFactory, (ConnectionObserver)pooledConnection);
            ch.pipeline().addFirst(new ChannelHandler[]{bootstrap.config().handler()});
        }

        @Override
        public String toString() {
            return "{ bootstrap=" + this.bootstrap + ", activeConnections=" + this.activeConnections + ", inactiveConnections=" + this.inactiveConnections + '}';
        }
    }

    static interface PoolFactory {
        public ChannelPool newPool(Bootstrap var1, ChannelPoolHandler var2, ChannelHealthChecker var3);
    }
}

