

#include <gnutls/gnutls.h>
#include <gnutls/x509.h>

/* This function will try to verify the peer's certificate, and
 * also check if the hostname matches, and the activation, expiration dates.
 */
void verify_certificate( gnutls_session session, const char* hostname)
{
   int status;

   /* This verification function uses the trusted CAs in the credentials
    * structure. So you must have installed one or more CA certificates.
    */
   status = gnutls_certificate_verify_peers(session);

   if (status == GNUTLS_E_NO_CERTIFICATE_FOUND) {
      printf("No certificate was sent\n");
      return;
   }

   if (status & GNUTLS_CERT_INVALID)
      printf("The certificate chain is broken.\n");

   if (status & GNUTLS_CERT_CORRUPTED)
      printf("The certificate is corrupted.\n");

   if (status & GNUTLS_CERT_REVOKED)
      printf("The certificate has been revoked.\n");

   if (status & GNUTLS_CERT_NOT_TRUSTED) {
      printf("The certificate is not trusted.\n");
      return;
   }


   if ( gnutls_certificate_type_get(session) == GNUTLS_CRT_X509) {
      const gnutls_datum* cert_list;
      int cert_list_size;
      gnutls_x509_crt cert;

      if ( gnutls_x509_crt_init( &cert) < 0) {
         printf("error in initialization\n");
         return;
      }

      cert_list = gnutls_certificate_get_peers( session, &cert_list_size);
      if ( cert_list == NULL) {
         printf("No certificate was found!\n");
         return;
      }

      /* Here we only check the first certificate in the given
       * chain.
       */
      if ( gnutls_x509_crt_import( cert, &cert_list[0], GNUTLS_X509_FMT_DER) < 0) {
         printf("error parsing certificate\n");
         return;
      }

      /* Beware here we do not check for errors.
       */
      if ( gnutls_x509_crt_get_expiration( cert) < time(0)) {
         printf("The certificate has expired\n");
         return;
      }

      if ( gnutls_x509_crt_get_activation_time( cert) > time(0)) {
         printf("The certificate is not yet activated\n");
         return;
      }

      if ( !gnutls_x509_crt_check_hostname( cert, hostname)) {
         printf("The certificate does not match hostname\n");
         return;
      }

      gnutls_x509_crt_deinit( cert);

   }

   printf("The certificate is trusted.\n");
   return;
}

